/***************************************************************************
 *                                                                         *
 *    LIBDSK: General floppy and diskimage access library                  *
 *    Copyright (C) 2001  John Elliott <jce@seasip.demon.co.uk>            *
 *                                                                         *
 *    This library is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU Library General Public          *
 *    License as published by the Free Software Foundation; either         *
 *    version 2 of the License, or (at your option) any later version.     *
 *                                                                         *
 *    This library is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    *
 *    Library General Public License for more details.                     *
 *                                                                         *
 *    You should have received a copy of the GNU Library General Public    *
 *    License along with this library; if not, write to the Free           *
 *    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,      *
 *    MA 02111-1307, USA                                                   *
 *                                                                         *
 ***************************************************************************/

/* This driver provides limited support for DSK files generated by IBM's 
 * SaveDskF utility, intended to be unpacked with their LoadDskF.
 *
 * TO DO
 * ~~~~~
 * 1. DSKF images only contain sectors used by the filesystem. Currently 
 *   attempts to access sectors outside this range will error; they should
 *   return blank sectors.
 *
 * 2. Sector writes do not update the checksum or other header fields. This
 *   should probably be done by a synchronisation function in dsk_close().
 *
 * 3. A second class needs to be added (cf dsk and edsk) so that images in
 *   the older dskf format (magic = 58AA) can be created.
 *
 * 4. Compression support (see compdskf.c) 
 *
 */

#include <stdio.h>
#include "libdsk.h"
#include "drvi.h"
#include "drvdskf.h"


/* This struct contains function pointers to the driver's functions, and the
 * size of its DSK_DRIVER subclass */

DRV_CLASS dc_dskf = 
{
	sizeof(DSKF_DSK_DRIVER),
	"dskf",
	"IBM LoadDskF driver",
	dskf_open,	/* open */
	dskf_creat,	/* create new */
	dskf_close,	/* close */
	dskf_read,	/* read sector, working from physical address */
	dskf_write,	/* write sector, working from physical address */
	dskf_format,	/* format track, physical */
	NULL,		/* get geometry */
	NULL,		/* sector ID */
	dskf_xseek,	/* seek to track */
	dskf_status,	/* drive status */
};

dsk_err_t dskf_open(DSK_DRIVER *self, const char *filename)
{
	int magic;
	DSKF_DSK_DRIVER *dskfself;
	char *comment;

	/* Sanity check: Is this meant for our driver? */
	if (self->dr_class != &dc_dskf) return DSK_ERR_BADPTR;
	dskfself = (DSKF_DSK_DRIVER *)self;

	dskfself->dskf_fp = fopen(filename, "r+b");
	if (!dskfself->dskf_fp) 
	{
		dskfself->dskf_readonly = 1;
		dskfself->dskf_fp = fopen(filename, "rb");
	}
	if (!dskfself->dskf_fp) return DSK_ERR_NOTME;

	/* Try to load the header */
	if (fread(dskfself->dskf_header, 1, sizeof(dskfself->dskf_header), 
			dskfself->dskf_fp) < sizeof(dskfself->dskf_header))
	{
		fclose(dskfself->dskf_fp);
		return DSK_ERR_NOTME;
	}
	magic = dskfself->dskf_header[1] * 256 + dskfself->dskf_header[0];
	/* Check the magic number.  */
	if (magic != 0x58AA && magic != 0x59AA)
	{
		fclose(dskfself->dskf_fp);
		return DSK_ERR_NOTME;
	}
	dskfself->dskf_comment = dskfself->dskf_header[36] + 
			           256 * dskfself->dskf_header[37];
	dskfself->dskf_datastart = dskfself->dskf_header[38] + 
			           256 * dskfself->dskf_header[39];

	if (dskfself->dskf_datastart == 0)
		dskfself->dskf_datastart = 512;

/* Record exact size, so we can tell if we're writing off the end
 * of the file. Under Windows, writing off the end of the file fills the 
 * gaps with random data, which can cause mess to appear in the directory;
 * and under UNIX, the entire directory is filled with zeroes. */
        if (fseek(dskfself->dskf_fp, 0, SEEK_END)) return DSK_ERR_SYSERR;
        dskfself->dskf_filesize = ftell(dskfself->dskf_fp);

/* Load the comment, if any */
	if (dskfself->dskf_comment < dskfself->dskf_datastart)
	{
		int clen;

		if (fseek(dskfself->dskf_fp, dskfself->dskf_comment, SEEK_SET)) 
		{
			fclose(dskfself->dskf_fp);
			return DSK_ERR_SYSERR;
		}
		clen =	dskfself->dskf_datastart - dskfself->dskf_comment + 1;
		comment = dsk_malloc(clen);
		if (comment != NULL)
		{
			memset(comment, 0, clen);
			fread(comment, 1, clen - 1, dskfself->dskf_fp);
			dsk_set_comment(self, comment);
			dsk_free(comment);
		}
	}	

	return DSK_ERR_OK;
}


dsk_err_t dskf_creat(DSK_DRIVER *self, const char *filename)
{
	DSKF_DSK_DRIVER *dskfself;
	
	/* Sanity check: Is this meant for our driver? */
	if (self->dr_class != &dc_dskf) return DSK_ERR_BADPTR;
	dskfself = (DSKF_DSK_DRIVER *)self;
/*

	dskfself->dskf_fp = fopen(filename, "w+b");
	dskfself->dskf_readonly = 0;
	if (!dskfself->dskf_fp) return DSK_ERR_SYSERR;
	dskfself->dskf_filesize = 0;
	return DSK_ERR_OK;
	*/
	return DSK_ERR_NOTIMPL;
}


dsk_err_t dskf_close(DSK_DRIVER *self)
{
	DSKF_DSK_DRIVER *dskfself;

	if (self->dr_class != &dc_dskf) return DSK_ERR_BADPTR;
	dskfself = (DSKF_DSK_DRIVER *)self;

	/* XXX Update the checksum! */

	if (dskfself->dskf_fp) 
	{
		if (fclose(dskfself->dskf_fp) == EOF) return DSK_ERR_SYSERR;
		dskfself->dskf_fp = NULL;
	}
	return DSK_ERR_OK;	
}


dsk_err_t dskf_read(DSK_DRIVER *self, const DSK_GEOMETRY *geom,
                             void *buf, dsk_pcyl_t cylinder,
                              dsk_phead_t head, dsk_psect_t sector)
{
	DSKF_DSK_DRIVER *dskfself;
	long offset;

	if (!buf || !self || !geom || self->dr_class != &dc_dskf) return DSK_ERR_BADPTR;
	dskfself = (DSKF_DSK_DRIVER *)self;

	if (!dskfself->dskf_fp) return DSK_ERR_NOTRDY;

	/* Convert from physical to logical sector. However, unlike the dg_* 
	 * functions, this _always_ uses "SIDES_ALT" mapping; this is the 
	 * mapping that both the Linux and NT floppy drivers use to convert 
	 * offsets back into C/H/S. */
	/* XXX Use the header not the passed geometry */
	offset = (cylinder * geom->dg_heads) + head;	/* Drive track */
	offset *= geom->dg_sectors;
	offset += (sector - geom->dg_secbase);
	offset *=  geom->dg_secsize;
	offset +=  dskfself->dskf_datastart;
	
	if (fseek(dskfself->dskf_fp, offset, SEEK_SET)) return DSK_ERR_SYSERR;

	if (fread(buf, 1, geom->dg_secsize, dskfself->dskf_fp) < geom->dg_secsize)
	{
		return DSK_ERR_NOADDR;
	}
	return DSK_ERR_OK;
}


static dsk_err_t seekto(DSKF_DSK_DRIVER *self, unsigned long offset)
{
	/* 0.9.5: Fill any "holes" in the file with 0xE5. Otherwise, UNIX would
	 * fill them with zeroes and Windows would fill them with whatever
	 * happened to be lying around */
	if (self->dskf_filesize < offset)
	{
		if (fseek(self->dskf_fp, self->dskf_filesize, SEEK_SET)) return DSK_ERR_SYSERR;
		while (self->dskf_filesize < offset)
		{
			if (fputc(0xE5, self->dskf_fp) == EOF) return DSK_ERR_SYSERR;
			++self->dskf_filesize;
		}
	}
	if (fseek(self->dskf_fp, offset, SEEK_SET)) return DSK_ERR_SYSERR;
	return DSK_ERR_OK;
}

dsk_err_t dskf_write(DSK_DRIVER *self, const DSK_GEOMETRY *geom,
                             const void *buf, dsk_pcyl_t cylinder,
                              dsk_phead_t head, dsk_psect_t sector)
{
	DSKF_DSK_DRIVER *dskfself;
	unsigned long offset;
	dsk_err_t err;

	if (!buf || !self || !geom || self->dr_class != &dc_dskf) return DSK_ERR_BADPTR;
	dskfself = (DSKF_DSK_DRIVER *)self;

	if (!dskfself->dskf_fp) return DSK_ERR_NOTRDY;
	if (dskfself->dskf_readonly) return DSK_ERR_RDONLY;

	/* Convert from physical to logical sector. However, unlike the dg_* 
	 * functions, this _always_ uses "SIDES_ALT" mapping; this is the 
	 * mapping that both the Linux and NT floppy drivers use to convert 
	 * offsets back into C/H/S. */
	offset = (cylinder * geom->dg_heads) + head;	/* Drive track */
	offset *= geom->dg_sectors;
	offset += (sector - geom->dg_secbase);
	offset *=  geom->dg_secsize;
	offset +=  dskfself->dskf_datastart;

	err = seekto(dskfself, offset);
	if (err) return err;

	if (fwrite(buf, 1, geom->dg_secsize, dskfself->dskf_fp) < geom->dg_secsize)
	{
		return DSK_ERR_NOADDR;
	}
	if (dskfself->dskf_filesize < offset + geom->dg_secsize)
		dskfself->dskf_filesize = offset + geom->dg_secsize;
	return DSK_ERR_OK;
}


dsk_err_t dskf_format(DSK_DRIVER *self, DSK_GEOMETRY *geom,
                                dsk_pcyl_t cylinder, dsk_phead_t head,
                                const DSK_FORMAT *format, unsigned char filler)
{
/*
 * Note that we completely ignore the "format" parameter, since raw DSKF
 * images don't hold track headers.
 */
	DSKF_DSK_DRIVER *dskfself;
	unsigned long offset;
	unsigned long trklen;
	dsk_err_t err;

   (void)format;
	if (!self || !geom || self->dr_class != &dc_dskf) return DSK_ERR_BADPTR;
	dskfself = (DSKF_DSK_DRIVER *)self;

	if (!dskfself->dskf_fp) return DSK_ERR_NOTRDY;
	if (dskfself->dskf_readonly) return DSK_ERR_RDONLY;

	/* Convert from physical to logical sector. However, unlike the dg_* 
	 * functions, this _always_ uses "SIDES_ALT" mapping; this is the 
	 * mapping that both the Linux and NT floppy drivers use to convert 
	 * offsets back into C/H/S. */
	offset = (cylinder * geom->dg_heads) + head;	/* Drive track */
	trklen = geom->dg_sectors * geom->dg_secsize;
	offset *= trklen;
	offset +=  dskfself->dskf_datastart;

	err = seekto(dskfself, offset);
	if (err) return err;
	if (dskfself->dskf_filesize < offset + trklen)
		dskfself->dskf_filesize = offset + trklen;

	while (trklen--) 
		if (fputc(filler, dskfself->dskf_fp) == EOF) return DSK_ERR_SYSERR;	

	return DSK_ERR_OK;
}

	

dsk_err_t dskf_xseek(DSK_DRIVER *self, const DSK_GEOMETRY *geom,
                      dsk_pcyl_t cylinder, dsk_phead_t head)
{
	DSKF_DSK_DRIVER *dskfself;
	long offset;

	if (!self || !geom || self->dr_class != &dc_dskf) return DSK_ERR_BADPTR;
	dskfself = (DSKF_DSK_DRIVER *)self;

	if (!dskfself->dskf_fp) return DSK_ERR_NOTRDY;

	if (cylinder >= geom->dg_cylinders || head >= geom->dg_heads)
		return DSK_ERR_SEEKFAIL;

	/* Convert from physical to logical sector. However, unlike the dg_* 
	 * functions, this _always_ uses "SIDES_ALT" mapping; this is the 
	 * mapping that both the Linux and NT floppy drivers use to convert 
	 * offsets back into C/H/S. */
	offset = (cylinder * geom->dg_heads) + head;	/* Drive track */
	offset *= geom->dg_sectors * geom->dg_secsize;
	offset +=  dskfself->dskf_datastart;
	
	if (fseek(dskfself->dskf_fp, offset, SEEK_SET)) return DSK_ERR_SEEKFAIL;

	return DSK_ERR_OK;
}

dsk_err_t dskf_status(DSK_DRIVER *self, const DSK_GEOMETRY *geom,
                      dsk_phead_t head, unsigned char *result)
{
	DSKF_DSK_DRIVER *dskfself;

	if (!self || !geom || self->dr_class != &dc_dskf) return DSK_ERR_BADPTR;
	dskfself = (DSKF_DSK_DRIVER *)self;

	if (!dskfself->dskf_fp) *result &= ~DSK_ST3_READY;
	if (dskfself->dskf_readonly) *result |= DSK_ST3_RO;
	return DSK_ERR_OK;
}

