
/* 
 * Dos/PC Emulator
 * Copyright (C) 1991 Jim Hudgens
 * 
 * 
 * The file is part of GDE.
 * 
 * GDE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 * 
 * GDE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GDE; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */

#ifdef DOSS_SUPPORT

static char rcsid[]=
  "$Id: doss_file.c,v 1.2 1991/07/30 01:52:37 hudgens Exp hudgens $";

/* $Log: doss_file.c,v $
 * Revision 1.2  1991/07/30  01:52:37  hudgens
 * added copyright.
 *
 * Revision 1.1  1991/07/20  04:06:16  hudgens
 * Initial revision
 *
 *
 */



#include "gde.h"

#include <ctype.h>




/* 
  routine called at system initialization.
  causes initialization of all connected (i.e. configured)  disk drives
*/


void DEFUN(doss_fs_init,(m),PC_ENV *m)
{
    int i;
    int err;
    
    m->doss_drives = doss_sys_drives;
    m->doss_ndrives = 0;

    for (i=0; i<DOSS_MAX_DRIVES; i++)
	{
	   if (m->doss_drives[i].flags & DOSS_VALID_DRIVE)
	       {
#ifdef DEBUG
		  if (DEBUG_FS(m))
		      {
			 debug_printf(m,
				      "DOSS DRIVE %d valid as %c...\n",i,
				      i+'A');
			 switch(m->doss_drives[i].fs_type)
			     {
			      case unix_fs:
				debug_printf(m,
					     "\ttype unix_fs\tpath=%s\n",
				       m->doss_drives[i].path);
				break;
			      default:
				break;
			     }
		      }
#endif
		  /* call the fs dependent routine to do the initialization
		     for this drive.  Note that this is the general
		     form for accessing the fs dependant parts of 
		     the drive specification. 
		     
		     m->doss_drives is an array of drives
		     
		     m->doss_drives[i] is drive unit #i

		     m->doss_drives[i].fs_ops  is a field of this 
		        drive specification, which is a pointer to
			a structure of type doss_fs.  

		     m->doss_drives[i].fs_ops->initialize_fs 
		        is a pointer to a function, which is the 
			fs specific routine for filesystem initialization
		   */

		  err = (*m->doss_drives[i].fs_ops->initialize_fs)
		    (m,&m->doss_drives[i]);
		  
		  if (err != FSOP_SUCCESS)
		      {
			 m->doss_drives[i].flags &= ~DOSS_VALID_DRIVE;
		      }
		  else
		      {
			 m->doss_ndrives += 1;
			 m->doss_lastdrive = i;
		      }
		  

	       }
	}
    
}	     




/* 
 *
 *  doss_filetab_init(m)
 *   PC_ENV *m;
 *
 *  this is a routine which initializes the system wide file
 *  table.  
 *
 */

void DEFUN(doss_filetab_init,(m),
	   PC_ENV *m)
 {
    
    int i;
    

    /* make it visible outside this module*/
    m->doss_files = doss_sys_file;
   
    /* clear out the flags and the count */
    
    for (i=0; i<DOSS_MAX_SYS_FH; i++)
	{
	   m->doss_files[i].flags = 0;
	   m->doss_files[i].count = 0;
	}

    /* initialize all the predefined character devices */
       
    /* CON */
    m->doss_files[0].flags = 
      DOSS_FH_ISCHAR|DOSS_FH_VALID|DOSS_FH_ISCONIO;
    /* note that the latter flag is CONIN|CONOUT */
    
    m->doss_files[0].count = 3;
    strcpy(m->doss_files[0].path,"CON");
    
    /* PRN */
    m->doss_files[1].flags = DOSS_FH_ISCHAR|DOSS_FH_VALID;
    m->doss_files[1].count = 1;
    strcpy(m->doss_files[1].path,"PRN");
    
    /* AUX */
    m->doss_files[2].flags = DOSS_FH_ISCHAR|DOSS_FH_VALID;
    m->doss_files[2].count = 1;
    strcpy(m->doss_files[2].path,"AUX");


    m->doss_nfiles=3;
    

 }


    
    
/* 
 *  doss_fsop_fopen(sys,path,mode,fh)
 *    PC_ENV *m;
 *    char   *path;
 *    int    mode;
 *    int   *fh;
 *  open file whose name is in path, and with access
 *  defined by mode. 
 *  return the filehandle in fh.
 *
 */

int DEFUN(doss_fsop_fopen,(m,path,mode,fh),
	  PC_ENV *m  AND
	  char   *path AND
	  int    mode AND 
	  int   *fh)
 {
    
    int retval;
    int sfh;
    int ufh;
    int drive;
    int i;
    char newpath[DOSS_MAX_PATH * 2];


    ufh = doss_get_next_ufh(m);
    if (ufh == -1)
	{
	   syscall_errno = 4;
	   return SYSCALL_FAILURE;
	}
    
    doss_cvt_path(m, &drive, path, newpath);

    if ((m -> doss_drives [drive].flags & DOSS_VALID_DRIVE) == 0)
	{
	   /* book doesn't mention this one. */
	   syscall_errno = 15;  /* drive invalid */
	   return SYSCALL_FAILURE;
	}

    sfh = -1;
    /* look for an empty slot in the system file table. */
    for (i=0; i<DOSS_MAX_SYS_FH; i++)
	{
	   if (m->doss_files[i].flags & DOSS_FH_VALID) continue;
	   sfh = i;  /* found an empty file handle. */
	   break;
	}
    if (sfh < 0)  /* handle not found. */
	{
	   syscall_errno = 4;  /*no free handles */
	   return SYSCALL_FAILURE;
	}

    /* fill in the file handle's state. */
    m->doss_files[sfh].unit = drive;
    m->doss_files[sfh].flags = DOSS_FH_VALID;
    m->doss_files[sfh].access = mode;
    strcpy(m->doss_files[sfh].path,newpath);
    
    /* now call the fs specific code to do the open.  
       pass the drive structure
       pass the file_handle structure
     */

    retval = (*m->doss_drives[drive].fs_ops->open_file)
      (m, m->doss_drives+drive,
       m->doss_files+sfh);

    if (retval== FSOP_SUCCESS)
	{
	   doss_set_fh(m,ufh,sfh);
	   *fh  = ufh;
	   m->doss_files[sfh].count += 1;
	   m->doss_nfiles += 1;
	   return  SYSCALL_SUCCESS;
	}
    else
	{
	   /* free the file table entry by marking it invalid */
	   m->doss_files[sfh].flags = 0;
	   return SYSCALL_FAILURE;
	}
 }
    
	   
  
    
    
/* 
 *  doss_fsop_fclose(sys,fh)
 *    PC_ENV *m;
 *    int   fh;
 *
 *  close file  given by filehandle fh
 *
 */

int DEFUN(doss_fsop_fclose,(m,fh),
	  PC_ENV *m AND
	  int   fh)
 {
    
    int retval;
    int ufh = fh;
    int sfh;
    int drive;


    sfh = doss_ufh_to_sfh(m,ufh);

    drive = m->doss_files[sfh].unit;
   
    if ((m -> doss_drives [drive].flags & DOSS_VALID_DRIVE) == 0)
	{
	   /* book doesn't mention this one. */
	   syscall_errno = 15;  /* drive invalid */
	   return SYSCALL_FAILURE;
	}
    
    /* now call the fs specific code to do the close.  
       pass the drive structure
       pass the file_handle
     */

    retval = (*m->doss_drives[drive].fs_ops->close_file)
      (m,m->doss_drives+drive,
       m->doss_files+sfh);

    if (retval== FSOP_SUCCESS)
	{
	   doss_set_fh(m,ufh,0xff);   /* mark invalid */
	   m->doss_files[sfh].count -= 1;
	   if (m->doss_files[sfh].count == 0)
	     {
		m->doss_nfiles -= 1;
		m->doss_files[sfh].flags &= ~ DOSS_FH_VALID;
	     }
	   return  SYSCALL_SUCCESS;
	}
    else
	{
	   return SYSCALL_FAILURE;
	}
 }
    
	   
  
    
    
    
	 
    
/* 
 *  doss_fsop_fcreat(sys,path,attr,fh)
 *    PC_ENV *m;
 *    char   *path;
 *    int    attr;
 *    int   *fh;
 *  create file whose name is in path, and with attributes
 *  defined by attr,
 *  return the filehandle in fh.
 *
 */

int DEFUN(doss_fsop_fcreat,(m,path,attr,fh),
	  PC_ENV *m AND
	  char   *path AND
	  int    attr AND
	  int   *fh)
 {
    
    int retval;
    int sfh;
    int ufh;
    int drive;
    int i;
    char newpath[DOSS_MAX_PATH * 2];


    ufh = doss_get_next_ufh(m);
    if (ufh == -1)
	{
	   syscall_errno = 4;
	   return SYSCALL_FAILURE;
	}

    doss_cvt_path(m, &drive, path, newpath);
    
    if ((m -> doss_drives [drive].flags & DOSS_VALID_DRIVE) == 0)
	{
	   /* book doesn't mention this one. */
	   syscall_errno = 15;  /* drive invalid */
	   return SYSCALL_FAILURE;
	}

    sfh = -1;
    /* look for an empty slot in the system file table. */
    for (i=0; i<DOSS_MAX_SYS_FH; i++)
	{
	   if (m->doss_files[i].flags & DOSS_FH_VALID) continue;
	   sfh = i;  /* found an empty file handle. */
	   break;
	}
    if (sfh < 0)  /* handle not found. */
	{
	   syscall_errno = 4;  /*no free handles */
	   return SYSCALL_FAILURE;
	}
    
    /* fill in the file handle's state. */
    m->doss_files[sfh].unit = drive;
    m->doss_files[sfh].flags = DOSS_FH_VALID;
    m->doss_files[sfh].attrib = attr;
    strcpy(m->doss_files[sfh].path,newpath);
    
    /* now call the fs specific code to do the create
       pass the drive structure
       pass the file_handle structure
     */


    retval = (*m->doss_drives[drive].fs_ops->create_file)
      (m, m->doss_drives+drive,
       m->doss_files+sfh);

    if (retval== FSOP_SUCCESS)
	{
	   doss_set_fh(m,ufh,sfh);
	   *fh  = ufh;
	   m->doss_files[sfh].count += 1;
	   m->doss_nfiles += 1;
	   return  SYSCALL_SUCCESS;
	}
    else
	{
	   /* free the file table entry by marking it invalid */
	   m->doss_files[sfh].flags = 0;
	   return SYSCALL_FAILURE;
	}
 }
    

    
/* 
 *  doss_fsop_fread(m,fh,buf,n,nread)
 *    PC_ENV *m;
 *    int   fh;
 *    char *buff;
 *    int  n;
 *    int *nread;
 *
 *  read from file whose given file handle fh.
 *  use buffer buf, and count n.
 *  return the amount read in nread.
 *
 */

int DEFUN(doss_fsop_fread,(m,fh,buf,n,nread),
	  PC_ENV *m AND
	  int   fh  AND
	  void *buf AND
	  int  n    AND
	  int *nread)
 {
    
    int retval;
    int drive;
    int sfh,ufh=fh;
    int i;
    

    sfh = doss_ufh_to_sfh(m,ufh);

    /* check that we are passed a valid handle */
    if ( (m->doss_files[sfh].flags & DOSS_FH_VALID)== 0)
	{
	   syscall_errno = DOSS_EINVALHANDLE;
	   return SYSCALL_FAILURE;
	}
    else if ((m->doss_files[sfh].flags & DOSS_FH_ISCHAR)
	     && (m->doss_files[sfh].flags & DOSS_FH_ISCONIN) )
	{
	   i = 0;
	   while (i < n) 
	       {
		  if (doss_conio_in_chr(m,(char*)buf+i,0) == SYSCALL_SUCCESS)
		      {
			 i += 1;
		      }
		  else
		    break;
	       }
	   *nread = i;
	   retval = FSOP_SUCCESS;
	}
    else
	{
	   drive =  m->doss_files[sfh].unit;
	   retval = (*m->doss_drives[drive].fs_ops->read_file)
	     (m, m->doss_drives+drive,
	      m->doss_files+sfh,
	      buf,
	      n,
	      nread);
	}
    return  retval;
 }
    



/* 
 *  doss_fsop_fwrite(m,fh,buf,n,nwrite)
 *    PC_ENV *m;
 *    int   fh;
 *    char *buf;
 *    int  n;
 *    int *nwrite;
 *
 *  write to file whose given file handle fh.
 *  use buffer buf, and count n.
 *  return the amount read in nwrite.
 *
 */

int DEFUN(doss_fsop_fwrite,(m,fh,buf,n,nwrite),
	  PC_ENV *m AND
	  int   fh  AND
	  void *buf AND
	  int  n    AND
	  int *nwrite)
 {
    
    int retval;
    int drive;
    int i;
    int sfh,ufh=fh;
    
    sfh = doss_ufh_to_sfh(m,ufh);

    /* check that we are passed a valid handle */
    if ( (m->doss_files[sfh].flags & DOSS_FH_VALID)== 0)
	{
	   syscall_errno = DOSS_EINVALHANDLE;
	   return SYSCALL_FAILURE;
	}
    else if ((m->doss_files[sfh].flags & DOSS_FH_ISCHAR) &&
	     (m->doss_files[sfh].flags & DOSS_FH_ISCONOUT))
	{
	   i = 0;
	   while (i < n) 
	       {
		  doss_conio_out_chr(m,((char*)buf)[i],0);
		  i += 1;
	       }
	   *nwrite = n;
	   retval = FSOP_SUCCESS;
	}
    else
	{
	   drive =  m->doss_files[sfh].unit;

	   retval = (*m->doss_drives[drive].fs_ops->write_file)
	     (m, m->doss_drives+drive,
	      m->doss_files+sfh,
	      buf,
	      n,
	      nwrite);
	}
    

    return  retval;
 }
    



/* 
 *  doss_fsop_fseek(m,fh,whence,offset,at)
 *    PC_ENV *m;
 *    int   fh;
 *    int   whence,offset;
 *    int   *at;
 *
 *  given file handle fh, seek to position offset relative
 *  to whence (0=beginning, 1=current, 2=end).
 *  return the current pos in at
 *
 */

int DEFUN(doss_fsop_fseek,(m,fh,whence,offset,at),
	  PC_ENV *m AND
	  int   fh AND
	  int32 whence AND
	  int32 offset AND
	  int32 *at)
 {
    
    int retval;
    int drive;
    int ufh=fh;
    int sfh;
    
    sfh = doss_ufh_to_sfh(m,ufh);

    /* check that we are passed a valid handle */
    if ( (m->doss_files[sfh].flags & DOSS_FH_VALID)== 0)
	{
	   syscall_errno = DOSS_EINVALHANDLE;
	   return SYSCALL_FAILURE;
	}
    else if (m->doss_files[sfh].flags & DOSS_FH_ISCHAR)
	{
	   syscall_errno = DOSS_EINVALHANDLE;
	   retval = SYSCALL_FAILURE;
	}
    else
	{
	   drive =  m->doss_files[sfh].unit;
	   retval = (*m->doss_drives[drive].fs_ops->seek_file)
	     (m, m->doss_drives+drive,
	      m->doss_files+sfh,
	      whence,
	      offset,
	      at);
	}
    

    return  retval;
 }
    






/* 
 *  doss_fsop_fdelete(m,path)
 *    PC_ENV *m;
 *    char *path
 *
 *  delete file given by path.
 *
 */

int DEFUN(doss_fsop_fdelete,(m,path),
	  PC_ENV *m AND
	  char *path)
 {
    
    int retval;
    int drive;
    char newpath[DOSS_MAX_PATH * 2];

    doss_cvt_path(m, &drive, path, newpath);

    if ((m -> doss_drives [drive].flags & DOSS_VALID_DRIVE) == 0)
	{
	   /* book doesn't mention this one. */
	   syscall_errno = 15;  /* drive invalid */
	   return SYSCALL_FAILURE;
	}

    retval = (*m->doss_drives[drive].fs_ops->delete_file)
	     (m, m->doss_drives+drive,
	      newpath);

    return  retval;
 }
    



/* 
 *  doss_fsop_getftime(m,ufh,date,time)
 *    PC_ENV *m;
 *    int ufh;
 *    u_int16 *date,*time
 *  
 *    get date and time of file.
 *
 */

int DEFUN(doss_fsop_getftime,(m,ufh,date,time),
	  PC_ENV *m AND
	  int ufh   AND
	  u_int16 *date AND
	  u_int16 *time)
 {
    
    int retval;
    int drive;
    int sfh;

    sfh = doss_ufh_to_sfh(m,ufh);

    drive = m->doss_files[sfh].unit;
  
    if ((m -> doss_drives [drive].flags & DOSS_VALID_DRIVE) == 0)
	{
	   /* book doesn't mention this one. */
	   syscall_errno = 15;  /* drive invalid */
	   return SYSCALL_FAILURE;
	}


    retval = (*m->doss_drives[drive].fs_ops->get_attr)
	     (m, m->doss_drives+drive,
	      m->doss_files+sfh);

    *date = m->doss_files[sfh].mdate;
    *time = m->doss_files[sfh].mtime;

    return  retval;
 }
    



/* 
 *  doss_fsop_setftime(m,ufh,date,time)
 *    PC_ENV *m;
 *    int ufh;
 *    u_int16 *date,*time
 *  
 *    set date and time of file.
 *
 */

int DEFUN(doss_fsop_setftime,(m,ufh,date,time),
	  PC_ENV *m AND
	  int ufh   AND
	  u_int16 date AND
	  u_int16 time)
 {
    
    int retval;
    int sfh;
    int drive;

    sfh = doss_ufh_to_sfh(m,ufh);

    drive = m->doss_files[sfh].unit;

   
    if ((m -> doss_drives [drive].flags & DOSS_VALID_DRIVE) == 0)
	{
	   /* book doesn't mention this one. */
	   syscall_errno = 15;  /* drive invalid */
	   return SYSCALL_FAILURE;
	}


    /* get the old stuff first. */
    retval = (*m->doss_drives[drive].fs_ops->get_attr)
	     (m, m->doss_drives+drive,
	      m->doss_files+sfh);

    if (retval == FSOP_SUCCESS)
	{


	   m->doss_files[sfh].mdate = date;
	   m->doss_files[sfh].mtime = time;
	   
	   /* set the date and time. */
	   retval = (*m->doss_drives[drive].fs_ops->set_attr)
	     (m, m->doss_drives+drive,
	      m->doss_files+sfh);
	}
	   
    return  retval;
 }
    



/* 
 *  doss_fsop_getfattr(m,path,attr)
 *    PC_ENV *m;
 *    char *path
 *    u_int16 *attr
 *  
 *    get file attrib bits.
 *
 */

int DEFUN(doss_fsop_getfattr,(m,path,attr),
	  PC_ENV *m AND
	  char *path AND
	  u_int16 *attr)
 {
    
    int retval;
    int drive;
    struct doss_file_handle fh;
    char newpath[DOSS_MAX_PATH * 2];


    doss_cvt_path(m, &drive, path, newpath);
    if ((m -> doss_drives [drive].flags & DOSS_VALID_DRIVE) == 0)
	{
	   /* book doesn't mention this one. */
	   syscall_errno = 15;  /* drive invalid */
	   return SYSCALL_FAILURE;
	}

    /* fill in the file handle's state. */
    fh.unit = drive;
    strcpy(fh.path,newpath);

    retval = (*m->doss_drives[drive].fs_ops->get_attr)
	     (m, m->doss_drives+drive,
	      &fh);

    *attr = fh.attrib;

    return  retval;
 }
    



/* 
 *  doss_fsop_setftime(m,path,attr)
 *    PC_ENV *m;
 *    char *path
 *    u_int16 attr
 *  
 *    set file attrib bits.
 *
 */

int DEFUN(doss_fsop_setfattr,(m,path,attr),
	  PC_ENV *m AND
	  char *path AND
	  u_int16 attr)
 {
    
    int retval;
    int drive;
    struct doss_file_handle fh;
    char newpath[DOSS_MAX_PATH * 2];

    doss_cvt_path(m, &drive, path, newpath);

    if ((m -> doss_drives [drive].flags & DOSS_VALID_DRIVE) == 0)
	{
	   /* book doesn't mention this one. */
	   syscall_errno = 15;  /* drive invalid */
	   return SYSCALL_FAILURE;
	}


    /* fill in the file handle's state. */
    fh.unit = drive;
    strcpy(fh.path,newpath);

    /* get the old stuff first. */
    retval = (*m->doss_drives[drive].fs_ops->get_attr)
	     (m, m->doss_drives+drive,
	      &fh);

    if (retval == FSOP_SUCCESS)
	{

	   fh.attrib = attr & 0x27;  /* turn off dir/vol bits */
	   
	   /* set them bits. */
	   retval = (*m->doss_drives[drive].fs_ops->set_attr)
	     (m, m->doss_drives+drive,
	      &fh);
	}
	   
    return  retval;
 }
    

/*
 *
 *
 * 
 *
 *    
 */
 

int DEFUN(doss_fsop_ioctl_sf0,(m,fh,bits),
	  PC_ENV *m AND
	  int fh AND
	  u_int16 *bits)
 {
    
    int sfh,ufh=fh;
    

    sfh = doss_ufh_to_sfh(m,ufh);

    /* check that we are passed a valid handle */
    if ( (m->doss_files[sfh].flags & DOSS_FH_VALID)== 0)
	{
	   syscall_errno = DOSS_EINVALHANDLE;
	   return SYSCALL_FAILURE;
	}


    * bits = 0;
    
 
    if (m->doss_files[sfh].flags & DOSS_FH_ISCHAR)
	{
	   /* minor hackery */
	   switch(fh)
	       {
		case 0:
		  *bits |= DOSS_FH_ISCONIN  | DOSS_FH_ISCHAR;
		  break;
		case 1:
		case 2:
		  *bits |= DOSS_FH_ISCONOUT | DOSS_FH_ISCHAR; 
		  break;
	       }
	   
	}
    

    return SYSCALL_SUCCESS;
    
 }
    

#endif
