
/* 
 * Dos/PC Emulator
 * Copyright (C) 1991 Jim Hudgens
 * 
 * 
 * The file is part of GDE.
 * 
 * GDE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 * 
 * GDE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GDE; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */

#ifdef DOSS_SUPPORT

static char rcsid[]=
  "$Id: doss_dir.c,v 1.4 1991/07/30 01:51:04 hudgens Exp hudgens $";

/* $Log: doss_dir.c,v $
 * Revision 1.4  1991/07/30  01:51:04  hudgens
 * added copyright.
 *
 * Revision 1.3  1991/07/21  18:19:43  hudgens
 * Fixed problem in the CVT_PATH subroutine, which was causing incorrect
 * transformations of the path.  Specifically,  /foo/.. was not handled
 * correctly.
 *
 * Revision 1.2  1991/07/21  02:01:22  hudgens
 * fixed problem with cvt_path.
 *
 * Revision 1.1  1991/07/20  04:05:46  hudgens
 * Initial revision
 *
 *
 */

#include "gde.h"

int ucasefns = 0;


/* newpath should have storage for at least 2 * MAXPATHLEN. */

/* 
  make path canonical. 
  that is, remove all redundant information in the 
     path specfification.  
  Also, hide all drive information.
  Also, expand relative pathnames to fully qualified pathnames.
  Also, allow the pathnames to be made either fully uppercase
     or fully lowercase (as configured by the user).  
 */


void DEFUN(doss_cvt_path,
	   (m, drive, oldpath,  newpath),
	   PC_ENV *m AND  int *drive AND char *oldpath AND char *newpath)
 {
    
    char *p=oldpath;
    char *n=newpath;
    char *tln,*ln;
    
    /* should
       1. convert everything to proper (configurable) case.
       2. remove drive info or handle the default drive if there is
       no drive information present.
       3. if a relative filename (i.e. if it begins with something
       other than a \, then we want to add in the current path.
       
    */

    /* is drive information present? */

    if (oldpath[1] == ':') 
	{
	   *drive = (islower(oldpath[0])?oldpath[0]:tolower(oldpath[0])) - 'a';
	   p = oldpath+2;
	}
    else
	{
	   p = oldpath;
	   *drive = m->doss_curr_drive;
	}

    /* is the file/directory expressed using an absolute pathname? */

/*    printf("DCP:'%s'\n",oldpath);*/
    
    if ( *p != '\\' ) 
	{
	   /* relative filename.  Let's copy in the drive's default path. */
	   /* drive default paths should always be absolute... */

	   if ( strcmp(m->doss_drives[*drive].curr_path,"\\") != 0)
	     sprintf(n,"%s\\",m->doss_drives[*drive].curr_path);
	   else
	     sprintf(n,"\\");

	   ln = n + strlen(n) - 1;
	   n += strlen(n);
	}

    /* now begin the copy. */
    /* we are trying to eliminate constructs such as 

       foo\..\
       foo\.\
       foo\\

       also fixed:  
     
       foo\..

    */

    /* 
      XXX
      note that this simplistic algorithm is not extremely
      robust at all, but works in "normal" cases. 
      it fails, for example, on something like
       
         \\foo\\bar\\..\\..\\baz
       
      by converting it into:
       
         \\foo\\baz

    */

    while (*p)
	{
	   switch (*p)
	       {
		case '.':
		  if (strncmp(p,"..",2) == 0)
		      {
			 /* must back up to the second previous
			    backslash.  */
			 p += 2;
			 n = ln;  
			 tln = n-1;
			 while (tln >= newpath && *tln != '\\') tln--;
			 if (*tln == '\\' )  /* at previous */
			   {
			      n = tln;
			      tln--;
			      while (tln >= newpath && *tln != '\\') tln--;
			      if (*tln == '\\') 
				{
				   ln = tln;
				}
			   }
		      }
		  else if (strncmp(p,".\\",2) == 0)
		      {
			 p += 2;
		      }
		  else
		      {
			 *n = *p;
			 n += 1;
			 p += 1;
		      }
		  break;

		case '\\':
		  if (strncmp(p,"\\..\\",4) == 0) 
		      {
			 p += 3;  /* advance over the input. */
			 n = ln;  /* backup to last slash on output */
			 tln = n-1; 

			 /* 
			   now try to find previous \ in output.
			   attempt to backup "ln" backup to the previous
			   `\\' so as to handle ../.. situations.
			  */

			 while (tln >= newpath && *tln != '\\') tln--;
			 if (*tln == '\\') ln = tln;
		      }
		  else if (strcmp(p,"\\..")==0)   /* trailing... */
		      {
			 p += 2;
			 n = ln;
			 tln = n-1;
			 while (tln >= newpath && *tln != '\\') tln--;
			 if (*tln == '\\') ln = tln;
		      }
		  else if (strncmp(p,"\\.\\",3) == 0)
		      {
			 p += 2; /* advance over the unwanted input. */
		      }
		  else if (strcmp(p,"\\.") == 0)
		      {
			 p += 2; /* advance over the unwanted input. */
		      }
		  else if (strncmp(p,"\\\\",2) == 0)
		      {
			 p += 1; /* advance over unwanted input. */
		      }
		  else
		      {
			 *n = *p;  /* do the copy of the \ */
			 ln = n;   /* mark this as the "previous" \ */
			 n += 1;
			 p += 1;
		      }
		  break;
		default:	

		  /* do case folding on everything else */
		  if (ucasefns)
		    *n = (islower(*p)?toupper(*p):*p);
		  else
		    *n = (islower(*p)?*p:tolower(*p));
		  n += 1;
		  p += 1;
	       }
	}
    /* null terminate ... */
    *n = '\0';
    
 }



/* doss service #0x47:  get current directory.
   This is a simple service to implement, in that the information
   is stored in the drive table structure.

   Two inputs:
     1.  Drive specification (0..maxdrives)
	 [JCE] Not quite. 0 => current drive, 1+ => specified drives.
     2.  buffer to store the drive information.

 */


int DEFUN(doss_fsop_getcwd,(m,drive,path),
	   PC_ENV *m AND int drive AND     char *path)
 {
	if (drive == 0)
	{ 
		drive = m->doss_curr_drive;
	}
	else	
	{
		--drive;
	} 
	if (m -> doss_drives [drive].flags & DOSS_VALID_DRIVE)
	{
		/* according to _ADVANCED MSDOS_, the leading
		 backslash is eliminated, so we start at the second
		 character.  */
		sprintf(path,"%s",m->doss_drives[drive].curr_path + 1);
	}
	else
	{
		syscall_errno = 15;  /* drive invalid */
		return SYSCALL_FAILURE;
	}
	return SYSCALL_SUCCESS;
 }
    




/* doss service #0x3B:  set current directory.

   This is a fairly simple service to implement, 
   in that the information is copied
   in the drive table structure.  However, we must ascertain
   that the directory exists before setting it as our
   current directory.

   One inputs:
     path to directory

 */


int DEFUN(doss_fsop_setcwd,(m,path),
	  PC_ENV *m AND  char *path )
 {
    int drive;
    char newpath[DOSS_MAX_PATH*2];
        
    doss_cvt_path(m, &drive, path, newpath);

    
    if (m -> doss_drives [drive].flags & DOSS_VALID_DRIVE)
	{
	   /* XXX access path and see if valid */
	   /* fixme. */
	   sprintf(m->doss_drives[drive].curr_path,"%s",newpath);
#ifdef DEBUG
	   if(DEBUG_SVC(m))
	       {
		  debug_printf(m,"path=%c:%s\n",
			  drive+'A', 
			  m->doss_drives[drive].curr_path);
	       }
#endif	   
	}
    else
	{
	   syscall_errno = 15;  /* drive invalid */
	   return SYSCALL_FAILURE;
	}
    return SYSCALL_SUCCESS;
 }
    

	       
		    

		       
/* 
 *  doss_fsop_mkdir(m,path)
 *    PC_ENV *m;
 *    char   *path;
 *
 *  make directory given by path
 *
 */

int DEFUN(doss_fsop_mkdir,(m,path),PC_ENV *m AND char   *path)
 {
    
    int retval;
    int drive;
    char newpath[DOSS_MAX_PATH * 2];

    /* convert to canonical form */
    doss_cvt_path(m, &drive, path, newpath);


    /* check the validity of the drive */

    if ((m -> doss_drives [drive].flags & DOSS_VALID_DRIVE) == 0)
	{
	   /* book doesn't mention this one. */
	   syscall_errno = 15;  /* drive invalid */
	   return SYSCALL_FAILURE;
	}

    retval = (*m->doss_drives[drive].fs_ops->create_dir)
      (m,m->doss_drives+drive,newpath);

    if (retval== FSOP_SUCCESS)
	{
	   return SYSCALL_SUCCESS;
	}
    else
	{
	   /* syscall errno is set already in the lower routines. */
	   return SYSCALL_FAILURE;
	}
 }
    


/* 
 *  doss_fsop_rmdir(m,path)
 *    PC_ENV *m;
 *    char   *path;
 *
 *  remove directory given by path
 *
 */

int DEFUN(doss_fsop_rmdir,(m,path), PC_ENV *m AND char   *path)
 {
    
    int retval;
    int drive;
    char newpath[DOSS_MAX_PATH * 2];

    /* convert to canonical form */
    doss_cvt_path(m, &drive, path, newpath);


    /* check the validity of the drive */

    if ((m -> doss_drives [drive].flags & DOSS_VALID_DRIVE) == 0)
	{
	   /* book doesn't mention this one. */
	   syscall_errno = 15;  /* drive invalid */
	   return SYSCALL_FAILURE;
	}

    
    if (strcmp(m->doss_drives[drive].curr_path, newpath) == 0)
	{
	   syscall_errno = 6;
	   return SYSCALL_FAILURE;
	}

    /* attempt to delete the directory. */
    retval = (*m->doss_drives[drive].fs_ops->delete_dir)
      (m,m->doss_drives+drive,newpath);

    if (retval== FSOP_SUCCESS)
	{
	   return SYSCALL_SUCCESS;
	}
    else
	{
	   /* syscall errno is set already in the lower routines. */
	   return SYSCALL_FAILURE;
	}
 }



/*************************************************************
the following two routines are necessary to map the considerably
different semantics of msdos dir operations into those available
to unix.  In particular, there are 21 bytes available in the 
dta [0..20] which are reserved by dos on each find{first,next}
which hold state information for the next invocation.  This is acceptable
in a pure block oriented search (store the drive, and the starting
fat index of the directory, as well as the last name matched).  
However, in a mapping to unix, the "fat index of directory" turns
out to be problematic.  Since a program may have several dta's
which are alternated between (one per dir, etc), we cannot 
simply store the name of the last directory accessed, we have to
store a (something like a) mapping between dta_addr -> directory name,
for every active dta.  So each process gets a list
of "active dtas" which are searched to determine the directory
which corresponds to the current directory search.  
*************************************************************/

static 
  int DEFUN(doss_dtalst_srch,
	  (m,phdta,c_dtaseg,c_dtaoff),
	  PC_ENV *m AND struct doss_dtalst **phdta  AND
	  u_int16 c_dtaseg AND u_int16 c_dtaoff)
 {
    /* returns true and assigns info to phdta.
       else returns false and assigns NULL to phdta.
     */

    struct doss_dtalst *head;

    head = m -> doss_curr_proc -> dtalst;
    while (head != NULL)
	{
	   
	   if (head -> dta_seg  == c_dtaseg && head->dta_off == c_dtaoff)
	       {
		  *phdta  = head;
		  return 1;
	       }
	   head = head -> next ;
	}
    *phdta = NULL;
    return 0;
 }
    

static void 
  DEFUN(doss_dtalst_insert,(m,hdta), PC_ENV *m AND struct doss_dtalst *hdta)

 {
    struct doss_dtalst *head;

    head = m -> doss_curr_proc -> dtalst;
    hdta -> next = head;
    m -> doss_curr_proc ->dtalst = hdta;
 }


void DEFUN(doss_load_fatr_dta,(m,dta,nbytes),
    PC_ENV *m AND u_int8    *dta AND int nbytes )
 {
    u_int16 dta_seg,dta_off;
    u_int8 *dta_base;
    int i;
    
    dta_seg = m->doss_curr_proc->dta_seg;
    dta_off = m->doss_curr_proc->dta_off;
    dta_base = (u_int8*)m->mem_base + (dta_seg<<4) + dta_off;
    for(i=0; i<nbytes; i++)
      dta[i] = dta_base[i];
 }

    


/* 
 *  doss_fsop_dirread(m,pattern,attrib,first)
 *    PC_ENV *m;
 *    char   *pattern;
 *    int    attrib;
 *    int    first;
 *    
 *  search directory given by pattern (wild cards OK.)
 *  and write the first match, formatted to the DTA.
 *
 */

int DEFUN(doss_fsop_dirread,(m,pattern,attrib,first),
    PC_ENV *m AND char   *pattern AND int attrib AND int first)
 {
    
    int retval;
    int found = 0;

    /* holds state from call via "findfirst" to call via findnext */
    int drive;
    struct doss_file_handle dirp;
    char newpath[DOSS_MAX_PATH * 2];
    char filename[DOSS_MAX_PATH*2];
/*    char directory[DOSS_MAX_PATH*2];  unused? */
    struct doss_file_handle tfile;
    char tdirectory[DOSS_MAX_PATH*2];
    char tfilename[DOSS_MAX_PATH*2];
    u_int16 dtaseg,dtaoff;
    struct doss_dtalst *dtamtch;
    u_int8 dta[64];
    u_int8 *dtabase;

    /* get the dta segment. */
    dtaseg = m->doss_curr_proc->dta_seg;
    dtaoff = m->doss_curr_proc->dta_off;

    if (first)  /* findfirst */
	{
	   doss_cvt_path(m, &drive, pattern, newpath);
	   /* check the validity of the drive */
	   if ((m -> doss_drives [drive].flags & DOSS_VALID_DRIVE) == 0)
	       {
		  /* book doesn't mention this one. */
		  syscall_errno = 15;  /* drive invalid */
		  return SYSCALL_FAILURE;
	       }
	   /* break out the filename from the pathname */
	   doss_split_path(newpath, dirp.path, filename);
	   if(doss_dtalst_srch(m,&dtamtch, dtaseg, dtaoff))
	       {	
		  dtamtch->drive = drive;
		  dtamtch->attrib = attrib;
		  strcpy(dtamtch->dname,dirp.path);
		  strcpy(dtamtch->pattern,filename);
	       }
	   else
	       {
		  /* freed when process exits */
		  dtamtch=(struct doss_dtalst *)
		    malloc(sizeof(struct doss_dtalst));
		  if (dtamtch == NULL) 
		      {
			sys_fatal(errno,"dosread: malloc\n");
		      }
		  dtamtch->dta_seg = dtaseg;
		  dtamtch->dta_off = dtaoff;
		  doss_dtalst_insert(m,dtamtch);
		  dtamtch->attrib = attrib;
		  dtamtch->drive = drive;
		  strcpy(dtamtch->dname,dirp.path);
		  strcpy(dtamtch->pattern,filename);
	       }

	   /* now do an opendir. */
	   retval = (*m->doss_drives[drive].fs_ops->open_dir)
	     (m,m->doss_drives+drive, &dirp); 
	   if (retval != FSOP_SUCCESS) 
	       {
		  syscall_errno = 2;
		  return SYSCALL_FAILURE;
	       }
	}
    else  /* findnext */
	{
	   if(doss_dtalst_srch(m,&dtamtch, dtaseg, dtaoff))
	       {	
		  drive = dtamtch->drive;
		  attrib = dtamtch->attrib;
		  strcpy(dirp.path,dtamtch->dname);
		  strcpy(filename,dtamtch->pattern);
	       }
	   else 
	       {
		  syscall_errno = 2;
		  return SYSCALL_FAILURE;
	       }

	   /* now do an opendir. */
	   retval = (*m->doss_drives[drive].fs_ops->open_dir)
	     (m,m->doss_drives+drive, &dirp); 
	   if (retval != FSOP_SUCCESS) 
	       {
		  syscall_errno = 2;
		  return SYSCALL_FAILURE;
	       }

	   /* now load the previous dta's contents. */
	   doss_load_fatr_dta(m,dta,44);

	   /* now do an seekdir. */
	   retval = (*m->doss_drives[drive].fs_ops->seek_dir)
	     (m,m->doss_drives+drive, &dirp, dta); 

	   if (retval != FSOP_SUCCESS) 
	       {
		  syscall_errno = 2;
		  return SYSCALL_FAILURE;
	       }	
	}	   
    
    do
	{	
	   /* do an readdir on the directory portion of the name. */
	   retval = (*m->doss_drives[drive].fs_ops->read_dir)
	     (m,m->doss_drives+drive, &dirp, &tfile);
   
	   if (retval == FSOP_FAILURE) break;  /* eodir */
	   
	   doss_split_path(tfile.path,tdirectory, tfilename);

	   if ( strcmp(tfilename,"        .   ") == 0)
	     continue;

/*	   printf("DIR:'%s'\n",tfile.path);*/
	   
	   /* lets stat the file, and see what it is.  */
	   retval = (*m->doss_drives[drive].fs_ops->get_attr)
	     (m,m->doss_drives+drive, &tfile);

	   if (retval == FSOP_FAILURE) break; /* unknown problem. */

	   /* we have a contender, and most of the information that we 
	      need on it.  Now, first, lets see if it matches
	      the original pattern specification. */
	   
	   /* note that filename===file name part of the original 
	      search pattern, and tfilename is our contender for 
	      a match.  */
	   if (doss_match_filename(m,filename,tfilename))
	       {
		  if (doss_match_attrib(attrib,tfile.attrib))
		      {
			 doss_copy_fatr_dta(m,&tfile);
			 found = 1;
			 break;
		      }
	       }
	}
    while(1);

    /* save current location... */
    dtabase = (u_int8*)m->mem_base + (dtaseg<<4) + dtaoff;
    retval = (*m->doss_drives[drive].fs_ops->tell_dir)
      (m,m->doss_drives+drive, &dirp, dtabase);

    
    /* now do an closedir. */
    retval = (*m->doss_drives[drive].fs_ops->close_dir)
      (m,m->doss_drives+drive, &dirp); 
    if (retval != FSOP_SUCCESS) 
	{
	   syscall_errno = 2;
	   return SYSCALL_FAILURE;
	}

    if (! found) 
	{
	   syscall_errno = 0x12;
	   return SYSCALL_FAILURE;
	}
    else
	{
	   return SYSCALL_SUCCESS;
	}
 }
 


/* note that I have not experimented with this, and am currently
   under the impression that the effect of the attrib mask to the 
   findfirst/findnext system calls is an exclusive-OR of the effect
   of each of the bits.  Very bizarre...

*/

int DEFUN(doss_match_attrib,(pattr,fattr), int pattr AND int fattr)
 {

    int samebits = pattr & fattr & (DOSS_FS_ATTR_DIR
				    /*|DOSS_FS_ATTR_ARC*/
				    |DOSS_FS_ATTR_SYS
				    |DOSS_FS_ATTR_HID
				    /*|DOSS_FS_ATTR_RO */
				    /*|DOSS_FS_ATTR_VOL*/);

/*    printf("pattr=%x fattr=%x  samebits=%x\n",
	   pattr,fattr,samebits); */

    /* weird rule:  
       if looking for volume labels, 
       then only match volume labels.
       check first if we have a volume label,
       so that it does not pass this test.  
       Must be simpler way to do this.*/
	
    if (pattr & DOSS_FS_ATTR_VOL)
	{
	   if (fattr & DOSS_FS_ATTR_VOL) 	   
	     return 1;
	   else 
	     return 0;
	}
    else if (fattr & DOSS_FS_ATTR_VOL) 	   
      return 0;
    
    
    
	   
    /* notreached on volume labels */

    /* regular files apparently always match. */
    if (fattr == 0 ||
	fattr & (DOSS_FS_ATTR_RO|DOSS_FS_ATTR_ARC))
      return 1;


      /* if any bits are set in the pattern attribute and in 
	 the file's attributes, *and* it is one of the bits
	 specified by the mask, then we have a match, and return 
	 true. */
    else if (samebits)   
      return 1; 
    else
      return 0;

 }



void DEFUN(doss_copy_fatr_dta,(m,file),
	   PC_ENV *m AND struct doss_file_handle *file)
 {
    u_int16 dta_seg,dta_off;
    u_int16 low, high;
    u_int8 *dta_base;
    char filename[13];
    char directory[DOSS_MAX_PATH*2];
    int i,j;
    
    
    dta_seg = m->doss_curr_proc->dta_seg;
    dta_off = m->doss_curr_proc->dta_off;
    
    dta_base = (u_int8*)m->mem_base + (dta_seg<<4) + dta_off;
    
    /* break out the filename */
    doss_split_path(file->path,directory, filename);

    
    STORE_BYTE(dta_base, 21, file->attrib);
    STORE_WORD(dta_base, 22, file->mtime);
    STORE_WORD(dta_base, 24, file->mdate);
    low = file->size & 0xffff;
    high = (file->size>>16) & 0xffff;
    STORE_WORD(dta_base, 26, low);
    STORE_WORD(dta_base, 28, high);
    STORE_BYTE(dta_base, 42, 0);

    j=7;
    for (i=7; i>=0; i--)
      if (filename[i] != ' ')
	{
	   j=i;
	   break;
	}
    for (i=0; i<= j; i++)
	{
	   STORE_BYTE(dta_base, 30+i, (u_int8)( filename[i]));
	}

    j += 1;
    
    STORE_BYTE(dta_base, 30+j, '.');
    
    for (i=1; i<4; i++)
	{
	   if (filename[8+i] == ' ')
	     break;
	   STORE_BYTE(dta_base, 30+j+i, (u_int8)( filename[8+i]));
	}

    STORE_BYTE(dta_base, 30+i+j, 0);

#ifdef DEBUG
    if (DEBUG_FS(m))
      {
	 debug_printf(m,"DTA dump %x:%x to %x:%x\n",
		 dta_seg,
		 dta_off,
		 dta_seg,
		 dta_off+64);
	 dump_memory(m, dta_seg, dta_off, 64);
	 debug_printf(m,"-----------------------\n");
      }
#endif    
   
 }

    


/* split full filename into its path components and the final
   filename.  Canonicalize the filename:
     'foo.bar'  ->  'foo     .bar'  with embedded spaces.
*/


void DEFUN(doss_split_path,
	   (oldpath, directory, filename),
	   char *oldpath AND char *directory 
	   AND char *filename)
 {
    
    char *p = oldpath;
    char *le = oldpath;   /* this should be a '\\' */
    char *q = directory;
    char *ne = directory;
    int count;

/*    printf("DSP:'%s'\n",oldpath);*/
    /* process entire string "oldpath" */
    while (*p)
	{
	   switch (*p) 
	       {
		case '/':
		  *p = '\\';
		  /* fall through */
		case '\\':
		  /* save this position in both strings.  In the copied-to
		     string, this element will be replaced with a null
		     to terminate the string and separate the "directory"
		     from the "filename".  In the source string, it is a 
		     placeholder for the beginning of the filename part. */
		  le = p;
		  ne = q;
		  break;
		default:
		  break;
	       }
	   *q = *p;
	   p++;
	   q++;
	}

    *ne = '\0';   /* wipe out the last '\\' */
    /* done with directory. */
    p = le + 1;  /* point to beginning of filename. */
    q = filename;
    count = 0;
    while (*p)
	{
	   switch (*p)
	       {
		case '.':
		  while (count < 8) 
		      {
			 *q = ' ';
			 q++;
			 count++;
		      }
		  *q = *p;  /* copy the dot */
		  q++;
		  p++;
		  count = 0;
		  break;
		default:
		  *q = *p;
		  q++;
		  p++;
		  count++;
		  break;
	       }

	}
    while (count < 3)   /* pad the extension part, as well. */
	{
	   *q = ' ';
	   q++;
	   count++;
	}
    *q = '\0';
 }



void DEFUN(doss_split_filename,(name,base,ext),
	   char *name AND char *base AND char *ext)
 {
    char *p,*q;
    
    p = name;
    q = base;
    
    while(*p)
	{
	   switch(*p)
	       {
		case '.':
		  *q = '\0';
		  p += 1;
		  q = ext;
		  break;
		default:
		  *q = *p;
		  p += 1;
		  q += 1;
		  break;
	       }
	}
    *q = '\0';
 }
    
       
		      
		
    
    

int DEFUN(doss_match_filename,(m,pattern,filename),
    PC_ENV *m AND char *pattern AND char *filename)
 {
    char base[13],ext[13];
    char pbase[13],pext[13];

    strcpy(base,"            ");
    strcpy(pbase,"            ");
    strcpy(ext,"   ");
    strcpy(pext,"   ");
    
    doss_split_filename(pattern, pbase, pext);
    doss_split_filename(filename, base,  ext);
    
#ifdef DEBUG
    if (strlen(pattern)  != 12 &&
	strlen(pbase)    != 8  &&
	strlen(pext)     != 3  &&
	strlen(filename) != 12 &&
	strlen(base)     != 8  &&
	strlen(ext)      != 3)
	{
	   
	   debug_printf(m,"Error in doss_match_filename\n");
	   debug_printf(m,
		   "pattern='%s'  pbase='%s' pext='%s'\n",
		   pattern,pbase,pext);
	   debug_printf(m,
		   "filename='%s'  base='%s' ext='%s'\n",
		   filename,base,ext);
	}
#endif

    /* look for wildcards, and handle the no-wildcard case 
       with a stringcompare.  */

/*    printf("pattern=%s \t pbase=%s pext=%s \t base=%s ext=%s\n",
           pattern,pbase,pext,base,ext);  */
	
    if (index(pattern,'*') == NULL && index(pattern,'?') == NULL)
	{

	   if (strcmp(pbase,base) == 0 &&
	       strcmp(pext,ext) == 0)
	     return 1;
	   else
	     return 0;
	}
    
    /* we have at least one wildcard in the pattern.  */

    if (index(pattern,'*') != NULL)
	{
	   doss_rewrite_regex(pattern, pbase, pext); 
	   /* and overwrite pext ..*/
	}

/*    printf("pattern=%s \t pbase=%s pext=%s \t base=%s ext=%s\n",
	   pattern,pbase,pext,base,ext); */
    
    /* there are no '*'s in either of the patterns now.  
       note, that this could be fixed.  But this is the way msdos
       apparently handles wildcards.  When it encounters a '*' anywhere,
       it converts to a sequence of '?'s.  So that:

       foo*bar.foo ->  foo?????.foo
       foo*.bar    ->  foo?????.bar
       foo*.*      ->  foo?????.???

       very bizarre handling of wildcards, and quite easy to implement.
     */

    if (doss_match_regex(pbase,base) && doss_match_regex(pext,ext))
      return 1;
    else
      return 0;
    
    
 }


/* see comment above about the nature of these regexp's.  They are
   quite limited.  */

int DEFUN(doss_match_regex,(pattern,name),
	  char *pattern AND char *name)
 {
    char *p = pattern;
    char *q = name;
    
    while (*p)
	{
	   if (*p == '?' || *p == *q ) 

	     ;            /* do nada.  match the character and advance. */
	     
	   else if ( *p != *q )
	       {
		  return 0;
	       }
	   
	   p += 1;
	   q += 1;
	}
    return 1;
 }


     
void DEFUN(doss_rewrite_regex,(pattern, base, ext),
	   char *pattern AND char *base AND char *ext)
 {

    char *p,*q;
    int count,max;
    /* pattern will be 12 characters in length (in canonical form)
       and the two subpatterns will be 8 and 3 characters in length
       respectively. */
    count = 0;
    max = 8;
    p = pattern;
    q = base;
    while(*p)
	{
	   switch(*p)
	       {
		case '?':
		  /* no processing needed.  Copy over. */
		  *q = *p;
		  q++;
		  p++;
		  break;

		case '.':
		  *q = '\0';
		  max = 3;
		  count = 0;
		  p += 1;
		  q = ext;
		  break;

		case '*':
		  for(;count < max; count++)
		    {
		       *q = '?';
		       q++;
		       p++;
		    }
		  break;

		default:
		  count++;
		  *q = *p;
		  p += 1;
		  q += 1;
		  break;
	       }
	}
    *q = '\0';
 }
    



    

#endif
