
/* 
 * Dos/PC Emulator
 * Copyright (C) 1991 Jim Hudgens
 * 
 * 
 * The file is part of GDE.
 * 
 * GDE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 * 
 * GDE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GDE; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */

#ifdef BIOS_SUPPORT

static char rcsid[]=
  "$Id: bios_time.c,v 1.3 1992/04/13 00:32:22 hudgens Exp hudgens $";

/* $Log: bios_time.c,v $
 * Revision 1.3  1992/04/13  00:32:22  hudgens
 * Added prototypes based on work by GCHUNT, and cleaned up a little.
 *
 * Revision 1.2  1991/07/30  01:47:27  hudgens
 * added copyright.
 *
 * Revision 1.1  1991/07/20  04:04:48  hudgens
 * Initial revision
 *
 *
 */


/* interrupt 0x1a functions for bios emulation  */

#include "gde.h"
#include <math.h>
#include <sys/types.h>
#include <sys/time.h>



struct bios_timer
 {
    struct timeval  boot_time;  /* may be modified by set_time. */
 };

static struct bios_timer  sys_time;


/* helper functions... all static. */
/* XXX ---   get exact or at least better value here */

#define TICKSPERSEC   18.2
#define TICKSPERDAY   (TICKSPERSEC * 60  * 60  * 24)

/* convert timeval to ticks (seconds*TICKSPERSEC) */
static void DEFUN(tv_to_biostick,
	     (t,bt), 
	     struct timeval *t AND int *bt)
 {
    double rtime;

    rtime = t->tv_usec / 1000000.0;
    rtime += t->tv_sec;
    rtime *= TICKSPERSEC;  
    *bt = floor(rtime);
 }
    	
/* convert from bios-ticks (TICKSPERSEC) to timeval */
static void DEFUN(biostick_to_tv,(t,bt),struct timeval *t AND  int    *bt)
 {
    double rtime;
    
    rtime = *bt;
    rtime /= TICKSPERSEC;  /* now in format of real seconds. */
    t -> tv_sec = floor(rtime);
    rtime -= floor(rtime);
    rtime *= 1000000.0;
    t -> tv_usec = floor(rtime);
 }
    
/* convert timeval to seconds */
static void DEFUN(tv_to_seconds,(t,s),struct timeval *t AND double *s)
 {
    *s = t->tv_usec / 1000000.0;
    *s += t->tv_sec;
 }
    	
	
/* convert from real seconds  to timeval */
static void DEFUN(seconds_to_tv,(t,s),struct timeval *t AND double  *s)
 {
    double rtime;
    
    rtime = *s;
    t -> tv_sec = floor(rtime);
    rtime -= floor(rtime);
    rtime *= 1000000.0;
    t -> tv_usec = floor(rtime);
 }
    

static void DEFUN(add_timeval,(tr,t1,t2), 
	     struct timeval *t1 AND struct timeval *t2 AND struct timeval *tr)
 {
    /* tr = t1 + t2 */
    double sr,s1,s2;

    /* OOO  could be done inline, using integer arithmetic */

    tv_to_seconds(t1,&s1);
    tv_to_seconds(t2,&s2);
    sr = s1 + s2;
    seconds_to_tv(tr, &sr);
 }
    

static void DEFUN(diff_timeval,(tr, t1,t2),
	     struct timeval *t1 AND
	     struct timeval *t2 AND
	     struct timeval *tr)
 {
    /* tr = t1 - t2 */
    double sr,s1,s2;

    /* OOO  could be done inline, using integer arithmetic */
    tv_to_seconds(t1,&s1);
    tv_to_seconds(t2,&s2);
    sr = s1 - s2;
    seconds_to_tv(tr, &sr);
 }
    
    
/* this routine should be called at "boot time", and initialize
   the value of the global sys_time struct.  
 */

   
void DEFUN_VOID(init_bios_timer)
 {
    gettimeofday(&sys_time.boot_time, (struct timezone *) 0);
 }    
    
    
/* bios int 0x1a function 00 */
/*
  passed 3 integer pointers, fills them with the 
  most and least significant portions of the timer.
  if 24 hours has passed, then it also returns an
  indication in *ov.
  
  This subroutine is ignorant of the register passing conventions
  used for the interrupt.  A level closer to the PC emulator
  places these values in the appropriate registers.

*/


int DEFUN(intr_0x1a_getclock,(h,l,ov),
	  u_int16 *h AND u_int16 *l AND u_int8  *ov)
 {
    /* we do the following:
          get the current time into ctime.
	  subtract from boot_time.
	  convert resulting time difference into a scaled 
	   "ticks since we booted".

     */
    struct timeval ctime;
    struct timeval deltatime;
    int biosticks;
    
    gettimeofday(&ctime, (struct timezone *) 0);
    diff_timeval(&deltatime, &ctime, &sys_time.boot_time);
    tv_to_biostick(&deltatime, &biosticks);
    
    /* note that this reprents the number of ticks since we booted.
       The logic should change if this ever emulates an AT bios,
       since that returns (apparently) ticks since midnight. 
     */

    *l = biosticks&0xffff;
    *h = (biosticks>>16) & 0xffff;

    if (biosticks > TICKSPERDAY)
      *ov=1;
    else
      *ov=0;
    
    return 0;
 }
    



/* bios int 0x1a function 01 */
int DEFUN(intr_0x1a_setclock,(hi,lo),
    u_int16 hi AND u_int16 lo)
 {
    /* we do the following:
          get the current time into ctime.

	  subtract from boot_time.
	  
	  convert resulting time difference into a scaled 
	   "ticks since we booted".

     */


    struct timeval ctime;
    double  f_ctime,f_boottime,f_bt;
    int biosticks;
    
    gettimeofday(&ctime, (struct timezone *) 0);
    tv_to_seconds(&ctime,&f_ctime);
    tv_to_seconds(&sys_time.boot_time, &f_boottime);

    biosticks = lo & 0xffff;
    biosticks |= (hi << 16);
    f_bt = biosticks / TICKSPERSEC;   /* real seconds since boot */

    f_boottime = f_ctime - f_bt;
    seconds_to_tv(&sys_time.boot_time,&f_boottime);
 
    return 0;
 }
    

#ifdef TEST
#include "test/t_bios_time.c"
#endif


#endif
