/*
SDSDL: GEM video driver for SDL
Copyright 2012 John Elliott <jce@seasip.demon.co.uk>
Based on SDPSC9.VGA copyright 1987, 1999 Caldera Thin Clients, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/* Deals with opening and closing workstations */

#include "sdsdl_i.h"

#ifdef HAVE_WINDOWS_H
#include <windows.h>
#endif

static DRIVERDATA *st_root;
static unsigned st_planes = 15;
static SDL_Surface *st_surface = NULL;
static SDL_Rect     st_window = { 0, 0, 640, 480 };

void sd_setsurface(PSURFACE s, PRECT window)
{
	st_surface = (SDL_Surface *)s;
	st_window  = *((SDL_Rect *)window);
}

void sd_setwidth(unsigned w)
{
	st_window.w = w;
}

void sd_setheight(unsigned h)
{
	st_window.h = h;
}

void sd_setplanes(unsigned p)
{
	st_planes = p;
}

static void def_buttons(unsigned b)
{
}


static void def_motion(signed x, signed y)
{
}


static void def_cursor(signed x, signed y)
{
}


static void def_timer(void)
{
}


static int timer_func(void *p)
{
	DRIVERCOMMON *drvr = (DRIVERCOMMON *)p;

	while (drvr->timer_running)
	{
		SDL_Delay(TIMER_MSEC);
		if (drvr->timvec) 
		{
			(*drvr->timvec)();
		}
	}
	return 0;
}




/* Register a new device instance */
unsigned register_data(DRIVERDATA *p)
{
	unsigned h = 1;
	DRIVERDATA *q;

	/* Come up with a driver handle that doesn't match any 
	 * existing ones. */
	for (q = st_root; q != NULL; q = q->next)
	{
		if (h <= q->handle) h = 1 + q->handle;
	}	
	p->handle = h;
	p->next = st_root;
	st_root = p;
	return 0;
}


/* Unregister a device instance. */
void     unregister_data(DRIVERDATA *p)
{
	DRIVERDATA *q;

	for (q = st_root; q != NULL; q = q->next)
	{
		if (q->next == p) q->next = p->next;
	}
	if (st_root == p) st_root = p->next;
	p->next = NULL;
}


DRIVERDATA *lookup_handle(unsigned h)
{
	DRIVERDATA *q;

	for (q = st_root; q != NULL; q = q->next)
	{
		if (q->handle == h) return q;
	}
	return NULL;
}



unsigned sd_opnwk(unsigned work_in[11], unsigned *handle, unsigned work_out[57])
{
	DRIVERCOMMON *common;
	DRIVERDATA *self;
	Uint8 buttons;
	
	if (st_surface == NULL && SDL_Init(SDL_INIT_VIDEO) < 0)
	{
		*handle = 0;
		return 1;
	}
	common = malloc(sizeof(DRIVERCOMMON));
	if (!common)
	{
		*handle = 0;
		return 1;
	}
	self = malloc(sizeof(DRIVERDATA));
	if (!self)
	{
		*handle = 0;
		return 1;
	}
	memset(common, 0, sizeof(DRIVERCOMMON));
	memset(self,   0, sizeof(DRIVERDATA));

	self->xfm_mode = 1;
	self->common = common;
	common->timvec = def_timer;
	common->butvec = def_buttons;
	common->motvec = def_motion;
	common->curvec = def_cursor;
	common->btn_state  = 0;
	common->textmode = 0;

	buttons = SDL_GetMouseState(NULL, NULL);
	if (buttons & SDL_BUTTON(1)) common->btn_state |= 1;
	if (buttons & SDL_BUTTON(2)) common->btn_state |= 4;
	if (buttons & SDL_BUTTON(3)) common->btn_state |= 2;

	register_data(self);

	if (st_surface)
	{
		common->own_surface = 0;
		common->surface = st_surface;
		common->window = st_window;
	}
	else
	{
		common->own_surface = 1;
		common->window = st_window;
		common->window.x = 0;
		common->window.y = 0;
/* If we're creating our own window, try to use the native screen depth */
		common->surface = SDL_SetVideoMode(st_window.w, st_window.h, 
						0, SDL_SWSURFACE);
		if (!common->surface)
		{
			unregister_data(self);
			*handle = 0;
			return 1;
		}
		common->window.w = common->surface->w;
		common->window.h = common->surface->h;
	}
	if (common->surface->format->BytesPerPixel < 1 || 
	    common->surface->format->BytesPerPixel > 4)
	{
		fprintf(stderr, "Unsupported pixel format, %d bytes/pixel.\n",
			common->surface->format->BytesPerPixel);
		SDL_FreeSurface(common->surface);
		common->surface = NULL;
		*handle = 0;
		return 1;
	}
	common->backbuf = SDL_AllocSurface(SDL_HWSURFACE, 
		common->window.w,
		common->window.h,
		common->surface->format->BitsPerPixel,
		common->surface->format->Rmask,
		common->surface->format->Gmask,
		common->surface->format->Bmask,
		common->surface->format->Amask);

// User-defined pattern at 15bpp depth
	self->udpt_15bpp = SDL_AllocSurface(SDL_SWSURFACE, 
		16, 16, 
		15, 0x7C00, 0x3E0, 0x1F, 0);
	self->udpt_native = SDL_AllocSurface(SDL_SWSURFACE, 
		16, 16, 
		common->surface->format->BitsPerPixel,
		common->surface->format->Rmask,
		common->surface->format->Gmask,
		common->surface->format->Bmask,
		common->surface->format->Amask);

	*handle = self->handle;

	/* Load default colour map */
	sd_spalette(self->handle, 0);

	common->HIDE_CNT = 1;
	memcpy(self->ud_patrn, def_ud_patrn, 16 * sizeof(Uint16));
	memcpy(self->line_styl, def_line_sty, 6 * sizeof(Uint16));
	memcpy(self->dev_tab, DEV_TAB, sizeof(self->dev_tab));
	memcpy(self->inq_tab, INQ_TAB, sizeof(self->inq_tab));
	memcpy(self->siz_tab, SIZ_TAB, sizeof(self->siz_tab));
	self->faux_depth = st_planes;
	self->inq_tab[4] = st_planes;
	if (st_planes < 15)
	{
		self->dev_tab[13] = (1 << st_planes);
	}
	if (self->common->surface->format->BytesPerPixel == 1)
	{
		if (self->inq_tab[4] > 8) self->inq_tab[4] = 8;
		self->dev_tab[14] = 256;
		self->inq_tab[5] = 1;
	}
	self->udpt_np = 1;
	self->xfm_mode = work_in[10];
	common->timer_running = 1;
	common->timer_thread = SDL_CreateThread(timer_func, common);
	init_sound(common);

	text_init(self, work_in[6]);
	init_wk(self, work_in, work_out);
	sd_clrwk(self->handle);
	return 0;
}

unsigned sd_opnvwk(unsigned work_in[11], unsigned *handle, 
			unsigned work_out[57])
{
	DRIVERDATA *parent = lookup_handle(*handle);
	DRIVERDATA *self;
	DRIVERCOMMON *common;

	if (!parent)
	{
		*handle = 0;
		return 1;
	}
	common = parent->common;
	
	self = malloc(sizeof(DRIVERDATA));
	if (!self)
	{
		*handle = 0;
		return 1;
	}
	memcpy(self,   parent, sizeof(DRIVERDATA));

	self->common = common;
	self->handle = 0;
	self->next = NULL;
	register_data(self);

	*handle = self->handle;
// User-defined pattern at 15bpp depth
	self->udpt_15bpp = SDL_AllocSurface(SDL_SWSURFACE, 
		16, 16, 
		15, 0x7C00, 0x3E0, 0x1F, 0);

// User-defined pattern at native video depth
	self->udpt_native = SDL_AllocSurface(SDL_SWSURFACE, 
		16, 16, 
		common->surface->format->BitsPerPixel,
		common->surface->format->Rmask,
		common->surface->format->Gmask,
		common->surface->format->Bmask,
		common->surface->format->Amask);
	*handle = self->handle;

	common->old_cursor = SDL_GetCursor();

	memcpy(self->ud_patrn, def_ud_patrn, 16 * sizeof(Uint16));
	memcpy(self->line_styl, def_line_sty, 6 * sizeof(Uint16));
	self->udpt_np = 1;
	self->xfm_mode = work_in[10];
	self->faux_depth = st_planes;
	self->inq_tab[4] = st_planes;
	if (st_planes < 15)
	{
		self->dev_tab[13] = (1 << st_planes);
	}
	text_init(self, work_in[6]);
	init_wk(self, work_in, work_out);
	return 0;
}



void sd_clswk(unsigned handle)
{
	DRIVERDATA *self = lookup_handle(handle);

	if (!self) return;

	if (SDL_GetCursor() == self->common->cursor)
	{
		SDL_SetCursor(self->common->old_cursor);
		SDL_FreeCursor(self->common->cursor);
		self->common->cursor = NULL;
	}
	term_sound(self->common);
	self->common->timer_running = 0;
	SDL_WaitThread(self->common->timer_thread, NULL);

	SDL_FreeSurface(self->udpt_native);
	SDL_FreeSurface(self->udpt_15bpp);
	SDL_FreeSurface(self->common->backbuf);
	if (self->common->own_surface)
	{
		SDL_FreeSurface(self->common->surface);
	}
	self->common->surface = NULL;
	free(self->common);
	self->common = NULL;
	unregister_data(self);
	if (st_surface == NULL)
	{
		SDL_Quit();
	}
}

void sd_clsvwk(unsigned handle)
{
	DRIVERDATA *self = lookup_handle(handle);
	if (!self) return;
	SDL_FreeSurface(self->udpt_native);
	SDL_FreeSurface(self->udpt_15bpp);
	unregister_data(self);
}


void sd_delay(unsigned handle, unsigned usec)
{
#ifdef HAVE_WINDOWS_H
	Sleep (usec / 1000);
#else 
	struct timeval tv;

	tv.tv_sec = 0;
	tv.tv_usec = usec;
	select(0, NULL, NULL, NULL, &tv);
#endif
}


unsigned sd_extimv(unsigned handle, TIMVEC vec, TIMVEC *old)
{
	DRIVERDATA *self = lookup_handle(handle);

	if (self)
	{
		if (old) *old = self->common->timvec;
		self->common->timvec = vec;
		return TIMER_MSEC;
	}
	return 0;
}


