/*
SDSDL: GEM video driver for SDL
Copyright 2012 John Elliott <jce@seasip.demon.co.uk>
Based on SDPSC9.VGA copyright 1987, 1999 Caldera Thin Clients, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/* Raster operations */

#include "sdsdl_i.h"

void sd_trnfm(unsigned handle, PMFDB mfdb1, PMFDB mfdb2)
{
	unsigned plane, row, word;
	unsigned char *src, *dst;

	/* For simplicity, don't support trnfm direct to the screen */
	if (mfdb1->mp == NULL || mfdb1->mp == (void *)(-1)) return;
	if (mfdb2->mp == NULL || mfdb2->mp == (void *)(-1)) return;

	src = mfdb1->mp;
	dst = mfdb2->mp;
	for (plane = 0; plane < mfdb1->np; plane++)
	{
		for (row = 0; row < mfdb1->fh; row++)
		{
			for (word = 0; word < mfdb1->fww; word++)
			{
				unsigned char s0 = src[0];	
				unsigned char s1 = src[1];
				dst[0] = s1;
				dst[1] = s0;
				src += 2;
				dst += 2;	
			}
		}
	}
}

static void surface_from_mfdb1x32(MFDB *mfdb, SDL_Surface *s, Uint32 fg, Uint32 bg)
{
	int x, y;
	unsigned xmask;

	/* Start by blanking the surface */

	memset(s->pixels, 0, s->h * s->pitch);
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint32 *dest = (Uint32 *)(((Uint8 *)s->pixels) + y * s->pitch);
		Uint8  *src  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			if (src[0] & xmask) 
			{
				dest[x] = fg; 
			}
			else 
			{
				dest[x] = bg;
			}
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++src; }
		}
	}
}


static void surface_from_mfdb1x24(MFDB *mfdb, SDL_Surface *s, Uint32 fg, Uint32 bg)
{
	int x, y;
	unsigned xmask;

	/* Start by blanking the surface */

	memset(s->pixels, 0, s->h * s->pitch);
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint8 *dest =  (Uint8 *)(((Uint8 *)s->pixels) + y * s->pitch);
		Uint8  *src  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			if (src[0] & xmask) 
			{
				dest[x * 3    ] = fg & 0xFF; 
				dest[x * 3 + 1] = (fg >> 8) & 0xFF; 
				dest[x * 3 + 2] = (fg >> 16) & 0xFF; 
			}
			else 
			{
				dest[x * 3    ] = bg & 0xFF; 
				dest[x * 3 + 1] = (bg >> 8) & 0xFF; 
				dest[x * 3 + 2] = (bg >> 16) & 0xFF; 
			}
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++src; }
		}
	}
	fflush(stdout);
}




static void surface_from_mfdb1x16(MFDB *mfdb, SDL_Surface *s, Uint32 fg, Uint32 bg)
{
	int x, y;
	unsigned xmask;

	/* Start by blanking the surface */

	memset(s->pixels, 0, s->h * s->pitch);
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint16 *dest = (Uint16 *)(((Uint8 *)s->pixels) + y * s->pitch);
		Uint8  *src  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			if (src[0] & xmask) 
			{
				dest[x] = fg; 
			}
			else 
			{
				dest[x] = bg;
			}
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++src; }
		}
	}
}


/* As above, for 8bpp palettised surface */
static void surface_from_mfdb1x8(MFDB *mfdb, SDL_Surface *s, Uint8 fg, Uint8 bg)
{
	int x, y;
	unsigned xmask;

	/* Start by blanking the surface */

	memset(s->pixels, 0, s->h * s->pitch);
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint8 *dest = ((Uint8 *)s->pixels) + y * s->pitch;
		Uint8 *src  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			if (src[0] & xmask) dest[x] = fg; else dest[x] = bg;
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++src; }
		}
	}
}

/* Promote 1-plane surface to truecolour, with arbitrary fore- and background
 * colours */
static void surface_from_mfdb1(MFDB *mfdb, SDL_Surface *s, Uint32 fg, Uint32 bg)
{
	switch (s->format->BytesPerPixel)
	{
		case 1: surface_from_mfdb1x8(mfdb, s, fg, bg); break;
		case 2: surface_from_mfdb1x16(mfdb, s, fg, bg); break;
		case 3: surface_from_mfdb1x24(mfdb, s, fg, bg); break;
		case 4: surface_from_mfdb1x32(mfdb, s, fg, bg); break;
	}
}


static unsigned crdiff(SDL_Surface *s, Uint32 col1, Uint32 col2)
{
	Uint8 r1, g1, b1;
	Uint8 r2, g2, b2;
	unsigned dr, dg, db;

	SDL_GetRGB(col1, s->format, &r1, &g1, &b1);
	SDL_GetRGB(col2, s->format, &r2, &g2, &b2);
	db = (b1 < b2) ? b2 - b1 : b1 - b2;
	dg = (g1 < g2) ? g2 - g1 : g1 - g2;
	dr = (r1 < r2) ? r2 - r1 : r1 - r2;

	/* These are uint8s, so max diff is 768 */
	return dr + dg + db;
}

static void surface_to_mfdb8x8(DRIVERDATA *self, MFDB *mfdb, SDL_Surface *s)
{
	int x, y;
	unsigned xmask;
	Uint32 colours[256];
	Uint32 diff;
	int ink, n;
	unsigned maxcol;
	unsigned planesize;
	int nc;

	for (nc = 0; nc < 256; nc++)
	{
		map_colour_x(self, s, nc, NULL, &colours[nc]);
	}
	maxcol = self->dev_tab[13];
	planesize = mfdb->fww * 2 * mfdb->fh;
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint8 *src   = ((Uint8 *)s->pixels) + y * s->pitch;
		Uint8 *dest  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			ink = 0;
			diff = 0xFFFFFFFF;
			/* Find the best colour match */
			for (n = 0; n < maxcol; n++)
			{
				unsigned d = crdiff(s, src[0], colours[n]);
				if (diff > d) { ink = n; diff = d; }
				if (d == 0) break;	/* Perfect match */
			}
			for (n = 0; n < mfdb->np; n++)
			{
				if (ink & 1) dest[planesize * n] |= xmask;
				else	     dest[planesize * n] &= ~xmask;
				ink = ink >> 1;
			}
;
			++src;
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++dest; }
		}
	}
}




static void surface_to_mfdb8x16(DRIVERDATA *self, MFDB *mfdb, SDL_Surface *s)
{
	int x, y;
	unsigned xmask;
	Uint32 colours[256];
	Uint32 diff;
	int ink, n;
	unsigned maxcol;
	unsigned planesize;
	int nc;

	for (nc = 0; nc < 256; nc++)
	{
		map_colour_x(self, s, nc, NULL, &colours[nc]);
	}
	maxcol = self->dev_tab[13];
	planesize = mfdb->fww * 2 * mfdb->fh;
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint16 *src   = (Uint16 *)(((Uint8 *)s->pixels) + y * s->pitch);
		Uint8  *dest  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			ink = 0;
			diff = 0xFFFFFFFF;
			/* Find the best colour match */
			for (n = 0; n < maxcol; n++)
			{
				unsigned d = crdiff(s, src[0], colours[n]);
				if (diff > d) { ink = n; diff = d; }
				if (d == 0) break;	/* Perfect match */
			}
			for (n = 0; n < mfdb->np; n++)
			{
				if (ink & 1) dest[planesize * n] |= xmask;
				else	     dest[planesize * n] &= ~xmask;
				ink = ink >> 1;
			}
			++src;
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++dest; }
		}
	}
}


static void surface_to_mfdb8x24(DRIVERDATA *self, MFDB *mfdb, SDL_Surface *s)
{
	int x, y;
	unsigned xmask;
	Uint32 colours[256];
	Uint32 diff;
	Uint32 srcpix;
	int ink, n;
	unsigned maxcol;
	unsigned planesize;
	int nc;

	for (nc = 0; nc < 256; nc++)
	{
		map_colour_x(self, s, nc, NULL, &colours[nc]);
	}
	maxcol = self->dev_tab[13];
	planesize = mfdb->fww * 2 * mfdb->fh;
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint8 *src   = ((Uint8 *)s->pixels) + y * s->pitch;
		Uint8 *dest  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			srcpix = src[2];
			srcpix = (srcpix << 8) | src[1];
			srcpix = (srcpix << 8) | src[0];
			ink = 0;
			diff = 0xFFFFFFFF;
			/* Find the best colour match */
			for (n = 0; n < maxcol; n++)
			{
				unsigned d = crdiff(s, srcpix, colours[n]);
				if (diff > d) { ink = n; diff = d; }
				if (d == 0) break;	/* Perfect match */
			}
			for (n = 0; n < mfdb->np; n++)
			{
				if (ink & 1) dest[planesize * n] |= xmask;
				else	     dest[planesize * n] &= ~xmask;
				ink = ink >> 1;
			}
			src += 3;
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++dest; }
		}
	}
}




static void surface_to_mfdb8x32(DRIVERDATA *self, MFDB *mfdb, SDL_Surface *s)
{
	int x, y;
	unsigned xmask;
	Uint32 colours[256];
	Uint32 diff;
	int ink, n;
	unsigned maxcol;
	unsigned planesize;
	int nc;

	for (nc = 0; nc < 256; nc++)
	{
		map_colour_x(self, s, nc, NULL, &colours[nc]);
	}
	maxcol = self->dev_tab[13];
	planesize = mfdb->fww * 2 * mfdb->fh;
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint32 *src   = (Uint32 *)(((Uint8 *)s->pixels) + y * s->pitch);
		Uint8  *dest  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			ink = 0;
			diff = 0xFFFFFFFF;
			/* Find the best colour match */
			for (n = 0; n < maxcol; n++)
			{
				unsigned d = crdiff(s, src[0], colours[n]);
				if (diff > d) { ink = n; diff = d; }
				if (d == 0) break;	/* Perfect match */
			}
			for (n = 0; n < mfdb->np; n++)
			{
				if (ink & 1) dest[planesize * n] |= xmask;
				else	     dest[planesize * n] &= ~xmask;
				ink = ink >> 1;
			}
			++src;
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++dest; }
		}
	}
}




static void surface_to_mfdb15(MFDB *mfdb, SDL_Surface *s)
{
	int x, y;
	unsigned xmask;
	Uint16 pixel;
	unsigned planesize;
	int n;

	planesize = mfdb->fww * 2 * mfdb->fh;
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint16 *src   = (Uint16 *)(((Uint8 *)s->pixels) + y * s->pitch);
		Uint8  *dest  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			pixel = *src >> (15 - mfdb->np);
			for (n = 0; n < mfdb->np; n++)
			{
				if (pixel & 1) dest[planesize * n] |= xmask;
				else	       dest[planesize * n] &= ~xmask;
				pixel = pixel >> 1;
			}
			++src;
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++dest; }
		}
	}
}


static void surface_from_mfdb8x8(DRIVERDATA *self, MFDB *mfdb, SDL_Surface *s)
{
	int x, y, np;
	unsigned pmask, xmask, pixel;
	Uint32 colours[256];
	int nc;
	Uint32 planesize;

	for (nc = 0; nc < 256; nc++)
	{
		map_colour_x(self, s, nc, NULL, &colours[nc]);
	}

	planesize = mfdb->fww * mfdb->fh * 2;
	/* Start by blanking the surface */
	memset(s->pixels, 0, s->h * s->pitch);
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint8 *dest  = ((Uint8 *)s->pixels) + y * s->pitch;
		Uint8  *src  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			pixel = 0;
			pmask = 1;
	/* Get colours from the planes */
			for (np = 0; np < mfdb->np; np++)
			{
				if (src[np * planesize] & xmask) pixel |= pmask;
				pmask = pmask << 1;
			}
			dest[x] = colours[pixel];
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++src; }
		}
	}
}




/* Promote a palettised surface (with 8 planes or fewer) to truecolour, using
 * the native colour depth */

static void surface_from_mfdb8x16(DRIVERDATA *self, MFDB *mfdb, SDL_Surface *s)
{
	int x, y, np;
	unsigned pmask, xmask, pixel;
	Uint32 colours[256];
	int nc;
	Uint32 planesize;

	for (nc = 0; nc < 256; nc++)
	{
		map_colour_x(self, s, nc, NULL, &colours[nc]);
	}

	planesize = mfdb->fww * mfdb->fh * 2;
	/* Start by blanking the surface */

	memset(s->pixels, 0, s->h * s->pitch);
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint16 *dest = (Uint16 *)(((Uint8 *)s->pixels) + y * s->pitch);
		Uint8  *src  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			pixel = 0;
			pmask = 1;
	/* Get colours from the planes */
			for (np = 0; np < mfdb->np; np++)
			{
				if (src[np * planesize] & xmask) pixel |= pmask;
				pmask = pmask << 1;
			}
			dest[x] = colours[pixel];
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++src; }
		}
	}
}


static void surface_from_mfdb8x24(DRIVERDATA *self, MFDB *mfdb, SDL_Surface *s)
{
	int x, y, np;
	unsigned pmask, xmask, pixel;
	Uint32 colours[256];
	int nc;
	Uint32 planesize;

	for (nc = 0; nc < 256; nc++)
	{
		map_colour_x(self, s, nc, NULL, &colours[nc]);
	}

	planesize = mfdb->fww * mfdb->fh * 2;
	/* Start by blanking the surface */

	memset(s->pixels, 0, s->h * s->pitch);
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint8 *dest  = ((Uint8 *)s->pixels) + y * s->pitch;
		Uint8  *src  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			pixel = 0;
			pmask = 1;
	/* Get colours from the planes */
			for (np = 0; np < mfdb->np; np++)
			{
				if (src[np * planesize] & xmask) pixel |= pmask;
				pmask = pmask << 1;
			}
			dest[x*3  ] = colours[pixel] & 0xFF;
			dest[x*3+1] = (colours[pixel] >> 8) & 0xFF;
			dest[x*3+2] = (colours[pixel] >> 16) & 0xFF;
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++src; }
		}
	}
}







/* Promote a palettised surface (with 8 planes or fewer) to truecolour, using
 * the native colour depth */
static void surface_from_mfdb8x32(DRIVERDATA *self, MFDB *mfdb, SDL_Surface *s)
{
	int x, y, np;
	unsigned pmask, xmask, pixel;
	Uint32 colours[256];
	int nc;
	Uint32 planesize;

	for (nc = 0; nc < 256; nc++)
	{
		map_colour_x(self, s, nc, NULL, &colours[nc]);
	}

	planesize = mfdb->fww * mfdb->fh * 2;
	/* Start by blanking the surface */

	memset(s->pixels, 0, s->h * s->pitch);
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint32 *dest = (Uint32 *)(((Uint8 *)s->pixels) + y * s->pitch);
		Uint8  *src  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			pixel = 0;
			pmask = 1;
	/* Get colours from the planes */
			for (np = 0; np < mfdb->np; np++)
			{
				if (src[np * planesize] & xmask) pixel |= pmask;
				pmask = pmask << 1;
			}
			dest[x] = colours[pixel];
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++src; }
		}
	}
}



/* Promote generic surface to 555 truecolour */
static void surface_from_mfdb15(MFDB *mfdb, SDL_Surface *s)
{
	int x, y, np;
	unsigned pmask, xmask, pixel;

	Uint32 planesize = mfdb->fww * mfdb->fh * 2;

	/* Start by blanking the surface */

	memset(s->pixels, 0, s->h * s->pitch);
	for (y = 0; y < mfdb->fh; y++)
	{
		Uint16 *dest = (Uint16 *)(((Uint8 *)s->pixels) + y * s->pitch);
		Uint8  *src  = ((Uint8 *)mfdb->mp) + 2 * y * mfdb->fww;
		xmask = 0x80;
		for (x = 0; x < mfdb->fwp; x++)
		{
			pixel = 0;
			pmask = 1;
	/* Get colours from the planes */
			for (np = 0; np < mfdb->np; np++)
			{
				if (src[np * planesize] & xmask) pixel |= pmask;
				pmask = pmask << 1;
			}
			while (np < 15)
			{
				pixel = (pixel << 1) | (pixel & 1);
				++np;
			}
			dest[x] = pixel;
			xmask = xmask >> 1;
			if (xmask == 0) { xmask = 0x80; ++src; }
		}
	}
}




static SDL_Surface *surface_from_mfdb(DRIVERDATA *self, MFDB *mfdb)
{
	SDL_Surface *s;

	if (mfdb->mp == NULL) 
	{
		mfdb->np = self->faux_depth;
		return self->common->surface;
	}
	if (mfdb->mp == ((void *)-1)) 
	{
		mfdb->np = self->faux_depth;
		return self->common->backbuf;
	}
	if (self->common->surface->format->BitsPerPixel == 16 ||
	    mfdb->np <= 8)
	{
		/* SDL_Blit will convert between surfaces with a 
		 * different number of bytes to the pixel, but not if
		 * they have the same number of bytes but a different number
		 * of bits. ie, RGB 555 vs RGB 565 */
		s = SDL_AllocSurface(SDL_SWSURFACE, mfdb->fwp, mfdb->fh,
	                self->common->surface->format->BitsPerPixel,
       	        	self->common->surface->format->Rmask,
       	        	self->common->surface->format->Gmask,
                	self->common->surface->format->Bmask,
                	self->common->surface->format->Amask);
		if (s->format->BytesPerPixel == 1)
		{
			SDL_SetColors(s,
			    self->common->surface->format->palette->colors, 0,
			    self->common->surface->format->palette->ncolors);
		}
}
	else
	{
		/* Allocate a 15bpp pattern */
		s = SDL_AllocSurface(SDL_SWSURFACE, mfdb->fwp, mfdb->fh,
					 15, 0x7C00, 0x3E0, 0x1F, 0);
	}
	if (!s) 
	{
		fprintf(stderr, "Failed to allocate surface %d x %d\n",
				mfdb->fwp, mfdb->fh);
		return NULL;
	}
	if (SDL_MUSTLOCK(s)) SDL_LockSurface(s);

//	if (mfdb->np == 1)
//	{
//		surface_from_mfdb1(mfdb, s, 0xFFFF, 0);
//	} else
	if (mfdb->np <= 8)
	{
		switch (s->format->BytesPerPixel)
		{
			case 1: surface_from_mfdb8x8 (self, mfdb, s); break;
			case 2: surface_from_mfdb8x16(self, mfdb, s); break;
			case 3: surface_from_mfdb8x24(self, mfdb, s); break;
			case 4: surface_from_mfdb8x32(self, mfdb, s); break;
		}
	}
	else
	{
		surface_from_mfdb15(mfdb, s);
	}

	if (SDL_MUSTLOCK(s)) SDL_UnlockSurface(s);
	return s;
}

static void surface_to_mfdb(DRIVERDATA *self, MFDB *mfdb, SDL_Surface *s)
{
	if (mfdb->mp == NULL || mfdb->mp == ((void *)-1)) return;

	if (s == NULL)
	{
		fprintf(stderr, "Internal error: mfdb->mp = %p but surface is null\n", mfdb->mp);
		return;
	}

	if (SDL_MUSTLOCK(s)) SDL_LockSurface(s);
	if (mfdb->np <= 8)
	{
		switch (s->format->BytesPerPixel)
		{
			case 1: surface_to_mfdb8x8 (self, mfdb, s); break;
			case 2: surface_to_mfdb8x16(self, mfdb, s); break;
			case 3: surface_to_mfdb8x24(self, mfdb, s); break;
			case 4: surface_to_mfdb8x32(self, mfdb, s); break;
		}
	}
	else 
	{
		surface_to_mfdb15(mfdb, s);
	}
	if (SDL_MUSTLOCK(s)) SDL_UnlockSurface(s);

	SDL_FreeSurface(s);
}

static void get_rect(signed *pts, SDL_Rect *rc)
{
	arb_corner(pts, ULLR);

	rc->x = pts[0]; if (rc->x < 0) rc->x = 0;
	rc->y = pts[1]; if (rc->y < 0) rc->y = 0;
	rc->w = pts[2] - pts[0] + 1; if (rc->w < 1) rc->w = 1;
	rc->h = pts[3] - pts[1] + 1; if (rc->h < 1) rc->h = 1;
}


/* On an 8-bit palettised display, we don't assume REV_VID */

static void rop8(int rop, Uint8 *src, Uint8 *dest)
{
	switch (rop & 15)
	{
		case 0: dest[0] = 0; break;
		case 1: dest[0] = src[0] & dest[0]; break;
		case 2: dest[0] = src[0] & (~dest[0]); break;
		case 3: dest[0] = src[0]; break;
		case 4: dest[0] = (~src[0]) & dest[0]; break;
		case 5: break;
		case 6: dest[0] = (src[0] ^ dest[0]); break;	
		case 7: dest[0] = src[0] | dest[0]; break;	
		case 8: dest[0] = ~(src[0] | dest[0]); break;	
		case 9: dest[0] = ~(src[0] ^ dest[0]); break;	
		case 10: dest[0] = ~dest[0]; break;
		case 11: dest[0] = src[0] | (~dest[0]); break;
		case 12: dest[0] = ~src[0]; break;
		case 13: dest[0] = (~src[0]) | dest[0]; break;
		case 14: dest[0] = ~(src[0] & dest[0]); break;
		case 15: dest[0] = ~0; break;
	}
}



/* Note: Because this is a truecolour display, we have to assume REV_VID 
 * holds, and use the REV_VID version of all these operations. */
static void rop16(int rop, Uint16 *src, Uint16 *dest)
{
	switch (rop & 15)
	{
		case 0: dest[0] = ~0; break;
		case 1: dest[0] = src[0] | dest[0]; break;
		case 2: dest[0] = src[0] | (~dest[0]); break;
		case 3: dest[0] = src[0]; break;
		case 4: dest[0] = (~src[0]) | dest[0]; break;
		case 5: break;
		case 6: dest[0] = ~(src[0] ^ dest[0]); break;	
		case 7: dest[0] = src[0] & dest[0]; break;	
		case 8: dest[0] = ~(src[0] & dest[0]); break;	
		case 9: dest[0] = ~(src[0] ^ dest[0]); break;	
		case 10: dest[0] = ~dest[0]; break;
		case 11: dest[0] = src[0] ^ (~dest[0]); break;
		case 12: dest[0] = ~src[0]; break;
		case 13: dest[0] = (~src[0]) & dest[0]; break;
		case 14: dest[0] = ~(src[0] | dest[0]); break;
		case 15: dest[0] = 0; break;
	}
}



static void rop32(int rop, Uint32 *src, Uint32 *dest)
{
	switch (rop & 15)
	{
		case 0: dest[0] = ~0; break;
		case 1: dest[0] = src[0] | dest[0]; break;
		case 2: dest[0] = src[0] | (~dest[0]); break;
		case 3: dest[0] = src[0]; break;
		case 4: dest[0] = (~src[0]) | dest[0]; break;
		case 5: break;
		case 6: dest[0] = ~(src[0] ^ dest[0]); break;	
		case 7: dest[0] = src[0] & dest[0]; break;	
		case 8: dest[0] = ~(src[0] & dest[0]); break;	
		case 9: dest[0] = ~(src[0] ^ dest[0]); break;	
		case 10: dest[0] = ~dest[0]; break;
		case 11: dest[0] = src[0] ^ (~dest[0]); break;
		case 12: dest[0] = ~src[0]; break;
		case 13: dest[0] = (~src[0]) & dest[0]; break;
		case 14: dest[0] = ~(src[0] | dest[0]); break;
		case 15: dest[0] = 0; break;
	}
}



static void blit_surface(DRIVERDATA *self, SDL_Surface *src, SDL_Rect *rcSrc,
			SDL_Surface *dest, SDL_Rect *rcDest, int rop)
{
	int dx, dy, sx, sy, row, col, backward;
	SDL_Surface *temp = NULL;

	/* Apply clipping... */
	if (dest == self->common->surface)
	{
		/* [1.0.1] Clip to physical screen */
		if (rcDest->x >= self->common->window.w) return;
		if (rcDest->y >= self->common->window.h) return;
		if (rcDest->x + rcDest->w > self->common->window.w)
		{
			rcDest->w = self->common->window.w - rcDest->x;
		}
		if (rcDest->y + rcDest->h > self->common->window.h)
		{
			rcDest->w = self->common->window.h - rcDest->y;
		}

		if (rcDest->x < self->rectClip.x)
		{
			if (rcDest->x + rcDest->w <= self->rectClip.x) return;

			dx = self->rectClip.x - rcDest->x;
			rcDest->x += dx;
			rcSrc->x  += dx;
			rcDest->w -= dx;
			rcSrc->w  -= dx;
		}
		if (rcDest->y < self->rectClip.y)
		{
			if (rcDest->y + rcDest->h <= self->rectClip.y) return;

			dy = self->rectClip.y - rcDest->y;
			rcDest->y += dy;
			rcSrc->y  += dy;
			rcDest->h -= dy;
			rcSrc->h  -= dy;
		}
		if (rcDest->x + rcDest->w >= self->rectClip.x + self->rectClip.w)
		{
			if (rcDest->x >= self->rectClip.x + self->rectClip.w) return;
		
			dx = rcDest->w - (self->rectClip.x + self->rectClip.w - rcDest->x);
			rcDest->w -= dx;
			rcSrc->w  -= dx; 	
		}
		if (rcDest->y + rcDest->h >= self->rectClip.y + self->rectClip.h)
		{
			if (rcDest->y >= self->rectClip.y + self->rectClip.h) return;
		
			dy = rcDest->h - (self->rectClip.y + self->rectClip.h - rcDest->y);
			rcDest->h -= dy;
			rcSrc->h  -= dy; 	
		}
	}
	if (rcSrc->w > rcDest->w) rcSrc->w = rcDest->w;
	if (rcSrc->h > rcDest->h) rcSrc->h = rcDest->h;


	if (dest->format->BytesPerPixel != src->format->BytesPerPixel)
	{
		/* Coerce source to match destination */
		temp = SDL_AllocSurface(SDL_SWSURFACE, src->w, 
				src->h, 
				dest->format->BitsPerPixel,
				dest->format->Rmask,
				dest->format->Gmask,
				dest->format->Bmask,
				dest->format->Amask);
		if (!temp) return;
		if (dest->format->BytesPerPixel == 1)
		{
			SDL_SetColors(temp,
				dest->format->palette->colors, 0,
				dest->format->palette->ncolors);
		}
		if (SDL_BlitSurface(src, rcSrc, temp, rcSrc))
		{
			fprintf(stderr, "Blit failed: %s\n", SDL_GetError());
		}
		src = temp;
	}
	if (rcSrc->y == rcDest->y) backward = (rcSrc->x > rcDest->x);
	else			 backward = (rcSrc->y > rcDest->y);

	sy = backward ? (rcSrc->y  + rcSrc->h  - 1) : rcSrc->y;
	dy = backward ? (rcDest->y + rcDest->h - 1) : rcDest->y;

	for (row = 0; row < rcSrc->h; row++)
	{
		Uint8 *sptr, *dptr;

		sx = backward ? (rcSrc->x  + rcSrc->w  - 1) : rcSrc->x;
		dx = backward ? (rcDest->x + rcDest->w - 1) : rcDest->x;
		sptr = concat_x(self, src, sx, sy);
		dptr = concat_x(self, dest, dx, dy);

		assert(sptr != NULL);
		assert(dptr != NULL);
		for (col = 0; col < rcSrc->w; col++)
		{
			switch(src->format->BytesPerPixel)
			{
				case 1: rop8 (rop, sptr, dptr); break;
				case 2: rop16(rop, (Uint16 *)sptr, (Uint16 *)dptr); break;
				case 3:
				{
					Uint32 s, d;
					memcpy(&s, sptr, 3);
					memcpy(&d, dptr, 3);
					rop32(rop, &s, &d);
					memcpy(sptr, &s, 3);
					memcpy(dptr, &d, 3);
				}
				break; 
				case 4: rop32(rop, (Uint32 *)sptr, (Uint32 *)dptr); break;
			}
			if (backward) 
			{
				sptr -= src->format->BytesPerPixel;
				dptr -= dest->format->BytesPerPixel;
			}
			else
			{
				sptr += src->format->BytesPerPixel;
				dptr += dest->format->BytesPerPixel;
			}
		}
		if (backward) { --sy; --dy; } else { ++sy; ++dy; }
	} 


	if (temp) SDL_FreeSurface(temp);
}


void sd_rocpyfm(unsigned handle, unsigned mode, signed *pts, MFDB *src,
		MFDB *dest)
{
	SDL_Surface *ssrc, *sdest;
	SDL_Rect rcSrc, rcDest;
	DRIVERDATA *self = lookup_handle(handle);

	if (!self) return;

	get_rect( pts,    &rcSrc);
	get_rect(&pts[4], &rcDest);

	ssrc  = surface_from_mfdb(self, src);
	sdest = surface_from_mfdb(self, dest);

	if (!ssrc || !sdest) return;

	if (src->mp == NULL || src->mp == (void *)-1)
	{
		dest->np = self->faux_depth;	
	}

	if (sdest == self->common->surface && self->CLIP)
	{
		self->rectClip.x += self->common->window.x;
		self->rectClip.y += self->common->window.y;
		SDL_SetClipRect(sdest, &self->rectClip);
		self->rectClip.x -= self->common->window.x;
		self->rectClip.y -= self->common->window.y;
	}
	if (mode == 3)
	{
		/* Optimised straight-through blit */
		if (ssrc == self->common->surface)
		{
			rcSrc.x += self->common->window.x;
			rcSrc.y += self->common->window.y;
		}
		if (sdest == self->common->surface)
		{
			rcDest.x += self->common->window.x;
			rcDest.y += self->common->window.y;
		}
		if (SDL_BlitSurface(ssrc, &rcSrc, sdest, &rcDest))
		{
			fprintf(stderr, "Blit failed: %s\n", SDL_GetError());
		}
		if (ssrc == self->common->surface)
		{
			rcSrc.x -= self->common->window.x;
			rcSrc.y -= self->common->window.y;
		}
		if (sdest == self->common->surface)
		{
			rcDest.x -= self->common->window.x;
			rcDest.y -= self->common->window.y;
		}
	}
	else	/* Doing it 'properly', with logic ops */
	{
		if (SDL_MUSTLOCK(ssrc))  SDL_LockSurface(ssrc);
		if (SDL_MUSTLOCK(sdest)) SDL_LockSurface(sdest);
		blit_surface(self, ssrc, &rcSrc, sdest, &rcDest, mode);
		if (SDL_MUSTLOCK(sdest)) SDL_UnlockSurface(sdest);
		if (SDL_MUSTLOCK(ssrc))  SDL_UnlockSurface(ssrc);
	}
	if (sdest == self->common->surface)
	{
		SDL_SetClipRect(sdest, NULL);
		update_rect(self, pts[4], pts[6], pts[5], pts[7]);
	} 
	if (ssrc != self->common->surface)
	{
		surface_to_mfdb(self, src,  ssrc);
	}
	if (sdest != self->common->surface)
	{
		surface_to_mfdb(self, dest, sdest);
	}
}



void sd_rtcpyfm(unsigned handle, unsigned mode, signed *pts, MFDB *src,
		MFDB *dest, unsigned fg, unsigned bg)
{
	SDL_Surface *ssrc, *sdest, *smask;
	SDL_Rect rcSrc, rcDest;
	DRIVERDATA *self = lookup_handle(handle);
	Uint32 fgpixel, bgpixel;
	
	if (!self) return;
	mode &= 3;

	if (fg >= self->dev_tab[13]) fg = 1;
	if (bg >= self->dev_tab[13]) bg = 1;

	get_rect( pts,    &rcSrc);
	get_rect(&pts[4], &rcDest);

	switch (self->common->surface->format->BytesPerPixel)
	{
		case 1:
		ssrc = SDL_AllocSurface(SDL_SWSURFACE, src->fwp, src->fh,
				8, 0, 0, 0, 0);
		if (!ssrc) return;
		smask = SDL_AllocSurface(SDL_SWSURFACE, src->fwp, src->fh,
				8, 0, 0, 0, 0);
		if (!smask) return;
		SDL_SetColors(ssrc, 
			self->common->surface->format->palette->colors, 0,
			self->common->surface->format->palette->ncolors);
		SDL_SetColors(smask, 
			self->common->surface->format->palette->colors, 0,
			self->common->surface->format->palette->ncolors);

		if (SDL_MUSTLOCK(ssrc)) SDL_LockSurface(ssrc);
		if (SDL_MUSTLOCK(smask)) SDL_LockSurface(smask);
		map_colour(self, fg, NULL, &fgpixel);
		map_colour(self, bg, NULL, &bgpixel);
		switch (mode)
		{
			case 1: surface_from_mfdb1x8(src, ssrc,  fgpixel, bgpixel); 
				break;
			case 2: surface_from_mfdb1x8(src, ssrc,  fgpixel, 0); 
				surface_from_mfdb1x8(src, smask, 0xFF, 0); 
				break;
			case 3: surface_from_mfdb1x8(src, smask, 0, 0xFF); 
				break;
			case 0: surface_from_mfdb1x8(src, ssrc,  0, bgpixel); 
				surface_from_mfdb1x8(src, smask, 0, 0xFF); 
				break;
		}
		break;	// end of 1bpp case

		default:	
		ssrc = SDL_AllocSurface(SDL_SWSURFACE, src->fwp, src->fh,
	                self->common->surface->format->BitsPerPixel,
       	        	self->common->surface->format->Rmask,
       	        	self->common->surface->format->Gmask,
                	self->common->surface->format->Bmask,
                	self->common->surface->format->Amask);

		if (!ssrc) 
		{
			fprintf(stderr, "Failed to create surface: %d x %d pixeals\n", src->fwp, src->fh);
			return;
		}
		smask = SDL_AllocSurface(SDL_SWSURFACE, src->fwp, src->fh,
	                self->common->surface->format->BitsPerPixel,
       	        	self->common->surface->format->Rmask,
       	        	self->common->surface->format->Gmask,
                	self->common->surface->format->Bmask,
                	self->common->surface->format->Amask);
		if (!smask) 
		{
			SDL_FreeSurface(ssrc);
			fprintf(stderr, "Failed to create surface: %d x %d pixeals\n", src->fwp, src->fh);
			return;
		}
		if (SDL_MUSTLOCK(ssrc)) SDL_LockSurface(ssrc);
		if (SDL_MUSTLOCK(smask)) SDL_LockSurface(smask);
		map_colour(self, fg, NULL, &fgpixel);
		map_colour(self, bg, NULL, &bgpixel);
		switch (mode)
		{
			case 1: surface_from_mfdb1(src, ssrc,  fgpixel, bgpixel); 
				break;
			case 2: surface_from_mfdb1(src, ssrc,  fgpixel, 0); 
				surface_from_mfdb1(src, smask, 0, 0xFFFFFFFF); 
				break;
			case 3: surface_from_mfdb1(src, smask, 0, 0xFFFFFFFF); 
				break;
			case 0: surface_from_mfdb1(src, ssrc,  0, bgpixel); 
				surface_from_mfdb1(src, smask, 0xFFFFFFFF, 0); 
				break;
		}
	}
	if (SDL_MUSTLOCK(smask)) SDL_UnlockSurface(smask);
	if (SDL_MUSTLOCK(ssrc))  SDL_UnlockSurface(ssrc);
	sdest = surface_from_mfdb(self, dest);

	if (sdest == self->common->surface && self->CLIP)
	{
		self->rectClip.x += self->common->window.x;
		self->rectClip.y += self->common->window.y;
		SDL_SetClipRect(sdest, &self->rectClip);
		self->rectClip.x -= self->common->window.x;
		self->rectClip.y -= self->common->window.y;
	}
	if (mode == 1)
	{
		/* Optimised straight-through blit */
		if (ssrc == self->common->surface)
		{
			rcSrc.x += self->common->window.x;
			rcSrc.y += self->common->window.y;
		}
		if (sdest == self->common->surface)
		{
			rcDest.x += self->common->window.x;
			rcDest.y += self->common->window.y;
		}
		if (SDL_BlitSurface(ssrc, &rcSrc, sdest, &rcDest))
		{
			fprintf(stderr, "Blit failed: %s\n", SDL_GetError());
		}
		if (ssrc == self->common->surface)
		{
			rcSrc.x -= self->common->window.x;
			rcSrc.y -= self->common->window.y;
		}
		if (sdest == self->common->surface)
		{
			rcDest.x -= self->common->window.x;
			rcDest.y -= self->common->window.y;
		}
	}
	else	
	{
		if (SDL_MUSTLOCK(ssrc))  SDL_LockSurface(ssrc);
		if (SDL_MUSTLOCK(sdest)) SDL_LockSurface(sdest);
		if (SDL_MUSTLOCK(smask)) SDL_LockSurface(smask);
		switch (mode)
		{
			case 1: blit_surface(self, ssrc, &rcSrc, sdest, &rcDest, 3);
				break;

			case 0:
			case 2:
			if (self->common->surface->format->BytesPerPixel == 1)
			{
				blit_surface(self, smask, &rcSrc, sdest, &rcDest, 4);
				blit_surface(self, ssrc, &rcSrc, sdest, &rcDest, 7);

			}
			else
			{
				blit_surface(self, smask, &rcSrc, sdest, &rcDest, 7);
				blit_surface(self, ssrc, &rcSrc, sdest, &rcDest, 1);
			}
			break;

			case 3:	/* Invert */
			blit_surface(self, smask, &rcSrc, sdest, &rcDest, 9);
			break;

		}
		if (SDL_MUSTLOCK(smask)) SDL_UnlockSurface(smask);
		if (SDL_MUSTLOCK(sdest)) SDL_UnlockSurface(sdest);
		if (SDL_MUSTLOCK(ssrc))  SDL_UnlockSurface(ssrc);
	}
	if (sdest == self->common->surface)
	{
		SDL_SetClipRect(sdest, NULL);
		update_rect(self, pts[4], pts[6], pts[5], pts[7]);
	} 
/*	There should not be any need to update the source surface */
	if (ssrc != self->common->surface)
	{
		/* surface_to_mfdb(self, src,  ssrc); */
		SDL_FreeSurface(ssrc);
	} 
	if (sdest != self->common->surface)
	{
		surface_to_mfdb(self, dest, sdest);
	}
	SDL_FreeSurface(smask);
}


