
/*************************************************************************
**       Copyright 1999, Caldera Thin Clients, Inc.                     ** 
**       This software is licenced under the GNU Public License.        **
**       Please see LICENSE.TXT for further information.                ** 
**                                                                      ** 
**                  Historical Copyright                                ** 
**									**
**									**
**									**
**  Copyright (c) 1987, Digital Research, Inc. All Rights Reserved.	**
**  The Software Code contained in this listing is proprietary to	**
**  Digital Research, Inc., Monterey, California and is covered by U.S.	**
**  and other copyright protection.  Unauthorized copying, adaptation,	**
**  distribution, use or display is prohibited and may be subject to 	**
**  civil and criminal penalties.  Disclosure to others is prohibited.	**
**  For the terms and conditions of software code use refer to the 	**
**  appropriate Digital Research License Agreement.			**
**									**
*************************************************************************/

#include "sdsdl_i.h"
#include <math.h>

static void BOX_FILL(DRIVERDATA *self, signed xy[4], int update);
static void HABLINE(DRIVERDATA *self, signed x1, signed x2, signed y1);
static void HLINE_CLIP(DRIVERDATA *self, signed x1, signed x2, signed y);

static void box_hline8(DRIVERDATA *self, int y, int x, int w, Uint16 pattern, Uint32 bg);
static void box_hline16(DRIVERDATA *self, int y, int x, int w, Uint16 pattern, Uint32 bg);
static void box_hline24(DRIVERDATA *self, int y, int x, int w, Uint16 pattern, Uint32 bg);
static void box_hline32(DRIVERDATA *self, int y, int x, int w, Uint16 pattern, Uint32 bg);

signed vec_len  (signed dx, signed dy)
{
        double dxd = dx, dyd = dy;
        double l = (dxd * dxd) + (dyd * dyd);

        return (long)sqrt(l);
}


/*
cseg
	public	GET_DS
	public	ABLINE
	public	HABLINE
	public	DIS_CUR
	public	HIDE_CUR
	public	MOV_CUR
	public	SMUL_DIV
	public	CLC_FLIT
	public	XFM_CRFM
	public	XFM_UDFL
	public	VEX_BUTV
	public	VEX_MOTV
	public	VEX_CURV
	public	RECTFILL
if mono_port
	public	next_seg_pgdown
next_seg_pgdown:
	sub	di, plane_size			;wrap back to 0 offset base
	push	dx
	push	ax
	mov	dx, plane_sel_port
	mov	al, ss:current_bank
	inc	al
	out	dx, al
	mov	ss:current_bank, al
	pop	ax
	pop	dx
	ret
endif
*/

/***************************************************************************
*
* SMUL_DIV (m1,m2,d1)
*
*	 ( ( ( m1 * m2 ) / d1 ) + 1 / 2	
*	m1 = signed 16 bit integer
*	m2 = snsigned 15 bit integer
*	d1 = signed 16 bit integer
*
****************************************************************************/
signed SMUL_DIV(signed m1, signed m2, signed d1)
{
        long long mr = (m1 * m2);
        long dq = (mr / d1);
        long dr = (mr % d1);
        int tick = (mr < 0) ? -1 : 1;

        if (d1 < 0) { tick = (-tick); d1 = (-d1); }
        if (dr < 0) dr = (-dr);
        if ((2 * dr) > d1) dq += tick;

        return dq;
}



/*
eject
EJECT
;************************************************************************
;TENNIS 								*
;	Entry	CX - delta count for ABLINE (count includes last point) *
;	Exit	CX is decremented by one if:				*
;			XOR writing mode and the line about to be	*
;			drawn is not the last of the poly line		*
;			else Cx is left alone				*
;	Purpose:  Xor does not Xor itself at intersection points of	*
;		  polline or plydpl of fill				*
;************************************************************************
tennis:
	cmp	WRT_MODE, 2		; check if xor
	jnz	jimmy	 
	cmp	LSTLIN, 0		; check if XOR and last line of pollin
	jnz	jimmy			; if false go around 
	cmp	cx, 1
	jz	jimmy
	dec	cx
jimmy:	ret
;
EJECT
*/
/****************************************************************
*Subroutine	HABLINE						*
*	Entry:	X1-coordinate					*
*		Y1-coordinate					*
*		X2-coordinate					*
*		patptr - pointer to fill pattern table		*
*								*	
*	Purpose:						*
*		This routine will draw a line from (X1,Y1) to	*
*		(X2,Y1) using a horizontal line algorithm.	*
*****************************************************************/
static void HABLINE(DRIVERDATA *self, signed x1, signed x2, signed y1)
{
	signed xy[4];

	xy[0] = x1;
	xy[1] = y1;
	xy[2] = x2;
	xy[3] = y1;
	BOX_FILL(self, xy, 1);
}


/******************************************************************************
*
* PROCEDURE:	RECTFIL		Fill Rectangle
*
* Inputs:	X1,Y1 = upper left
*		X2,Y2 = lower right
*
*******************************************************************************/

void rectfill(DRIVERDATA *self, signed xy[4], int update)
{
	int n;
	/* Clip to screen */

	for (n = 0; n < 4; n++) 
	{
		if (xy[n] < 0) xy[n] = 0;
		if (n % 2)
		{
			if (xy[n] >= self->common->window.h)
				xy[n] = self->common->window.h - 1;
		}
		else
		{
			if (xy[n] >= self->common->window.w)
				xy[n] = self->common->window.w - 1;
		}
	}
	if (self->CLIP)
	{
		/* See if the left-hand edge is clipped */
		if (xy[0] < self->rectClip.x)
		{
			/* Completely off the left-hand edge? */
			if (xy[2] < self->rectClip.x) return;
			xy[0] = self->rectClip.x;
		}
		/* Repeat for the right-hand edge */
		if (xy[2] >= self->rectClip.x + self->rectClip.w)
		{
			/* Completely off the right-hand edge? */
			if (xy[0] > self->rectClip.x + self->rectClip.w)
				return;
			xy[2] = self->rectClip.x + self->rectClip.w - 1;
		}
		/* See if the top edge is clipped */
		if (xy[3] < self->rectClip.y)
		{
			/* Completely off the top edge? */
			if (xy[1] < self->rectClip.y) return;
			xy[3] = self->rectClip.y;
		}
		/* Repeat for the bottom edge */
		if (xy[1] >= self->rectClip.y + self->rectClip.h)
		{
			/* Completely off the bottom edge? */
			if (xy[3] > self->rectClip.y + self->rectClip.h)
				return;
			xy[1] = self->rectClip.y + self->rectClip.h - 1;
		}
	}
	BOX_FILL(self, xy, update);
}

/******************************************************************************
* BOX_FILL
*	Fill a rectangular area with pattern
*	As rectfill, but assumes clipping has been done already
*
*	Entry	X1,Y1 = upper left corner  inclusive
*		X2,Y2 = lower right corner inclusive
*
******************************************************************************/
static void BOX_FILL(DRIVERDATA *self, signed xy[4], int update)
{
	SDL_Rect rect;
	Uint32 bg;
	int y, x;

	map_colour(self, 0, NULL, &bg);	/* Background */

	rect.x = xy[0];
	rect.y = xy[3];
	rect.w = xy[2] - xy[0] + 1;
	rect.h = xy[1] - xy[3] + 1;

/* Optimisation: If we are doing a solid fill in write mode 0 (replace) or 
 * 1 (trans), then we can use the SDL built-in rectangle function. Otherwise
 * do it by hand. */
	if (self->fill_style == 1 && self->WRT_MODE < 2)	/* Solid fill */
	{
		rect.x += self->common->window.x;
		rect.y += self->common->window.y;
		if (!SDL_FillRect(self->common->surface, &rect, self->FG_BP_1))
		{
			if (update) update_rect(self, xy[0], xy[2], xy[3], xy[1]);
			return;
		}
		rect.x -= self->common->window.x;
		rect.y -= self->common->window.y;
	}
/* Colour pattern? */
	if (self->fill_style == 4 && self->udpt_np > 1 && self->WRT_MODE < 2)
	{
		SDL_Rect sr, dr;

		sr.x = sr.y = 0;
		sr.w = sr.h = 16;
		dr.w = dr.h = 16;
/* Clip to the rectangle we want to fill */
		rect.x += self->common->window.x;
		rect.y += self->common->window.y;
		SDL_SetClipRect(self->common->surface, &rect);	
		rect.x -= self->common->window.x;
		rect.y -= self->common->window.y;
		for (y = (xy[3] & ~0x0F); y <= (xy[1] & ~0x0F); y += 16)
		{
			for (x = (xy[0] & ~0x0F); x <= (xy[2] & ~0x0F); x += 16)
			{
				int err;
 
				dr.x = self->common->window.x + x;
				dr.y = self->common->window.y + y;
				err = SDL_BlitSurface(self->udpt_native, &sr,
						self->common->surface, &dr);
				if (err)
				{
					fprintf(stderr, "User-defined fill (x=%d y=%d w=%d h=%d) -> (x=%d y=%d) failed: (%d) %s\n", 
	sr.x, sr.y, sr.w, sr.h, dr.x, dr.y, err, SDL_GetError());
				}
			}	
		}

		SDL_SetClipRect(self->common->surface, NULL);	
		if (update) update_rect(self, xy[0], xy[2], xy[3], xy[1]);
		return;
	}
	if (update)
	{
		if (SDL_MUSTLOCK(self->common->surface))
			SDL_LockSurface(self->common->surface);
	}
	switch (self->common->surface->format->BytesPerPixel)
	{
		case 1:
		for (y = xy[3]; y <= xy[1]; y++)
		{
			Uint16 pattern = self->patptr[y & self->patmsk];

			box_hline8(self, y, xy[0], xy[2] - xy[0] + 1, pattern, bg);
		}
		break;
		case 2:
		for (y = xy[3]; y <= xy[1]; y++)
		{
			Uint16 pattern = self->patptr[y & self->patmsk];

			box_hline16(self, y, xy[0], xy[2] - xy[0] + 1, pattern, bg);
		}
		break;
		case 3:
		for (y = xy[3]; y <= xy[1]; y++)
		{
			Uint16 pattern = self->patptr[y & self->patmsk];

			box_hline24(self, y, xy[0], xy[2] - xy[0] + 1, pattern, bg);
		}
		break;
		case 4:
		for (y = xy[3]; y <= xy[1]; y++)
		{
			Uint16 pattern = self->patptr[y & self->patmsk];

			box_hline32(self, y, xy[0], xy[2] - xy[0] + 1, pattern, bg);
		}
		break;
		default:
			break;	
	}
	if (update)
	{
		if (SDL_MUSTLOCK(self->common->surface))
			SDL_UnlockSurface(self->common->surface);
		update_rect(self, xy[0], xy[2], xy[3], xy[1]);
	}
}

Uint8 *concat(DRIVERDATA *self, signed x, signed y)
{
	return concat_x(self, self->common->surface, x, y);
}


Uint8 *concat_x(DRIVERDATA *self, SDL_Surface *s, signed x, signed y)
{
	if (x < 0 || y < 0)
	{
		return NULL;
	}
	if (s == self->common->surface)
	{
		x += self->common->window.x;
		y += self->common->window.y;
	}
 	if (x >= s->w || y >= s->h)
	{
		fprintf(stderr, "CONCAT out of range: x=%d w=%d  y=%d h=%d\n",
			x, s->w, y, s->h);
		return NULL;
	}
	return ((Uint8 *)s->pixels) 
		   + y * s->pitch 
		   + x * s->format->BytesPerPixel;
}


/* Draw a patterned horizontal line: 8bpp */
static void box_hline8(DRIVERDATA *self, int y, int x, int w, Uint16 pattern, Uint32 bg)
{
	unsigned n;
	Uint8 *buf = concat(self, x, y);
	Uint16 xmask = 0x8000 >> (x & 15);

	if (!buf) return;
	for (n = 0; n < w; n++)
	{
		if (pattern & xmask)
		{
			switch(self->WRT_MODE)
			{
				case 0: case 1:
					*buf = self->FG_BP_1;
					break;
				case 2: 		
					*buf ^= 0xFF;
					break;
			}
		}
		else 
		{
			switch (self->WRT_MODE)
			{
				case 0: *buf = bg;
					break;
				case 3: *buf = self->FG_BP_1;
					break;
			}
		}
		buf++;
		xmask = xmask >> 1;
		if (xmask == 0) xmask = 0x8000;
	}
}



/* Draw a patterned horizontal line: 16bpp */
static void box_hline16(DRIVERDATA *self, int y, int x, int w, Uint16 pattern, Uint32 bg)
{
	unsigned n;
	Uint16 *buf = (Uint16 *)concat(self, x, y);
	Uint16 xmask = 0x8000 >> (x & 15);

	if (!buf) return;

	for (n = 0; n < w; n++)
	{
		if (pattern & xmask)
		{
			switch(self->WRT_MODE)
			{
				case 0: case 1:
					*buf = self->FG_BP_1;
					break;
				case 2: 		
					*buf ^= 0xFFFF;
					break;
			}
		}
		else 
		{
			switch (self->WRT_MODE)
			{
				case 0: *buf = bg;
					break;
				case 3: *buf = self->FG_BP_1;
					break;
			}
		}
		++buf;	
		xmask = xmask >> 1;
		if (xmask == 0) xmask = 0x8000;
	}
}

/* Draw a patterned horizontal line: 24bpp */
static void box_hline24(DRIVERDATA *self, int y, int x, int w, Uint16 pattern, Uint32 bg)
{
	unsigned n;
	Uint8 *buf = concat(self, x, y);
	Uint16 xmask = 0x8000 >> (x & 15);

	if (!buf) return;

	for (n = 0; n < w; n++)
	{
		if (pattern & xmask)
		{
			switch(self->WRT_MODE)
			{
				case 0: case 1:
					memcpy(buf, &self->FG_BP_1, 3);
					break;
				case 2: 		
					buf[0] ^= 0xFF;
					buf[1] ^= 0xFF;
					buf[2] ^= 0xFF;
					break;
			}
		}
		else 
		{
			switch (self->WRT_MODE)
			{
				case 0: memcpy(buf, &bg, 3);
					break;
				case 3: memcpy(buf, &self->FG_BP_1, 3);
					break;
			}
		}
		buf += 3;	
		xmask = xmask >> 1;
		if (xmask == 0) xmask = 0x8000;
	}
}




/* Draw a patterned horizontal line: 32bpp */
static void box_hline32(DRIVERDATA *self, int y, int x, int w, Uint16 pattern, Uint32 bg)
{
	unsigned n;
	Uint32 *buf = (Uint32 *)concat(self, x, y);
	Uint16 xmask = 0x8000 >> (x & 15);

	if (!buf) return;


	for (n = 0; n < w; n++)
	{
		if (pattern & xmask)
		{
			switch(self->WRT_MODE)
			{
				case 0: case 1:
					*buf = self->FG_BP_1;
					break;
				case 2: 		
					*buf ^= 0xFFFFFFFF;
					break;
			}
		}
		else 
		{
			switch (self->WRT_MODE)
			{
				case 0: *buf = bg;
					break;
				case 3: *buf = self->FG_BP_1;
					break;
			}
		}
		++buf;	
		xmask = xmask >> 1;
		if (xmask == 0) xmask = 0x8000;
	}
}


void box_hline(DRIVERDATA *self, signed y, signed x, unsigned w, Uint16 pattern, Uint32 bg)
{
	/* Sanity clipping */
	if (y < 0 || y >= self->common->window.h) return;
	if (x < 0) x = 0;
	if (x + w > self->common->window.w)
	{
		w = self->common->window.w - x;
	}
	switch (self->common->surface->format->BytesPerPixel)
	{
		case 1: box_hline8(self, y, x, w, pattern, bg);
			break;
		case 2: box_hline16(self, y, x, w, pattern, bg);
			break;
		case 3: box_hline24(self, y, x, w, pattern, bg);
			break;
		case 4: box_hline32(self, y, x, w, pattern, bg);
			break;
		default:
			break;	
	}
}



#if 0
;******************************************************************************
;
; VEX_BUTV
;
; Exchange button change vector
;	
;	Entry	contrl(7),contrl(8) = pointer to user defined routine
;
;	Exit	contrl(9),contrl(10) = old pointer to button routine
;
;
;******************************************************************************
VEX_BUTV:	
	mov	si, offset userbut
	call	exchg_vector
	ret
exchg_vector:
	pushf
	cli
	push	es
	push	ds
	mov	ax, seg contrl
	mov	ds, ax
	les	di,contrl_ptr
	mov	ax,[si]			;get the offset 
	mov	es:word ptr 18[di],ax
	mov	ax,2[si]
	mov	es:word ptr 20[di],ax	;get the segment 
	mov	ax,contrl+14
	mov	[si],ax
	mov	ax,contrl+16
	mov	2[si],ax		;load the new offset/segment 
	pop	ds
	pop	es
	popf	
	ret
;******************************************************************************
;
; VEX_MOTV
;
; Exchange coordinate change vector
;	
;	Entry	contrl(7),contrl(8) = pointer to user defined routine
;
;	Exit	contrl(9),contrl(10) = old pointer to button routine
;
;
;******************************************************************************
VEX_MOTV:	
	mov	si, offset usermot
	call	exchg_vector
	ret
;******************************************************************************
;
; VEX_CURV
;
; Exchange cursor draw vector
;	
;	Entry	contrl(7),contrl(8) = pointer to user defined routine
;
;	Exit	contrl(9),contrl(10) = old pointer to button routine
;
;
;******************************************************************************
VEX_CURV:	
	mov	si, offset usercur
	call	exchg_vector
	ret

#endif

/***********************************************************************
*XFM_CRFM
*        Transform the user defined cursor form to device spec format
*
*	Entry	CONTRL[7]
*		CONTRL[8] = long pointer to input data
*
*	Exit	
*		
*
************************************************************************/
void sd_scform(unsigned handle, unsigned *form)
{
        DRIVERDATA *self = lookup_handle(handle);
	Uint8 data[32];
	Uint8 mask[32];
	Sint16 xhot, yhot;
	int n;

        if (!self) return;
	if (form[2] != 1) return;	/* Cursor must be mono */

	if (self->common->cursor)
	{
		if (SDL_GetCursor() == self->common->cursor) 
			SDL_SetCursor(self->common->old_cursor);
		SDL_FreeCursor(self->common->cursor);
		self->common->cursor = NULL;
	}
	xhot = (signed)(form[0]);
	yhot = (signed)(form[1]);
	/* form[2] == plct */
	/* form[3] == mask colour */
	/* form[4] == data colour */
	for (n = 0; n < 16; n++)
	{
		mask[2*n  ] =  (form[n+5] >> 8);
		mask[2*n+1] =  (form[n+5] & 0xFF);
		data[2*n  ] =  (form[n+21] >> 8);
		data[2*n+1] =  (form[n+21] & 0xFF);
	}
	self->common->cursor = SDL_CreateCursor(data, mask, 16, 16, xhot, yhot);
	SDL_SetCursor(self->common->cursor);
}

/***********************************************************************
*XFM_UDFL
*        Transform the user defined fill pattern to device spec format
*
*	Entry	INTIN[0] - INTIN[15] = fill pattern
*		
*
************************************************************************/
static int sd_udpat4 (DRIVERDATA *self, unsigned *data, unsigned np);
static int sd_udpat16(DRIVERDATA *self, unsigned *data);


int sd_udpat(unsigned handle, unsigned *data, unsigned planes)
{
	int result, n;
	DRIVERDATA *self = lookup_handle(handle);
	unsigned fakedata[0xF0];
	SDL_Rect rc;

	rc.x = rc.y = 0; rc.w = rc.h = 16;

	if (!self) return -1;

	if (planes == 1)
	{
		self->udpt_np = 1;
		for (n = 0; n < 16; n++) self->ud_patrn[n] = data[n];
		return 0;		
	}
	if (planes > 15) planes = 15;

	/* Ensure udpt_native has the same palette as the screen */
	if (self->udpt_native->format->BytesPerPixel == 1)
	{
		SDL_SetColors(self->udpt_native,
		  self->common->surface->format->palette->colors, 0,
		  self->common->surface->format->palette->ncolors);
	}


/* Special case for 4-plane VGA bitmap */
	if (planes >= 2 && planes <= 4)
	{
		/* Take the first plane in case we need it for mono drawing */
		for (n = 0; n < 16; n++) self->ud_patrn[n] = data[n];

		/* OK, now to generate packed-pixel bitmap from multiplane data */
		if (SDL_MUSTLOCK(self->udpt_15bpp))
			SDL_LockSurface(self->udpt_15bpp);
		result = sd_udpat4(self, data, planes);
		if (SDL_MUSTLOCK(self->udpt_15bpp))
			SDL_UnlockSurface(self->udpt_15bpp);

		if (SDL_BlitSurface (self->udpt_15bpp, &rc, self->udpt_native, &rc))
		{
			fprintf(stderr, "vsf_udpat failed: %s\n", SDL_GetError());
		}
		self->udpt_np = 15;
		return result;
	}

/* If passed fewer than 15 planes, stretch the bitmap out to 15 planes. */
	if (planes < 15)
	{
		for (n = 0; n < 15; n++)
		{
			memcpy(&fakedata[n * 16],
			       &data[((n * planes) / 15) * 16],
			       16 * sizeof(unsigned));
		}
		planes = 15;
		data = fakedata;
	}


	self->udpt_np = planes;
	/* Take the first plane in case we need it for mono drawing */
	for (n = 0; n < 16; n++) self->ud_patrn[n] = data[n];

	/* OK, now to generate packed-pixel bitmap from multiplane data */
	if (SDL_MUSTLOCK(self->udpt_15bpp))
		SDL_LockSurface(self->udpt_15bpp);
	result = sd_udpat16(self, data);
	if (SDL_MUSTLOCK(self->udpt_15bpp))
		SDL_UnlockSurface(self->udpt_15bpp);
	if (SDL_BlitSurface (self->udpt_15bpp, &rc, self->udpt_native, &rc))
	{
		fprintf(stderr, "vsf_udpat failed: %s\n", SDL_GetError());
	}
	
	return result;
}

static int sd_udpat4(DRIVERDATA *self, unsigned *data, unsigned planes)
{
	int x, y, np;
	unsigned pmask, xmask, pixel;
	const Uint32 *colours;

	static Uint32 colours4 [4]  = { 0x7FFF, 0x7C00, 0x03E0, 0x0000 };
	static Uint32 colours8 [8]  = { 0x7FFF, 0x7C00, 0x03E0, 0x7FE0, 
				        0x001F, 0x7C1F, 0x03FF, 0x0000 };
	static Uint32 colours16[16] = { 0x7FFF, 0x7C00, 0x03E0, 0x7FE0,
					0x001F, 0x7C1F, 0x3FFF, 0x5EF7,
					0x3DEF, 0x5C00, 0x02E0, 0x5EE0,
					0x0017, 0x5C17, 0x02F7, 0x0000 };

	switch (planes)
	{
		case 2: colours = colours4; break;
		case 3: colours = colours8; break;
		case 4: colours = colours16; break;
	}

	/* Start by blanking the surface */

	memset(self->udpt_15bpp->pixels, 0, 16 * self->udpt_15bpp->pitch);
	for (y = 0; y < 16; y++)
	{
		Uint16 *dest = (Uint16 *)(((Uint8 *)self->udpt_15bpp->pixels)
				+ y * self->udpt_15bpp->pitch);

		xmask = 0x8000;
		for (x = 0; x < 16; x++)
		{
			pixel = 0;
			pmask = 1;
	/* Get colours from the planes */
			for (np = 0; np < planes; np++)
			{
				if (data[np*16+y] & xmask) pixel |= pmask;
				pmask = pmask << 1;
			}
			dest[x] = colours[pixel];
			xmask = xmask >> 1;
		}
	}
	return 0;
}



static int sd_udpat16(DRIVERDATA *self, unsigned *data)
{
	int x, y, np;
	unsigned pmask, xmask, pixel;

	/* Start by blanking the surface */

	memset(self->udpt_15bpp->pixels, 0, 16 * self->udpt_15bpp->pitch);
	for (y = 0; y < 16; y++)
	{
		Uint16 *dest = (Uint16 *)(((Uint8 *)self->udpt_15bpp->pixels)
				+ y * self->udpt_15bpp->pitch);

		xmask = 0x8000;
		for (x = 0; x < 16; x++)
		{
			pixel = 0;
			pmask = 1;
	/* Get colours from the planes */
			for (np = 0; np < 15; np++)
			{
				if (data[np*16+y] & xmask) pixel |= pmask;
				pmask = pmask << 1;
			}
			dest[x] = pixel;
			xmask = xmask >> 1;
		}
	}
	return 0;
}




#if 0
XFM_UDFL:
	mov	dx, CONTRL+6		;get the intin count
	mov	cx, dx			; save away the count
	xor	al, al
xfm_udfl_size_ck:
	inc	al
	sub	dx, 16
	jc	xfm_udfl_bad
	jz	xfm_udfl_size
	jmps	xfm_udfl_size_ck
xfm_udfl_size:
	cmp	al, 1			; 1 is always ok
	jz	xfm_udfl_ok
	cmp	al, num_planes
	jz	xfm_udfl_ok1
xfm_udfl_bad:				;must be integral of 16
	mov	ax, -1
	ret
xfm_udfl_ok:
	mov	udpt_np, 0
	jmps	xfm_udfl_ok2
xfm_udfl_ok1:	
	mov	udpt_np, 32
xfm_udfl_ok2:
	mov	dx, cx
	push	ds			;make sure es=ds
	pop	es
	mov	di,offset ud_patrn	;point at dest
	mov	si,offset INTIN
	mov	cx, dx
	rep	movsw			;copy the data
	mov	si,offset ud_patrn
if (msb_first and byte_swap)
	mov	cx, dx			;swap 16 words
xfm_udfl_loop:
	mov	ax,[si]
	xchg	ah,al
	mov	[si],ax
	inc	si
	inc	si
	loop	xfm_udfl_loop
	ret
endif	
if (msb_first and not byte_swap)
	ret
endif
if (not msb_first and not byte_swap)
	mov	cx, dx
xfm_udfl_loop:
	push	cx
	mov	ax,[si]
	mov	cx,16
xfm_udfl_ilp:
	rol	ax,1
	rcr	bx,1
	loop	xfm_udfl_ilp
	mov	[si],bx
	pop	cx
	inc	si
	inc	si
	loop	xfm_udfl_loop
	ret
endif
eject

#endif

static int compare_ipts(const void *p1, const void *p2)
{
        const unsigned *l1 = (const unsigned *)p1;
        const unsigned *l2 = (const unsigned *)p2;

        return (*l1) - (*l2);
}


/***********************************************************************
*CLC_FLIT
*        Find the Intersection points for each raster line y
*
*	Entry	CONTRL[1] = vectex count
*		PTSIN[]   = verticies
*		y	  = scan line to intersect
*		fill_int  = 0
*
*	Exit	fill_int  = count of intersections
*		INTIN  = x values of intersections
*
************************************************************************/

void CLC_FLIT(DRIVERDATA *self, signed rowy, unsigned count, signed *ptsin)
{
	signed *ipts = malloc(2 * (10+count) * sizeof(signed));
	signed *vertex = ptsin;
	signed *ipt = ipts;
	unsigned npt;

	if (!ipts) return;

/* First, build a list of intersection points */
	for (npt = 0; npt < count-1; ++npt, vertex += 2)
	{
		signed dy, dy1, dy2, dx, ax;
		signed y0 = vertex[1];
		signed y1 = vertex[3];

/* The corresponding lines are commented out in the Digital Research source.
 *
 *              if (y1 & 0x80000000) continue;  // look at the flag if not 
 *                                              // a required edge
 *              y0 &= 0x7FFFFFFF;               // get rid of flag if any
 */
                dy = y1 - y0;
                if (!dy) 	/* Horizontal edge: No intersection */
                {
                        /* vertex[1] |= 0x80000000;  // DRI again */
                        continue;
                }
		dy1 = rowy - y0;
		dy2 = rowy - y1;
		/* Line segment completely above or completely below rowy */
		if ((dy1 < 0 && dy2 < 0) || (dy1 >= 0 && dy2 >= 0))
		{
                        /* DRI commented out 
                         * if (dy1 >= 0)        vertex[1] |= 0x80000000; */
			continue;
		}
		dx = vertex[2] - vertex[0];
		ax = (2 * dx * dy1) / dy;
		if (ax < 0) ax = -((1 - ax) / 2);
		else ax = (ax+1) / 2;
		ax += vertex[0];
		*(ipt++) = ax;
		++self->fill_intersect;
	}
	if (self->fill_intersect < 2)	/* < 2 intersections, go home */
	{
		free(ipts);
		return;
	}
	ipt = ipts;
	if (self->fill_intersect == 2)	/* if 2 then simple sort and draw */
	{
		if (ipts[0] < ipts[1])
		{ 
			HLINE_CLIP(self, ipts[0], ipts[1], rowy);
		}
		else	
		{
			HLINE_CLIP(self, ipts[1], ipts[0], rowy);
		}
		free(ipts);
		return;
	}
        /* Sort the intersections' X coordinates. GEM used bubblesort
         * here but the C library gives us qsort. */
        qsort(ipts,  self->fill_intersect, sizeof(unsigned), compare_ipts);

        /* Draw line segments */
        self->fill_intersect /= 2;
        ipt = ipts;
        for (npt = 0; npt < self->fill_intersect; npt++)
        {
		HLINE_CLIP(self, ipt[0], ipt[1], rowy);
                ipt += 2;
        }
        free(ipts);
}


void update_rect(DRIVERDATA *self, signed xmin, signed xmax,
			signed ymin, signed ymax)
{
//	SDL_Rect rc;

        if (ymin > ymax || xmin > xmax || ymin < 0 || xmin < 0) return;
        if (self->CLIP)
        {
                if (xmin < self->rectClip.x)
                {
                        if (xmax < self->rectClip.x) return;
                        else xmin = self->rectClip.x;
                }
                if (xmax > self->rectClip.x + self->rectClip.w - 1)
                {
                        if (xmin > self->rectClip.x + self->rectClip.w - 1)
				return;
                        else xmax = self->rectClip.x + self->rectClip.w - 1;
                }
                if (ymin < self->rectClip.y)
                {
                        if (ymax < self->rectClip.y) return;
                        else ymin = self->rectClip.y;
                }
                if (ymax > self->rectClip.y + self->rectClip.h - 1)
                {
                        if (ymin > self->rectClip.y + self->rectClip.h - 1)
				return;
                        else ymax = self->rectClip.y + self->rectClip.h - 1;
                }
        }
	if (xmin < 0) xmin = 0;
	if (ymin < 0) ymin = 0;
	if (xmax < 0) return;
	if (ymax < 0) return;
        if (xmin >= self->common->window.w) return;
        if (ymin >= self->common->window.h) return;
        if (xmax >= self->common->window.w) 
		xmax = self->common->window.w - 1;
        if (ymax >= self->common->window.h) 
		ymax = self->common->window.h - 1;
/*
	rc.x = xmin;
	rc.y = ymin;
	rc.w = xmax - xmin + 1;
	rc.h = ymax - ymin + 1;
*/
	assert(xmin <= xmax);
	assert(ymin <= ymax);
	if (xmin > xmax || ymin > ymax) return;

        SDL_UpdateRect(self->common->surface, 
			xmin + self->common->window.x,
			ymin + self->common->window.y, 
			xmax-xmin+1, ymax-ymin+1);

}




#if 0
	ret
full_fill_sort:
	mov	dx, cx
	mov	di, offset INTIN
full_fill_sort_outlp:
	push	cx			; save the full count
	mov	si, di
	mov	ax, word ptr 0[si]	; get the initial lower word
full_fill_sort_inlp:
	mov	bx, word ptr 2[si]
	cmp	ax, bx			; is ax <= bx
	jle	full_fill_sort_inlp_1
	mov	word ptr 0[si], bx	; if ax > bx then exchange
	mov	word ptr 2[si], ax
	mov	bx, ax			; set up for next word
full_fill_sort_inlp_1:
	mov	ax, bx			; already have lower word
	inc	si
	inc	si			; point upwards
	loop	full_fill_sort_inlp
	pop	cx
	dec	dx
	jnz	full_fill_sort_outlp
	mov	cx, fill_int
	shr	cx, 1			; make count of word pairs
full_fill_drawlp:
	mov	ax, [di]
	mov	bx, 2[di]
	mov	X1, ax
	mov	X2, bx
	push	di
	push	cx
	call	HLINE_CLIP
	pop	cx
	pop	di
	add	di,4			; point to the next pair
	loop	full_fill_drawlp
	ret
#endif

static void HLINE_CLIP(DRIVERDATA *self, signed x1, signed x2, signed y)
{
	signed mxclip;

/* [1.0.1] Do this always: even if clipping is turned off, we still need to 
 * clip to screen boundaries */
	if (x1 < self->rectClip.x) 
	{
		/* Completely to the left of clipping rectangle? */
		if (x2 < self->rectClip.x) 
		{
			return; 
		}
		x1 = self->rectClip.x;
	}
	mxclip = self->rectClip.x + self->rectClip.w - 1;
	if (x2 > mxclip)
	{
		/* Completely to the right of clipping rectangle? */
		if (x1 > mxclip)
		{
			return; 
		}
		x2 = mxclip;
	}
	HABLINE(self, x1, x2, y);
}


