/*
SDSDL: GEM video driver for SDL
Copyright 2012 John Elliott <jce@seasip.demon.co.uk>
Based on SDPSC9.VGA copyright 1987, 1999 Caldera Thin Clients, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/


#include "sdsdl_i.h"
#include <SDL_events.h>

/* Mapping of SDL keyboard to GEM keyboard functions */

unsigned sd_qkeys(unsigned handle)
{
	unsigned result = 0;
	SDLMod mod;

        DRIVERDATA *self = lookup_handle(handle);
        if (!self) return 0;

	mod = SDL_GetModState();

	if (mod & KMOD_SHIFT) result |= 1;
	if (mod & KMOD_CTRL ) result |= 2;
	if (mod & KMOD_ALT  ) result |= 4;
	return result;
}

/* UK keyboard layout. XXX Other layouts? */
static unsigned keymap[] =
{
		/* unshifted */	/* shifted */   /* ctrl */	/*alt */
	SDLK_ESCAPE, 0x011b, 	0x0000,		0x011b,		0x0000,
	SDLK_1, 0x0231, 	0x0221,		0x0000,		0x7800,
	SDLK_2, 0x0332, 	0x0322,		0x0300,		0x7900,
	SDLK_3, 0x0433, 	0x049C,		0x0000,		0x7a00,
	SDLK_4, 0x0534, 	0x0524,		0x0000,		0x7b00,
	SDLK_5, 0x0635, 	0x0625,		0x0000,		0x7c00,
	SDLK_6, 0x0736, 	0x075E,		0x071e,		0x7d00,
	SDLK_7, 0x0837, 	0x0826,		0x0000,		0x7e00,
	SDLK_8, 0x0938, 	0x092a,		0x0000,		0x7f00,
	SDLK_9, 0x0a39, 	0x0a28,		0x0000,		0x8000,
	SDLK_0, 0x0b30, 	0x0b29,		0x0000,		0x8100,
	SDLK_MINUS,  0x0c2d, 	0x0c5f,		0x0c1f,		0x8200,
	SDLK_EQUALS, 0x0d3d, 	0x0d2b,		0x0000,		0x8300,
	SDLK_BACKSPACE, 0x0e08,	0x0e08,		0x0e7f,		0x0000,
	SDLK_TAB,	0x0f09, 0x0f00,		0x0000,		0x0000,
	SDLK_q, 0x1071, 	0x1051,		0x1011,		0x1000,
	SDLK_w, 0x1177, 	0x1157,		0x1117,		0x1100,
	SDLK_e, 0x1265, 	0x1245,		0x1205,		0x1200,
	SDLK_r, 0x1372, 	0x1352,		0x1312,		0x1300,
	SDLK_t, 0x1474, 	0x1454,		0x1414,		0x1400,
	SDLK_y, 0x1579, 	0x1559,		0x1519,		0x1500,
	SDLK_u, 0x1675, 	0x1655,		0x1615,		0x1600,
	SDLK_i, 0x1769, 	0x1749,		0x1709,		0x1700,
	SDLK_o, 0x186F, 	0x184F,		0x180F,		0x1800,
	SDLK_p, 0x1970, 	0x1950,		0x1910,		0x1900,
	SDLK_LEFTBRACKET,  0x1A5B, 0x1A7B,	0x1A1B,		0x1A00,
	SDLK_RIGHTBRACKET, 0x1B5D, 0x1B7D,	0x1B1D,		0x1B00,
	SDLK_RETURN, 0x1c0d,	0x1c0d,		0x1c0a,		0x1C00,
	SDLK_KP_ENTER,  0x1C0D,	0x1C0D,		0x1C0A,		0x1C00,
	SDLK_a, 0x1E61, 	0x1E41,		0x1E01,		0x1E00,
	SDLK_s, 0x1F73, 	0x1F53,		0x1F13,		0x1F00,
	SDLK_d, 0x2064, 	0x2044,		0x2004,		0x2000,
	SDLK_f, 0x2166, 	0x2146,		0x2106,		0x2100,
	SDLK_g, 0x2267, 	0x2247,		0x2207,		0x2200,
	SDLK_h, 0x2368, 	0x2348,		0x2308,		0x2300,
	SDLK_j, 0x246A, 	0x244A,		0x240A,		0x2400,
	SDLK_k, 0x256B, 	0x254B,		0x250B,		0x2500,
	SDLK_l, 0x266C, 	0x264C,		0x260C,		0x2600,
	SDLK_SEMICOLON, 0x273b, 0x273a,		0x0000,		0x2700,
	SDLK_QUOTE,	0x2827, 0x2840, 	0x0000,		0x2800,
	SDLK_BACKQUOTE, 0x2960,	0x297E,		0x0000,		0x0000,
/* 2A = left shift */
	SDLK_HASH,	0x2B23, 0x297e,		0x2B1C,		0x2900,
	SDLK_z,	0x2C7A,		0x2C5A,		0x2C1A,		0x2C00,	
	SDLK_x,	0x2D78,		0x2D58,		0x2D18,		0x2D00,	
	SDLK_c,	0x2E63,		0x2E43,		0x2E03,		0x2E00,
	SDLK_v,	0x2F76,		0x2F56,		0x2F16,		0x2F00,
	SDLK_b, 0x3062,		0x3042,		0x3012,		0x3000,
	SDLK_n,	0x316E,		0x314E,		0x310E,		0x3100,
	SDLK_m,	0x326D,		0x324D,		0x320D,		0x3200,
	SDLK_COMMA, 0x332C,	0x333C,		0x0000,		0x3300,
	SDLK_PERIOD, 0x342e,	0x343e,		0x0000,		0x3400,
	SDLK_SLASH, 0x352F,	0x353F,		0x0000,		0x3500,
	SDLK_KP_DIVIDE, 0x352F,  0x352F,	0x0000,		0x0000,	
/* 36 = right shift */
	SDLK_KP_MULTIPLY, 0x372A, 0x372A,	0x0000,		0x0000,
	SDLK_SPACE, 0x3920,     0x3920,		0x3920,		0x3920,
/* 3A = caps lock */
	SDLK_F1, 0x3b00,	0x5400,		0x5E00,		0x6800,	
	SDLK_F2, 0x3c00,	0x5500,		0x5F00,		0x6900,	
	SDLK_F3, 0x3d00,	0x5600,		0x6000,		0x6A00,	
	SDLK_F4, 0x3e00,	0x5700,		0x6100,		0x6B00,	
	SDLK_F5, 0x3f00,	0x5800,		0x6200,		0x6C00,	
	SDLK_F6, 0x4000,	0x5900,		0x6300,		0x6D00,	
	SDLK_F7, 0x4100,	0x5a00,		0x6400,		0x6E00,	
	SDLK_F8, 0x4200,	0x5b00,		0x6500,		0x6F00,	
	SDLK_F9, 0x4300,	0x5c00,		0x6600,		0x7000,	
	SDLK_F10, 0x4400,	0x5d00,		0x6700,		0x7100,	
/* 45 = numlock 46 = scroll lock */
	SDLK_KP7, 0x4700,	0x4737,		0x7700,		0x0000,
	SDLK_HOME, 0x4700,	0x4700,		0x0000,		0x0000,
	SDLK_KP8, 0x4800,	0x4838,		0x0000,		0x0000,
	SDLK_UP,  0x4800,	0x4800,		0x0000,		0x0000,
	SDLK_KP9, 0x4900,	0x4939,		0x8400,		0x0000,
	SDLK_PAGEUP, 0x4900,	0x4900,		0x0000,		0x0000,
	SDLK_KP_MINUS, 0x4A2D,	0x4A2D,		0x0000,		0x0000,
	SDLK_KP4, 0x4B00,	0x4B34,		0x7300,		0x0000,
	SDLK_LEFT, 0x4B00,	0x4B00,		0x0000,		0x0000,
	SDLK_KP5, 0x4C00,	0x4C35,		0x0000,		0x0000,
	SDLK_KP6, 0x4D00,	0x4D36,		0x7400,		0x0000,
	SDLK_RIGHT, 0x4D00,	0x4D00,		0x0000,		0x0000,
	SDLK_KP_PLUS, 0x4E2B,	0x4E2B,		0x0000,		0x0000,
	SDLK_KP1, 0x4F00,	0x4F31,		0x7500,		0x0000,
	SDLK_END, 0x4F00,	0x4F00,		0x7500,		0x0000,
	SDLK_KP2, 0x5000,	0x5032,		0x0000,		0x0000,
	SDLK_DOWN, 0x5000,	0x5000,		0x0000,		0x0000,
	SDLK_KP3, 0x5100,	0x5133,		0x7600,		0x0000,
	SDLK_PAGEDOWN, 0x5100,	0x5100,		0x0000,		0x0000,
	SDLK_KP0, 0x5200,	0x5230,		0x0000,		0x0000,
	SDLK_INSERT, 0x5200,	0x5200,		0x0000,		0x0000,
	SDLK_KP_PERIOD, 0x5300,	0x532E,		0x0000,		0x0000,
	SDLK_DELETE, 0x5300,	0x5300,		0x0000,		0x0000,
/* 54 = SysRQ 55,56 unused */
	SDLK_F11, 0x5700,	0x7000,		0x7a00,		0x8400,	
	SDLK_F12, 0x5800,	0x7100,		0x7b00,		0x8500,	

	SDLK_BACKSLASH, 0x565C, 0x567C,		0x561C,		0x5600,
};

static unsigned xlt_key(int key, SDLMod mod)
{
	int max = sizeof(keymap) / sizeof(keymap[0]);
	int n;

	for (n = 0; n < max; n += 5)
	{
		if (key == keymap[n])
		{
			if      (mod & KMOD_ALT)   return keymap[n+4];
			else if (mod & KMOD_CTRL)  return keymap[n+3];
			else if (mod & KMOD_SHIFT) return keymap[n+2];
			else                       return keymap[n+1];
		}
	}
	return 0;
}

int handle_keydown(DRIVERDATA *self, SDL_Event *ev)
{
	unsigned token;

	if (self->common->textmode)
	{
		return 0;
	}
	if (self->common->kbd_queue_len >= KBD_QUEUE_MAX)
	{
		return 0;	/* Keyboard buffer full */
	}
	token = xlt_key(ev->key.keysym.sym, ev->key.keysym.mod);
	if (!token)
	{
		return 0;	/* Token not recognised by GEM */
	}
	self->common->kbd_queue[self->common->kbd_queue_len] = token;
	++self->common->kbd_queue_len;
}


int handle_keyup(DRIVERDATA *self, SDL_Event *ev)
{
	if (self->common->textmode)
	{
		return 0;
	}
	return 0;	/* For a first pass, ignore all key-up events */	
}



unsigned sd_pollkey(DRIVERDATA *self, unsigned *result, signed max)
{
	/* Limit 'max' to the length of the keyboard queue */
	if (max > self->common->kbd_queue_len)
	{
		max = self->common->kbd_queue_len;
	}
	if (!max) return 0;
	/* Pop the first 'max' entries off the queue */
	memcpy(result, self->common->kbd_queue, max * sizeof(unsigned));
	/* If there are entries left, move them to the start */
	if (max < self->common->kbd_queue_len)
	{
		memmove(self->common->kbd_queue,
		       &self->common->kbd_queue[max],
			(self->common->kbd_queue_len - max) * sizeof(unsigned));
	}
	self->common->kbd_queue_len -= max;
	return max;
}


unsigned sdrq_choice  (unsigned handle, unsigned chc_number, unsigned value,
				unsigned *val_out, unsigned *status)
{
	DRIVERDATA *self = lookup_handle(handle);
	unsigned buf;

	if (!self)
	{
		*status = 0;
		return 0;
	}
	*val_out = value;
	if (self->chc_mode)	/* Sample */
	{
		if (!sd_pollkey(self, &buf, 1))
		{
			*status = 0;
			return 0;
		}
		/* Was it a function key? */
		if (buf >= 0x3B00 && buf <= 0x4400)
		{
			*status = 1;
			*val_out = (buf - 0x3A00) >> 8;
			return 1;
		}
		/* Normal key */
		*status = 2;
		val_out[1] = buf;
		return 2;
	}
	else	/* Request */
	{
		while (1)
		{
			if (sd_pollkey(self, &buf, 1) &&
				buf >= 0x3B00 && buf <= 0x4400)
			{
				*status = 1;
				*val_out = (buf - 0x3A00) >> 8;
				return 1;
			}
			SDL_PumpEvents();
		}
	}
}



