/************************************************************************
 *       Copyright 1999, Caldera Thin Clients, Inc.                     *
 *       This software is licenced under the GNU Public License.        *
 *       Please see LICENSE.TXT for further information.                *
 *                                                                      *
 *                  Historical Copyright                                *
 *                                                                      *
 *                                                                      *
 *									*
 *  Copyright (c) 1987, Digital Research, Inc. All Rights Reserved.	*
 *  The Software Code contained in this listing is proprietary to	*
 *  Digital Research, Inc., Monterey, California and is covered by U.S.	*
 *  and other copyright protection.  Unauthorized copying, adaptation,	*
 *  distribution, use or display is prohibited and may be subject to 	*
 *  civil and criminal penalties.  Disclosure to others is prohibited.	*
 *  For the terms and conditions of software code use refer to the 	*
 *  appropriate Digital Research License Agreement.			*
 *									*
 ************************************************************************/

#include "sdsdl_i.h"

extern Uint8 byte_mask_table[9];

/*
HORZ_OFF	equ	2
; Special effects masks.
THICKEN		equ	1h
LIGHT		equ	2h
SKEW    	equ	4h
UNDER		equ	8h
OUTLINE		equ	10h
SHADOW		equ	20h
ROTATE		equ	0c0h
ROT90		equ	040h
ROT180		equ	080h
ROT270		equ	0c0h
ROTODD		equ	040h
ROTHIGH		equ	080h
SCALE		equ	100h
		dseg
;;;		extrn	points_out:word		; jn, justified text fix
		extrn	fi_flags:word
		extrn	lt_hoff:byte
		extrn	rt_hoff:byte
		extrn	off_htbl:word
		extrn	CONTRL:word
		extrn	INTIN:word
		extrn	INTOUT:word
		extrn	PTSIN:word
		extrn	FG_BP_1:word
		extrn	NEXT_PAT:word
		extrn	patmsk:word
		extrn	patptr:word 
		extrn	SOLID:word
		extrn	DITHER:word
		extrn	DITHRMSK:word
		extrn	X1:word
		extrn	X2:word
		extrn	Y1:word
		extrn	Y2:word
		extrn	SPECIAL:word
		extrn	T_SCLSTS:word
		extrn	MONO_STA:word
		extrn	XACC_DDA:word
		extrn	DDA_INC:word
		extrn	WRT_MODE:word
		extrn	TEXT_BP:word
		extrn	MAP_COL:word
		extrn	LN_MASK:word
		extrn	plane_loop_count:byte
if ( num_planes gt 1 ) and ( not segment_access )
		extrn	plane_port_tbl:byte
		extrn	plane_read_tbl:byte
endif
		extrn	WEIGHT:word		;bolding ammount for char
		extrn	L_OFF:word,R_OFF:word	;offsets for italic chars
		extrn	CHAR_DEL:word		;additional width to add
		extrn	DESTX:word,DESTY:word	;destination pixel address
		extrn	ACTDELY:word		;form height after scale
		extrn	CLIP:word		;clipping flag
		extrn	XMN_CLIP:WORD,XMX_CLIP:WORD
		extrn	YMN_CLIP:WORD,YMX_CLIP:WORD
		extrn	FLIP_Y:word
		extrn	width:word
		extrn	rot_case:word
		extrn	PTSOUT:word

		extrn	FIR_CHR:word
		extrn	f_i_last:word
		extrn	OFF_TBL:dword
		extrn	FOFF:word
		extrn	FWIDTH:word
		extrn	DELY:word
		extrn	txbuf1:word
		extrn	txbuf2:word
		extrn	buff_ptr:word
		extrn	source_seg:word
		extrn	tempdely:word		;temporary char height storage
		extrn	foffindex:word		;temporary char offset index
		extrn	JUST_DEL_WORD:word
		extrn	JUST_DEL_WORD_REM:word
		extrn	JUST_DEL_CHAR:word
		extrn	JUST_DEL_CHAR_REM:word
		extrn	JUST_DEL_SIGN:word
		extrn	JUST_NEG:word

		extrn	byte_mask_table:byte
		extrn	text_ix:byte		;text color index
		extrn	text_mode:byte		;text write mode
		extrn	font_off:word		;temp storage for font offset
		extrn	port_sel:word
		just_width	dw	0
		extrn	line_table:word
		extrn	pixel_table:word

		cseg
		extrn	chk_fnt:near
		extrn	txt_blt_branch_table:word
		extrn	double_table:word
		extrn	txtblt_xor_rr_s:near
		extrn	txtblt_xor_rl_s:near
		extrn	d_gtext:near
		extrn	ACT_SIZ:near
		extrn	RECTFILL:near
		extrn	CONCAT:near

		extrn	next_dest:word
		extrn	next_source:word
		extrn	rot_height:word
		extrn	SPECIAL_CS:word

		public	clr_skew_next_line_0
		public	clr_skew_next_line_90
		public	clr_skew_next_line_180
		public	clr_skew_next_line_270
		public	clr_skew_next_pixel_0
		public	clr_skew_next_pixel_90
		public	clr_skew_next_pixel_180
		public	clr_skew_next_pixel_270
		public	text_blit
		public	text_move
		public	clr_skew
		if not GSX
		 public	dqt_just
		endif
		public	d_justif	
		public	text_rotate
		public	text_scale
		public	text_grey
		public	text_bold
		public	text_italic
*/
/****************************************************************
* TEXT_BLIT							*
* moves the character from temp form to screen			*
* Inputs:							*
*		ES:DI		Pointer to the screen		*
*		foffindex	Source form lines to skip	*
*		tempdely	Char height in pels		*
*		bl		Char buffer form width in bytes	*
*		si		char width in pels		*
*		ax		source x			*
*		bp		dest x				*
* Uses:		all						*
* Returns:	none						*
****************************************************************/

void text_blit8(DRIVERDATA *self, Uint8 *dest, unsigned src_pitch,
			unsigned char_width, unsigned src_x, unsigned dest_x,
			unsigned mode)
{
	Uint8 mask, val;
	unsigned y, x, xb;
	Uint8 *scr;
	Uint32 bg;

        map_colour(self, 0, NULL, &bg);

	self->font_off = self->buff_ptr + self->foffindex * src_pitch;

	for (y = 0; y < self->tempdely; y++)
	{
		scr = (Uint8 *)(&dest[y * self->common->surface->pitch]);
		xb = (src_x / 8) + y * src_pitch;
		mask = 0x80 >> (src_x & 7);
		val = self->font_off[xb];
		for (x = 0; x < char_width; x++)
		{
			if (val & mask)
			{
				switch (mode)
				{
					case 0: 
					case 1: scr[x] = self->text_colour; break;
					case 2: scr[x] ^= 0xFF; break;
				}
			}	
			else
			{ 
				switch (mode)
				{
					case 0: scr[x] = bg; break;
					case 3: scr[x] = self->text_colour; break;
				}
			}
			mask = mask >> 1;
			if (!mask)
			{
				mask = 0x80;
				val = self->font_off[++xb];
			}
		}
	}	
}




void text_blit16(DRIVERDATA *self, Uint8 *dest, unsigned src_pitch,
			unsigned char_width, unsigned src_x, unsigned dest_x,
			unsigned mode)
{
	Uint8 mask, val;
	unsigned y, x, xb;
	Uint16 *scr;
	Uint32 bg;

        map_colour(self, 0, NULL, &bg);

	self->font_off = self->buff_ptr + self->foffindex * src_pitch;

	for (y = 0; y < self->tempdely; y++)
	{
		scr = (Uint16 *)(&dest[y * self->common->surface->pitch]);
		xb = (src_x / 8) + y * src_pitch;
		mask = 0x80 >> (src_x & 7);
		val = self->font_off[xb];
		for (x = 0; x < char_width; x++)
		{
			if (val & mask)
			{
				switch (mode)
				{
					case 0: 
					case 1: scr[x] = self->text_colour; break;
					case 2: scr[x] ^= 0xFFFF; break;
				}
			}	
			else
			{ 
				switch (mode)
				{
					case 0: scr[x] = bg; break;
					case 3: scr[x] = self->text_colour; break;
				}
			}
			mask = mask >> 1;
			if (!mask)
			{
				mask = 0x80;
				val = self->font_off[++xb];
			}
		}
	}	
}


void text_blit24(DRIVERDATA *self, Uint8 *dest, unsigned src_pitch,
			unsigned char_width, unsigned src_x, unsigned dest_x,
			unsigned mode)
{
	Uint8 mask, val;
	unsigned y, x, xb;
	Uint8 *scr;
	Uint32 bg;

        map_colour(self, 0, NULL, &bg);

	self->font_off = self->buff_ptr + self->foffindex * src_pitch;

	for (y = 0; y < self->tempdely; y++)
	{
		scr = (Uint8 *)(&dest[y * self->common->surface->pitch]);
		xb = (src_x / 8) + y * src_pitch;
		mask = 0x80 >> (src_x & 7);
		val = self->font_off[xb];
		for (x = 0; x < char_width; x++)
		{
			if (val & mask)
			{
				switch (mode)
				{
					case 0: 
					case 1: scr[x*3]   = self->text_colour &0xFF; 
						scr[x*3+1] = (self->text_colour >> 8) & 0xFF;
						scr[x*3+2] = (self->text_colour >> 16) & 0xFF;
						break;
					case 2: scr[x*3]   ^= 0xFF;
						scr[x*3+1] ^= 0xFF;
						scr[x*3+2] ^= 0xFF; break;
				}
			}	
			else
			{ 
				switch (mode)
				{
					case 0: scr[x*3]   = bg & 0xFF; 
						scr[x*3+1] = (bg >> 8) & 0xFF;
						scr[x*3+2] = (bg >> 16) & 0xFF;
						break;
					case 3: scr[x*3]   = self->text_colour &0xFF; 
						scr[x*3+1] = (self->text_colour >> 8) & 0xFF;
						scr[x*3+2] = (self->text_colour >> 16) & 0xFF;
						break;
				}
			}
			mask = mask >> 1;
			if (!mask)
			{
				mask = 0x80;
				val = self->font_off[++xb];
			}
		}
	}	
}




void text_blit32(DRIVERDATA *self, Uint8 *dest, unsigned src_pitch,
			unsigned char_width, unsigned src_x, unsigned dest_x,
			unsigned mode)
{
	Uint8 mask, val;
	unsigned y, x, xb;
	Uint32 *scr;
	Uint32 bg;

        map_colour(self, 0, NULL, &bg);

	self->font_off = self->buff_ptr + self->foffindex * src_pitch;

	for (y = 0; y < self->tempdely; y++)
	{
		scr = (Uint32 *)(&dest[y * self->common->surface->pitch]);
		xb = (src_x / 8) + y * src_pitch;
		mask = 0x80 >> (src_x & 7);
		val = self->font_off[xb];
		for (x = 0; x < char_width; x++)
		{
			if (val & mask)
			{
				switch (mode)
				{
					case 0: 
					case 1: scr[x] = self->text_colour; break;
					case 2: scr[x] ^= 0xFFFFFFFF; break;
				}
			}	
			else
			{ 
				switch (mode)
				{
					case 0: scr[x] = bg; break;
					case 3: scr[x] = self->text_colour; break;
				}
			}
			mask = mask >> 1;
			if (!mask)
			{
				mask = 0x80;
				val = self->font_off[++xb];
			}
		}
	}	
}




void text_blit(DRIVERDATA *self, Uint8 *dest, unsigned src_pitch,
			unsigned char_width, unsigned src_x, 
			unsigned dest_x, unsigned mode)
{
        switch (self->common->surface->format->BytesPerPixel)
        {
		case 1: text_blit8(self, dest, src_pitch, char_width, src_x, dest_x, mode); break;
		case 2: text_blit16(self, dest, src_pitch, char_width, src_x, dest_x, mode); break;
		case 3: text_blit24(self, dest, src_pitch, char_width, src_x, dest_x, mode); break;
		case 4: text_blit32(self, dest, src_pitch, char_width, src_x, dest_x, mode); break;
	}
}


/*

;si = CHAR WIDTH
;bp = destx
;ax = sourcex
;calculate the right mask right mask = dh
	mov	cx, 7
	mov	bx, bp
	add	bx, si			;right x = DESTX+DELX
	and	bx, cx 
	mov	dh, byte_mask_table[bx] ;dh = right mask
;calculate the left mask
;left mask = dl
	mov	bx, bp			;get dest x in bx
	and	bx, cx
	mov	dl, byte_mask_table[bx]
	not	dl
;calculate rotate value
;rotate value = bh 0000drrr	d=0 if right 1 if left rrr = count 0 - 7
	mov	ch, al
	and	ch, cl			;mask off all but 3 lsbits
	mov	bh, bl
	sub	bh, ch			;if destx >= source x
	jns	text_blit_clc_height
	add	bh, 16			;get the left rotate index
;calculate the vertical loop count
;vertical loop count = cl
text_blit_clc_height:
	mov	cx, tempdely
	or	cl, cl			;vertical loop count must be
	jnz	text_blit_clc_ok	;non-zero
	inc	cl
;calculate the middle byte count
;middle byte count = ch
text_blit_clc_ok:
	push	ax
	sub	bl, 8			;left count max = 8-xposn
	neg	bl
	mov	ah, bh
	xor	bh, bh
	sub	si, bx			;delx - leftx 
	jnc	text_blit_clc_middle
	or	dl, dh			;if left only then must modify masks
	mov	dh, 0ffh		;the right mask is now none
	xor	ch, ch			;the middle count is 0
	jmps	text_blit_op_branch
text_blit_clc_middle:
	mov	bx, si
	shr	bx, 1
	shr	bx, 1
	shr	bx, 1			;divide out the low order bits
	mov	ch, bl
text_blit_op_branch:
	pop	si			;get the source x back
	shr	si, 1
	shr	si, 1
	shr	si, 1 
	add	si, font_off		;offset from the top
	pop	bp			;get the char form width	
	mov	bl, text_mode
	mov	al, ah			;get the rotate flag
	and	al, 8
	shr	al, 1
	or	bl, al
	xor	bh, bh
	and	ah, 11110111b		;mask off the rotate flag
	shl	bx, 1			;index in by words
	push	ds
	mov	ds, source_seg
	call	txt_blt_branch_table[bx]
	pop	ds
	ret
*/

/****************************************************************
* TEXT_MOVE							*
*								*
* moves the character from font form to temp			*
* Inputs:							*
*		ES:DI	Pointer to the char buffer		*
*		DELY	Char height in pels			*
*		foff	font offset 				*
*		fseg	font segment				*
*		bl	Char buffer form width in bytes		*
*		si	char width in pels			*
*		ax	source x				*
* Uses:								*
*		all						*
* Returns:							*
*		dx = to next form				*
****************************************************************/
signed text_move(DRIVERDATA *self, Uint8 *dest, unsigned dest_pitch,  /* bl */
		unsigned src_width/* si */, unsigned src_x /* ax */)
{
	unsigned lines;
	Uint8 rmask, lmask;
	Sint8 sadj;
	Uint8 midc;
	Uint8 *saddr;

/* Clear out the character buffer */
	memset(dest, 0, dest_pitch * self->font_inf.form_height);

/* si = CHAR WIDTH			;calculate the masks , rotate counts
 * bp = destx = 0
 * ax = sourcex */

	rmask = byte_mask_table[src_width & 7];	/* right x = DESTX + DELX */

/* left mask = dl */
	lmask = 0;
	sadj = 0;				/* get dest x in bx */
/* calculate rotate value
 * rotate value = bh 0000drrr	d=0 if right 1 if left rrr = count 0 - 7 */

	sadj = -(src_x & 7);
	if (sadj < 0) sadj += 16;

/* calculate the vertical loop count */
/* vertical loop count = lines */
	lines = self->font_inf.form_height;
/* calculate the middle byte count */
/* middle byte count = ch */

	if (src_width < 8)
	{
		lmask |= rmask;	/* All done on the left mask */
		rmask = 0xFF;	/* No right mask */
		midc = 0;		/* No middle bytes */
	}
	else
	{
		midc = (src_width - 8) >> 3;
	}
	saddr = self->font_inf.dat_table + (src_x >> 3);

	if (sadj & 8)
	{
/* text_move_left */
		while (lines)
		{
			txtblt_xor_rl_i(self, dest, saddr, sadj & 7, lmask, rmask, midc);
			dest += dest_pitch;
			saddr += self->font_inf.form_width;
			--lines;
		}
	}
	else
	{
/* text_move_right */
		while(lines)
		{
			txtblt_xor_rr_i(self, dest, saddr, sadj & 7, lmask, rmask, midc);
			saddr += self->font_inf.form_width;
			dest += dest_pitch;
			--lines;
		}
	}
	return dest_pitch * self->font_inf.form_height;
}
/*
void showbuf(Uint8 *buf, int pitch)
{
	Uint8 mask = 0x80;
	Uint8 bt;
	int x;

	int m = pitch * 8;
	if (m > 64) m = 64;

	bt = buf[0];
	for (x = 0; x < m; x++)
	{
		if (bt & mask) fprintf(stderr, "1"); else fprintf(stderr, "0");
		mask = mask >> 1;
		if (!mask)
		{
			mask = 0x80;
			++buf;
			bt = buf[0];
		}
	}
	fprintf(stderr, "\n");
}
*/

/****************************************************************
* TEXT_ITALIC							*
* Italicizes the form passed to it				*
* Inputs:							*
*		ES:DI	Pointer to the char buffer		*
*		cx	Char height in pels			*
*		bl	Char buffer width in bytes		*
*		dx	Char width in pels			*
* Uses:		all						*
* Returns:	none						*
*****************************************************************/

void text_italic(DRIVERDATA *self, Uint8 *buffer, unsigned src_height/*cx*/,
			unsigned src_pitch/*bl*/, unsigned src_width /*dx*/)
{
	Uint8 *sptr, *dptr, *tmpbuf;
	Uint8 leftx, rightx;
	unsigned row;

	sptr = buffer + (src_pitch * (src_height - 1));	/* Bottom row of char */
	tmpbuf = malloc(src_pitch * 2);	/* Working buffer */

	rightx = (src_width & 7);	/* rightx */
	leftx = 0;			/* leftx */
/*
 * es:di = pointer to ptsin	ds:si = pointer to bottom of char form
 * bh = leftx  = 00000xxx		bl = rightx = 00000xxx
 * cx = vert scan count in pels	ax = char form width in bytes
 * bp = char width in pels
 */
/* This code is executed every two scan lines */
	row = src_height;
	dptr = tmpbuf;
	while (row)
	{
		Uint8 rmask = 0;
		Uint8 lmask = 0;
		Uint8 midc = 0;
 
		memset(tmpbuf, 0, 2 * src_pitch);
						/* Clear out temp buffer */
		lmask = 0;				/* Left mask */
		rmask = byte_mask_table[rightx];	/* Right mask */

		if (src_width + leftx < 8)		/* Fits in a single byte */
		{
			lmask |= rmask;		/* Left | Right mask */
			rmask = 0xFF;		/* No right mask */
			midc = 0;			/* No middle bytes */
		}
		else
		{
			midc = (src_width / 8);		/* Middle bytes */
		}
		txtblt_xor_rr_i(self, dptr, sptr, leftx, lmask, rmask, midc);	
		memcpy(sptr, tmpbuf, src_pitch);	
		--row;
		if (!row) break;
		sptr -= src_pitch;
		/* Repeat for the next row, leaving bh / dh / dl / ch 
		 * unchanged */
		memset(tmpbuf, 0, 2 * src_pitch);	/* Clear out temp buffer */
		txtblt_xor_rr_i(self, dptr, sptr, leftx, lmask, rmask, midc);	
		memcpy(sptr, tmpbuf, src_pitch);	
		--row;
		if (!row) break;
		sptr -= src_pitch;
		++leftx;
		++rightx;
		if (leftx >= 7)
		{
			++dptr;
		}
		leftx &= 7;
		rightx &= 7;	
	}	
	free(tmpbuf);
}


/****************************************************************
* TEXT_BOLD							*
* Bolds the form passed to it					*
* Inputs:							*
*		ES:DI	Pointer to the char buffer		*	
*		cx	Char height				*
*		bl	Char buffer width in bytes		*
* Uses:		all						*
* Returns:	bx, cx, dx, ds					*
*****************************************************************/

void text_bold(DRIVERDATA *self, Uint8 *buffer, unsigned src_height,
			unsigned src_pitch)
{
	unsigned row, col, pass;
	Uint8 *cur;
	Uint8 b0, val;
	
	for (row = 0; row < src_height; row++)
	{
		cur = buffer + row * src_pitch;
		for (pass = 0; pass < self->WEIGHT; pass++)
		{
			b0 = 0;
			for (col = 0; col < src_pitch; col++)
			{
				val = b0 | (cur[col] >> 1);
				b0 = (cur[col] & 1) << 7;
				cur[col] |= val;
			}	
		}
	}
}


/****************************************************************
* TEXT_GREY							*
* Greys the form passed to it					*
* Inputs:							*
*		ES:DI	Pointer to the char buffer		*
*		cx	Char height				*
*		bl	Char buffer width in bytes		*
* Uses:		ax, bx, cx, di, ds				*
* Returns:	bx, cx, dx					*
*****************************************************************/
void text_grey(DRIVERDATA *self, Uint8 *buffer, unsigned src_height,
			unsigned src_pitch)
{
	unsigned y, x;
	Uint8 mask = 0xAA;

	for (y = 0; y < src_height; y++)
	{
		for (x = 0; x < src_pitch; x++)
		{
			*buffer &= mask;
			++buffer;
		}
		mask = ~mask;
	}
}

static signed scale_up(DRIVERDATA *self, signed v0)
{
	signed v1 = 0;
	unsigned col;
	unsigned dda = self->XACC_DDA;
	unsigned dda_inc = self->DDA_INC;
	if (!v0) return 0;
	for (col = 0; col < v0; col++)
	{
		dda += dda_inc;
		if (dda & 0x10000) ++v1;
		++v1;
		dda &= 0xFFFF;
	}	
	return v1;
}

static signed scale_down(DRIVERDATA *self, signed v0)
{
	signed v1 = 0;
	unsigned col;
	unsigned dda = self->XACC_DDA;
	unsigned dda_inc = self->DDA_INC;
	if (!v0) return 0;
	for (col = 0; col < v0; col++)
	{
		dda += dda_inc;
		if (dda & 0x10000) ++v1;
		dda &= 0xFFFF;
	}	
	return v1;
}




/****************************************************************
* TEXT_SCALEx2							*
* Doubles the form passed to it					*
* Inputs:							*
*		ES:DI	Pointer to the char buffer		*
*		cx	Char height				*
*		bl	Char buffer width in bytes		*
* 		si	Char width in pels			*
*		dx	Source form size in bytes		*
* Uses:		ax, bx, cx, di, ds				*
* Returns:	bl, si						*
*****************************************************************/
static void text_scalex2(DRIVERDATA *self, Uint8 *buffer, unsigned *src_pitch,
			unsigned *char_width, unsigned src_size)
{
/* doubles the current character in size */
	unsigned new_pitch, new_charp, new_extra;
	Uint8 *sptr, *dptr;
	unsigned row, col;

/* 11/6/86 horizontal offset enhancement */
	self->lt_hoff *= 2;
	self->rt_hoff *= 2;

	/* AX = source pitch */
	char_width[0] *= 2;
	/* New cell width, bytes */
	new_pitch = (char_width[0] + self->CHAR_DEL + 7) / 8;
	/* New char width, bytes */
	new_charp = (char_width[0] + 7) / 8; 
	/* And the difference between them */
	new_extra = (new_pitch - new_charp);
	
	self->buff_ptr += src_size;	/* Dest. buffer */

	for (row = 0; row < self->ACTDELY; row++)
	{
/*
; si = first temp buffer
; di = second temp buffer
; src_pitch = source form width
; new_charp = character width in bytes (no effects)
; dx = height of dest
; new_extra = dest (skew + bold)  */
		sptr = buffer + row * src_pitch[0];
		dptr = self->buff_ptr + 2 * row * new_pitch;
		for (col = 0; col < new_charp; col++)
		{
			Uint16 ax = self->common->double_table[*sptr];
			++sptr;
			*dptr++ = (ax >> 8) & 0xFF;
			++col;
			if (col >= new_charp) break;
			*dptr++ = ax & 0xFF;
		}
		for (col = 0; col < new_extra; col++)
		{
			*dptr++ = 0;
		}
/* Duplicate the line */
		memmove(dptr, dptr - new_pitch, new_pitch);
	}
	src_pitch[0] = new_pitch;
}


static void text_scale_up_x(DRIVERDATA *self, Uint8 *dptr, Uint8 *sptr, 
			unsigned width, unsigned dda)
{
	unsigned col;
	Uint8 smask = 0x80;
	Uint8 dmask = 0x80;
	Uint8 val;

	dptr[0] = 0;
	for (col = 0; col < width; col++)
	{
		val = (sptr[0] & smask);
		
		dda += self->DDA_INC;
		if (dda & 0x10000)
		{
			dda &= 0xFFFF;
			if (val) dptr[0] |= dmask; 
			dmask = dmask >> 1;
			if (!dmask) { dmask = 0x80; ++dptr; dptr[0] = 0; }
		}		
		if (val) dptr[0] |= dmask;
		dmask = dmask >> 1;
		if (!dmask) { dmask = 0x80; ++dptr; dptr[0] = 0; }
		smask = smask >> 1;
		if (!smask) { smask = 0x80; ++sptr; }
	}
}

static void text_scale_down_x(DRIVERDATA *self, Uint8 *dptr, Uint8 *sptr, 
			unsigned width, unsigned dda)
{
	unsigned col;
	Uint8 smask = 0x80;
	Uint8 dmask = 0x80;
	Uint8 val;

	dptr[0] = 0;
	for (col = 0; col < width; col++)
	{
		val = (sptr[0] & smask);
		
		dda += self->DDA_INC;
		if (dda & 0x10000)
		{
			dda &= 0xFFFF;
			if (val) dptr[0] |= dmask; 
			dmask = dmask >> 1;
			if (!dmask) { dmask = 0x80; ++dptr; dptr[0] = 0; }
		}		
		smask = smask >> 1;
		if (!smask) { smask = 0x80; ++sptr; }
	}
}



/****************************************************************
* TEXT_SCALE							*
* Scales the form passed to it					*
* Inputs:							*
*		ES:DI	Pointer to the char buffer		*
*		cx	Char height				*
*		bl	Char buffer width in bytes		*
* 		si	Char width in pels			*
*		dx	Source form size in bytes		*
* Uses:		ax, bx, cx, di, ds				*
* Returns:	bl, si						*
*****************************************************************/
void text_scale(DRIVERDATA *self, Uint8 *buffer, unsigned *src_pitch,
			unsigned *char_width, unsigned src_size)
{
	unsigned new_pitch, new_charp, new_extra;
	Uint8 *si, *di;
	unsigned row, col;
	unsigned dda, dda0;
	unsigned new_width;
	unsigned ydda;

	if (self->T_SCLSTS == 1)	/* Is this a scale up? */
	{
/* Optimise the scale x 2 case */
		if (self->DDA_INC == 0xFFFF)
		{
			text_scalex2(self, buffer, src_pitch, char_width,
					src_size);
			return;
		}

		/* Scaling up */
		self->buff_ptr += src_size;	/* Dest. buffer */
		/* Scale up offsets */
		self->lt_hoff = scale_up(self, self->lt_hoff);
		self->rt_hoff = scale_up(self, self->rt_hoff);
		new_width     = scale_up(self, *char_width);

		/* Update the accumulator */
		dda0 = dda = self->XACC_DDA;
		for (col = 0; col < *char_width; col++)
		{
			dda = (dda + self->DDA_INC) & 0xFFFF;
		}
		self->XACC_DDA = dda;
		dda = dda0;

		new_pitch = (new_width + self->CHAR_DEL + 7) / 8;
		/* New char width, bytes */
		new_charp = (new_width + 7) / 8; 
		/* And the difference between them */
		new_extra = (new_pitch - new_charp);

		ydda = 0x7FFF;
		di = self->buff_ptr;
		for (row = 0; row < self->ACTDELY; row++)
		{
			si = buffer + row * src_pitch[0];
/* Scale up row */
			text_scale_up_x(self, di, si, *char_width, dda0);
			ydda += self->DDA_INC;
			di += new_pitch;
			if (ydda & 0x10000)
			{
				memmove(di, di - new_pitch, new_pitch);
				di += new_pitch;
				ydda &= 0xFFFF;
			}
		}
		src_pitch[0] = new_pitch;
		char_width[0] = new_width;
		return;
	}
	/* Scaling down */
	self->buff_ptr += src_size;	/* Dest. buffer */
	/* Scale up offsets */
	self->lt_hoff = scale_down(self, self->lt_hoff);
	self->rt_hoff = scale_down(self, self->rt_hoff);
	new_width     = scale_down(self, *char_width);

	/* Update the accumulator */
	dda0 = dda = self->XACC_DDA;
	for (col = 0; col < *char_width; col++)
	{
		dda = (dda + self->DDA_INC) & 0xFFFF;
	}
	self->XACC_DDA = dda;
	dda = dda0;

	new_pitch = (new_width + self->CHAR_DEL + 7) / 8;
	/* New char width, bytes */
	new_charp = (new_width + 7) / 8; 
	/* And the difference between them */
	new_extra = (new_pitch - new_charp);

	ydda = 0x7FFF;
	di = self->buff_ptr;
	for (row = 0; row < self->ACTDELY; row++)
	{
		si = buffer + row * src_pitch[0];
		ydda += self->DDA_INC;
		if (ydda & 0x10000)
		{
			text_scale_down_x(self, di, si, *char_width, dda0);
			di += new_pitch;
			ydda &= 0xFFFF;
		}
	}
	src_pitch[0] = new_pitch;
	char_width[0] = new_width;
}


/****************************************************************
* TEXT_ROTATE							*
* Inputs:							*
*	bl = character buffer width, in bytes			*
*	cx = height of the character cell			*
*	dx = offset to the next character			*
*	si = character width -- all horizontal effects applied	*
*	es:di -> temporary character buffer			*
*****************************************************************/
void text_rotate(DRIVERDATA *self, Uint8 *buffer, unsigned rot_height,
			unsigned *src_pitch, unsigned src_width)
{
	Uint8 *sptr, *dptr;

	signed next_source;
	signed next_dest;
	unsigned dest_height = src_width;
	Uint8 smask, dmask;
	Uint8 value;
	unsigned row, col;

/* The source and destination pointers must be modified.  The source will
 * be somewhere in the temporary buffer and the destination will be somewhere
 * else in the second temporary buffer.  Exactly where depends on the rotation
 * angle. */
	sptr = buffer;
	dptr = self->buff_ptr = self->txbuf2;

/* Text rotated 180 degrees is handled dptrfferently from text rotated either 
 * 90 or 270 degrees.*/
	if (self->SPECIAL & ROTODD)
	{
/* Set up the increment values to the next source line and next destination
 * line. */
		if (!dest_height) ++dest_height;
				/* If no width, give it width */
		next_source = src_pitch[0]; 
		next_dest = (rot_height + 7) >> 3; /* Dest pitch */
		if (!(self->SPECIAL & ROTHIGH))
		{
/* 90 degrees:  the starting source will be the beginning of the
 * temporary character buffer and the starting destination will be the
 * beginning of the last scan line in the second buffer.
 */
			dptr += (next_dest * (dest_height - 1)); /* Last row */
			next_dest = -next_dest;
		}
		else	/* !self->SPECIAL & ROTHIGH */
		{
/* 270 degrees:  the starting source will be the beginning of the last scan
 * line in the temporary character buffer and the starting destination will
 * be the beginning of the second buffer.
 */
			next_source = -next_source;
			sptr += (src_pitch[0] * (rot_height - 1));
		}
/* Top of the outer 90/270 degree rotation loop. */
		smask = 0x80;
		for (row = 0; row < dest_height; row++)
		{
			Uint8 *src = sptr;
			Uint8 *dst = dptr;
			value = 0;
			dmask = 0x80;
			for (col = 0; col < rot_height; col++)
			{
				if (src[0] & smask) value |= dmask;
				dmask = dmask >> 1;		
				if (!dmask)
				{
					*dst = value;
					++dst;
					value = 0;
					dmask = 0x80;
				}	
				src += next_source;
			}
			if (dmask != 0x80) *dst = value;
			smask = smask >> 1;
			if (smask == 0)
			{
				smask = 0x80;	
				++sptr;
			}
			dptr += next_dest;
		}
	}
	else
	{
/*
; 180 degrees:  the starting source will be the beginning of the temporary
; character buffer and the starting destination will be the last used byte
; in the second temporary buffer. */
		col = (rot_height * src_pitch[0]);
		dptr += (col - 1);
		dmask = 0x01 << (src_width & 7);
		smask = 0x80;
		value = 0;

		++col;
		while (dptr > self->buff_ptr)
		{
			if (sptr[0] & smask) value |= dmask;
			smask = smask >> 1;
			if (smask == 0) { smask = 0x80; ++sptr; }
			dmask = (dmask << 1) & 0xFF;
			if (dmask == 0) 
			{
				*dptr = value;
				--dptr;
				value = 0;
				dmask = 1;
			}
		} 
	}	/* End of rotation */

/* Swap height and width information in the input registers if the rotation
 * is either 90 or 270 degrees. */
	if (self->SPECIAL & ROTODD)
	{
		self->tempdely = src_width;		/* New height */
		src_pitch[0] = (rot_height + 7) >> 3;	/* New pitch */
	}
}
static unsigned d_justif_calc(DRIVERDATA *self, signed x, signed y, 
		unsigned count, unsigned *text, signed width, 
		unsigned wordspace, unsigned charspace, unsigned *sizecount,
		unsigned *sizes, unsigned fx);


/*******************************************
*d_justif
*	justified text entry point
*	Entry
*********************************************/

void sd_justified(unsigned handle, signed x, signed y, unsigned count,
		  unsigned *text, signed width, unsigned wordspace, 
                  unsigned charspace, unsigned *sizecount, unsigned *sizes,
		  unsigned fx)
{
	DRIVERDATA *self = lookup_handle(handle);

	if (!self) return;
	chk_fnt(self);
	wordspace |= 1;	/* JN 11-21-87, force interword spacing */
	if (!d_justif_calc(self, x, y, count, text, width,
			   wordspace, charspace, sizecount, sizes, fx)) 
	{
		return;
	}
	sd_gtext(handle, x, y, count, text, fx);
}

static unsigned d_justif_sub(DRIVERDATA *self, unsigned a)
{
	if (a >= self->font_inf.first_ade && a <= self->font_inf.last_ade)
		return a;

	a = chk_ade(self, a);
	if (a == 0) a = ' ';
	return a;
}


/*******************************************
*d_justif_calc
*	justified text entry point
*
*	Entry
;********************************************/

static unsigned d_justif_calc(DRIVERDATA *self, signed x, signed y, 
		unsigned count, unsigned *text, signed width, 
		unsigned wordspace, unsigned charspace, unsigned *sizecount,
		unsigned *sizes, unsigned fx)
{
	unsigned wcount = 0;	/* Init the width counter */
	unsigned sp_count = 0;	/* Count of spaces */
	unsigned m = 0;
	unsigned curch = 0;
	unsigned sp_width = 0;
	unsigned just_neg = 0;
	signed adjust = 0;
	int n;

	*sizecount = 0;
	if (!count) return 0;	/* No text to measure */

/* si = text */
/* es:bx = self->font_inf.off_table */

	if (!(wordspace & 0x8000))
	{
		for (m = 0; m < count; m++)
		{
/* d_justif_calc_width_loop: */
			curch = d_justif_sub(self, text[m]);
			if (curch == ' ') ++sp_count;	/* Increase count of spaces */

			curch -= self->font_inf.first_ade;
			wcount += (self->font_inf.off_table[curch + 1]);
			wcount -= (self->font_inf.off_table[curch]);
/* 11/6/86 DH */
/* added to horizontal offset calculation */
			if (self->font_inf.flags & HORZ_OFF)
			{
				wcount -= self->font_inf.hor_table[curch * 2];	
				wcount -= self->font_inf.hor_table[curch * 2 + 1];	
			}
		}
		curch = d_justif_sub(self, ' ') - self->font_inf.first_ade;
		sp_width += (self->font_inf.off_table[curch + 1]);
		sp_width -= (self->font_inf.off_table[curch]);
		if (self->font_inf.flags & HORZ_OFF)
		{
			sp_width -= self->font_inf.hor_table[curch * 2];	
			sp_width -= self->font_inf.hor_table[curch * 2 + 1];	
		}
/* d_justif_calc_spc_nohorz: 
 *
 * width of string with no attributes
 * width = bp
 * space width = di  sp_width
 * space count = ch
*/
		if (self->SPECIAL & SCALE)
		{
			if (self->DDA_INC == 0xFFFF)
			{
				wcount *= 2;
				sp_width *= 2;
			}
			else
			{
				wcount = ACT_SIZ(self, wcount);
				sp_width = ACT_SIZ(self, sp_width);
			}
		}
		if (self->SPECIAL & THICKEN)
		{
			wcount += count * self->font_inf.thicken;
		}
		if (fx != 11) return wcount;
		if (self->SPECIAL & ROTODD)
		{
			/* Make allowances for aspect ratio */
			width = (width * XSIZE) / YSIZE;
		}
		self->width = width;
		if (width >= wcount)
		{
/******
*This is where max space and min space can be set by application
*******/
			just_neg = 0;
			self->JUST_DEL_SIGN = 1;
			adjust = width - wcount;
			if (!charspace) sp_width = 1000;
		}
		else
		{
			self->JUST_DEL_SIGN = -1;
			adjust = wcount - width;
			just_neg = 1; /* Set the negative flag */
/******
*This is where max space and min space can be set by application
*******/ 
			sp_width /= 2;
		}
		self->JUST_DEL_WORD     = 0;
		self->JUST_DEL_WORD_REM = 0;
		self->JUST_DEL_CHAR     = 0;
		self->JUST_DEL_CHAR_REM = 0;
		if (wordspace && sp_count != 0)	/* Are there some spaces? */
		{
			int pelsperspace = (adjust / sp_count);
			if (pelsperspace > sp_width)
			{
				self->JUST_DEL_WORD = sp_width;
			}
			else
			{
				self->JUST_DEL_WORD     = pelsperspace;
				self->JUST_DEL_WORD_REM = adjust % sp_count;
				if (just_neg) 
				{
					self->JUST_DEL_WORD = -pelsperspace;
				}
				return wcount;
			}
		}
/* inter character only is allowed
 * width     = ax	adjust
 * dir       = si	just_neg	
 * numspaces = cx	sp_count
 * max space = di	sp_width
 */
		if (charspace)
		{
			adjust -= (sp_width * sp_count);	/* Deduct width of spaces */
			if (count > 1)
			{
				self->JUST_DEL_CHAR     = adjust / (count-1);
				self->JUST_DEL_CHAR_REM = adjust % (count-1);
			}
		}
		if (just_neg) 
		{
			self->JUST_DEL_WORD = -self->JUST_DEL_WORD;
			self->JUST_DEL_CHAR = -self->JUST_DEL_CHAR;
		}
		return wcount;
	}
	else
/*
 *
 * returns the width of each character in the string in dev coords
 * after having justified the string
 */
	{
		signed just_width = 0;
		// dx = sizes

		*sizecount = 0;
		for (m = 0; m < count; m++)
		{
			curch = d_justif_sub(self, text[m]);
			if (curch == ' ') ++sp_count;	/* Increase count of spaces */

			curch -= self->font_inf.first_ade;
			wcount  = (self->font_inf.off_table[curch + 1]);
			wcount -= (self->font_inf.off_table[curch]);
/* 11/6/86 DH */
/* added to horizontal offset calculation */
			if (self->font_inf.flags & HORZ_OFF)
			{
				wcount -= self->font_inf.hor_table[curch * 2];	
				wcount -= self->font_inf.hor_table[curch * 2 + 1];	
			}
			just_width += wcount;
			sizes[*sizecount] = wcount;
			++(*sizecount);
		}
		curch = d_justif_sub(self, ' ') - self->font_inf.first_ade;
		sp_width += (self->font_inf.off_table[curch + 1]);
		sp_width -= (self->font_inf.off_table[curch]);
		if (self->font_inf.flags & HORZ_OFF)
		{
			sp_width -= self->font_inf.hor_table[curch * 2];	
			sp_width -= self->font_inf.hor_table[curch * 2 + 1];	
		}
		wcount = just_width;
/* width of string with no attributes
 * width = bp
 * space width = di  sp_width
 * space count = ch
 */
		if (self->SPECIAL & SCALE)
		{
			if (self->DDA_INC == 0xFFFF)
			{
				wcount *= 2;
				sp_width *= 2;
				for (n = 0; n < sizecount[0]; n++)
				{
					sizes[n] *= 2;
				}
			}
			else	/* Not doubling */
			{
				wcount       = ACT_SIZ(self, wcount);
				if (self->T_SCLSTS & 1)	/* Scale up? */
				{
					unsigned scaler = 0x7FFF;
					unsigned size = 0;
					for (n = 0; n < sizecount[0]; n++)
					{
						adjust = 0;
						size = sizes[n];
						while (size)
						{
							scaler += self->DDA_INC;
							++adjust;
							if (scaler > 0x10000)
							{
								++adjust;
								scaler &= 0xFFFF;
							}
							--size;
						}
						sizes[n] = adjust;
					}
				}
				else	/* Scaling down */
				{
					unsigned scaler = 0x7FFF;
					for (n = 0; n < sizecount[0]; n++)
					{
						unsigned size = 0;
						adjust = 0;
						size = sizes[n];
						while (size)
						{
							scaler += self->DDA_INC;
							if (scaler > 0x10000)
							{
								++adjust;
								scaler &= 0xFFFF;
							}
							--size;
						}
						sizes[n] = adjust;
					}

				}

			}	/* end if not doubling */
		}		/* end if scaled */ 
		if (self->SPECIAL & THICKEN)
		{
			wcount += self->WEIGHT * count;
			for (n = 0; n < sizecount[0]; n++)
			{
				sizes[n] += self->WEIGHT;
			}
		}
		/* 'width' = width desired */
		just_neg = 0;		/* si */
		self->JUST_DEL_SIGN = 1;	/* bx */
		/*ax = width - wcount */
		/* bx = 1 */
		if (width >= wcount)	/* Spaces will be widened */
		{
			adjust = width - wcount;
			if (!charspace) sp_width = 1000;
		}
		else
		{
			just_neg = 1;
			self->JUST_DEL_SIGN = -1;
			adjust = wcount - width;
			sp_width /= 2;	
		}
		self->JUST_DEL_WORD = 0;
		self->JUST_DEL_WORD_REM = 0;
		self->JUST_DEL_CHAR = 0;
		self->JUST_DEL_CHAR_REM = 0;
		if ((wordspace & 1) && sp_count) /* Is inter word justify allowed? */
		{
			if ((adjust / sp_count) > sp_width)
			{
				self->JUST_DEL_WORD = sp_width;
			}
			else	/* Can all be done with word spacing */
			{
				self->JUST_DEL_WORD = adjust / sp_count;
				self->JUST_DEL_WORD_REM = adjust % sp_count;
				goto d_justif_calc_inq_calc_exit;
			}
		}
		/* Need to use inter character spacing */
/* width     = ax    adjust
 * dir       = si    JUST_NEG
 * numspaces = cx    sp_count
 * max space = di    sp_width 
 */
		if (charspace & 1)
		{
			adjust -= (sp_width * sp_count);	/* Get space used by words */
			if (sizecount[0] > 1)
			{
				self->JUST_DEL_CHAR = adjust / (sizecount[0] - 1);
				self->JUST_DEL_CHAR_REM = adjust % (sizecount[0] - 1);
			}
		}
d_justif_calc_inq_calc_exit:
		if (just_neg)
		{
			self->JUST_DEL_CHAR = -self->JUST_DEL_CHAR;
			self->JUST_DEL_WORD = -self->JUST_DEL_WORD;
		}
		/* bx->text 
		   si->sizes */

		*sizecount = count;
		for (n = 0; n < count; n++)
		{
			curch = text[n];
			if (curch == ' ')
			{
				sizes[n] += self->JUST_DEL_WORD;
				if (self->JUST_DEL_WORD_REM) 
				{
					sizes[n] += self->JUST_DEL_SIGN;
					--self->JUST_DEL_WORD_REM;
				}
			}
			else
			{
				sizes[n] += self->JUST_DEL_CHAR;
				if (self->JUST_DEL_CHAR_REM)
				{
					sizes[n] += self->JUST_DEL_SIGN;
					--self->JUST_DEL_CHAR_REM;
				}
			}
		}
		if (charspace & 0x8000) return 0;
		return width;
	}
}
/*
;
dqt_calc_sub:
	cmp	ax,FIR_CHR
	jc	dqt_calc_sub1
	cmp	ax,f_i_last
	ja	dqt_calc_sub1
	ret
;
dqt_calc_sub1:
	push	bx
	push	es
	push	ds
	push	di
	push	si
	push	dx
	push	cx
	push	ax
	call	check_ade
	and	ax,ax
	pop	ax
	jnz	dqt_calc_sub2
	mov	ax,' '
dqt_calc_sub2:
	pop	cx
	pop	dx
	pop	si
	pop	di
	pop	ds
	pop	es
	les	bp,dword ptr OFF_TBL
	mov	bx,FIR_CHR
	shl	bx,1
	sub	bp,bx
	pop	bx
	ret
;
;
*/

/****************************************************************
* dqt_justified - inquire offsets for justified text		*
****************************************************************/


static signed dqt_just_setup(DRIVERDATA *self, signed x, signed y, 
		signed req_width, unsigned count, unsigned *str, 
		unsigned word_space, unsigned char_space, signed *offsets);

static signed dqt_calc(DRIVERDATA *self, unsigned ch);


void sd_qtjustified(unsigned handle, signed x, signed y, signed req_width,
		unsigned count, unsigned *str, unsigned word_space, 
		unsigned char_space, signed *offsets)
{
	DRIVERDATA *self = lookup_handle(handle);
	signed xpos;
	signed *curpt;
	int n;

	if (!self) return;

	chk_fnt(self);
	if (!dqt_just_setup(self, x, y, req_width, count, str, word_space, 
				char_space, offsets))
	{
		return;
	}
	self->FLIP_Y = 1;

	memset(offsets, 0, sizeof(signed) * 2 * count);
	/* si = str  bp = &off_tbl[-FIR_CHR] */

	xpos = 0;	/* Width starts at 0 */
	curpt = &offsets[0];
	if (self->rot_case & 1)	/* Vertical */
	{
		curpt = &offsets[1];
	}
	if (self->rot_case == 0 || self->rot_case == 3)
	{
		for (n = 0; n < count; n++)
		{
			xpos += dqt_calc(self, str[n]);	
			*curpt = xpos;
			curpt += 2;	
		}
	}
	else
	{
		for (n = 0; n < count; n++)
		{
			xpos -= dqt_calc(self, str[n]);
			*curpt = xpos;
			curpt += 2;	
		}
	}
}

static signed dqt_calc(DRIVERDATA *self, unsigned ch)
{
	signed char_w;

	if (ch < self->font_inf.first_ade || ch > self->font_inf.last_ade)
	{
		if (!chk_ade(self, ch)) ch = ' ';
	}
	ch -= self->font_inf.first_ade;
	char_w =  self->font_inf.off_table[ch + 1]  
		- self->font_inf.off_table[ch];
	if (self->SPECIAL & SCALE)
	{
		char_w = ACT_SIZ(self, char_w);
	}
	if (self->SPECIAL & THICKEN)
	{
		char_w += self->WEIGHT;
	}
	if (ch == ' ')
	{
		char_w += self->JUST_DEL_WORD;
		if (self->JUST_DEL_WORD_REM)
		{
			char_w += self->JUST_DEL_SIGN;
			--self->JUST_DEL_WORD_REM;
		}
	}
	else
	{
		char_w += self->JUST_DEL_CHAR;
		if (self->JUST_DEL_CHAR_REM)
		{
			char_w += self->JUST_DEL_SIGN;
			--self->JUST_DEL_CHAR_REM;
		}
	}
	return char_w;
}





static signed dqt_just_setup(DRIVERDATA *self, signed x, signed y, 
		signed req_width, unsigned count, unsigned *str, 
		unsigned word_space, unsigned char_space, signed *offsets)
{
	unsigned spaces = 0;	/* CH: Count of spaces */
	unsigned width = 0;	/* BP: Total width */
	signed char_w;
	signed space_w;		/* DI: Width of a space */
	unsigned n;
	signed delta;
	unsigned just_neg = 0;

	if (!count) return 0;
	for (n = 0; n < count; n++)
	{
		unsigned curch = str[n];
		if (curch < self->font_inf.first_ade || 
		    curch > self->font_inf.last_ade)
		{
			if (!chk_ade(self, curch)) curch = ' ';
		}
		if (curch == ' ') ++spaces;
		curch -= self->font_inf.first_ade;
		char_w =  self->font_inf.off_table[curch + 1]  
			- self->font_inf.off_table[curch];
		width += char_w;
	}
	if (' ' < self->font_inf.first_ade || 
	    ' ' > self->font_inf.last_ade)
	{
		chk_ade(self, ' ');
	}
	space_w = self->font_inf.off_table[' ' + 1 - self->font_inf.first_ade]
		- self->font_inf.off_table[' '     - self->font_inf.first_ade];
	if (self->SPECIAL & SCALE)
	{
		width   = ACT_SIZ(self, width);
		space_w = ACT_SIZ(self, space_w);
	}
	if (self->SPECIAL & THICKEN)
	{
		width += self->WEIGHT * count;
	}
	if (req_width <= 0) return 0;

	if (self->SPECIAL & ROTODD)
	{
		signed tmp = (req_width * XSIZE);

		req_width = tmp / YSIZE;
		/* Round to nearest */
		if (((tmp % YSIZE) * 2) >= YSIZE) ++req_width;
	}
	self->width = req_width;
	just_neg = 0;	
	self->JUST_DEL_SIGN = 1;
	/* di = space width */
	if (req_width < width)
	{
		self->JUST_DEL_SIGN = -1;
		delta = width - req_width;
		just_neg = 1;
		space_w /= 2;
	}
	else
	{
		delta = req_width - width;
		space_w *= 2;
	}
	/* CX = count of spaces */
	self->JUST_DEL_WORD = 0;
	self->JUST_DEL_WORD_REM = 0;
	self->JUST_DEL_CHAR = 0;
	self->JUST_DEL_CHAR_REM = 0;
	if (word_space)
	{
		if (spaces)
		{
			if ((delta / spaces) <= space_w)
			{
/* I'm not sure what to make of this. That's why I've left it as a goto. It
 * seems that if we're reducing the size of a space to align the text, don't
 * adjust the size of characters. */
				self->JUST_DEL_WORD_REM = (delta % spaces);
				self->JUST_DEL_WORD = delta / spaces;
				goto dqt_just_setup_end;
			}
			else
			{
				self->JUST_DEL_WORD = space_w;
			}
		}
	}
	else
	{
		spaces = 0;
	}
	if (char_space)
	{
		delta -= (space_w * spaces);
		if (count > 0)
		{
			self->JUST_DEL_CHAR = delta / count;
			self->JUST_DEL_CHAR_REM = delta % count;
		}
	}
dqt_just_setup_end:
	if (just_neg)
	{
		self->JUST_DEL_WORD = -self->JUST_DEL_WORD;
		self->JUST_DEL_CHAR = -self->JUST_DEL_CHAR;
	}
	return -1;
}
	
/****************************************************************
 * CLR_SKEW							*
 ****************************************************************/

void clr_skew(DRIVERDATA *self)
{
	Uint32 save_bp   = self->FG_BP_1;
	Uint16 save_mask = self->LN_MASK;
	unsigned rotate, row;
	Uint16 skewmask;
	signed xy[4];

/* If replace mode, use the background color. */
	if (self->WRT_MODE == 0)
	{
		/* set to background color */
		map_colour(self, 0, NULL, &self->FG_BP_1);
	}
/*
 * Set the rotation case index:  0 for 0 degrees, 2 for 90 degrees, 4 for
 * 180 degrees, and 6 for 270 degrees.  Prepare for the output loop. */

	self->LN_MASK = 0xFFFF;

	rotate = (self->SPECIAL & 0xC0) >> 5; /* Rotation case index */
	skewmask = 0x5555;		/* Skewing mask */

	/* Top of the line output loop. */

	for (row = self->ACTDELY; row > 0; row--)
	{
		xy[0] = self->X1;
		xy[1] = self->Y1;
		xy[2] = self->X2;
		xy[3] = self->Y2;
		rectfill(self, xy, 1);

		switch(rotate)
		{
			case 0: /*   0 */ --self->Y1; --self->Y2; break;
			case 1: /*  90 */ --self->X1; --self->X2; break;
			case 2:	/* 180 */ ++self->Y1; ++self->Y2; break;
			case 3: /* 270 */ ++self->X1; ++self->X2; break;
		}
		skewmask = ((skewmask << 1) & 0xFFFE) | 
			   ((skewmask >> 15) & 1);
		if (skewmask & 1) switch (rotate)
		{
			case 0: /*   0 */ ++self->X1; ++self->X2; break;
			case 1: /*  90 */ --self->Y1; --self->Y2; break;
			case 2: /* 180 */ --self->X1; --self->X2; break;
			case 3: /* 270 */ ++self->Y1; ++self->Y2; break;
		}
	}
	self->LN_MASK = save_mask;
	self->FG_BP_1 = save_bp;
}
