/************************************************************************
 *       Copyright 1999, Caldera Thin Clients, Inc.                     * 
 *       This software is licenced under the GNU Public License.        *
 *       Please see LICENSE.TXT for further information.                * 
 *                                                                      * 
 *                  Historical Copyright                                * 
 *									*
 *									*
 *									*
 *  Copyright (c) 1987, Digital Research, Inc. All Rights Reserved.	*
 *  The Software Code contained in this listing is proprietary to	*
 *  Digital Research, Inc., Monterey, California and is covered by U.S.	*
 *  and other copyright protection.  Unauthorized copying, adaptation,	*
 *  distribution, use or display is prohibited and may be subject to 	*
 *  civil and criminal penalties.  Disclosure to others is prohibited.	*
 *  For the terms and conditions of software code use refer to the 	*
 *  appropriate Digital Research License Agreement.			*
 *									*
 ************************************************************************/

#include "sdsdl_i.h"


static signed text_attr_just(DRIVERDATA *self);

int MONO8XHT(DRIVERDATA *self)
{
/* Dummy: Optimised blit for monospaced characters 8 pixels wide */
	return 0;
}


const Uint8 byte_mask_table[] = { 0xFF, 0x7F, 0x3F, 0x1F, 0x0F, 0x07, 0x03, 0x01, 0x00 };


/****************************************************************
* PRE_BLT_0							*
* Input:	none						*
* Input/Output:							*
*	dx = offset to the next character			*
* 	bp = destination x position				*
* 	si = character width					*
* Output:							*
*	carry flag:  set if character is clipped out [ret]	*
*	char_count:  number of characters to process		*
*	ax = source offset from left edge of form		*
*	di = bitmap address					*
* Preserved:	bx						*
* Trashed:	cx						*
****************************************************************/

typedef struct pre_blt_state
{
	int justify;	/* Nonzero if justified text */
	signed nxtchar;	/* INOUT Offset to the next character */
	signed xdest;	/* INOUT Destination X-coordinate */
	signed wchar;	/* INOUT Character width */
	signed srcoff;	/* OUT source offset from left edge of form */
	Uint8 *addr;	/* OUT bitmap address */

} PREBLTSTATE;


typedef int (*PREBLTFUNC)(DRIVERDATA *self, PREBLTSTATE *state, unsigned *char_count);

typedef int (*POSTBLTFUNC)(DRIVERDATA *self, signed x, signed offset);

static int pre_blt_0  (DRIVERDATA *self, PREBLTSTATE *state, unsigned *char_count);
static int pre_blt_90 (DRIVERDATA *self, PREBLTSTATE *state, unsigned *char_count);
static int pre_blt_180(DRIVERDATA *self, PREBLTSTATE *state, unsigned *char_count);
static int pre_blt_270(DRIVERDATA *self, PREBLTSTATE *state, unsigned *char_count);

static int post_blt_0  (DRIVERDATA *self, signed x, signed offset);
static int post_blt_90 (DRIVERDATA *self, signed x, signed offset);
static int post_blt_180(DRIVERDATA *self, signed x, signed offset);
static int post_blt_270(DRIVERDATA *self, signed x, signed offset);


static PREBLTFUNC pre_blt_table[] = 
{
	pre_blt_0, pre_blt_90, pre_blt_180, pre_blt_270
};

static POSTBLTFUNC post_blt_table[] = 
{
	post_blt_0, post_blt_90, post_blt_180, post_blt_270
};

/*
line_table	dw	clr_skew_next_line_0
		dw	clr_skew_next_line_90
		dw	clr_skew_next_line_180
		dw	clr_skew_next_line_270

pixel_table	dw	clr_skew_next_pixel_0
		dw	clr_skew_next_pixel_90
		dw	clr_skew_next_pixel_180
		dw	clr_skew_next_pixel_270
		public	lt_hoff
		public	rt_hoff
		public	off_htbl
		public	seg_htbl
		public	poff_tbl
		public	text_ix
		public	JUST_NEG
		public	buff_ptr
		public	JUST_DEL_CHAR_REM
		public	JUST_DEL_CHAR
		public	line_table
		public	tempdely
		public	JUST_DEL_WORD
		public	JUST_DEL_WORD_REM
		public	foffindex
		public	port_sel
		public	pixel_table
		public	text_mode
		public	source_seg
		public	JUST_DEL_SIGN

		public	byte_mask_table
		cseg
		extrn	chk_fnt:near
;;;		extrn	set_fnthdr:near
		extrn	text_move:near
		extrn	text_scale:near
		extrn	text_italic:near
		extrn	text_bold:near
		extrn	text_grey:near
		extrn	text_rotate:near
		extrn	text_blit:near
		extrn	clr_skew_next_line_0:near
		extrn	clr_skew_next_line_90:near
		extrn	clr_skew_next_line_180:near
		extrn	clr_skew_next_line_270:near
		extrn	clr_skew_next_pixel_0:near
		extrn	clr_skew_next_pixel_90:near
		extrn	clr_skew_next_pixel_180:near
		extrn	clr_skew_next_pixel_270:near

		extrn	double_table:word
		extrn	d_gtext:near
		extrn	ACT_SIZ:near
		extrn	RECTFILL:near
		extrn	CONCAT:near
if (num_planes gt 1) and (not segment_access)
		extrn	EGA_KLUG:near
endif

next_dest	rw	1
next_source	rw	1
rot_height	rw	1
SPECIAL_CS	rw	1
extrn	txtblt_rep_rr:near
extrn	txtblt_tran_rr:near
extrn	txtblt_xor_rr_s:near
extrn	txtblt_itran_rr:near
extrn	txtblt_rep_rl:near
extrn	txtblt_tran_rl:near
extrn	txtblt_xor_rl_s:near
extrn	txtblt_itran_rl:near
;
;text blit branch table
;	format three bits (rot dir ) (mode msb) (mode lsb) 

txt_blt_mode_table	dw	offset txt_blt_rep
			dw	offset txt_blt_tran
			dw	offset txt_blt_xor
			dw	offset txt_blt_tran

txt_blt_branch_table	dw	offset	txtblt_rep_rr	;replace right source
			dw	offset	txtblt_tran_rr ;trans right source
			dw	offset	txtblt_tran_rr ;xor right source
			dw	offset	txtblt_itran_rr ;invtrans right source

			dw	offset	txtblt_rep_rl ;replace left source
			dw	offset	txtblt_tran_rl ;trans left source
			dw	offset	txtblt_tran_rl ;xor left source
			dw	offset	txtblt_itran_rl ;invtrans left source
			public	rot_height
			public	next_dest
			public	txt_blt_branch_table
			public	next_source
			public	SPECIAL_CS
*/
unsigned set_fnthdr(DRIVERDATA *self, unsigned ch)
{
	if (ch >= self->font_inf.first_ade && ch <= self->font_inf.last_ade)
	{
		return ch;
	}
	if (!chk_ade(self, ch)) ch = ' ';

	self->font_off = self->font_inf.dat_table + 
			 self->foffindex * self->font_inf.form_width;
	return ch;
}

/****************************************************************
 * TEXT_BLT							*
 ****************************************************************/


void text_blt(DRIVERDATA *self, unsigned count, unsigned *text, int justified)
{
/*
 * If skewing and the writing mode is replace or reverse transparent, force the
 * mode to transparent.  In the latter case, substitute background color for
 * the foreground.
 */
	self->foffindex = 0;
	self->text_mode = self->WRT_MODE;
	if (self->SPECIAL & SKEW)  	/* Outputting italics? */
	{
		switch (self->WRT_MODE)
		{
			case 0:	self->text_mode = 1; break; /* Replace mode */
			case 3: self->text_mode = 1;	/* Reverse trans */
				map_colour(self, 0, &self->sdlc_text, 
						&self->text_colour);
				break;
		}
	}
	text_blt_main(self, count, text, self->text_mode, justified); 
}


static void paint_character8(DRIVERDATA *self, unsigned srcx, Uint8 *dest, 
				unsigned width, unsigned mode)
{
	Uint32 x, y, xb;
	Uint8 mask, bt;
	Uint32 bg;

	map_colour(self, 0, NULL, &bg);

	for (y = 0; y < self->tempdely; y++)
	{
		Uint8 *buf = (Uint8 *)(&dest[y * self->common->surface->pitch]);
		mask = 0x80 >> (srcx & 7);
		xb = (srcx / 8) + y * self->font_inf.form_width;
		bt = self->font_off[xb];
		for (x = 0; x < width; x++)
		{
			if (bt & mask)
			{
				switch (mode)
				{
					case 0:
					case 1:	buf[x] = self->text_colour;
						break;
					case 2: buf[x] ^= 0xFFFF; break;
				}
			}
			else switch (mode)
			{
				case 0: buf[x] = bg; break;
				case 3: buf[x] = self->text_colour; break;
			}
			mask = mask >> 1;
			if (!mask)
			{
				mask = 0x80;
				bt = self->font_off[++xb];
			}
		}
	}
}


static void paint_character16(DRIVERDATA *self, unsigned srcx, Uint8 *dest, 
				unsigned width, unsigned mode)
{
	Uint32 x, y, xb;
	Uint8 mask, bt;
	Uint32 bg;

	map_colour(self, 0, NULL, &bg);

	for (y = 0; y < self->tempdely; y++)
	{
		Uint16 *buf = (Uint16 *)(&dest[y * self->common->surface->pitch]);
		mask = 0x80 >> (srcx & 7);
		xb = (srcx / 8) + y * self->font_inf.form_width;
		bt = self->font_off[xb];
		for (x = 0; x < width; x++)
		{
			if (bt & mask)
			{
				switch (mode)
				{
					case 0:
					case 1:	buf[x] = self->text_colour;
						break;
					case 2: buf[x] ^= 0xFFFF; break;
				}
			}
			else switch (mode)
			{
				case 0: buf[x] = bg; break;
				case 3: buf[x] = self->text_colour; break;
			}
			mask = mask >> 1;
			if (!mask)
			{
				mask = 0x80;
				bt = self->font_off[++xb];
			}
		}
	}
}


static void paint_character24(DRIVERDATA *self, unsigned srcx, Uint8 *dest, 
				unsigned width, unsigned mode)
{
	Uint32 x, y, xb;
	Uint8 mask, bt;
	Uint32 bg;
	Uint8 tc[3], bgc[3];

	map_colour(self, 0, NULL, &bg);

	tc[0] = self->text_colour & 0xFF;
	tc[1] = (self->text_colour >> 8) & 0xFF;
	tc[2] = (self->text_colour >> 16) & 0xFF;
	bgc[0] = bg & 0xFF;
	bgc[1] = (bg >> 8) & 0xFF;
	bgc[2] = (bg >> 16) & 0xFF;
	for (y = 0; y < self->tempdely; y++)
	{
		Uint8 *buf = (Uint8 *)(&dest[y * self->common->surface->pitch]);
		mask = 0x80 >> (srcx & 7);
		xb = (srcx / 8) + y * self->font_inf.form_width;
		bt = self->font_off[xb];
		for (x = 0; x < width; x++)
		{
			if (bt & mask)
			{
				switch (mode)
				{
					case 0:
					case 1:	buf[x * 3    ] = tc[0];
						buf[x * 3 + 1] = tc[1];
						buf[x * 3 + 2] = tc[2];
						break;
					case 2: buf[x * 3    ] ^= 0xFF;
						buf[x * 3 + 1] ^= 0xFF;
						buf[x * 3 + 2] ^= 0xFF;
						break;
				}
			}
			else switch (mode)
			{
				case 0: buf[x * 3    ] = bgc[0];
					buf[x * 3 + 1] = bgc[1]; 
					buf[x * 3 + 2] = bgc[2]; 
					break;
				case 3:	buf[x * 3    ] = tc[0];
					buf[x * 3 + 1] = tc[1];
					buf[x * 3 + 2] = tc[2];
					break;
			}
			mask = mask >> 1;
			if (!mask)
			{
				mask = 0x80;
				bt = self->font_off[++xb];
			}
		}
	}
}



static void paint_character32(DRIVERDATA *self, unsigned srcx, Uint8 *dest, 
				unsigned width, unsigned mode)
{
	Uint32 x, y, xb;
	Uint8 mask, bt;
	Uint32 bg;

	map_colour(self, 0, NULL, &bg);

	for (y = 0; y < self->tempdely; y++)
	{
		Uint32 *buf = (Uint32 *)(&dest[y * self->common->surface->pitch]);
		mask = 0x80 >> (srcx & 7);
		xb = (srcx / 8) + y * self->font_inf.form_width;
		bt = self->font_off[xb];
		for (x = 0; x < width; x++)
		{
			if (bt & mask)
			{
				switch (mode)
				{
					case 0:
					case 1:	buf[x] = self->text_colour;
						break;
					case 2: buf[x] ^= 0xFFFFFFFF; break;
				}
			}
			else switch (mode)
			{
				case 0: buf[x] = bg; break;
				case 3: buf[x] = self->text_colour; break;
			}
			mask = mask >> 1;
			if (!mask)
			{
				mask = 0x80;
				bt = self->font_off[++xb];
			}
		}
	}
}



void paint_character(DRIVERDATA *self, unsigned srcx, Uint8 *dest, 
				unsigned width, unsigned mode)
{
	switch (self->common->surface->format->BytesPerPixel)
	{
		case 1: paint_character8 (self, srcx, dest, width, mode); break;
		case 2: paint_character16(self, srcx, dest, width, mode); break;
		case 3: paint_character24(self, srcx, dest, width, mode); break;
		case 4: paint_character32(self, srcx, dest, width, mode); break;
	}
}

/* main entry into the text blit */
void text_blt_main(DRIVERDATA *self, unsigned count, unsigned *text, 
			int mode, int justified)
{
	Uint32 ch_destx, di;
	Uint32 ch_index;
	Uint32 ch_srcx = 0;
	Uint32 ch_width;
	Uint32 ch;
	Uint32 src_pitch;
	Uint32 src_size;
	Uint8 *dest;
	signed xmin, xmax, ymin, ymax;

	ymin = self->DESTY;
	ymax = self->DESTY + self->ACTDELY - 1;
	xmin = self->DESTX;
	xmax = self->DESTX;

	/* clip the string in y direction */
/* Handle special effects, rotation, and scaling separately. */
	if (!(self->SPECIAL & ~UNDER))
	{
		Uint32 offset = 0;		/* AX: Top offset */
		Uint32 dely = self->ACTDELY;	/* DX: Char height */
		Uint32 desty = self->DESTY;	/* BP: Character destination Y */
		Uint32 maxy = (desty + dely) - 1; /* BX: Bottom row of character */
		Uint32 mxclip = self->rectClip.y + self->rectClip.h - 1;	
		/* Clipping at bottom? */	
		if ((desty + self->ACTDELY - 1) > mxclip)
		{
			/* Clipped completely out? */
			if (desty > mxclip) return;
			dely = (mxclip - desty) + 1;
			maxy = mxclip;
		}
		/* Clipping at top? */
		if (desty < self->rectClip.y)
		{
			Uint32 newdely;

			desty = self->rectClip.y;
			if (maxy < self->rectClip.y) return;
			newdely = (maxy - self->rectClip.y) + 1;
			self->foffindex = dely - newdely;
			if (self->foffindex)
			{
				offset = self->font_inf.form_width * self->foffindex;	
			}
			dely = newdely;	
		}
		chk_fnt(self);
		self->font_off = self->font_inf.dat_table + offset;
		self->tempdely = dely;
		self->DESTY = desty;

// Should be handled by sd_justified
//		if (justified)	
//		{
//			text += 2;	/* Justified text has 2 integer */
//		}			/* parameters at the start. */
		ch_destx = self->DESTX;
		di = self->rectClip.x;	
		while (count)
		{
/* Find the first character to draw */
			ch = *text++;	/* Next character to draw */
			self->isspace = ch = set_fnthdr(self, ch);

/* Find offset of character */
			ch_index = ch - self->font_inf.first_ade;
/* 11/6/86 DH
   added horizontal offset code here */
			if (self->font_inf.flags & HORZ_OFF)
			{
				self->rt_hoff = self->font_inf.hor_table[ch_index+1];
				ch_destx -=           self->font_inf.hor_table[ch_index];
			}
			ch_srcx  = self->font_inf.off_table[ch_index];	/* Offset into font */
			ch_width = self->font_inf.off_table[ch_index + 1] - ch_srcx; /* Char width */

			if (ch_destx < self->rectClip.x)
			{
				Uint32 bp = ch_destx + ch_width - 1; /* Character right X */
				
				if (bp < self->rectClip.x)
				{
					if (justified)
					{
						if (self->isspace == ' ')
						{
							ch_width += self->JUST_DEL_WORD;
							if (self->JUST_DEL_WORD_REM)
							{
								ch_width += self->JUST_DEL_SIGN;
								--self->JUST_DEL_WORD_REM;
							}
						}
						else	/* isspace == ' ' */
						{
							ch_width += self->JUST_DEL_CHAR;
							if (self->JUST_DEL_CHAR_REM) 
							{
								ch_width += self->JUST_DEL_SIGN;
								--self->JUST_DEL_CHAR_REM;
							}
						} /* end if space */
					}	/* end if justified */
					ch_destx += ch_width;
					if (self->font_inf.flags & HORZ_OFF)
					{
						ch_destx -= self->rt_hoff;
					}
					--count;
					if (!count) return;
					continue;	
				}
				++bp;			/* New char width */
				ch_srcx += (ch_width - bp);	/* New char X */
				ch_destx = self->rectClip.x;
				ch_width = bp;		/* New char width */
			}	/* end if left clip */
			break;
		}	/* end while */
		/* OK, we've dealt with clipping at the left edge */


/*
; ch_srcx = first char source startx [DX]
; bx = first char dest startx
; ch_width = first char width [AX]
; cx = char count
	push	si
	push	cx
	mov	si, ax			;si = DELX = CHAR WIDTH
	mov	bp, bx 			;bp = DESTX
	push	dx			
;calculate dest address
;dest pointer = es:di
*/
		if (ch_destx >= self->common->window.w) return; /* Off screen */
		dest = concat(self, ch_destx, self->DESTY);
		if (!dest)
		{
			return;
		}
		do
		{
			Uint32 rclip;
			signed just_extra = 0;

			rclip = self->rectClip.x + self->rectClip.w - 1;
/* 
 this is the main char out loop for text
 si = CHAR WIDTH
 bp = destx
 ax = sourcex
 calculate the right mask right mask = dh
*/
			if (ch_destx > rclip) break;	/* Beyond clip area */
			if (ch_destx + ch_width > rclip)
			{
				count = 1;	/* Only 1 character will fit */
				ch_width = (rclip - ch_destx) + 1;
			}
			if (justified)
			{
				if (self->isspace == ' ')
				{
					just_extra =  self->JUST_DEL_WORD;
					if (self->JUST_DEL_WORD_REM)
					{
						just_extra += self->JUST_DEL_SIGN;
						--self->JUST_DEL_WORD_REM;
					}
				}
				else
				{
					just_extra =  self->JUST_DEL_CHAR;
					if (self->JUST_DEL_CHAR_REM)
					{
						just_extra += self->JUST_DEL_SIGN;
						--self->JUST_DEL_CHAR_REM;
					}
				}
			}
			ch_destx += ch_width;
			ch_destx += just_extra;

			xmax = ch_destx;
			paint_character(self, ch_srcx, dest, ch_width, mode);
			--count;
			if (!count) break;
			dest += ((ch_width + just_extra) * self->common->surface->format->BytesPerPixel);
			ch = *text++;	/* Next character to draw */
			self->isspace = ch = set_fnthdr(self, ch);

/* Find offset of character */
			ch_index = ch - self->font_inf.first_ade;
			if (self->font_inf.flags & HORZ_OFF)
			{
				self->rt_hoff = self->font_inf.hor_table[ch_index+1];
				ch_destx -=     self->font_inf.hor_table[ch_index];
			}
			ch_srcx  = self->font_inf.off_table[ch_index];	/* Offset into font */
			ch_width = self->font_inf.off_table[ch_index + 1] - ch_srcx; /* Char width */
		} while (count);

	}
	else
	{
		PREBLTFUNC pre_blt;
		POSTBLTFUNC post_blt;
		PREBLTSTATE state;


/****************************************************************
* TEXT_BLT_SPECIAL						*
*****************************************************************/
		chk_fnt(self);
/* Save the address of the appropriate pre- and post-blt routines (a function
 * of rotation). */
		pre_blt  = pre_blt_table[(self->SPECIAL & ROTATE) >> 6];
		post_blt = post_blt_table[(self->SPECIAL & ROTATE) >> 6];
/* Perform preliminary clipping against the non-baseline axis of the text. */

		self->foffindex = 0;
		if (!(self->SPECIAL & ROTODD))
		{
			signed height = self->ACTDELY;
			signed desty = self->DESTY;
			signed clipy = self->rectClip.y + self->rectClip.h - 1;

			if ((desty + height - 1) > clipy)	/* Clipping at the bottom */
			{
				if (desty > clipy) return; 	/* Clipped completely out */

				height = (clipy - desty) + 1;	/* New cell height */
			}
/* At least part of the cell is above the bottom of the clipping rectangle.
 * Check against the top of the clipping rectangle. */
			if (desty < self->rectClip.y)
			{
				/* Clipped completely out? */
				if (desty + height - 1 < self->rectClip.y) return;

				/* No. */
				self->foffindex = self->rectClip.y - desty;
				height -= self->foffindex;
				desty = self->rectClip.y;
			}
			self->y_position = desty;
			self->tempdely = height;
			ch_destx = self->DESTX;	/* Cell left X */
		}
		else
		{
/* Clip in the x direction.  First, compare the cell against the clipping
 * window right edge. */
			signed char_w = self->ACTDELY;
			signed leftx  = self->DESTX;
			signed clipx = self->rectClip.x + self->rectClip.w - 1;

			signed rightx = (leftx + char_w - 1); /* Right edge of cell */

			self->form_offset = 0;

			if (leftx + char_w - 1 > clipx)	/* Right edge clipped */
			{
				/* Clipped completely out? */
				if (leftx > clipx) return;
				rightx = clipx;	/* New cell right edge */
				char_w = (clipx - leftx);	/* New cell width */
			}
/* At least part of the cell is left of the right edge  of the clipping
 * rectangle.  Check against the left edge of the clipping rectangle. */
			clipx = self->rectClip.x;
			if (leftx < clipx)
			{
				leftx = clipx;		/* New left edge */
				if (rightx < clipx) return;	/* Clipped out */
					
				self->form_offset = char_w - rightx;
				char_w = rightx;
			}
			self->form_width = char_w;
			self->y_position = self->DESTY;
			ch_destx = self->DESTX;	/* Cell left X */
		}
/*
; Prepare for the character output loop.
spec_clip_done:

		mov	ax, graph_plane
		mov	es, ax			; init the segment register

		mov	cx, CONTRL + 6		; cx = number of characters
		mov	char_count, cx		; save the character count
		mov	si, offset INTIN	; ds:si -> string
		cmp	CONTRL, 11		; if "justified text" was
		jnz	spec_char_loop		;   requested, bump pointer
		add	si, 4			;   by two words
*/
/* Top of the character output loop.  Get the character, the offset
 * into the font form, and the width of the character in the font. */

		xmin = xmax = ch_destx;
		ymin = ymax = self->y_position;
		switch ((self->SPECIAL & ROTATE) >> 5)
		{
			case 0:	
			case 2: ymax += self->ACTDELY; break;
			case 1:	
			case 3: xmax += self->ACTDELY; break;
		}
		while (count)
		{
/* 11/6/86 horizontal offset enhancement */
			self->lt_hoff = self->rt_hoff = 0;
			ch = *text++;	/* Get a character */
			ch = self->isspace = set_fnthdr(self, ch);
			ch_index = ch - self->font_inf.first_ade;

/* 11/6/86 horizontal offset enhancement */
			if (self->font_inf.flags & HORZ_OFF)
			{
				self->lt_hoff = self->font_inf.hor_table[ch_index];
				self->rt_hoff = self->font_inf.hor_table[ch_index + 1];
			}
			ch_srcx  = self->font_inf.off_table[ch_index];
			ch_width = self->font_inf.off_table[ch_index + 1] - ch_srcx;

			if (!ch_width)
			{
				--count;
				continue;
			}
/*		mov	si, ax			; si = character width        ch_width
		mov	ax, dx			; ax = offset into font form  ch_srcx
*/
/* Point the destination to the temporary buffer. */
			self->buff_ptr = self->txbuf1;

/* If doubling, adjust the character buffer form width. */

			di = ch_width;
			if (!(self->SPECIAL & SCALE))
			{
				di += self->CHAR_DEL;	
			}
			src_pitch = (di + 7) / 8;
/* Move the character from the font form to the temporary buffer. */
			src_size = text_move (self, self->buff_ptr, src_pitch, ch_width, ch_srcx); 

			/* dx -> src_size */
/* Apply scaling, if necessary. */

			if (self->SPECIAL & SCALE)
			{
				text_scale(self, self->buff_ptr, &src_pitch, &ch_width, 
					   src_size);
			}
/*
 * The status of the registers when calling the special effects routines
 * is as follows:
 *	bl = character buffer width, in bytes
 *	cx = height of the character cell
 *	dx = character width, in pixels
 *	es:di -> temporary character buffer
 */
			state.nxtchar = ch_width; /* INOUT Offset to the next character */

/* Apply skewing, if necessary. */

			if (self->SPECIAL & SKEW)
			{
				text_italic(self, self->buff_ptr, self->ACTDELY, src_pitch,
						state.nxtchar);
				ch_width += self->L_OFF;
				ch_width += self->R_OFF;
			}
/* Apply bolding, if necessary.  Bump the character width only if the font is
 * not monospaced. */

			if (self->SPECIAL & THICKEN)
			{
				text_bold(self, self->buff_ptr, self->ACTDELY, src_pitch);
				ch_width += self->WEIGHT;
/*
 * [JCE] SDPSC9 fix. Moving the line above to before the monospace check fixes 
 * a bug whereby upside-down bold monospaced text doesn't print; though even
 * after the fix has been applied, upside-down bold monospaced text comes 
 * out wider than right-way-up bold monospaced text.
 *
 * Test BASIC2 program:
 *
 *	CLS
 *	FOR e = 0 TO 7
 *	PRINT
 *	PRINT EFFECTS(e);e;".    ";ANGLE(180);"abcd"
 *	PRINT
 *	PRINT
 *	NEXT
 */
				if (!self->MONO_STATUS)
				{
					state.nxtchar += self->WEIGHT;
				}
			}
/* Apply lightening, if necessary. */

			if (self->SPECIAL & LIGHT)
			{
				text_grey(self, self->buff_ptr, self->ACTDELY, src_pitch);
			}
/* Rotate, if necessary. */
			if (self->SPECIAL & ROTATE)
			{
				text_rotate(self, self->buff_ptr, self->ACTDELY, &src_pitch,
					state.nxtchar);

			}
/* Special effects and attributes have been bound. */

			state.justify = justified;
			state.xdest = ch_destx; /* INOUT Destination X-coordinate */
			state.wchar = ch_width; /* INOUT Character width */

/* Perform the necessary pre-blt operations, based on the rotation angle.
 * Output the character and perform the necessary post-blt operations. */

			if (!(*pre_blt)(self, &state, &count))
			{
//	fprintf(stderr, "xdest=%d wchar=%d\n", state.xdest, state.wchar);
				text_blit(self, state.addr, src_pitch, 
						state.wchar, state.srcoff,
						state.xdest, mode);
				if (justified) 
				{
					ch_width += text_attr_just(self); 
					state.nxtchar += text_attr_just(self); 
				}
				if (state.xdest < xmin) xmin = state.xdest;
				if (state.xdest + state.wchar > xmax) xmax = state.xdest + state.wchar;
				if (self->y_position < ymin) ymin = self->y_position;
				if (self->y_position + self->tempdely > ymax) ymax = self->y_position + self->tempdely;
				ch_destx = (*post_blt)(self, state.xdest, state.nxtchar);
				if (ch_destx < xmin) xmin = ch_destx;
				if (ch_destx > xmax) xmax = ch_destx;
				if (self->y_position < ymin) ymin = self->y_position;
				if (self->y_position + self->tempdely > ymax) ymax = self->y_position + self->tempdely;
			}
/* Done with the character.  Prepare for the next one. */
			--count;
		}	/* End while (count) */
		/* GEM had the following code here, but nothing then looked at ch_destx.

		if (self->SPECIAL & SKEW)
		{
			ch_destx += self->R_OFF;	
		} 
		*/

	} /* End if special */

/* All of the characters which can be output have been written.  All that is
  left is to underline them, if requested. */
	if (self->SPECIAL & UNDER)
	{
		const Uint16  *s_patptr = self->patptr;
		unsigned s_patmsk = self->patmsk;
		unsigned s_nxtpat = self->NEXT_PAT;
		Uint32   s_fg_bp  = self->FG_BP_1;
		unsigned s_fill_style = self->fill_style;
		signed xy[4];

		self->NEXT_PAT = 0;
		if (self->SPECIAL & LIGHT)
		{
			self->fill_style = 2;
			self->patmsk = dithrmsk;
			self->patptr = &dither[12];
		}
		else
		{
			self->fill_style = 1;
			self->patmsk = 0;
			self->patptr = &solid;
		}	
		self->FG_BP_1 = self->text_colour;
		xy[0] = self->X1;
		xy[1] = self->Y1;
		xy[2] = self->X2 - 1;
		xy[3] = self->Y2;
		rectfill(self, xy, 1);
		self->fill_style = s_fill_style;
		self->FG_BP_1  = s_fg_bp;
		self->NEXT_PAT = s_nxtpat;
		self->patmsk = s_patmsk;
		self->patptr = s_patptr;
	}
	update_rect(self, xmin, xmax, ymin, ymax);
}


/* Subroutine to adjust for justified text. */
static signed text_attr_just(DRIVERDATA *self)
{
	signed spacing = 0;

	if (self->isspace == ' ')
	{
		spacing += self->JUST_DEL_WORD;
		if (self->JUST_DEL_WORD_REM)
		{
			spacing += self->JUST_DEL_SIGN;
			--self->JUST_DEL_WORD_REM;
		}
	}
	else
	{
		spacing += self->JUST_DEL_CHAR;
		if (self->JUST_DEL_CHAR_REM)
		{
			spacing += self->JUST_DEL_SIGN;
			--self->JUST_DEL_CHAR_REM;
		}
	}
	return spacing;
}

/****************************************************************
* PRE_BLT_0							*
* Input:	none						*
* Input/Output:							*
*	dx = offset to the next character			*
* 	bp = destination x position				*
* 	si = character width					*
* Output:							*
*	carry flag:  set if character is clipped out		*
*	char_count:  number of characters to process		*
*	ax = source offset from left edge of form		*
*	di = bitmap address					*
* Preserved:	bx						*
* Trashed:	cx						*
****************************************************************/

static int pre_blt_0  (DRIVERDATA *self, PREBLTSTATE *state, unsigned *char_count)
{
	signed cell_r;

/* Find out if clipping is necessary on the left side. */
	state->xdest -= self->lt_hoff;	
	state->srcoff = 0;				/* Source offset from left */
	cell_r = (state->xdest + state->wchar) - 1;	/* Rightmost pixel */

	if (state->xdest < self->rectClip.x)	/* Not clipped at left */
	{
		state->srcoff = cell_r - self->rectClip.x;	/* Source offset */
		if (state->srcoff < 0)
		{	/* Completely outside clipping rectangle */
/* No character will be drawn (clipped).  Bump to the next position and return
 * a status indicating clipped. */
			if (state->justify) 
			{
				state->nxtchar += text_attr_just(self);
				state->xdest += state->nxtchar;
			}
			return 1;
		}	
		/* Clipping is necessary on the left side.  Do it. */
		state->wchar = state->srcoff + 1;
		state->srcoff = self->rectClip.x - state->xdest;
		state->nxtchar -= state->srcoff;
		state->xdest = self->rectClip.x;	

	}
/* Find out if clipping is necessary on the right side. */
	if (cell_r >= self->rectClip.x + self->rectClip.w - 1)
	{
		/* Completely out at the right-hand side? */
		if (state->xdest > self->rectClip.x + self->rectClip.w - 1)
		{
			*char_count = 1;
			return 1;
		}
/* Clipping is necessary on the right side.  Do it. */
		state->wchar = self->rectClip.x + self->rectClip.w - state->xdest;
	}
/* Get the new bitmap address and return a status indicating that a character
 * should be drawn. */
	state->addr = concat(self, state->xdest, self->y_position);
	if (!state->addr) 
	{
		assert(0);
		return 1;
	}
	return 0;
}

/****************************************************************
* PRE_BLT_90							*
* Input:							*
*	tempdely:  form height					*
* Input/Output:							*
*	y_position:  screen y coordinate			*
*	dx = offset to the next character			*
* Output:							*
*	carry flag:  set if character is clipped out		*
*	char_count:  number of characters to process		*
*	ax = source offset from left edge of form		*
* 	si = character form "width"				*
*	di = bitmap address					*
* Preserved:	bx, bp						*
* Trashed:	cx						*
*****************************************************************/
static int pre_blt_90 (DRIVERDATA *self, PREBLTSTATE *state, unsigned *char_count)
{
	signed cell_b;
	signed ymx_clip;

/* Subtract the form height from the y position to obtain the screen y of the
 * character's upper left corner.  The offset to the next character must be
 * similarly patched to return the y position for the next pass through. */
	self->y_position += self->lt_hoff;
	self->y_position -= self->tempdely;
	state->nxtchar -= self->tempdely;
/* Find out if clipping is necessary on the top. */
	state->srcoff = 0;
	cell_b = self->y_position + state->wchar - 1;	/* bottom edge of cell */
	if (self->y_position < self->rectClip.y)
	{
		if (cell_b < self->rectClip.y)
		{
			*char_count = 1;	/* All clipped out */
			return 1;
		}
		state->wchar = (cell_b - self->rectClip.y + 1);   /* New height */
		state->srcoff = (self->rectClip.y - self->y_position); 
						/* New left offset */
		state->nxtchar += state->srcoff;		/* New distance to next */
		self->y_position = self->rectClip.y;	     /* New Y-coord */
	}
/* Find out if clipping is necessary on the bottom. */
	ymx_clip = self->rectClip.y + self->rectClip.h - 1;
	if (cell_b > ymx_clip)
	{
		if (self->y_position > ymx_clip)
		{
			if (state->justify)
			{
				self->y_position -= text_attr_just(self);
			}
			return 1;
		}
		state->wchar = (ymx_clip - self->y_position) + 1;
	}

/* Get the new bitmap address and return a status indicating that a character
 * should be drawn. */
	
	self->foffindex = state->srcoff;	
	self->tempdely = state->wchar;
	state->srcoff = self->form_offset;
	state->wchar = self->form_width;
/* Get the new bitmap address and return a status indicating that a character
 * should be drawn. */
	state->addr = concat(self, state->xdest, self->y_position);
	if (!state->addr) 
	{
		assert(0);
		return 1;
	}
	return 0;
}




/****************************************************************
* PRE_BLT_180							*
* Input:	none						*
* Input/Output:							*
* 	bp = destination x position				*
* 	si = character form width				*
* Output:							*
*	carry flag:  set if character is clipped out		*
*	char_count:  number of characters to process		*
*	ax = source offset from left edge of form		*
*	dx = offset to the next character			*
*	di = bitmap address					*
* Preserved:	bx						*
* Trashed:	cx						*
****************************************************************/


static int pre_blt_180(DRIVERDATA *self, PREBLTSTATE *state, unsigned *char_count)
{
	signed cell_r;
	signed cell_w;

/*  Subtract the form width from the destination x position so that the
 * character will be placed correctly.  The distance to the next character
 * must also be updated. */
	state->xdest += self->lt_hoff;
	state->xdest -= state->wchar;

	state->nxtchar = - (self->R_OFF + self->L_OFF);

/* Find out if clipping is necessary on the left side. */

	state->srcoff = 0;
	cell_r = state->xdest + state->wchar - 1; /* di = right edge of cell */

	if (state->xdest < self->rectClip.x)
	{
		if (cell_r < self->rectClip.x)
		{
			*char_count = 1;
			return 1;
		}
/* Clipping is necessary on the left side.  Do it. */
		state->wchar = (cell_r - self->rectClip.x) + 1;
		state->srcoff = self->rectClip.x - state->xdest;
		state->nxtchar += state->srcoff;
		state->xdest = self->rectClip.x;
	}
/* Find out if clipping is necessary on the right side. */
	cell_w = self->rectClip.x + self->rectClip.w - 1;
	if (cell_r > cell_w)
	{
		if (state->xdest > cell_w)
		{
			if (state->justify) 
			{
				state->nxtchar += text_attr_just(self);
				state->xdest += state->nxtchar;
			}
			return 1;
		}
		state->wchar = (cell_w - state->xdest + 1);
	}
/* Get the new bitmap address and return a status indicating that a character
 * should be drawn. */
	state->addr = concat(self, state->xdest, self->y_position);
	if (!state->addr) 
	{
		assert(0);
		return 1;
	}
	return 0;
}


/****************************************************************
* PRE_BLT_270							*
****************************************************************/
static int pre_blt_270(DRIVERDATA *self, PREBLTSTATE *state, unsigned *char_count)
{
	signed cell_b;
	signed cell_h;

/* Get the character y position and the "height" (screen sense) of the
 * character. */
	self->y_position -= self->lt_hoff;
	state->wchar = self->tempdely;	/* character form 'height' */
	
/* Find out if clipping is necessary on the top. */
	cell_b = self->y_position + self->tempdely - 1; /* bottom edge of cell */
	if (self->y_position < self->rectClip.y)	/* Clipping at top */
	{
		if (cell_b < self->rectClip.y)	/* Clipped completely out */
		{
			if (state->justify) 
			{
				self->y_position += text_attr_just(self);
			}
			return 1;
		}
/* Clipping is necessary on the top.  Do it. */
		self->tempdely = (cell_b - self->rectClip.y + 1); /* visible height */
		state->srcoff = self->rectClip.y - self->y_position;
		state->nxtchar -= state->srcoff;
		self->y_position = self->rectClip.y;
	
	}
/* Find out if clipping is necessary on the bottom. */
	cell_h = self->rectClip.y + self->rectClip.h - 1;
	if (cell_b > cell_h)	/* Clipped at the bottom? */
	{
		if (self->y_position > cell_h)
		{
			*char_count = 1;	/* Clipped completely out */
			return 1;
		}
/* Clipping is necessary on the bottom.  Do it. */
		self->tempdely = (cell_h - self->y_position) + 1;
	}	
		
/* Get the new bitmap address and return a status indicating that a character
 * should be drawn. */
	state->srcoff = self->form_offset;
	state->wchar = self->form_width;
	state->addr = concat(self, state->xdest, self->y_position);
	if (!state->addr) 
	{
		assert(0);
		return 1;
	}
	return 0;
}

/****************************************************************
* POST_BLT_0							*
* Input:							*
*	dx = offset to the next character			*
* Input/Output:							*
* 	bp = destination x position				*
* Output:	none						*
* Preserved:	ax, bx, cx, dx, si, di				*
* Trashed:	none						*
*****************************************************************/
static int post_blt_0  (DRIVERDATA *self, signed x, signed offset)
{
/* For no rotation, move to the right. */
	x += (offset - self->rt_hoff);
	return x;	
}


/****************************************************************
* POST_BLT_90							*
* Input:							*
*	dx = offset to the next character			*
* Input/Output:							*
*	y_position:  screen y position				*
* Output:	none						*
* Preserved:	ax, bx, cx, dx, si, di, bp			*
* Trashed:	none						*
*****************************************************************/

static int post_blt_90 (DRIVERDATA *self, signed x, signed offset)
{
/* For 90 degree rotation, move up. */
	self->y_position -= (offset - self->rt_hoff);
	
	return x;
}


/****************************************************************
* POST_BLT_180							*
* Input:							*
*	dx = offset to the next character			*
* Input/Output:							*
* 	bp = destination x position				*
* Output:	none						*
* Preserved:	ax, bx, cx, dx, si, di				*
* Trashed:	none						*
*****************************************************************/

static int post_blt_180(DRIVERDATA *self, signed x, signed offset)
{
/* For 180 degree rotation, move to the left. */
	x -= (offset - self->rt_hoff);
	return x;	
}


/****************************************************************
* POST_BLT_270							*
* Input:							*
*	dx = offset to the next character			*
* Input/Output:							*
*	y_position:  screen y position				*
* Output:	none						*
* Preserved:	ax, bx, cx, dx, si, di, bp			*
* Trashed:	none						*
*****************************************************************/

static int post_blt_270 (DRIVERDATA *self, signed x, signed offset)
{
/* For 270 degree rotation, move down. */
	self->y_position += (offset - self->rt_hoff);
	return x;
}	


