/*
SDSDL: GEM video driver for SDL
Copyright 2012 John Elliott <jce@seasip.demon.co.uk>
Based on SDPSC9.VGA copyright 1987, 1999 Caldera Thin Clients, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "sdsdl_i.h"

/* Contour fill is new in SDSDL; the GEM drivers don't support it. */

/* This code is to manage a FIFO queue of 'seed' points -- points above or 
 * below the current line, marking areas into which the fill can extend */
typedef struct seeds
{
	int count;
	int max;
	int *pts;
} SEEDS;


void addseed(SEEDS *s, int x, int y)
{
	if (s->pts == NULL)
	{
		s->pts = calloc(10, sizeof(int));
		s->count = 0;
		s->max = 5;
		if (!s->pts) return;
	}
	if (s->count >= s->max)
	{
		int *pts2 = realloc(s->pts, 4 * s->max * sizeof(int));
		if (!pts2) return;
		s->pts = pts2;
		s->max *= 2;	
	}
	s->pts[2 * s->count    ] = x;	
	s->pts[2 * s->count + 1] = y;	
	++s->count;
}

int popseed(SEEDS *s, signed *x, signed *y)
{
	if (!s->count) return 0;

	*x = s->pts[0];
	*y = s->pts[1];

	--s->count;

	if (s->count)
	{
		memmove(&s->pts[0], &s->pts[2], sizeof(int) * s->count * 2);
	}
	return 1;
}



static inline int isedge(signed pen, Uint32 pixel, Uint32 match)
{
	/* For positive pen, 'pixel' specifies the edge colour */
	if (pen >= 0)	return (pixel == match);
	/* Otherwise, 'pixel' specifies the colour to replace */
	else		return (pixel != match);
}


static inline int isedge24(signed pen, Uint32 pixel, Uint8 *match)
{
	Uint32 value = match[0] + 256 * match[1] + 65536 * match[2];

	/* For positive pen, 'pixel' specifies the edge colour */
	if (pen >= 0)	return (pixel == value);
	/* Otherwise, 'pixel' specifies the colour to replace */
	else		return (pixel != value);
}


/* Maintain separate fill functions for different colour depths */


static void sd_contourfill8(DRIVERDATA *self, signed x0, signed y0, 
				signed pen, Uint32 pixel, Uint32 bg,
				SDL_Rect *bounds, SDL_Surface *workbuf)
{
	Uint8 *buf  = NULL, *buf2 = NULL, *buf3 = NULL;
	signed x, xmin, xmax;
	signed xy[4];
	SDL_Rect rc;
	SEEDS seeds = { 0 };

	buf = (Uint8 *)concat_x(self, workbuf, x0, y0);

	if (pen < 0) pixel = *buf;
	/* Start with one seed point, the one that is passed in. As each 
	 * row is processed, it is removed from the head of the queue, and
	 * any rows that it gives rise to are added to the tail of the 
	 * queue. */
	addseed(&seeds, x0, y0);
	while (popseed(&seeds, &x0, &y0))
	{
		buf = (Uint8 *)concat_x(self, workbuf, 0, y0);
		if (y0 > 0)	
			buf2 = (Uint8 *)concat_x(self, workbuf, 0, y0 - 1);
		else	buf2 = NULL;
	
		if (y0 < workbuf->h - 1)
			buf3 = (Uint8 *)concat_x(self, workbuf, 0, y0 + 1);
		else	buf3 = NULL;
	
		xmin = xmax = x0;
		/* Find the left and right hand ends of this line */
		for (x = x0 - 1; x >= 0; x--)
		{
			if (isedge(pen, pixel, buf[x])) break;
			xmin = x; 
		}
		for (x = x0 + 1; x < workbuf->w; x++)
		{
			if (isedge(pen, pixel, buf[x])) break;
			xmax = x; 
		}
/* Inflate the bounding rectangle if necessary */
		if (xmin < bounds->x)
		{
			bounds->w += (bounds->x - xmin);
			bounds->x = xmin;
		}
		if (bounds->w < (xmax - bounds->x + 1))
		{
			bounds->w = xmax - bounds->x + 1; 
		}
		if (y0 < bounds->y)
		{
			bounds->h += (bounds->y - y0);
			bounds->y = y0;
		}
		if (bounds->h < (y0 - bounds->y + 1))
		{
			bounds->h = (y0 - bounds->y + 1);
		}
		
//	box_hline(self, y0, xmin, (xmax - xmin + 1), pattern, bg);

/* Fill the area we've just done with the border colour, so that 
 * future scans see it as border and don't try to re-enter it */
		rc.x = xmin;
		rc.y = y0;
		rc.w = (xmax - xmin + 1);
		rc.h = 1;
		SDL_FillRect(workbuf, &rc, (pen < 0) ? ~pixel : pixel);

/* Now scan the areas above and below */
		if (buf2)
		{
			for (x = xmin; x <= xmax; x++)
			{
				if (isedge(pen, pixel, buf2[x])) continue;
				
				addseed(&seeds, x, y0 - 1);
				/* And skip over adjacent seed points */	
				while (!isedge(pen, pixel, buf2[x]) && x <= xmax) ++x;
			}
		}
		if (buf3)
		{
			for (x = xmin; x <= xmax; x++)
			{
				if (isedge(pen, pixel, buf3[x])) continue;
				
				addseed(&seeds, x, y0 + 1);
				/* And skip over adjacent seed points */	
				while (!isedge(pen, pixel, buf3[x]) && x <= xmax) ++x;
			}
		}
/* Now fill the current line segment with its proper colour & pattern */
		xy[0] = xmin;
		xy[1] = y0;
		xy[2] = xmax;
		xy[3] = y0;
		rectfill(self, xy, 0);
	}
	if (seeds.pts) 
	{
		free(seeds.pts);
		seeds.pts = NULL;
		seeds.count = seeds.max = 0;
	}
}



static void sd_contourfill16(DRIVERDATA *self, signed x0, signed y0, 
				signed pen, Uint32 pixel, Uint32 bg,
				SDL_Rect *bounds, SDL_Surface *workbuf)
{
	Uint16 *buf  = NULL, *buf2 = NULL, *buf3 = NULL;
	signed x, xmin, xmax;
	signed xy[4];
	SDL_Rect rc;
	SEEDS seeds = { 0 };

	buf = (Uint16 *)concat_x(self, workbuf, x0, y0);

	if (pen < 0) pixel = *buf;
	/* Start with one seed point, the one that is passed in. As each 
	 * row is processed, it is removed from the head of the queue, and
	 * any rows that it gives rise to are added to the tail of the 
	 * queue. */
	addseed(&seeds, x0, y0);
	while (popseed(&seeds, &x0, &y0))
	{
		buf = (Uint16 *)concat_x(self, workbuf, 0, y0);
		if (y0 > 0)	
			buf2 = (Uint16 *)concat_x(self, workbuf, 0, y0 - 1);
		else	buf2 = NULL;
	
		if (y0 < workbuf->h - 1)
			buf3 = (Uint16 *)concat_x(self, workbuf, 0, y0 + 1);
		else	buf3 = NULL;
	
		xmin = xmax = x0;
		/* Find the left and right hand ends of this line */
		for (x = x0 - 1; x >= 0; x--)
		{
			if (isedge(pen, pixel, buf[x])) break;
			xmin = x; 
		}
		for (x = x0 + 1; x < workbuf->w; x++)
		{
			if (isedge(pen, pixel, buf[x])) break;
			xmax = x; 
		}
/* Inflate the bounding rectangle if necessary */
		if (xmin < bounds->x)
		{
			bounds->w += (bounds->x - xmin);
			bounds->x = xmin;
		}
		if (bounds->w < (xmax - bounds->x + 1))
		{
			bounds->w = xmax - bounds->x + 1; 
		}
		if (y0 < bounds->y)
		{
			bounds->h += (bounds->y - y0);
			bounds->y = y0;
		}
		if (bounds->h < (y0 - bounds->y + 1))
		{
			bounds->h = (y0 - bounds->y + 1);
		}
		
//	box_hline(self, y0, xmin, (xmax - xmin + 1), pattern, bg);

/* Fill the area we've just done with the border colour, so that 
 * future scans see it as border and don't try to re-enter it */
		rc.x = xmin;
		rc.y = y0;
		rc.w = (xmax - xmin + 1);
		rc.h = 1;
		SDL_FillRect(workbuf, &rc, (pen < 0) ? ~pixel : pixel);

/* Now scan the areas above and below */
		if (buf2)
		{
			for (x = xmin; x <= xmax; x++)
			{
				if (isedge(pen, pixel, buf2[x])) continue;
				
				addseed(&seeds, x, y0 - 1);
				/* And skip over adjacent seed points */	
				while (!isedge(pen, pixel, buf2[x]) && x <= xmax) ++x;
			}
		}
		if (buf3)
		{
			for (x = xmin; x <= xmax; x++)
			{
				if (isedge(pen, pixel, buf3[x])) continue;
				
				addseed(&seeds, x, y0 + 1);
				/* And skip over adjacent seed points */	
				while (!isedge(pen, pixel, buf3[x]) && x <= xmax) ++x;
			}
		}
/* Now fill the current line segment with its proper colour & pattern */
		xy[0] = xmin;
		xy[1] = y0;
		xy[2] = xmax;
		xy[3] = y0;
		rectfill(self, xy, 0);
	}
	if (seeds.pts) 
	{
		free(seeds.pts);
		seeds.pts = NULL;
		seeds.count = seeds.max = 0;
	}
}


static void sd_contourfill24(DRIVERDATA *self, signed x0, signed y0, 
				signed pen, Uint32 pixel, Uint32 bg,
				SDL_Rect *bounds, SDL_Surface *workbuf)
{
	Uint8 *buf  = NULL, *buf2 = NULL, *buf3 = NULL;
	signed x, xmin, xmax;
	signed xy[4];
	SDL_Rect rc;
	SEEDS seeds = { 0 };

	buf = (Uint8 *)concat_x(self, workbuf, x0, y0);

	if (pen < 0) pixel = buf[0] + 256 * buf[1] + 65536 * buf[2];
	/* Start with one seed point, the one that is passed in. As each 
	 * row is processed, it is removed from the head of the queue, and
	 * any rows that it gives rise to are added to the tail of the 
	 * queue. */
	addseed(&seeds, x0, y0);
	while (popseed(&seeds, &x0, &y0))
	{
		buf = (Uint8 *)concat_x(self, workbuf, 0, y0);
		if (y0 > 0)	
			buf2 = (Uint8 *)concat_x(self, workbuf, 0, y0 - 1);
		else	buf2 = NULL;
	
		if (y0 < workbuf->h - 1)
			buf3 = (Uint8 *)concat_x(self, workbuf, 0, y0 + 1);
		else	buf3 = NULL;
	
		xmin = xmax = x0;
		/* Find the left and right hand ends of this line */
		for (x = x0 - 1; x >= 0; x--)
		{
			if (isedge24(pen, pixel, &buf[x*3])) break;
			xmin = x; 
		}
		for (x = x0 + 1; x < workbuf->w; x++)
		{
			if (isedge24(pen, pixel, &buf[x*3])) break;
			xmax = x; 
		}
/* Inflate the bounding rectangle if necessary */
		if (xmin < bounds->x)
		{
			bounds->w += (bounds->x - xmin);
			bounds->x = xmin;
		}
		if (bounds->w < (xmax - bounds->x + 1))
		{
			bounds->w = xmax - bounds->x + 1; 
		}
		if (y0 < bounds->y)
		{
			bounds->h += (bounds->y - y0);
			bounds->y = y0;
		}
		if (bounds->h < (y0 - bounds->y + 1))
		{
			bounds->h = (y0 - bounds->y + 1);
		}
		
//	box_hline(self, y0, xmin, (xmax - xmin + 1), pattern, bg);

/* Fill the area we've just done with the border colour, so that 
 * future scans see it as border and don't try to re-enter it */
		rc.x = xmin;
		rc.y = y0;
		rc.w = (xmax - xmin + 1);
		rc.h = 1;
		SDL_FillRect(workbuf, &rc, (pen < 0) ? ~pixel : pixel);

/* Now scan the areas above and below */
		if (buf2)
		{
			for (x = xmin; x <= xmax; x++)
			{
				if (isedge24(pen, pixel, &buf2[x*3])) continue;
				
				addseed(&seeds, x, y0 - 1);
				/* And skip over adjacent seed points */	
				while (!isedge24(pen, pixel, &buf2[x*3]) && x <= xmax) ++x;
			}
		}
		if (buf3)
		{
			for (x = xmin; x <= xmax; x++)
			{
				if (isedge24(pen, pixel, &buf3[x*3])) continue;
				
				addseed(&seeds, x, y0 + 1);
				/* And skip over adjacent seed points */	
				while (!isedge24(pen, pixel, &buf3[x*3]) && x <= xmax) ++x;
			}
		}
/* Now fill the current line segment with its proper colour & pattern */
		xy[0] = xmin;
		xy[1] = y0;
		xy[2] = xmax;
		xy[3] = y0;
		rectfill(self, xy, 0);
	}
	if (seeds.pts) 
	{
		free(seeds.pts);	
		seeds.pts = NULL;
		seeds.count = seeds.max = 0;
	}
}




static void sd_contourfill32(DRIVERDATA *self, signed x0, signed y0, 
				signed pen, Uint32 pixel, Uint32 bg,
				SDL_Rect *bounds, SDL_Surface *workbuf)
{
	Uint32 *buf  = NULL, *buf2 = NULL, *buf3 = NULL;
	signed x, xmin, xmax;
	signed xy[4];
	SDL_Rect rc;
	SEEDS seeds = { 0 };

	buf = (Uint32 *)concat_x(self, workbuf, x0, y0);

	if (pen < 0) pixel = *buf;
	/* Start with one seed point, the one that is passed in. As each 
	 * row is processed, it is removed from the head of the queue, and
	 * any rows that it gives rise to are added to the tail of the 
	 * queue. */
	addseed(&seeds, x0, y0);
	while (popseed(&seeds, &x0, &y0))
	{
		buf = (Uint32 *)concat_x(self, workbuf, 0, y0);
		if (y0 > 0)	
			buf2 = (Uint32 *)concat_x(self, workbuf, 0, y0 - 1);
		else	buf2 = NULL;
	
		if (y0 < workbuf->h - 1)
			buf3 = (Uint32 *)concat_x(self, workbuf, 0, y0 + 1);
		else	buf3 = NULL;
	
		xmin = xmax = x0;
		/* Find the left and right hand ends of this line */
		for (x = x0 - 1; x >= 0; x--)
		{
			if (isedge(pen, pixel, buf[x])) break;
			xmin = x; 
		}
		for (x = x0 + 1; x < workbuf->w; x++)
		{
			if (isedge(pen, pixel, buf[x])) break;
			xmax = x; 
		}
/* Inflate the bounding rectangle if necessary */
		if (xmin < bounds->x)
		{
			bounds->w += (bounds->x - xmin);
			bounds->x = xmin;
		}
		if (bounds->w < (xmax - bounds->x + 1))
		{
			bounds->w = xmax - bounds->x + 1; 
		}
		if (y0 < bounds->y)
		{
			bounds->h += (bounds->y - y0);
			bounds->y = y0;
		}
		if (bounds->h < (y0 - bounds->y + 1))
		{
			bounds->h = (y0 - bounds->y + 1);
		}
		
//	box_hline(self, y0, xmin, (xmax - xmin + 1), pattern, bg);

/* Fill the area we've just done with the border colour, so that 
 * future scans see it as border and don't try to re-enter it */
		rc.x = xmin;
		rc.y = y0;
		rc.w = (xmax - xmin + 1);
		rc.h = 1;
		SDL_FillRect(workbuf, &rc, (pen < 0) ? ~pixel : pixel);

/* Now scan the areas above and below */
		if (buf2)
		{
			for (x = xmin; x <= xmax; x++)
			{
				if (isedge(pen, pixel, buf2[x])) continue;
				
				addseed(&seeds, x, y0 - 1);
				/* And skip over adjacent seed points */	
				while (!isedge(pen, pixel, buf2[x]) && x <= xmax) ++x;
			}
		}
		if (buf3)
		{
			for (x = xmin; x <= xmax; x++)
			{
				if (isedge(pen, pixel, buf3[x])) continue;
				
				addseed(&seeds, x, y0 + 1);
				/* And skip over adjacent seed points */	
				while (!isedge(pen, pixel, buf3[x]) && x <= xmax) ++x;
			}
		}
/* Now fill the current line segment with its proper colour & pattern */
		xy[0] = xmin;
		xy[1] = y0;
		xy[2] = xmax;
		xy[3] = y0;
		rectfill(self, xy, 0);
	}
	if (seeds.pts) 
	{
		free(seeds.pts);
		seeds.pts = NULL;
		seeds.count = seeds.max = 0;
	}
}




void sd_contourfill(unsigned handle, signed x, signed y, signed pen)
{
	Uint32 pixel = 0, bg = 0;
	SDL_Rect bounds, wbrc;
	SDL_Surface *workbuf;
        DRIVERDATA *self = lookup_handle(handle);
        if (!self) return;

/* If the initial point is outside the clipping rectangle, nothing happens */
	if (x < self->rectClip.x || x >= self->rectClip.x + self->rectClip.w) 
		return;
	if (y < self->rectClip.y || y >= self->rectClip.y + self->rectClip.h) 
		return;

/* Allocate a copy of the screen, for the drawing code to scribble on. It will
 * fill as it goes with solid colour, to ensure that the fill algorithm 
 * does not attempt to revisit already-filled areas. The real screen will 
 * fill the same areas, but with the currently selected fill style and index. */
        workbuf = SDL_AllocSurface(SDL_SWSURFACE,
                self->common->window.w,
                self->common->window.h,
                self->common->surface->format->BitsPerPixel,
                self->common->surface->format->Rmask,
                self->common->surface->format->Gmask,
                self->common->surface->format->Bmask,
                self->common->surface->format->Amask);

	if (!workbuf) return; 

	if (self->common->surface->format->BytesPerPixel == 1)
	{
		SDL_SetColors(workbuf,
			self->common->surface->format->palette->colors, 0,
			self->common->surface->format->palette->ncolors);

	}

	wbrc = self->common->window;
	wbrc.x = 0;
	wbrc.y = 0;
	if (SDL_BlitSurface(self->common->surface, &self->common->window, 
			workbuf, &wbrc))
	{
		fprintf(stderr, "Contour fill failed: %s\n", SDL_GetError());
		return;
	}

	map_colour(self, 0, NULL, &bg);
	if (pen >= 0)
	{
		if (pen >= self->dev_tab[13])
		{
			pen = 1;
		}
		map_colour(self, pen, NULL, &pixel);
	}
	bounds.x = x;
	bounds.y = y;
	bounds.w = 1;
	bounds.h = 1;
        if (SDL_MUSTLOCK(self->common->surface))
                SDL_LockSurface(self->common->surface);

	self->FG_BP_1 = self->fill_colour;
	self->LSTLIN = 0;
        switch (self->common->surface->format->BytesPerPixel)
        {
		case 1: sd_contourfill8(self, x, y, pen, pixel, bg, &bounds, workbuf); break;
		case 2: sd_contourfill16(self, x, y, pen, pixel, bg, &bounds, workbuf); break;
		case 3: sd_contourfill24(self, x, y, pen, pixel, bg, &bounds, workbuf); break;
		case 4: sd_contourfill32(self, x, y, pen, pixel, bg, &bounds, workbuf); break;
	}
        if (SDL_MUSTLOCK(self->common->surface))
                SDL_UnlockSurface(self->common->surface);
// 
///	SDL_BlitSurface(workbuf, &wbrc, 
///			self->common->surface, &self->common->window);

	bounds.x += self->common->window.x;
	bounds.y += self->common->window.y;
	SDL_UpdateRects(self->common->surface, 1, &bounds);
	SDL_FreeSurface(workbuf);
}
