/*
SDSDL: GEM video driver for SDL
Copyright 2012 John Elliott <jce@seasip.demon.co.uk>
Based on SDPSC9.VGA copyright 1987, 1999 Caldera Thin Clients, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/* This file handles colour management */

#include "sdsdl_i.h"

/* Map the first 256 colour numbers to well-known colours, using the same 
 * mapping as SD256 under FreeGEM. */

static Uint8 fixedmap[768] = 
{
/* The first 16 match the VGA driver */
	0xFF, 0xFF, 0xFF,	/* 0 = white */
	0x00, 0x00, 0x00,	/* 1 = black */
	0xFF, 0x00, 0x00,	/* 2 = red */
	0x00, 0xFF, 0x00,	/* 3 = green */
	0x00, 0x00, 0xFF,	/* 4 = blue */
	0x00, 0xFF, 0xFF,	/* 5 = cyan */
	0xFF, 0xFF, 0x00,	/* 6 = yellow */
	0xFF, 0x00, 0xFF,	/* 7 = magenta */
	0xC0, 0xC0, 0xC0,	/* 8 = light grey */
	0x80, 0x80, 0x80,	/* 9 = dark grey */
	0xA8, 0x00, 0x00,	/* A = red */
	0x00, 0xA8, 0x00,	/* B = green */
	0x00, 0x00, 0xA8,	/* C = blue */
	0x00, 0xA8, 0xA8,	/* D = cyan */
	0xA8, 0xA8, 0x00,	/* E = yellow */
	0xA8, 0x00, 0xA8,	/* F = magenta */
/* The next 24 are a grey scale. */
	0x0A, 0x0A, 0x0A,	/* 10 = 4% */
	0x14, 0x14, 0x14,	/* 11 = 8% */
	0x1F, 0x1F, 0x1F,	/* 12 = 12% */
	0x29, 0x29, 0x29,	/* 13 = 16% */
	0x33, 0x33, 0x33,	/* 14 = 20% */
	0x3D, 0x3D, 0x3D,	/* 15 = 24% */
	0x47, 0x47, 0x47,	/* 16 = 28% */	
	0x52, 0x52, 0x52,	/* 17 = 32% */
	0x5C, 0x5C, 0x5C,	/* 18 = 36% */
	0x66, 0x66, 0x66,	/* 19 = 40% */
	0x70, 0x70, 0x70,	/* 1A = 44% */
	0x7A, 0x7A, 0x7A,	/* 1B = 48% */
	0x85, 0x85, 0x85,	/* 1C = 52% */
	0x8F, 0x8F, 0x8F,	/* 1D = 56% */
	0x99, 0x99, 0x99,	/* 1E = 60% */
	0xA3, 0xA3, 0xA3,	/* 1F = 64% */
	0xAD, 0xAD, 0xAD,	/* 20 = 68% */	
	0xB8, 0xB8, 0xB8,	/* 21 = 72% */
	0xC2, 0xC2, 0xC2,	/* 22 = 76% */
	0xCC, 0xCC, 0xCC,	/* 23 = 80% */
	0xD6, 0xD6, 0xD6,	/* 24 = 84% */
	0xE0, 0xE0, 0xE0,	/* 25 = 88% */
	0xEB, 0xEB, 0xEB,	/* 26 = 92% */
	0xF5, 0xF5, 0xF5,	/* 27 = 96% */
/* The next 216 are a 6x6x6 colour cube. */
	0x00, 0x00, 0x00,
	0x33, 0x00, 0x00,
	0x66, 0x00, 0x00,
	0x99, 0x00, 0x00,
	0xCC, 0x00, 0x00,
	0xFF, 0x00, 0x00,
	0x00, 0x33, 0x00,
	0x33, 0x33, 0x00,
	0x66, 0x33, 0x00,
	0x99, 0x33, 0x00,
	0xCC, 0x33, 0x00,
	0xFF, 0x33, 0x00,
	0x00, 0x66, 0x00,
	0x33, 0x66, 0x00,
	0x66, 0x66, 0x00,
	0x99, 0x66, 0x00,
	0xCC, 0x66, 0x00,
	0xFF, 0x66, 0x00,
	0x00, 0x99, 0x00,
	0x33, 0x99, 0x00,
	0x66, 0x99, 0x00,
	0x99, 0x99, 0x00,
	0xCC, 0x99, 0x00,
	0xFF, 0x99, 0x00,
	0x00, 0xCC, 0x00,
	0x33, 0xCC, 0x00,
	0x66, 0xCC, 0x00,
	0x99, 0xCC, 0x00,
	0xCC, 0xCC, 0x00,
	0xFF, 0xCC, 0x00,
	0x00, 0xFF, 0x00,
	0x33, 0xFF, 0x00,
	0x66, 0xFF, 0x00,
	0x99, 0xFF, 0x00,
	0xCC, 0xFF, 0x00,
	0xFF, 0xFF, 0x00,

	0x00, 0x00, 0x33,
	0x33, 0x00, 0x33,
	0x66, 0x00, 0x33,
	0x99, 0x00, 0x33,
	0xCC, 0x00, 0x33,
	0xFF, 0x00, 0x33,
	0x00, 0x33, 0x33,
	0x33, 0x33, 0x33,
	0x66, 0x33, 0x33,
	0x99, 0x33, 0x33,
	0xCC, 0x33, 0x33,
	0xFF, 0x33, 0x33,
	0x00, 0x66, 0x33,
	0x33, 0x66, 0x33,
	0x66, 0x66, 0x33,
	0x99, 0x66, 0x33,
	0xCC, 0x66, 0x33,
	0xFF, 0x66, 0x33,
	0x00, 0x99, 0x33,
	0x33, 0x99, 0x33,
	0x66, 0x99, 0x33,
	0x99, 0x99, 0x33,
	0xCC, 0x99, 0x33,
	0xFF, 0x99, 0x33,
	0x00, 0xCC, 0x33,
	0x33, 0xCC, 0x33,
	0x66, 0xCC, 0x33,
	0x99, 0xCC, 0x33,
	0xCC, 0xCC, 0x33,
	0xFF, 0xCC, 0x33,
	0x00, 0xFF, 0x33,
	0x33, 0xFF, 0x33,
	0x66, 0xFF, 0x33,
	0x99, 0xFF, 0x33,
	0xCC, 0xFF, 0x33,
	0xFF, 0xFF, 0x33,
	0x00, 0x00, 0x66,
	0x33, 0x00, 0x66,
	0x66, 0x00, 0x66,
	0x99, 0x00, 0x66,
	0xCC, 0x00, 0x66,
	0xFF, 0x00, 0x66,
	0x00, 0x33, 0x66,
	0x33, 0x33, 0x66,
	0x66, 0x33, 0x66,
	0x99, 0x33, 0x66,
	0xCC, 0x33, 0x66,
	0xFF, 0x33, 0x66,
	0x00, 0x66, 0x66,
	0x33, 0x66, 0x66,
	0x66, 0x66, 0x66,
	0x99, 0x66, 0x66,
	0xCC, 0x66, 0x66,
	0xFF, 0x66, 0x66,
	0x00, 0x99, 0x66,
	0x33, 0x99, 0x66,
	0x66, 0x99, 0x66,
	0x99, 0x99, 0x66,
	0xCC, 0x99, 0x66,
	0xFF, 0x99, 0x66,
	0x00, 0xCC, 0x66,
	0x33, 0xCC, 0x66,
	0x66, 0xCC, 0x66,
	0x99, 0xCC, 0x66,
	0xCC, 0xCC, 0x66,
	0xFF, 0xCC, 0x66,
	0x00, 0xFF, 0x66,
	0x33, 0xFF, 0x66,
	0x66, 0xFF, 0x66,
	0x99, 0xFF, 0x66,
	0xCC, 0xFF, 0x66,
	0xFF, 0xFF, 0x66,
	0x00, 0x00, 0x99,
	0x33, 0x00, 0x99,
	0x66, 0x00, 0x99,
	0x99, 0x00, 0x99,
	0xCC, 0x00, 0x99,
	0xFF, 0x00, 0x99,
	0x00, 0x33, 0x99,
	0x33, 0x33, 0x99,
	0x66, 0x33, 0x99,
	0x99, 0x33, 0x99,
	0xCC, 0x33, 0x99,
	0xFF, 0x33, 0x99,
	0x00, 0x66, 0x99,
	0x33, 0x66, 0x99,
	0x66, 0x66, 0x99,
	0x99, 0x66, 0x99,
	0xCC, 0x66, 0x99,
	0xFF, 0x66, 0x99,
	0x00, 0x99, 0x99,
	0x33, 0x99, 0x99,
	0x66, 0x99, 0x99,
	0x99, 0x99, 0x99,
	0xCC, 0x99, 0x99,
	0xFF, 0x99, 0x99,
	0x00, 0xCC, 0x99,
	0x33, 0xCC, 0x99,
	0x66, 0xCC, 0x99,
	0x99, 0xCC, 0x99,
	0xCC, 0xCC, 0x99,
	0xFF, 0xCC, 0x99,
	0x00, 0xFF, 0x99,
	0x33, 0xFF, 0x99,
	0x66, 0xFF, 0x99,
	0x99, 0xFF, 0x99,
	0xCC, 0xFF, 0x99,
	0xFF, 0xFF, 0x99,
	0x00, 0x00, 0xCC,
	0x33, 0x00, 0xCC,
	0x66, 0x00, 0xCC,
	0x99, 0x00, 0xCC,
	0xCC, 0x00, 0xCC,
	0xFF, 0x00, 0xCC,
	0x00, 0x33, 0xCC,
	0x33, 0x33, 0xCC,
	0x66, 0x33, 0xCC,
	0x99, 0x33, 0xCC,
	0xCC, 0x33, 0xCC,
	0xFF, 0x33, 0xCC,
	0x00, 0x66, 0xCC,
	0x33, 0x66, 0xCC,
	0x66, 0x66, 0xCC,
	0x99, 0x66, 0xCC,
	0xCC, 0x66, 0xCC,
	0xFF, 0x66, 0xCC,
	0x00, 0x99, 0xCC,
	0x33, 0x99, 0xCC,
	0x66, 0x99, 0xCC,
	0x99, 0x99, 0xCC,
	0xCC, 0x99, 0xCC,
	0xFF, 0x99, 0xCC,
	0x00, 0xCC, 0xCC,
	0x33, 0xCC, 0xCC,
	0x66, 0xCC, 0xCC,
	0x99, 0xCC, 0xCC,
	0xCC, 0xCC, 0xCC,
	0xFF, 0xCC, 0xCC,
	0x00, 0xFF, 0xCC,
	0x33, 0xFF, 0xCC,
	0x66, 0xFF, 0xCC,
	0x99, 0xFF, 0xCC,
	0xCC, 0xFF, 0xCC,
	0xFF, 0xFF, 0xCC,
	0x00, 0x00, 0xFF,
	0x33, 0x00, 0xFF,
	0x66, 0x00, 0xFF,
	0x99, 0x00, 0xFF,
	0xCC, 0x00, 0xFF,
	0xFF, 0x00, 0xFF,
	0x00, 0x33, 0xFF,
	0x33, 0x33, 0xFF,
	0x66, 0x33, 0xFF,
	0x99, 0x33, 0xFF,
	0xCC, 0x33, 0xFF,
	0xFF, 0x33, 0xFF,
	0x00, 0x66, 0xFF,
	0x33, 0x66, 0xFF,
	0x66, 0x66, 0xFF,
	0x99, 0x66, 0xFF,
	0xCC, 0x66, 0xFF,
	0xFF, 0x66, 0xFF,
	0x00, 0x99, 0xFF,
	0x33, 0x99, 0xFF,
	0x66, 0x99, 0xFF,
	0x99, 0x99, 0xFF,
	0xCC, 0x99, 0xFF,
	0xFF, 0x99, 0xFF,
	0x00, 0xCC, 0xFF,
	0x33, 0xCC, 0xFF,
	0x66, 0xCC, 0xFF,
	0x99, 0xCC, 0xFF,
	0xCC, 0xCC, 0xFF,
	0xFF, 0xCC, 0xFF,
	0x00, 0xFF, 0xFF,
	0x33, 0xFF, 0xFF,
	0x66, 0xFF, 0xFF,
	0x99, 0xFF, 0xFF,
	0xCC, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF,
};


/* For 256-colour modes, the palette needs to be reordered so that XORing 
 * 0xFF with a pixel inverts it. */

static Uint8 pal256[768] = 
{
/* Firstly: 8 bright colours */
	0xFF, 0xFF, 0xFF,	/* 0 = white */
	0xFF, 0x00, 0x00,	/* 1 = red */
	0x00, 0xFF, 0x00,	/* 2 = green */
	0xFF, 0xFF, 0x00,	/* 3 = yellow */
	0x00, 0x00, 0xFF,	/* 4 = blue */
	0xFF, 0x00, 0xFF,	/* 5 = magenta */
	0x00, 0xFF, 0xFF,	/* 6 = cyan */
	0xC0, 0xC0, 0xC0,	/* 7 = light grey */
/* Then 12 greys */
	0x0A, 0x0A, 0x0A,	/* 8 = 4% */
	0x14, 0x14, 0x14,	/* 9 = 8% */
	0x1F, 0x1F, 0x1F,	/* A = 12% */
	0x29, 0x29, 0x29,	/* B = 16% */
	0x33, 0x33, 0x33,	/* C = 20% */
	0x3D, 0x3D, 0x3D,	/* D = 24% */
	0x47, 0x47, 0x47,	/* E = 28% */	
	0x52, 0x52, 0x52,	/* F = 32% */
	0x5C, 0x5C, 0x5C,	/* 10 = 36% */
	0x66, 0x66, 0x66,	/* 11 = 40% */
	0x70, 0x70, 0x70,	/* 12 = 44% */
	0x7A, 0x7A, 0x7A,	/* 13 = 48% */
/* Then the 6x6x6 colour cube */
	0x00, 0x00, 0x00,
	0x33, 0x00, 0x00,
	0x66, 0x00, 0x00,
	0x99, 0x00, 0x00,
	0xCC, 0x00, 0x00,
	0xFF, 0x00, 0x00,
	0x00, 0x33, 0x00,
	0x33, 0x33, 0x00,
	0x66, 0x33, 0x00,
	0x99, 0x33, 0x00,
	0xCC, 0x33, 0x00,
	0xFF, 0x33, 0x00,
	0x00, 0x66, 0x00,
	0x33, 0x66, 0x00,
	0x66, 0x66, 0x00,
	0x99, 0x66, 0x00,
	0xCC, 0x66, 0x00,
	0xFF, 0x66, 0x00,
	0x00, 0x99, 0x00,
	0x33, 0x99, 0x00,
	0x66, 0x99, 0x00,
	0x99, 0x99, 0x00,
	0xCC, 0x99, 0x00,
	0xFF, 0x99, 0x00,
	0x00, 0xCC, 0x00,
	0x33, 0xCC, 0x00,
	0x66, 0xCC, 0x00,
	0x99, 0xCC, 0x00,
	0xCC, 0xCC, 0x00,
	0xFF, 0xCC, 0x00,
	0x00, 0xFF, 0x00,
	0x33, 0xFF, 0x00,
	0x66, 0xFF, 0x00,
	0x99, 0xFF, 0x00,
	0xCC, 0xFF, 0x00,
	0xFF, 0xFF, 0x00,

	0x00, 0x00, 0x33,
	0x33, 0x00, 0x33,
	0x66, 0x00, 0x33,
	0x99, 0x00, 0x33,
	0xCC, 0x00, 0x33,
	0xFF, 0x00, 0x33,
	0x00, 0x33, 0x33,
	0x33, 0x33, 0x33,
	0x66, 0x33, 0x33,
	0x99, 0x33, 0x33,
	0xCC, 0x33, 0x33,
	0xFF, 0x33, 0x33,
	0x00, 0x66, 0x33,
	0x33, 0x66, 0x33,
	0x66, 0x66, 0x33,
	0x99, 0x66, 0x33,
	0xCC, 0x66, 0x33,
	0xFF, 0x66, 0x33,
	0x00, 0x99, 0x33,
	0x33, 0x99, 0x33,
	0x66, 0x99, 0x33,
	0x99, 0x99, 0x33,
	0xCC, 0x99, 0x33,
	0xFF, 0x99, 0x33,
	0x00, 0xCC, 0x33,
	0x33, 0xCC, 0x33,
	0x66, 0xCC, 0x33,
	0x99, 0xCC, 0x33,
	0xCC, 0xCC, 0x33,
	0xFF, 0xCC, 0x33,
	0x00, 0xFF, 0x33,
	0x33, 0xFF, 0x33,
	0x66, 0xFF, 0x33,
	0x99, 0xFF, 0x33,
	0xCC, 0xFF, 0x33,
	0xFF, 0xFF, 0x33,
	0x00, 0x00, 0x66,
	0x33, 0x00, 0x66,
	0x66, 0x00, 0x66,
	0x99, 0x00, 0x66,
	0xCC, 0x00, 0x66,
	0xFF, 0x00, 0x66,
	0x00, 0x33, 0x66,
	0x33, 0x33, 0x66,
	0x66, 0x33, 0x66,
	0x99, 0x33, 0x66,
	0xCC, 0x33, 0x66,
	0xFF, 0x33, 0x66,
	0x00, 0x66, 0x66,
	0x33, 0x66, 0x66,
	0x66, 0x66, 0x66,
	0x99, 0x66, 0x66,
	0xCC, 0x66, 0x66,
	0xFF, 0x66, 0x66,
	0x00, 0x99, 0x66,
	0x33, 0x99, 0x66,
	0x66, 0x99, 0x66,
	0x99, 0x99, 0x66,
	0xCC, 0x99, 0x66,
	0xFF, 0x99, 0x66,
	0x00, 0xCC, 0x66,
	0x33, 0xCC, 0x66,
	0x66, 0xCC, 0x66,
	0x99, 0xCC, 0x66,
	0xCC, 0xCC, 0x66,
	0xFF, 0xCC, 0x66,
	0x00, 0xFF, 0x66,
	0x33, 0xFF, 0x66,
	0x66, 0xFF, 0x66,
	0x99, 0xFF, 0x66,
	0xCC, 0xFF, 0x66,
	0xFF, 0xFF, 0x66,
	0x00, 0x00, 0x99,
	0x33, 0x00, 0x99,
	0x66, 0x00, 0x99,
	0x99, 0x00, 0x99,
	0xCC, 0x00, 0x99,
	0xFF, 0x00, 0x99,
	0x00, 0x33, 0x99,
	0x33, 0x33, 0x99,
	0x66, 0x33, 0x99,
	0x99, 0x33, 0x99,
	0xCC, 0x33, 0x99,
	0xFF, 0x33, 0x99,
	0x00, 0x66, 0x99,
	0x33, 0x66, 0x99,
	0x66, 0x66, 0x99,
	0x99, 0x66, 0x99,
	0xCC, 0x66, 0x99,
	0xFF, 0x66, 0x99,
	0x00, 0x99, 0x99,
	0x33, 0x99, 0x99,
	0x66, 0x99, 0x99,
	0x99, 0x99, 0x99,
	0xCC, 0x99, 0x99,
	0xFF, 0x99, 0x99,
	0x00, 0xCC, 0x99,
	0x33, 0xCC, 0x99,
	0x66, 0xCC, 0x99,
	0x99, 0xCC, 0x99,
	0xCC, 0xCC, 0x99,
	0xFF, 0xCC, 0x99,
	0x00, 0xFF, 0x99,
	0x33, 0xFF, 0x99,
	0x66, 0xFF, 0x99,
	0x99, 0xFF, 0x99,
	0xCC, 0xFF, 0x99,
	0xFF, 0xFF, 0x99,
	0x00, 0x00, 0xCC,
	0x33, 0x00, 0xCC,
	0x66, 0x00, 0xCC,
	0x99, 0x00, 0xCC,
	0xCC, 0x00, 0xCC,
	0xFF, 0x00, 0xCC,
	0x00, 0x33, 0xCC,
	0x33, 0x33, 0xCC,
	0x66, 0x33, 0xCC,
	0x99, 0x33, 0xCC,
	0xCC, 0x33, 0xCC,
	0xFF, 0x33, 0xCC,
	0x00, 0x66, 0xCC,
	0x33, 0x66, 0xCC,
	0x66, 0x66, 0xCC,
	0x99, 0x66, 0xCC,
	0xCC, 0x66, 0xCC,
	0xFF, 0x66, 0xCC,
	0x00, 0x99, 0xCC,
	0x33, 0x99, 0xCC,
	0x66, 0x99, 0xCC,
	0x99, 0x99, 0xCC,
	0xCC, 0x99, 0xCC,
	0xFF, 0x99, 0xCC,
	0x00, 0xCC, 0xCC,
	0x33, 0xCC, 0xCC,
	0x66, 0xCC, 0xCC,
	0x99, 0xCC, 0xCC,
	0xCC, 0xCC, 0xCC,
	0xFF, 0xCC, 0xCC,
	0x00, 0xFF, 0xCC,
	0x33, 0xFF, 0xCC,
	0x66, 0xFF, 0xCC,
	0x99, 0xFF, 0xCC,
	0xCC, 0xFF, 0xCC,
	0xFF, 0xFF, 0xCC,
	0x00, 0x00, 0xFF,
	0x33, 0x00, 0xFF,
	0x66, 0x00, 0xFF,
	0x99, 0x00, 0xFF,
	0xCC, 0x00, 0xFF,
	0xFF, 0x00, 0xFF,
	0x00, 0x33, 0xFF,
	0x33, 0x33, 0xFF,
	0x66, 0x33, 0xFF,
	0x99, 0x33, 0xFF,
	0xCC, 0x33, 0xFF,
	0xFF, 0x33, 0xFF,
	0x00, 0x66, 0xFF,
	0x33, 0x66, 0xFF,
	0x66, 0x66, 0xFF,
	0x99, 0x66, 0xFF,
	0xCC, 0x66, 0xFF,
	0xFF, 0x66, 0xFF,
	0x00, 0x99, 0xFF,
	0x33, 0x99, 0xFF,
	0x66, 0x99, 0xFF,
	0x99, 0x99, 0xFF,
	0xCC, 0x99, 0xFF,
	0xFF, 0x99, 0xFF,
	0x00, 0xCC, 0xFF,
	0x33, 0xCC, 0xFF,
	0x66, 0xCC, 0xFF,
	0x99, 0xCC, 0xFF,
	0xCC, 0xCC, 0xFF,
	0xFF, 0xCC, 0xFF,
	0x00, 0xFF, 0xFF,
	0x33, 0xFF, 0xFF,
	0x66, 0xFF, 0xFF,
	0x99, 0xFF, 0xFF,
	0xCC, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF,
/* Now the other twelve greys */
	0x85, 0x85, 0x85,	/* 1C = 52% */
	0x8F, 0x8F, 0x8F,	/* 1D = 56% */
	0x99, 0x99, 0x99,	/* 1E = 60% */
	0xA3, 0xA3, 0xA3,	/* 1F = 64% */
	0xAD, 0xAD, 0xAD,	/* 20 = 68% */	
	0xB8, 0xB8, 0xB8,	/* 21 = 72% */
	0xC2, 0xC2, 0xC2,	/* 22 = 76% */
	0xCC, 0xCC, 0xCC,	/* 23 = 80% */
	0xD6, 0xD6, 0xD6,	/* 24 = 84% */
	0xE0, 0xE0, 0xE0,	/* 25 = 88% */
	0xEB, 0xEB, 0xEB,	/* 26 = 92% */
	0xF5, 0xF5, 0xF5,	/* 27 = 96% */
/* Dark versions of the 8 EGA colours */
	0x80, 0x80, 0x80,	/* F8 = dark grey */
	0xA8, 0x00, 0x00,	/* F9 = dark red */
	0x00, 0xA8, 0x00,	/* FA = dark green */
	0xA8, 0xA8, 0x00,	/* FB = dark yellow */
	0x00, 0x00, 0xA8,	/* FC = dark blue */
	0xA8, 0x00, 0xA8,	/* FD = dark magenta */
	0x00, 0xA8, 0xA8,	/* FE = dark cyan */
	0x00, 0x00, 0x00,	/* FF = black */
};


static Uint8 map256[256] = 
{
		0,			/* White -> 0 */
		0xFF,			/* Black -> 0FFh (from white -> 0) */
		1,			/* Red   -> 1 */
		2,			/* Green -> 2 */
		4,			/* Blue  -> 4 */
		6,			/* Cyan  -> 6 */
		3,			/* Yellow-> 3 */
		5,			/* Magenta->5 */
		7,			/* Grey   ->8 */
		248,			/* DGrey  ->248 */
		249,			/* Dark red */
		250,			/* Dark green */
		252,			/* Dark blue */
		254,			/* Dark cyan */
		251,			/* Dark yellow */
		253,			/* Dark magenta */
/* 
 * The next 24 colours are a grey scale.
 */
		  8,  9, 10, 11,12,  13, 14, 15, 16, 17, 18, 19,
		236,237,238,239,240,241,242,243,244,245,246,247,

/* The last 216 are a 6x6x6 colour cube. */

		20, 21, 22, 23, 24, 25,	/* First block: B=0 */
		26, 27, 28, 29, 30, 31, 
		32, 33, 34, 35, 36, 37, 
		38, 39, 40, 41, 42, 43,
		44, 45, 46, 47, 48, 49,
		50, 51, 52, 53, 54, 55,
	
		56, 57, 58, 59, 60, 61,	/* Second block: B=13 */
		62, 63, 64, 65, 66, 67, 
		68, 69, 70, 71, 72, 73,
		74, 75, 76, 77, 78, 79,
		80, 81, 82, 83, 84, 85,
		86, 87, 88, 89, 90, 91, 

		 92, 93, 94, 95, 96, 97, /* Third block: B=26 */
		 98, 99,100,101,102,103,
		104,105,106,107,108,109,
		110,111,112,113,114,115,
		116,117,118,119,120,121, 
		122,123,124,125,126,127, 

		128,129,130,131,132,133,	/* Fourth block: B=37 */
		134,135,136,137,138,139,
		140,141,142,143,144,145,
		146,147,148,149,150,151, 
		152,153,154,155,156,157, 
		158,159,160,161,162,163,

		164,165,166,167,168,169,	/* Fifth block: B=50 */
		170,171,172,173,174,175,
		176,177,178,179,180,181, 
		182,183,184,185,186,187, 
		188,189,190,191,192,193,
		194,195,196,197,198,199,

		200,201,202,203,204,205,	/* Sixth block: B=63 */
		206,207,208,209,210,211, 
		212,213,214,215,216,217, 
		218,219,220,221,222,223,
		224,225,226,227,228,229,
		230,231,232,233,234,235,
};



unsigned sd_spalette(unsigned handle, unsigned palette)
{
	DRIVERDATA *self = lookup_handle(handle);
	int n;

	/* Selecting any palette has the effect of resetting the colour map
	 * to default values */
	if (self)
	{
		if (self->common->surface->format->BytesPerPixel == 1)
		{
			SDL_Colour clr[256];
			memcpy(self->colour_map, pal256, sizeof(pal256));	
			for (n = 0; n < 768; n++)
			{
				self->req_col[n] = (self->colour_map[n] * 1000) / 255;
			}
		
			for (n = 0; n < 256; n++)
			{	
				clr[n].r = pal256[n*3];
				clr[n].g = pal256[n*3+1];
				clr[n].b = pal256[n*3+2];
			}
			SDL_SetColors(self->common->surface, clr, 0, 256);
		}
		else
		{
			memcpy(self->colour_map, fixedmap, sizeof(fixedmap));	
			for (n = 0; n < 768; n++)
			{
				self->req_col[n] = (self->colour_map[n] * 1000) / 255;			}
		}
	}
	return 0;
}

/* The equivalent of the MAP_COL array in GEM drivers: given a GEM colour 
 * index, retrieve its RGB values (in c) and the bit pattern with which to
 * draw it (in pixel) */
void map_colour(DRIVERDATA *self, unsigned short gem_colour, SDL_Colour *c, Uint32 *pixel)
{
	map_colour_x(self, self->common->surface, gem_colour, c, pixel);
}


/* map_colour_x acts as map_colour, but on an arbitrary SDL surface */
void map_colour_x(DRIVERDATA *self, SDL_Surface *s, unsigned short gem_colour, SDL_Colour *c, Uint32 *pixel)
{
	SDL_Colour tmp;

	if (!c) c = &tmp;

	if (s->format->BytesPerPixel == 1)
	{
		gem_colour = map256[gem_colour & 0xFF];
		c->r = self->colour_map[gem_colour * 3];
		c->g = self->colour_map[gem_colour * 3 + 1];
		c->b = self->colour_map[gem_colour * 3 + 2];
	}
	else if (gem_colour < 256)
	{
		c->r = self->colour_map[gem_colour * 3];
		c->g = self->colour_map[gem_colour * 3 + 1];
		c->b = self->colour_map[gem_colour * 3 + 2];

	}
	else
	{
		/* And all others as 555 truecolour */
		c->b = ((gem_colour         & 0x1F) * 255) / 31;
		c->g = (((gem_colour >>  5) & 0x1F) * 255) / 31;
		c->r = (((gem_colour >> 10) & 0x1F) * 255) / 31;
	}
	*pixel = SDL_MapRGB(s->format, c->r, c->g, c->b);
}

/* Get the R/G/B mapping of a colour. */
unsigned sd_qcolor    (unsigned handle, unsigned index, unsigned flag, 
			unsigned rgb[3])
{
	SDL_Colour tmp;
	Uint32 pixel;

	DRIVERDATA *self = lookup_handle(handle);
	if (index > 0xFFFF || !self) return (unsigned)-1;

	map_colour(self, index, &tmp, &pixel);
	if (flag == 0 && index < 256)
	{
		rgb[0] = self->req_col[index * 3];
		rgb[1] = self->req_col[index * 3 + 1];
		rgb[2] = self->req_col[index * 3 + 2];
	}
	else
	{
		rgb[0] = (1000 * tmp.r) / 255;
		rgb[1] = (1000 * tmp.g) / 255;
		rgb[2] = (1000 * tmp.b) / 255;
	}
	return pixel;	
}

/* Set the R/G/B mapping of a colour. We only allow 256 colours to be 
 * remapped. */
void sd_setcolor(unsigned handle, unsigned index, const unsigned rgb[3])
{
	DRIVERDATA *self = lookup_handle(handle);
	if (index > 0xFF || !self) return;

	if (self->common->surface->format->BytesPerPixel == 1)
	{
		index = map256[index];
	}
	self->req_col[index*3  ] = rgb[0];
	self->req_col[index*3+1] = rgb[1];
	self->req_col[index*3+2] = rgb[2];

	self->colour_map[index*3  ] = (rgb[0] * 255) / 1000;
	self->colour_map[index*3+1] = (rgb[1] * 255) / 1000;
	self->colour_map[index*3+2] = (rgb[2] * 255) / 1000;


	if (self->common->surface->format->BytesPerPixel == 1)
	{
		SDL_Colour clr;
		
		clr.r = self->colour_map[index * 3];	
		clr.g = self->colour_map[index * 3 + 1];	
		clr.b = self->colour_map[index * 3 + 2];	
	
		SDL_SetColors(self->common->surface, &clr, index, 1);
	}

}

/* Given a pixel and a colour, find the corresponding GEM colour index */ 
unsigned unmap_colour(DRIVERDATA *self, SDL_Color *cr, Uint32 pel)
{
	int n;
	SDL_PixelFormat *fmt = self->common->surface->format;

	if (self->common->surface->format->BytesPerPixel == 1)
	{
		for (n = 0; n < 256; n++)
		{
			if (map256[n] == pel) return n;
		}
	}
	else
	{
		/* If RGB values match a 'known' ink, use that */
		for (n = 0; n < 256; n++)
		{
			if ((self->colour_map[n*3  ] >> fmt->Rloss) == 
				(cr->r >> fmt->Rloss) && 
			    (self->colour_map[n*3+1] >> fmt->Gloss) == 
				(cr->g >> fmt->Gloss) && 
			    (self->colour_map[n*3+2] >> fmt->Bloss) == 
				(cr->b >> fmt->Bloss))
			{
				return n;
			}
		}
	}
	/* Otherwise return it in 555 truecolour */
	return (((cr->r >> 3) & 0x1F) << 10) | 
	       (((cr->g >> 3) & 0x1F) << 5)  | 
	        ((cr->b >> 3) & 0x1F);
}


unsigned get_pixel(SDL_Surface *s, int x, int y, SDL_Color *cr)
{
	Uint32 pixel = 0;
	SDL_PixelFormat *fmt;
	Uint8 *buf;

        if (x < 0 || y < 0 || x >= s->w || y >= s->h)
        {
                return 0;
        }
        buf = ((Uint8 *)s->pixels) + y * s->pitch
                   + x * s->format->BytesPerPixel;

	fmt = s->format;
	if (SDL_MUSTLOCK(s)) SDL_LockSurface(s);
	switch (fmt->BytesPerPixel)
	{
		case 1: pixel = buf[0]; 
			if (cr)
			{
				*cr = fmt->palette->colors[pixel];
			}
			break;
		case 2: pixel = ((Uint16 *)buf)[0]; 
			if (cr)
			{
				cr->r = ((pixel & fmt->Rmask) >> fmt->Rshift) << (fmt->Rloss);
				cr->g = ((pixel & fmt->Gmask) >> fmt->Gshift) << (fmt->Gloss);
				cr->b = ((pixel & fmt->Bmask) >> fmt->Bshift) << (fmt->Bloss);
			}
			break;
		case 3: 
		case 4: pixel = ((Uint32 *)buf)[0];
			if (cr)
			{
				cr->r = ((pixel & fmt->Rmask) >> fmt->Rshift) << (fmt->Rloss);
				cr->g = ((pixel & fmt->Gmask) >> fmt->Gshift) << (fmt->Gloss);
				cr->b = ((pixel & fmt->Bmask) >> fmt->Bshift) << (fmt->Bloss);
			}
			break;
	}
	if (SDL_MUSTLOCK(s)) SDL_UnlockSurface(s);

	return pixel;
}



void sd_get_pixel(unsigned handle, signed x, signed y, unsigned *pel, 
	unsigned *colour)
{
	DRIVERDATA *self = lookup_handle(handle);
	Uint32 pixel;
	SDL_Color cr;

	if (!self) return;

	pixel = get_pixel(self->common->surface, 
				x + self->common->window.x, 
				y + self->common->window.y, 
				&cr);

	*pel = pixel;
	*colour = unmap_colour(self, &cr, pixel);
}



