
/*
 *
 * Each GLK window is represented by an entry in a GEM widget tree
 * (this is also the way GEM windows are represented internally within GEM,
 *  but I digress). 
 *
 * Each widget has the INDIRECT flag set. Its spec then points to the 
 * managing window structure. The window structure then contains the
 * ultimate spec - the 16-bit segment:offset address of a USERBLK 
 * structure in low memory. The ub_code in that structure points 
 * (via what Microsoft calls a thunk) to a window drawing
 * subroutine. The ub_parm member points back at the window.
 *
 *                             +-----------------------------+
 *                             |                             |
 *                             v                             |
 * +------------+       +------------+     +-------------+   |
 * | GEM object |------>| GLK window |---->| GEM USERBLK |   |
 * +------------+       +------------+     |             |---+
 *                                         +-------------+
 * 
 * "Pair" windows don't have a USERBLK. They are represented by
 * a GEM IBOX object - an invisible, transparent rectangle:
 *
 * +------------+       +------------+     
 * | GEM object |------>| GLK window | ==== IBOX spec
 * +------------+       +------------+     
 *
 */

/* All the structures have a C++ class derived from them. The C++ class
 * does the "real" work, but the structure is what is passed to C 
 * interfaces. We guarantee that every glk_*_struct is really a 
 * glk_*_class; the class has things like vtables in it, which GEM 
 * and GLK wouldn't like.
 *
 * Note: This means it is essential (in a constructor or whatever) to
 *      set the gw_size member of the base class. That way GEM (or 
 *      more accurately my DJGPP bindings) knows how much space
 *      the object actually occupies.
 *
 */

class glk_pair_window;

#define MAXWINDOW 255	/* GEM allows an object tree to be at most 8 
                         * levels deep. If we allow an object to have at
                         * most 2 descendants, then we allow at most 255
                         * objects (1+2+4+8+16+32+64+128). */

struct glk_window_struct
{
	LONG	gw_gemspec;	/* GEM object spec for the window */
	LONG	gw_size;	/* Size of this structure */

	/* Window area, parent and child are done in the GEM object tree */
	glui32	gw_type;	/* GLK type */
	glui32	gw_rock;	/* GLK user-defined parameter */
};

class glk_window_class : protected glk_window_struct
{
public:
	glk_window_class(glk_pair_window *parent, glui32 method, glui32 size);
	glk_window_class(glk_pair_window *parent, WORD obj, glui32 method, glui32 size);
	virtual ~glk_window_class();

	virtual void close(stream_result_t *result);
	virtual void set_size();
	virtual WORD on_paint(PARMBLK *pb);

	__inline__ glk_window_struct *get_winid() { return (glk_window_struct *)this; }
	__inline__ operator glk_window_struct *() { return get_winid(); }

	glui32 get_rock();
	void set_rock(glui32 rock);
	
	virtual void set_arrangement(glui32 method, glui32 size, glk_window_class *keywin);
	virtual void get_arrangement(glui32 *methodptr, glui32 *sizeptr, glk_window_struct **keywinptr);

	glui32 get_type(void);		/* GLK type */

	glk_pair_window *get_parent();
	glk_window_class *get_sibling();

	static glk_window_class *from_object(WORD object);
	WORD get_object(void);		/* No. of GEM object in glk_tree */
					/* corresponding to this window */

	void set_object(WORD object);	/* Move to another object */
	glk_stream_class *get_stream();
	glk_stream_class *get_echo();
	void set_echo(glk_stream_class *s);

	/* Text output functions. Some windows (such as Blank 
         * and Graphics) won't do anything here. Others (such as Text)
         * will. */
	virtual void   put_char(unsigned char c);
	virtual glui32 get_style();
	virtual void   set_style(glui32 style);
	virtual void   clear();
	virtual void   move_cursor(glui32 x, glui32 y);
	void redraw_window(void);

	glk_window_class *get_key();
	void set_key(glk_window_class *key);

	/* Get window dimensions, in pixels. Be warned: get_area() calls
         * the GEM AES and so MUST NOT be called from on_paint(), because
         * the AES is not re-entrant. */
	VOID get_area(GRECT *r);
	VOID set_area(GRECT *r);

	/* Get GLK-specific size */
	virtual void get_size(glui32 *x, glui32 *y);

	virtual void request_char_event(void);
	virtual void request_line_event(char *buf, glui32 maxlen, glui32 initlen);
	virtual void request_mouse_event(void);
	virtual void cancel_char_event(void);
	virtual void cancel_line_event(event_t *ev);
	virtual void cancel_mouse_event(void);
	virtual WORD get_cursor(GRECT *r, GRECT *area);
	virtual WORD on_key(event_t *ev, WORD key);
	virtual void line_append_key(WORD key);

	virtual glui32 preferred_height(glui32 i);
	virtual glui32 preferred_width(glui32 i);

	void set_disp_rock(gidispatch_rock_t r);
	gidispatch_rock_t get_disp_rock(void);
	void set_buf_rock(gidispatch_rock_t r);
	gidispatch_rock_t get_buf_rock(void);

	glk_window_class *get_child(int nChild);

	void set_reg_index(int idx);
	int  get_reg_index(void);
	glk_pair_window *split_yourself(glui32 method, glui32 size);
	virtual void arrange(void);	/* Arrange children */

	int is_input_wanted(void);
	int is_dirty(void);
	void set_dirty(int d = 1);
private:
	void common_construct(glk_pair_window *parent, WORD object, glui32 method, glui32 size);
	glk_window_class *key;
protected:
	glk_window_class *keyholder;

	virtual void goodbye(glk_window_class *key);

	WORD find_free_gemob(void);
	int reg_index;
	gidispatch_rock_t disp_rock, buf_rock;

	glk_stream_class *wdw_stream;
	glk_stream_class *echo_stream;

#define REQ_CHAR 1
#define REQ_LINE 2
#define REQ_MOUSE 4
#define REQ_MORE 8
#define REQ_KEY    (REQ_CHAR | REQ_LINE | REQ_MORE)

	int dirty;
	int reqd_event;
	char *ev_inbuf;
	glui32 ev_inlen, ev_inmax, ev_inpos;
	GRECT input_rect;

	virtual int check_page(void); /* Check if the window needs to do *
                                       * a [more] thing                  */
};

typedef glk_window_class *winclass_t;





class glk_textgrid_window : public glk_window_class
{
public:
	glk_textgrid_window(glk_pair_window *parent, glui32 method, glui32 size);
	virtual WORD on_paint(PARMBLK *pb);

	/* Get GLK-specific size */
	virtual void get_size(glui32 *x, glui32 *y);
	virtual glui32 preferred_height(glui32 i);
	virtual glui32 preferred_width(glui32 i);
	virtual void   clear();
	virtual void   move_cursor(glui32 x, glui32 y);
	virtual void   put_char(unsigned char c);
	virtual void arrange(void);	/* Arrange this window */
	virtual glui32 get_style();
	virtual void   set_style(glui32 s);

private:
	glui32 cur_style;
	WORD cell_x, cell_y;
	int cur_x, cur_y;
	int buf_w, buf_h;
	char **buffer;
	glui32 **style;
};




class glk_text_window : public glk_window_class
{


public:
	glk_text_window(glk_pair_window *parent, glui32 method, glui32 size);
	virtual void   put_char(unsigned char c);
	virtual WORD   on_paint(PARMBLK *pb);
	virtual void   set_size(void);
	virtual glui32 get_style();
	virtual void   set_style(glui32 style);
	virtual WORD   get_cursor(GRECT *r, GRECT *area);
	void           paint_input_rect(void);
	virtual void   line_append_key(WORD key);
	virtual void   cancel_line_event(event_t *ev);
	virtual glui32 preferred_height(glui32 i);
	virtual glui32 preferred_width(glui32 i);
	virtual void   clear();
	virtual void   arrange(void);	/* Arrange this window */
	/* Get GLK-specific size */
	virtual void   get_size(glui32 *x, glui32 *y);
	virtual void   request_char_event(void);
	virtual void   request_line_event(char *buf, glui32 maxlen, glui32 initlen);
	virtual WORD   on_key(event_t *ev, WORD key);
protected:
	void paint_more_rect(GRECT *w);
	virtual int check_page(void); /* Check if the window needs to do *
                                       * a [more] thing                  */
private:
	void compact(void);
	glk_text_para **parabuf;
	int cur_para;		/* Current output paragraph */
	int last_para;		/* Last paragraph painted */
	int prev_last_para;	/* Previous value of last_para */
	int max_para;		/* No. of available paragraphs */

	int cur_page;
	GRECT winrect;		/* Window rectangle */
};


class glk_graphics_window : public glk_window_class
{
public:
	glk_graphics_window(glk_pair_window *parent, glui32 method, glui32 size);
};

class glk_pair_window : public glk_window_class
{
public:
	glk_pair_window(glk_pair_window *parent, glk_window_class *key, 
                        WORD object, glui32 method, glui32 size);
	virtual void arrange(void);	/* Arrange children */
	void add_child(glk_window_class *w);	/* Add a child window */
	
	virtual void set_arrangement(glui32 method, glui32 size, glk_window_class *keywin);
	virtual void get_arrangement(glui32 *methodptr, glui32 *sizeptr, glk_window_struct **keywinptr);
	virtual void close(stream_result_t *result);
	virtual void goodbye(glk_window_class *key);

private:
	glui32 lost_key;
	glui32 size;
	glui32 splitdir, splitdiv;
};


