
/* Deals with registering objects. Note that most objects get registered
 * twice - once for the library, and once for the dispatch layer */

#include "gemglk.h"

typedef gidispatch_rock_t (*REGFUNC)   (void *obj, glui32 objclass);
typedef void              (*UNREGFUNC) (void *obj, glui32 objclass, gidispatch_rock_t objrock);
typedef gidispatch_rock_t (*RREGFUNC ) (void *array, glui32 len, char *typecode);
typedef void              (*RUNREGFUNC)(void *array, glui32 len, char *typecode, gidispatch_rock_t objrock);

REGFUNC    gl_reg    = NULL;
UNREGFUNC  gl_unreg  = NULL;
RREGFUNC   gl_rreg   = NULL;
RUNREGFUNC gl_runreg = NULL;

//
// Functions required by the GLK dispatch layer
//

void gidispatch_set_object_registry(gidispatch_rock_t (*regi)(void *obj, glui32 objclass), 
                                               void (*unregi)(void *obj, glui32 objclass, gidispatch_rock_t objrock))
{
	gl_reg   = regi;
	gl_unreg = unregi;	
}

gidispatch_rock_t gidispatch_get_objrock(void *obj, glui32 objclass)
{
	gidispatch_rock_t dummy;

	switch(objclass)
	{
		case gidisp_Class_Window: 
			return ((glk_window_class *)obj)->get_disp_rock();
		case gidisp_Class_Stream:
			return ((glk_stream_class *)obj)->get_disp_rock();
		case gidisp_Class_Fileref:
			return ((glk_fileref_class *)obj)->get_disp_rock();
		default:
			glk_assert(FALSE);
	}
	return dummy;
}


void gidispatch_set_retained_registry(RREGFUNC regi, RUNREGFUNC unregi)
{
	gl_rreg   = regi;
	gl_runreg = unregi;
}


//
// Functions required by us
//


void register_window(glk_window_class *w)
{
	if (gl_reg) w->set_disp_rock( (*gl_reg)(w->get_winid(), gidisp_Class_Window) );
	
	w->set_reg_index(window_registry.add(w));
}

void unregister_window(glk_window_class *w)
{
	if (gl_unreg) (*gl_unreg)(w->get_winid(), gidisp_Class_Window, w->get_disp_rock());

	window_registry.remove(w->get_reg_index());
}


void register_stream(glk_stream_class *s)
{
	if (gl_reg) s->set_disp_rock( (*gl_reg)((strid_t)s, gidisp_Class_Stream) );
	
	s->set_reg_index(stream_registry.add(s));
}

void unregister_stream(glk_stream_class *s)
{
	if (gl_unreg) (*gl_unreg)((strid_t)s, gidisp_Class_Stream, s->get_disp_rock());

	stream_registry.remove(s->get_reg_index());
}


void register_fileref(glk_fileref_class *f)
{
	if (gl_reg) f->set_disp_rock( (*gl_reg)((frefid_t)f, gidisp_Class_Fileref) );
	
	f->set_reg_index(fileref_registry.add(f));

}

void unregister_fileref(glk_fileref_class *f)
{
	if (gl_unreg) (*gl_unreg)((frefid_t)f, gidisp_Class_Fileref, f->get_disp_rock());

	fileref_registry.remove(f->get_reg_index());

}



gidispatch_rock_t register_memory(void *array, glui32 len, char *typecode)
{
	static gidispatch_rock_t dummy;

	if (gl_rreg) return (*gl_rreg)(array, len, typecode);
	return dummy;
}

void unregister_memory(void *array, glui32 len, char *typecode, gidispatch_rock_t rock)
{
	if (gl_runreg) (*gl_runreg)(array, len, typecode, rock);
}


////////////////////////////////////////////////////////////////////////////
//
// Implementation of a generic object registry
//


typedef void * pvoid;

glk_registry::glk_registry()
{
	members = new pvoid[8];
	for (int n = 0; n < 8; n++) members[n] = NULL;
	count = 8;
}

glk_registry::~glk_registry()
{
	delete members;
}

int glk_registry::add(void *member)
{
	int n;

	for (n = 0; n < count; n++) 
	{
		if (members[n] == NULL) 
		{
			members[n] = member;
			return n;
		}
	}
	pvoid *tmp = new pvoid[2*count];
	for (n = 0; n < count; n++) 
	{
		tmp[n]       = members[n];
		tmp[n+count] = NULL;
	}
	delete members;
	members = tmp;
	n = count;

	count *= 2;
	members[n] = member;
	return n;
}

void *glk_registry::get(int index)
{
	if (index < 0 || index >= count) return NULL;
	return members[index];
}


int glk_registry::index_of(void *object)
{
	for (int n = 0; n < count; n++)
	{
		if (members[n] == object) return n;
	}
	return -1;
}


void *glk_registry::iterate(void *object)
{
	int n;

	if (object == NULL) n = 0;
	else 
	{
		n = index_of(object);
		if (n < 0) return NULL;
		++n;
	}

	while (n < count)
	{
		if (members[n]) return members[n];
		++n;
	}
	return NULL;
}


void glk_registry::remove(int index)
{
	if (index < 0 || index >= count) return;
	members[index] = NULL;
}

