
/* The glk_text_para class deals with storing and rendering one paragraph
 * of text */

#include "gemglk.h"


glk_text_para::glk_text_para()
{
	buf   = new char[128];
	style = new glui32[128];
	rects = new GRECT[128];
	baseh = new glui32[128];
	base2 = new glui32[128];

	buf_length = 0;
	buf_max = 128;
	cur_style = 0;
}

glk_text_para::~glk_text_para()
{
	delete buf;
	delete style;
	delete rects;
	delete baseh;
	delete base2;
}



int glk_text_para::put_char(char c)
{
	/* If the dynamic text buffer overflows, double its size */
	char str[2];

	glk_assert(buf_max != 0);

	if (buf_length == buf_max)
	{
		char    *nb = new char  [2 * buf_length];
		glui32  *ns = new glui32[2 * buf_length];
		GRECT   *ng = new GRECT [2 * buf_length];
		glui32  *nh = new glui32[2 * buf_length];
		glui32  *n2 = new glui32[2 * buf_length];

		memcpy(nb, buf,   buf_length);
		memcpy(ns, style, buf_length * sizeof(glui32));
		memcpy(ng, rects, buf_length * sizeof(GRECT));
		memcpy(nh, baseh, buf_length * sizeof(glui32));
		memcpy(n2, base2, buf_length * sizeof(glui32));

		delete buf;
		delete style;
		delete rects;
		delete baseh;
		delete base2;

		buf   = nb;
		style = ns;
		rects = ng;
		baseh = nh;
		base2 = n2;
		buf_max *= 2;
	}
	glk_assert(buf_length < buf_max);	

/* Set all stats for the character except its position - that gets 
 * calculated by arrange() */

	GRECT *gr = &rects[buf_length];

	gr->g_x = gr->g_y = 0;
	str[0] = c;
	str[1] = 0;
	baseh[buf_length] = dflt_style[wintype_TextBuffer][cur_style]
                                .text_size(str, &gr->g_w, &gr->g_h);
	style[buf_length] = cur_style;
	buf  [buf_length] = c;
	base2[buf_length] = 0;
	++buf_length;
	return 0;
}

void glk_text_para::set_style(glui32 w)
{
	cur_style = w;
}

glui32 glk_text_para::get_style()
{
	return cur_style;
}


void glk_text_para::get_height(WORD *h, WORD *tlh)
{
	if (buf_length == 0) 
	{
		*h   = 0;
		*tlh = 0;
		return;		
	}	

	*tlh = base2[0];
	*h   = rects[buf_length-1].g_y + rects[buf_length-1].g_h;
}

/* Work out where the next character would be put */
void glk_text_para::get_curpos(WORD *x, WORD *y)
{
	if (buf_length == 0) 
	{
		*x = 0;
		*y = 0;
		return;		
	}	
	*x = rects[buf_length-1].g_x + rects[buf_length-1].g_w;
	*y = rects[buf_length-1].g_y;
}


void glk_text_para::on_paint(WORD x0, WORD y0, PARMBLK *pb)
{
	WORD x,y;
	int rs,re,n;
	char txt[81];
	glk_style *st;

	UNUSED(pb);

	rs = 0;
	while (rs < buf_length)
	{
		/* Find a range of characters with the same style and 
                 * Y-coordinate */
		find_range(&rs, &re);

		if (re - rs > 80) re = rs + 80;

		for (n = rs; n < re; n++)
		{
			if (rects[n].g_y != rects[rs].g_y)
			{
				re = n;
				break;
			}
		}
		x = rects[rs].g_x;
		y = rects[rs].g_y;

		glk_assert(re > rs);
		st = &dflt_style[wintype_TextBuffer][style[rs]];

		/* Print this range */
		memcpy(txt, buf + rs, re - rs);
		txt[re-rs] = 0;

		st->paint(x + x0, y + y0 + base2[rs], txt);

		rs = re;
	}
}


/* Find a range of characters with the same style */
void glk_text_para::find_range(int *rs, int *re)
{
	int n;
	glui32 s;

	n = *rs;
	s = style[n];

	if (n >= buf_length) 
	{
		*re = buf_length; 
		*rs = buf_length; 
		return; 
	}
	while (n < buf_length)
	{
		++n;
		if (n == buf_length || style[n] != s) break;
	}
	*re = n;
}

WORD glk_text_para::measure_text(int rs, int re, WORD *lw, WORD *lh)
{
	WORD y;

	*lw = *lh = y = 0;
	for (int n = rs; n < re; n++)
	{
		*lw += rects[n].g_w;
		if (*lh < rects[n].g_h) *lh  = rects[n].g_h;  
		if (y   < (long)baseh[n]    )   y  = baseh[n];
	}

	return y;
}


void glk_text_para::set_xy(int rs, int re, WORD x, WORD y)
{
	for (int n = rs; n < re; n++)
	{
		rects[n].g_y = y;
		rects[n].g_x = x;
		x += rects[n].g_w;
	}
}



void glk_text_para::layout(WORD w)
{
	WORD h = 0, x0, x=0, y=0;
	WORD lw, lh, maxlh, baseh, tl, ls;
	int rs = 0, re = 0;
	int wrap = 0;
	char txt[81];
	glui32 s;

	/* Make the empty paragraph "\n" equivalent to " \n", so that
         * we're sure of getting a blank line */

	if (buf_length == 0) put_char(' ');

	x0 = x;
	baseh = 0;	/* Max height of this line above baseline */
	maxlh = 0;	/* Maximum text height on this line */
	ls = 0;		/* Index of line start character */
	while (rs < buf_length)
	{
		/* Find a range of characters with the same style */
		find_range(&rs, &re);		

/* rs is the index of the first character to print. re is the index of the
 * first character not to print. */	

		if (re == rs || rs >= buf_length) break;

/* The VDI buffers hold 80 characters. So we can't measure or print a string
 * longer than that.
 *
 * nb: This could be changed by recompiling the bindings with a bigger buffer.
 */

		if (re - rs > 80) re = rs + 80;

/* Pull out the bit we want as an ASCIIZ string */

		memcpy(txt, &buf[rs], re - rs);
		txt[re - rs] = 0;
		s = style[rs];

/* Find out the space occupied by those characters */

		measure_text(rs, re, &lw, &lh);

		/* If it won't fit, trim it down */		
		if( (lw + x0 - x ) > w)
		{

/* Cut down the string until we have either: 
         a string that will fit
    or   the empty string
 */
			while ( ((lw + x0 - x) > w) && (re > rs) )
			{
				txt[strlen(txt)-1] = 0;
				--re;
				measure_text(rs, re, &lw, &lh);
			}

/* If it's empty, there's no room for even a single character */

			if (re == rs) 
			{

/* If we're also right at the left margin, it won't ever fit. So make
 * the best of a bad job by printing a single character, then wrapping.
 */

				if (x0 == x)   
				{              
					++re;  
					txt[0] = buf[rs];
					txt[1] = 0;
				}

/* If we're NOT at the left margin, try again on the next line */

				wrap = 1; 
			}

/* We have a text string that will fit in the available space. This is not
  * the full string. */

			else	
			{
/* Does it have a space in it? If so, break it at the space.
 * 
 * This could perhaps be extended to other whitespace characters.
 */
				char *split = strrchr(txt, ' ');
				
				if (split) 
				{

/* Found a space. Chop off everything to the right of the space */

					*split = 0;
					re = rs + strlen(txt) + 1;
				}
				else 

/* No space was found. If we aren't at the left margin, then wrap to the 
 * next line and try again. If we are, then this string will never wrap 
 * nicely. Wrap it nastily. */
				{
					if (x0 > x) 
					{
						re = rs;
						wrap = 1;
					}				
				}
			}

		}

/* By now, we could have re==rs and wrap==1 - in which case, don't print 
 * anything, just let it rejig the coordinates and try again at the start
 * of a line.*/

		if (re > rs)	/* There still is some text.  */
		{
			set_xy(rs, re, x0, y);

			/* Work out how much space it took up */
			tl = measure_text(rs, re, &lw, &lh);
			/* Get new maximum line heights */
			if (lh > maxlh) maxlh = lh;
			if (tl > baseh) baseh = tl;

			/* Update X */
			x0 += lw;
		}
		if (wrap)
		{			
			for (int n = ls; n < re; n++) base2[n] = baseh;
			ls    = re;
			h    += maxlh;
			y    += maxlh;
			x0    = x;
			maxlh = 0;
			baseh = 0;
			wrap  = 0;
		}

		rs = re;
	}
	/* Set the baseline position for characters on the last line, if any */
	for (int n = ls; n < re; n++) base2[n] = baseh;
}


void glk_text_para::arrange(GRECT *win_rect)
{
	char str[2];
	GRECT *gr;

	str[1] = 0;

	for (int n = 0; n < buf_length; n++)
	{
		str[0]   = buf[n];
		gr       = &rects[n];
		baseh[n] = dflt_style[wintype_TextBuffer][style[n]].text_size(str, &gr->g_w, &gr->g_h);
	}
	layout(win_rect->g_w);
}

//////////////////////////////////////////////////////////////////////////////
