/*****************************************************************************
*   FONTPULL - extract fonts from a GEM driver or TOS ROM                    *
*   Copyright 1999, John Elliott                                             *
*                                                                            *
* This program is free software; you can redistribute it and/or              *
* modify it under the terms of the GNU General Public License                *
* as published by the Free Software Foundation; either version 2             *
* of the License, or (at your option) any later version.                     *
*                                                                            *
* This program is distributed in the hope that it will be useful,            *
* but WITHOUT ANY WARRANTY; without even the implied warranty of             *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *
* GNU General Public License for more details.                               *
*                                                                            *
* You should have received a copy of the GNU General Public License          *
* along with this program; if not, write to the Free Software                *
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.*
*                                                                           *
******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

typedef unsigned char byte;

/* Output file type flags:
 * 
 * atarimode:  Words are MSB first 
 * tosmode:    File is the TOS 1.00 ROM
 * drivermode: File is an EXE-format driver 
 */

int dest_atari, dest_tosrom, dest_driver;

/* Input file type flags: 
 * 
 * src_driver:  EXE-format driver
 * src_tosrom:  The TOS 1.00 ROM
 * src_psf:     PSF (Linux console font)
 * src_atari:   Byteswapped GEM font
 * src_rawfont: Straight dump of VGA character generator
 */

int src_driver, src_tosrom, src_psf, src_atari, src_rawfont;

byte *dest_mem;	/* Memory image of font file */
long dest_len;	/* Length of font file */

byte *src_mem;	/* Memory image of source font file */
long src_len;		/* and its length */

unsigned int peekwbe(byte *base, long offset);
unsigned int peekwle(byte *base, long offset);
unsigned int peekw  (byte *base, long offset, int bigendian);
int load_file(char *s, byte **mem, long *len);

char *nicename(char *s)	/* Create a DOS-friendly filename */
{
	static char buf[20];
	
	sprintf(buf, "%.8s", s);
	s = strrchr(buf, ' ');
	if (s) *s = 0;
	if (strlen(buf) < 5) strcat(buf, "_");
	return buf;
}


void transw(long srco, long dsto)
{
	int w = peekw(src_mem, srco, src_atari);

	if (dest_atari)
	{
		dest_mem[dsto  ] = (w >> 8);
		dest_mem[dsto+1] = (w & 0xFF);
	}
	else
	{
	        dest_mem[dsto+1] = (w >> 8);
                dest_mem[dsto  ] = (w & 0xFF);
        }
}

unsigned int peekw(byte *base, long offset, int bigendian)
		/* Read a word value */
{
	return bigendian ? peekwbe(base, offset) : peekwle(base,offset);
}

unsigned int peekwle(byte *base, long offset)
{
	return ((unsigned int)base[offset+1] << 8) | base[offset];
}


unsigned int peekwbe(byte *base, long offset)
{
        return ((unsigned int)base[offset] << 8) | base[offset+1];
}


unsigned int peekd(byte *base, long offset, int bigendian)
					/* Read a doubleword address. On the */
{					/* PC, assume it's in segment:offset */
					/* form */
	unsigned int seg;

	if (bigendian)
	{
		seg = peekwbe(base, offset);
		return (seg << 16) + peekwbe(base, offset+2);
	}
	seg = peekwle(base, offset+2);
	return (seg << 4) + peekwle(base, offset);
}


/* Starting at offset "o", find the next "System" font in a PC-GEM driver */

long drv_find_font(long o, void *mem, long len)
{
	long count;

	for (count = o; count < len - 8; count++)
	{
		if (!strcasecmp(mem + count, "System")) return count - 4;
	}
	return -1;
}


/* Find a font of cell size w x h in the target file */
long find_font(int w, int h)
{
	if (dest_tosrom)
	{
		if (w == 6 && h ==  6) return 0x125CAL;
		if (w == 8 && h ==  8) return 0x12CA6L;
		if (w == 8 && h == 16) return 0x13702L;

		return -1;
	}
	if (dest_driver)
	{
		long offset  = 0;
		do
		{
			offset = drv_find_font(offset, dest_mem, dest_len);

			if (offset == -1) return -1;
			if (peekwle(dest_mem, offset+52) == w &&
			    peekwle(dest_mem, offset+82) == h) return offset;
			offset += 5;
		}
		while (offset != -1);
	}
	if (peekw(dest_mem, 52, dest_atari) == w &&
	    peekw(dest_mem, 82, dest_atari) == h) return 0;

	return -1;
}

/* Given the address of a font, find its bitmap table */

byte *data_for(void *base, long offs, int bigendian)
{
	long charset = peekd(base, offs + 76, bigendian);

	if (base == src_mem  && src_tosrom)  charset -= 0xFC0000L;
	if (base == dest_mem && dest_tosrom) charset -= 0xFC0000L;
	if (base == dest_mem && dest_driver) charset += 16 * peekwle(base, 8);
	if (base == src_mem  && src_driver)  charset += 16 * peekwle(base, 8);

	return base + charset;	
}

void bitmap_copy(byte *dest, byte *src, int h)
{
	int x,y;
	
	for (y = 0; y < h; y++)
		for (x = 0; x < 256; x++)
		{
			dest[y*256 + x] = src[y + x*h];
		}
        printf("8x%d font: Done.\n", h);

}

void do_rawfont(void)
{
	int   h    = src_len / 256;
	long  offs = find_font(8, h);
	byte *data;
	
	if (offs == -1)
	{
		fprintf(stderr,"8x%d font: Couldn't find a matching font.\n", h);
		return;
	}
	if (peekw(dest_mem, offs + 36, dest_atari) != 0 ||
	    peekw(dest_mem, offs + 38, dest_atari)  != 255)
	{
                fprintf(stderr,"8x%d font: Target is not the full range.\n", h);
                return;
	}

	data = data_for(dest_mem, offs, dest_atari);
	bitmap_copy(data, src_mem, h);	
}

void do_psf(void)
{
        int   h    = src_mem[3];
        long  offs = find_font(8, h);
        byte *data;

	if (src_mem[2])
	{
		fprintf(stderr, "PSF file must be in format 0\n");
		return;
	}

        if (offs == -1)
        {
                fprintf(stderr,"8x%d font: Couldn't find a matching font.\n", h);
                return;
        }

        if (peekw(dest_mem, offs + 36, dest_atari) != 0 ||
            peekw(dest_mem, offs + 38, dest_atari)  != 255)
        {
                fprintf(stderr,"8x%d font: Target is not the full range.\n", h);
                return;
        }


	data = data_for(dest_mem, offs, dest_atari);
	bitmap_copy(data, src_mem + 4, h);
}

void do_gem(long fbase)
{
	int w,h,wb, n;
	long dstoff;
	byte *sf, *df;

        w  = peekw(src_mem, fbase + 52, src_atari);
	wb = peekw(src_mem, fbase + 80, src_atari);
        h  = peekw(src_mem, fbase + 82, src_atari);

	dstoff = find_font(w, h);
	if (dstoff == -1)
	{
		fprintf(stderr, "%dx%d font: No font of this size in target.\n", w,h);
		return;
	}

        if (((peekw(dest_mem, dstoff + 36, dest_atari) != 
              peekw(src_mem,  fbase  + 36, src_atari)) ||
             peekw(dest_mem, dstoff + 38, dest_atari)  != 
              peekw(src_mem,  fbase  + 38, src_atari)))
        {
                fprintf(stderr,"%dx%d font: Character ranges differ.\n",
				w, h);
                return;
        }
	/* Copy font metrics */
	for (n = 40; n < 70; n += 2)
	{
		transw(fbase+n, dstoff+n);
	}

	df = data_for(dest_mem, dstoff, dest_atari);
	sf = data_for(src_mem,  fbase,  src_atari);

	memcpy(df, sf, h * wb);
	fprintf(stderr, "%dx%d font: Copied.\n", w, h);
}

void do_tosrom(void)
{
	src_atari = 1;
	do_gem(0x125CAL);
	do_gem(0x12CA6L);
	do_gem(0x13702L);
}


void do_driver(void)
{
	long offs = 0;

	while ((offs = drv_find_font(offs, src_mem, src_len)) != -1)
	{
		do_gem(offs);
		offs += 5;
	}
}


int main(int argc, char **argv)
{
	FILE *fp;

	if (argc < 3)
	{
		fprintf(stderr, "Syntax: %s driverfile fontfile "
                                "{outfile}\n", argv[0]);
		return 1;	
	}
	if (load_file(argv[1], &dest_mem, &dest_len)) return 1;
	if (load_file(argv[2], &src_mem,  &src_len))  return 2;

	if      (dest_mem[0] == 'M'  && dest_mem[1] == 'Z')   dest_driver = 1;
	else if (dest_mem[0] == 0x60 && dest_mem[1] == 0x1E)  dest_tosrom = 1;
	else if (dest_mem[0] == 0    && dest_mem[2] == 0)     dest_atari  = 1;

	if      (src_mem[0]  == 'M'  && src_mem[0] == 'Z')    src_driver  = 1;
	if 	(src_mem[0]  == 0x60 && src_mem[1] == 0x1E)   src_tosrom  = 1;
	if	(src_mem[0]  == 0x36 && src_mem[1] == 0x04)   src_psf     = 1;
	if      (src_len == 2048 || src_len == 4096 || src_len == 3584) 
							      src_rawfont = 1;
	if      (src_mem[0] == 0     && src_mem[2] == 0)      src_atari   = 1;

	/* This gives 3x5 = 15 possible transformations... */

	if      (src_rawfont) do_rawfont();
	else if (src_driver)  do_driver();
	else if (src_psf)     do_psf();
	else if (src_tosrom)  do_tosrom();
	else                  do_gem(0);


	if (argc >= 4) fp = fopen(argv[3], "wb");
	else	       fp = fopen(argv[1], "wb");

	if (!fp) { perror("Output file"); return 3; }
	if (fwrite(dest_mem, 1, dest_len, fp) < dest_len) perror("Output file");
	fclose(fp);
	free(src_mem);
	free(dest_mem);	

	return 0;
}


int load_file(char *s, byte **mem, long *len)
{
        FILE *fp = fopen(s, "rb");
        if (!fp)
        {
                perror(s);
                return 1;
        }
        fseek(fp, 0, SEEK_END);
        *len = ftell(fp);
        fseek(fp, 0, SEEK_SET);
        *mem = malloc(*len);
        if (!*mem)
        {
                fclose(fp);
                fprintf(stderr, "Can't malloc %ld bytes for %s\n",
                        *len, s);
                return 2;
        }
        if (fread(*mem, 1, *len, fp) < *len)
        {
                fclose(fp);
                free(*mem);
                perror(s);
                return 3;
        }
        fclose(fp);
	return 0;
}
