/*****************************************************************************
*   FONTPULL - extract fonts from a GEM driver or TOS ROM                    *
*   Copyright 1999, John Elliott                                             *
*                                                                            *
* This program is free software; you can redistribute it and/or              *
* modify it under the terms of the GNU General Public License                *
* as published by the Free Software Foundation; either version 2             *
* of the License, or (at your option) any later version.                     *
*                                                                            *
* This program is distributed in the hope that it will be useful,            *
* but WITHOUT ANY WARRANTY; without even the implied warranty of             *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *
* GNU General Public License for more details.                               *
*                                                                            *
* You should have received a copy of the GNU General Public License          *
* along with this program; if not, write to the Free Software                *
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.*
*                                                                           *
******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/* File type flags:
 * 
 * atarimode:  Words are MSB first 
 * tosmode:    File is the TOS 1.00 ROM
 * drivermode: File is an EXE-format driver 
 */

int atarimode, tosmode, drivermode;

unsigned char *driver;	/* Memory image of font file */
long driver_len;	/* Length of font file */
char *output_format;	/* File format to output as */

unsigned int peekwbe(long offset);

char *nicename(char *s)	/* Create a DOS-friendly filename */
{
	static char buf[20];
	
	sprintf(buf, "%.8s", s);
	s = strrchr(buf, ' ');
	if (s) *s = 0;
	if (strlen(buf) < 5) strcat(buf, "_");
	return buf;
}

unsigned int peekw(long offset)	/* Read a word value */
{
	if (atarimode) return peekwbe(offset);
	return ((unsigned int)driver[offset+1] << 8) | driver[offset];
}


unsigned int peekwbe(long offset)	/* Read a big-endian word value */
{
        return ((unsigned int)driver[offset] << 8) | driver[offset+1];
}


unsigned int peekd(long offset)		/* Read a doubleword address. On the */
{					/* PC, assume it's in segment:offset */
					/* form */
	unsigned int seg;

	if (atarimode)
	{
		seg = peekw(offset);
		return (seg << 16) + peekw(offset+2);
	}
	seg = peekw(offset+2);
	return (seg << 4) + peekw(offset);
}


/* Dump a font */
void font_out(long offset)
{
	long charset, ctbl;
	int x,y,w,h, cw, fww, lwc, pts, maxade, minade;
	FILE *fp;
	char fname[150], *face;

	printf("\nFont at %lx\n",    offset);
	printf("Font ID:    %d\n", peekw(offset));
	printf("Font size:  %d\n", pts = peekw(offset+2));
	printf("Font name:  %s\n", face = driver + offset + 4);
	printf("MinADE:     %d\n", minade = peekw(offset+36));
	printf("MaxADE:     %d\n", maxade = peekw(offset+38));
	printf("Top:        %d\n", peekw(offset+40));
	printf("Ascent:     %d\n", peekw(offset+42));
	printf("Half:       %d\n", peekw(offset+44));
	printf("Descent:    %d\n", peekw(offset+46));
	printf("Bottom:     %d\n", peekw(offset+48));
	printf("Char width: %d\n", peekw(offset+50));
	printf("Cell width: %d\n", cw = peekw(offset+52));
	printf("Offset L:   %d\n", peekw(offset+54));
	printf("Offset R:   %d\n", peekw(offset+56));
	printf("Thicken:    %d\n", peekw(offset+58));
	printf("Underline:  %d\n", peekw(offset+60));
	printf("Lighten:    %x\n", peekw(offset+62));
	printf("Skew:       %x\n", peekw(offset+64));
	printf("Flags:      %x\n", peekw(offset+66));
	printf("H offsets:  %x\n", peekd(offset+68));
	printf("C offsets:  %lx\n", ctbl = peekd(offset+72));
	printf("Bitmaps:    %lx\n", charset = peekd(offset+76));
	printf("Form W:     %d\n", fww = peekw(offset+80));
	printf("Form H:     %d\n", h = peekw(offset+82));

	if (tosmode)
	{
		ctbl    -= 0xFC0000L;
		charset -= 0xFC0000L;
	}
	else if (drivermode)
	{
		/* Include the EXE header in our calculations */
		charset += peekw(8)*16;
		ctbl    += peekw(8)*16;
	}
	/* If the file format is suitable, dump it out as a PSF file */
	if (maxade == 255 && minade == 0 && cw == 8 && fww == 256)
	{
		sprintf(fname, "%s%d.psf", nicename(face), pts);
		fp = fopen(fname, "wb");
		if (fp)
		{
			fprintf(fp, "%c%c%c%c", 0x36, 0x04, 0x00, h);
			for (x = 0; x < 256; x++)
			{
				for (y = 0; y < h; y++)
				{
					fputc(driver[charset + 256*y+x] ,fp);
				}
			}
			fclose(fp);
		}	
	} 

#ifdef USE_PBM
	if (output_format)
	{
        	sprintf(fname, "pbmto%s > %s%d.%s", output_format, 
                                                  nicename(face), pts, output_format);
	        fp = popen(fname, "w");
       		if (!fp) return;
	}
	else
#endif
	{
		sprintf(fname, "%s%d.pbm", nicename(face), pts);
		fp = fopen(fname, "w");
		if (!fp) return;
	}


	/* Dump out the bitmap as a PBM*/
	
	w = 8 * fww;
	fprintf(fp, "P1\n");
	fprintf(fp, "%d %d\n", w, h);
	lwc = 0;
	for (y = 0; y < h; y++)
	{
		for (x = 0; x < fww/2; x++)
		{
			unsigned int w = peekwbe(charset + y*fww + 2*x);
			unsigned int m = 0x8000;

			while (m)
			{
				if (w & m) fprintf(fp,"1"); 
				else fprintf(fp,"0");	
				m = m >> 1;
			}
			++lwc;
			if (lwc == 4) { fprintf(fp,"\n"); lwc = 0; }
		}
	}
	fprintf(fp, "\n\n");
#ifdef USE_PBM
	if (output_format) pclose(fp);
	else
#endif
		fclose(fp);
}


int main(int argc, char **argv)
{
	FILE *fp;
	long count;

	if (argc < 2)
	{
		fprintf(stderr, "Syntax: %s fontfile "
#ifdef USE_PBM
				"{output-format}"
#endif
				"\n", argv[0]);
		return 1;	
	} 
#ifdef USE_PBM
	if (argc >= 3) 	output_format = argv[2];
	else
#endif
			output_format = NULL;

	fp = fopen(argv[1], "rb"); 
	if (!fp)
	{
		perror(argv[1]);
		return 1;
	}
	fseek(fp, 0, SEEK_END);
	driver_len = ftell(fp);
	fseek(fp, 0, SEEK_SET);
	driver = malloc(driver_len);
	if (!driver)
	{
		fclose(fp);
		fprintf(stderr, "Can't malloc %ld bytes for the driver\n",
			driver_len);
		return 2;
	}
	if (fread(driver, 1, driver_len, fp) < driver_len)
	{
		fclose(fp);
		free(driver);
		perror(argv[1]);
		return 3;
	}
	fclose(fp);

	if (driver[0] == 'M'  && driver[1] == 'Z')   
		drivermode = 1; /* driver */
	else if (driver[0] == 0x60 && driver[1] == 0x1E)  
		tosmode    = 1; /* TOS ROM */
	else if (driver[0] == 0 && driver[2] == 0)        
		atarimode  = 1; /* TOS font */

	if (tosmode) 	/* TOS ROM image */
	{
		atarimode = 1;
		printf("TOS ROM image\n");
		font_out(0x125CA);
		font_out(0x12CA6);
		font_out(0x13702);
	}
	else if (drivermode)	/* Driver */
	{
		printf("PC-GEM driver\n");
		printf("EXE header occupies %d bytes\n",  peekw(8)*16);
		for (count = 0; count < driver_len - 8; count++)
		{
			if (!strcasecmp(driver + count, "System"))
			{
				font_out(count-4);
			}
		}
	}
	else 
	{
		printf("%s font file\n", atarimode ? "Atari" : "PC");
		font_out(0);	/* Font */
	}
	free(driver);	

	return 0;
}
