/************************************************************************

    VGA2VESA v1.0.0 - Patch the Windows 2.x VGA driver for VESA
		     800x600 mode.

    Copyright (C) 2011, 2014  John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Assume all the files we are patching fit in a 40000-byte buffer. */
#define BUFFSIZE 40000
typedef unsigned char byte;

#define SAVE 0xBF	/* Location of save area: 7FF0 for EGA, BFF0 for VGA */
#define OLDHEIGHT 0xE0, 0x01

byte buf[BUFFSIZE];
unsigned drvsize;

#define EGAMODE  0x10	/* EGA  640x350 */
#define VGAMODE  0x12	/* VGA  640x480 */
#define VESAMODE 0x6A	/* VESA 800x600 */

/* The patches. Each patch comes in two or three sets: the byte sequence
 * to search for, and one or two sequences to replace it with. */

/* First patch: Setting the video mode. Replace */

byte find1[] = { 0xCD, 0x10, 		/* INT 10h */
                 0xAA, 			/* STOSB */
                 0xB8, VGAMODE, 0x00, 	/* MOV AX, EGAMODE */
                 0xCD, 0x10, 		/* INT 10h */
                 0xB8, 0x00, 0x0F, 	/* MOV AX, 0F00h */
                 0xCD, 0x10, 		/* INT 10h */
                 0x3C, VGAMODE };	/* CMP AL, EGAMODE */

byte vesa1[]  = { 0xCD, 0x10, 		/* As above, but for VESAMODE */
                 0xAA, 
                 0xB8, VESAMODE, 0x00, 
                 0xCD, 0x10, 
                 0xB8, 0x00, 0x0F, 
                 0xCD, 0x10, 
                 0x3C, VESAMODE };

/* Second patch: Device properties table. First word is video segment and
 * remains unchanged; second, third and fourth are width, height, bytes per
 * row */
byte find2a[] = { 0x00, 0xA0, 0x80, 0x02,  OLDHEIGHT, 0x50, 0x00 };/* 640x480 */
byte vesa2a[] = { 0x00, 0xA0, 0x20, 0x03, 0x58, 0x02, 0x64, 0x00 };/* 800x600 */

/* Third patch: GDIINFO structure. First two words are aspect ratio,
 * second two are width and height */
byte find2b[] = { 0xD0, 0x00, 0x9C, 0x00, 0x80, 0x02,  OLDHEIGHT };/* 640x480 */
byte vesa2b[] = { 0xD0, 0x00, 0x9C, 0x00, 0x20, 0x03, 0x58, 0x02 };/* 800x600 */

/* Fourth patch: Vertical resolution is hardcoded here and there  */
byte find2c1[] = { 0x81, 0xE9,  OLDHEIGHT };	/* SUB CX, OLDHEIGHT */
byte vesa2c1[] = { 0x81, 0xE9, 0x58, 0x02 };	/* SUB CX, 600 */
byte find2c2[] = { 0xBF, 0xDF, 0x01 };		/* MOV DI, 479 */
byte vesa2c2[] = { 0xBF, 0x57, 0x02 };		/* MOV DI, 599 */
byte find2c3[] = { 0x3D, OLDHEIGHT  };		/* CMP AX, OLDHEIGHT */
byte vesa2c3[] = { 0x3D, 0x58, 0x02 };		/* CMP AX, 600 */

/* As is horizontal resolution */
byte find2d1[] = { 0x2D, 0x80, 0x02 };	/* SUB AX, 640 */
byte vesa2d1[] = { 0x2D, 0x20, 0x03 };	/* SUB AX, 800 */
byte find2d2[] = { 0x05, 0x80, 0x02 };	/* ADD AX, 640 */
byte vesa2d2[] = { 0x05, 0x20, 0x03 };	/* ADD AX, 800 */
byte find2d3[] = { 0xBA, 0x80, 0x02 };	/* MOV DX, 640 */
byte vesa2d3[] = { 0xBA, 0x20, 0x03 };	/* MOV DX, 800 */
byte find2d4[] = { 0xb8, 0x47, 0x00 };	/* MOV AX, 80-9 */
byte vesa2d4[] = { 0xb8, 0x5B, 0x00 };	/* MOV AX, 100-9 */
byte find2d5[] = { 0xb8, 0x4B, 0x00 };	/* MOV AX, 80-5 */
byte vesa2d5[] = { 0xb8, 0x5F, 0x00 };	/* MOV AX, 100-5 */
byte find2d6[] = { 0x83, 0xE9, 0x50 };	/* SUB CX, 80 */
byte vesa2d6[] = { 0x83, 0xE9, 0x64 };	/* SUB CX, 100 */
byte find2d7[] = { 0xBE, 0x7F, 0x02 };	/* MOV SI, 639 */
byte vesa2d7[] = { 0xBE, 0x1f, 0x03 };	/* MOV SI, 799 */

/* Patch the subroutine that calculates the position of the pointer in 
 * video RAM. Originally this would use shifts and multiplies to add AX*50h 
 * to SI. Calculating AX*64h the same way would require more space, so bite 
 * the bullet and use MUL. */
byte find2h[] = { 0xD1, 0xE0, 	/* SHL AX, 1 */
		  0xD1, 0xE0, 	/* SHL AX, 1 */
		  0xD1, 0xE0, 	/* SHL AX, 1 */
		  0xD1, 0xE0,	/* SHL AX, 1 */
		  0x03, 0xF0 	/* ADD SI, AX */
};
byte vesa2h[] = { 0x52, 		/* PUSH DX */
		  0xba, 0x64, 0x00, 	/* MOV DX, 100 */
		  0xF7, 0xE2, 		/* MUL DX */
		  0x5A, 		/* POP DX */
		  0x03, 0xF0, 		/* ADD SI, AX */
		  0xC3 			/* RET */
};

/* Register save area:
   xFF0: current_brush     (8 bytes)
   xFF8: tonys_bar_n_grill (1 byte)
   xFF9: shadow_mem_status (1 byte, EGA only) 
   xFFA: known_word
   xFFC: shadowed_mode
   xFFD: plane_index
   xFFE: enable_test
   xFFF: saved_latches
 */
byte find2i[] = { 0x8D, 0x3E, 0xF0, SAVE };	/* LEA DI, [SAVE+0] */
byte vesa2i[] = { 0x8D, 0x3E, 0xF0, 0xFF };	/* LEA DI, [0xFFF0] */
byte find2j[] = { 0xA0, 0xF0, SAVE };		/* MOV AL, [SAVE+0] */
byte vesa2j[] = { 0xA0, 0xF0, 0xFF };		/* MOV AL, [0xFFF0] */
byte find2k[] = { 0x26, 0x88, 0x3E, 0xF8, SAVE };/* MOV ES:[SAVE+8], BH */
byte vesa2k[] = { 0x26, 0x88, 0x3E, 0xF8, 0xFF };/* MOV ES:[0xFFF8], BH */
byte find2l[] = { 0x26, 0xA0, 0xF8, SAVE };	/* MOV AL, ES:[SAVE+8] */
byte vesa2l[] = { 0x26, 0xA0, 0xF8, 0xFF };	/* MOV AL, ES:[0xFFF8] */
byte find2m[] = { 0x26, 0xA2, 0xF8, SAVE };	/* MOV ES:[SAVE+8], AL */
byte vesa2m[] = { 0x26, 0xA2, 0xF8, 0xFF };	/* MOV ES:[0xFFF8], AL */
// EGA-only
//byte find2n[] = { 0x26, 0xA0, 0xF9, SAVE };	/* MOV AL, ES:[SAVE+9] */
//byte vesa2n[] = { 0x26, 0xA0, 0xF9, 0xFF };	/* MOV AL, ES:[0xFFF9] */
//byte find2o[] = { 0x26, 0xA2, 0xF9, SAVE };	/* MOV ES:[SAVE+9], AL */
//byte vesa2o[] = { 0x26, 0xA2, 0xF9, 0xFF };	/* MOV ES:[0xFFF9], AL */
byte find2p[] = { 0x26, 0x80, 0x0E, 0xF9, SAVE, 0x04 };/* OR ES:[SAVE+9], 4 */
byte vesa2p[] = { 0x26, 0x80, 0x0E, 0xF9, 0xFF, 0x04 };/* OR ES:[0xFFF9], 4 */
byte find2p1[]= { 0x26, 0x88, 0x0E, 0xF9, SAVE };/* MOV ES:[SAVE+9], CL */
byte vesa2p1[]= { 0x26, 0x88, 0x0E, 0xF9, 0xFF };/* MOV ES:[0xFFF9], CL */
byte find2q[] = { 0x26, 0xA3, 0xFA, SAVE };	/* MOV ES:[SAVE+A], AX */
byte vesa2q[] = { 0x26, 0xA3, 0xFA, 0xFF };	/* MOV ES:[0xFFFA], AX */
byte find2r[] = { 0x26, 0xA1, 0xFA, SAVE };	/* MOV AX, ES:[SAVE+A] */
byte vesa2r[] = { 0x26, 0xA1, 0xFA, 0xFF };	/* MOV AX, ES:[0xFFFA] */
byte find2s[] = { 0x26, 0x89, 0x1E, 0xFA, SAVE };/* MOV ES:[SAVE+A], BX */
byte vesa2s[] = { 0x26, 0x89, 0x1E, 0xFA, 0xFF };/* MOV ES:[0xFFFA], BX */
byte find2t[] = { 0x88, 0x26, 0xFC, SAVE };	/* MOV DS:[SAVE+C], AH */
byte vesa2t[] = { 0x88, 0x26, 0xFC, 0xFF };	/* MOV DS:[0xFFFC], AH */
byte find2u[] = { 0x26, 0x8A, 0x26, 0xFC, SAVE };/* MOV AH, DS:[SAVE+C] */
byte vesa2u[] = { 0x26, 0x8A, 0x26, 0xFC, 0xFF };/* MOV AH, DS:[0xFFFC] */
byte find2v[] = { 0x26, 0xC6, 0x06, 0xFC, SAVE };/* BYTE PTR ES:[SAVE+C] */
byte vesa2v[] = { 0x26, 0xC6, 0x06, 0xFC, 0xFF };/* BYTE PTR ES:[0xFFFC] */
byte find2w[] = { 0xFD, SAVE };	/* SAVE+D  blunt instrument!*/
byte vesa2w[] = { 0xFD, 0xFF };	/* FFFE */
byte find2x[] = { 0xFE, SAVE };	/* SAVE+E  blunt instrument!*/
byte vesa2x[] = { 0xFE, 0xFF };	/* FFFE */
byte find2y[] = { 0x26, 0xA0, 0xFF, SAVE };	/* MOV AL, ES:[SAVE+F] */
byte vesa2y[] = { 0x26, 0xA0, 0xFF, 0xFF };	/* MOV AL, ES:[0xFFFF] */
byte find2z[] = { 0x26, 0xA2, 0xFF, SAVE };	/* MOV ES:[SAVE+F], AL */
byte vesa2z[] = { 0x26, 0xA2, 0xFF, 0xFF };	/* MOV ES:[0xFFFF], AL */


/* The driver hardcodes the size of a plane of video RAM here and there */
byte find3[] = { 0xBF, 0x00, 0x96 };	/* plane size 38400 */
byte vesa3[] = { 0xBF, 0x60, 0xEA };	/* plane size 60000 */

byte find4[] = { 0x81, 0xC6, 0xA0, 0x96 };	/* plane size+2 rows 38560 */
byte vesa4[] = { 0x81, 0xC6, 0x28, 0xEB };	/* plane size+2 rows 60200 */
byte find4a[] = { 0xBE, 0xA0, 0x96 };	/* plane size+2 rows 38560 */
byte vesa4a[] = { 0xBE, 0x28, 0xEB };	/* plane size+2 rows 60200 */
byte find4b[] = { 0xBF, 0xA0, 0x96 };	/* plane size+2 rows 38560 */
byte vesa4b[] = { 0xBF, 0x28, 0xEB };	/* plane size+2 rows 60200 */

/* More hardcoded line pitches in Windows 2.x */
byte find4c[] = { 0xb8, 0x50, 0x00 };		/* MOV AX, 80 */
byte vesa4c[] = { 0xb8, 0x64, 0x00 };		/* MOV AX, 100 */
byte find4d[] = { 0x81, 0xC7, 0x50, 0x00 };	/* ADD DI, 80 */
byte vesa4d[] = { 0x81, 0xC7, 0x64, 0x00 };	/* ADD DI, 100 */
byte find4e[] = { 0x81, 0xC7, 0x4F, 0x00 };	/* ADD DI, 79 */
byte vesa4e[] = { 0x81, 0xC7, 0x63, 0x00 };	/* ADD DI, 99 */
byte find4f[] = { 0xba, 0x50, 0x00 };		/* MOV DX, 80 */
byte vesa4f[] = { 0xba, 0x64, 0x00 };		/* MOV DX, 100 */
byte find4g[] = { 0xbb, 0x50, 0x00 };		/* MOV BX, 80 */
byte vesa4g[] = { 0xbb, 0x64, 0x00 };		/* MOV BX, 100 */
/* Don't do this one: the MOV CX, 80 is used to draw the mouse pointer and 
 * is not related to screen drawing 
 * byte find4h[] = { 0xb9, 0x50, 0x00 };		// MOV CX, 80 
 * byte vesa4h[] = { 0xb9, 0x64, 0x00 };		// MOV CX, 100
 */

byte find4i[] = { 0x81, 0xFB,  OLDHEIGHT };	/* CMP BX, OLDHEIGHT */
byte vesa4i[] = { 0x81, 0xFB, 0x58, 0x02 };	/* CMP BX, 600 */
byte find4j[] = { 0x81, 0xFA,  OLDHEIGHT };	/* CMP DX, OLDHEIGHT */
byte vesa4j[] = { 0x81, 0xFA, 0x58, 0x02 };	/* CMP DX, 600 */

byte find4k[] = { 0xBF, 0x50, 0x00 };	/* MOV DI, 80 */
byte vesa4k[] = { 0xBF, 0x64, 0x00 };	/* MOV DI, 100 */
byte find4l[] = { 0xBE, 0x50, 0x00 };	/* MOV SI, 80 */
byte vesa4l[] = { 0xBE, 0x64, 0x00 };	/* MOV SI, 100 */
byte find4m[] = { 0xBB, 0x30, 0x02 };	/* MOV BX, 560 (80 * 7) */
byte vesa4m[] = { 0xBB, 0xBC, 0x02 };	/* MOV BX, 100 (100 * 7) */

byte find9[] = "IBM Personal System/2 Model 50 Color Display";
byte vesa9[] = "VESA 800x600 (based on original VGA driver) ";

/* Apply a patch to the buffer */
void replace(const char *txt, byte *a, byte *b, int len)
{
    unsigned n, found = 0;
    for (n = 0; n <= drvsize-len; n++)
    {
	/* Search for 'find' pattern */
        if (!memcmp(buf+n, a, len))
        {
	    /* If it's found, log it and substitute 'replace' pattern */
            printf("Patching %s at %04x\n", txt, n);
            memcpy(buf+n, b, len);
	    found = 1;
        }
    } 
    if (!found)
    {
	fprintf(stderr, "!! Not applied: %s\n", txt);
    }
}

/* Copy a file. Used to create .GRB and .LGO files to go with the driver. */
int copy(const char *s, const char *d)
{
    FILE *fp1, *fp2;
    int c;

    fp1 = fopen(s, "rb");
    if (!fp1)
    {
        fprintf(stderr, "Cannot open %s\n", s);
        return 1; 
    }
    fp2 = fopen(d, "wb");
    if (!fp2)
    {
        fprintf(stderr, "Cannot open %s\n", d);
        fclose(fp1);
        return 1; 
    }
    while ( (c = fgetc(fp1)) != EOF)
    {
        fputc(c, fp2);
    }
    fclose(fp2);
    fclose(fp1);
    return 0;
} 



/* Create VESA800.DRV */
void pat_vesa(void)
{
    printf("Creating VESA800V.DRV from IBMPS250.DRV:\n");

    replace("mode set",     find1,  vesa1,  sizeof(find1));
    replace("resolution 1", find2a, vesa2a, sizeof(find2a));
    replace("resolution 2", find2b, vesa2b, sizeof(find2b));

    replace("y resolution 3", find2c1, vesa2c1, sizeof(find2c1));
    replace("y resolution 4", find2c2, vesa2c2, sizeof(find2c2));
    replace("y resolution 5", find2c3, vesa2c3, sizeof(find2c3));
    replace("x resolution 1", find2d1, vesa2d1, sizeof(find2d1));
    replace("x resolution 2", find2d2, vesa2d2, sizeof(find2d2));
    replace("x resolution 3", find2d3, vesa2d3, sizeof(find2d3));
    replace("x resolution 4", find2d4, vesa2d4, sizeof(find2d4));
    replace("x resolution 5", find2d5, vesa2d5, sizeof(find2d5));
    replace("x resolution 6", find2d6, vesa2d6, sizeof(find2d6));
    replace("x resolution 7", find2d7, vesa2d7, sizeof(find2d7));
    replace("concat",       find2h, vesa2h, sizeof(find2h));

    replace("register save area 1",find2i, vesa2i, sizeof(find2i));
    replace("register save area 2",find2j, vesa2j, sizeof(find2j));
    replace("register save area 3",find2k, vesa2k, sizeof(find2k));
    replace("register save area 4",find2l, vesa2l, sizeof(find2l));
    replace("register save area 5",find2m, vesa2m, sizeof(find2m));
//    replace("register save area 6",find2n, vesa2n, sizeof(find2n));
//    replace("register save area 7",find2o, vesa2o, sizeof(find2o));
    replace("register save area 8",find2p, vesa2p, sizeof(find2p));
    replace("register save area 8",find2p1, vesa2p1, sizeof(find2p1));
    replace("register save area 9",find2q, vesa2q, sizeof(find2q));
    replace("register save area A",find2r, vesa2r, sizeof(find2r));
    replace("register save area B",find2s, vesa2s, sizeof(find2s));
    replace("register save area C",find2t, vesa2t, sizeof(find2t));
    replace("register save area D",find2u, vesa2u, sizeof(find2u));
    replace("register save area E",find2v, vesa2v, sizeof(find2v));
    replace("register save area F",find2w, vesa2w, sizeof(find2w));
    replace("register save area G",find2x, vesa2x, sizeof(find2x));
    replace("register save area H",find2y, vesa2y, sizeof(find2y));
    replace("register save area I",find2z, vesa2z, sizeof(find2z));

    replace("plane size SUB DI", find3,  vesa3,  sizeof(find3));
    replace("plane size ADD SI", find4,  vesa4,  sizeof(find4));
    replace("plane size MOV SI", find4a, vesa4a, sizeof(find4a));
    replace("plane size MOV DI", find4b, vesa4b, sizeof(find4b));

    replace("line pitch MOV AX",   find4c, vesa4c, sizeof(find4c));
    replace("line pitch ADD DI",   find4d, vesa4d, sizeof(find4d));
    replace("line pitch ADD DI 2", find4e, vesa4e, sizeof(find4e));
    replace("line pitch MOV DX",   find4f, vesa4f, sizeof(find4f));
    replace("line pitch MOV BX",   find4g, vesa4g, sizeof(find4g));
//    replace("line pitch MOV CX",   find4h, vesa4h, sizeof(find4h));
 
    replace("y resolution I", find4i, vesa4i, sizeof(find4i));
    replace("y resolution J", find4j, vesa4j, sizeof(find4j));
    replace("x resolution K", find4k, vesa4k, sizeof(find4k));
    replace("x resolution L", find4l, vesa4l, sizeof(find4l));
    replace("x resolution M", find4m, vesa4m, sizeof(find4m));

    replace("driver desc", find9, vesa9, sizeof(vesa9)-1); 
}



/* Load EGAHIRES.DRV, patch it, write out VESA800.DRV */
int fixvesa()
{
    FILE *fp;

    memset(buf, 0, sizeof(buf));
    fp = fopen("ibmps250.drv", "rb");
    if (fp == NULL)
    {
        fprintf(stderr, "Cannot open IBMPS250.DRV for read.\n");
        return 1;
    }
    drvsize = fread(buf, 1, sizeof(buf), fp); 
    fclose(fp);

    pat_vesa();

    fp = fopen("vesa800v.drv", "wb");
    if (fp == NULL)
    {
        fprintf(stderr, "Cannot open VESA800V.DRV for write.\n");
        return 1;
    }
    if (fwrite(buf, 1, drvsize, fp) != drvsize)
    {
        fprintf(stderr, "Could not write to VESA800V.DRV.\n");
        fclose(fp);
        return 1;
    }
    fclose(fp);
    return 0;
}



int main(int argc, char **argv)
{
	if (fixvesa()) return 1;
	if (copy("egahires.grb", "vesa800v.grb")) return 1;
	if (copy("cga.lgo", "vesa800v.lgo")) return 1;
	return 0;
}
