/************************************************************************

    CPPREP v1.00 - Replacement for MODE CON CP PREPARE supporting various
                  file formats.

    Copyright (C) 2006  John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*************************************************************************/

/* This file is called to load a font. It determines its format, hands 
 * off to the correct load routine, and also synthesizes fonts that 
 * aren't in the original. For example, if you loaded only an 8x16 font, the
 * 8x8 and 8x14 fonts would be generated. */

#include "cpprep.h"

void load_font(int page, char *filename, int *haveFonts)
{
	unsigned char magic[8];
	FILE *fp;

	if (!filename) 
	{
		return;
	}

	fp = fopen(filename, "rb");
	if (!fp)
	{
		fprintf(stderr, "Cannot open font: %s\n", filename);
		exit(1);
	}	
	memset(magic, 0, sizeof(magic));
	fread(magic, 1, sizeof(magic), fp);
	fseek(fp, 0, SEEK_SET);
	
	/* Now to load the file */
	if (!memcmp(magic, "\377FONT   ",      8))
		load_cpi(filename, fp, page, haveFonts, 0);
	else if (!memcmp(magic, "\377FONT.NT", 8))
		load_cpi(filename, fp, page, haveFonts, 1);
	else if (!memcmp(magic, "\177DRFONT ", 8))
		load_cpi(filename, fp, page, haveFonts, 2);
	else if (magic[0] == 0x36 && magic[1] == 0x04)
		load_psf1(filename, fp, page, haveFonts);
	else if (magic[0] == 0x72 && magic[1] == 0xB5 &&
		 magic[2] == 0x4a && magic[3] == 0x86)
		load_psf2(filename, fp, page, haveFonts);
	else if (magic[0] == 0x81 && magic[1] == 0xFC &&
		 magic[4] == 0x77 && magic[5] == 0x02 &&
		 magic[6] == 0xcd && magic[7] == 0x20) /* CPX */
		load_cpx(filename, fp, page, haveFonts);
	else	load_raw(filename, fp, page, haveFonts);
	fclose(fp);
}


int  extrapolate(int page, int haveFonts)
{
	unsigned char far *f16 = fontbits(page, 16);
	unsigned char far *f14 = fontbits(page, 14);
	unsigned char far *f08 = fontbits(page,  8);
	int n, y;

	/* We have no fonts and cannot extrapolate at all */
	if (haveFonts == 0)	return 0;

	/* If we don't have the 8x16 font, create it. From the 8x14 if
	 * that's there, otherwise the 8x8. */
	if (!(haveFonts & 4))
	{
		if (haveFonts & 2)	/* Have 8x14 font */
		{
			for (n = 0; n < 256; n++)
			{
				for (y = 0; y < 16; y++)
				{
					if (y == 0 || y == 15) 
						f16[n*16+y] = 0;
					else	f16[n*16+y] = f14[n*14+y-1];
				}
			}
		}
		else			/* We must have the 8x8 font */
		{
			for (n = 0; n < 256; n++)
			{
				for (y = 0; y < 8; y++)
				{
					f16[n*16+2*y]   = f08[n*8+y];
					f16[n*16+2*y+1] = f08[n*8+y];
				}
			}
		}
		haveFonts |= 4;
	}
	/* If we don't have the 8x14, rather brutally truncate the 8x16,
	  which we now know to exist */
	if (!(haveFonts & 2))
	{
		for (n = 0; n < 256; n++)
		{
			for (y = 0; y < 14; y++)
			{
				f14[n*14+y] = f16[n*16+y+1];
			}
		}
		haveFonts |= 2;				
	}
	/* And if we don't have the 8x8, that comes from the 8x16 too */
	if (!(haveFonts & 1))
	{
		for (n = 0; n < 256; n++)
		{
			for (y = 0; y < 8; y++)
			{
				f08[n*8+y] = f16[n*16+2*y] | f16[n*16+2*y+1];
			}
		}
		haveFonts |= 1;				
	}
	return haveFonts;
}

