/* stdio.c
 * Copyright (C) 2000, Tsurishaddai Williamson, tsuri@earthlink.net
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**********************************************************************/

#include "stdio.h"
#include <stdarg.h>
#include <string.h>
#include "printf.h"
#include "gets.h"

/* CONSOLE printf(), vprintf(), puts() REPLACEMENT */

#pragma mark struct SPrintfState
typedef struct SPrintfState SPrintfState;
typedef SPrintfState *SPrintfStatePtr;
struct SPrintfState {
	char *next;
	char *end;
};

static int SPrintfPutc(int c, void *output)
{
	SPrintfStatePtr s = (SPrintfStatePtr)output;

	if (s->next == s->end)
		goto error;

	*(s->next)++ = c;
	*(s->next) = 0;

	return 1;

error:
	return 0;

}

int
	ConsoleVSNPrintf(char *buffer,
	                 int size,
	                 const char *format,
	                 void *ap)
{
	SPrintfState s;

	s.next = buffer;
	s.end = &buffer[size - 1];

	return _printf(&s, SPrintfPutc, format, ap);

}

int ConsoleSNPrintf(char *buffer, int size, const char *format, ...)
{
	va_list ap;
	int result;

	va_start(ap, format);
	result = ConsoleVSNPrintf(buffer, size, format, ap);
	va_end(ap);

	return result;

}

int ConsoleVSPrintf(char *buffer, const char *format, void *ap)
{
	SPrintfState s;

	s.next = buffer;
	s.end = &buffer[255];

	return _printf(&s, SPrintfPutc, format, ap);

}

int ConsoleSPrintf(char *buffer, const char *format, ...)
{
	va_list ap;
	int result;

	va_start(ap, format);
	result = ConsoleVSPrintf(buffer, format, ap);
	va_end(ap);

	return result;

}

int ConsoleVFPrintf(FILE *file, const char *format, void *ap)
{
	extern int fputc(int c, FILE *file);

	return _printf(file, (int (*)(int, void *))fputc, format, ap);

}

int ConsoleFPrintf(FILE *file, const char *format, ...)
{
	va_list ap;
	int result;

	va_start(ap, format);
	result = ConsoleVFPrintf(file, format, ap);
	va_end(ap);

	return result;

}

static int ConsolePutc(int c, void *output)
{
#pragma unused(output)

	if (c == '\n')
		ConsoleOutput('\r');

	return ConsoleOutput(c);

}

int ConsoleVPrintf(const char *format, void *ap)
{

	return _printf(0, ConsolePutc, format, ap);

}

int ConsolePrintf(const char *format, ...)
{
	va_list ap;
	int result;

	va_start(ap, format);
	result = ConsoleVPrintf(format, ap);
	va_end(ap);

	return result;

}

int ConsolePuts(const char *s)
{

	while(*s != 0)
		ConsolePutc(*s++, 0);

	return ConsolePutc('\n', 0);

}

/* CONSOLE gets() REPLACEMENT */

int ConsoleGetchar(void)
{
	int c;

	while ((c = ConsoleInput(10)) == kConsoleNotReady)
		;

	return c;

}

static int ConsoleProbe(void *input, int waitSeconds)
{
#pragma unused(input)

	return ConsoleInput(waitSeconds);

}

int gGetsStatus;

char *ConsoleGets(char *buffer, int size)
{

	return _gets(0,
	             ConsoleProbe,
	             0,
	             ConsolePutc,
	             buffer,
	             size,
	             &gGetsStatus);

}

/* FOpenPath() opens a file using a search path. */
FILE *FOpenPath(const char *file, const char *mode)
{
	char fileName[kMaxFileName];
	FILE *f;
	int i;

	/* Search for the first instance of the file. */
	f = 0;
	for (i = 0; gFOpenPath[i] != 0; i++) {
		sprintf(fileName, "%s%s", gFOpenPath[i], file);
		if ((f = fopen(fileName, "r")) != 0)
			break;
	}

	/* If the file was not found, open it here. */
	if (f == 0)
		f = fopen(file, mode);
	/* If the file was found, open it here. */
	else if (strcmp(mode, "r")) {
		/* Reopen if the specified mode is not "r". */
		fclose(f);
		f = fopen(fileName, mode);
	}

	/* All done, no error, return the FILE pointer. */
	/* Return zero if there was an error. */
	return f;

}
