// SNA2GIF.CPP (v1.0)
// Extract screen from snapshot and save it as GIF.
//
// Marco Vieth, 24.6.1995 (18.6.1995)
//


#include <stdlib.h>		// exit
#include <fstream.h>
#include <alloc.h>		// farmalloc
#include <dos.h>		// FP_SEG
#include <string.h>
#include <ctype.h>


//#define DEBUG
//#define TEST_OPT	// Optimierung, die nicht viel bringt

extern "C" {
#include "gifsave.h"
}

typedef unsigned char byte;
typedef unsigned short int word;
typedef unsigned long ulong;
typedef byte far fbyte;		// far byte


typedef int ErrCode;
#define OK		0
#define ERROR		1


#define THISPROG "SNA2GIF (v1.0) - Extract GIF screen from snapshot file"
#define COPYRIGHT "(c) Marco Vieth, 24.6.1995"

/*
      The Graphics Interchange Format(c) is the Copyright property of
      CompuServe Incorporated. GIF(tm) is a Service Mark property of
      CompuServe Incorporated.
*/



class membase {
public:
  enum {
    blk_size = 0x4000 	// block size of one memory block
  };

public:
  static ErrCode allocblock(const long size, fbyte **mempool, word *memseg);
  static ErrCode freeblock(fbyte *mempool);
  static ErrCode get_block(ifstream& is, fbyte *blk);
};


ErrCode membase::get_block(ifstream& is, fbyte *blk) {
  byte buf[blk_size];
  is.read(buf, blk_size);
  if (is.gcount() != blk_size) return ERROR;
  _fmemcpy(blk, (fbyte *)buf, blk_size);
  return OK;
}

ErrCode membase::allocblock(const long size, fbyte **mempool, word *memseg) {
  if ((*mempool = (fbyte *)farmalloc(size + 15L)) == NULL)  return ERROR;
  *memseg = FP_SEG(*mempool);
  if (FP_OFF(*mempool)) { (*memseg)++; } /* Add one paragraph, o.k. because +15l */
  return OK;
}

ErrCode membase::freeblock(fbyte *mempool) {
  farfree(mempool);
  return OK;
}





class screen {
  enum { s_max_multi = 6 };
  byte *cpc_palette;

  int width;	// pixels across the screen (640, 320, 160)
  int height;	// pixles down the screen (200)
  int numcolours;	// number of different colours (2, 4, 16, 32?)
  int bits_pr_prim_colour;	// bits per primary colour
  //
  byte ga_inkval[17];
  static byte ga_mode;
  byte ga_mmode[s_max_multi];
  static byte crtc_data[18];
  static fbyte *mem_ptr;
  fbyte *mem_bptr;
  //
  static word cpc_offset;
  static int pix_p_byte;	// pixels per byte
  static int scr_hori;		// screen characters horizontal
  static int scr_verti;		// screen characters vertical
  static int scr_char_raster;	// screen character raster
  static int overscan;		// overscan flag
  //
  ErrCode load_sshot(char *fname);
  ErrCode compress(void);
public:
  ErrCode init(char *filename);
#ifdef TEST_OPT
  static int gpixel0(int x, int y);
  static int gpixel1(int x, int y);
  static int gpixel2(int x, int y);
  static int gpixel0o(int x, int y);
  static int gpixel1o(int x, int y);
  static int gpixel2o(int x, int y);
#else
  static int gpixel(int x, int y);
#endif
  ErrCode create(char *filename);
  screen(int colour_flg);
  ~screen(void);
};


byte screen::ga_mode = 0;
word screen::cpc_offset = 0;
byte screen::crtc_data[18];
fbyte *screen::mem_ptr;
int screen::pix_p_byte = 0;
int screen::scr_hori = 0;
int screen::scr_verti = 0;
int screen::scr_char_raster = 0;
int screen::overscan = 0;




#define S_ID_LG 8
#define SNA_VERSION	2
#define SSIZE64		64
#define SSIZE128	128

ErrCode screen::load_sshot(char *fname) {
  const char ident[S_ID_LG]  = "MV - SNA";

  ifstream is(fname, ios::binary);

  if (!is) {
    cerr << "Couldn't open " << fname << endl;
    return ERROR;
  }

  byte ident_buf[S_ID_LG];
  is.read(ident_buf, S_ID_LG);
  if (memcmp(ident_buf, ident, S_ID_LG)) {
    cerr << "ERROR: Not a Snapshot file.\n";
    return ERROR;
  }

  is.seekg(0x10);
  int sna_ver = is.get();
  if ((sna_ver == 0) || (sna_ver > SNA_VERSION)) {
    cerr << "Version " << sna_ver << " not supported.\n";
    return ERROR;
  }


  is.seekg(0x2f);
  for (int i = 0; i < 17; i++)  ga_inkval[i] = is.get() & 0x1f;
  ga_mode = is.get() & 0x03;	// screen mode
  if (ga_mode == 3) ga_mode = 0;

  is.seekg(0x43);
  for (i = 0; i < 18; i++)  crtc_data[i] = is.get();


  is.seekg(0x6b);
  // Miscellaneous
  int by = is.get(); word dump_size = (is.get()<<8)+by;
  if ((dump_size != SSIZE64) && (dump_size != SSIZE128)) {
    cerr << "Snapshot size " << dump_size << "KB not supported.\n";
    return ERROR;
  }

  // only for versions > 1
  if (sna_ver > 1) {
    by = is.get();		// get CPC version

    by = is.get();		// get interrupt position 0..5
//    zif_set_intnum(by % 6);
    for (int i = 0; i < screen::s_max_multi; i++) {
      ga_mmode[i] = is.get() & 0x03;		// get the multimodes
    }
  } else {
    for (int i = 0; i < screen::s_max_multi; i++) {
      ga_mmode[i] = ga_mode;		// init multimodes
    }
  }

// ---------

  byte scr_start_hi = crtc_data[12];
  word scrbank = ((scr_start_hi & 0x30)<<10);   // ma12,ma13 as a14,a15

  is.seekg(0x100 + scrbank);
  if (membase::get_block(is, mem_ptr)) {
    cerr << "ERROR reading snapshot file.\n";
    return ERROR;
  }

  scrbank += 0x4000;	// maybe we need another block for overscan
  is.seekg(0x100 + scrbank);
  if (membase::get_block(is, mem_ptr + 0x4000)) {  // so we load it now
    cerr << "ERROR reading snapshot file.\n";
    return ERROR;
  }
  return OK;
}




/* CPC-Green-table: defines colors 0..31 on VGA-Card (red,green,blue) */
/* CPC-Palette-table: transfer GA_Out to PC-Color */
static byte cpc_palette_green[32*3] = {
  0x00, 0x24, 0x00,  0x00, 0x24, 0x00,  0x00, 0x30, 0x00,  0x00, 0x3C, 0x00,
  0x00, 0x0C, 0x00,  0x00, 0x18, 0x00,  0x00, 0x1E, 0x00,  0x00, 0x2A, 0x00,
  0x00, 0x18, 0x00,  0x00, 0x3c, 0x00,  0x00, 0x3a, 0x00,  0x00, 0x3e, 0x00,
  0x00, 0x16, 0x00,  0x00, 0x1a, 0x00,  0x00, 0x28, 0x00,  0x00, 0x2c, 0x00,
  0x00, 0x0c, 0x00,  0x00, 0x30, 0x00,  0x00, 0x2e, 0x00,  0x00, 0x32, 0x00,
  0x00, 0x0a, 0x00,  0x00, 0x0e, 0x00,  0x00, 0x1c, 0x00,  0x00, 0x20, 0x00,
  0x00, 0x12, 0x00,  0x00, 0x36, 0x00,  0x00, 0x34, 0x00,  0x00, 0x38, 0x00,
  0x00, 0x10, 0x00,  0x00, 0x14, 0x00,  0x00, 0x22, 0x00,  0x00, 0x26, 0x00
};


/* CPC-Color-table: defines colors 0..31 on VGA-Card (red,green,blue) */
/* CPC-Palette-table: transfer GA_Out to PC-Color */
static byte cpc_palette_col[32*3] = {
  0x29, 0x29, 0x29,	/* white */
  0x29, 0x29, 0x29,	/* (same as white) */
  0x1c, 0x39, 0x20,	/* seegruen */
  0x3c, 0x31, 0x14,	/* pastellgelb */
  0x00, 0x00, 0x2b,	/* blue */
  0x3f, 0x00, 0x28,	/* purpur */
  0x00, 0x28, 0x32,	/* bluegreen */
  0x3f, 0x26, 0x26,	/* purple (rosa)*/
  0x3f, 0x00, 0x38,	/* (same as purpur) */
  0x3c, 0x31, 0x14,	/* (same as 'pastellgelb') */
  0x3a, 0x3a, 0x00,	/* light yellow */
  0x3e, 0x3e, 0x3e,	/* light white */
  0x3f, 0x00, 0x00,	/* light red */
  0x3f, 0x00, 0x3a,	/* light magenta */
  0x3c, 0x2a, 0x00,	/* orange */
  0x3b, 0x1a, 0x3b,	/* past. magenta*/
  0x00, 0x00, 0x2b,	/* (same as blue) */
  0x1c, 0x39, 0x20,	/* (same as 'seegruen') */
  0x00, 0x39, 0x00,	/* lightgreen */
  0x00, 0x2f, 0x3f,	/* light bluegreen */
  0x00, 0x00, 0x00,	/* black */
  0x00, 0x00, 0x3c,	/* lightblue */
  0x00, 0x2e, 0x00,	/* green */
  0x1e, 0x1e, 0x3e,	/* skyblue */
  0x30, 0x00, 0x30,	/* magenta */
  0x20, 0x3d, 0x21,	/* past. green */
  0x2b, 0x3c, 0x00,	/* limonengruen */
  0x1f, 0x1a, 0x3f,	/* past.bluegr.*/
  0x29, 0x05, 0x05,	/* red */
  0x30, 0x00, 0x3f,	/* mauve (violett) */
  0x30, 0x2b, 0x00,	/* yellow */
  0x26, 0x26, 0x3c	/* past. blue */
};

ErrCode screen::create(char *filename) {
  // Create and set up the GIF-file
  cout << "Screen information:\n";
  cout << "scr_mode " << (int)ga_mode << ", scr_base=" << hex <<
    (word)((crtc_data[12] & 0x30)<<10) << ", scr_offset=" << cpc_offset <<
    ", overscan=" << ((overscan) ? "yes" : "no") << endl;
  cout << "pixel_p_byte=" << dec << pix_p_byte <<
    ", horiz.chars=" << scr_hori << ", vert.chars=" <<
    scr_verti << ", raster_p_char=" << scr_char_raster << endl;
  cout << "width=" << width << ", height=" << height << ", numcolours="
    << numcolours << "\n";
  int err = OK;
  if ((err = GIF_Create(filename, width, height, numcolours, bits_pr_prim_colour))
    != GIF_OK) {
    // Set each color according to the values extracted from the palette
    cerr << "ERROR: ";
    if (err == GIF_ERRCREATE) cerr << "Couldn't create GIF file\n";
    else if (err == GIF_ERRWRITE) cerr << "Writing to GIF file\n";
    else if (err == GIF_OUTMEM) cerr << "Out of memory for GIF colour table\n";
    return ERROR;
  }
  for (int i = 0; i < numcolours; i++) {
    byte *colptr = cpc_palette + (ga_inkval[i] * 3);
    GIF_SetColor(i, *colptr, *(colptr+1), *(colptr+2));	// RGB
  }
  err = compress();
  //  Finish it all and close the file
  if (GIF_Close()) {
    cerr << "ERROR: Writing to GIF file\n";
    err = ERROR;
  }
  return (err);
}


ErrCode screen::init(char *filename) {
  word memseg = 0;
  if (membase::allocblock(0x8000, &mem_bptr, &memseg)) {
    cerr << "ERROR: Cannot allocate memory\n";
    return ERROR;
  }
  mem_ptr = (fbyte *)MK_FP(memseg, 0);
  if (load_sshot(filename)) return ERROR;
  byte scr_start_hi = crtc_data[12], scr_start_lo = crtc_data[13];
  cpc_offset = ((scr_start_hi & 0x03)<<9)  /* ma8,ma9   as a9,a10 */
      | (scr_start_lo<<1);  	/* ma0-ma7  as a1-a8 */
  cpc_offset &= 0x7fe;
  overscan = ((scr_start_hi & 0x0c) == 0x0c);  // b2,b3=1 => 32K for overscan
  scr_hori = crtc_data[1] * 2;
//  byte scr_total_verti = crtc_data[4] + 1;
  scr_verti = crtc_data[6] & 0x7f;
  scr_char_raster = (crtc_data[9] & 0x1f) + 1;
  switch (ga_mode) {
    case 0:
      pix_p_byte = 2;
      numcolours = 16;
      bits_pr_prim_colour = 6;
    break;
    case 1:
      pix_p_byte = 4;
      numcolours = 4;
      bits_pr_prim_colour = 6;
    break;
    case 2:
      pix_p_byte = 8;
      numcolours = 2;
      bits_pr_prim_colour = 6;
    break;
    default:
      cerr << "Error screen mode!!";
      break;
  }
  width = pix_p_byte * scr_hori;
  height = scr_verti * scr_char_raster;
  return OK;
}


/*-------------------------------------------------------------------------
 *
 *  NAME:           gpixel()
 *
 *  DESCRIPTION:    Callback function. Near version of getpixel()
 *
 *                  If this program is compiled with a model using
 *                  far code, Borland's getpixel() can be used
 *                  directly.
 *
 *  PARAMETERS:     As for getpixel()
 *
 *  RETURNS:        As for getpixel()
 *
 */

#ifndef TEST_OPT
// get pixel for all modi, overscan yes/no
// pix_p_byte depends on mode! (always 2, 4, 8!)
int screen::gpixel(int x, int y) {
  word addr = (y % scr_char_raster) * 0x800 +
    ((((y / scr_char_raster) * scr_hori) + cpc_offset) & 0x7ff) +
    ((overscan) ? (((((y / scr_char_raster) * scr_hori) + cpc_offset) / 0x800) * 0x4000) : 0) +
    (x / pix_p_byte);
  if (!overscan) addr &= 0x3fff;

  switch(ga_mode) {
    case 0:	// 1 = pix_p_byte-1
      int by = mem_ptr[addr] >> (1 - (x & 1));
      return( ((by<<3) & 0x08) | ((by>>2) & 0x04) |
	((by>>1) & 0x02) | ((by>>6) & 0x01) );

    case 1:	// 3 = pix_p_byte-1
      by = mem_ptr[addr] >> (3 - (x & 3));
      return( ((by & 0x01) << 1) | ( (by & 0x10) >> 4) );

    case 2:	// 7 = pix_p_byte-1
      return( (mem_ptr[addr] >> (7 - (x & 7))) & 0x01 );
  }
  return 0;	// to avoid warning
}
#endif


#ifdef MUELL
// get pixel OLD for all modi, overscan yes/no
int screen::gpixel_old(int x, int y) {
  word addr = (y % scr_char_raster) * 0x800 +
    ((((y / scr_char_raster) * scr_hori) + cpc_offset) & 0x7ff) +
    ((overscan) ? (((((y / scr_char_raster) * scr_hori) + cpc_offset) / 0x800) * 0x4000) : 0) +
    (x / pix_p_byte);
  if (!overscan) addr &= 0x3fff;

  int by = mem_ptr[addr];
  switch(ga_mode) {
    case 0:
      by >>= ((pix_p_byte - (x % pix_p_byte)) - 1);
      by = ((by<<3) & 0x08) | ((by>>2) & 0x04) |
	((by>>1) & 0x02) | ((by>>6) & 0x01);  // b0,b4,b2,b6
    break;

    case 1:
      by >>= ((pix_p_byte - (x % pix_p_byte)) - 1);
      by = ((by & 0x01) << 1) | ( (by & 0x10) >> 4);
    break;

    case 2:
      by >>= ((pix_p_byte - (x % pix_p_byte)) - 1);
      by &= 0x01;
    break;

    default:
      by = 0;
    break;
  }
  return by;
}
#endif


#ifdef TEST_OPT
// folgende Optimierung bringt nicht viel.
#define PIX_P_BYTE (2)
// get pixel, mode 0, no overscan
int screen::gpixel0(int x, int y) {
  word addr = ((y % scr_char_raster) * 0x800 +
    ((((y / scr_char_raster) * scr_hori) + cpc_offset) & 0x7ff) +
    (x / PIX_P_BYTE)) & 0x3fff;
//  int by = (mem_ptr[addr] >> ((PIX_P_BYTE - (x % PIX_P_BYTE)) - 1)); old
  int by = (mem_ptr[addr] >> ((PIX_P_BYTE-1) - (x & (PIX_P_BYTE-1))));
  return(((by<<3) & 0x08) | ((by>>2) & 0x04) |
    ((by>>1) & 0x02) | ((by>>6) & 0x01));  // b0,b4,b2,b6
}

// get pixel, mode 0, overscan
int screen::gpixel0o(int x, int y) {
  word addr = (y % scr_char_raster) * 0x800 +
    ((((y / scr_char_raster) * scr_hori) + cpc_offset) & 0x7ff) +
    ((((y / scr_char_raster) * scr_hori) + cpc_offset) / 0x800) * 0x4000 +
    (x / PIX_P_BYTE);
  int by = (mem_ptr[addr] >> ((PIX_P_BYTE-1) - (x & (PIX_P_BYTE-1))));
  return(((by<<3) & 0x08) | ((by>>2) & 0x04) |
    ((by>>1) & 0x02) | ((by>>6) & 0x01));  // b0,b4,b2,b6
}
#undef PIX_P_BYTE

#define PIX_P_BYTE (4)
// get pixel, mode 1, no overscan
int screen::gpixel1(int x, int y) {
  word addr = ((y % scr_char_raster) * 0x800 +
    ((((y / scr_char_raster) * scr_hori) + cpc_offset) & 0x7ff) +
    (x / PIX_P_BYTE)) & 0x3fff;
  int by = (mem_ptr[addr] >> ((PIX_P_BYTE-1) - (x & (PIX_P_BYTE-1))));
  return((by & 0x01) << 1) | ( (by & 0x10) >> 4);
}

// get pixel, mode 1, overscan
int screen::gpixel1o(int x, int y) {
  word addr = (y % scr_char_raster) * 0x800 +
    ((((y / scr_char_raster) * scr_hori) + cpc_offset) & 0x7ff) +
    ((((y / scr_char_raster) * scr_hori) + cpc_offset) / 0x800) * 0x4000 +
    (x / PIX_P_BYTE);
  int by = (mem_ptr[addr] >> ((PIX_P_BYTE-1) - (x & (PIX_P_BYTE-1))));
  return((by & 0x01) << 1) | ( (by & 0x10) >> 4);
}
#undef PIX_P_BYTE

#define PIX_P_BYTE (8)
// get pixel, mode 2, no overscan
int screen::gpixel2(int x, int y) {
  word addr = ((y % scr_char_raster) * 0x800 +
    ((((y / scr_char_raster) * scr_hori) + cpc_offset) & 0x7ff) +
    (x / PIX_P_BYTE)) & 0x3fff;
  int by = (mem_ptr[addr] >> ((PIX_P_BYTE-1) - (x & (PIX_P_BYTE-1))));
  return(by & 0x01);
}

// get pixel, mode 2, overscan
int screen::gpixel2o(int x, int y) {
  word addr = (y % scr_char_raster) * 0x800 +
    ((((y / scr_char_raster) * scr_hori) + cpc_offset) & 0x7ff) +
    ((((y / scr_char_raster) * scr_hori) + cpc_offset) / 0x800) * 0x4000 +
    (x / PIX_P_BYTE);
  return( (mem_ptr[addr] >> ((PIX_P_BYTE-1) - (x & (PIX_P_BYTE-1)))) & 0x01);
}
#undef PIX_P_BYTE
#endif




ErrCode screen::compress(void) {
  //  Store the entire screen as an image using the user defined
  //  callback function gpixel() to get pixel values from the screen
  int err = OK;

#ifdef TEST_OPT
  int (*f_ptr)(int, int) = screen::gpixel0;
  if (overscan) {
    if (ga_mode == 1) f_ptr = screen::gpixel1o;
    else if (ga_mode == 2) f_ptr = screen::gpixel2o;
    else f_ptr = screen::gpixel0o;
  } else {
    if (ga_mode == 1) f_ptr = screen::gpixel1;
    else if (ga_mode == 2) f_ptr = screen::gpixel2;
    else f_ptr = screen::gpixel0;
  }
  if ( (err = GIF_CompressImage(0, 0, -1, -1, f_ptr)) != GIF_OK) {
#else
  if ( (err = GIF_CompressImage(0, 0, -1, -1, screen::gpixel)) != GIF_OK) {
#endif
    cerr << "ERROR: ";
    if (err == GIF_OUTMEM) cerr << "Out of memory while converting to GIF\n";
    else cerr << "Writing to GIF file\n";
    return ERROR;
  }
  return OK;
}

screen::screen(int colour_flg) {
  cpc_palette = (colour_flg) ? cpc_palette_col : cpc_palette_green;
}

screen::~screen(void) {
  if (membase::freeblock(mem_bptr)) {
    cerr << "ERROR: Freeing memory\n";
  }
}


static ErrCode convert(char *fname, int colour_flg) {
//  char *gif_name = "test.gif";
  char fn[80] = "";
  screen scr(colour_flg);

  strcpy(fn, fname);
  if (strchr(fn, '.') == 0)  strcat(fn, ".SNA");
  cout << fn << " -> ";
  if (scr.init(fn)) return ERROR;
  *strchr(fn, '.') = '\0';
  strcat(fn, ".GIF");
  cout << fn << endl;
  return (scr.create(fn));
}



static ErrCode usage(char *pname) {
  char *pn;

  if ((pn = strrchr(pname, '\\')) == NULL) {
    pn = pname;
  }
  else {
    pn++;
  }
  cout << "Usage: " << pn << " [/pc|/pg] file[.sna] [more files...]\n";
  cout << "Examples:\n";
  cout << pn << " test.sna - extract test.gif\n";
  cout << pn << " *.sna    - convert all *.sna to *.gif\n";
  cout << "/pg=palette green, /pc=palette colour (default)\n";
  cout << endl;
  cout << "The Graphics Interchange Format(c) is the Copyright property of\n";
  cout << "CompuServe Incorporated. GIF(tm) is a Service Mark property of\n";
  cout << "CompuServe Incorporated.\n";
  cout << endl;
  cout << pn << " is written by\n";
  cout << "Marco Vieth, Auf dem Uekern 4, D-33165 Lichtenau, Germany\n";
  cout << "It uses GIFSAVE.C, which is written by\n";
  cout << "Sverre H. Huseby, Bjoelsengt. 17, N-0468 Oslo, Norway\n";
  return OK;
}


int main(int argc, char **argv) {
  int colour = 1;
  cout << THISPROG << endl;
  cout << COPYRIGHT << endl;
  if (argc < 2) {
    (void)usage(argv[0]);
    exit(ERROR);
  }
  cout << "GIF(c), GIF(tm) is Copyright property of CompuServe Incorporated\n";
  int err = OK;
  for (int i = 1; i < argc; i++) {
    if ((argv[i][0] == '-') || (argv[i][0] == '/')) {
      switch(tolower(argv[i][1])) {
	case 'p':
	  int opt2 = tolower(argv[i][2]);
	  if ((opt2 == 'c') || (opt2 == 'g'))
	    colour = opt2 == 'c';
	  else
	    cerr << "Error: unknown option " << argv[i] << endl;
	break;
	default:
	  cerr << "Error: unknown option " << argv[i] << endl;
	break;
      }
    } else {
      struct find_t ffblk;
      int done = _dos_findfirst(argv[i], _A_NORMAL, &ffblk);
      if ( done && (i == 1) ) cout << "No files " << argv[i] << " found.";
      while (!done) {
	if ((err = convert(ffblk.name, colour)) > 0) {
	  cout << "ERROR converting " << argv[i] << endl;
	  return err;
	}
	done = _dos_findnext(&ffblk);
      }
      cout << endl;
    }
  }
  return err;
}
// End of SNA2GIF.CPP
