//: PCPARA.CPP (v1.1)  -- transfer program for the parallel adapter
//. Marco Vieth, 21.2.1995 (25.5.1994)
//

#include <iostream.h>
#include <iomanip.h>	// setw()
#include <fstream.h>
#include <sys/timeb.h>

#include <stdlib.h>	// for exit
#include <dos.h>	// inportb
#include <string.h>	// for strlen
#include <ctype.h>	// toupper
#include <conio.h>	// gotoxy


#define FAST		// use fast LPT port I/O

//#define DEBUG
//#define WITH_DSK	// with save disk? (not implemented yet)


typedef unsigned char byte;
typedef signed char schar;	// signed !! for (ix+zz), zz>0x80
typedef unsigned short int word;
typedef unsigned long ulong;
typedef byte far fbyte;		// far byte


typedef int ErrCode;
#define OK		0
#define ERROR		1

#define THISPROG "PCPARA (v1.2) - Parallel Adapter Transfer (PC side)"
#define COPYRIGHT "(c) Marco Vieth, 18.2.1996"





#define DEF_LPT_ADDR 0x378	// default lpt addr
// DEF_LPT_ADDR: (w) b0 = 1 -> set BUSY on CPC
// DEF_LPT_ADDR+1:  (r) b7..b3=data from CPC
//   b7     = CPC_STROBE
//   b6..b3 = data from CPC (4 bit)
//   b2..b0 = unused


#define ACTIVE 0 		// negative logic !
#define INACTIVE 1


#define RECV_STROBE 0x80

#define CONIND 0x40
#define DATAREQ 0x20

#define TIMEOUT 1000

#define CONIND_TO 1000		// 1000 connect indication timeout
#define CONRES_TO 50000U	// 10000 connect respond timeout
#define DATAREQ_TO 50000	// 40000 data request timeout
#define DATAACK_TO 60000U	// data ack (not request) timeout


// ------------------------------------------------------------------------

static word cpcbusy_port = DEF_LPT_ADDR;
  // normally 0x378 (cpc_busy_port), lpt addr
static word in_port = DEF_LPT_ADDR + 1;
  // normally cpcbusy_port+1 (0x379, in port)

// ------------------------------------------------------------------------


class prompt {
  static char prompt_txt[4];
  static int actprompt;
  static int x, y;
public:
  static inline void print(void);
  static int c_break(void);
};

// static variables:
char prompt::prompt_txt[4] = {'-', '\\', '|', '/'};
int prompt::actprompt = 0;
int prompt::x = 0;
int prompt::y = 0;

static inline void prompt::print(void) {
  x = wherex();
  y = wherey();
  gotoxy(78, 1);
  cout << prompt_txt[(actprompt++) & 0x03] << flush;
  gotoxy(x, y);
  x = 0;
}

int prompt::c_break(void) {
  const int abort = 0;
  cout.flush();
  if (x > 0) gotoxy(x, y);
  cerr << "\nControl-C pressed. Program aborted." << endl;
  outportb(cpcbusy_port, INACTIVE);
  // set adapter inactive
  return(abort);
}

// ------------------------------------------------------------------------

// general :


//static char *input(void) {
//  static char line[80];
//  cin.get(line,80,'\n');	// better than cin >> line
//  return line;
//}


//static double diff_time(const struct timeb *start_t, const struct timeb
//  *end_t) {
//  double diff_t = (end_t->time + end_t->millitm / 1000.0) -
//    (start_t->time + start_t->millitm / 1000.0);
//  return (diff_t);
//}

static void print_time(const struct timeb *start_t, const struct timeb *end_t,
  const ulong count, const char *msg) {
  cout.fill('0');
#ifdef DEBUG
  cout << "start: " << start_t->time << "." << setw(3) << start_t->millitm << endl;
  cout << "end  : " << end_t->time << "." << setw(3) << end_t->millitm << endl;
#endif
  long sec = end_t->time - start_t->time;
  int msec = end_t->millitm - start_t->millitm;
  if (msec < 0) {
    sec--;
    msec = -msec;
  }
  cout << count << " bytes " << msg << " in " << sec << "." << setw(3)
    << msec << " sec. (" << ((sec > 0) ? (count / sec) : 0) << " bytes/sec)\n";
}


// ------------------------------------------------------------------------


// ------------------------------------------------------------------------


class receive {
  static enum {err_noerr = 0, err_nibble_to = 0x100};

  static int get_nibble(void);
  static int get_byte(void);
  static int get_buffer(byte *buf);
public:
  static ErrCode get_string(char *str);
  static void terminal(char *str);
  static ErrCode get_disk_image(byte *buf);
  static ErrCode get_file(void);
};

// receive a nibble. nibble >= 0x100 -> error
int receive::get_nibble(void) {
  const word RECV1_TO = 4000;	// receive timeout
  int err = err_noerr;
  const word maxcount = 200;
  word timeout = 0;

  int nibble = 0;
  outportb(cpcbusy_port, ACTIVE); // ready to receive
  while (((nibble = inportb(in_port)) & RECV_STROBE)) { // wait for strobe active (=0)
    if ((--timeout) >= maxcount) {
      timeout = 0;	// restart
      prompt::print();
    }
  // delay(1); don't use this !!! It slows down the transfer significantly!
  }
  outportb(cpcbusy_port, INACTIVE); // data received

  timeout = 0;
  while (!(inportb(in_port) & RECV_STROBE)) { // wait until strobe inactive (=0)
    if ((++timeout) >= RECV1_TO)  { err = err_nibble_to; break; }
  }
  return(((nibble >> 3) & 0x0f) + err);
}

// receive a byte. byte >= 0x100 -> error
int receive::get_byte(void) {
  int in_byte = get_nibble();	// receive low nibble
  if (in_byte < err_nibble_to) {
    in_byte += (get_nibble() << 4);	// receive high nibble
  }
  return(in_byte);
}


int receive::get_buffer(byte *buf) {
  int len = 0;
  int in_byte;

  if ((in_byte = get_byte()) < err_nibble_to) {
    len = in_byte;
  }
  else {
    cerr << "Error: timeout when reading length\n";
    return(len);
  }

  for (int i = 0; i < len; i++) {
    if ((in_byte = get_byte()) < err_nibble_to) {
      buf[i] = in_byte;
    }
    else {
      cerr << "Error: timeout when reading data item " << i << endl;
      len = 0;
      break;
    }
  }
  return(len);
}


ErrCode receive::get_string(char *str) {
  int len = get_buffer((byte *)str);
  str[len] = '\0';	// set end marker
  return ((len != 0) ? OK : ERROR);
}



void receive::terminal(char *str) {
  cout << "Terminal Mode\n";
  while (get_string(str) != ERROR) {
    cout << str << endl;
  }
}


#ifdef WITH_DSK
union diskinfo_u {
  enum { diskinfo_max = 0x100 };	// const
  struct {
    char ident[34];	//"MV - CPCEMU Disk-File\r\nDisk-Info\r\n"
    byte empty[14];   // free space (0) (up to position 0x30)
    byte tracks;	// number of tracks 40, 80, ?
    byte heads;		// number of heads 1 or 2
    word tsize;		// track-size in bytes (including track-info)
		// computed as spt * (0x0080 << bps) +  IMG_DISK_INFO
  } d;
  byte array[diskinfo_max];
} diskinfo;




struct sectorid_s {
  byte track;
  byte head;
  byte sector;
  byte bps;
};

struct sectorinfo_s {
  struct sectorid_s sid;
  // track-number, head-number, sector number (including offset),
  // BPS (bytes per sector)
  byte state1;	// state 1 errors
  byte state2;	// state 2 errors
  byte unused1;
  byte unused2;
};


union trackinfo_u {
  enum { trackinfo_max = 0x100, sectorinfo_max = 18 };	// const
  struct {
    char ident[12];	// "Track-Info\r\n"
    byte empty[4];	// free space (0) (up to position 0x10)
    byte track;
    byte head;
    byte unused1, unused2;
    // track format parameter:
    byte bps;
    byte spt;
    byte gap3;		// gap#3 format
    byte fill;		// filler byte
    // sector ID's ...
    struct sectorinfo_s si[sectorinfo_max];
  } t;
  byte array[trackinfo_max];
} trackinfo;
#endif	// WITH_DSK




ErrCode receive::get_disk_image(byte *buf) {
  char *fn = (char *)(buf + 4);  	// after DSK:
  cout << "Receiving disk image " << fn << "...\n";

  ofstream os(buf, ios::binary);
  if (os == 0) {
    cerr << "Error opening file " << fn << endl;
    return ERROR;
  }

#ifdef WITH_DSK
  // create disk info
  memset(diskinfo.array, 0, diskinfo_u::diskinfo_max);
  static char d_ident[] = "MV - CPCEMU Disk-File\r\nDisk-Info\r\n";
  strcpy(diskinfo.d.ident, d_ident);
  diskinfo.d.tracks = 40;	// set default values
  diskinfo.d.heads = 1;
  diskinfo.d.tsize = 0x1300;
  os.write(diskinfo.array, diskinfo_u::diskinfo_max);



  // create track info
    // in reality we've to wait for the sector ID's from the CPC!
  memset(trackinfo.array, 0, trackinfo_u::trackinfo_max);
  static char t_ident[] = "Track-Info\r\n";
  strcpy(trackinfo.t.ident, t_ident);
  trackinfo.t.track = 0;	// set default values
  trackinfo.t.head = 0;
  trackinfo.t.bps = 2;
  trackinfo.t.spt = 9;
  trackinfo.t.gap3 = 0x4e;
  trackinfo.t.fill = 0xe5;
  os.write(trackinfo.array, trackinfo_u::trackinfo_max);
#endif	// WITH_DSK


  cerr << "Not implemented yet.\n";
  return ERROR;
}



ErrCode receive::get_file(void) {
   const int recv_buf_max = 0x100;
    // up to to 0x100-1 possible bytes, since the length is coded in one byte
  byte buf[recv_buf_max];
  char *fn = (char *)buf;

  if (get_string(fn) != OK) {
    cerr << "Error receiving filename.\n";
    return ERROR;
  }

  strupr(fn);
  if (!strcmp(fn, "TRM:")) {
    terminal(fn);
    return OK;
  } else if ((strlen(fn) > 4) && (!strncmp(fn, "DSK:", 3))) {
    get_disk_image(buf);
    return OK;
  }

  {
    ifstream is(buf, ios::binary);
    if (is != 0) {
      cerr << "File " << fn << " already exist. Overwrite." << endl;
    }
  }

  ofstream os(buf, ios::binary);
  if (os == 0) {
    cerr << "Error opening file " << fn << endl;
    return ERROR;
  }

  cout << "Receiving file " << fn << endl;

  ulong count = 0;
  int block = 0;
  int lg;
  struct timeb start_t;
  ftime(&start_t);
  while ((lg = get_buffer(buf)) > 0) {
    cout << "\rBlock " << block++ << " received   ";
    count += lg;
    os.write(buf, lg);
  }
  struct timeb end_t;
  ftime(&end_t);
  cout << endl;
//  double diff_t = diff_time(&start_t, &end_t);
//  cout << count << " bytes received in " << diff_t << " sec. ("
//   << (count / diff_t) << " bytes/sec)\n";
  print_time(&start_t, &end_t, count, "received");
  return OK;
}


// ------------------------------------------------------------------------




// sending & receiving:

enum err_enum {
  ERR_OK, ERR_LENGTH0, ERR_TIMEOUT, ERR_TOUT2, ERR_TOUT3
};

static char *error_msg[] = {
  "Ok", "Length is 0", "Timeout" , "Timeout2", "Timeout3"
};


// ...


class send {
  static byte busy_state;

  static ErrCode put_byte(byte sb);
  static ErrCode put_buffer(byte *buf, byte len);
  static ErrCode put_string(char *str);
public:
  static ErrCode put_file(char *fn);
  static ErrCode terminal(void);
  static ErrCode flush_peer(void);
  static ErrCode init_connection(void);
};


byte send::busy_state = INACTIVE;


ErrCode send::put_byte(byte sb) {
  ErrCode err = OK;
  byte local_busy_state = busy_state ^ 1;
  word timeout = 0;

  while ((inportb(in_port) & CONIND)) { // wait for connect indication (=0)
    timeout++;
    if (timeout >= CONIND_TO) {
      timeout = 0;
      prompt::print();
    }
  }

#ifdef MUELL
/*
  asm	mov dx,cpcbusy_port
  asm	mov al,local_busy_state
  asm	out dx,al
  asm	mov cx,CONRES_TO
wait1:
  asm 	mov dx,in_port
  asm	in al,dx
  asm	test al,CONIND
  asm   jnz short wait2
  asm	dec cx
  asm	jnz short wait1
  asm	mov err,ERR_TIMEOUT
wait2:
*/
#else
  outportb(cpcbusy_port, local_busy_state); // connect respond (toggle signal)
  timeout = 0;
  while (!(inportb(in_port) & CONIND)) { // wait until peer got the bit (=1)
    timeout++;
    if (timeout >= CONRES_TO) {
      err = ERR_TIMEOUT;
      break;
    }
  }
#endif


  if (!err) {
    for (int i = 0; i < 8; i++) {		// number of bits

#ifdef FAST
      asm	cli
      asm	mov cx,DATAREQ_TO

      asm	mov al,local_busy_state
      asm	xor al,1
      asm	mov dx,cpcbusy_port
      asm	out dx,al
      asm	mov ah,sb
      asm	and ah,1
      asm	mov dx,in_port
label1:
      asm	in al,dx
      asm	test al,DATAREQ
      asm       jz short label2
      asm	dec cx
      asm	jnz short label1
      asm	mov ah,INACTIVE
      asm	mov err,ERR_TOUT2
label2:
      asm     	mov dx,cpcbusy_port
      asm	mov al,ah
      asm     	out dx,al
      asm	mov local_busy_state,ah
      asm	sti
#else
      timeout = 0;
      disable();
      outportb(cpcbusy_port, local_busy_state ^= 1); // data trigger
      local_busy_state = sb & 0x01;
      while (inportb(in_port) & DATAREQ) { // wait for data request (=0)
	timeout++;
	if (timeout >= DATAREQ_TO) {
	  err = ERR_TOUT2; local_busy_state = INACTIVE; break;
	}
      }
      outportb(cpcbusy_port, local_busy_state); // set data (must be fast enough !)
      enable();
#endif


      // delay(1-?); for slow transfer

      timeout = 0;
      if (!err) {
	while (!(inportb(in_port) & DATAREQ)) { // wait for data ack (=1)
	  timeout++;
	  if (timeout >= DATAACK_TO) {  err = ERR_TOUT3; break; }
	}
      }

      // delay(1-?); for slow transfer

      sb >>= 1;		// next bit
    }
  }
  local_busy_state = INACTIVE;
  outportb(cpcbusy_port, local_busy_state); // inactive
  busy_state = local_busy_state;
  return (err);
}


ErrCode send::put_buffer(byte *buf, byte len) {
  ErrCode err;

  while ((err = put_byte(len)) > 0) {
    cerr << hex << len << " : Error " << error_msg[err] << endl;
  }

  for (int i = 0; i < len; i++) {
    while ((err = put_byte(buf[i]))>0) {
      cerr << hex << buf[i] << " : Error " << error_msg[err] << endl;
    }
  }
  return OK;
}

ErrCode send::put_string(char *str) {
  return(put_buffer((byte *)str, strlen(str)));
}


#define BUFSIZE	0x80		// up to 0xff-1 possible

ErrCode send::put_file(char *fn) {
  byte buf[BUFSIZE];

  cout << "File " << fn << endl;

  ifstream is(fn, ios::binary);
  if (is == 0) {
    cerr << "File " << fn << " not found.\n";
    return ERROR;
  }

  char *pn;
  if ((pn = strrchr(fn, '\\')) == NULL) {
    pn = fn;
  }
  else {
    pn++;
  }
  cout << "Sending file " << pn << endl;
  if (put_string(pn) > 0) {
    cerr << "Error sending filename.\n";
    return ERROR;
  }

  ulong count = 0;
  int block = 0;
  int lg;
  struct timeb start_t;
  ftime(&start_t);
  while ((is.read(buf, BUFSIZE), lg = is.gcount()) > 0) {
    cout << "\rSending block " << block++ << " ...  ";
    count += lg;
    if (put_buffer(buf, lg) > 0) {
      cerr << "Error sending block.\n";
      return ERROR;
    }
  }
  put_byte(0);		// length = 0 -> end of file
  struct timeb end_t;
  ftime(&end_t);
  cout << endl;
//  double diff_t = diff_time(&start_t, &end_t);
//  cout << count << " bytes sent in " << diff_t << " sec. ("
//   << (count / diff_t) << " bytes/sec)\n";

  print_time(&start_t, &end_t, count, "sent");
  return OK;
}



#ifdef MUELL
/*
static ErrCode send_test(void) {
  int i, err;

  for (i = 0; i < 0x100; i++) {
    err = send_byte(i);
    if (err) { printf(" Error %d sending %02X\n", err, i); }
    else printf("%02X, ",i);
  }
  return OK;
}
*/
#endif



#define FLUSH_TO	1000

ErrCode send::flush_peer(void) {
  int err = 0;
  int timeout;

  timeout = 0;
  while (!(inportb(in_port) & DATAREQ)) { // wait for data ack inactive (=1)
    prompt::print();
    delay(2);
    outportb(cpcbusy_port, busy_state ^=1); // toggle state
    timeout++;
    if (timeout >= FLUSH_TO)  {  err = 1; break; }
  }
  busy_state = INACTIVE;
  outportb(cpcbusy_port, busy_state); // inactive
  return (err);
}

static ErrCode check_peer(void) {
  byte by;

  by = inportb(in_port);
  if (!(by & CONIND)) {
    cerr << "CPC is already waiting ..." << endl;
  }
  else {
    cerr << "Start the receiver on the CPC now ..." << endl;
  }

  if (!(by & DATAREQ)) {
    cerr << "Error: CPC is already requesting data" << endl;
    cerr << "Trying to flush it ...\n";
    if (send::flush_peer()) { return(ERROR); }
  }
  return OK;
}

ErrCode send::terminal(void) {
  char str[255];

  cout << "CPC Terminal" << endl;
  cout << "(Exit with CTRL-Z + Return)" << endl;

  put_string("TRM:");

  while ((cin >> str) > 0) {
    strcat(str,"\n\r");
    put_string(str);
  }

  put_byte(0);		// length = 0 -> end of file
  return OK;
}


ErrCode send::init_connection(void) {
  busy_state = INACTIVE;
  outportb(cpcbusy_port, busy_state);		// inactive
  if (check_peer()) {
    cout << "Maybe there's no connection." << endl;
    return ERROR;
  }
  return OK;
}


word get_lpt_addr(int *lpt_port) {
  word far *bios_addr = (word far *)MK_FP(0x0040, 0x0008); // addr of LPT1
  word lpt_addr = ((*lpt_port >= 0) && (*lpt_port <= 3)) ?
    bios_addr[*lpt_port] : 0;
  if (lpt_addr == 0) {
    cerr << "LPT" << (*lpt_port + 1) << " does not exist!" << endl;
    *lpt_port = 0;
    lpt_addr = bios_addr[*lpt_port];
  }
  return (lpt_addr);
}


void do_test(void) {
  int old_p = 0x1ff, p;
  int b_state = INACTIVE;
  long count = 0;
  long errors = 0;

  cout << "Test the adapter (use CPCPARA(...test) on the CPC):\n";
  cout << "Press any key to exit.\n";
  cout << "Nibbles from cpc:\n";
  cout.flush();
  outportb(cpcbusy_port, b_state);
  while (!kbhit()) {
    p = (inportb(in_port) >> 3) & 0x1f;
    if (p != ((inportb(in_port) >> 3) & 0x1f)) { // wait for bits to settle down
      p = (inportb(in_port) >> 3) & 0x1f;
    }
    if (old_p != p) {
      cout << hex << p << " ";
      count++;
      if ((p != ((old_p+1) & 0x1f)) && (old_p != 0x1ff)) {
	errors++;
	cout << "\a";  // beep
      }
      old_p = p;
      outportb(cpcbusy_port, b_state ^= 1);
    }
  }
  outportb(cpcbusy_port, INACTIVE);
  cout << endl << "Received nibbles: " << dec << count << " / Errors: " <<
    errors << endl;
}


static ErrCode usage(char *pname) {
  char *pn;
  if ((pn = strrchr(pname, '\\')) == NULL) pn = pname;
  else  pn++;
  cout << "Usage: " << pn << " /s sendfile | /r | [/t] [/l 1..3]\n";
  cout << "Examples:\n";
  cout << pn << " /s <file>\t - send <file> to CPC (1-bit seriell)\n";
  cout << pn << " /s TRM:\t - send PC terminal input (1-bit seriell)\n";
  cout << pn << " /r  \t - receive data (4-bit parallel)\n";
  cout << "(You will need a fast PC to send to CPC - a 386/33 is enough)\n";
  cout << pn << " /t  \t - test adapter\n";
  cout << pn << " /l x\t - use port LPTx, x=1..3\n";
  cout << endl;
  return OK;
}


#define M_RECEIVE	0
#define M_SEND		1

#define M_TEST 		3


#ifdef MUELL
/*
static ErrCode test(void) {
  struct timeb start_t;
  ftime(&start_t);
  ulong count = 1;
  //
  delay(500);
  struct timeb end_t;
  ftime(&end_t);
  cout << endl;

  print_time(&start_t, &end_t, count, "");
  return ERROR;
}
*/
#endif


int main(int argc, char **argv) {
  int i, err = 0, rc = OK;
  int mode = M_RECEIVE;
  char *sname;
  int lpt_port = 0;

  setcbrk(1);
  ctrlbrk(prompt::c_break);

  cout << THISPROG << endl;
  cout << COPYRIGHT << endl;

#ifdef MUELL
/*
  if (test()) return ERROR;
*/
#endif

  if (argc < 2) {
    (void)usage(argv[0]);
    exit(ERROR);
  }

  for (i = 1; (i < argc) && !err; i++) {
    if ( (argv[i][0] == '-') || (argv[i][0] == '/') ) {
      switch ( toupper(argv[i][1]) ) {
	case 'R' :
	  mode = M_RECEIVE;
	break;

	case 'S' :
	  if ((argc - i) >= 2) {
	     mode = M_SEND;
	     sname = argv[++i];
	  }
	  else  err++;
	break;

	case 'L' :
	  if ((argc - i) >= 2) {
	    lpt_port = atoi(argv[++i]) - 1;
	  }
	  else  err++;
	break;

	case '?' :
	case 'H' :
	  (void)usage(argv[0]);
	break;

	case 'T' :
	  mode = M_TEST;
	break;

	default :
	  err++;
	break;
      }
    } else { err++; }
  }

  if (!err) {
    word lpt_addr = get_lpt_addr(&lpt_port);
    cpcbusy_port = lpt_addr;
    in_port = lpt_addr + 1;
    cout << "Using LPT" << (lpt_port + 1) << " (0x" << hex << lpt_addr << dec
      << ")." << endl;
    switch (mode) {
      case M_RECEIVE :
	do {
	  rc = receive::get_file();
	} while (!kbhit());
      break;

      case M_SEND :
	if (!send::init_connection()) {
	  strupr(sname);
	  if (!strcmp(sname,"TRM:"))  {  rc = send::terminal(); }
	  else {
	    rc = send::put_file(sname);
	  }
	}
      break;

      case M_TEST :
	do_test();
      break;

      default:
	(void)usage(argv[0]);
	rc = ERROR;
      break;
    }
  }
  else
  {
    (void)usage(argv[0]);
    rc = ERROR;
  }
  return (rc);
}
// end of pcpara.cpp
