#!/usr/bin/perl -w
# dskinfo1.pl - show info of DSK file
# Marco Vieth, 27.11.1999
#
# 0.01  27.11.1999 first tests
# 0.011 08.03.2006 reworked
#
  $VERSION = '0.011';
  use strict;
  use Getopt::Std ();


###


sub do_show_info1($$) {
  my($fname, $out_flg) = @_;
  my $dsk = Amstrad::CPC::DiskImage->new();
  $dsk->open($fname, '<') || return;
  $dsk->print_image_info() || return;
  $dsk->close() || return;
  return 1;
}


sub do_format1($$) {
  my($fname, $format1) = @_;
  if ($fname !~ /\.dsk$/o) {
    warn "WARNING: Can only create .dsk and not '$fname'!\n";
    return;
  }
  my $dsk = Amstrad::CPC::DiskImage->new();
  $dsk->open($fname, '>') || return;
  $dsk->format_image($format1) || return;
  $dsk->close() || return;
  return 1;
}



#
# main
#
sub main() {
  my %opts = (
    'o' => 0,
    'F' => '',
  );
  if (!Getopt::Std::getopts("DF:o:hd:", \%opts) or ($#ARGV <= -1) or exists($opts{'h'})) {
    print STDERR "diskinfo1.pl v$::VERSION -- Show contents of DSK file\n";
    require File::Basename;
    print STDERR "Usage: ". File::Basename::basename($0) ." [<options>] file\n";
    #print STDERR "-O file  : optional output file (default: '$opts{'O'}')\n";
    print STDERR "-F type  : format an image with type\n";
    print STDERR "-D       : dup STDOUT to STDERR\n";
    print STDERR "-o num   : output flags... (default: '$opts{'o'}')\n";
    print STDERR "-h       : help\n";
    print STDERR "-d level : set debug level\n";
    exit 1;
  }

  if ($opts{'D'}) { #dup flag
    open(STDERR, ">&STDOUT") || die "Cannot dup STDOUT to STDERR!";
    select((select(*STDERR), $| = 1)[0]); # autoflush needed on some weird Perl 5.6.1 on Windows...
    $| = 1; # set autoflush for STDOUT (normally not needed, but to see messages at correct places...)
  }

  #binmode(STDOUT); # we need to output special characters in strings...

  my $rc = 1;
  foreach my $file (@ARGV) {
    if ($opts{'F'}) { # format?
      do_format1($file, $opts{'F'});
    } else {
      do_show_info1($file, $opts{'o'});
    }
  }
  return ($rc) ? 0 : 1;
}

exit(main());


#
#
#

package Amstrad::CPC::DiskImage;

#  $VERSION = '0.01';
  use strict;


sub new {
  my $class = shift;
  my $self = bless({}, ref($class) || $class);
  $self->_init(@_);
}


sub get_fname { $_[0]->{'fname'} }

sub get_fh { $_[0]->{'fh'} }


sub _init {
  my $self = shift;
  #$self->{'bla'} = 0;
  return $self;
}


sub _file_open($) {
  local *FH;
  open(FH, $_[0]) || return undef();
  return *FH;
}

sub _fread_blk($$) {
  my($fh, $size) = @_;
  my($n, $buf);
  if (($n = read($fh, $buf, $size)) != $size) {
    #if ($errout) {
      warn "WARNING: fread_blk: read $n bytes instead of $size\n";
    #}
    return undef;
  }
  #if ($::g_debug) { Bm::debug_msg("fread_blk: bytes read: '$n', length(buf)='". length($buf) ."'", 2); }
  return \$buf;
}


sub _fwrite_blk($$) {
  my($fh, $bufref) = @_;
  my $n;
  if (($n = print $fh ($$bufref)) != 1) {
    warn "WARNING: fwrite_blk: written $n blocks instead of 1 with length". length($$bufref) ."\n";
    return;
  }
  #if ($::g_debug) { Bm::debug_msg("fwrite_blk: blocks written: '$n', length(block)='". length($$bufref) ."'", 2); }
  return $n;
}


sub open {
  my($self, $fname, $mode) = @_;
  my $fh = _file_open(($mode || '<') . $fname) || (warn("WARNING: $!: '$fname'\n"), return);
  binmode($fh) || (warn("WARNING: Cannot set binary mode for '$fname'"), return);
  $self->{'fname'} = $fname;
  $self->{'fh'} = $fh;
  return 1;
}



#BEGIN AD1  Amstrad CPC 178K Data Side 1 (5.25")
#DENSITY MFM,LOW CYLINDERS 40 SIDES 1 SECTORS 9,512 SKEW 2
#SIDE1 0 0c1h, 0c2h, 0c3h, 0c4h, 0c5h, 0c6h, 0c7h, 0c8h, 0c9h
#BSH 3 BLM 7 EXM 0 DSM 179 DRM 63 AL0 0c0H AL1 0 OFS 0
#END

#BEGIN AD2  Amstrad CPC 178K Data Side 2 (5.25")
#DENSITY MFM,LOW CYLINDERS 40 SIDES 2 SECTORS 9,512 SKEW 2
#SIDE1 0 0c1h, 0c2h, 0c3h, 0c4h, 0c5h, 0c6h, 0c7h, 0c8h, 0c9h
#SIDE2 0 0c1h, 0c2h, 0c3h, 0c4h, 0c5h, 0c6h, 0c7h, 0c8h, 0c9h
#ORDER EAGLE
#BSH 3 BLM 7 EXM 0 DSM 179 DRM 63 AL0 0C0H AL1 0 OFS 40
#END

#BEGIN AS1  Amstrad CPC 169K System Side 1 (5.25")
#DENSITY MFM,LOW CYLINDERS 40 SIDES 1 SECTORS 9,512 SKEW 2
#SIDE1 0 041h, 042h, 043h, 044h, 045h, 046h, 047h, 048h, 049h
#BSH 3 BLM 7 EXM 0 DSM 170 DRM 63 AL0 0C0H AL1 0 OFS 2
#END


# The CP/M 3.1 directory has four types of entry:
#http://members.iinet.net.au/~daveb/cpm/format31.html


# CP/M System Alteration Guid?
#

# CP/M Disk Parameter Block (DPB)
#BSH refers to the block shift or the number of left shifts needed to translate an allocation block number to a relative sector address.
#
#BLM refers to the block mask or a bit mask of ones corresponding to the number of bits specified by BSH. For example, if BSH is 3, BLM is 7 (3 bits).
#
#EXM refers to the extent mask or a bit mask of ones used in delimiting extents, or groups of 128 128-byte records.
#
#DSM refers to the total number of sectors present on a diskette. This quantity, like all of the CP/M system variables, assumes a 128-byte sector.
#
#DRM refers to the total number of directory entries available on a diskette. DRM is also related to the AL0 and AL1 fields.
#
#AL0 and AL1 form a bit mask, beginning with the most significant bit in byte AL0 and ending with the least significant bit in AL1. These two bytes map the first 16 allocation blocks of the disk. A 1 bit in a position indicates that an allocation block is reserved, usually for the directory (see DRM, above).
#
#OFS indicates the number of tracks that are used for storing the CP/M system and BIOS at the #beginning of a diskette.
#
#SOFS  indicates the number of sectors that are used for storing the CP/M system and BIOS at the beginning of a diskette. SOFS is used when the number of sectors used by the CP/M is not an integral  number of tracks, such as for the Coleco Adam.
#


sub get_format_descr($) {
  my($self, $format1) = @_;
  my $format_descr_r = {
    data => {
      tracks => 40, # number of tracks (1-85)
      heads => 1, # number of heads/sides (1-2)
      #head => 0, # head number??
      bps => 2, # Bytes per Sector (1-5)
      spt => 9, # Sectors per Track (1-18)
      gap3 => 0x4e, # gap between ID and data
      fill => 0xe5, # filler byte
      first_sec => 0xc1, # first sector number
    },

    # double sided data
    data2 => {
      _ref => 'data',
      heads => 2,
    },

    system => {
      _ref => 'data',
      first_sec => 0x41,
    },

    # double sided system
    system2 => {
      _ref => 'system',
      heads => 2,
    },

    vortex => {
      _ref => 'data',
      tracks => 80,
      heads => 2,
      first_sec => 0x01,
    },

    '3dos' => {
      _ref => 'data',
      first_sec => 0x00,
    },
  };
  my $fo_r = $format_descr_r->{$format1};
  if (!$fo_r) {
    warn "WARNING: Unknown format: '$format1'\n";
    return;
  }
  if ($fo_r->{'_ref'}) {
    my $ref_format = $fo_r->{'_ref'};
    my $ref_fo_r = $format_descr_r->{$ref_format};
    foreach (keys %$ref_fo_r) {
      if (!defined $fo_r->{$_}) {
        $fo_r->{$_} = $ref_fo_r->{$_}; # get parameter from reference format
      }
    }
    delete $fo_r->{'_ref'};
  }
  return $fo_r;
}


#($$;$)
sub create_disk_info {
  my($self, $tracks, $heads, $tsize, $tsizes_r) = @_;

  my $disk_ident1 = "MV - CPCEMU Disk-File\r\n";
  my $e_disk_ident1 = "EXTENDED CPC DSK File\r\n";
  my $disk_ident2 = "Disk-Info\r\n"; # part 2

  my $di_r = {
    ident => (($tsizes_r) ? $e_disk_ident1 : $disk_ident1) . $disk_ident2,
    creator => 'diskinfo', #max 14 chars
    tracks => $tracks,
    heads => $heads,
    tsize => $tsize, # standard
    tsizes => $tsizes_r, # extended: list of tsizes
  };
  return $di_r;
}


sub _pack_disk_info($) {
  my($di_r) = @_;
  my(@di_txt) = qw(ident creator tracks heads tsize);
  my $disk_info = pack('a34A14CCv', (@{$di_r}{@di_txt})); # get from hash slice
  if ($di_r->{'tsizes'}) { # extended: individual tsizes?
    my @tsizes_hi = map { ($_ >> 8) & 0xff } @{$di_r->{'tsizes'}};
    $disk_info .= pack('C*', @tsizes_hi);
  }

  my $disk_info_size = 0x100;
  $disk_info .= chr(0x00) x ($disk_info_size - length($disk_info)); # fill up
  return \$disk_info;
}


sub _unpack_disk_info($) {
  my($disk_info_r) = @_;

  my(@di_txt) = qw(ident creator tracks heads tsize);
  my $di_r;
  (@{$di_r}{@di_txt}) = unpack('a34A14CCv', $$disk_info_r); # put into hash slice

  my $disk_ident1 = "MV - CPCEMU Disk-File\r\n";
  my $e_disk_ident1 = "EXTENDED CPC DSK File\r\n";
  my $disk_ident2 = "Disk-Info\r\n"; # part 2
  if (($di_r->{'ident'} ne ($disk_ident1 . $disk_ident2)) && ($di_r->{'ident'} ne ($e_disk_ident1 . $disk_ident2))) {
    warn "WARNING: Disk ident not found: '$di_r->{'ident'}'\n";
    return;
  }

  if (!$di_r->{'tsize'}) { # no common tsize specified -> Extended format
    my $tsize_num = $di_r->{'tracks'} * $di_r->{'heads'}; # number of track sizes
    my @tsizes = map { $_ * 0x100 } unpack("x52C${tsize_num}", $$disk_info_r); # get high bytes of track sizes (offset 52) and compute lengths
    $di_r->{'tsizes'} = \@tsizes;
  }
  return $di_r;
}



sub read_disk_info {
  my($self) = @_;
  my $disk_info_size = 0x100;
  my $disk_info_r = _fread_blk($self->get_fh, $disk_info_size) || return;
  _unpack_disk_info($disk_info_r);
}


#($)
sub write_disk_info {
  my($self, $di_r) = @_;
  my $disk_info_r = _pack_disk_info($di_r) || return;
  _fwrite_blk($self->get_fh, $disk_info_r);
}


sub print_disk_info($) {
  my($self, $di_r) = @_;
  my(@di_txt) = qw(ident creator tracks heads tsize);

  print "Disk_Info\n";
  foreach (@di_txt) {
    my $val = $di_r->{$_};
    $val =~ s/\r\n/\\r\\n/go; # for ident
    printf "%-9s = '%s'\n", ucfirst($_), $val;
  }
  print "\n";

  if ($di_r->{'tsizes'}) {
    print "Format    = 'Extended DSK format'\n";
    print "Tsizes    = '@{$di_r->{'tsizes'}}'\n";
  }

  print "\n";
  return 1;
}


###

sub _create_sector_id($$$$) {
  my($fo_r, $track, $head, $sec) = @_;
  my $si_r = {
      track => $track, # track number in ID
      head => $head,  # head number in ID
      sector => $fo_r->{'first_sec'} + $sec, # sector number (with offset)
      bps => $fo_r->{'bps'},
      state1 => 0, # state 1 errors
      state2 => 0, # state 2 errors
      ssize => 0, # sector size, extended: (0x0080 << $ti_r->{'bps'})
  };
  return $si_r;
}


#($$$)
sub create_track_info {
  my($self, $fo_r, $track, $head) = @_;
  my $track_ident = "Track-Info\r\n";

  my $ti_r = {
    ident => $track_ident,
    track => $track,
    head => $head,
    data_rate => 0,
    rec_mode => 0,
    bps => $fo_r->{'bps'},
    spt => $fo_r->{'spt'},
    gap3 => $fo_r->{'gap3'},
    fill => $fo_r->{'fill'},
  };
  for (my $sec = 0; $sec < $fo_r->{'spt'}; $sec++) {
    $ti_r->{'sec_info'}[$sec] = _create_sector_id($fo_r, $track, $head, $sec) || return;
  }
  return $ti_r;
}


sub _pack_track_info($) {
  my($ti_r) = @_;
  my(@ti_txt) = qw(ident track head data_rate rec_mode bps spt gap3 fill);
  my $track_info = pack('a12x4CCCCCCCC', (@{$ti_r}{@ti_txt}));
  my(@si_txt) = qw(track head sector bps state1 state2 ssize);
  for (my $sec = 0; $sec < $ti_r->{'spt'}; $sec++) {
    my $si_r = $ti_r->{'sec_info'}[$sec];
    my $sector_info = pack('C6v', (@{$si_r}{@si_txt}));
    $track_info .= $sector_info;
  }

  my $track_info_size = 0x100;
  my $entry =
  $track_info .= chr(0x00) x ($track_info_size - length($track_info)); # fill up
  return \$track_info;
}


sub _unpack_track_info($) {
  my($track_info_r) = @_;
  my(@ti_txt) = qw(ident track head data_rate rec_mode bps spt gap3 fill);
  # data_rate, rec_mode are extended version 2:
  # data_rate: 0=unknown, 1=single or double density, 2=high density, 3=extended density
  # recording_mode: 0=unknown, 1=FM, 2=MFM
  #
  my $ti_r;
  (@{$ti_r}{@ti_txt}) = unpack('a12x4CCCCCCCC', $$track_info_r); # put into hash slice

  my $track_ident = "Track-Info\r\n";
  if ($ti_r->{'ident'} ne $track_ident) {
    warn "WARNING: Track ident not found: '$ti_r->{'ident'}'\n";
    return;
  }

  my $sect_num = $ti_r->{'spt'}; # number of sectors
  my $off1 = 24;                 # sectors start with offset 24
  #my(@si_txt) = qw(s_trk s_hd s_sec s_bps st1 st2 ssize);
  my(@si_txt) = qw(track head sector bps state1 state2 ssize);
  while ($sect_num-- > 0) {
    my $si_r;
    (@{$si_r}{@si_txt}) = unpack("x${off1}C6v", $$track_info_r); # get sector info into hash slice
    push @{$ti_r->{'sec_info'}}, $si_r;
    $off1 += 8;	# add length
  }
  return $ti_r;
}


sub read_track_info {
  my($self) = @_;
  my $track_info_size = 0x100;
  my $track_info_r = _fread_blk($self->get_fh, $track_info_size) || return;
  _unpack_track_info($track_info_r);
}


sub write_track_info {
  my($self, $ti_r) = @_;
  my $track_info_r = _pack_track_info($ti_r) || return;
  _fwrite_blk($self->get_fh, $track_info_r);
}


sub print_track_info($) {
  my($self, $ti_r) = @_;
  my(@ti_txt) = qw(ident track head data_rate rec_mode bps spt gap3 fill);

  print "Track_Info\n";
  foreach (@ti_txt) {
    my $val = $ti_r->{$_};
    $val =~ s/\r\n/\\r\\n/go; # for ident
    printf "%-9s = '%s'\n", ucfirst($_), $val;
  }
  #my(@si_txt) = qw(s_trk s_hd s_sec s_bps st1 st2 ssize);
  my(@si_txt) = qw(track head sector bps state1 state2 ssize);

  for (my $i = 0; $i < $ti_r->{'spt'}; $i++) {
    print "Sector: ". join(' ', map { $_ ."='". $ti_r->{'sec_info'}[$i]{$_} ."'" } @si_txt) . "\n";
  }
  print "\n";
  return 1;
}


## old: c_tsize = c_ts.spt * (0x0080 << c_ts.bps) + 0x100; //IMG_DISK_INFO;
#c_tsize = 0x100; //IMG_DISK_INFO;
#      for (int i = 0; i < c_ts.spt; i++) {
#        c_tsize += c_ts.ss[i].e_ssize;
#          // compute track size by adding all sector sizes
#      }

sub format_image {
  my($self, $format1) = @_;
  my $fo_r = $self->get_format_descr($format1) || return;
  print "Formatting image ". $self->get_fname() ." with $format1 format\n";
  my $tsize = $fo_r->{'spt'} * (0x0080 << $fo_r->{'bps'}) + 0x100; # 0x100 track info size
  {
    my $di_r = $self->create_disk_info($fo_r->{'tracks'}, $fo_r->{'heads'}, $tsize, undef) || return;
    $self->write_disk_info($di_r) || return;
  }
  my $ssize = (0x0080 << $fo_r->{'bps'}); # sector size
  my $empty_sec_data = pack('C', $fo_r->{'fill'}) x $ssize;
  for (my $track = 0; $track < $fo_r->{'tracks'}; $track++) {
    for (my $head = 0; $head < $fo_r->{'heads'}; $head++) {
      #print "Formatting track $track head $head...\n";
      my $ti_r = $self->create_track_info($fo_r, $track, $head) || return;
      $self->write_track_info($ti_r) || return;
      for (my $sec = 0; $sec < $fo_r->{'spt'}; $sec++) {
        _fwrite_blk($self->get_fh, \$empty_sec_data) || return;
      }
    }
  }
  return 1;
}


sub print_image_info {
  my($self) = @_;
  print "DSK_file  = '". $self->get_fname() ."'\n\n";
  my $di_r = $self->read_disk_info() || return;
  $self->print_disk_info($di_r) || return;
  my $track_info_size = 0x100;
  for (my $track = 0; $track < $di_r->{'tracks'}; $track++) {
    for (my $head = 0; $head < $di_r->{'heads'}; $head++) {
      my $ti_r = $self->read_track_info() || return;
      $self->print_track_info($ti_r) || return;
      my $tsize = $di_r->{'tsize'};
      if (!$tsize) { # no common tsize -> assume extended
        $tsize = @{$di_r->{'tsizes'}}[$track * $di_r->{'heads'} + $head];
      }
      my $data_r = _fread_blk($self->get_fh, $tsize - $track_info_size) || return;
      print "Track_data = '". length($$data_r) ."'\n\n";
    }
  }
  return 1;
}


#sub seek_track {
#  my($self) = @_;
#  return 1;
#}


sub read_sector_id {
  my($self) = @_;
  return 1;
}


sub read_sector {
  my($self) = @_;
  return 1;
}


sub write_sector {
  my($self) = @_;
  return 1;
}


sub close {
  my($self) = @_;
  my $fh = $self->get_fh();
  close($fh) || (warn("WARNING: $!: '". $self->get_fname ."'\n"), return);
  $self->{'fh'} = undef();
  return 1;
}




#############

# http://members.iinet.net.au/~daveb@iinet.net.au/cpm/amsform.html
# http://members.iinet.net.au/~daveb/cpm/format31.html
#0U F1 F2 F3 F4 F5 F6 F7 F8 T1 T2 T3 EX S1 S2 RC   .FILENAMETYP....
#AL AL AL AL AL AL AL AL AL AL AL AL AL AL AL AL   ................
#
# Entry number = ((32*S2)+EX) / (exm+1)
# The total number of records used in this extent is (EX & exm) * 128 + RC
#
sub _unpack_dir_entry($) {
  my($dir_entry_r) = @_;
  my(@de_txt) = qw(user fname ftype extend_lo last_rec_bytes extent_hi records allocation);
  my $de_r;
  (@{$de_r}{@de_txt}) = unpack('Ca8a3CCCCC8', $$dir_entry_r);
  return $de_r;
}

# 1024 byte CP/M block size
# 64 directory entries


#
# BLS: data block allocaton size (1024, 2048, 4096, 8192, 16384)
# SPT: total number of 128-bytes records on each track
# BSH: log2 BLS - 7
# BLM: BLS / 128 - 1
# EXM: if DSM<256 then BLS/1024-1 else BLS/2048-1
# DSM: total size of disc in bloack excluding any reserved tracks
# DRM: total number of directory entries-1
# AL0,1: bit significant representation of number of diectory blocks (0x0080=1, 0x00c0=2,...)
# CKS: length of checksum vector, normally DRM/4+1, or 0 if not used
# OFF: number of reserved tracks (also the track where the directory starts)

#############



1;
#
#

__END__
