/*
 * p/n:
 * Program name:    Dump Formatter symbol table routines
 * Author:          Alan Ballard
 * Backup:
 *
 * Description:
 *    This component contains routines used to process the symbol
 *    table from an object file.
 */

#include	<a.out.h>
#include        <ctype.h>
#include        <stdio.h>

#define MAGIC   exp.a_magic
#define	BADMAG	MAGIC!=A_MAGIC1 && MAGIC!=A_MAGIC2  \
		&& MAGIC!=A_MAGIC3 && MAGIC!=A_MAGIC4
#define SYMLEN  sizeof(p->n_name)

char    *malloc();           /* C allocation routines */
char	*realloc();
int     compare();           /* comparison routine for sort */

struct  nlist *map = NULL;   /* symbol table */
int     mapsize;             /* number of entries in table */

/*
 * getsymtab reads the map (symbol table) for the specified
 * file.
 */
getsymtab(mapfile)
char    *mapfile;
{
        FILE    *fi;         /* map file file descr */
        int     o;           /* offset in file */
        register i, n, c;
        struct  nlist   sym; /* symbol being processed */
        struct  exec    exp; /* for object header */

        fi = fopen(mapfile,"r");
        if (fi == NULL)
                error(1, "cannot open %s", mapfile);
        fread((char *)&exp, 1, sizeof(MAGIC), fi);   /* get magic no. */
        if (BADMAG)
                error(1,"%s-- bad format", mapfile);
        fseek(fi, 0, 0);
        fread((char *)&exp, 1, sizeof(struct exec), fi);

        o = exp.a_text + exp.a_data;
        if ((exp.a_flag & 01) == 0)
                o += exp.a_reloc;
        fseek(fi, o, 1);

        n = exp.a_syms / sizeof(struct nlist);
        if (n == 0)
                error(1,"%s-- no symbol table",  mapfile);

        i = 0;
        while (--n >= 0) {
                if (fread((char *)&sym, 1, sizeof(sym), fi) !=
                    sizeof(sym))
                        error(1,
                          "addr: %s-- symbol table truncated", mapfile);

                sym.n_name[0] &= 0177;  /* zero high bit for local symbols */

                switch (sym.n_type&N_TYPE) {

                case N_UNDF:
                        /*  ignore undefined symbols */
                        continue;

                default:
                case N_ABS:
                        c = 'a';
                        break;

                case N_TEXT:
                        c = 't';
                        break;

                case N_DATA:
                        c = 'd';
                        break;

                case N_BSS:
                        c = 'b';
                        break;
                }

                if (sym.n_type&N_EXT)
                        c = toupper(c);
                sym.n_type = c;

                if (map == NULL)
                    map = (struct nlist *)malloc(sizeof(struct nlist) );
                else
                    map = (struct nlist *)realloc(map,
                                             (i+1)*sizeof(struct nlist));

                if (map == NULL)
                        error(2, "out of memory on %s", mapfile);
                map[i++] = sym;
        }
        mapsize = i;

        fclose(fi);

        /*
         *  Sort the map in increasing address order for convenient
         *  address lookup.
         */
        qsort(map, mapsize, sizeof(struct nlist), compare);

        return (1);
}

/*
 *  compare is comparison routine called from qsort to order a pair
 *  of symbols.
 */
compare(p1, p2)
struct nlist *p1, *p2;
{
        return (p1->n_value - p2->n_value);
}

/*
 *  lookup tries to find the specified string in the map and
 *  returns the symbol pointer.
 */
struct nlist *lookup(str)
char    *str;        /* null-terminated symbol */
{
        register struct nlist *p;
        register cmplen;
        cmplen = strlen(str);
        if (cmplen > SYMLEN)
            /* The term is too long for a symbol */
            return (NULL);
        for (p = map; p < &map[mapsize]; p++)
            if (strncmp(str, p->n_name, cmplen) == 0)
                if (cmplen == SYMLEN || p->n_name[cmplen]  == '\0')
                    return(p);
        return (NULL);
}

/*
 *  symlook finds the specified symbol in the map and returns its
 *  address or -1.
 */
symlook(str)
char    *str;
{
        register struct nlist *p;
        p = lookup(str);
        if (p == NULL)
                return (-1);
        else
                return (p->n_value);
}

/*
 *  addrlook finds the symbol that is closest (lower than)
 *  given address.
 *
 *  It returns the symbol list element.
 */
struct nlist *addrlook(addr)
int     addr;
{
        register struct nlist *p;
        for (p = map; p < &map[mapsize]; p++)
                if (p->n_value == addr)
                        return(p);
                else if (p->n_value > addr)
                        break;
        if (p != map)
                --p; /* back up to previous */
        return (p);
}

/*
 *  symf outputs the values of an address as a string of the form
 *    symbol(t)+offset
 */
symf(addr)
int     addr;
{
        extern FILE *out;
        register struct nlist *sym;
        int offs;

        sym = addrlook(addr);
        fprintf(out,"%-.8s", sym->n_name);
        offs = addr - sym->n_value;
        if (offs)
                fprintf(out,"+%06x", offs);
}
