/*
 * The PLI command is a user interface to the SEND command, designed
 * explicitly for the purpose of submitting PL/I programs for Compile,
 * Compile and Link, or Compile, Link and Go type runs.
 *
 * The valid flags are:
 *
 *   -c   Compile the given files.
 *   -l   Compile and Link the given files.
 *   -g   Compile all but the last file given.  Compile, Link and
 *        Go the last file.
 *   -r   Don't prompt for information; use what is in the .../pli
 *        file.
 *   -s   Instead of invoking the SEND command, place the PLI command
 *        output on the standard output.
 *   -q   Instead of invoking SEND, invoke GATH so that the generated
 *        runstream is placed on the standard output.
 */

#include  "pli.h"
#include  <stdio.h>

char parmf[]    "/.../pli";
char pfile[51];
char tempfile[] "/tmp/pliXXXXX";
char jobcf[]    "/usr/lib/model/jobcard";
char plicf[]    "/usr/lib/model/plic";
char pliclf[]   "/usr/lib/model/plicl";
char pliclgf[]  "/usr/lib/model/pliclg";

char usage[]    "Usage: pli -clgrsq file1 file2 ...\n";

extern int cancel();



main(argc, argv) char **argv;
{
        /* Get the command line parms */
        while(--argc>0 && *argv[1]=='-') {
                argv++;
                while(*++*argv) switch(**argv) {
                        case 'c':
                                cflag++;
                                continue;
                        case 'l':
                                lflag++;
                                continue;
                        case 'g':
                                gflag++;
                                continue;
                        case 'r':
                                rflag++;
                                continue;
                        case 's':
                                sflag++;
                                continue;
                        case 'q':
                                qflag++;
                                continue;
                        default:
                                fprintf(stderr, "%s", usage);
                                exit(1);
                }
        }
        if (argc<1) {
                fprintf(stderr, "%s", usage);
                exit(1);
        }
        /* set up mode flag for C-CL-CLG */
        if      (gflag) mode = 2;
        else if (lflag) mode = 1;
        else            mode = 0;

        mktemp(tempfile); /* create and open output file */
        outfile = fopen(tempfile,"w");
        if ((signal(INTR, 1)&01)==0) signal(INTR, cancel);

        getdefaults(argc, argv);
        if (rflag && pm.account[0]==NULL) {
                fprintf(stderr,"Missing .../pli file; -r flag invalid.\n")
                ;
                cancel();
        }
        if (!rflag) {
                prompt_for_parms();
                save_parms();
        }
        putkeys();
        putjobcard();
        putexecs();
        fprintf(outfile, "//\n");
        fclose(outfile);

        callsend();
        unlink(tempfile);    /* cleanup that temp file */
}


getdefaults(argc, argv) char **argv;

/* read the default parms file, and for each file given on the   */
/* command line, fill the 'files' array with an index to the     */
/* appropriate filename and default member name in 'pm'          */
{
        char *dir;
        int  pfd;

        dir = getenv("HOME");        /* user's home directory */
        strcpy(pfile, dir);
        strcat(pfile, parmf);  /* make parm file name   */
        if ((pfd=open(pfile, 0))<0) {
                /* file not found, give possible defaults */
                strcpy(pm.outclass, "A");
                strcpy(pm.cputime, "20");
                strcpy(pm.linkops, "NCAL");
                strcpy(pm.e1bldg,  "NO");
                strcpy(pm.cops  ,  "INCLUDE");
        }
        else {
                if (read(pfd, &pm, sizeof(pm))<0) {
                        fprintf(stderr, "Error reading %s\n", pfile);
                        cancel();
                }
                read(pfd, &dfiles, MAXFILES*sizeof(*dfiles));
                close(pfd);
        }
        argv++;
        for (filecnt=0; filecnt<argc; filecnt++)
                files[filecnt] = enter(argv[filecnt]);
}


enter(fn) char *fn;

/* find or enter the given UNIX filename in 'dfiles' and return */
/* the index to the structure array                             */
{
        int i, j, min;

        for (i=0; i<MAXFILES; i++) {
                if(cmpend(fn,dfiles[i].filename)==0) {
                        dfiles[i].ltime = time();
                        strcpy(dfiles[i].filename, fn);
                        return(i);
                }
                else if(dfiles[i].filename[0]==NULL) {
                        /* reached end of list, add this one */
                        strcpy(dfiles[i].filename, fn);
                        dfiles[i].ltime = time();
                        return(i);
                }
        }
        /* end of a full list ... find oldest file and replace */
        min = HUGE;
        j = 0;
        for (i=0; i<MAXFILES; i++)
                if(dfiles[i].ltime < min) {
                        min = dfiles[i].ltime;
                        j = i;
                }
        strcpy(dfiles[j].filename, fn);
        dfiles[j].memb[0] = NULL;
        dfiles[j].ltime     = time();
        return(j);
}


cmpend(f1, f2)  char *f1, *f2;

/* Compare the last pathname given in filenames f1 and f2.  */
/* Return 0 if they are equal.                              */
{
        register char *s1, *s2;

        s1 = f1;
        while(*++s1) ;
        while(*--s1 != '/' && s1 != f1) ;
        if(*s1 == '/') ++s1;

        s2 = f2;
        while(*++s2) ;
        while(*--s2 != '/' && s2 != f2) ;
        if(*s2 == '/') ++s2;

        return(strcmp(s1, s2));
}


prompt_for_parms()

/* Use the quickscreen 3270 interface to prompt for the user parms */
/* and display defaults                                            */
{
        qsopen();       /* open terminal for full-screen io   */
        switch(mode) {
                case 0:
                        jobparms("COMPILE");
                        plicparms();
                        break;
                case 1:
                        jobparms("COMPILE AND LINK");
                        pliclparms();
                        break;
                case 2:
                        jobparms("COMPILE, LINK AND GO");
                        pliclgparms();
                        break;
        }
        qsclose();      /* set the terminal back to line mode */
}


nonnull(s, curspos, mesg) char *s, *mesg; int *curspos;

/* Return 1 if the given string is non-null.  As a side effect,     */
/* remove all blanks in the string.  If the string is NULL, set     */
/* the value of curspos to point to the field, put an appropriate   */
/* error message into mesg, and return 0.                           */
{
        strip(s);
        if(*s == NULL) {
                *curspos = qspos(s);
                cpright(mesg, "ENTER REQUIRED PARM", 30);
                panel (noerase,bell,nowait);
                return(0);
        }
        else
                return(1);
}


strip(s) char *s;

/* Strip any blanks from the given string */
{
        register char *t, *f;

        t = f = s;
        while(*f != NULL) {
                if (*f == ' ') f++;
                else
                        *t++ = *f++;
        }
        *t = NULL;
}


nonnumb(s, curspos, mesg) char *s, *mesg; int *curspos;

/* Return 1 if the given string is numeric. If the string is not   */
/* an integer number, set the value of curspos to point to the      */
/* field, put an appropriate error message into mesg, and return 0. */
{
        while(*s != NULL) {
                if(*s < '0'  ||  *s > '9') {
                        *curspos = qspos(s);
                        cpright(mesg, "FIELD MUST BE NUMERIC", 30);
                        panel (noerase,bell,nowait);
                        return(0);
                }
                s++;
        }
        return(1);
}


checkyn(s, curspos, mesg) char *s, *mesg; int *curspos;

/* Return 1 if the given string contains 'YES' or 'NO'.  Otherwise    */
/* set the value of curspos to point to the field, put an appropriate */
/* error message into mesg, and return 0.                             */
{
        if (strcmp(s, "NO")!=0 && strcmp(s, "YES")!=0 &&
            strcmp(s, "no")!=0 && strcmp(s, "yes")!=0) {
                *curspos = qspos(s);
                cpright(mesg, "SPECIFY \"yes\" OR \"no\"", 30);
                panel (noerase,bell,nowait);
                return(0);
        }
        return(1);
}


cpright(s1, s2, len) char *s1, *s2;

/* This thing copies string s2 into s1, right justifying the string */
/* according to the given length of s1.                             */
{
        int i, j;
        char *s;

        j = len - strlen(s2);
        s = s1;
        for(i=0; i<j; i++) *s++ = ' ';
        strcpy(s, s2);
}


char *lower(s) char *s;

/* Translate the given string to lower case and return address of */
/* the string.                                                    */
{
        register char *t;

        t = s;
        while(*t != NULL) {
                if (*t >= 'A' && *t <= 'Z')
                        *t = *t - 'A' + 'a';
                t++;
        }
        return(s);
}


save_parms()

/* save the user's parms on the .../pli file */
{
        int pfd;

        if ((pfd = creat(pfile,0644))<0) {
                fprintf(stderr,"Could not save parms in %s\n",pfile);
                return;
        }
        write(pfd, &pm, sizeof(pm));
        write(pfd, &dfiles, MAXFILES*sizeof(*dfiles));
        close(pfd);
        return;
}

putkeys()

/* Write all the keyword definitions in SEND format to the temp file */
/* Note: this does not write the MEMBER keyword for object and load  */
/* library output.  That's done by the 'putexecs' routine.           */
{
        int itime;
        fprintf(outfile, "~&JOBNAME=%s\n",  pm.jobname );
        fprintf(outfile, "~&ACCOUNT=%s\n",  pm.account );
        fprintf(outfile, "~&BIN=%s\n",      pm.bin     );
        fprintf(outfile, "~&NAME=%s\n",     pm.name    );
        itime = atoi(pm.cputime);
        fprintf(outfile, "~&CPUTIME=(%d,%d)\n",itime/60,itime%60);

        if(strcmp(lower(pm.e1bldg), "yes")==0 && pm.unixfile[0]==NULL)
                fprintf(outfile, "~&E1BLDG=%s\n",   pm.e1bldg  );
        if(pm.unixfile[0]!=NULL)
                fprintf(outfile, "~&OUTCLASS=A\n");
        else
                fprintf(outfile, "~&OUTCLASS=%s\n", pm.outclass);
        fprintf(outfile, "~&UNIXFILE=%s\n", pm.unixfile);
        fprintf(outfile, "~&UNIXID=%s\n", getlogin());

        fprintf(outfile, "~&COPS=%s\n",     pm.cops    );
        fprintf(outfile, "~&PLINPT1=%s\n",  pm.plinpt1 );
        fprintf(outfile, "~&PLINPT2=%s\n",  pm.plinpt2 );
        if (mode != 2)
               fprintf(outfile, "~&OBJECT=%s\n",   pm.object  );
        fprintf(outfile, "~&LINKIN1=%s\n",  pm.linkin1 );
        fprintf(outfile, "~&LINKIN2=%s\n",  pm.linkin2 );
        fprintf(outfile, "~&LOADMOD=%s\n",  pm.loadmod );
        fprintf(outfile, "~&LINKOPS=%s\n",  pm.linkops );
        fprintf(outfile, "~&TESTDT1=%s\n",  pm.testdt1 );
        fprintf(outfile, "~&TESTDT2=%s\n",  pm.testdt2 );
        fprintf(outfile, "~&TESTDT3=%s\n",  pm.testdt3 );
        fprintf(outfile, "~&DATADD1=%s\n",  pm.datadd1 );
        fprintf(outfile, "~&DATADD2=%s\n",  pm.datadd2 );
        fprintf(outfile, "~&DATADD3=%s\n",  pm.datadd3 );
        fprintf(outfile, "~&GOOPS=%s\n",  pm.goops );
        return;
}


putjobcard()

/* This simple little routine copies the model jobcard file to the */
/* temporary output file.                                          */
{
        int  fd, i;
        char buf[BUFSIZE];

        if((fd = open(jobcf, 0))<0) {
                fprintf(stderr,"Open failed for %s\n",jobcf);
                cancel();
        }
        while((i=read(fd, buf, BUFSIZE))>0)
                fwrite(buf,i,1,outfile);
        close(fd);
        return;
}


putexecs()

/* This one output's the &MEMBER and &FILENAME keyword assignments */
/* and then copies the appropriate model jcl file to the temporary */
/* output file for each user file.                                 */
{
        int fd, i, fnum;
        char buf[BUFSIZE];

        for (fnum=0; fnum<filecnt; fnum++) {
                fprintf(outfile, "~&MEMBER=%s\n",
                        dfiles[files[fnum]].memb);
                fprintf(outfile, "~&FILENAME=%s\n",
                        dfiles[files[fnum]].filename);
                switch(mode) {
                case 0:
                        if((fd=open(plicf, 0))<0) {
                                fprintf(stderr,"Open failed for %s\n",
                                plicf);
                                cancel();
                        }
                        break;
                case 1:
                        if((fd=open(pliclf, 0))<0) {
                                fprintf(stderr,"Open failed for %s\n",
                                pliclf);
                                cancel();
                        }
                        break;
                case 2:
                        if(fnum<filecnt-1) {
                            if((fd=open(plicf, 0))<0) {
                                    fprintf(stderr,"Open failed for %s\n",
                                    plicf);
                                    cancel();
                            }
                        }
                        else {
                            if((fd=open(pliclgf, 0))<0) {
                                    fprintf(stderr,"Open failed for %s\n",
                                    pliclgf);
                                    cancel();
                            }
                            fprintf(outfile, "~&LINKOPS=\n");
                        }
                        break;
                }
                while((i=read(fd, buf, BUFSIZE))>0)
                        fwrite(buf,i,1,outfile);
                close(fd);
        }
        return;
}

callsend()

/* If sflag, then write the temp file to the standard output.  Else */
/* if qflag then invoke the GATH command passing it the temp file.  */
/* Else, invoke the SEND command passing it the temp file.          */
{
        int  fd, i, try, m;
        char buf[BUFSIZE];
        if (sflag) {
                fd = open(tempfile, 0);
                while((i=read(fd, buf, BUFSIZE))>0)
                        fprintf(stderr,"%s\n",buf);
                close(fd);
                return;
        }
        for (try=0; try<10; try++) {
                if ((i=fork()) >= 0) break;
                sleep(2);
        }
        if (i < 0) {
                fprintf(stderr, "Cannot fork to invoke SEND.\n");
                return;
        }
        if (i == 0) {     /* child process */
                if (qflag)
                        execlp("gath","gath",tempfile, 0);
                else
                        execlp("send","send",tempfile, 0);
                fprintf(stderr, "SEND invocation failed.\n");
        }
        /* parent process waits for SEND to finish */
        while(i != wait(&m));
        return;
}

cancel()

/* some terminal error occured ... cleanup the temp file and exit */
{
        fclose(outfile);
        unlink(tempfile);
        exit(1);
}
