/*****************************************************************************
 *****************************************************************************
 Copyright (c) 1999 - 2000, Intel Corporation 

 All rights reserved.

 Redistribution and use in source and binary forms, with or without 
 modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.

 3. Neither the name of Intel Corporation nor the names of its contributors 
    may be used to endorse or promote products derived from this software 
    without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 

 *****************************************************************************
 ****************************************************************************/

 /***************************************************************************
 **                                                                        **
 ** INTEL CORPORATION                                                      **
 **                                                                        **
 ** This software is supplied under the terms of the license included      **
 ** above.  All use of this software must be in accordance with the terms  **
 ** of that license.                                                       **
 **                                                                        **
 **  Module Name:                                                          **
 **    ians_kernel.c                                                       **
 **                                                                        **
 **  Abstract:                                                             **
 **    This module contains all the functions needed for the interface of  **
 **    the iANS module with the kernel.                                    **
 **                                                                        **
 **  Environment:                                                          **
 **    Kernel Mode (linux 2.2.x)                                           **
 **                                                                        **
 ***************************************************************************/

#define _IANS_MAIN_MODULE_C_
//////////////////////////////////////////////////////////
//must allways stay the first lines in the file!
#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif MODVERSIONS
//////////////////////////////////////////////////////////
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/malloc.h>


#include <linux/init.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/dma.h>

#include <linux/malloc.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>

#include <linux/version.h>
#include <linux/rtnetlink.h>



char kernel_version [] = UTS_RELEASE;


#include "base_comm.h"
#include "incg_utils.h"
#include "ians_base.h"
#include "ians_kernel.h"
#include "incg_net.h"
#include "ans_interface.h"
#include "incg_flow.h"
#include "pubDefs.h"
#include "status.h"



#define netif_stop_queue(dev)    (   set_bit(0, &(dev)->tbusy))
#define netif_wake_queue(dev)    { clear_bit(0, &(dev)->tbusy); \
                                                        mark_bh(NET_BH); }
#define MAX_ADAPTERS 100

static struct device *control_g = NULL;
static struct device *vAdapters_g[MAX_ADAPTERS];
static struct device *baseDrivers_g[MAX_ADAPTERS];

int globalRcvProbeNotOnTime;

// Forward declaration
#define DEST_ADDR_PTR(osMsg, flow)  \
 (flow == TRANSMIT ? (MAC_ADDR *)(osMsg->data) : \
                     (MAC_ADDR *)(osMsg->mac.ethernet->h_dest));

#define SRC_ADDR_PTR(osMsg, flow) \
  (flow == TRANSMIT ? ((MAC_ADDR *)(osMsg->data) + 1) :\
		       (MAC_ADDR *)(osMsg->mac.ethernet->h_source));

struct device *iansInitEtherdev( const char *name );

static int iansVadapterOpen(struct device *dev);

static int iansVadapterStop ( struct device *dev );

static int iansVadapterSend (struct sk_buff *skb, struct device *dev);

static struct enet_statistics *iansVadapterGetStats ( struct device *dev); 

static void  iansVadapterSetMulticastList ( struct device *dev );

static int iansVadapterDoIoctl ( struct device *dev, struct ifreq *rq, int cmd);
static int iAnsOpenReceive (struct sk_buff *skb, struct device *dev, struct packet_type *pt);


#ifdef INCG_DEBUG
	unsigned long GlobalDebugLevel;
#endif


static struct packet_type iansPacketType =
{
	IANS_FRAME_TYPE,                        // Note : no need for htons !!!
        NULL,                                   // Null = All devices 
	iAnsOpenReceive, // was iansReceive,  // receive function
        NULL,                                   // Private to the packet type          
        NULL,                                   // next
};





inline int _convertOsMCListToAnsMCList(struct dev_mc_list *osList,
                                int osCount,
                                MULTICAST_LIST *ansList)
{

    struct dev_mc_list *currList;
    int res;

    debugIoctlFine(("start"));    
    
    ASSERT(osList && ansList);

    for(currList = osList; currList != NULL; currList = currList->next) {
        res = ANSAddMulticastAddress (ansList,
                    (MAC_ADDR*)currList->dmi_addr,
                    currList->dmi_addrlen,
                    currList->dmi_users,
                    NULL);

        if (res != FLOW_OK) {
	  if (res == FLOW_GP_RESOURCES)
	    debugIoctlErr(("FLOW_GP_RESOURCES "));
	  debugIoctlErr(("iNCGAddrAddMulticastAddress failed !!!"));
	  return IANS_STAM;
        }
    }

    ASSERT(osCount == ansList->NumAddresses);


    debugIoctlFine(("end"));

    return IANS_OK;
}


inline void _convertOsPFToAnsPF(unsigned short osFlags, PACKET_FILTER *packetFilter){

    if(osFlags & IFF_PROMISC) {
        *packetFilter |= PF_PROMISCUOUS_BIT;
    }
    else {
        *packetFilter &= ~(PF_PROMISCUOUS_BIT);
    }

    if(osFlags & IFF_ALLMULTI) {
        *packetFilter |= PF_ALLMCA_BIT;
    }
    else {
        *packetFilter &= ~(PF_ALLMCA_BIT);
    }
}




inline void _convertOsMessageToAnsMessage(pOsMessage_t osMsg, message_t *ansMsg, IANS_FLOW_MODE flow) {

 
  ASSERT(osMsg && ansMsg);
  
  ansMsg->pOsMessage = osMsg;
  ansMsg->pProtocol = &(osMsg->protocol);  
  ansMsg->Head = osMsg->head;                 
  ansMsg->Data = osMsg->data;                
  ansMsg->Len = osMsg->tail - osMsg->data;      
  ansMsg->pDest = DEST_ADDR_PTR(osMsg, flow);
  ansMsg->pSrc = SRC_ADDR_PTR(osMsg, flow); 

}





// --------------------------------------------------------------------------------
/* 
*  Here we accept the control IOCTLS. 
*/


static int doControlIoctl ( struct device *dev, struct ifreq *rq, int cmd)
{
  int res;

  char *data;

  // ---- Lock the bh ----
  start_bh_atomic();

  data = rq->ifr_ifru.ifru_data;

#ifdef TESTING_LOCKS
  configSetLockAction(CONFIG_ACTION);
#endif

  if ( cmd != SIOCDEVPRIVATE ) {
      res = IANS_OS_ERROR;
      debugAll(("Wrong Ioctl"));
      goto exitDoControlIoctl;
    } 

  if ( dev != control_g ) {
    debugAll ( ("Got control IOCTL on non control device : dev->name %s\n",dev->name));
    res = IANS_OS_ERROR;
    goto exitDoControlIoctl;

  }
  if ( data == NULL ) {
    debugAll (( "Got null data in control IOCTL" ));
    res = IANS_OS_ERROR;
    goto exitDoControlIoctl;
  }

  ANSSetConfigExecuteCommand(1);
  res=ANSDoConfigIoctl( data );
  ANSSetConfigExecuteCommand(0);
  
  //currently only commands to add a virtual adapter
  //IANS_DEBUG ( "Add an adapter on eth :  %s\n" , rq->ifr_ifru.ifru_data );
  //addVadapterOnBase ( rq->ifr_ifru.ifru_data );

 exitDoControlIoctl:
#ifdef TESTING_LOCKS
  configReleseLock(CONFIG_ACTION);
#endif
	
  // ---- Free BH lock ----
  end_bh_atomic();


  return res;
}


// --------------------------------------------------------------------------------


// Creates the ANS subtitude for the kernel's struct device
// for an adapter.

int iansInitDev(const char *baseName, device_t **pDev){

  struct device *dev;
  
  if (  (dev = dev_get( baseName ) ) == NULL ) {
	IANS_ERROR (("Can't find the base driver.\n"));
	return 	IANS_MEMBER_NAME_WRONG;
  }

  //create the related Ans device.
  if ( (*pDev = kmalloc( sizeof(device_t) , GFP_KERNEL) ) == NULL ) {
    IANS_ERROR ( "failed to allocate ans device_t" );
    return IANS_MEM_ERROR;
  }
  
  (*pDev)->pOsDev = dev;   
  (*pDev)->name = dev->name;
  (*pDev)->mac = *( (MAC_ADDR *)(dev->dev_addr) );
  (*pDev)->open = baseOpenDevice;
  (*pDev)->close = baseCloseDevice;
  //(*pDev)->send = (int (*) (void *, void *) )dev->hard_start_xmit;
  (*pDev)->doIoctl = baseDoIoctl; 
  (*pDev)->setMacAddr = baseSetMacAddr;
  (*pDev)->getStats = baseGetStats;
  (*pDev)->setMulticastList = baseSetMulticastList;
	
  return IANS_OK;
}



BOOLEAN iansBaseDevIsOpen(device_t *dev) {

  ASSERT(dev);
  ASSERT(dev->pOsDev);

 //printk("TEST: dev->start %d\n", dev->pOsDev->start);
  
  if ((dev->pOsDev->start) || (dev->pOsDev->flags & IFF_UP))
    return TRUE;
  return FALSE;
 

}




unsigned long iansGetDeviceBaseAddr(device_t *dev) {

      ASSERT(dev);

      return dev->pOsDev->base_addr;
}



//----------------------------------------------------------------------
//Alocates a kernel's struct device for a vadapter

struct device *iansDevAlloc( const char *name ) {
       
	int allocSize = sizeof(struct device)+16;
	struct device *dev;
	
	allocSize &= ~3;	
	if ( (dev = kmalloc( allocSize , GFP_ATOMIC) ) == NULL ) {
                IANS_ERROR ( "kmalloc failed" );
		return NULL;
	}
	
	memset( dev , 0 , allocSize );

	
	dev->name=(char *)(dev+1);	// Name string space 
	iAnsStrCpy(dev->name,name);
     

	if ( dev_get( dev->name) != NULL ) {
	  debugAll ( ( "device with this name already exists\n" ) );
	  kfree(dev);
	  return NULL;
	}

	return dev;
}

// --------------------------------------------------------------------------------

struct device *iansInitEtherdev( const char *name ) {
  
	struct device *dev;
	
	if ( (  dev = iansDevAlloc ( name ) ) == NULL )
		return NULL;

	ether_setup(dev); 	
	
	register_netdevice(dev);

	return dev;
}


int iansIfReceive(device_t *dev, message_t *msg) {
      struct sk_buff *skb;
      int FlowStatus = FLOW_DOESNT_EXIST;

      debugRxFine(("start"));
      
      ASSERT(dev && msg);
      skb = msg->pOsMessage;
      ASSERT(skb);

      // set to vAdapter device
      skb->dev = (struct device *) dev->pOsDev;

      ASSERT(skb->dev);
         
      if (skb->protocol == IANS_FRAME_TYPE)
	debugRxErr(("Packet has Intel protocol and not original"));

#ifdef TESTING_LOCKS
      configReleseLock(RX_ACTION);
#endif

      netif_rx (skb);	  // pass the packet up the stack 
      
      debugRxFine(("after netif_rx"));

      FlowStatus = FLOW_TRANSFERED;

      debugRxFine(("end"));
            
      return ( FlowStatus );

}

void iansIfStop(device_t *ansDev) {

  netif_stop_queue(GET_OS_DEV(ansDev));

}


void iansIfWake(device_t *ansDev) {

 netif_wake_queue(GET_OS_DEV(ansDev));

}


// --------------------------------------------------------------------------------

static int dummyFunction1 ( struct device *dev ) {
	debugAll (( "got open/stop on control device" ));
	return 0;
}
static struct enet_statistics *dummyFunction2 ( struct device *dev ) {
	debugAll (( "got get_statistics for control device" ));
	return NULL;
}
static void  dummyFunction3 ( struct device *dev ) {
	debugAll (( "got set_multicast for control device" ));
}
static int dummyFunction4 (struct sk_buff *skb, struct device *dev) {
	debugAll (( "got hard_start_xmit for control device" ));
	return 0;
}
// --------------------------------------------------------------------------------

__initfunc( int init_module(void) ) 
{
        int res=0;

	res = ANSConfigCreate();
	if (res!=IANS_OK)
		return -1;

	memset ( vAdapters_g , 0 , sizeof ( struct device * ) * MAX_ADAPTERS );
	memset ( baseDrivers_g , 0 , sizeof ( struct device * ) * MAX_ADAPTERS );	

	if ( ( control_g = iansInitEtherdev( "ians" ) ) == NULL ) {
		IANS_ERROR ( "Init etherdev failed. module not loaded.\n" );
		return -1;
	}

	control_g->open               =  dummyFunction1;
	control_g->hard_start_xmit    =  dummyFunction4;
	control_g->stop               =  dummyFunction1;
	control_g->get_stats          =  dummyFunction2;
	control_g->set_multicast_list =  dummyFunction3;

	control_g->do_ioctl           =  doControlIoctl;
	control_g->start              =  0;

	dev_add_pack ( & iansPacketType );
        
	debugAll (( "ANS loaded.\n" ));

	return 0;
}


// --------------------------------------------------------------------------------

void cleanup_module(void) {
  
	int res;
 
	unregister_netdev( control_g );
	     
        //remove all the topology.
	res = ANSRemoveAll();
	if (res != IANS_OK) {
		debugConfigFine((" Failed to remove topology"));
	}
	  
	
	dev_remove_pack( & iansPacketType );

	if (globalRcvProbeNotOnTime > 0)
	  debugInfoAll(("globalRcvProbeNotOnTime %d \n",globalRcvProbeNotOnTime));

	debugAll (( "module removed.\n" ));
}


static int iansVadapterOpen(struct device *dev) {
  device_t *ansDev = NULL; 

  MOD_INC_USE_COUNT;
  dev->start = 1;

  ansDev = ANSGetVadapterAnsDevice(dev->name);
  ASSERT(ansDev);

  ansDev->open(ansDev);
       	
#ifdef TESTING_LOCKS
	configReleseLock(IOCTL_ACTION);
#endif
	return 0;
}




static  int iansVadapterStop ( struct device *dev )
{
 
#ifdef TESTING_LOCKS
	configSetLockAction(IOCTL_ACTION);
#endif
	
      	debugIoctlFine ( ( "iansVadapterStop()\n" ) );
	MOD_DEC_USE_COUNT;
	dev->start = 0;
       
#ifdef TESTING_LOCKS
	configReleseLock(IOCTL_ACTION);
#endif
	return 0;
}


static struct net_device_stats *iansVadapterGetStats ( struct device *dev )
{
 device_t *ansDev = NULL;
 
  debugIoctlFine(("start"));

  ASSERT(dev);

#ifdef TESTING_LOCKS
 configSetLockAction(IOCTL_ACTION);
#endif

 ansDev = ANSGetVadapterAnsDevice(dev->name);
 ASSERT(ansDev);

  debugIoctlFine(("end"));

#ifdef TESTING_LOCKS
	configReleseLock(IOCTL_ACTION);
#endif
        
 return (struct net_device_stats *) ansDev->getStats(ansDev);
      
} 


// This function handles both the packet filter and the multicast list.
static void  iansVadapterSetMulticastList ( struct device *dev )
{
  MULTICAST_LIST multicastList;
  PACKET_FILTER  packetFilter;
  device_t *ansDev = NULL;
  int res;

#ifdef TESTING_LOCKS
    configSetLockAction(IOCTL_ACTION);
#endif

    debugIoctlFine(("start"));
    
    ASSERT(dev);

    memset( &multicastList , 0 , sizeof(MULTICAST_LIST) );


    if(dev->mc_list){
        ASSERT(dev->mc_count);
        res = _convertOsMCListToAnsMCList(dev->mc_list, dev->mc_count, &multicastList);
        if(res != IANS_OK) {
            debugIoctlErr(("can't execute set multicast list"));
            return;
        }
    }

    _convertOsPFToAnsPF(dev->flags, &packetFilter);

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    ASSERT(ansDev);

    res = ansDev->setMulticastList(ansDev, &multicastList, packetFilter);
    if(res != IANS_OK) {
            debugIoctlErr(("Set multicast list failed!"));
            return;
    }

    debugIoctlFine(("end"));
    
#ifdef TESTING_LOCKS
    configReleseLock(IOCTL_ACTION);
#endif
}


static int iansVadapterDoIoctl ( struct device *dev, struct ifreq *rq, int cmd) {
	device_t *ansDev = NULL;
	int res;

	debugIoctlFine(("start"));

	ASSERT(dev);
	
	// ---- Lock the bh ----
	start_bh_atomic();

#ifdef TESTING_LOCKS
	configSetLockAction(IOCTL_ACTION);
#endif
      
	debugIoctlFine ( ( "iansVadapterDoIoctl() : commad = %d\n" , cmd ) );

	if ( cmd == SIOCDEVPRIVATE )  // should not get private ioctls here.
	  {
	    // ---- Free BH lock ----
	    end_bh_atomic();
	    return 0;
	  }
	ansDev = ANSGetVadapterAnsDevice(dev->name);
	ASSERT(ansDev);

	res = ansDev->doIoctl(ansDev, rq, cmd);
	
#ifdef TESTING_LOCKS
	configReleseLock(IOCTL_ACTION);
#endif

	// ---- Free BH lock ----
	end_bh_atomic();

	debugIoctlFine(("end"));

	return res;
}


void iansVadapterSetMac(device_t *ansDev) {
  struct device *dev = NULL;

  debugIoctlFine(("start"));
  
  ASSERT(ansDev);

  dev = ansDev->pOsDev;
  ASSERT(dev);
  iNCGMacAddrCopy(dev->dev_addr, &(ansDev->mac)); 
  //*((MAC_ADDR *) dev->dev_addr) = ansDev->mac;
  
  debugIoctlFine(("end"));
  
}




static int iAnsOpenReceive (struct sk_buff *skb, struct device *dev, struct packet_type *pt)
{
	int status = 0;
	static message_t msg;

	debugRxFine(("start"));

	ASSERT(skb && dev);
       
#ifdef TESTING_LOCKS
	configSetLockAction(RX_ACTION);
#endif

	if (skb->protocol != IANS_FRAME_TYPE) {
		debugRxErr(("Protocol not set to IANS frame type!\n"));
		debugRxErr(("Protocol is set to %x\n", skb->protocol));
		status = 0;
   	}
	else {
	        _convertOsMessageToAnsMessage(skb, &msg, RECEIVE); 
	        status = ANSReceive(&msg, dev->name,(pPacket_t) pt);
	}

	if (status == 1)
	  {
	    // there are 2 scenarios for this error:
	    //1. The Kernel has made a mistake and sent us a
            // packet that does not have our protocol.

	    //2. OR the packet has the ANS protocol but the 
      	    // adapter sending up this packet is not a member.
	    // This is eather a probe packet or some other packet
	    // that we don't recognize.
	    
	    // drop it.
	    dev_kfree_skb (skb);
	    status = 0;
	  }

	status = 0;


#ifdef TESTING_LOCKS
	configReleseLock(RX_ACTION);
#endif


	debugRxFine(("end"));
 
	return status;
}



int iansRegisterDev(const char *name, device_t **ansVadapterDev) {
  struct device  *newVadapterDev;
	
  ASSERT(name);

  debugConfigFine(("start"));

  
  if ( ( newVadapterDev = iansInitEtherdev( name ) ) == NULL ) {
		debugConfigErr(("fail to register vadapter"));
		return IANS_VADAPTER_REGISTER_FAIL;
  }


  debugConfigFine(("register vadapter %s", name));

  // The ANS-specific entries in the device structure.
  newVadapterDev->open               =  iansVadapterOpen;
  newVadapterDev->hard_start_xmit    =  iansVadapterSend;
  newVadapterDev->stop               =  iansVadapterStop;
  newVadapterDev->get_stats          =  iansVadapterGetStats;
  newVadapterDev->set_multicast_list =  iansVadapterSetMulticastList;
  newVadapterDev->do_ioctl           =  iansVadapterDoIoctl;
  newVadapterDev->start              =  0;


 if ( (*ansVadapterDev = kmalloc( sizeof(device_t), GFP_KERNEL) ) == NULL ) {
    IANS_ERROR ( "failed to allocate ans device_t" );
    return IANS_MEM_ERROR;
 }
  
  (*ansVadapterDev)->pOsDev = newVadapterDev;   
  (*ansVadapterDev)->name = newVadapterDev->name;
  (*ansVadapterDev)->IfReceive = (void (*)(void *))&netif_rx;
  
  return IANS_OK;

}


void iansUnregisterDev(device_t *dev,BOOLEAN close) {
	char* name;
	struct device * devExist;

	
	ASSERT(dev);

	name = dev->name;
	ASSERT(name);
	if  (( devExist = dev_get( dev->name) )) {
		if (devExist != dev->pOsDev) {
			debugConfigFine(("unregister wasn't executed since devices are different"));
		}
		else {
			
			if (close)
				dev_close ( devExist );

			if (ANSGetConfigExecuteCommand() > 0)
				rtnl_unlock();
			
			unregister_netdev ( devExist );

			if (ANSGetConfigExecuteCommand() > 0)
				rtnl_lock();        

			debugConfigFine(("after unregister_netdev"));
		    if (dev_get( name) != NULL)
				debugConfigErr(("unregister FAILED"));
		}
	} 
	else 
	  iNCGLogPrint (( "try to unregister an non-exist device (%s) ", dev->name ) );
		
}



static int iansVadapterSend (struct sk_buff *skb, struct device *dev) {
  device_t *ansDev;
  static message_t msg;  //automatic variable to save allocation time. 
  int res;

  debugTxFine(("start"));
  
  ASSERT(skb && dev);
  
#ifdef TESTING_LOCKS
  configSetLockAction(TX_ACTION);
#endif
  
  ASSERT(dev->name);

  ansDev = ANSGetVadapterAnsDevice(dev->name);

  ASSERT(ansDev);
  ASSERT(((ansDev->pOsDev) == dev)) 
  
  _convertOsMessageToAnsMessage(skb, &msg, TRANSMIT);

  res = ansDev->send(&msg, ansDev);

  debugTxFine(("end"));
  
  return res;

}





void  iansConvertProbeMessageToAnsMessage(pOsMessage_t osMsg, message_t *ansMsg, IANS_FLOW_MODE flow) {


  _convertOsMessageToAnsMessage(osMsg, ansMsg, flow);

}


int iansGetSizeofStats(void){
  return (int)sizeof(struct net_device_stats);
}


void iansAddStats(pstats_t destStats, pstats_t srcStats) {

  ASSERT(destStats && srcStats);

  destStats->rx_packets += srcStats->rx_packets;
  destStats->tx_packets += srcStats->tx_packets;
  destStats->rx_bytes += srcStats->rx_bytes;
  destStats->tx_bytes += srcStats->tx_bytes;
  destStats->rx_errors += srcStats->rx_errors;
  destStats->tx_errors += srcStats->tx_errors;
  destStats->rx_dropped += srcStats->rx_dropped;
  destStats->tx_dropped += srcStats->tx_dropped;
  destStats->multicast += srcStats->multicast;
  destStats->collisions += srcStats->collisions; 
  destStats->rx_length_errors += srcStats->rx_length_errors;
  destStats->rx_over_errors += srcStats->rx_over_errors;
  destStats->rx_crc_errors += srcStats->rx_crc_errors;
  destStats->rx_frame_errors += srcStats->rx_frame_errors;
  destStats->rx_fifo_errors += srcStats->rx_fifo_errors;
  destStats->rx_missed_errors += srcStats->rx_missed_errors;

  /* detailed tx_errors */
  destStats->tx_aborted_errors += srcStats->tx_aborted_errors;
  destStats->tx_carrier_errors += srcStats->tx_carrier_errors;
  destStats->tx_fifo_errors += srcStats->tx_fifo_errors;
  destStats->tx_heartbeat_errors += srcStats->tx_heartbeat_errors;
  destStats->tx_window_errors += srcStats->tx_window_errors;
	

  /* for cslip etc */
  destStats->rx_compressed += srcStats->rx_compressed;
  destStats->tx_compressed += srcStats->tx_compressed;
 
}


void iansResetStats(pstats_t stats) {

  ASSERT(stats);
  memset((void *)(stats), 0, sizeof(struct net_device_stats));  

}
