/*****************************************************************************
 *****************************************************************************
 Copyright (c) 1999 - 2000, Intel Corporation 

 All rights reserved.

 Redistribution and use in source and binary forms, with or without 
 modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.

 3. Neither the name of Intel Corporation nor the names of its contributors 
    may be used to endorse or promote products derived from this software 
    without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 

 *****************************************************************************
 ****************************************************************************/

 /***************************************************************************
 **                                                                        **
 ** INTEL CORPORATION                                                      **
 **                                                                        **
 ** This software is supplied under the terms of the license included      **
 ** above.  All use of this software must be in accordance with the terms  **
 ** of that license.                                                       **
 **                                                                        **
 **  Module Name:                                                          **
 **    ians_base.c                                                         **
 **                                                                        **
 **  Abstract:                                                             **
 **    This module contains all the functions needed for the interface of  **
 **    the iANS module with the base driver.                               **
 **                                                                        **
 **  Environment:                                                          **
 **    Kernel Mode  (linux 2.2.x)                                          **
 **                                                                        **
 ***************************************************************************/

//////////////////////////////////////////////////////////
//must allways stay the first lines in the file!
#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif MODVERSIONS
//////////////////////////////////////////////////////////
#include <linux/kernel.h>
#include <linux/socket.h>
#include <linux/if.h>
#include <linux/ioctl.h>
#include <linux/netdevice.h>
#include <linux/string.h>

#include "incg_utils.h"
#include "ians_base.h"
#include "base_comm.h"
#include "base_comm_os.h"

//#include "incg_misc.h"
#include "incg_gp_mem.h"
#include "incg_types.h"

#include "ans_interface.h"
#include "pubDefs.h"
#include "status.h"

int _convertAnsMCListToOsMCList(struct dev_mc_list **osList, int *osCount, MULTICAST_LIST *ansList);
void _convertAnsPFToOsPF(unsigned short *osFlags, PACKET_FILTER packetFilter);
int _addOSMCList(struct dev_mc_list **mcList, int *mcCount);
void _freeOSMCList(struct dev_mc_list **mcList, int *mcCount);
void iAnsNotify(struct device *dev, int IndType);





int baseOpenDevice(device_t *ansDev) {

  struct device *dev  = (struct device *) GET_OS_DEV(ansDev);
  ASSERT(dev);

  //calles the open device function in base driver.
  if (dev->open ( dev ) != 0) {
    debugConfigErr(("Error failed to open member"));
    return IANS_MEMBER_OPEN_FAILED;
  }

  return IANS_OK;

}


int baseCloseDevice(device_t *ansDev) {

  struct device *dev = (struct device *) GET_OS_DEV(ansDev);
  ASSERT(dev);
  
  if (dev->stop ( dev ) != 0) 
    return IANS_MEMBER_CLOSE_FAILED;
	
  return IANS_OK;
 
}



int baseSend(device_t *ansDev, message_t *msg) {

  struct device *dev = (struct device *) GET_OS_DEV(ansDev);
  struct sk_buff *skb = (struct sk_buff *) GET_OS_MSG(msg); 

  ASSERT(dev);
  ASSERT(skb);

  if (dev->hard_start_xmit (skb ,dev ) != 0)
    return IANS_STAM;

  return IANS_OK;
}


int baseDoIoctl(device_t *ansDev, preq_t ifr, int cmd) {

  struct device *dev = (struct device *) GET_OS_DEV(ansDev);
  ASSERT(dev && ifr);
  
  if (dev->do_ioctl ( dev,(struct ifreq * )ifr, cmd ) != 0)
    return IANS_STAM;

  return IANS_OK;
}


int baseSetMacAddr(device_t *ansDev, sockAddr_t *addr) {

  struct device *dev = (struct device *) GET_OS_DEV(ansDev);
  ASSERT(dev && addr);

  if (dev->set_mac_address ( dev, (struct sockaddr *) addr) != 0)     
    return IANS_STAM;
  
  return IANS_OK;
		    
}


pstats_t  baseGetStats(device_t *ansDev) {

  struct device *dev = (struct device *) GET_OS_DEV(ansDev);
  struct net_device_stats *stats;
  
  ASSERT(dev);
  
  stats = dev->get_stats ( dev );

  if (stats == NULL) 	
    return NULL;
  
   return (pstats_t ) stats;

}


int baseIdentify (device_t *ansBase ) {
        
	struct device *base = (struct device *) GET_OS_DEV(ansBase);  
	int status;
	struct ifreq ifr;
	IANS_BD_PARAM_IDENTIFY id;

 	id.Header.Opcode = IANS_OP_BD_IDENTIFY;
	memcpy(id.iANSSignature, IntelCopyrightString, IANS_SIGNATURE_LENGTH); 	
	id.iANSCommVersion = IANS_COMM_VERSION_MINOR;
	id.iANSCommVersion |= (IANS_COMM_VERSION_MAJOR << 16);
	ifr.ifr_data = (void *) &id;

        
	if (base->do_ioctl == NULL) {
		debugAll(("base->do_ioctl == NULL"));
		return IANS_OS_ERROR;
	}

	if ( ( status = base->do_ioctl(base, &ifr, IANS_BASE_SIOC) ) < 0 ) {
		// the driver did not recognize the ioctl
	  iAnsKernelPrint(INGC_ERR_LEVEL,
			  "base did not recognize identify ioctl\n");
		return status;		
	}
	return 0;
}


int baseSetExtMode (device_t *ansBase )
{
        struct device *base = (struct device *) GET_OS_DEV(ansBase);
	int status;
	struct ifreq ifr;
	IANS_BD_PARAM_EXT_SET_MODE request;


	debugConfigFine(("start"));

	request.Header.Opcode = IANS_OP_EXT_SET_MODE;
	request.BDIansStatusReport = IANS_REQUEST_SUPPORT;
#ifdef IANS_TAGGING	
	request.BDIansAttributedMode = IANS_REQUEST_SUPPORT;
#else
	request.BDIansAttributedMode = IANS_DONT_SUPPORT;
#endif	
	request.BDIansRoutingMode = IANS_ROUTING_RX_PROTOCOL;

	ifr.ifr_data = (void *) &request;

	if ( ( status = base->do_ioctl(base, &ifr, IANS_BASE_SIOC) ) < 0 ) {
		// the driver did not recognize the ioctl
		iAnsKernelPrint(INGC_ERR_LEVEL,
				"base failed IANS_OP_EXT_SET_MODE\n");
		return status;
	}
	

	debugConfigFine(("end   status=%d",status));
	

	return 0;
}


int baseGetExtendedCapability ( device_t *ansBase, IANS_BD_PARAM_EXT_CAP *cap) {
	struct device *base = (struct device *) GET_OS_DEV(ansBase);
	int status;
	struct ifreq ifr;

	debugConfigFine(("start"));
	
	cap->Header.Opcode = IANS_OP_EXT_GET_CAPABILITY;
	
	ifr.ifr_data = (void *) cap;
	
	if ( ( status = base->do_ioctl(base, &ifr, IANS_BASE_SIOC) ) < 0 ) {
		// the driver did not recognize the ioctl
	  iAnsKernelPrint(INGC_ERR_LEVEL,
			  "base did not recognize IANS_OP_EXT_GET_CAPABILITY\n");	
		return status;
	}

	debugConfigFine(("end"));

	return 0;
}


int baseGetStatusInfo ( device_t *ansBase, IANS_BD_IOC_PARAM_STATUS *stat) {
        struct device *base = (struct device *) GET_OS_DEV(ansBase);
	int status;
	struct ifreq ifr;

	//printk("ENTERED baseGetStatusInfo\n");
	
	stat->Header.Opcode = IANS_OP_EXT_GET_STATUS;
	
	ifr.ifr_data = (void *) stat;
	
	if ( ( status = base->do_ioctl(base, &ifr, IANS_BASE_SIOC) ) < 0 ) {
		// the driver did not recognize the ioctl
	  iAnsKernelPrint(INGC_ERR_LEVEL,
			  "base did not recognize IANS_OP_EXT_GET_STATUS\n");
		return status;
	}

	return 0;

}


#ifdef IANS_BINDING_BASED_VLAN
//base driver fills the cap structure with its vlan capabilities.
int baseGetVlanCapability ( device_t *ansBase, IANS_BD_PARAM_IVLAN_CAP *cap)
{
       struct device *base = (struct device *) GET_OS_DEV(ansBase);
       int status;
       struct ifreq ifr;
    
       cap->Header.Opcode = IANS_OP_IVLAN_ID_GET_CAPABILITY;

       ifr.ifr_data = (void *) cap;

       if ( ( status = base->do_ioctl(base, &ifr, IANS_BASE_SIOC) ) < 0 ) {
		// the driver did not recognize the ioctl
	 iAnsKernelPrint(INGC_ERR_LEVEL,
			 "base did not recognize IANS_OP_IVLAN_ID_GET_CAPABILITY\n");
	 
		return status;
	}
	return 0;
}

int baseSetVlanMode ( device_t *ansBase, IANS_BD_MODE  mode )
{
        struct device *base = (struct device *) GET_OS_DEV(ansBase);
        int status;
	struct ifreq ifr;
	IANS_BD_PARAM_IVLAN_SET_MODE request;

	request.Header.Opcode = IANS_OP_IVLAN_ID_SET_MODE;
	if (mode == IANS_BD_ON)
	  request.VlanIDRequest = IANS_REQUEST_SUPPORT;
	else
	  request.VlanIDRequest = IANS_DONT_SUPPORT;
	request.VlanIDFilteringRequest = IANS_DONT_SUPPORT;

	ifr.ifr_data = (void *) &request;

	if ( ( status = base->do_ioctl(base, &ifr, IANS_BASE_SIOC) ) < 0 ) {
		// the driver did not recognize the ioctl
	  iAnsKernelPrint(INGC_ERR_LEVEL,
			  "base did not recognize IANS_BD_PARAM_IVLAN_SET_MODE\n");   
		return status;
	}
	return 0;
}

int baseSetVlanIdTable(device_t *ansBase, UINT16 *vlanIdTable, UINT32 vlanNum){
        struct device *base = (struct device *) GET_OS_DEV(ansBase);
        int status;
	struct ifreq ifr;
	IANS_BD_PARAM_IVLAN_TABLE request;

	ASSERT(vlanIdTable);

	request.Header.Opcode = IANS_OP_IVLAN_ID_SET_TABLE;

	request.VLanIDNum = vlanNum;
	request.VLanIDTable = vlanIdTable;  

	ifr.ifr_data = (void *) &request;

	if ( ( status = base->do_ioctl(base, &ifr, IANS_BASE_SIOC) ) < 0 ) {
		// the driver did not recognize the ioctl
		iAnsKernelPrint(INGC_ERR_LEVEL, "base failed IANS_OP_IVLAN_ID_SET_TABLEIANS_OP_IVLAN_ID_SET_TABLE\n");
		return status;
	}
	return 0;
}

#endif


#ifdef IANS_TAGGING
int baseGetTaggingCapability ( device_t *ansBase, IANS_BD_PARAM_ITAG_CAP *cap)
{
       struct device *base = (struct device *) GET_OS_DEV(ansBase);
       int status;
       struct ifreq ifr;  
       
       cap->Header.Opcode = IANS_OP_ITAG_GET_CAPABILITY;

	ifr.ifr_data = (void *) cap;

	if ( ( status = base->do_ioctl(base, &ifr, IANS_BASE_SIOC) ) < 0 ) {
		// the driver did not recognize the ioctl
		iAnsKernelPrint(INGC_ERR_LEVEL,
				"base failed IANS_OP_ITAG_GET_CAPABILITY\n");
		return status;
	}
	return 0;

       
}

int baseSetTaggingMode ( device_t *ansBbase, IANS_BD_TAGGING_MODE TagMode )
{
       struct device *base = (struct device *) GET_OS_DEV(ansBase);
       int status;
       struct ifreq ifr;  
       IANS_BD_PARAM_ITAG_SET_MODE request;
   
       request.Header.Opcode = IANS_OP_ITAG_SET_MODE;
       request.SetTagMode = TagMode;

       ifr.ifr_data = (void *) &request;
	
	if ( ( status = base->do_ioctl(base, &ifr, IANS_BASE_SIOC) ) < 0 ) {
		// the driver did not recognize the ioctl
		iAnsKernelPrint(INGC_ERR_LEVEL,
				"base failed IANS_OP_ITAG_SET_MODE\n");
		return status;
	}
	return 0;
       
}
#endif


int baseUndoSettings (device_t *ansBase) {
       struct device *base = (struct device *) GET_OS_DEV(ansBase);
       int status;
       struct ifreq ifr;
       IANS_BD_PARAM_HEADER request;
   
       request.Opcode = IANS_OP_BD_DISCONNECT;
      

       ifr.ifr_data = (void *) &request;
	
	if ( ( status = base->do_ioctl(base, &ifr, IANS_BASE_SIOC) ) < 0 ) {
		// the driver did not recognize the ioctl
		iAnsKernelPrint(INGC_ERR_LEVEL,
				"base failed IANS_OP_BD_DISCONNECT\n");
		return status;
	}
	return IANS_OK;
  
}

int baseSetMulticastList(device_t *ansDev,
                         MULTICAST_LIST *multicastList,
                         PACKET_FILTER packetFilter)
{

    struct device *dev = (struct device *) GET_OS_DEV(ansDev) ;
    int res;

    _freeOSMCList(&dev->mc_list, &dev->mc_count);

    res = _convertAnsMCListToOsMCList(&dev->mc_list, &dev->mc_count, multicastList);
    if(res != IANS_OK) {
        debugIoctlErr(("_convertAnsMCListToOsMCList failed!!!"));
        return res;
    }

    ASSERT(dev->mc_count == multicastList->NumAddresses);

    _convertAnsPFToOsPF(&dev->flags, packetFilter);

   dev->set_multicast_list ( dev );

   return IANS_OK; 
}



void _freeOSMCList(struct dev_mc_list **mcList, int *mcCount) {

    struct dev_mc_list* tempMCList;

    while(*mcList) {
        tempMCList = *mcList;
        *mcList = (*mcList)->next;
        iNCGGPMemFree((void **)(&tempMCList));
    }

    *mcList = NULL;
    *mcCount = 0;
}


int _addOSMCList(struct dev_mc_list **mcList, int *mcCount) {

    struct dev_mc_list* tempMCList = NULL;
    
    iNCGGPSizedMemAlloc ((void **)(&tempMCList) , sizeof(struct dev_mc_list) );
    if (tempMCList == NULL) {
        return IANS_MEM_ERROR; 
    }

    tempMCList->next = *mcList;
    *mcList  = tempMCList;      
    *mcCount = *mcCount + 1;
    return IANS_OK;
}



int _convertAnsMCListToOsMCList(struct dev_mc_list **osList,
                                int *osCount,
                                MULTICAST_LIST *ansList)
{

    MULTICAST_ENTRY *currAnsEntry = ansList->pAddressesList;
    struct dev_mc_list *currOsList = NULL;
    int addrCount = 0;
    int res;

    while(currAnsEntry){

        res = _addOSMCList(&currOsList, &addrCount);
        if(res != IANS_OK) {
            debugIoctlErr(("_addOSMCList failed!!!"));
            _freeOSMCList(&currOsList, &addrCount);
            return res;
        }

        iNCGMacAddrCopy( currOsList->dmi_addr, currAnsEntry->MacAddr.MacAddr );
        currOsList->dmi_addrlen = currAnsEntry->dmiAddrLen;
        currOsList->dmi_users = currAnsEntry->dmiUsers;

        currAnsEntry = currAnsEntry->pNextEntry;
    }

    *osCount = addrCount;
    *osList = currOsList;
    return IANS_OK;
}


void _convertAnsPFToOsPF(unsigned short *osFlags, PACKET_FILTER packetFilter) {

    if(packetFilter & PF_PROMISCUOUS_BIT) {
        *osFlags |= IFF_PROMISC;
    }
    else {
        *osFlags &= ~(IFF_PROMISC);
    }

    if(packetFilter & PF_ALLMCA_BIT) {
        *osFlags |= IFF_ALLMULTI;
    }
    else {
        *osFlags &= ~(IFF_ALLMULTI);
    }
}


int baseSetNotifyCB(device_t *ansBase) {
  struct device *base = (struct device *) GET_OS_DEV(ansBase);
  int status;
  IANS_BD_ANS_SET_CB request;
  struct ifreq ifr;

  request.Header.Opcode = IANS_OP_ANS_SET_CB;
  request.notify = iAnsNotify;

  ifr.ifr_data = (void *) &request;
	
  if ( ( status = base->do_ioctl(base, &ifr, IANS_BASE_SIOC) ) < 0 ) {
    // the driver did not recognize the ioctl
    iAnsKernelPrint(INGC_ERR_LEVEL, "base failed IANS_OP_ANS_SET_CB\n");
    return status;
  }
  return 0;
   
}



void iAnsNotify(struct device *dev, int IndType) { 
IANS_BD_IOC_PARAM_STATUS status;
 device_t *ansDev; 

 //printk("ENTERED NOTIFICATION FUNCTION FOR DEVICE: %s\n",dev->name);
  ASSERT(dev);
  ASSERT(dev->name);

  ansDev = ANSGetMemberAnsDevice(dev->name);

  ASSERT(ansDev);

  switch (IndType) {
  case IANS_IND_EXT_STATUS_CHANGE: 
    //printk("GOING TO GET STATUS\n");
    if ( ( baseGetStatusInfo ( ansDev, &status ) ) != 0) {
      debugIndicateErr(("Collect status info failed"));
      return;
    }
    ANSNotify(ansDev, IndType, &status);
    break;
    
  case IANS_IND_XMIT_QUEUE_FULL:  // stop transmit through this member
       ANSNotify(ansDev, IndType, NULL);
       break;
       
  case IANS_IND_XMIT_QUEUE_READY:  // start transmit through this member
       ANSNotify(ansDev, IndType, NULL);
       break;
       
  default: 
   
    debugIndicateErr(("Not supported indication type"));
   
    
  }

}

