/*****************************************************************************
 *****************************************************************************
 Copyright (c) 1999 - 2000, Intel Corporation 

 All rights reserved.

 Redistribution and use in source and binary forms, with or without 
 modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.

 3. Neither the name of Intel Corporation nor the names of its contributors 
    may be used to endorse or promote products derived from this software 
    without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 

 *****************************************************************************
 ****************************************************************************/

 /***************************************************************************
 **                                                                        **
 ** INTEL CORPORATION                                                      **
 **                                                                        **
 ** This software is supplied under the terms of the license included      **
 ** above.  All use of this software must be in accordance with the terms  **
 ** of that license.                                                       **
 **                                                                        **
 **  THIS FILE IS INCLUDED BY ians_core.o. HENCE, IT SHOULD NOT BE         **
 **  MODIFIED!                                                             **
 **                                                                        **
 **  Module Name:                                                          **
 **    cfg.h                                                               **
 **                                                                        **
 **  Abstract:                                                             **
 **     A CONF_HANDLER is a pointer to a struct that holds a list of       **
 **     adapters installed in the PCI bus of the server, and a list of     **
 **     Teams created by the user. Each Adapter in the list holds the      **
 **     necessary information for binding it as a member of a Team.	   **
 **	Each Team holds lists of Members and Virtual adapters that are     **
 **     attached To it.                                        		   **
 **                                                                        **
 **     In BATCH mode, the user tasks are not done immediately but rather  **
 **     written to a batch file that can be executed afterwards.           **
 **                                                                        **   
 **	In the IMMEDIATE mode - all operations are executed ay Ans on the  **
 **     spot.	                                                           **
 **                                                                        **
 **  Environment:                                                         **
 **    Kernel Mode  (Linux 2.2.x)                                          **
 **                                                                        **
 ***************************************************************************/

#ifndef _CONFIG_API_H
#define _CONFIG_API_H


#include "pubDefs.h"
#include <stdio.h>

/**
 * operation mode of the configure module
 */
typedef enum {
   IMMEDIATE,
   BATCH
} CONF_MODE;

/**
 * init data structures and adapters' list.
 * if IANS is already loaded, get its current structure, <BR>
 * otherwise - init new structure
 */
CONF_HANDLER cfgInit(CONF_MODE mode);

/**
 * free the whole structure
 */
void cfgFree(CONF_HANDLER cfg);

/**
 * free and rebuild adapter list
 */
iansStatus_t cfgRefreshAdapterList(CONF_HANDLER cfg);

/**
 * get the current configuration of ans
 */
iansStatus_t cfgRefresh(CONF_HANDLER cfg);

/**
 * traverse list of adapters
 */
ADAPTER_HANDLER cfgGetFirstAdapter(CONF_HANDLER cfg);
ADAPTER_HANDLER cfgGetNextAdapter(CONF_HANDLER cfg);

/**
 * traverse list of teams
 */
TEAM_HANDLER cfgGetFirstTeam(CONF_HANDLER cfg);
TEAM_HANDLER cfgGetNextTeam(CONF_HANDLER cfg);

/**
 * allocate a new team with the given teamId
 * assign the newly allocated team to the team handler
 */
iansStatus_t cfgAddTeam(CONF_HANDLER cfg, const char* teamId, TEAM_HANDLER* team);

/**
 * get the team with the given Id
 */
TEAM_HANDLER cfgGetTeam(CONF_HANDLER cfg, const char* teamId);

/**
 * get the adapter with the given name
 */
ADAPTER_HANDLER cfgGetAdapterByName(CONF_HANDLER cfg, const char* adapterName);

/**
 * get the virtual adapter with the given name
 */
VADAPTER_HANDLER cfgGetVadapterByName(CONF_HANDLER cfg, const char* adapterName);

/**
 * delete a team
 */
iansStatus_t cfgRemoveTeam(CONF_HANDLER cfg, TEAM_HANDLER team);

/**
 * create member with the given adapter and add it to the given team
 */
iansStatus_t cfgAddMember(CONF_HANDLER cfg, TEAM_HANDLER team, ADAPTER_HANDLER adapter, IANS_MEMBER_ATTR priority);

/**
 * remove the member with the given adapter from the given team
 */
iansStatus_t cfgRemoveMember(CONF_HANDLER cfg, TEAM_HANDLER team, ADAPTER_HANDLER adapter);

/**
 * add virtual adapter with the given name and vlan id to the given team
 * in BATCH mode - if no vlan mode is defined, the vlan IDs are ignored in the script
 */
iansStatus_t cfgAddVadapter(CONF_HANDLER cfg, TEAM_HANDLER team, const char* vadapterName,int vlanId);

/**
 * the same as above, with a frame type
 * frameIndex is one of the following:
 *		0 - Ethernet_II
 *		1 - Ethernet_802.2
 *		2 - Ethernet_802.3
 *		3 - Ethernet_SNAP
 */
iansStatus_t cfgAddVadapterWithFrame(CONF_HANDLER cfg, TEAM_HANDLER team, 
									 const char* logicalName, int frameIndex, int vlanId);

/**
 * remove the given virtual adapter from the given team
 */
iansStatus_t cfgRemoveVadapter(CONF_HANDLER cfg, TEAM_HANDLER team, VADAPTER_HANDLER vadapter);

iansStatus_t cfgSetProbeEnable(CONF_HANDLER cfg, TEAM_HANDLER team, char on);

iansStatus_t cfgSetProbeAddrMode(CONF_HANDLER cfg, TEAM_HANDLER team, IANS_PROBE_ADDR_MODES mode);

iansStatus_t cfgSetProbeCheckTimeout(CONF_HANDLER cfg, TEAM_HANDLER team, int seconds);

iansStatus_t cfgSetProbeSendTime(CONF_HANDLER cfg, TEAM_HANDLER team, int seconds);

iansStatus_t cfgSetProbeMaxRetryCount(CONF_HANDLER cfg, TEAM_HANDLER team, int times);

iansStatus_t cfgSetProbeReceiveTimeout(CONF_HANDLER cfg, TEAM_HANDLER team, int seconds);

iansStatus_t cfgSetProbeRxBackCycles(CONF_HANDLER cfg, TEAM_HANDLER team, int nnnn); 

iansStatus_t cfgSetProbeBurstSize(CONF_HANDLER cfg, TEAM_HANDLER team, int size); 

#ifdef IANS_BINDING_BASED_VLAN 
   
/**
 * set the vlan mode of the given team
 */
iansStatus_t cfgSetVlanMode(CONF_HANDLER cfg, TEAM_HANDLER team, IANS_VLAN_MODE mode);
 
#endif

/**
 * set the teaming mode of the team
 */
iansStatus_t cfgSetTeamingMode(CONF_HANDLER cfg, TEAM_HANDLER team, IANS_TEAM_MODE mode);

/**
 * bind a virtual adapter up to IP.
 * addr and mask - 4 bytes - dot decimal notation
 */
iansStatus_t cfgBindVadapterToIP(CONF_HANDLER cfg, VADAPTER_HANDLER vadapter,
								 unsigned long addr[], unsigned long mask[]);

/**
 * bind a virtual adapter up to IP.
 * net - network number to bind to (0-255)
 */
iansStatus_t cfgBindVadapterToIPX(CONF_HANDLER cfg, VADAPTER_HANDLER vadapter,
								  unsigned long net);

/**
 * return pointer to the protocol structure
 */
PROTOCOL_PARAMETERS* cfgGetProtocolBoundToVadapter(CONF_HANDLER cfg, VADAPTER_HANDLER vadapter);

/**
 * write structure topology and data to file.
 */
iansStatus_t cfgPrint2File(CONF_HANDLER cfg, const char* filename);

/**
 * parse file and apply it to empty configutarion structure
 */
iansStatus_t cfgApplyFile(CONF_HANDLER cfg, const char* filename);

iansStatus_t cfgSetDebugLevel(int topic, int level, int function);

/*******   adapter functions   *****/

/**
 * get the PCI slot of the NIC of the adapter
 */
int adapterGetSlot(ADAPTER_HANDLER adpater); 

/**
 * get the card type of the adapter
 */
CONF_CARD_TYPE adapterGetCardType(ADAPTER_HANDLER adpater); 

/**
 * return TRUE if driver is currently loaded for this adapter
 */
char adapterIsDriverUp(ADAPTER_HANDLER adpater);

/**
 * return the member that uses this adapter (NULL if the adapter is not used)
 */
MEMBER_HANDLER adapterGetMember(ADAPTER_HANDLER adpater); 

/**
 * return the team that one of its members uses this adapter (NULL if the adapter is not used)
 */
TEAM_HANDLER adapterGetTeam(ADAPTER_HANDLER adpater); 

/**
 * convert the data of the adapter (Card name and slot) to a string
 */
iansStatus_t adapterToString(ADAPTER_HANDLER adapter, char* outString);

/**
 * get string representing adapter name
 */
iansStatus_t adapterGetName(ADAPTER_HANDLER adapter, char* name);

/**
 * set the "selected" field of an adapter
 * TRUE - the adapter was selected for a team (by the user or as default suggestion)
 * FALSE - otherwise
 */
void adapterSetSelected(ADAPTER_HANDLER adapter, char selection);

/**
 * Get the "selected" field
 * TRUE - the adapter was selected for a team (by the user or as default suggestion)
 * FALSE - otherwise
 */
char adapterGetSelection(ADAPTER_HANDLER adapter);

/**
 * description format: <FrameType name>  <logical name>
 * if the frame type is not supported - return < 0;
 */
int adapterGetLogicalBoardDescription(ADAPTER_HANDLER adapter, int frameIndex, char* outString);





/*******   member functions   *****/

/**
 * Get the priority of the member in its team
 */
IANS_MEMBER_ATTR memberGetPriority(MEMBER_HANDLER member);

/**
 * return the team that this members belongs to
 */
TEAM_HANDLER memberGetTeam(MEMBER_HANDLER member);

/**
 * return the adapter used by this member
 */
ADAPTER_HANDLER memberGetAdapter(MEMBER_HANDLER member);

/**
 * set the priority of the member within its team
 */
iansStatus_t memberSetRole(CONF_HANDLER cfg, MEMBER_HANDLER member, IANS_MEMBER_ATTR priority);


/*******   virtual adapter functions   *****/

/**
 * return the logical name of the virtual adapter
 */
char* vadapterGetName(VADAPTER_HANDLER vadapter);

/**
 * return the vlan Id (0 on non-vlan mode)
 */
int vadapterGetVlanId(VADAPTER_HANDLER vadapter);

/**
 * return the index of the frame type of this adapter
 * Frame index is one of the following:
 *		0 - Ethernet_II
 *		1 - Ethernet_802.2
 *		2 - Ethernet_802.3
 *		3 - Ethernet_SNAP
 */
int vadapterGetFrameIndex(VADAPTER_HANDLER vadapter);

/**
 * return the team that the virtual adapter belongs to
 */
TEAM_HANDLER vadapterGetTeam(VADAPTER_HANDLER vadapter);


/**
 * traverse members of the Team
 */
MEMBER_HANDLER teamGetFirstMember(TEAM_HANDLER team);
MEMBER_HANDLER teamGetNextMember(TEAM_HANDLER team);

/**
 * traverse virtual adapters of the Team
 */
VADAPTER_HANDLER teamGetFirstVadapter(TEAM_HANDLER team);
VADAPTER_HANDLER teamGetNextVadapter(TEAM_HANDLER team);

#ifdef IANS_BINDING_BASED_VLAN
/**
 * get the vlan tagging mode (IEEE or ISL)
 */
IANS_VLAN_MODE teamGetVlanMode(TEAM_HANDLER team);

#endif

/**
 * get the name (sometimes called teamId)
 */
const char* teamGetName(TEAM_HANDLER team);

/**
 * get the teaming mode (AFT, ALB, GEC, FEC, 3AD)
 */
IANS_TEAM_MODE teamGetTeamingMode(TEAM_HANDLER team);

/**
 * assign the available frame types to the boolean array AvailableFrames.
 * assign the previous choice for that team to the boolean array PrevChosen.
 */
iansStatus_t teamGetFrameTypeInfo(TEAM_HANDLER team, char AvailableFrames[], char PrevChosen[]);

/**
 * in immediate mode - execute the commit command
 */
iansStatus_t teamCommit(CONF_HANDLER cfg, TEAM_HANDLER team);

/**
 * in the batch mode - build the batch file and write it to the given file
 */
iansStatus_t teamBuildScript(TEAM_HANDLER team, FILE* scriptFile);

#endif // _CONFIG_API_H


