/*
 * COPYRIGHT:        See COPYRIGHT.TXT
 * PROJECT:          Ext2 File System Driver for WinNT/2K/XP
 * FILE:             shutdown.c
 * PROGRAMMER:       Matt Wu <mattwu@163.com>
 * HOMEPAGE:         http://ext2.yeah.net
 * UPDATE HISTORY: 
 */

/* INCLUDES *****************************************************************/

#include "ntifs.h"
#include "ext2fs.h"

/* GLOBALS ***************************************************************/

extern PEXT2_GLOBAL Ext2Global;

/* DEFINITIONS *************************************************************/

#ifdef ALLOC_PRAGMA
#pragma alloc_text(PAGE, Ext2ShutDown)
#endif

NTSTATUS
Ext2ShutDown (IN PEXT2_IRP_CONTEXT IrpContext)
{
    NTSTATUS                Status;

    PKEVENT                 Event;

    PIRP                    Irp;
    PIO_STACK_LOCATION      IrpSp;

    PEXT2_VCB               Vcb;
    PLIST_ENTRY             ListEntry;

    BOOLEAN                 GlobalResourceAcquired = FALSE;

    __try
    {
        ASSERT(IrpContext);
    
        ASSERT((IrpContext->Identifier.Type == EXT2ICX) &&
            (IrpContext->Identifier.Size == sizeof(EXT2_IRP_CONTEXT)));

        Irp = IrpContext->Irp;
    
        IrpSp = IoGetCurrentIrpStackLocation(Irp);

        if (!ExAcquireResourceExclusiveLite(
                &Ext2Global->Resource,
                IrpContext->IsSynchronous ))
        {
            Status = STATUS_PENDING;
            __leave;
        }
            
        GlobalResourceAcquired = TRUE;

        Event = ExAllocatePool( NonPagedPool, sizeof(KEVENT));
        KeInitializeEvent(Event, NotificationEvent, FALSE );

        for (ListEntry = Ext2Global->VcbList.Flink;
             ListEntry != &(Ext2Global->VcbList);
             ListEntry = ListEntry->Flink )
        {
            Vcb = CONTAINING_RECORD(ListEntry, EXT2_VCB, Next);

            if (ExAcquireResourceExclusiveLite(
                &Vcb->MainResource,
                TRUE ))
            {

                Status = Ext2FlushVolume(Vcb, TRUE);
                
                CcPurgeCacheSection(&Vcb->SectionObject,
                                    NULL,
                                    0,
                                    FALSE );

                Ext2DiskShutDown(Vcb);

                ExReleaseResourceForThreadLite(
                    &Vcb->MainResource,
                    ExGetCurrentResourceThread());
            }
        }
    }

    __finally
    {
        if (GlobalResourceAcquired)
        {
            ExReleaseResourceForThreadLite(
                &Ext2Global->Resource,
                ExGetCurrentResourceThread() );
        }

        if (!IrpContext->ExceptionInProgress)
        {
            if (Status == STATUS_PENDING)
            {
                Ext2QueueRequest(IrpContext);
            }
            else
            {
                IrpContext->Irp->IoStatus.Status = Status;
                
                Ext2CompleteRequest(
                    IrpContext->Irp,
                    (CCHAR)
                    (NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
                
                Ext2FreeIrpContext(IrpContext);
            }
        }
    }

    return Status;
}