/*
 * This is version 1.0 of "uredir", a program to redirect
 * UDP packets to a port on another host.  It works for me
 * under RedHat 4.2 (2.0.33), your mileage may vary.
 *
 * This is free software.  You can use it, modify it, give
 * it away, or sell it, but you cannot claim that you or
 * anybody else wrote it, and you cannot claim or attempt
 * to claim actual or perceived damages from me, my heirs,
 * my estate, or my company based on use of this code.
 *
 * Constructive comments and/or buckets of money are welcome.
 *
 * id est (mrn@shaw.wave.ca)
 */

#include <stdio.h>			/* NULL */
#include <signal.h>			/* signal(), SIGHUP, ... */
#include <string.h>			/* bzero(), strerror() */
#include <getopt.h>			/* getopt() */
#include <syslog.h>			/* openlog(), syslog() */
#include <sys/socket.h>		/* socket() */
#include <arpa/inet.h>		/* htonl(), ntohs(), INADDR_ANY, ... */
#include <netdb.h>			/* gethostbyname(), getservbyname(), ... */
#include <errno.h>			/* errno, EINTR, ... */
#include <sys/resource.h>	/* setpriority() */
#include <unistd.h>			/* fork(), chdir(), setsid() */
#include <stdlib.h>			/* atol() */
#include <sys/stat.h>		/* umask() */


#define MAX_BUFFER 4096


static char *progname = NULL,
            *target_host = NULL;

static int debug = 0,
           foreground = 0,
           s[2] = { -1, -1 };

static long local_port,
            target_port;

static unsigned long target_addr;

static void die(int),
            usage(void);


static void
daemon(void)
{
	(void) signal(SIGHUP,  SIG_IGN);
	(void) signal(SIGQUIT, SIG_IGN);
	(void) signal(SIGTERM, die);
	(void) signal(SIGTTIN, SIG_IGN);
	(void) signal(SIGTTOU, SIG_IGN);
	(void) signal(SIGTSTP, SIG_IGN);

	switch (fork()) {

	case -1:

		syslog(LOG_WARNING, "fork(), error %d, %m", errno);
		exit(1);
		/*NOTREACHED*/

	case 0:		/* child goes on */

		break;

	default:	/* and the parent exits */

		exit(0);
		/*NOTREACHED*/
	}

	if (setsid() < 0) {

		syslog(LOG_WARNING, "setsid(), error %d, %m", errno);
		exit(1);
		/*NOTREACHED*/
	}

	(void) chdir("/tmp");

	(void) umask(022);

	(void) setpriority(PRIO_PROCESS, 0, 10);

}	/* daemon(void) */


static void
die(int not_used)
{
	syslog(LOG_WARNING, "caught TERM signal, exiting ...");

	closelog();

	if (s[0] > -1) {

		(void) shutdown(s[0], 2);
		(void) close(s[0]);
	}

	if (s[1] > -1) {

		(void) shutdown(s[1], 2);
		(void) close(s[1]);
	}

	exit(0);
	/*NOTREACHED*/

}	/* die(int) */


static void
parse_args(int argc, char *argv[])
{
	extern int optind;
	int c;
	struct servent *s;

	while ((c = getopt(argc, argv, "dfh")) != -1)
		switch (c) {

		case 'd':

			debug = 1;
			break;

		case 'f':

			foreground = 1;
			break;

		case 'h':

			usage();
			exit(0);
			/*NOTREACHED*/

		default:

			(void) fprintf(stderr, "\n%s: unknown flag \"%c\"\n",
				progname, c);

			usage();
			exit(1);
			/*NOTREACHED*/
		}

	if ((argc - optind) != 3) {

		usage();
		exit(1);
		/*NOTREACHED*/
	}

	if ((s = getservbyname(argv[optind], "udp")) != NULL)
		local_port = ntohs(s->s_port);
	else
		local_port = atol(argv[optind]);

	if (debug)
		syslog(LOG_WARNING, "local port = %ld", local_port);

	optind++;

	target_host = argv[optind];

	if (debug)
		syslog(LOG_WARNING, "target host = %s", target_host);

	optind++;

	if ((s = getservbyname(argv[optind], "udp")) != NULL)
		target_port = ntohs(s->s_port);
	else
		target_port = atol(argv[optind]);

	if (debug)
		syslog(LOG_WARNING, "target port = %ld", target_port);

}	/* parse_args(int, char *[]) */


static void
usage(void)
{
	(void) fprintf(stderr,
		"\n%s: redirect udp packets to a port on another host\n\n",
		progname);

	(void) fprintf(stderr,
		"usage: %s [-d] [-f] [-h] local_port target_host target_port\n\n",
			progname);

	(void) fprintf(stderr, " -d    send debugging info to syslog\n");
	(void) fprintf(stderr, " -f    run in foreground\n");
	(void) fprintf(stderr, " -h    print this message\n\n");

	(void) fprintf(stderr,
		" local_port    UDP service (eg. echo, daytime) or port number\n");
	(void) fprintf(stderr,
		" target_host   name or IP address of target host\n");
	(void) fprintf(stderr,
		" target_port   UDP service (eg. echo, daytime) or port number\n\n");

}	/* usage(void) */


int
main(int argc, char *argv[])
{
	int bytes, len;
	struct hostent *hp;
	struct sockaddr_in sa[2];
	unsigned char buffer[MAX_BUFFER];

	progname = argv[0];

	if (getuid()) {

		(void) fprintf(stderr, "\n%s: must be root\n", progname);
		exit(1);
		/*NOTREACHED*/
	}

	openlog(progname, LOG_NDELAY | LOG_PID, LOG_WARNING);

	parse_args(argc, argv);

	s[0] = socket(AF_INET, SOCK_DGRAM, 0);
	if (s[0] < 0) {

		(void) fprintf(stderr,
			"\n%s: socket() failed [0], error %d, %s\n",
			progname, errno, strerror(errno));

		syslog(LOG_WARNING, "socket() [0], error %d, %m", errno);

		exit(1);
		/*NOTREACHED*/
	}

	bzero(&sa[0], sizeof(sa[0]));
	sa[0].sin_family = AF_INET;
	sa[0].sin_addr.s_addr = htonl(INADDR_ANY);
	sa[0].sin_port = htons(local_port);

	if (bind(s[0], (struct sockaddr *)&sa[0], sizeof(sa[0])) < 0) {

		(void) fprintf(stderr,
			"\n%s: bind() failed, error %d, %s\n",
			progname, errno, strerror(errno));

		syslog(LOG_WARNING, "bind(), error %d, %m", errno);

		(void) close(s[0]);
		exit(1);
		/*NOTREACHED*/
	}

	s[1] = socket(AF_INET, SOCK_DGRAM, 0);
	if (s[1] < 0) {

		(void) fprintf(stderr,
			"\n%s: socket() failed [1], error %d, %s\n",
			progname, errno, strerror(errno));

		syslog(LOG_WARNING, "socket() [1], error %d, %m", errno);

		(void) close(s[0]);
		exit(1);
		/*NOTREACHED*/
	}

	bzero(&sa[1], sizeof(sa[1]));
	sa[1].sin_family = AF_INET;

	if ((target_addr = inet_addr(target_host)) != INADDR_NONE)
		bcopy((char *)&target_addr, (char *)&sa[1].sin_addr,
			sizeof(target_addr));
	else if ((hp = gethostbyname(target_host)) == NULL) {

		(void) fprintf(stderr,
			"%s: gethostbyname(%s) failed [1], error %d",
			progname, target_host, h_errno);
		herror(" ");

		syslog(LOG_WARNING, "gethostbyname(%s), error %d",
			target_host, h_errno);

		(void) close(s[1]);
		(void) close(s[0]);
		exit(1);
		/*NOTREACHED*/

	} else bcopy(hp->h_addr, (char *)&sa[1].sin_addr, hp->h_length);

	sa[1].sin_port = htons(target_port);

	if (!foreground)
		daemon();

	while (1) {

		fd_set fds;
		int i, max_fd;

		FD_ZERO(&fds);
		FD_SET(s[0], &fds);
		FD_SET(s[1], &fds);

		max_fd = (s[0] > s[1]) ? s[0] : s[1];

		if ((i = select(max_fd + 1, &fds, NULL, NULL, NULL)) <= 0) {

			if (i < 0 && errno != EINTR) {

				syslog(LOG_WARNING, "select(), error %d, %m", errno);

				(void) close(s[0]);
				(void) close(s[1]);
				exit(1);
				/*NOTREACHED*/
			}

			continue;
		}

		for (i = 0; i < 2; i++)
			if (FD_ISSET(s[i], &fds)) {

				len = sizeof(sa[i]);
				bytes = recvfrom(s[i], buffer, MAX_BUFFER - 1, 0,
					(struct sockaddr *)&sa[i], &len);

				if (bytes <= 0) {

					if (bytes < 0 && errno != EINTR) {

						syslog(LOG_WARNING, "recvfrom(), error %d, %m",
							errno);

						(void) close(s[0]);
						(void) close(s[1]);
						exit(1);
						/*NOTREACHED*/
					}

					continue;
				}

				if (debug)
					syslog(LOG_WARNING,
						"received %d bytes from %s, port %d",
						bytes, inet_ntoa(sa[i].sin_addr),
						ntohs(sa[i].sin_port));

				len = sizeof(sa[(i)?0:1]);
				if (sendto(s[(i)?0:1], (char *)buffer, bytes, 0,
					(struct sockaddr *)&sa[(i)?0:1], len) != bytes)
						syslog(LOG_WARNING,
							"sendto(), error %d, %m", errno);

				if (debug)
					syslog(LOG_WARNING,
						"sent %d bytes to %s, port %d",
						bytes, inet_ntoa(sa[(i)?0:1].sin_addr),
						ntohs(sa[(i)?0:1].sin_port));
			}
	}
	/*NOTREACHED*/

}	/* main(int, char *[]) */

/* uredir.c */
