#include "sophie.h"
#include "sophie_syslog.h"

#define	STRNCMP(A, B, X)	(strncasecmp(A, B, X) == 0)

static int show_config_details(CISavi3 *pSAVI, CIEngineConfig *pConfig);
static void trim(char *trim_string);
static void sophie_check_config(void);
static int sophie_isnumeric(char *check);
static int parse_config(char *line, char *option, int option_size, char *value, int value_size);
static void sophie_savi_set_defaults(void);


CISavi3 *pSAVI;

void sophie_init(void)
{
	const char *ClientName = "sophie";

	CISweepClassFactory2 *pFactory;
	HRESULT hr;

	hr = DllGetClassObject(
		(REFIID)&SOPHOS_CLASSID_SAVI,
		(REFIID)&SOPHOS_IID_CLASSFACTORY2,
		(void **) &pFactory);

	if (!(hr == SOPHOS_S_OK))
	{
		sophie_print(0, "%s Could not initialize SAVI class/object", ERRSTR);
		exit(EXIT_FAILURE);
	}
	else
	{
		hr = pFactory->pVtbl->CreateInstance(pFactory, NULL, &SOPHOS_IID_SAVI3, (void **) &pSAVI );
			
		pFactory->pVtbl->Release(pFactory);
	
		if (hr == SOPHOS_S_OK)
		{
			hr = pSAVI->pVtbl->InitialiseWithMoniker(pSAVI, ClientName);

			if (SOPHOS_FAILED(hr))
			{
				sophie_print(0, "%s Failed to initialize SAVI [%ld]", ERRSTR, (long) hr);
				pSAVI->pVtbl->Release(pSAVI);
				pSAVI = NULL;
				exit(EXIT_FAILURE);
			}

			sophie_set_engine_config(pSAVI);
		}
	}
}

void sophie_config_reset(void)
{
	memset(config.saviconfig, 0, sizeof(config.saviconfig));
	config.maxproc = -1;
	memset(config.pidfile, 0, sizeof(config.pidfile));
	memset(config.socketfile, 0, sizeof(config.socketfile));
	config.umask = -1;
	memset(config.user, 0, sizeof(config.user));
	memset(config.group, 0, sizeof(config.group));
	config.timeout = -1;
	memset(config.logname, 0, sizeof(config.logname));
	config.logfacility = -1;
	config.logpriority = -1;
	config.error_strings = -1;
	config.timestamps = -1;
	config.show_virusname = -1;
	config.callbacks = -1;
	config.limit_classif = -1;
	config.limit_nextfile = -1;
	config.limit_decompr = -1;
	config.socket_check = -1;
#ifdef SOPHIE_NET
	config.net_port = -1;
	memset(config.net_tempdir, 0, sizeof(config.net_tempdir));
#endif
}

void sophie_lookup_syslog_facility(int syslog_int, char *syslog_char, int syslog_char_size)
{
	CODE slog;
	int s = 0;

	for(;;)
	{
		slog = facilitynames[s];
		if (slog.c_name == NULL)
			break;

		if (slog.c_val == syslog_int)
		{
			strncpy(syslog_char, slog.c_name, syslog_char_size-1);
			break;
		}

		s++;
	}
}

void sophie_lookup_syslog_priority(int syslog_int, char *syslog_char, int syslog_char_size)
{
	CODE slog;
	int s = 0;

	for(;;)
	{
		slog = prioritynames[s];
		if (slog.c_name == NULL)
			break;

		if (slog.c_val == syslog_int)
		{
			strncpy(syslog_char, slog.c_name, syslog_char_size-1);
			break;
		}

		s++;
	}
}

int sophie_load_config(char *config_file)
{
	FILE *cfgfp = NULL;
	char line[256];
	char option[64];
	char value[192];
	int i = 0;
	int found = 0;
	
	CODE slog;

	if (config_file[0] != '\0')
	{
		if (!file_exists(config_file))
		{
			sophie_print(0, "%s Could not open '%s' config file (does not exist)", WARNSTR, config_file);
			return(0);
		}
		
		cfgfp = fopen(config_file, "r");
		if (!cfgfp)
		{
			sophie_print(0, "%s fopen() failed [%s]", WARNSTR, strerror(errno));
			return(0);
		}

		found = 1;
	}
	else
	{
		sophie_print(0, "%s Sophie configuration file not specified - falling back to defaults", NOTESTR);

		while(default_configs[i] != NULL)
		{
			if (!file_exists(default_configs[i]))
			{
				sophie_print(1, "%s Default Sophie config file '%s' not found", NOTESTR, default_configs[i]);
				i++;
				continue;
			}
			
			cfgfp = fopen(default_configs[i], "r");
			if (!cfgfp)
			{
				sophie_print(0, "%s Error fopen()ing '%s' [%s] - moving on", WARNSTR, strerror(errno));
				i++;
				continue;
			}
			
			found = 1;
			sophie_print(0, "%s Using Sophie configuration file '%s'", NOTESTR, default_configs[i]);
			break;
		}
	}

	if (found != 1)
	{
		sophie_print(0, "%s Could not open any Sophie configuration file", ERRSTR);
		return(0);
	}

	while(fgets(line, sizeof(line)-1, cfgfp))
	{
		if (parse_config(line, option, sizeof(option), value, sizeof(value)))
		{
			if (STRNCMP(option, "saviconfig", 10))
				strncpy(config.saviconfig, value, sizeof(config.saviconfig)-1);

			if ( (STRNCMP(option, "maxproc", 7)) && (sophie_isnumeric(value)) )
				config.maxproc = atoi(value);

			if (STRNCMP(option, "pidfile", 7))
				strncpy(config.pidfile, value, sizeof(config.pidfile)-1);

			if (STRNCMP(option, "socketfile", 10))
				strncpy(config.socketfile, value, sizeof(config.socketfile)-1);

			if ( (STRNCMP(option, "umask", 5)) && (sophie_isnumeric(value)) )
				config.umask = atoi(value);

			if (STRNCMP(option, "user", 4))
				strncpy(config.user, value, sizeof(config.user)-1);

			if (STRNCMP(option, "group", 5))
				strncpy(config.group, value, sizeof(config.group)-1);

			if ( (STRNCMP(option, "timeout", 7)) && (sophie_isnumeric(value)) )
				config.timeout = atoi(value);

			if (STRNCMP(option, "logname", 7))
				strncpy(config.logname, value, sizeof(config.logname)-1);

			if (STRNCMP(option, "logfacility", 11))
			{
				int s = 0;
				for(;;)
				{
					slog = facilitynames[s];
					if (slog.c_name == NULL)
						break;

					if (STRNCMP(slog.c_name, value, strlen(slog.c_name)))
					{
						config.logfacility = slog.c_val;
						break;
					}
					
					s++;
				}
			}

			if (STRNCMP(option, "logpriority", 11))
			{
				int s = 0;
				for(;;)
				{
					slog = prioritynames[s];
					if (slog.c_name == NULL)
						break;

					if (STRNCMP(slog.c_name, value, strlen(slog.c_name)))
					{
						config.logpriority = slog.c_val;
						break;
					}
					
					s++;
				}
			}

			if (STRNCMP(option, "error_strings", 13))
			{
				if (STRNCMP(value, "yes", 3))
					config.error_strings = 1;
				else if (STRNCMP(value, "no", 2))
					config.error_strings = 0;
			}

			if (STRNCMP(option, "timestamps", 10))
			{
				if (STRNCMP(value, "yes", 3))
					config.timestamps = 1;
				else if (STRNCMP(value, "no", 2))
					config.timestamps = 0;
			}

			if (STRNCMP(option, "show_virusname", 14))
			{
				if (STRNCMP(value, "yes", 3))
					config.show_virusname = 1;
				else if (STRNCMP(value, "no", 2))
					config.show_virusname = 0;
			}

			if (STRNCMP(option, "callbacks", 9))
			{
				if (STRNCMP(value, "yes", 3))
					config.callbacks = 1;
				else if (STRNCMP(value, "no", 2))
					config.callbacks = 0;
			}

			if ( (STRNCMP(option, "limit_classif", 13)) && (sophie_isnumeric(value)) )
				config.limit_classif = atoi(value);

			if ( (STRNCMP(option, "limit_nextfile", 14)) && (sophie_isnumeric(value)) )
				config.limit_nextfile = atoi(value);

			if ( (STRNCMP(option, "limit_decompr", 13)) && (sophie_isnumeric(value)) )
				config.limit_decompr = atoi(value);

			if (STRNCMP(option, "socket_check", 12))
			{
				if (STRNCMP(value, "yes", 3))
					config.socket_check = 1;
				else if (STRNCMP(value, "no", 2))
					config.socket_check = 0;
			}

#ifdef SOPHIE_NET
			if ( (STRNCMP(option, "net_port", 8)) && (sophie_isnumeric(value)) )
				config.net_port = atoi(value);

			if (STRNCMP(option, "net_tempdir", 11))
				strncpy(config.net_tempdir, value, sizeof(config.net_tempdir)-1);
#endif
		}
	}

	fclose(cfgfp);

	sophie_check_config();

	return(1);
}

static void sophie_check_config(void)
{
	if (config.saviconfig[0] == '\0')
		sophie_print(0, "%s config.saviconfig is empty: SAVI options will be set to default", WARNSTR);

	if (config.maxproc == -1)
	{
		sophie_print(0, "%s config.maxproc is set to -1: setting to %d", NOTESTR, DEFAULT_MAX_PROC);
		config.maxproc = DEFAULT_MAX_PROC;
	}
	
	if (config.pidfile[0] == '\0')
	{
		sophie_print(0, "%s config.pidfile is empty: setting to %s", NOTESTR, DEFAULT_PIDFILE);
		strncpy(config.pidfile, DEFAULT_PIDFILE, sizeof(config.pidfile)-1);
	}

	if (config.socketfile[0] == '\0')
	{
		sophie_print(0, "%s config.socketfile is empty: setting to %s", NOTESTR, DEFAULT_SOCKETFILE);
		strncpy(config.socketfile, DEFAULT_SOCKETFILE, sizeof(config.socketfile)-1);
	}

	if (config.umask == -1)
	{
		sophie_print(0, "%s config.umask is set to -1: setting to %d", NOTESTR, DEFAULT_UMASK);
		config.umask = DEFAULT_UMASK;
	}

	if (config.user[0] == '\0')
	{
		sophie_print(0, "%s config.user is not set. User MUST be set in order for Sophie to continue. Exiting...", ERRSTR);
		exit(EXIT_FAILURE);
	}

	if (config.group[0] == '\0')
	{
		sophie_print(0, "%s config.group is not set. Group MUST be set in order for Sophie to continue. Exiting...", ERRSTR);
		exit(EXIT_FAILURE);
	}

	if (config.timeout == -1)
	{
		sophie_print(0, "%s config.timeout is set to -1: setting to %d", NOTESTR, DEFAULT_TIMEOUT);
		config.timeout = DEFAULT_TIMEOUT;
	}

	if (config.logname[0] == '\0')
	{
		sophie_print(0, "%s config.logname is empty: setting to %s", NOTESTR, DEFAULT_LOGNAME);
		strncpy(config.logname, DEFAULT_LOGNAME, sizeof(config.logname)-1);
	}

	if (config.logfacility == -1)
	{
		sophie_print(0, "%s config.logfacility is set to -1: setting to %d", NOTESTR, DEFAULT_LOGFACILITY);
		config.logfacility = DEFAULT_LOGFACILITY;
	}

	if (config.logpriority == -1)
	{
		sophie_print(0, "%s config.logpriority is set to -1: setting to %d", NOTESTR, DEFAULT_LOGPRIORITY);
		config.logpriority = DEFAULT_LOGPRIORITY;
	}

	if (config.error_strings == -1)
	{
		sophie_print(0, "%s config.error_strings is set to -1: setting to %d", NOTESTR, DEFAULT_ERROR_STRINGS);
		config.error_strings = DEFAULT_ERROR_STRINGS;
	}

	if (config.timestamps == -1)
	{
		sophie_print(0, "%s config.timestamps is set to -1: setting to %d", NOTESTR, DEFAULT_TIMESTAMPS);
		config.timestamps = DEFAULT_TIMESTAMPS;
	}

	if (config.show_virusname == -1)
	{
		sophie_print(0, "%s config.show_virusname is set to -1: setting to %d", NOTESTR, DEFAULT_SHOW_VIRUSNAME);
		config.show_virusname = DEFAULT_SHOW_VIRUSNAME;
	}

	if (config.callbacks == -1)
	{
		sophie_print(0, "%s config.callbacks is set to -1: setting to %d", NOTESTR, DEFAULT_CALLBACKS);
		config.callbacks = DEFAULT_CALLBACKS;
	}

	if (config.limit_classif == -1)
	{
		sophie_print(0, "%s config.limit_classif is set to -1: setting to %d", NOTESTR, DEFAULT_LIMIT_CLASSIF);
		config.limit_classif = DEFAULT_LIMIT_CLASSIF;
	}

	if (config.limit_nextfile == -1)
	{
		sophie_print(0, "%s config.limit_nextfile is set to -1: setting to %d", NOTESTR, DEFAULT_LIMIT_NEXTFILE);
		config.limit_nextfile = DEFAULT_LIMIT_NEXTFILE;
	}

	if (config.limit_decompr == -1)
	{
		sophie_print(0, "%s config.limit_decompr is set to -1: setting to %d", NOTESTR, DEFAULT_LIMIT_DECOMPR);
		config.limit_decompr = DEFAULT_LIMIT_DECOMPR;
	}

	if (config.socket_check == -1)
	{
		sophie_print(0, "%s config.socket_check is set to -1: setting to %d", NOTESTR, DEFAULT_SOCKET_CHECK);
		config.socket_check = DEFAULT_SOCKET_CHECK;
	}

#ifdef SOPHIE_NET
	if (config.net_port == -1)
	{
		sophie_print(0, "%s config.net_port is set to -1: setting to %d", NOTESTR, DEFAULT_NET_PORT);
		config.net_port = DEFAULT_NET_PORT;
	}

	if (config.net_tempdir[0] == '\0')
	{
		sophie_print(0, "%s config.net_tempdir is empty: setting to %s", NOTESTR, DEFAULT_NET_TEMPDIR);
		strncpy(config.net_tempdir, DEFAULT_NET_TEMPDIR, sizeof(config.net_tempdir)-1);
	}
#endif
}

static int sophie_isnumeric(char *check)
{
	char *p;
	char ch;
	
	p = check;
	while(*p != '\0')
	{
		if (!isdigit((int) *p))
			return(0);
		ch = *p++;
	}

	return(1);
}

static int parse_config(char *line, char *option, int option_size, char *value, int value_size)
{
	char *p;
	char ch;

	/* Strip newline */
	if (strchr(line, '\n'))
		*strchr(line, '\n') = '\0';
	if (strchr(line, '\r'))
		*strchr(line, '\r') = '\0';

	/* Continue if we don't need to parse this line */
	/* We don't parse if line is empty, contains '#' at the beginning or doesn't have ':' in it */
	if ( (line[0] == '\0') || (line[0] == '#') || (!strchr(line, ':')) )
		return(0);

	/* Find a ':' first */
	p = strchr(line, ':');
		
	/* Find the beginning of the value for the option */
	while( (*p == ':') || (*p == ' ') || (*p == '\t') )
		ch = *p++;

	/* Copy the value to the 'value' var */
	strncpy(value, p, value_size-1);

	/* Null terminate the string at ':' sign */
	*strchr(line, ':') = '\0';

	/* Copy the option name to the 'option' var */
	strncpy(option, line, option_size-1);

	/* Trim spaces */
	trim(option);
	trim(value);

	/* Continue if the value or option are empty */
	if ( (option[0] == '\0') || (value[0] == '\0') )
		return(0);

	return(1);
}

void sophie_show_settings(void)
{
	CIEnumEngineConfig *pConfigEnum = NULL;
	CIEngineConfig *pConfig = NULL;
	U32 pcFetched;
	HRESULT hr;

	hr = pSAVI->pVtbl->GetConfigEnumerator(	pSAVI,
											(REFIID)&SOPHOS_IID_ENUM_ENGINECONFIG,
											(void **)&pConfigEnum
										 );

	if (SOPHOS_FAILED(hr))
	{
		sophie_print(0, "%s Could not get SAVI configuration data", WARNSTR);
	}

	hr = pConfigEnum->pVtbl->Reset(pConfigEnum);

	if (SOPHOS_FAILED(hr))
	{
		sophie_print(0, "%s Could not reset configuration data enumerator", WARNSTR);
	}

	sophie_print(0, "+-----------------------------------+------+-----+");
	sophie_print(0, "| Configuration parameter           | Type |Value|");
	sophie_print(0, "+-----------------------------------+------+-----+");
	
	while (pConfigEnum->pVtbl->Next(pConfigEnum, 1, (void **) &pConfig, &pcFetched) == SOPHOS_S_OK)
	{
		show_config_details(pSAVI, pConfig);
		pConfig->pVtbl->Release(pConfig);
	}

	pConfigEnum->pVtbl->Release(pConfigEnum);
	
	sophie_print(0, "+-----------------------------------+------+-----+");

}

static int show_config_details(CISavi3 *pSAVI, CIEngineConfig *pConfig)
{
	const U32 arraySize = 100;
	TCHAR name[100];
	TCHAR value[100];
	U32 type;
	HRESULT hr;

	name[0] = '\0';
	value[0] = '\0';
	type = 0;

	hr = pConfig->pVtbl->GetName(pConfig, arraySize, name, NULL);
	if (SOPHOS_FAILED(hr))
	{
		sophie_print(0, "%s Couldn't get options name [0x%08lx]", NOTESTR, (unsigned long) hr);
		return 0;
	}

	hr = pConfig->pVtbl->GetType(pConfig, &type);
	if (SOPHOS_FAILED(hr))
	{
		sophie_print(0, "%s Couldn't get size [0x%08lx]", NOTESTR, (unsigned long) hr);
		return 0;
	}

	hr = pSAVI->pVtbl->GetConfigValue(pSAVI, name, type, arraySize, value, NULL);
	if (SOPHOS_FAILED(hr))
	{
		sophie_print(0, "%s Couldn't get value of '%s' [0x%08lx]", NOTESTR, name, (unsigned long) hr);
		return 0;
	}

	if (strlen(value) > 2)
		sophie_print(0, "| %-34s|  %3ld | %s", name, (unsigned long) type, value);
	else
		sophie_print(0, "| %-34s|  %3ld | %3s |", name, (unsigned long) type, value);

	return 1;
}

static void sophie_savi_set_defaults(void)
{
	HRESULT hr;
	
	hr = pSAVI->pVtbl->SetConfigDefaults(pSAVI);
	if (SOPHOS_FAILED(hr))
		sophie_print(0, "%s Failed to SetConfigDefaults()", WARNSTR);
	else
		sophie_print(1, "%s Default SAVI configuration options set", NOTESTR);
}

void sophie_set_engine_config(CISavi3 *pSAVI)
{
	FILE *cfg;
	int stype;
	char cfgbuf[128];
	char option[64];
	char value[256]; /* if someone gives really long path, for example */
	HRESULT hr;

	if (config.saviconfig[0] == '\0')
	{
		sophie_savi_set_defaults();
		return;
	}

	if (!file_exists(config.saviconfig))
	{
		sophie_print(0, "%s SAVI options file '%s' does not exist. Using defaults.", WARNSTR, config.saviconfig);
		sophie_savi_set_defaults();
		return;
	}

	memset(cfgbuf, 0, sizeof(cfgbuf));
	memset(option, 0, sizeof(option));
	memset(value, 0, sizeof(value));

	/* Open configuration file */
	cfg = fopen(config.saviconfig, "r");
	if (!cfg)
	{
		sophie_print(0, "%s Failed to fopen() SAVI options file '%s' [%s]", WARNSTR, config.saviconfig, strerror(errno));
		return;
	}

	sophie_print(0, "%s Setting configuration options - please wait...", NOTESTR);

	/* Read line by line, and set config values */
	while(fgets(cfgbuf, sizeof(cfgbuf)-1, cfg))
	{
		if (parse_config(cfgbuf, option, sizeof(option), value, sizeof(value)))
		{
			/* FIXME: At this point, we'll only set the config options that take int as a value. */
			/* String options will be added later, if needed (for VirusDataDir, VirusDataName and IdeDir) */

			/* Set the configuration option */
		
			/* Crap... */
			if (STRNCMP(option, "MaxRecursionDepth", 17))
				stype = SOPHOS_TYPE_U16;
			else if (STRNCMP(option, "Grp", 3))
				stype = SOPHOS_TYPE_OPTION_GROUP;
			else
				stype = SOPHOS_TYPE_U32;

			hr = pSAVI->pVtbl->SetConfigValue(pSAVI, option, stype, value);
			if (SOPHOS_FAILED(hr))
				sophie_print(0, "%s Unable to setup configuration value '%s' to '%s'", WARNSTR, option, value);
			else
				sophie_print(1, "%s Option '%s' set to '%s'", NOTESTR, option, value);
		}
	}

	fclose(cfg);

	sophie_print(0, "%s Configuration options set", NOTESTR);
}

/* Trim leading/trailing spaces */
static void trim(char *trim_string)
{
	char *p;
	char ch;
	unsigned int len;
	
	p = trim_string;
	
	/* Strip leading spaces/tabs */
	while( (*p == ' ') || (*p == '\t') )
		ch = *p++;
	
	/* Save the result */
	trim_string = p;

	/* Get the new string length */
	len = strlen(trim_string);
	
	/* Position the pointer to the end of the string */
	p = &trim_string[len-1];
	
	/* FIXME: Sort this crap out later */
	while( (*p == ' ') || (*p == '\t') )
			ch = *p--;

	/* Lame, but works - it's late :) */
	ch = *p++;
	*p = '\0';
}
