# Module: moduleHistoryTruncated
# Author: Craig H. Rowland <crowland@psionic.com>
# Created: 11-1-98
#
# Send all changes/modifications/bugfixes to the above address.  
# 
# This software is Copyright(c) 1997-98 Craig H. Rowland
# 
# Disclaimer:
# 
# All software distributed by Craig H. Rowland ("the author") and
# Psionic Systems is distributed AS IS and carries NO WARRANTY or
# GUARANTEE OF ANY KIND. End users of the software acknowledge that         
# they will not hold the author, Psionic Systems, and any employer of    
# the author liable for failure or non-function of a software
# product. YOU ARE USING THIS PRODUCT AT YOUR OWN RISK
# 
# Licensing restrictions apply. See the license that came with this
# file for more information or visit http://www.psionic.com for more
# information.
# 
# This software is NOT GPL NOR PUBLIC DOMAIN so please read the license   
# before modifying or distributing. Contact the above address if you have
# any questions.
# 
# Description:
# 
# This module checks what history file should exist in their home directory
# (by referencing /etc/passwd) and then checks if it:
# 
# a) Exists
# b) Is not zero bytes
# c) Is not a link (i.e. linked to /dev/null)
# 
# $Id: moduleHistoryTruncated.py,v 1.3 1999/03/22 04:57:23 crowland Exp crowland $


from hostSentryCore import *
import hostSentryConfig
import hostSentryUser
import hostSentryLog
import hostSentryDB

import sys
import string
import re
import pwd
import posixpath
import os

# This is a listing of popular shells and the name of the
# history file they make. I need to find more of these to 
# make a complete table (some may even be wrong)
shellDict = { 'bash': '.bash_history', 'csh': '.history', 'tcsh': 
'.history', 'zsh': '.history', 'ksh': '.history'}


class moduleHistoryTruncated(hostSentryCore):

	def __init__(self):
		self.setLogLevel()
		self.__result = None

	def login(self, userObj, loginStamp):
		if self.getLogLevel() > 0:
			hostSentryLog.log('debug: moduleHistoryTruncated: login: processing user: ' + userObj.getUsername())		

	def logout(self, userObj, logoutStamp):

		if self.getLogLevel() > 0:
			hostSentryLog.log('debug: moduleHistoryTruncated: logout: processing user: ' + userObj.getUsername())		

		loginIP, loginHostname, loginTTY, loginTime, logoutTime =  string.split(logoutStamp, '@')

		try:
			pwInfo = pwd.getpwnam(userObj.getUsername())
		except:
			hostSentryLog.log('adminalert: moduleHistoryTruncated: logout: Cannot find user: ' + userObj.getUsername() + ' in passwd database: ' + sys.exc_value[0])
			raise hostSentryError('adminalert: moduleHistoryTruncated: logout: Cannot find user: ' + userObj.getUsername() + ' in passwd database: ' + sys.exc_value[0])

		try:
			userPath = pwInfo[5]
			userShell = pwInfo[6]
			if self.getLogLevel() > 0:
				hostSentryLog.log('debug: moduleHistoryTruncated: logout: looking for history file in: %s' % userPath)
			# No shell for this user
			if userShell == '':
				return
		except:
			return


		try:
			# This breaks off just the shell which is what we need to do a hash table
			# lookup.
			path, shell = posixpath.split(userShell)

			if self.getLogLevel() > 0:
				hostSentryLog.log('debug: moduleHistoryTruncated: logout: found shell: %s' % shell)

			# This looks up the proper history file for our shell and makes
			# the path
			historyFile = shellDict.get(shell)
			historyPath = userPath + '/' + historyFile

			# This will check the history file for a link, non-existent, or zero bytes.
			if historyFile != None:
				if self.getLogLevel() > 0:
					hostSentryLog.log('debug: moduleHistoryTruncated: logout: looking for history file: %s' % historyFile)
				if posixpath.islink(historyPath):
					hostSentryLog.log('securityalert: moduleHistoryTruncated: history file for user: %s is a symbolic link: %s' % (userObj.getUsername(), historyFile))
					self.setResult('securityalert: moduleHistoryTruncated: history file for user: %s is a symbolic link: %s' % (userObj.getUsername(), historyFile))
					return
				elif not posixpath.isfile(historyPath):
					hostSentryLog.log('securityalert: moduleHistoryTruncated: history file for user: %s is missing: %s' % (userObj.getUsername(), historyFile))
					self.setResult('securityalert: moduleHistoryTruncated: history file for user: %s is missing: %s' % (userObj.getUsername(), historyFile))
					return
				elif os.stat(historyPath)[6] == 0:
					hostSentryLog.log('securityalert: moduleHistoryTruncated: history file for user: %s is zero bytes: %s' % (userObj.getUsername(), historyFile))
					self.setResult('securityalert: moduleHistoryTruncated: history file for user: %s is zero bytes: %s' % (userObj.getUsername(), historyFile))
					return
		except:			
			hostSentryLog.log('adminalert: moduleHistoryTruncated: logout: Error reading history file for processing: ' + sys.exc_value[0])
			raise hostSentryError('adminalert: moduleHistoryTruncated: logout: Error reading history file for processing: ' + sys.exc_value[0])

	def setResult(self, result):
		self.__result = result

	def getResult(self):
		return self.__result


if __name__ == '__main__':

	test = moduleHistoryTruncated()
	test.setLogLevel(99)
	user = hostSentryUser.hostSentryUser()
	user.setUsername('test')
#	user.setUsername('hostSentrytest')
	user.setTotalLogins(1)

	config = hostSentryConfig.hostSentryConfig()
	config.configInit()
	dbFile = config.parseToken('DB_FILE')

	db = hostSentryDB.hostSentryDB(dbFile)
	db.store(user)
	db.close()

	test.login(user, '192.168.2.1@somewhere.com@tty0001@999999999@')
	test.logout(user, '192.168.2.1@somewhere.com@tty0001@999999999@98989898')

	db = hostSentryDB.hostSentryDB(dbFile)
	db.delete(user.getUsername())
	db.close()
