/*
 *	Copyright (c) 1996 The CAD lab of the
 *	Siberian State Academy of Telecommunication
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#include <setjmp.h>
#include <errno.h>
#include "netchat.h"
#include "callsign.h"

extern void cleanup();

struct conf cf[MAXHOSTS];
int myport, userport, hn = 0;

static int cmdpool = 0;
static int sock, fromlen;
static struct sockaddr_in from;
static int smon = -1;
static struct sockaddr_un nmon;
static jmp_buf after_alarm;
static time_t timenow;
static char *psysreply = NULL;

void
cmd_done(sig)
	int sig;
{
	sigset_t mask;

	do {
		if (cmdpool) {
			int status, ret;
			while ((ret = (int)wait(&status)) < 0 && errno == EINTR);
			if (ret < 0) syslog(LOG_ERR, "wait: %m");
			cmdpool--;
		} else {
			syslog(LOG_WARNING, "unexpected signal %d", sig);
			break;
		}
		sigemptyset(&mask);
		sigpending(&mask);
	} while (sigismember(&mask, sig));
	(void)signal(sig, cmd_done);
}

void
initmon(sig)
	int sig;
{
	if (smon == -1) {
		if ((smon = socket(AF_UNIX, SOCK_DGRAM, 0)) < 0)
			syslog(LOG_ERR, "socket: %m");
		else {
			nmon.sun_family = AF_UNIX;
			sprintf(nmon.sun_path, "%s/%s.IN", MONITORDIR, MYTTY);
		}
	} else syslog(LOG_WARNING, "unexpected signal %d", sig);
	(void)signal(sig, initmon);
}

/*
 * Init network functions. Return -1 on error or 0 for success.
 */
int
initnet(port)
	int port;
{
	struct sockaddr_in sin;
	int on = 1;

	if (port) myport = htons(port);
	else {
		struct servent *sp;

		if ((sp = getservbyname(NETCHAT, "udp")) == NULL) {
			syslog(LOG_ERR, "getservbyname: %s/udp: Unknown service", NETCHAT);
			return -1;
		}
		myport = sp->s_port;
	}

	if (initcf(configfile) < 0) return -1;

	if ((sock = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		syslog(LOG_ERR, "socket: %m");
		return -1;
	}
	if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (void *)&on, sizeof(on)) < 0) {
		syslog(LOG_ERR, "setsockopt: %m");
		return -1;
	}
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = INADDR_ANY;
	sin.sin_port = myport;
	if (bind(sock, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		syslog(LOG_ERR, "bind: %m");
		return -1;
	}
	return 0;
}

int
initcf(cfile)
	char *cfile;
{
	int i;
	register char *ptr;
	char *line, *host, *nick;

	for (i = 0; i < hn; i++) {
		free(cf[i].hin);
		free(cf[i].nick);
	}
	hn = 0;
	cf[hn].hin = (struct sockaddr_in *)malloc(sizeof(struct sockaddr_in));
	if (cf[hn].hin == NULL) {
		syslog(LOG_ERR, "malloc: %m");
		return -1;
	}
	cf[hn].hin->sin_family = AF_INET;
	cf[hn].hin->sin_addr.s_addr = inet_addr(LOOPBACK);
	cf[hn].hin->sin_port = 0;
	if ((cf[hn].nick = strdup(username)) == NULL) {
		syslog(LOG_ERR, "strdup: %m");
		return -1;
	}
	cf[hn].alive = 0;
	hn++;

	if ((ptr = (char *)loadfile(cfile)) == NULL) {
		syslog(LOG_WARNING, "initcf: %s: %m", cfile);
		return 0;
	}
	for (; (line = strchr(ptr, '\n')) != NULL && hn < MAXHOSTS; ptr = line) {
		for (*line++ = 0; *ptr && (u_char)*ptr <= 0x20; ptr++);
		if (!*ptr || *ptr == '#') continue;
		for (host = ptr; (u_char)*ptr > 0x20; ptr++);
		if (*ptr) for (*ptr++ = 0; *ptr && (u_char)*ptr <= 0x20; ptr++);
		nick = ptr;
		cf[hn].hin = (struct sockaddr_in *)malloc(sizeof(struct sockaddr_in));
		if (cf[hn].hin == NULL) {
			syslog(LOG_ERR, "malloc: %m");
			return -1;
		}
		if ((ptr = strchr(host, ':')) != NULL) {
			*ptr++ = '\0';
			cf[hn].hin->sin_port = htons(atoi(ptr));
		} else	cf[hn].hin->sin_port = myport;

		if (atoi(host) > 0) {
			cf[hn].hin->sin_family = AF_INET;
			cf[hn].hin->sin_addr.s_addr = inet_addr(host);
		} else {
			struct hostent *hp;

			if ((hp = gethostbyname(host)) == NULL) {
				syslog(LOG_ERR, "gethostbyname: %s: Unknown host", host);
				return -1;
			}
			cf[hn].hin->sin_family = hp->h_addrtype;
			memmove((caddr_t)&cf[hn].hin->sin_addr, hp->h_addr_list[0], hp->h_length);
		}
		if ((cf[hn].nick = strdup(*nick ? nick : host)) == NULL) {
			syslog(LOG_ERR, "strdup: %m");
			return -1;
		}
		cf[hn++].alive = 0;
	}
	if (dflag > 1 && hn > 1) {
		printf("Host table:\n");
		for (i = 0; i < hn; i++)
			printf("host=%s  port=%d  nick=%s  alive=%x\n",
			       inet_ntoa(cf[i].hin->sin_addr),
			       (int)ntohs(cf[i].hin->sin_port),
			       cf[i].nick, cf[i].alive);
	}
	return 0;
}

void
monitor(const char *fmt, ...)
{
	int len;
	char buf[256];
	va_list ap;

	if (!dflag && smon < 0) return;

	va_start(ap, fmt);
	len = vsnprintf(buf, sizeof(buf), fmt, ap);
	va_end(ap);

	if (dflag) {
		fputs(buf, stdout);
		fflush(stdout);
	} else if (smon != -1) {
		len = sendto(smon, buf, len, 0, (struct sockaddr *)&nmon,
			     sizeof(struct sockaddr_un));
		if (len < 0 && errno != ENOBUFS) {
			close(smon);
			smon = -1;
			unlink(nmon.sun_path);
		}
	}
}

char *
waitmsg()
{
	struct timeval timeout = {1, 0};
	fd_set ready;
	register ret, sec;
	static char buf[256];

	if (dflag) {
		fputs("wait for message...", stdout);
		fflush(stdout);
	}
	sec = TRICKLE;
	do {
		FD_ZERO(&ready);
		FD_SET(sock, &ready);
		while ((ret = select(sock + 1, &ready, NULL, NULL, &timeout)) < 0 &&
		       errno == EINTR);
		if (ret < 0) {
			syslog(LOG_ERR, "select: %m");
			cleanup();
		}
		if (!ret && psysreply) {
			monitor("%s: %s: %s\n", LOOPBACK, username, psysreply);
			msgremote(0, "%s: %s", username, psysreply);
			msglocal(NULL, 0, " %s", psysreply);
		}
		psysreply = NULL;
	} while (!ret && --sec);
	if (!ret) {
		if (dflag) {
			fputs(" send trickle marker\n", stdout);
			fflush(stdout);
		}
		return NULL;	/* timeout */
	}
	fromlen = sizeof(from);
	while ((ret = recvfrom(sock, (char *)buf, sizeof(buf)-1, 0,
			(struct sockaddr *)&from, &fromlen)) < 1) {
		if (ret < 0 && errno != EINTR) {
			syslog(LOG_ERR, "recvfrom: %m");
			return NULL;
		}
	}
	buf[ret] = '\0';
	if (dflag) fputc('\n', stdout);
	monitor("%s: %s\n", inet_ntoa(from.sin_addr), buf);
	return buf;
}

/*
 * Send message to remote host by idx; zero idx -- send to all hosts.
 */
void
msgremote(int idx, ...)
{
	int i, maxidx, len;
	char *p, buf[100];
	va_list ap;

	cf[0].alive = timenow;

	if (!idx) {
		idx++;
		maxidx = hn;
	} else	maxidx = idx + 1;

	if (userport) len = sprintf(buf, "%d%%", userport);
	else len = 0;

	va_start(ap, idx);
	p = va_arg(ap, char *);
	len += vsnprintf(&buf[len], sizeof(buf)-len, p, ap);
	va_end(ap);

	for (i = idx; i < maxidx; i++) {
		if (len > 1 && !cf[i].alive) continue;
		if (sendto(sock, (char *)buf, len, 0,
			   (struct sockaddr *)cf[i].hin, sizeof(struct sockaddr)) < 0)
			syslog(LOG_ERR, "sendto: %m");
	}
}

/*
 * Send message to local user; zero user -- send to all users.
 */
void
msglocal(char *user, int idx, ...)
{
	int len;
	char *p, buf[100];
	struct loginfo *info;
	va_list ap;

	(void)strcpy(buf, cf[idx].nick);
	len = strlen(strcat(buf, ":"));

	va_start(ap, idx);
	p = va_arg(ap, char *);
	len += vsnprintf(&buf[len], sizeof(buf)-len, p, ap);
	va_end(ap);

	if (user) user = getlogname(user);

	while ((info = getinfo()) != NULL) {
		if (!user) {
			if ((info->flags & INCONF) && info->port != myport) {
				cf[0].hin->sin_port = info->port;
				if (sendto(sock, (char *)buf, len, 0,
					   (struct sockaddr *)cf[0].hin,
					   sizeof(struct sockaddr)) < 0)
					syslog(LOG_ERR, "sendto: %m");
			}
			continue;
		}
		if (!strcasecmp(info->name, user)) {
			closeinfo();
			if (info->flags & (MESGNO|INRECVF|INSENDF|EXECEXT)) {
				msgreply("<%s unreachable -- %s>", info->name, flag2str(info->flags));
				return;
			}
			cf[0].hin->sin_port = info->port;
			if (sendto(sock, (char *)buf, len, 0,
				   (struct sockaddr *)cf[0].hin,
				   sizeof(struct sockaddr)) < 0)
				syslog(LOG_ERR, "sendto: %m");
			return;
		}
	}
	if (user) msgreply("<%s -- No such user>", user);
}

void
msgreply(const char *fmt, ...)
{
	int len;
	char buf[100];
	va_list ap;

	va_start(ap, fmt);
	len = vsnprintf(buf, sizeof(buf), fmt, ap);
	va_end(ap);

	if (sendto(sock, (char *)buf, len, 0,
		   (struct sockaddr *)&from, fromlen) < 0)
		syslog(LOG_ERR, "sendto: %m");
}

int
parsearg(txt, arg)
	char *txt;
	struct args *arg;
{
	register char *p, *p1;

	arg->cmd = arg->host = arg->user = arg->txt = NULL;
	for (p = txt; (u_char)*p > 0x20; p++);
	if (*p) for (*p++ = '\0'; (u_char)*p <= 0x20 && *p; p++);
	if ((p1 = strchr(txt, ':')) != NULL) {
		*p1++ = '\0';		/* host:[user] */
		if (!*txt) return -1;
		arg->host = txt;
		if (*p1) arg->user = p1;
	} else	arg->cmd = txt;		/* cmd */
	if (!*p) return (arg->host != NULL ? -1 : 0);
	if (arg->host) {
		arg->txt = p;		/* host:[user] message... */
		return 0;
	}
	for (p1 = p; (u_char)*p1 > 0x20; p1++);
	*p1 = '\0';
	if ((p1 = strchr(p, ':')) == NULL) return -1;	/* bad format */
	*p1++ = '\0';
	if (*p) arg->host = p;			/* cmd host:[user] */
	if (*p1) arg->user = p1;
	return 0;
}

struct cmd *
getcmd(name)
	char *name;
{
	char *p, *q;
	struct cmd *c, *found;
	int nmatches, longest;

	longest = nmatches = 0;
	found = 0;
	for (c = cmdtab; p = c->name; c++) {
		for (q = name; *q == *p++; q++)
			if (*q == 0) return c;	/* exact match? */
		if (!*q) {			/* the name was a prefix */
			if (q - name > longest) {
				longest = q - name;
				nmatches = 1;
				found = c;
			} else if (q - name == longest) nmatches++;
		}
	}
	return (nmatches > 1 ? ((struct cmd *)-1) : found);
}

void
hostcmd(idx, user, txt)
	int idx;
	char *user, *txt;
{
	int i;
	struct args arg;
	struct cmd *c;

	if (parsearg(txt, &arg) < 0) {
		msgreply("<Bad format. Use `cmd host:[user]' or `host:[user] message...'>");
		return;
	}
	if (!idx && arg.host) {
		for (i = 1; i < hn; i++)
			if (!strcasecmp(arg.host, cf[i].nick)) break;
		if (i == hn) {
			msgreply("<%s -- No such host>", arg.host);
			return;
		}
		if (!cf[i].alive) {
			msgreply("<%s -- Host not answer>", arg.host);
			return;
		}
	} else i = 0;
	if (!arg.cmd) {
		/* Private message */
		if (!idx) /* Send to remote host */
			msgremote(i, "%s> %s:%s %s", user, arg.host,
				  arg.user ? arg.user : "", arg.txt);
		else	/* Send to local user(s) */
			msglocal(arg.user, idx, "%s> %s", user, arg.txt);
		return;
	}
	c = getcmd(arg.cmd);
	if (c == (struct cmd *)-1) msgreply("<%s -- Ambiguous command; ?-help>", arg.cmd);
	else if (c == 0) msgreply("<%s -- Invalid command; ?-help>", arg.cmd);
	else {
		if (c->host && !arg.host) msgreply("<%s -- Missing host arg>", c->name);
		else if (c->user && !arg.user) msgreply("<%s -- Missing user arg>", c->name);
		else if (c->txt && !arg.txt) msgreply("<%s -- Missing text arg>", c->name);
		else if (cmdpool < MAXCMDPOOL) {
			pid_t pid = fork();
			if (!pid) {
				syslog(LOG_DEBUG, "%s: %s> %s %s:%s %s",
				       inet_ntoa(cf[idx].hin->sin_addr), user,
				       arg.cmd, arg.host ? arg.host : "",
				       arg.user ? arg.user : "", arg.txt ? arg.txt : "");
				(*c->func)(idx, i, &arg);
				sleep(1);
				exit(0);
			} else if (pid < 0) {
				syslog(LOG_ERR, "fork: %m");
				msgreply("<I'm busy, try later>");
			} else cmdpool++;
		} else {
			syslog(LOG_DEBUG, "cmd denied: cmdpool exhausted");
			msgreply("<I'm busy, try later>");
		}
	}
}

void
hostalive(idx)
	int idx;
{
	register i;

	timenow = time(NULL);
	if (idx) {
		cf[idx].alive = timenow;
		if (timenow - cf[0].alive > TRICKLE) {
			userport = 0;
			msgremote(0, ">");
		}
	} else {
		userport = 0;
		msgremote(0, ">");
	}
	for (i = 1; i < hn; i++) {
		if (i != idx && timenow - cf[i].alive > MAXSILENT)
			cf[i].alive = 0;
	}
}

void
mainloop()
{
	int i, flag;
	char *msg, *txt;

again:
	while ((msg = waitmsg()) != NULL) {
		if (*msg == '<') {
			if (!strncmp(msg, GETINFOMARKER, sizeof(GETINFOMARKER)-1))
				msgreply("Not implemented yet");
			continue;
		}
		if ((txt = strpbrk(msg, ":>")) == NULL) continue;
		if (*txt == '>') {
			*txt = '\0';
			flag = 1;
		} else	flag = 0;
		for (txt++; *txt && (u_char)*txt <= 0x20; txt++);

		for (i = 0; i < hn; i++) {
			if (from.sin_addr.s_addr == cf[i].hin->sin_addr.s_addr)
				break;
		}
		if (i == hn) {
			syslog(LOG_WARNING, "message from unwanted host %s",
			       inet_ntoa(from.sin_addr));
			continue;
		}
		if (i) hostalive(i);
		if (!*msg || !*txt) continue;
		userport = 0;
		if (!flag) {
			if (i) 	msglocal(NULL, i, msg);
			else {
				msgremote(i, msg);
				if (psysmode) psysreply = psys(msg, txt);
			}
			continue;
		}
		if (i) {
			char *p;
			if ((p = strchr(msg, '%')) != NULL) {
				*p++ = '\0';
				from.sin_port = atoi(msg);
				msg = p;
			}
		} else userport = from.sin_port;
		hostcmd(i, msg, txt);
	}
	/* send trickle to all hosts */
	hostalive(0);
	goto again;
}

static void
onalarm()
{
	longjmp(after_alarm, 1);
}

FILE *
opendata(remote, cmd)
	int remote;
	char *cmd;
{
	int s, sock;
	struct sockaddr_in sin;
	sigfunc oldalrm;

	if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		syslog(LOG_ERR, "socket: %m");
		return NULL;
	}
	if (!remote || !cmd) {
		if (connect(s, (struct sockaddr *)&from, fromlen) < 0) {
			close(s);
			syslog(LOG_WARNING, "connect: %m");
			return NULL;
		}
		return fdopen(s, "w");
	}
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = INADDR_ANY;
	sin.sin_port = 0;
	if (bind(s, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		syslog(LOG_ERR, "bind: %m");
		return NULL;
	}
	sock = sizeof(sin);
	if (getsockname(s, (struct sockaddr *)&sin, &sock) < 0) {
		syslog(LOG_ERR, "getsockname: %m");
		return NULL;
	}
	if (listen(s, 1) < 0) syslog(LOG_ERR, "listen: %m");

	userport = sin.sin_port;
	msgremote(remote, cmd);
	msgreply("<%s requested, wait for reply>", inet_ntoa(cf[remote].hin->sin_addr));

	oldalrm = signal(SIGALRM, onalarm);
	alarm(MAXWAIT);
	if (setjmp(after_alarm)) {
		(void)signal(SIGALRM, oldalrm);
		close(s);
		syslog(LOG_WARNING, "accept: Connection timed out");
		return NULL;
	}
	sock = accept(s, 0, 0);
	alarm(0);
	(void)signal(SIGALRM, oldalrm);
	close(s);
	if (sock < 0) {
		syslog(LOG_ERR, "accept: %m");
		return NULL;
	}
#ifdef	IP_TOS
	s = IPTOS_THROUGHPUT;
	if (setsockopt(sock, IPPROTO_IP, IP_TOS, (char *)&s, sizeof(int)) < 0)
		syslog(LOG_ERR, "setsockopt: %m");
#endif
	return fdopen(sock, "r");
}

void
remotedata(remote, arg)
	int remote;
	struct args *arg;
{
	FILE *fp_in, *fp_out;
	int c, bytes = 0;
	char temp[100], buf[8192];

	if (!remote) return;

	/* Request data connection */
	sprintf(temp, "> %s %s:%s", arg->cmd, arg->host, arg->user ? arg->user : "");
	if ((fp_in = opendata(remote, temp)) == NULL) {
		msgreply("<%s not responding>", arg->host);
		return;
	}
	if (mktemp(strcpy(temp, TMPFILEMASK)) == NULL) {
		fclose(fp_in);
		syslog(LOG_ERR, "mktemp: %m");
		return;
	}
	if ((fp_out = fopen(temp, "w")) == NULL) {
		fclose(fp_in);
		syslog(LOG_ERR, "fopen: %m");
		return;
	}
	while ((c = read(fileno(fp_in), buf, sizeof(buf))) > 0) {
		if (write(fileno(fp_out), buf, c) != c) break;
		bytes += c;
		if (dflag) fprintf(stdout, "\r%d bytes received", bytes);
	}
	fclose(fp_in);
	fchmod(fileno(fp_out), 0644);
	fclose(fp_out);
	if (c < 0) {
		if (errno != EPIPE) syslog(LOG_ERR, "netin: %m");
	} else {
		msgreply("%s DISPLAY=%s", CMDSERVMARKER, temp);
		sleep(1);
	}
	unlink(temp);
}
