#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <stdlib.h>
#include <string.h>

/* Scan the directory dirname calling select to make a list of selected
 * directory entries then sort using qsort and compare routine dcomp.
 * Returns the number of entries and a pointer to a list of pointers to
 * struct dirent (through namelist). Returns -1 if there were any errors.
 */
#undef DIRSIZ
#define DIRSIZ(dp) \
    ((sizeof (struct dirent) - (MAXNAMLEN+1)) + (((dp)->d_namlen+1 + 3) &~ 3))

int
_scandir(dirname, namelist, select, dcomp)
	char *dirname;
	struct dirent ***namelist;
	int (*select) (struct dirent *);
	int (*dcomp) (const void *, const void *);
{
	register struct dirent *d, *p, **names;
	register size_t nitems;
	struct stat stb;
	long arraysz;
	DIR *dirp;

	if ((dirp = opendir(dirname)) == NULL)
		return -1;
	if (fstat(dirp->dd_fd, &stb) < 0)
		return -1;

	/*
	 * estimate the array size by taking the size of the directory file
	 * and dividing it by a multiple of the minimum size entry. 
	 */
	arraysz = (stb.st_size / 24);
	names = (struct dirent **)malloc(arraysz * sizeof(struct dirent *));
	if (names == NULL)
		return -1;

	nitems = 0;
	while ((d = readdir(dirp)) != NULL) {
		if (select != NULL && !(*select)(d))
			continue;	/* just selected names */
		/*
		 * Make a minimum size copy of the data
		 */
#if defined(SVR4)
		p = (struct dirent *)malloc(d->d_reclen);
#else	/* BSD, Linux */
		p = (struct dirent *)malloc(DIRSIZ(d));
#endif
		if (p == NULL) return -1;
		(void)strcpy(p->d_name, d->d_name);
		/*
		 * Check to make sure the array has space left and
		 * realloc the maximum size.
		 */
		if (++nitems >= arraysz) {
			if (fstat(dirp->dd_fd, &stb) < 0)
				return -1;	/* just might have grown */
			arraysz = stb.st_size / 12;
			names = (struct dirent **)realloc((char *)names,
				arraysz * sizeof(struct dirent *));
			if (names == NULL)
				return -1;
		}
		names[nitems-1] = p;
	}
	closedir(dirp);
	if (nitems && dcomp != NULL)
		qsort(names, nitems, sizeof(struct dirent *), dcomp);
	*namelist = names;
	return nitems;
}

/*
 * Alphabetic order comparison routine for those who want it.
 */
int
_alphasort(d1, d2)
	const void *d1;
	const void *d2;
{
	return (strcmp((*(struct dirent **)d1)->d_name,
		       (*(struct dirent **)d2)->d_name));
}

/*
 * Free memory allocated by scandir()
 */
void
freedir(dplist, listlen)
	struct dirent **dplist;
	int listlen;
{
	register i;
	if (dplist != NULL) {
		for (i = 0; i < listlen; i++) free(dplist[i]);
		free(dplist);
	}
}
