/*
 *	Copyright (c) 1996 The CAD lab of the
 *	Siberian State Academy of Telecommunication
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <syslog.h>
#include <pwd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include "netchat.h"

void usage();
void cleanup();
void reloadcf();
extern void cmd_done();
extern void initmon();

char *configfile;
char *helpfile;
char *username;
char *psysmode;
char *orgdir;
int dflag;


void daemon(int a, int b)
{
    int c;

    switch(fork()) {
	case -1:
	    perror("fork()");
	    exit(1);
	    break;
	case 0:
	    c = open("/dev/tty", O_RDWR);
	    if(c >= 0) {
		ioctl(c, TIOCNOTTY, (char *)0);
		close(c);
	    }    
	    return;
	default:
	    exit(0);
    }
}


int
main(argc, argv)
	int argc;
	char *argv[];
{
	int op, port;
	struct passwd *pwd;
	extern int optind, opterr;
	extern char *optarg;

	if ((pwd = getpwnam(BBS)) == NULL && (pwd = getpwnam("tns")) == NULL) {
		fprintf(stderr, "No %s in this system\n", BBS);
		exit(1);
	}
	if (geteuid() != pwd->pw_uid) {
		if (getuid() != 0) {
			fprintf(stderr, "No permission. %s or root required\n", BBS);
			exit(1);
		}
		if (setuid(pwd->pw_uid) < 0) {
			fprintf(stderr, "Can't set the user ID to %s\n", BBS);
			exit(1);
		}
	}
	configfile = NETCHATCONF;
	helpfile = NETCHATHELP;
	username = NETCHATNAME;
	psysmode = NULL;
	orgdir = pwd->pw_dir;
	opterr = port = dflag = 0;
	while ((op = getopt(argc, argv, "c:h:f:n:p:d?")) != EOF)
		switch (op) {
			case 'c':	/* config file */
				configfile = optarg;
				break;
			case 'h':	/* help file */
				helpfile = optarg;
				break;
			case 'f':	/* psys language file; enable psys */
				psysmode = optarg;
				break;
			case 'n':	/* my name */
				username = optarg;
				break;
			case 'p':	/* port number */
				if ((port = atoi(optarg)) < 1) usage();
				break;
			case 'd':	/* debug mode */
				dflag++;
				break;
			case '?':
			default:
				usage();
		}

	initcallsign(orgdir);

	(void)signal(SIGHUP, dflag ? cleanup : SIG_IGN);
	(void)signal(SIGINT, dflag ? cleanup : reloadcf);
	(void)signal(SIGTERM, cleanup);
	(void)signal(SIGCHLD, cmd_done);
	(void)signal(SIGQUIT, SIG_IGN);
	(void)signal(SIGPIPE, SIG_IGN);
	(void)signal(SIGUSR2, SIG_IGN);
	(void)signal(SIGUSR1, dflag ? SIG_IGN : initmon);

	if (!dflag) daemon(1, 0);
	openlog(NETCHAT, LOG_PID|LOG_CONS, LOG_DAEMON);

	if (initnet(port) < 0 || saveinfo(INCONF) < 0) {
		if (!dflag) syslog(LOG_ERR, "%s aborted", NETCHAT);
		exit(1);
	}
	if (psysmode && initpsys(psysmode) < 0) psysmode = NULL;

	if (!dflag) syslog(LOG_NOTICE, "%s listen port %d", NETCHAT, ntohs(myport));
	mainloop();
}

void
cleanup()
{
	FILE *fp;

	purgeinfo();
	if (!dflag) {
		if ((fp = fopen(TNSMONPID, "r")) != NULL) {
			char buf[40];
			if (fgets(buf, sizeof(buf), fp) != NULL)
				kill(atoi(buf), SIGUSR1);
		}
		syslog(LOG_NOTICE, "%s stoped", NETCHAT);
	} else printf("Interrupted\n");
	exit(0);
}

void
reloadcf(sig)
	int sig;
{
	if (initcf(configfile) < 0) {
		purgeinfo();
		if (!dflag) syslog(LOG_ERR, "%s aborted", NETCHAT);
		exit(1);
	}
	syslog(LOG_NOTICE, "reloadcf: %s", configfile);
	(void)signal(sig, reloadcf);
}

void
usage()
{
	fprintf(stderr, "%s for TNSDrive\n", NC);
	fprintf(stderr, "usage: netchat [-d] [-c conf] [-h help] [-f file] [-n name] [-p port]\n");
	fprintf(stderr, "where:\n\
  -d      \tDebug mode\n\
  -c conf \tConfig file, default %s\n\
  -h help \tHelp file, default %s\n\
  -f file \tPsys dictionary file, default psys disabled\n\
  -n name \tSelf name for conference, default \"%s\"\n\
  -p port \tPort number, default %s/udp\n",
		NETCHATCONF,
		NETCHATHELP,
		NETCHATNAME,
		NETCHAT);
	exit(1);
}
