/*
 *	Copyright (c) 1995 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <termios.h>
#include <errno.h>
#include <setjmp.h>

#include "drive.h"
#include "variables.h"
#include "loginfo.h"
#include "udpserv.h"
#include "monitor.h"

#define	MAXUDPWAIT		3	/* sec for response */
#define	MAXINPUTSTRLEN		256
#define	LINE1HELP		"\
  F1-help  F2-who  F3-info  F4-chat  ESC-return  DEL-logoff  INS-display_file"
#define	LINE2HELP		"\
  HOME-set_variable  END-get_variable  PageUp+5min  PageDn-5min     or assist"

extern int errno;

static struct termios tbufsave;
static int sock;

void
ttysetraw(rate)
	int rate;
{
	int speed;
	struct termios tbuf;
	static int speed_tab[] = {
	0, 50, 75, 110, 134, 150, 200, 300, 600, 1200, 1800, 2400, 4800, 9600,
	19200, 38400, 57600, 76800, 115200, 153600, 230400, 307200, 460800 };

	tcgetattr(0, &tbufsave);
	speed = (int)cfgetospeed(&tbufsave);
	if (speed > 0 && speed < 23) speed = speed_tab[speed];
	if (speed < rate)
		printf("Connected at %d (user at %d) - might lose data\n",
		       speed, rate);
	tbuf = tbufsave;
	cfmakeraw(&tbuf);
	tbuf.c_iflag = (IXANY | IGNPAR | IXON);
	tbuf.c_lflag = ISIG;
	tbuf.c_oflag = (OPOST | ONLCR);
	tbuf.c_cc[VINTR] = 0x03;    /* ^C */
	tbuf.c_cc[4] = 1;
	tbuf.c_cc[5] = 5;
	tcsetattr(0, TCSAFLUSH, &tbuf);
}

void
ttyrestore()
{
	tcsetattr(0, TCSAFLUSH, &tbufsave);
}

static int
outstr(int fd, ...)
{
	register char *cp;
	register len;
	unsigned char sb[1024];
	va_list ap;

	va_start(ap, fd);
	cp = va_arg(ap, char *);
	len = vsnprintf(sb, sizeof(sb), cp, ap);
	va_end(ap);
	return write(fd, sb, len);
}

static int
inputchar()
{
	int ch = 0;
	while (read(0, &ch, 1) < 0 && errno == EINTR);
	return ch & 0377;
}

int
eraseline(i)
	register i;
{
	if (i) while (i) {
		i--;
		write(STDOUT_FILENO, "\b \b", 3);
	}
	return i;
}

char *
inputstr()
{
	int ch;
	int touch = FALSE;
	static int i = 0;
	static char strbuf[MAXINPUTSTRLEN];

	strbuf[i] = '\0';
	if (i) outstr(STDOUT_FILENO, strbuf);
	while (1) {
		while((ch = inputchar()) < ' ' || ch == 127) {
			if (ch == '\r' || ch == '\n') break;
			touch = TRUE;
			if (ch == '\b' || ch == 127) { /* Backspace */
				if (i) {
					i--;
					eraseline(1);
				} else write(STDOUT_FILENO, "\007", 1);
			} else if (ch == 21) {	/* Control-U, erase line */
				if (i) i = eraseline(i);
				else write(STDOUT_FILENO, "\007", 1);
			} else write(STDOUT_FILENO, "\007", 1);
		}
		if (ch == '\r' || ch == '\n') break;
		if (!touch) {
			i = eraseline(i);
			touch = TRUE;
		}
		if (i + 1 <= MAXINPUTSTRLEN) {
			strbuf[i++] = ch;
			strbuf[i] = '\0';
			write(STDOUT_FILENO, &ch, 1);
		} else write(STDOUT_FILENO, "\007", 1);
	}
	strbuf[i] = '\0';
	if (i > 0) return strbuf;
	return NULL;
}

int
initudpsock()
{
	int s, on = 1;
	struct sockaddr_in sin;

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("socket");
		return -1;
	}
	if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (void *)&on, sizeof(on)) < 0) {
		close(s);
		perror("setsockopt SO_REUSEADDR");
		return -1;
	}
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = htonl(INADDR_ANY);
	sin.sin_port = 0;
	if (bind(s, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		close(s);
		perror("bind");
		return -1;
	}
	return s;
}

char *
fn_tty(tty)
	char *tty;
{
	register i;
	static char buf[12];

	for (i = 0; *tty && i < sizeof(buf)-1; tty++)
		if (*tty != '/') buf[i++] = *tty;
	buf[i] = '\0';
	return buf;
}

int
grabtty(user)
	struct loginfo *user;
{
	int sockin, rval;
	struct sockaddr_un in;

	if ((sockin = socket(AF_UNIX, SOCK_DGRAM, 0)) < 0) {
		perror("socket");
		return -1;
	}
	in.sun_family = AF_UNIX;
	sprintf(in.sun_path, "%s/%s.IN", MONITORDIR, fn_tty(user->tty));
	unlink(in.sun_path);
	if (bind(sockin, (struct sockaddr *)&in,
		 sizeof(struct sockaddr_un)) < 0) {
		close(sockin);
		perror("bind");
		return -1;
	}
	if ((sock = initudpsock()) < 0) {
		close(sockin);
		return -1;
	}
	if (kill(user->pid, SIGUSR1) < 0) {
		close(sockin);
		perror("kill");
		return -1;
	}
	sleep(1);
	ttysetraw(user->baud);
	rval = showtty(sockin, user);
	ttyrestore();
	close(sock);
	close(sockin);
	unlink(in.sun_path);
	return rval;
}

static jmp_buf dontanswer;

static void
waitforanswer()
{
	alarm(0);
	signal(SIGALRM, SIG_DFL);
	longjmp(dontanswer, 1);
}

char *
getvalue(int port, ...)
{
	register i;
	int len;
	char *cp;
	va_list ap;
	struct sockaddr_in to;
	struct sockaddr_in from;
	static char buf[1024];

	va_start(ap, port);
	cp = va_arg(ap, char *);
	strcpy(buf, GETINFOMARKER);
	len = strlen(buf);
	(void)vsprintf(&buf[len], cp, ap);
	va_end(ap);
	len = strlen(buf);

	to.sin_family = AF_INET;
	to.sin_addr.s_addr = inet_addr(LOOPBACK);
	to.sin_port = port;

	if (setjmp(dontanswer)) {
		fprintf(stderr, "recvfrom: Operation timed out\n");
		return NULL;
	}
	signal(SIGALRM, waitforanswer);
	alarm(MAXUDPWAIT);
	if (sendto(sock, buf, len, 0, (struct sockaddr *)&to, sizeof(to)) < 0) {
		perror("sendto");
		return NULL;
	}
	len = sizeof(from);
	while ((i = recvfrom(sock, buf, sizeof(buf)-1, 0,
			(struct sockaddr *)&from, &len)) < 1)
		if (i < 0 && errno != EINTR) {
			perror("recvfrom");
			alarm(0);
			signal(SIGALRM, SIG_DFL);
			return NULL;
		}
	alarm(0);
	signal(SIGALRM, SIG_DFL);
	buf[i] = '\0';
	return buf;
}

int
setvalue(int port, ...)
{
	register len;
	char *cp;
	va_list ap;
	struct sockaddr_in to;
	char buf[1024];

	va_start(ap, port);
	cp = va_arg(ap, char *);
	strcpy(buf, CMDSERVMARKER);
	len = strlen(buf);
	(void)vsprintf(&buf[len], cp, ap);
	va_end(ap);
	len = strlen(buf);

	to.sin_family = AF_INET;
	to.sin_addr.s_addr = inet_addr(LOOPBACK);
	to.sin_port = port;

	return sendto(sock, buf, len, 0, (struct sockaddr *)&to, sizeof(to));
}

int
settimeleft(port, min, up)
	int port, min, up;
{
	char *p;
	int nb, hh, mm, ss;

	if ((p = getvalue(port, USERTIMELEFT)) == NULL) return -1;
	if ((nb = sscanf(p, "%d:%d:%d", &hh, &mm, &ss)) < 2) return -1;
	if (nb == 2) {
		mm = hh;
		hh = 0;
	}
	mm += hh * 60;
	if (up) mm += min;
	else {
		mm -= min;
		if (mm < 2) mm = 2;
	}
	return setvalue(port, "%s=%d", USERTIMELEFT, mm);
}

void
showstate(port)
	int port;
{
#define	NKW	8
	register i;
	char *p, *line, buf[256];
	char *kw[NKW] = { CALLNUMBER, USERPRIVLEVEL, USERGROUP, USERSTATUS,
			INTERRUPT, TIMERATE, USERTIMELEFT, USERSTATE };

	for (i = 0, buf[0] = 0; i < NKW; i++) {
		if (buf[0] != '\0') strcat(buf, " ");
		strcat(buf, kw[i]);
	}
	write(STDOUT_FILENO, "\n", 1);
	for (i = 0, p = getvalue(port, buf);
	     p != NULL && i < NKW; i++, p = line) {
		if ((line = strchr(p, '\t')) != NULL) *line++ = '\0';
		outstr(STDOUT_FILENO, "%s%10s: %s%s\n", BEGSTRING, kw[i], p, ENDSTRING);
	}
#undef	NKW
}

showtty(sockin, user)
	int sockin;
	struct loginfo *user;
{
	register nb;
	int ch, sockout, inchat = 0;
	char *p;
	fd_set allsock, ready;
	struct sockaddr_un out;
	struct timeval timeout = {1, 0};	/* 1 sec */
	char sockbuf[8192];
	extern int redraw;

	if ((sockout = socket(AF_UNIX, SOCK_DGRAM, 0)) < 0) {
		perror("socket");
		return -1;
	}
	if ((p = getvalue(user->port, USERSTATE)) != NULL)
		outstr(STDOUT_FILENO, "User state: %s%s%s\n",
		       BOLDSTRING, p, ENDSTRING);

	outstr(STDOUT_FILENO, "\n%s%s%s\n", BEGSTRING, LINE1HELP, ENDSTRING);
	outstr(STDOUT_FILENO, "%s%s%s\n", BEGSTRING, LINE2HELP, ENDSTRING);

	out.sun_family = AF_UNIX;
	sprintf(out.sun_path, "%s/%s.OUT", MONITORDIR, fn_tty(user->tty));

	FD_ZERO(&allsock);
	FD_SET(0, &allsock);
	FD_SET(sockin, &allsock);

	while (1) {
		ready = allsock;
		if ((nb = select(sockin + 1, &ready, NULL, NULL, &timeout)) <= 0) {
			if (!nb) {
				if (redraw && kill(user->pid, 0)) break;
				continue;
			}
			if (errno == EINTR) continue;
			close(sockout);
			perror("select");
			return -1;
		}
		if (FD_ISSET(sockin, &ready)) {
			while ((nb = read(sockin, sockbuf, sizeof(sockbuf)-1)) < 0 &&
			       errno == EINTR);
			write(STDOUT_FILENO, sockbuf, nb);
		}
		if (FD_ISSET(0, &ready)) {
			ch = inputchar();
			switch(ch) {
			case ESC:
				switch(get_arrow_key(inputchar)) {
				case KEYMAP_ESCAPE:	/* return */
					if (!inchat) goto end_show;
					inchat = 0;
					if (kill(user->pid, SIGUSR1) < 0)
						perror("kill");
					else 	sendto(sockout, ".", 1, 0,
						       (struct sockaddr *)&out, sizeof(struct sockaddr_un));
					continue;
				case KEYMAP_F1:		/* help */
					outstr(STDOUT_FILENO, "\n%s%s%s\n",
					       BEGSTRING, LINE1HELP, ENDSTRING);
					outstr(STDOUT_FILENO, "%s%s%s\n",
					       BEGSTRING, LINE2HELP, ENDSTRING);
					break;
				case KEYMAP_F2:		/* who */
					outstr(STDOUT_FILENO, "\n%s\%-6.6s  %.15s  %-24.24s %d  pid %d%s\n",
					       BEGSTRING,
					       user->tty, ctime(&user->ltime)+4,
					       user->name, user->baud, user->pid,
					       ENDSTRING);
					break;
				case KEYMAP_F3:		/* info */
					if (inchat) goto bad_func;
					showstate(user->port);
					break;
				case KEYMAP_F4:		/* chat */
					if (inchat || (p = getvalue(user->port, "USERSTATE")) == NULL)
						goto bad_func;
					if (strcmp(p, "BBSing")) {
						outstr(STDOUT_FILENO, "\n%sUser state: %s\n", BEGSTRING, p);
						outstr(STDOUT_FILENO, "\007%sAre you sure?", BEGSTRING);
						ch = inputchar();
						if (ch == '\r' || ch == '\n' ||
						    ch == '\b' || ch == 0x7f) ch = 'N';
						outstr(STDOUT_FILENO, " %c%s\n", ch, ENDSTRING);
						if (ch != 'y' && ch != 'Y') break;
					}
					if (kill(user->pid, SIGUSR2) < 0)
						perror("kill");
					else 	inchat = 1;
					break;
				case KEYMAP_PAGE_UP:	/* incr 5 min */
					if (inchat) goto bad_func;
					settimeleft(user->port, 5, 1);
					break;
				case KEYMAP_PAGE_DOWN:	/* decr 5 min */
					if (inchat) goto bad_func;
					settimeleft(user->port, 5, 0);
					break;
				case KEYMAP_HOME:	/* set var to value */
					if (inchat) goto bad_func;
					outstr(STDOUT_FILENO, "\n%sSet variable: ", BEGSTRING);
					p = inputstr();
					if (p != NULL) {
						if (strchr(p, '=')) setvalue(user->port, p);
						else outstr(STDOUT_FILENO, "\007  Usage: VAR=value");
					}
					outstr(STDOUT_FILENO, "%s\n", ENDSTRING);
					break;
				case KEYMAP_END:	/* get value of var */
					if (inchat) goto bad_func;
					outstr(STDOUT_FILENO, "\n%sGet variable: ", BEGSTRING);
					p = inputstr();
					if (p != NULL) {
						if (!strchr(p, '='))
							p = getvalue(user->port, p);
						else {
							p = NULL;
							outstr(STDOUT_FILENO, "\007  Usage: VAR");
						}
					}
					outstr(STDOUT_FILENO, "%s\n", ENDSTRING);
					if (p != NULL)
						outstr(STDOUT_FILENO, "%s%s%s\n",
						       BEGSTRING, p, ENDSTRING);
					break;
				case KEYMAP_INS:	/* insert(exec) file */
					outstr(STDOUT_FILENO, "\n%sDisplay file: ", BEGSTRING);
					p = inputstr();
					outstr(STDOUT_FILENO, "%s\n", ENDSTRING);
					if (p != NULL)
						setvalue(user->port, "DISPLAY=%s", p);
					break;
bad_func:
				default:
					write(STDOUT_FILENO, "\007", 1);
				}
				break;
			case 0x7f:	/* logoff user */
				kill(user->pid, SIGTERM);
				break;
			default:	/* assist user */
				sendto(sockout, (void *)&ch, 1, 0,
				       (struct sockaddr *)&out, sizeof(struct sockaddr_un));
			}
		}
	}
end_show:
	close(sockout);
	return 0;
}
