/*
 *	Copyright (c) 1994,1995 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>

#include "compat.h"
#include "drive.h"
#include "sysmsg.h"
#include "group.h"
#include "rematch.h"

extern int errno;

char filesdir[MAXPATHLEN+1];

static char dirdescfile[MAXPATHLEN+1];
static char **dirmask = NULL;
static int nmasks = 0;
struct dt_ent {
	char *dirname;
	time_t mtime;
};
static struct desc_ent {
	int nlen;
	char *name;
	char *desc;
} *dirdesc = NULL;
static int ndescs = 0;

freedirtree()
{
	register i;

	dirdescfile[0] = '\0';
	if (dirmask != NULL) {
		for (i = 0; i < nmasks; i++) free(dirmask[i]);
		free(dirmask);
		dirmask = NULL;
	}
	nmasks = 0;
	if (dirdesc != NULL) {
		for (i = 0; i < ndescs; i++) {
			free(dirdesc[i].name);
			free(dirdesc[i].desc);
		}
		free(dirdesc);
		dirdesc = NULL;
	}
	ndescs = 0;
}

setdirdescfile(str)
	char *str;
{
	int i, n;
	char *top, *ptr, *line, *name, *desc;
	struct stat st;

	if (*str != '/') {
		strncpy(dirdescfile, str, sizeof(dirdescfile)-1);
		dirdescfile[sizeof(dirdescfile)-1] = '\0';
		return;
	}
	if (stat(str, &st) < 0 || !st.st_size) return;
	if ((top = ptr = malloc((int)st.st_size + 2)) == NULL) return;
	if ((n = open(str, O_RDONLY)) < 0) {
		free(top);
		return;
	}
	while ((i = read(n, ptr, (int)st.st_size)) < 0 && errno == EINTR);
	close(n);
	if (i != st.st_size) {
		free(top);
		return;
	}
	if (ptr[i-1] != '\n') ptr[i++] = '\n';
	ptr[i] = '\0';

	ndescs = 0;
	dirdesc = NULL;
	for (; (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#') continue;
		name = ptr;
		while ((u_char)*ptr > 0x20) ptr++;
		if (*ptr) for (*ptr++ = 0; *ptr && (u_char)*ptr <= 0x20; ptr++);
		if (!*ptr || isdigitstr(ptr)) continue;
		desc = ptr;
		if ((ptr = strrchr(name, '.')) != NULL)
			if (isdigitstr(&ptr[1])) *ptr = '\0';
		dirdesc = realloc(dirdesc, (ndescs + 1) * sizeof(struct desc_ent));
		if (dirdesc == NULL) break;
		dirdesc[ndescs].nlen = strlen(name);
		dirdesc[ndescs].name = strdup(name);
		dirdesc[ndescs].desc = strdup(desc);
		ndescs++;
	}
	free(top);
	return;
}

static char *
getdirdesc(dirtop, dirname, buf)
	char *dirtop, *dirname, *buf;
{
	register i, l;
	register char *ptr;
	FILE *fp;

	if (ndescs) {
		sprintf(buf, "%s/%s", dirtop, dirname);
		l = strlen(buf);
		for (i = 0; i < ndescs; i++)
			if (dirdesc[i].nlen == l &&
			    !strcmp(dirdesc[i].name, buf))
				return dirdesc[i].desc;
		*buf = '\0';
		return buf;		
	}
	if (dirdescfile[0] == '\0') {
		*buf = '\0';
		return buf;
	}
	sprintf(buf, "%s/%s/%s", dirtop, dirname, dirdescfile);
	if ((fp = sfopen(buf, "r")) == NULL) {
		*buf = '\0';
		return buf;
	}
	while (fgets(buf, 256, fp) != NULL) {
		buf[255] = '\0';
		if ((ptr = strchr(buf, '\n')) != NULL) *ptr = '\0';
		for (ptr = buf; *ptr && (u_char)*ptr <= 0x20; ptr++);
		if (*ptr != '\0' && *ptr != '#') {
			sfclose(fp);
			return ptr;
		}
	}
	sfclose(fp);
	*buf = '\0';
	return buf;
}

int
adddirmask(mask)
	char *mask;
{
	dirmask = realloc(dirmask, (nmasks + 1) * sizeof(char *));
	if (dirmask != NULL) dirmask[nmasks++] = strdup(mask);
	return 0;
}

int
isdirmasked(path)
	char *path;
{
	register i;

	if (nmasks) {
		for (i = 0; i < nmasks; i++)
			if (rematch(dirmask[i], path)) break;
		if (i < nmasks) return TRUE;
	}
	return !isgrpright(GSID_FILE, path, GRF_S);
}

static int
sortdtbyname(d1, d2)
	register struct dt_ent *d1;
	register struct dt_ent *d2;
{
	if (d1->dirname[0] == '.') return -1;
	return (strcmp(d1->dirname, d2->dirname));
}

int
godirtree(dirtop)
	register char *dirtop;
{
	register n, i;
	register char *ptr;
	DIR *dirp;
	struct dirent *d;
	struct stat st;
	struct dt_ent dirs[MAXCONFLINES];
	char ch, buf[1024];
	int files, lncnt, rval, fidx[MAXSCRLINES];
	time_t date;
	extern char working[];

	if ((dirp = opendir(dirtop)) == NULL) return -1;

	clearline();
	putstr("\r%s-", makeshowpath(dirtop));
	files = n = i = 0;
	
	while (n < MAXCONFLINES && (d = readdir(dirp)) != NULL) {
		if (d->d_name[0] == '.' &&
		    (d->d_name[1] != '.' || d->d_name[2])) continue;
		sprintf(buf, "%s/%s", dirtop, d->d_name);
		if (isdirmasked(buf)) continue;
		if (!(n % 10)) putstr("\b%c", working[++i & 3]);
		if (stat(buf, &st) < 0 || !(st.st_mode & S_IRUSR)) continue;
		if ((st.st_mode & S_IFMT) == S_IFDIR) {
			if (!(st.st_mode & S_IXUSR)) continue;
			if ((dirs[n].dirname = strdup(d->d_name)) == NULL)
				continue;
			dirs[n++].mtime = st.st_mtime;
		} else if ((st.st_mode & S_IFMT) == S_IFREG) files++;
	}
	(void)closedir(dirp);
	if (!n) return 0;

	qsort(dirs, n, sizeof(struct dt_ent), sortdtbyname);
	for (i = 0; i < n; i++) {
		sprintf(buf, "%s/%s", dirtop, dirs[i].dirname);
		if (!strncmp(filesdir, buf, strlen(buf))) break;
	}
	if (i < n) {
		if ((i = godirtree(buf)) < 0)
			LOGIT(LOG_ERR, "can't opendir \"%s\": %m", buf);
		if (i > 0) {
			for (i = 0; i < n; i++) free(dirs[i].dirname);
			return 1;
		}
	}
	filesdir[0] = '\0';

select_again:
	lncnt = 1;
	for (i = 0; i < n; i++) {
		if (lncnt == 1) {
			lncnt += 2;
			memset(fidx, 0, sizeof(fidx));
			fidx[0] = -1;
			snprintf(buf, MAXSTRINGLENGTH-12, "%s %s",
				 sysmsg(MSG_AFILEAREAHEAD), makeshowpath(dirtop));
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[H\033[J\033[1;37;44m %-68.68s %4d/%-4d\033[K\033[40m\n\n", buf, n, files);
			else {
				clearline();
				putstr("\n %s %4d/%-4d\n\n", buf, n, files);
			}
		}
		fidx[lncnt-1] = i+1;
		sprintf(buf, "%s/%s", dirtop, dirs[i].dirname);
		ptr = getrcent(FILE_RC, buf);
		if (ptr != NULL)
			date = atoi(ptr);
		else	date = lastcalltime;
		if (dirs[i].mtime > date) ch = '*';
		else ch = ' ';
		ptr = getdirdesc(dirtop, dirs[i].dirname, buf);
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;37m%4d \033[31m%c \033[33m%-32.32s  \033[0;36m%-.38s\n",
			       i+1, ch, dirs[i].dirname, ptr);
		else	putstr("%4d) %c %-31.31s  %-.38s\n",
			       i+1, ch, dirs[i].dirname, ptr);
more_again:
		rval = morenum(MSG_SELECTFAREA, &lncnt, n);
		if (rval == -1) goto quit_list;
		if (rval > 0) {
			if (rval > n) {
				rval = fidx[(rval-n-1)/2];
				if (rval < 0) goto quit_list;
				if (!rval) {
					lncnt = scrlines;
					goto more_again;
				}
			}
			--rval;
			if (!strcmp(dirs[rval].dirname, "..")) {
				for (i = 0; i < n; i++) free(dirs[i].dirname);
				return 0;
			}
			sprintf(buf, "%s/%s", dirtop, dirs[rval].dirname);
			if ((i = godirtree(buf)) < 0)
				LOGIT(LOG_ERR, "can't opendir \"%s\": %m", buf);
			if (i <= 0) goto select_again;
			for (i = 0; i < n; i++) free(dirs[i].dirname);
			return 1;
		}
		if (rval == -2)	i -= (scrlines - 2) * 2;
		else if (rval == -3) i = n - (scrlines - 2) - 1;
		else if (rval == -4) i = -1;
		if (i < -1) i = -1;
	}
	lncnt = scrlines;
	goto more_again;

quit_list:
	(void)strcpy(filesdir, dirtop);
	for (i = 0; i < n; i++) free(dirs[i].dirname);
	return 1;
}
