/*
 *	Copyright (c) 1994,1995 The CAD lab of the
 *	Siberian State Academy of Telecommunication, RU
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

/*
 * Copyright (c) 1985, 1989, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <time.h>
#include <sys/time.h>
#include <sys/file.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#include <arpa/ftp.h>
#include <arpa/telnet.h>

#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#ifdef	__STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <setjmp.h>
#include <ctype.h>

#include "compat.h"
#include "drive.h"
#include "sysmsg.h"
#include "ftp.h"
#include "parseftp.h"
#include "zmodem.h"
#include "zmdm.h"
#include "group.h"
#include "loginfo.h"

extern struct loginfo process;
extern pid_t pid;
extern int dailydlsize, dailyulsize, userdlsize, userulsize, dailydllimit, ulmultiplier, dlulratio;
extern char *fromfield();
extern void *xferlog();

static	struct	sockaddr_in hisctladdr;
static	struct	sockaddr_in data_addr;
static	int	data;
static	int	abrtflag;
static	struct	sockaddr_in myctladdr;
static	FILE	*cin, *cout;
static	char	reply_string[BUFSIZ];	/* last line of previous reply */

static	int	verbose;
static	int	rcode;
static	int	sendport;
static	u_short	defport;
static	int	cpend;
static	jmp_buf	recvabort;

char	*FTPserver = NULL;	/* connected FTP server */
char	FTPpwd[256];		/* current FTP work directory */
char	TransferFile[512];

char *ftp_connect(char *, int);
void recvrequest(FILE *, char *, long);
FILE *dataconn(void);
void ptransfer(char *, long, struct timeval *, struct timeval *);
void tvsub(struct timeval *, struct timeval *, struct timeval *);
void abort_remote(FILE *);
void lostpeer(void);

void
cmdabort()
{
	abrtflag++;
}

static int
#ifdef	__STDC__
command(char *fmt, ...)
#else
command(fmt, va_alist)
	char *fmt;
	va_dcl
#endif
{
	va_list ap;
	int r;
	sigfunc oldintr;

	abrtflag = 0;
	if (cout == NULL) {
		putstr("No control connection for command\n");
		FTPserver = NULL;
		FTPpwd[0] = '\0';
		rcode = -1;
		return (0);
	}
	oldintr = signal(SIGINT, cmdabort);
#ifdef	__STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	vfprintf(cout, fmt, ap);
	va_end(ap);
	fprintf(cout, "\r\n");
	(void) fflush(cout);
	cpend = 1;
	r = getreply(!strcmp(fmt, "QUIT"));
	if (abrtflag && oldintr != SIG_IGN) (*oldintr)(SIGINT);
	(void) signal(SIGINT, oldintr);
	return (r);
}

initftparea()
{
	struct servent *sp;
	if ((sp = getservbyname("ftp", "tcp")) == 0) {
		LOGIT(LOG_WARNING, "ftp/tcp: unknown service, assume port 21");
		defport = DEFAULT_PORT;
	} else	defport = ntohs(sp->s_port);
	FTPpwd[0] = '\0';
	TransferFile[0] = '\0';
}

int
open_ftpbyent(ftp)
	struct ftpserv_ent *ftp;
{
	if (!isgrpright(GSID_FTP, ftp->host, GRF_R)) {
		nopermission(GRF_R);
		return 0;
	}
	FTPserver = ftp_connect(ftp->host, ftp->port);
	if (FTPserver != NULL) {
		if (ftp_login(ftp->user, ftp->pswd) == 1) return 1;
		ftp_disconnect();
	}
	anykey();
	return 0;
}

int
open_ftpbyname(host)
	char *host;
{
	if (!isgrpright(GSID_FTP, host, GRF_R)) {
		nopermission(GRF_R);
		return 0;
	}
	FTPserver = ftp_connect(host, DEFAULT_PORT);
	if (FTPserver != NULL) {
		if (ftp_login(NULL, NULL) == 1) return 1;
		ftp_disconnect();
	}
	return 0;
}

static dir_file_info ftpfile[MAXCONFLINES];
static int ftpfiles = 0;

int
open_ftpdir()
{
	register i;
	register char *ptr, *line;
	struct stat st;
	int fd;
	static char *cachebuf = NULL;
	static int cachesize = 0;

	if (FTPserver == NULL) {
		warning(MSG_OPENFTPFIRST, 1);
		return -1;
	}
	strcpy(tmpbuf, FTPserver);
	strcat(tmpbuf, FTPpwd);
	if (!isgrpright(GSID_FTP, tmpbuf, GRF_S)) {
		nopermission(GRF_S);
		return -1;
	}

	if (stat(ftpcachfile, &st) < 0) {
		FILE *fp;
		if ((fp = fopen(ftpcachfile, "w")) == NULL) return -1;
		recvrequest(fp, NULL, 0);
		fclose(fp);
		if (stat(ftpcachfile, &st) < 0) {
			warning(MSG_NOFILESAVAIL, 1);
			return -1;
		}
	}
	if (!st.st_size) {
		unlink(ftpcachfile);
		warning(MSG_NOFILESAVAIL, 1);
		return -1;
	}
	if (cachesize == (int)st.st_size) return 0;

	if (cachebuf != NULL) free(cachebuf);
	if ((cachebuf = malloc((int)st.st_size + 2)) == NULL) return -1;
	if ((fd = open(ftpcachfile, O_RDONLY)) < 0) return -1;
	while ((i = read(fd, cachebuf, (int)st.st_size)) < 0 && errno == EINTR);
	close(fd);
	if (i != (int)st.st_size) return -1;
	if (cachebuf[i-1] != '\n') cachebuf[i++] = '\n';
	cachebuf[i] = '\0';

	for (i = 0, ptr = cachebuf;
	     i < MAXCONFLINES && (line = strchr(ptr, '\n')) != NULL;
	     ptr = ++line) {
		*line = '\0';
		if (line > ptr && *(line-1) == '\r') *(line-1) = '\0';
		if (*ptr && parse_ls_line(ptr, &ftpfile[i]) == 0) i++;
	}
	cachesize = (int)st.st_size;
	ftpfiles = i;
	if (!i) {
		warning(MSG_NOFILESAVAIL, 1);
		return -1;
	}
	return 0;
}

void
FTP_open()
{
	char *ptr;

	if ((ptr = prompt_str(MSG_ENTERFTPHOST, NULL, 0)) != NULL) {
		putchr('\n');
		open_ftpbyname(ptr);
	}
}

void
FTP_select()
{
	int i, n, lncnt, rval, fidx[MAXSCRLINES];
	char *ptr, *ptr2, *line, *defuser, *defpswd;
	struct ftpserv_ent ftp[MAXCONFLINES];

	if (nchr > 5) nchr = 0;
	sprintf(tmpbuf, "%s/%s", dirpath, FTPLISTCONF);
	if ((ptr = loadfile(tmpbuf)) == NULL) return;

	defuser = DEFAULT_LOGIN;
	defpswd = fromfield(0);

	for (n = 0; n < MAXCONFLINES && (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#' || *ptr == '.') continue;
		ptr2 = ptr;
		while ((u_char)*ptr > 0x20) ptr++;
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr; ptr++);
		if (*ptr && isdigit(*ptr)) {
			if (atoi(ptr) > privlevel) continue;
			while ((u_char)*ptr > 0x20) ptr++;
			if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr; ptr++);
		}
		if (!*ptr) continue;
		ftp[n].desc = ptr;
		if ((ptr = strchr(ptr2, '@')) != NULL) {
			*ptr++ = '\0';
			ftp[n].host = ptr;
		} else {
			ftp[n].host = ptr2;
			ptr2 = NULL;
		}
		ftp[n].user = defuser;
		ftp[n].pswd = defpswd;
		if (ptr2 != NULL) {
			ftp[n].user = ptr2;
			if ((ptr = strchr(ptr2, ':')) != NULL) {
				*ptr++ = '\0';
				ftp[n].pswd = ptr;
			}
		}
		ftp[n].port = 0;
		if ((ptr = strchr(ftp[n].host, ':')) != NULL) {
			*ptr++ = '\0';
			if (isdigitstr(ptr)) ftp[n].port = atoi(ptr);
			else {
				struct servent *sp;
				if ((sp = getservbyname(ptr, "tcp")) == 0) {
					badline(tmpbuf, ptr);
					continue;
				}
				ftp[n].port = ntohs(sp->s_port);
			}
		}
		if (!ftp[n].port) ftp[n].port = defport;
		if (isgrpright(GSID_FTP, ftp[n].host, GRF_S)) n++;
	}
	if (!n) {
		warning(MSG_NOFTPAVAILABLE, 1);
		return;
	}
	if (nchr) {
		rval = prompt_num(0, 0, n);
		if (!rval) return;
		if (open_ftpbyent(&ftp[rval-1])) return;
	}

list_again:
	lncnt = 1;
	for (i = 0; i < n; i++) {
		if (lncnt == 1) {
			lncnt += 2;
			memset(fidx, 0, sizeof(fidx));
			fidx[0] = -1;
			snprintf(tmpbuf, MAXSTRINGLENGTH-8, "%s %s",
				 sysmsg(MSG_FTPOPENHEADER), makeshowpath(dirpath));
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[H\033[J\033[1;37;44m %-72.72s %5d\033[K\033[40m\n\n", tmpbuf, n);
			else {
				clearline();
				putstr("\n %s (%4d)\n\n", tmpbuf, n);
			}
		}
		fidx[lncnt-1] = i+1;
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;37m%4d \033[33m%-24.24s  \033[0;36m%-.48s\n",
			       i+1, ftp[i].host, ftp[i].desc);
		else	putstr("%4d) %-24.24s %-.48s\n",
			       i+1, ftp[i].host, ftp[i].desc);
more_again:
		rval = morenum(MSG_SELECTFTPHOST, &lncnt, n);
		if (rval == -1) return;
		if (rval > 0) {
			if (rval > n) {
				rval = fidx[(rval-n-1)/2];
				if (rval < 0) return;
				if (!rval) {
					lncnt = scrlines;
					goto more_again;
				}
			}
			if (open_ftpbyent(&ftp[rval-1])) return;
			goto list_again;
		}
		if (rval == -2)	i -= (scrlines - 2) * 2;
		else if (rval == -3) i = n - (scrlines - 2) - 1;
		else if (rval == -4) i = -1;
		if (i < -1) i = -1;
	}
	lncnt = scrlines;
	goto more_again;
}

void
FTP_list()
{
	int i, startidx, lncnt, rval, fidx[MAXSCRLINES];
	char ch, *ptr;
	struct tm *tm;

	if (open_ftpdir() < 0) return;

	LOGIT(LOG_INFO, "FTP: list");

	if (nchr) {
		startidx = atoi(getstr(0, 1, ECHODISABLE));
		if (startidx >= ftpfiles) startidx = ftpfiles - scrlines/2;
		if (startidx < 1) startidx = 1;
	} else	startidx = 0;

list_again:
	lncnt = 1;
	for (i = 0; i < ftpfiles; i++) {
		if (lncnt == 1) {
			lncnt += 2;
			memset(fidx, 0, sizeof(fidx));
			fidx[0] = -1;
			snprintf(tmpbuf, MAXSTRINGLENGTH-8, "%s %s",
				 sysmsg(MSG_SELECTFILESHEAD), FTPpwd);
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[H\033[J\033[1;37;44m %-72.72s %5d\033[K\033[40m\n\n", tmpbuf, ftpfiles);
			else {
				clearline();
				putstr("\n %s (%4d)\n\n", tmpbuf, ftpfiles);
			}
		}
		if (i+1 < startidx) continue;

		fidx[lncnt-1] = i+1;
		switch (ftpfile[i].mode & S_IFMT) {
			case S_IFDIR:	ptr = "<DIR>"; break;
			case S_IFLNK:	ptr = "<LINK>"; break;
			case S_IFREG:	ptr = "<FILE>"; break;
			default:	ptr = "<UNKN>"; break;
		}
		if (ftpfile[i].mtime > lastcalltime) ch = '*';
		else ch = ' ';
		tm = localtime(&ftpfile[i].mtime);
		if (!tm->tm_hour && !tm->tm_min && !tm->tm_sec)
			strftime(tmpbuf, 13, "%b %d  %Y", tm);
		else	strftime(tmpbuf, 13, "%b %d %R", tm);
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;37m%4d \033[0;36m%-6.6s\033[32m%9d  %-12.12s\033[1;31m%c \033[33m%-.42s\n",
			       i+1, ptr, ftpfile[i].size, tmpbuf, ch, ftpfile[i].name);
		else	putstr("%4d) %-6.6s%9d  %-12.12s%c %-.42s\n",
			       i+1, ptr, ftpfile[i].size, tmpbuf, ch, ftpfile[i].name);
more_again:
		rval = morenum(MSG_SELECTFILE, &lncnt, ftpfiles);
		if (rval == -1) return;
		if (rval > 0) {
			if (rval > ftpfiles) {
				rval = fidx[(rval-ftpfiles-1)/2];
				if (rval < 0) return;
				if (!rval) {
					lncnt = scrlines;
					goto more_again;
				}
			}
			rcode = 250;
			switch (ftpfile[rval-1].mode & S_IFMT) {
				case S_IFDIR:
					startidx = !ftp_changedir(ftpfile[rval-1].name);
					break;
				case S_IFLNK:
					verbose = 0;
					startidx = !ftp_changedir(ftpfile[rval-1].name);
					verbose = 1;
					if (!startidx)
						startidx = ftp_get(ftpfile[rval-1].name);
					break;
				case S_IFREG:
					startidx = ftp_get(ftpfile[rval-1].name);
					break;
				default:
					warning(MSG_CANTOPENFILE, 1);
					rcode = 500;
					startidx = -1;
			}
			if (startidx > 0) return;
			startidx = 0;
			if (rcode != 250) anykey();
			goto list_again;
		}
		if (rval == -2)	i -= (scrlines - 2) * 2;
		else if (rval == -3) i = ftpfiles - (scrlines - 2) - 1;
		else if (rval == -4) i = -1;
		if (i < -1) i = -1;
	}
	lncnt = scrlines;
	goto more_again;
}

void
FTP_raw()
{
	if (open_ftpdir() < 0) return;
	LOGIT(LOG_INFO, "FTP: raw");
	display(1, ftpcachfile);
}

void
FTP_stat()
{
	if (FTPserver == NULL) {
		warning(MSG_OPENFTPFIRST, 1);
		return;
	}
	LOGIT(LOG_INFO, "FTP: stat");
	command("STAT");
}

void
FTP_pwd()
{
	if (FTPserver == NULL) {
		warning(MSG_OPENFTPFIRST, 1);
		return;
	}
	LOGIT(LOG_INFO, "FTP: pwd");
	if (command("PWD") == ERROR && rcode == 500) command("XPWD");
}

void
FTP_cdup()
{
	if (FTPserver == NULL) {
		warning(MSG_OPENFTPFIRST, 1);
		return;
	}
	LOGIT(LOG_INFO, "FTP: cdup");
	if (command("CDUP") == ERROR && rcode == 500) command("XCUP");
	if (rcode == 250) ftp_pwd("..");
}

void
FTP_cwd()
{
	char *ptr;

	if (FTPserver == NULL) {
		warning(MSG_OPENFTPFIRST, 1);
		return;
	}
	if ((ptr = prompt_str(MSG_CHANGEFTPCWD, NULL, 0)) == NULL) return;
	putchr('\n');
	LOGIT(LOG_INFO, "FTP: cwd \"%s\"", ptr);
	ftp_changedir(ptr);
}

void
FTP_get()
{
	if (open_ftpdir() < 0) return;
	ftp_get(NULL);
}

void
FTP_new()
{
	if (open_ftpdir() < 0) return;
	if (setnewfilestime() < 0) return;
	globalflags |= NEWFILES;
	ftp_get(NULL);
	globalflags &= ~NEWFILES;
}

/*
 * Lookup "host" and try to connect into "port".
 * Return hostname connected to or NULL.
 */
char *
ftp_connect(host, port)
	char *host;
	int port;
{
	struct hostent *hp = 0;
	int s, len;
	sigfunc oldintr;
	static char hostnamebuf[80];

	LOGIT(LOG_INFO, "FTP: trying %s", host);
	unlink(ftpcachfile);
	putchr('\n');
	ftp_disconnect();
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;36m");
	putstr("trying %s...\n", host);
	memset((char *)&hisctladdr, 0, sizeof (hisctladdr));
	hisctladdr.sin_addr.s_addr = inet_addr(host);
	if (hisctladdr.sin_addr.s_addr != -1) {
		hisctladdr.sin_family = AF_INET;
		strncpy(hostnamebuf, host, sizeof(hostnamebuf));
	} else {
		hp = gethostbyname(host);
		if (hp == NULL) {
			fprintf(stderr, "%s: Unknown host\n", host);
			rcode = -1;
			return NULL;
		}
		hisctladdr.sin_family = hp->h_addrtype;
		memmove((caddr_t)&hisctladdr.sin_addr,
				hp->h_addr_list[0], hp->h_length);
		strncpy(hostnamebuf, hp->h_name, sizeof(hostnamebuf));
	}
	s = socket(hisctladdr.sin_family, SOCK_STREAM, 0);
	if (s < 0) {
		perror("socket");
		rcode = -1;
		return NULL;
	}
	hisctladdr.sin_port = htons(port);
	oldintr = signal(SIGINT, SIG_IGN);
	backuptimer();
	while (connect(s, (struct sockaddr *)&hisctladdr, sizeof (hisctladdr)) < 0) {
		if (hp && hp->h_addr_list[1]) {
			int oerrno = errno;
			char *ia;

			ia = inet_ntoa(hisctladdr.sin_addr);
			errno = oerrno;
			putstr("connect to address %s: %s\n", ia, strerror(errno));
			hp->h_addr_list++;
			memmove((caddr_t)&hisctladdr.sin_addr,
					hp->h_addr_list[0], hp->h_length);
			putstr("trying %s...\n", inet_ntoa(hisctladdr.sin_addr));
			close(s);
			s = socket(hisctladdr.sin_family, SOCK_STREAM, 0);
			if (s < 0) {
				perror("socket");
				rcode = -1;
				restoretimer(1);
				(void)signal(SIGINT, oldintr);
				return NULL;
			}
			continue;
		}
		perror("connect");
		rcode = -1;
		restoretimer(1);
		(void)signal(SIGINT, oldintr);
		goto bad;
	}
	restoretimer(1);
	(void)signal(SIGINT, oldintr);

	len = sizeof (myctladdr);
	if (getsockname(s, (struct sockaddr *)&myctladdr, &len) < 0) {
		perror("getsockname");
		rcode = -1;
		goto bad;
	}
#ifdef IP_TOS
	len = IPTOS_LOWDELAY;
	if (setsockopt(s, IPPROTO_IP, IP_TOS, (char *)&len, sizeof(int)) < 0)
		LOGIT(LOG_ERR, "can't setsockopt IPTOS_LOWDELAY: %m");
#endif
	cin = fdopen(s, "r");
	cout = fdopen(s, "w");
	if (cin == NULL || cout == NULL) {
		perror("fdopen");
		if (cin) fclose(cin);
		if (cout) fclose(cout);
		rcode = -1;
		goto bad;
	}
	putstr("connected to %s\n", hostnamebuf);
	LOGIT(LOG_INFO, "FTP: connected to %s", hostnamebuf);
	if (getreply(0) > 2) { 	/* read startup message from server */
		if (cin) fclose(cin);
		if (cout) fclose(cout);
		rcode = -1;
		goto bad;
	}
#ifdef SO_OOBINLINE
	len = 1;
	if (setsockopt(s, SOL_SOCKET, SO_OOBINLINE, (char *)&len, sizeof(int)) < 0)
		LOGIT(LOG_ERR, "can't setsockopt SO_OOBINLINE: %m");
#endif
	sendport = -1;
	data = -1;
	abrtflag = 0;
	verbose = 1;
	return hostnamebuf;
bad:
	close(s);
	return NULL;
}

/*
 * Terminate ftp connection.
 */
ftp_disconnect()
{
	if (FTPserver != NULL) {
		putstr("closing connection with %s\n", FTPserver);
		(void) command("QUIT");
		if (cout) fclose(cout);
		cout = NULL;
		if (cin) fclose(cin);
		cin = NULL;
		data = -1;
		FTPserver = NULL;
		FTPpwd[0] = '\0';
		TransferFile[0] = '\0';
	}
}

/*
 * Login into previously connected host under "username" and "password".
 * Return -1 on error and 0 for success.
 */
int
ftp_login(user, pswd)
	char *user, *pswd;
{
	int n;
	char *ptr;

	clearline();
	putchr('\n');
	if (user == NULL) {
		if ((user = prompt_str(MSG_ENTERUSERNAME, DEFAULT_LOGIN, 25)) == NULL)
			return 0;
		clearline();
	}
	LOGIT(LOG_INFO, "FTP: user \"%s\"", user);
	n = command("USER %s", user);
	if (n == CONTINUE) {
		if (pswd == NULL) {
			if (!strcasecmp(user, DEFAULT_LOGIN))
				pswd = fromfield(0);
			if ((pswd = prompt_str(MSG_ENTERPASSWORD, pswd, 25)) == NULL)
				return 0;
			clearline();
		}
		LOGIT(LOG_DEBUG, "FTP: password \"%s\"", pswd);
		n = command("PASS %s", pswd);
	}
	if (n == CONTINUE) {
		putstr("\rAccount:");
		ptr = getstr(40, 0, ECHODISABLE);
		clearline();
		LOGIT(LOG_DEBUG, "FTP: account \"%s\"", ptr);
		n = command("ACCT %s", ptr);
	}
	if (n != COMPLETE) {
		if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;36m");
		putstr("login failed!\n");
		FTPpwd[0] = '\0';
		LOGIT(LOG_WARNING, "FTP: %s login failed", FTPserver);
		return -1;
	}
	command("TYPE I");
	LOGIT(LOG_NOTICE, "FTP: logged in %s", FTPserver);
	ptr = getrcent(FTP_RC, FTPserver);
	ftp_pwd("/");
	if (ptr != NULL && !strcmp(FTPpwd, "/") && strcmp(ptr, "/"))
		ftp_changedir(ptr);
	return 1;
}

/*
 * Get current working directory on remote host.
 * Return -1 on error.
 */
int
ftp_pwd(dir)
	char *dir;
{
	char *ptr;

	unlink(ftpcachfile);
	if (command("PWD") == ERROR && rcode == 500) command("XPWD");
	if (rcode == 257 && (ptr = strchr(reply_string, ' ')) != NULL) {
		*ptr++ = '\0';
		if (atoi(reply_string) != rcode) return -1; /* ??? */
		if ((dir = strchr(ptr, '"')) != NULL) ptr = &dir[1];
		if ((dir = strrchr(ptr, '"')) != NULL) *dir = '\0';
		else {	/* VMS, Novell or other shit */
			for (dir = ptr; (dir = strtok(dir, " \t")) != NULL; dir = NULL)
				if (strpbrk(dir, "/\\") != NULL) break;
			if (dir != NULL) ptr = dir;
			if ((dir = strpbrk(ptr, "\r\n")) != NULL) *dir = '\0';
		}
		strcpy(FTPpwd, ptr);
	} else {
		int dirlen = strlen(dir);
		if (dirlen > 1 && dir[dirlen-1] == '/') dir[--dirlen] = '\0';
		if (*dir == '/') strcpy(FTPpwd, dir);
		else if (!strcmp(dir, ".")) return 0;
		else if (!strcmp(dir, "..")) {
			if ((ptr = strrchr(FTPpwd, '/')) == NULL) return -1;
			if (ptr == FTPpwd) return 0;
			*ptr = '\0';
		} else {
			int pwdlen = strlen(FTPpwd);
			if (sizeof(FTPpwd) - pwdlen < dirlen) return -1;
			if (pwdlen > 1) strcat(FTPpwd, "/");
			strcat(FTPpwd, dir);
		}
	}
	putrcent(FTP_RC, FTPserver, FTPpwd);
	return 0;
}

/*
 * Change current working directory on remote host.
 * Return -1 on error.
 */
int
ftp_changedir(dir)
	char *dir;
{
	if (FTPserver == NULL) return -1;
	if (command("CWD %s", dir) == ERROR && rcode == 500)
		command("XCWD %s", dir);
	if (rcode != 250) return -1;
	return ftp_pwd(dir);
}

struct ftpget_ent {
	char *name;
	long size;
	time_t mtime;
};

int
ftp_get(name)
	register char *name;
{
	int i, nfiles, dlsize, dltime, lncnt, perm, tmpvar;
	char *ptr;
	char buf[512], temp[10];
	struct ftpget_ent file[MAXCONFLINES];

	if (FTPserver == NULL) {
		warning(MSG_OPENFTPFIRST, 1);
		return 0;
	}
	strcpy(tmpbuf, FTPserver);
	strcat(tmpbuf, FTPpwd);
	if (!isgrpright(GSID_FTP, tmpbuf, GRF_R)) {
		nopermission(GRF_R);
		return 0;
	}
	TransferFile[0] = '\0';
	if (!((termflags & RIPMOUSE) && name != NULL))
		if ((name = prompt_str(MSG_ENTERFILENAME, name, 0)) == NULL)
			return 0;

	LOGIT(LOG_INFO, "FTP: get \"%s\"", name);
	for (nfiles = lncnt = dlsize = dltime = 0, name = strcpy(buf, name);
	     (name = strtok(name, " \t")) != NULL && nfiles < MAXCONFLINES;
	     name = NULL) {
		for (i = 0; i < ftpfiles; i++) {
			if ((ftpfile[i].mode & S_IFMT) == S_IFDIR ||
			    ((globalflags & NEWFILES) && ftpfile[i].mtime < newfilestime) ||
			    fnmatch(name, ftpfile[i].name, FNM_NOESCAPE))
				continue;

			if ((ftpfile[i].mode & S_IFMT) == S_IFLNK) {
				verbose = 0;
				tmpvar = command("SIZE %s", ftpfile[i].name);
				verbose = 1;
				if (tmpvar != COMPLETE) continue;
				sscanf(reply_string, "%*s %d", &ftpfile[i].size);
			}
			ptr = fixfname(ftpfile[i].name, FALSE);
			if (strlen(ptr) < 2) continue;
			strcpy(tmpbuf, FTPserver);
			if (ftpfile[i].name[0] != '/') {
				strcat(tmpbuf, FTPpwd);
				if (strlen(FTPpwd) > 1) strcat(tmpbuf, "/");
			}
			strcat(tmpbuf, ftpfile[i].name);
			perm = isgrpright(GSID_FTP, tmpbuf, GRF_R);
			if (!lncnt) putchr('\n');
			tmpvar = ((int)ftpfile[i].size/128+1)*128*11/(process.baud+1);
			if (termflags & (ANSITERM | RIPTERM)) {
				sprintf(tmpbuf, "\033[1;33m%s \033[0;32m(%dkb, %s)",
					ptr, (int)ftpfile[i].size/1024, sec2str(tmpvar));
				strcpy(temp, "\033[1;31m");
			} else {
				sprintf(tmpbuf, "%s (%dkb, %s)",
					ptr, (int)ftpfile[i].size/1024, sec2str(tmpvar));
				temp[0] = '\0';
			}
			if (!perm) {
				LOGIT(LOG_INFO, "DL \"%s\" (%dkb, %s): permission denied",
				    ftpfile[i].name, (int)ftpfile[i].size, sec2str(tmpvar));
				putstr("%s %s- %s\n", tmpbuf, temp, sysmsg(MSG_NOPERMISSION));
			} else if ((dltime + tmpvar) > gettimer()) {
				LOGIT(LOG_INFO, "DL \"%s\" (%dkb, %s): exceed time limit",
				    ftpfile[i].name, (int)ftpfile[i].size, sec2str(tmpvar));
				putstr("%s %s- %s\n", tmpbuf, temp, sysmsg(MSG_EXCTIMELIMIT));
			} else if (dailydllimit > 0 &&
				   dlsize + dailydlsize + (int)ftpfile[i].size > dailydllimit) {
				LOGIT(LOG_INFO, "DL \"%s\" (%dkb, %s): exceed size limit",
				    ftpfile[i].name, (int)ftpfile[i].size, sec2str(tmpvar));
				putstr("%s %s- %s\n", tmpbuf, temp, sysmsg(MSG_EXCSIZELIMIT));
			} else if (dlulratio > 0 && userdlsize > dailydllimit &&
				   (dlsize + (int)ftpfile[i].size + userdlsize)/userulsize > dlulratio) {
				LOGIT(LOG_INFO, "DL \"%s\" (%dkb, %s): exceed DL/UL ratio",
				    ftpfile[i].name, (int)ftpfile[i].size, sec2str(tmpvar));
				putstr("%s %s- %s\n", tmpbuf, temp, sysmsg(MSG_EXCDLULRATIO));
			} else {
				dltime += tmpvar;
				dlsize += (int)ftpfile[i].size;
				file[nfiles].name = ftpfile[i].name;
				file[nfiles].size = ftpfile[i].size;
				file[nfiles].mtime = ftpfile[i].mtime;
				nfiles++;
				if (termflags & (ANSITERM | RIPTERM))
					putstr("\033[0;36m#%d  %s\033[0;37m\n", nfiles, tmpbuf);
				else	putstr("#%d  %s\n", nfiles, tmpbuf);
			}
			lncnt++;
		}
	}
	if (!nfiles) {
		clearline();
		if (!lncnt) warning(MSG_NOSUCHFILE, 1);
		return 0;
	}

	LOGIT(LOG_NOTICE, "%s started for Download", PROTO_NAME);
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;37m");
	sprintf(tmpbuf, "%d fls, %d kb, %s, %s at %d", nfiles, dlsize/1024,
		sec2str(dltime), PROTO_NAME, process.baud);
	putchr('\n');
	putstr(sysmsg(MSG_BEGINDOWNLOAD), tmpbuf);
	putchr('\n');
	setxferproto(PROTO_NAME);
	globalflags |= SENDINPROGRESS;
	process.flags |= INRECVF;
	saveinfo(&process);

	backuptimer();
	nfiles = ZM_sendbatch(file, nfiles);
	sigalarm();
	restoretimer(1);

	process.flags &= ~INRECVF;
	saveinfo(&process);

	setxferproto(NULL);
	if (!nfiles) {
		protobyebye();
		LOGIT(LOG_NOTICE, "%s failed", PROTO_NAME);
		putstr("\n\n%s\n", sysmsg(MSG_DLABORTED));
	} else	putstr("\n\n%s (%d files transfered)\n", sysmsg(MSG_DLCOMPLETE), nfiles);
	globalflags &= ~SENDINPROGRESS;
	return nfiles;
}

static void
onchld()
{
	if (pid) {
		if (wait(0) == -1) perror("wait");
		pid = 0;
	}
	(void) signal(SIGCHLD, SIG_IGN);
}

ZM_cleanup()
{
	if (pid) {
		kill(pid, SIGINT);
		onchld();
	}
	fd_exit();
	sigalarm();
	restoretimer(1);
	kill(0, SIGINT);
}

/*
 * Create temporary local file and receive remote file in background into it.
 * Return file stream to read from or NULL.
 */
FILE *
ftp_getbackground(remote, pos)
	char *remote;
	long pos;
{
	FILE *fp;
	char temp[1024];

	(void) sprintf(temp, "%s/.zmftp", homedir);
	(void) signal(SIGCHLD, onchld);

	if ((pid = fork()) == 0) {
		(void) signal(SIGQUIT, SIG_IGN);
		(void) signal(SIGHUP, SIG_IGN);
		if ((fp = fopen(temp, "w")) != NULL) {
			verbose = 0;
			if (pos) lseek(fileno(fp), (off_t)pos, SEEK_SET);
			recvrequest(fp, remote, pos);
			fclose(fp);
		}
		exit(0);
	}
	if (pid < 0) {
		pid = 0;
		(void) signal(SIGCHLD, SIG_IGN);
		LOGIT(LOG_ERR, "can't fork() ftp: %m");
		return NULL;
	}
	sleep(1);
	fp = fopen(temp, "r");
	unlink(temp);
	return fp;
}

int
getreply(expecteof)
	int expecteof;
{
	int c, n, n1;
	int dig;
	int originalcode = 0, continuation = 0;
	sigfunc oldintr;
	char *cp;

	if (verbose) {
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;36m\rwait...\r");
		else	putstr("\rwait...\r");
	}
	oldintr = signal(SIGINT, cmdabort);
	n = n1 = 0;
	for (;;) {
		dig = rcode = 0;
		cp = reply_string;
		while ((c = getc(cin)) != '\n') {
			if (c == IAC) {     /* handle telnet commands */
				switch (c = getc(cin)) {
				case WILL:
				case WONT:
					c = getc(cin);
					fprintf(cout, "%c%c%c", IAC, DONT, c);
					(void) fflush(cout);
					break;
				case DO:
				case DONT:
					c = getc(cin);
					fprintf(cout, "%c%c%c", IAC, WONT, c);
					(void) fflush(cout);
					break;
				default:
					break;
				}
				continue;
			}
			dig++;
			if (c == EOF) {
				if (expecteof) {
					(void) signal(SIGINT,oldintr);
					rcode = 221;
					return (0);
				}
				lostpeer();
				if (verbose)
					putstr("Service not available, remote server has closed connection\n");
				FTPserver = NULL;
				FTPpwd[0] = '\0';
				rcode = 421;
				return (4);
			}
			if (verbose && c != '\r' && dig > 4 &&
			    (n == '5' || (n == '2' && (n1 == '1' || n1 == '3' || n1 == '5'))))
				putchr(c);
			if (dig < 4 && isdigit(c))
				rcode = rcode * 10 + (c - '0');
			if (dig == 4 && c == '-') {
				if (continuation) rcode = 0;
				continuation++;
			}
			if (n == 0) {
				n = c;
				if (verbose && (termflags & (ANSITERM | RIPTERM))) {
					if (n == '5') putstr("\033[31m");
					else putstr("\033[37m");
				}
			} else	if (n1 == 0) n1 = c;
			if (cp < &reply_string[sizeof(reply_string) - 1])
				*cp++ = c;
		}
		if (verbose && (n == '5' || (n == '2' && (n1 == '1' || n1 == '3' || n1 == '5'))))
			putchr(c);
		if (continuation && rcode != originalcode) {
			if (originalcode == 0)
				originalcode = rcode;
			continue;
		}
		*cp = '\0';
		if (n != '1') cpend = 0;
		(void) signal(SIGINT,oldintr);
		if (rcode == 421 || originalcode == 421) lostpeer();
		if (abrtflag && oldintr != cmdabort && oldintr != SIG_IGN)
			(*oldintr)(SIGINT);
		return (n - '0');
	}
}

int
empty(mask, sec)
	fd_set *mask;
	int sec;
{
	struct timeval t;

	t.tv_sec = (long) sec;
	t.tv_usec = 0;
	return (select(32, mask, NULL, NULL, &t));
}

void
abortrecv()
{

	abrtflag = 0;
	longjmp(recvabort, 1);
}

void
recvrequest(fp, remote, restart)
	FILE *fp;
	char *remote;
	long restart;
{
	FILE *din = 0;
	sigfunc oldintr;
	int c, d;
	char buf[8192];
	long bytes = 0;
	struct timeval start, stop;

	oldintr = NULL;
	if (setjmp(recvabort)) {
		while (cpend) (void) getreply(0);
		if (data >= 0) {
			(void) close(data);
			data = -1;
		}
		if (oldintr) (void) signal(SIGINT, oldintr);
		rcode = -1;
		return;
	}
	oldintr = signal(SIGINT, abortrecv);
	if (initconn()) {
		(void) signal(SIGINT, oldintr);
		rcode = -1;
		return;
	}
	if (setjmp(recvabort)) goto abort;
	if (remote != NULL) {
		LOGIT(LOG_NOTICE, "FTP: receiving \"%s\" at %ld", remote, restart);
		if (restart && command("REST %ld", restart) != CONTINUE)
			return;
		if (command("RETR %s", remote) != PRELIM) {
			(void) signal(SIGINT, oldintr);
			return;
		}
	} else if (command("LIST") != PRELIM) {
		(void) signal(SIGINT, oldintr);
		return;		
	}
	if ((din = dataconn()) == NULL) goto abort;
	(void) gettimeofday(&start, (struct timezone *)0);

	errno = d = 0;
	while ((c = read(fileno(din), buf, sizeof(buf))) > 0) {
		if ((d = write(fileno(fp), buf, c)) != c)
			break;
		bytes += c;
		if (verbose)
			putstr("\r%ld bytes received", bytes);
		else	LOGIT(LOG_DEBUG, "FTP: %ld bytes received", bytes);
	}
	if (c < 0) {
		if (errno != EPIPE) perror("netin");
		bytes = -1;
	}
	(void) signal(SIGINT, oldintr);
	(void) gettimeofday(&stop, (struct timezone *)0);
	(void) fclose(din);
	(void) getreply(0);
	if (bytes > 0) ptransfer(remote, bytes, &start, &stop);
	return;
abort:

/* abort using RFC959 recommended IP,SYNC sequence  */

	(void) gettimeofday(&stop, (struct timezone *)0);
	(void) signal(SIGINT, SIG_IGN);
	if (!cpend) {
		rcode = -1;
		(void) signal(SIGINT, oldintr);
		return;
	}
	abort_remote(din);
	rcode = -1;
	if (data >= 0) {
		(void) close(data);
		data = -1;
	}
	if (din) (void) fclose(din);
	if (bytes > 0) ptransfer(remote, bytes, &start, &stop);
	(void) signal(SIGINT, oldintr);
}

/*
 * Need to start a listen on the data channel before we send the command,
 * otherwise the server's connect may fail.
 */
int
initconn()
{
	char *p, *a;
	int result, len, tmpno = 0;
	int on = 1;

noport:
	data_addr = myctladdr;
	if (sendport) data_addr.sin_port = 0;	/* let system pick one */ 
	if (data != -1) (void) close(data);
	data = socket(AF_INET, SOCK_STREAM, 0);
	if (data < 0) {
		perror("socket");
		if (tmpno) sendport = 1;
		return (1);
	}
	if (!sendport)
		if (setsockopt(data, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof (on)) < 0) {
			LOGIT(LOG_ERR, "can't setsockopt SO_REUSEADDR: %m");
			goto bad;
		}
	if (bind(data, (struct sockaddr *)&data_addr, sizeof (data_addr)) < 0) {
		perror("bind");
		goto bad;
	}
	len = sizeof (data_addr);
	if (getsockname(data, (struct sockaddr *)&data_addr, &len) < 0) {
		perror("getsockname");
		goto bad;
	}
	if (listen(data, 1) < 0) perror("listen");
	if (sendport) {
		a = (char *)&data_addr.sin_addr;
		p = (char *)&data_addr.sin_port;
#define	UC(b)	(((int)b)&0xff)
		result = command("PORT %d,%d,%d,%d,%d,%d",
		      UC(a[0]), UC(a[1]), UC(a[2]), UC(a[3]),
		      UC(p[0]), UC(p[1]));
		if (result == ERROR && sendport == -1) {
			sendport = 0;
			tmpno = 1;
			goto noport;
		}
		return (result != COMPLETE);
	}
	if (tmpno) sendport = 1;
#ifdef IP_TOS
	on = IPTOS_THROUGHPUT;
	if (setsockopt(data, IPPROTO_IP, IP_TOS, (char *)&on, sizeof(int)) < 0)
		LOGIT(LOG_ERR, "can't setsockopt IPTOS_THROUGHPUT: %m");
#endif
	return 0;
bad:
	(void) close(data);
	data = -1;
	if (tmpno) sendport = 1;
	return 1;
}

FILE *
dataconn()
{
	struct sockaddr_in from;
	int s, fromlen = sizeof (from), tos;

	s = accept(data, (struct sockaddr *) &from, &fromlen);
	if (s < 0) {
		perror("accept");
		(void) close(data);
		data = -1;
		return (NULL);
	}
	(void) close(data);
	data = s;
#ifdef IP_TOS
	tos = IPTOS_THROUGHPUT;
	if (setsockopt(s, IPPROTO_IP, IP_TOS, (char *)&tos, sizeof(int)) < 0)
		LOGIT(LOG_ERR, "can't setsockopt IPTOS_THROUGHPUT: %m");
#endif
	return (fdopen(data, "r"));
}

void
ptransfer(file, bytes, t0, t1)
	char *file;
	long bytes;
	struct timeval *t0, *t1;
{
	struct timeval td;
	float s, bs;

	tvsub(&td, t1, t0);
	s = td.tv_sec + (td.tv_usec / 1000000.);
#define	nz(x)	((x) == 0 ? 1 : (x))
	bs = bytes / nz(s);
	if (file != NULL) {
		LOGIT(LOG_NOTICE, "FTP: received \"%s\" in %ld bytes (%.2gKb/s)",
		    file, bytes, bs / 1024.);
	}
	if (verbose) {
		putstr("\r%ld bytes received in %.2g seconds (%.2gKb/s)\n",
		       bytes, s, bs / 1024.);
	}
}

void
tvsub(tdiff, t1, t0)
	struct timeval *tdiff, *t1, *t0;
{

	tdiff->tv_sec = t1->tv_sec - t0->tv_sec;
	tdiff->tv_usec = t1->tv_usec - t0->tv_usec;
	if (tdiff->tv_usec < 0)
		tdiff->tv_sec--, tdiff->tv_usec += 1000000;
}


	void abort_remote(FILE *din)
{
	char buf[BUFSIZ];
	int nfnd;
	fd_set msk;

	/*
	 * send IAC in urgent mode instead of DM because 4.3BSD places oob mark
	 * after urgent byte rather than before as is protocol now
	 */
	sprintf(buf, "%c%c%c", IAC, IP, IAC);
	if (send(fileno(cout), buf, 3, MSG_OOB) != 3) perror("abort");
	fprintf(cout,"%cABOR\r\n", DM);
	(void) fflush(cout);
	FD_ZERO(&msk);
	FD_SET(fileno(cin), &msk);
	if (din) { 
		FD_SET(fileno(din), &msk);
	}
	if ((nfnd = empty(&msk, 10)) <= 0) {
		if (nfnd < 0) perror("abort");
		lostpeer();
	}
	if (din && FD_ISSET(fileno(din), &msk)) {
		while (read(fileno(din), buf, BUFSIZ) > 0)
			/* LOOP */;
	}
	if (getreply(0) == ERROR && rcode == 552) {
		/* 552 needed for nic style abort */
		(void) getreply(0);
	}
	(void) getreply(0);
}



	void lostpeer(void)
{
	if (cout != NULL) {
		(void) shutdown(fileno(cout), 1+1);
		(void) fclose(cout);
		cout = NULL;
	}
	if (data >= 0) {
		(void) shutdown(data, 1+1);
		(void) close(data);
		data = -1;
	}
}


/*** ZMODEM ***/

/* data subpacket size; maybe lowered during a session */
#define	MAX_SUBPACKETSIZE		1024	/* MANDATORY: 1024 bytes */
#define	MIN_SUBPACKETSIZE		64

char tx_data_subpacket[MAX_SUBPACKETSIZE];	/* transmit data buffer */
int subpacket_size;				/* current subpacket size */

/*
 * send from the current position in the file
 * all the way to end of file or until something goes wrong.
 * (ZNAK or ZRPOS received)
 */
int
ZM_sendfrom(fp, eof)
	FILE *fp;
	long eof;
{
	int pkts, n;
	int type = ZCRCW;	/* was ZCRCG */
	char zdata_frame[] = { ZDATA, 0, 0, 0, 0 };

	pkts = n = 0;
	while (ftell(fp) < eof) {
		if (!n) {
			/*
			 * put the file position in the ZDATA frame
			 */
			zdata_frame[ZP0] = ftell(fp) & 0xff;
			zdata_frame[ZP1] = (ftell(fp) >> 8) & 0xff;
			zdata_frame[ZP2] = (ftell(fp) >> 16) & 0xff;
			zdata_frame[ZP3] = (ftell(fp) >> 24) & 0xff;
			tx_header(zdata_frame);
		}

		/*
		 * adopt new subpacket size for good line quality
		 */
		if (++pkts > 2 && subpacket_size < MAX_SUBPACKETSIZE)
			subpacket_size *= 2;

		/*
		 * read a block from the file
		 */
		clearerr(fp);
		if ((n = fread(tx_data_subpacket, 1, subpacket_size, fp)) < 1) {
			if (ferror(fp)) {
				LOGIT(LOG_ERR, "%s: fread: %m", PROTO_NAME);
				return ZFERR;
			}
			if (feof(fp)) {
				if (ftell(fp) == eof) return ZACK; /* EOF */
				if (!pid) {
					LOGIT(LOG_WARNING, "%s ftp-data died?!", PROTO_NAME);
					return ZABORT;
				}
			}
			n = 0;
		}

		/*
		 * at end of file wait for an ACK
		 */
		if (ftell(fp) == eof) type = ZCRCW;

		tx_data(type, tx_data_subpacket, n);
		if (n)	LOGIT(LOG_DEBUG, "%s: %ld bytes transfered", PROTO_NAME, ftell(fp));
		else	LOGIT(LOG_DEBUG, "%s: sent idle frame", PROTO_NAME);

		if (type == ZCRCW) {
			int type;
			do {
				type = rx_header(10000);
				switch (type) {
					case ZRINIT:
						type = ZSKIP;
					case ZSKIP:
					case ZNAK:
					case ZRPOS:
					case ZABORT:
					case ZFERR:
						return type;
				}
			} while (type != ZACK);

			/*
			 * ZCRCW: need to transfer data header every time
			 */
			n = 0;
		}

		/* 
		 * characters from the other side
		 * check out that header
		 */
		while (rx_poll()) {
			if (rx_raw(0) == ZPAD) {
				int type;
				type = rx_header(1000);
				if (type != TIMEOUT && type != ACK)
					return type;
			}
		}
	}

	/*
	 * end of file reached.
	 * should receive something... so fake ZACK
	 */
	return ZACK;
}

/*
 * send a file; returns -1 when session is aborted or bytes transfered.
 * (using ZABORT frame)
 */
long
ZM_sendfile(name, size, mtime, rest)
	char *name;
	long size;
	time_t mtime;
	int rest;
{
	char *ptr;
	long pos;
	FILE *fp = NULL;
	char zfile_frame[] = { ZFILE, 0, 0, 0, 0 };
	char zeof_frame[] = { ZEOF, 0, 0, 0, 0 };
	int type, retries;
	
	zfile_frame[ZF0] = ZF0_ZCBIN;	/* conversion option */
	zfile_frame[ZF1] = ZF1_ZMNEWL;	/* management option */
	zfile_frame[ZF2] = ZF2_ZTNOR;	/* transport options */
	zfile_frame[ZF3] = 0;		/* extended option */

	/* enter the name and a '\0' */
	ptr = strcpy(tx_data_subpacket, fixfname(name, FALSE));
	ptr += strlen(ptr) + 1;

	/* file size, date, mode, etc */
	sprintf(ptr, "%ld %lo 0 0 %d 0", size, mtime, rest);
	ptr += strlen(ptr) + 1;

	do {
		/*
	 	 * send the header and the data
	 	 */
		tx_header(zfile_frame);
		tx_data(ZCRCW, tx_data_subpacket, ptr - tx_data_subpacket);
	
		/*
		 * wait for anything but an ZACK packet
		 */
		while ((type = rx_header(10000)) == ZACK);

		if (type == ZSKIP) return 0;	/* skipped file */

	} while (type != ZRPOS);

	subpacket_size = 0;	/* 0 -- start transfer sign */
	retries = 0;
	do {
		/*
		 * fetch pos from the ZRPOS header
		 */
		if (type == ZRPOS)
			pos = rxd_header[ZP0] | (rxd_header[ZP1] << 8) | (rxd_header[ZP2] << 16) | (rxd_header[ZP3] << 24);

		if (!subpacket_size) {
			/*
			 * begin data transfer
			 */
			if (pos >= size) break;	/* skipped file??? */

			/*
			 * set initial value for subpacket size
			 */
			subpacket_size = MAX_SUBPACKETSIZE / 2;

			/*
			 * start ftp-data receiving in background
			 */
			if ((fp = ftp_getbackground(name, pos)) == NULL)
				/* can't open temp file */
				return -1;
		}

		/*
		 * adopt new subpacket size for bad line quality
		 */
		if (pos < ftell(fp)) {
			if (++retries > 12) {
				/* too many retries; abort bad transfer */
				fclose(fp);
				if (pid) kill(pid, SIGINT);
				return -1;
			}
			if (retries > 4 && subpacket_size > MIN_SUBPACKETSIZE)
				subpacket_size /= 2;
		} else retries = 0;

		/*
 		 * seek to the right place in the file
		 */
		fseek(fp, pos, SEEK_SET);

		/*
		 * start sending from expected position
		 */
		type = ZM_sendfrom(fp, size);

		if (type == ZFERR || type == ZABORT) {
			/* abort transfer */
 			fclose(fp);
			if (pid) kill(pid, SIGINT);
			return -1;
		}

	} while (type == ZRPOS || type == ZNAK);

	/*
	 * file sent. send end of file frame
	 * and wait for zrinit. if it doesnt come then try again
	 */
	if (fp != NULL) fclose(fp);

	zeof_frame[ZP0] =  size        & 0xff;
	zeof_frame[ZP1] = (size >> 8)  & 0xff;
	zeof_frame[ZP2] = (size >> 16) & 0xff;
	zeof_frame[ZP3] = (size >> 24) & 0xff;

	do {
		tx_hex_header(zeof_frame);
		type = rx_header(10000);
	} while (type != ZRINIT);

	/* return transfered size */
	return (subpacket_size ? size : 0);
}

int
ZM_sendbatch(file, nfiles)
	struct ftpget_ent *file;
	int nfiles;
{
	int i, bytes, cps;
	time_t xftime;

	/*
	 * set the io device to transparent
	 */
	fd_init();	

	/*
	 * clear the input queue from any possible garbage
	 * this also clears a possible ZRINIT from an already started
	 * zmodem receiver. this doesn't harm because we reinvite to
	 * receive again below and it may be that the receiver whose
	 * ZRINIT we are about to wipe has already died.
	 */
	rx_purge();

	/*
	 * establish contact with the receiver
	 */
	i = 0;
	do {
		unsigned char zrqinit_header[] = { ZRQINIT, 0, 0, 0, 0 };
		i++;
		if (i > 5) {
			/* can't establish contact with receiver */
			fd_exit();
			return 0;
		}
		tx_raw('z');
		tx_raw('m');
		tx_raw(13);
		tx_hex_header(zrqinit_header);
	} while (rx_header(7000) != ZRINIT);

	/*
	 * decode receiver capability flags
	 * forget about encryption and compression.
	 */
	can_fcs_32			= (rxd_header[ZF0] & ZF0_CANFC32) != 0;
	escape_all_control_characters	= (rxd_header[ZF0] & ZF0_ESCCTL)  != 0;
	use_variable_headers		= (rxd_header[ZF1] & ZF1_CANVHDR) != 0;

	/* 
	 * and send each file in turn
	 */
	for (i = 0; i < nfiles; i++) {
		strcpy(TransferFile, FTPserver);
		if (file[i].name[0] != '/') {
			strcat(TransferFile, FTPpwd);
			if (strlen(FTPpwd) > 1) strcat(TransferFile, "/");
		}
		strcat(TransferFile, file[i].name);
		LOGIT(LOG_NOTICE, "DL \"%s\"", TransferFile);
		xftime = time(NULL);
		bytes = ZM_sendfile(file[i].name, file[i].size, file[i].mtime, nfiles - i);
		if (bytes < 0) break;
		if (bytes) {
			xftime = time(NULL) - xftime;
			cps = bytes/(xftime+1);
			LOGIT(LOG_NOTICE, "%s complete. %d kb, %d cps", PROTO_NAME, bytes/1024, cps);
			add_dl_stat(bytes);
			xferlog('D', bytes, cps, TransferFile);
		} else	LOGIT(LOG_NOTICE, "%s skiped file", PROTO_NAME);
		TransferFile[0] = '\0';
	}

	/*
	 * close the session
	 */
	{
		int type;
		unsigned char zfin_header[] = { ZFIN, 0, 0, 0, 0 };

		do {
			tx_hex_header(zfin_header);
			type = rx_header(10000);
		} while (type != ZFIN && type != ZCAN && type != TIMEOUT);
		
		/*
		 * these Os are formally required; but they don't do a thing
		 * unfortunately many programs require them to exit 
		 * (both programs already sent a ZFIN so why bother ?)
		 *
		 * if (type == ZFIN) {
		 *	tx_raw('O');
		 *	tx_raw('O');
		 * }
		 */
	}
	fd_exit();

	/* files transfered */
	return i;
}
