/*
 *	Copyright (c) 1997 Rinet Corp., Novosibirsk, Russia
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

/*
 *	Copyright (c) 1994,1995 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <pwd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef	__STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <signal.h>
#include <setjmp.h>
#include <syslog.h>

#include "compat.h"
#include "drive.h"
#include "sysmsg.h"
#include "group.h"
#include "rematch.h"
#include "scandir.h"
#include "loginfo.h"
#include "utmpent.h"
#include "variables.h"
#include "version.h"

/* system variables */
char *systemid;			/* package, version, target	*/
char *systemname;		/* name of this system		*/
char *systemhost;		/* hostname of this system	*/
char *sysmailto;		/* e-mail of system admin	*/
char *sysadmname;		/* name of system admin		*/
struct loginfo process;		/* user/system process info	*/

/* user variables */
char callsign[LOGINFO_NAMELEN];	/* user call signature			*/
char homedir[1024];		/* user home directory			*/
char dirpath[1024];		/* current menu directory path		*/
char term[LOGINFO_TTYLEN];	/* term type, ex. VT100, ANSI, RIP, etc	*/
char fntty[LOGINFO_TTYLEN];	/* tty name as file name (with no '/'s)	*/
int privlevel;			/* user access level (0-min, 99-max)	*/
time_t lastcalltime;		/* user last logged time		*/
int globalflags = IMMQUIT;	/* set of global flags			*/
int ncalls;			/* number of user calls			*/
char *usergroup = NULL;		/* group name of current user		*/
char userconf[40];		/* it config file name			*/

char *loginline;		/* login line description if any	*/
char *remote_addr;		/* remote IP address (if any)		*/
char *remote_host;		/* remote domain name (if any)		*/

/* internal variables */
char *editorworkfile;		/* editor work file		*/
char *talkaddrfile = NULL;	/* HOME/.talk.ttyXX addr file	*/
char *ftpcachfile = NULL;	/* ftp cach file		*/

char *shellprog;		/* Path to prefered system shell*/
char *editprog;			/* Path to prefered system editor*/
char *talkprog;			/* Path to external talk program*/
char *fingerprog;		/* Path to the finger program	*/
char *logoutprog = NULL;	/* Path to logout procedure	*/
char *mailerprog;		/* Path to ext. mailer program	*/
char *inewsprog;		/* Path to ext. inews program	*/
char *progloader;		/* Path to loader of user progs */

/* currently scaned menu directory */
struct dir_ent *menu_item = NULL;
int menu_num;

char tmpbuf[2048], inrunning[512];
char *stdfile = NULL;
jmp_buf	jmpint;
pid_t pid = 0;
int loggedin = FALSE;
int maxsessiontime = 0, restoretime = 0;
char **Argv;	/* pointer to argument vector */
char *LastArgv;	/* end of argv */

static FILE *fdlog;	/* logfile handler */
static int loglevel, sysloglevel;	/* logging levels */
static char *pidfile = NULL;	/* myself pid file */
static char *path_altuserconf = NULL;	/* for backup alternate user.conf */
static char *body_altuserconf = NULL;

extern int errno;
extern int usenetinuse;
extern int UsenetClose();
extern int menu_index;
extern char TransferFile[];

/* forward function declaration */
int isaccess(char *dir);
static void set_remote(void);
static void killpid(char *func, pid_t pid);


void
#ifdef	__STDC__
logit(int level, const char *fmt, ...)
#else
logit(level, fmt, va_alist)
	int level;
	const char *fmt;
	va_dcl
#endif
{
	va_list ap;
	char *p, buf[200];

#ifdef	__STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	if (level <= sysloglevel) vsyslog(level, fmt, ap);
	(void)strncpy(buf, fmt, sizeof(buf));
	buf[sizeof(buf)-1] = '\0';
	if ((p = strstr(buf, "%m")) != NULL) *p = '\0';
	vproctitle(buf, ap);
	if (level <= loglevel) {
		time_t now = time(NULL);
		(void)fprintf(fdlog, "%.15s drive[%s]: ", ctime(&now)+4,
			      process.tty);
		if (level < LOG_ERR)
			(void)fputs("PANIC: ", fdlog);
		else if (level < LOG_WARNING)
			(void)fputs("ERROR: ", fdlog);
		(void)vfprintf(fdlog, buf, ap);
		if (p != NULL)
			(void)fputs(strerror(errno), fdlog);
		(void)fputc('\n', fdlog);
		(void)fflush(fdlog);
	}
	va_end(ap);
}

int
main(argc, argv, envp)
	int argc;
	char **argv, **envp;
{
	int i, j;
	struct passwd *pwd;
	FILE *fp;
	char *ptr, buf[1024];

	void sigcatch(), review_priv();
	static int getconnectfile(), acceptfile();
	int (*menu_func)();
	extern int (*find_term())();
	extern int talkportid;
	extern int optind, opterr;
	extern char *optarg;

	/*
	 * Open an default system logging (via syslog).
	 * Log just system errors, not user activity.
	 */
	error = syslog;
	openlog("drive", LOG_CONS|LOG_PID, LOG_USER);
	sysloglevel = LOG_ERR;
	setlogmask(LOG_UPTO(sysloglevel));

	/*
	 * Determine myself.
	 */
	if ((pwd = getpwuid(getuid())) == NULL)	{ /* nonsens */
		LOGIT(LOG_CRIT, "getpwuid: %m");
		exit(1);
	}

	/*
	 * Make known settings of current process.
	 */
	memset(&process, 0, sizeof(struct loginfo));
	ptr = ttyname(0) + 5;	/* skip "/dev/" */
	(void)strncpy(process.tty, ptr, sizeof(process.tty));
	process.tty[sizeof(process.tty)-1] = '\0';
	for (i = 0; *ptr; ptr++) if (*ptr != '/') fntty[i++] = *ptr;
	fntty[i] = '\0';

	(void)strncpy(process.name, pwd->pw_name, sizeof(process.name));
	process.name[sizeof(process.name)-1] = '\0';
	process.ltime = time(NULL);
	process.pid = getpid();
	process.flags = HIDDENUSER;	/* trying to login */

	opterr = 0;
	while ((i = getopt(argc, argv, "s:")) != EOF) {
		switch (i) {
		case 's': /* imitate user connected speed (for debugging) */
			process.baud = atoi(optarg);
			(void)sprintf(tmpbuf, "CONNECT %d (argv)", process.baud);
			break;
		/* any more switches? nope yet */
		}
	}

	/* privlevel and menupath canbe passed to cmdline (for debugging) */
	if (argv[optind] != NULL)
		(void)strcpy(dirpath, argv[optind++]);
	else	(void)strcpy(dirpath, ".");

	if (argv[optind] != NULL)
		privlevel = atoi(argv[optind]);
	else	privlevel = 0;

	(void)strcpy(homedir, pwd->pw_dir);
	(void)strcpy(userconf, USERCONF);
	(void)strncpy(callsign, pwd->pw_name, sizeof(callsign));
	callsign[sizeof(callsign)-1] = '\0';

	umask(022);

	/*
	 * Configure main system parameters.
	 */
	initgetconf(pwd->pw_dir);
	systemid = strdupf("%s %s [%s]", package, version, target);
	systemname = strdup(getsysconf(SYSTEMNAME, "Unknown"));
	if (gethostname(buf, sizeof(buf)) < 0) (void)strcpy(buf, "Unknown");
	systemhost = strdup(getsysconf(MYHOSTNAME, buf));
	sysmailto = strdup(getsysconf(SYSMAILTO, "root"));
	sysadmname = strdup(getsysconf(SYSADMNAME, "System Administrator"));
	pidfile = strdupf(MYPIDFILE, strippath(argv[0]), fntty);
	if ((ptr = getsysconf(LOGFILE, NULL)) != NULL) {
		if (strstr(ptr, "%s") != NULL) {
			sprintf(buf, ptr, fntty);
			ptr = buf;
		}
		if ((fdlog = fopen(ptr, "a")) == NULL) {
			LOGIT(LOG_CRIT, "fopen: %s: %m", ptr);
			exit(1);
		}
		loglevel = LOG_ERR + atoi(getsysconf(LOGLEVEL, "0"));
		error = logit;
	}

	/*
	 * Make sure we have permission to home and menu (current) dir.
	 */
	if (!isaccess(dirpath)) {
		LOGIT(LOG_CRIT, "can't access \"%s\": %m", dirpath);
		exit(1);
	}
	if (!isaccess(homedir)) {
		LOGIT(LOG_CRIT, "can't access \"%s\": %m", homedir);
		exit(1);
	}

	/*
	 * Magic cookie with current process environment.
	 */
	ptr = getenv(TERM);	/* default terminal type */
	(void)strcpy(term, strlwr(ptr != NULL ? ptr : VT52));

	/* find login line description (name, phone num, location, etc) */
	if (getenv(LINE) != NULL) loginline = strdup(getenv(LINE));
	else if (ptr != NULL) loginline = strdup(ptr);
	else loginline = NULL;

	if (process.baud == 0) {
		/* I hope it found an TRUE connect speed returned by NAS */
		if (loginline != NULL && (ptr = strchr(loginline, ':')) != NULL) {
			*ptr++ = '\0';
			process.baud = atoi(ptr);
		} else if ((ptr = getenv("PRINTER")) != NULL) /* why not? */
			process.baud = atoi(ptr);

		if (process.baud == 0)
			/* tmpbuf will contain original modem string */
			process.baud = getconnectfile();
		else	sprintf(tmpbuf, "CONNECT %d (env)", process.baud);
	}

	/* get out all other parameters from environment at this point */
	editprog = strdup(get_env(EDITOR, getsysconf(EDITOR, DEFEDITOR)));
	talkprog = strdup(get_env(TALK, getsysconf(TALK, PROG_TALK)));
	fingerprog = strdup(get_env(FINGER, getsysconf(FINGER, PROG_FINGER)));

	/* save start and extend of argv for proctitle */
	Argv = argv;
	while (*envp) envp++;
	LastArgv = envp[-1] + strlen(envp[-1]);

	/*
	 * Init state.
	 */
	proctitle("INIT");

	/* Prepare user program loader */
	if ((progloader = getsysconf("LOADER", NULL)) != NULL)
		progloader = strdup(progloader);

	/* Prepare shell program */
	shellprog = strdup(getsysconf(SHELL, DEFSHELL));

	/* Prepare logout procedure */
	logoutprog = strdup(getsysconf(LOGOUT, DEFLOGOUT));

	/* Prepare mailer program */
	mailerprog = strdup(getsysconf(MAILER, DEFMAILER));

	/* Prepare inews program */
	inewsprog = strdup(getsysconf(INEWS, DEFINEWS));

	set_remote();	/* find address of telneted user */

	/* init languages data base */
	if (langinit(getsysconf(LANGUAGE, DEFLANGUAGE)) < 1) {
		LOGIT(LOG_CRIT, "can't initialize language");
		exit(1);
	}
	/* init user groups data base */
	if (initgrpbase(NULL) < 0) {
		LOGIT(LOG_CRIT, "can't initialize groups");
		exit(1);
	}
	/* open an monitoring file if configured (for debugging) */
	if (openmonitor() < 0) {
		LOGIT(LOG_CRIT, "can't initialize monitor");
		exit(1);
	}
	/* make process ID file */
	if ((fp = fopen(pidfile, "w")) == NULL) {
		LOGIT(LOG_CRIT, "can't write \"%s\": %m", pidfile);
		exit(1);
	}
	fprintf(fp, "%d\n", (int)process.pid);
	fprintf(fp, "%s\n", systemid);
	fclose(fp);

	/*
	 * Save current termio and setup terminal to raw mode.
	 */
	i = ttysave();	/* return tty speed */
	ttysetraw();

	/* need to know connect speed anyway */
	if (process.baud == 0) {
		process.baud = i;
		(void)sprintf(tmpbuf, "CONNECT %d (%s)", i,
			      remote_host ? remote_host : "tty");
	}

	/*
	 * First time initialization is done; I'm attached to remote.
	 */
	LOGIT(LOG_NOTICE, tmpbuf);
	LOGIT(LOG_INFO, systemid);

	sigcatch();	/* catch all required signals */
	settimer(LOGINTIMELEFT); /* limit user for login time */

	if (setjmp(jmpint)) {
		/*
		 * Restore after Ctrl-C.
		 */
		if (orgdir) chdir(orgdir);
		TransferFile[0] = '\0';
		closeinfo();
		globalflags &= ~INACTIVITY;
		menu_index = 0;
		nchr = 0;
		if (pid) {
			putstr("\n\n%s\n", sysmsg(MSG_INTERRUPTED));
			killpid("interrupt", pid);
			pid = 0;
			ttysetraw();
		}
		if (globalflags & IMMQUIT) quit(0);
		if (usenetinuse) UsenetClose();
		graphopen(1);
		process.flags &= ~(INCONF|INTALK|INRECVF|INSENDF|EXECEXT);
		saveinfo(&process);

		j = 0;	/* flag: after ^C */
	} else	{
		/*
		 * Log in the user.
		 */
		i = tnslogin();
		alarm(0);
		j = init_sysinfo();
		ptr = getuserconf(USERGROUP, USERDEFGROUP);
		if (!setgrpname(ptr)) ptr = "none";
		usergroup = strdup(ptr);
		LOGIT(LOG_NOTICE, "Login, %s <%s>", process.name, makestrstatus());
		LOGIT(LOG_INFO, "group \"%s\", privlevel %d, timeleft %s, terminal %s",
		      usergroup, privlevel, sec2str(j*60), term);
		if (!j) {
			if (!display(0, DAYLIMIT))
				putstr("\n%s\n", sysmsg(MSG_DAILYTIMEUP));
			LOGIT(LOG_NOTICE, "Daily time's up");
			quit(13);
		}
		init_selfinfo();
		menu_func = find_term(term);
		settimer(j);
		graphopen(0);
		globalflags = pid = 0;
		if (!i) process.flags |= HIDDENUSER;
		else process.flags &= ~HIDDENUSER;
		if (initIPC() < 0) {
			LOGIT(LOG_CRIT, "can't initialize IPC");
			quit(1);
		}
		dropusr1();
		settings();
		display(0, "%s/%s", homedir, STARTUPFILE);
		if (ncalls > 1) {
			if (ncalls > 7) ptr = WELCOME;
			else	ptr = ROOKIE;
		} else	ptr = NEWUSER;
		display(0, ptr);

		j = 1;	/* flag: after login */
	}

	/*
	 * Check for menu password at startup.
	 */
	do {
		/* system menu path */
		ptr = getsysconf(MENUPATH, DEFMENUPATH);
		if (*ptr != '/') sprintf(buf, "%s/%s", orgdir, skiplp(ptr));
		else strcpy(buf, ptr);
		if (!isaccess(buf)) strcpy(buf, ".");

		/* length of root portion of system menu path */
		if ((ptr = strrchr(buf, '/')) != NULL) i = ++ptr - buf;
		else i = 0;
		if (j) {
			j = 0;
			/* user menu path if any */
			if ((ptr = getuserconf(MENUPATH, NULL)) != NULL) {
				if (*ptr != '/') sprintf(tmpbuf, "%s/%s", orgdir, skiplp(ptr));
				else strcpy(tmpbuf, ptr);
				if (!isaccess(tmpbuf)) strcpy(tmpbuf, ".");
			} else strcpy(tmpbuf, buf);

			/* user menu path must in the system menu path */
			if (!strncmp(tmpbuf, buf, strlen(buf))) strcpy(buf, tmpbuf);
		}
		(void)strcpy(dirpath, &buf[i]);
	} while (!menupassword(dirpath));

	loggedin = TRUE;
	while (1) {
		free_dir(&menu_item, menu_num);
		globalflags = pid = 0;
		proctitle("IDLE");

		if ((menu_num = scan_dir(dirpath, &menu_item, acceptfile, 0)) < 0) {
			LOGIT(LOG_CRIT, "can't scan_dir \"%s\": %m", dirpath);
			quit(1);
		}
		review_priv();

		if ((i = (*menu_func)()) != -1) {
			if (i >= 0) {
				sprintf(buf, "%s/%s", dirpath, menu_item[i].name);
				if (!isgrpright(GSID_MENU, buf, GRF_R)) {
					nopermission(GRF_R);
					continue;
				}
				if (!menupassword(buf)) continue;
				if ((menu_item[i].mode & S_IFMT) == S_IFDIR) {
					if (menu_item[i].mode & S_IXUSR) {
						if ((j = readlink(buf, tmpbuf, sizeof(tmpbuf))) > 0) {
							tmpbuf[j] = 0;
							if (tmpbuf[0] == '/') strcpy(dirpath, tmpbuf);
							else dirlink(dirpath, tmpbuf);
						} else strcpy(dirpath, buf);
						(void)getarea();
					} else LOGIT(LOG_ERR, "access to \"%s\" denied", menu_item[i].name);
				} else	display(0, menu_item[i].name);
			} else	if (!display(0, HELPFILE)) warning(MSG_NOHELPAVAIL, 1);
		} else if (talkportid) {
			proctitle("TalkRespond");
			globalflags |= BEGINCHAT;
			talk(-2);
			talkportid = 0;
			putchr('\n');
			anykey();
		} else	(void)externaltalk();
	}
	/* Not reached */
	quit(0);
}

static int
acceptfile(path, file)
	char *path;
	struct dir_ent *file;
{
	register unsigned char *ptr, chr;

	/* first check for owner of menu item */
	if (file->uid > MAX_MENU_UID) return 0;

	/* check for file type of menu item */
	if ((file->mode & (S_IFREG | S_IFLNK | S_IFDIR)) == 0) return 0;

	/* check for user access */
	if ((ptr = strrchr(file->name, '.')) != NULL) {
		chr = atoi(++ptr);
		if (chr > privlevel) return 0;
	} else	chr = 0;
	file->priv = chr;

	/* find the mark character for menu item */
	for (chr = 0, ptr = file->name; *ptr && *ptr != '.'; ptr++) {
		if (!islower(*ptr) &&
		    *ptr != '_' && *ptr != '-' && *ptr != ' ' &&
		    !(*ptr >= 0xc0 && *ptr < 0xe0)) {
			chr = *ptr;
			break;
		}
	}
	if (!chr) return 0;
	file->key = chr;

	/* check for group permission */
	return isgrpright(GSID_MENU, path, GRF_S);
}

void
review_priv()
{
	register i, j;
	int priv, priv1, idx;

	for (i = 0; i < menu_num; i++) {
		if (!menu_item[i].key) continue;
		priv = menu_item[i].priv;
		idx = i;
		for (j = i + 1; j < menu_num; j++) {
			if (menu_item[j].key != menu_item[i].key)
				continue;
			priv1 = menu_item[j].priv;
			if (priv1 > priv) {
				priv = priv1;
				menu_item[idx].key = 0;
				idx = j;
			} else	menu_item[j].key = 0;
		}
	}
}

int
dirlink(dir, link)
	char *dir, *link;
{
	int i;
	char *ptr, *ptr2;

	if (!(i = strlen(link))) return 0;
	if (link[i-1] == '/') link[i-1] = 0;
	ptr2 = link;
	while ((ptr = strtok(ptr2, "/")) != NULL) {
		if (strcmp(ptr, "..") == 0 && (ptr2 = strrchr(dir, '/')) != NULL)
			*ptr2 = 0;
		else {
			(void)strcat(dir, "/");
			(void)strcat(dir, ptr);
		}
		ptr2 = NULL;
	}
	return 1;
}

int
isaccess(dirname)
	char *dirname;
{
	struct stat st;

	if (stat(dirname, &st) < 0) return FALSE;
	if ((st.st_mode & S_IFMT) != S_IFDIR) {
		errno = ENOTDIR;
		return FALSE;
	}
	return (access(dirname, F_OK|R_OK|X_OK) < 0 ? FALSE : TRUE);
}

int
run(fullname, av)
	char *fullname, *av[];
{
	int pstat, i;
	pid_t rval;
	sigfunc quitsave, intsave;
	sigset_t mask, omask;
	char *p, *ev[25];
	extern char *systemhost, *dszlogname, *makestrdotname();
	extern int langnumber, codepage;

	process.flags |= EXECEXT;
	saveinfo(&process);

	tmpbuf[0] = '\0';
	strcatf(tmpbuf, "PATH=%s:%s/bin:%s\n", orgdir, orgdir, homedir);
	strcatf(tmpbuf, "USER=%s\n", process.name);
	strcatf(tmpbuf, "GROUP=%s\n", usergroup);
	strcatf(tmpbuf, "HOME=%s\n", homedir);
	strcatf(tmpbuf, "CALLSIGN=%s\n", callsign);
	strcatf(tmpbuf, "DOTTEDNAME=%s\n", makestrdotname());
	strcatf(tmpbuf, "HOSTNAME=%s\n", systemhost);
	strcatf(tmpbuf, "EMAIL=%s\n", (char *)fromfield(FALSE));
	strcatf(tmpbuf, "MAILPATH=%s/%s\n", homedir, MAILBOX);

	if (*shellprog == '/') strcatf(tmpbuf, "SHELL=%s\n", shellprog);
	else strcatf(tmpbuf, "SHELL=%s/%s\n", orgdir, shellprog);

	if (*editprog == '/') strcatf(tmpbuf, "EDITOR=%s\n", editprog);
	else strcatf(tmpbuf, "EDITOR=%s/%s\n", orgdir, editprog);

	strcatf(tmpbuf, "TERM=%s\n", VT100);
	strcatf(tmpbuf, "%s=%d\n", USERTIMELEFT, gettimer()/60);
	strcatf(tmpbuf, "%s=%d\n", USERPRIVLEVEL, privlevel);
	strcatf(tmpbuf, "%s=%d\n", LANGNUMBER, langnumber);
	strcatf(tmpbuf, "%s=%d\n", CODEPAGE, codepage);
	strcatf(tmpbuf, "%s=%d\n", CONNECT, process.baud);

	if (remote_addr) strcatf(tmpbuf, "%s=%s\n", REMOTEADDR, remote_addr);
	if (remote_host) strcatf(tmpbuf, "%s=%s\n", REMOTEHOST, remote_host);
	if (loginline) strcatf(tmpbuf, "%s=%s\n", LINE, loginline);
	if (dszlogname) strcatf(tmpbuf, "DSZLOG=%s\n", dszlogname);

	for (i = 0, p = tmpbuf; (p = strtok(p, "\n")) != NULL && i < 24; p = NULL)
		ev[i++] = p;
	ev[i] = NULL;
	for (i = 0, inrunning[0] = '\0'; av[i] != NULL; i++) {
		pstat = sizeof(inrunning) - strlen(inrunning);
		if (pstat < strlen(av[i])+2) break;
		if (inrunning[0] != '\0') strcat(inrunning, " ");
		strcat(inrunning, av[i]);
	}
	proctitle(inrunning);
	cursoron();
	ttyrestore(1);

	sigemptyset(&mask);
	sigaddset(&mask, SIGCHLD);
	sigprocmask(SIG_BLOCK, &mask, &omask);
	switch (pid = vfork()) {
		case -1:
			LOGIT(LOG_CRIT, "can't fork() to exec \"%s\": %m", fullname);
			quit(1);
		case 0:
			sigprocmask(SIG_SETMASK, &omask, NULL);
			if (stdfile != NULL) freopen(stdfile, "w", stdout);
			setgid(getgid());
			setuid(getuid());
			execve(fullname, av, ev);
			_exit(127);
	}
	intsave = signal(SIGINT, SIG_IGN);
	quitsave = signal(SIGQUIT, SIG_IGN);
	LOGIT(LOG_DEBUG, "run: \"%s\" pid %d", fullname, pid);
	while ((rval = waitpid(pid, &pstat, 0)) == -1 && errno == EINTR);
	if (rval == -1) LOGIT(LOG_ERR, "can't waitpid() pid %d: %m", pid);
	else LOGIT(LOG_DEBUG, "run: ret status %d", pstat);
	pid = 0;
	stdfile = NULL;
	sigprocmask(SIG_SETMASK, &omask, NULL);
	(void)signal(SIGINT, intsave);
	(void)signal(SIGQUIT, quitsave);

	ttysetraw();
	cursoroff();
	process.flags &= ~(INCONF|INTALK|INRECVF|INSENDF|EXECEXT);
	saveinfo(&process);
	if (orgdir) chdir(orgdir);

	return (rval == -1 ? -1 : pstat);
}

static void
killpid(func, pid)
	char *func;
	pid_t pid;
{
	int ret;

	ret = kill(pid, SIGHUP);
	if (ret < 0 && errno != ESRCH)
		LOGIT(LOG_ERR, "%s: hangup pid %d: %m", func, pid);
	if (ret == 0) {
		sleep(1);
		kill(pid, SIGKILL);
	}
	LOGIT(LOG_DEBUG, "%s: pid %d killed", func, pid);
}

int
sigalarm()
{
	struct sigaction act;
	void onalarm();

	(void) signal(SIGALRM, onalarm);
	act.sa_handler = onalarm;
	sigemptyset(&act.sa_mask);
	act.sa_flags = SA_RESTART;
	return sigaction(SIGALRM, &act, NULL);
}

int
sig_set(sig, func)
	int sig;
	sigfunc func;
{
	struct sigaction act;

	(void) signal(sig, func);
	act.sa_handler = func;
	sigemptyset(&act.sa_mask);
	act.sa_flags = SA_RESTART;
	return sigaction(sig, &act, NULL);
}

void
sigcatch()
{
	void onhup(), onint(), onterm(), onpipe(), onsegv();
	extern void onusr1(), onusr2();

	(void) signal(SIGHUP, onhup);
	(void) signal(SIGINT, onint);
	(void) signal(SIGQUIT, onterm);
	(void) signal(SIGABRT, onterm);
	(void) signal(SIGTERM, onterm);
	(void) signal(SIGSEGV, onsegv);
	(void) signal(SIGTRAP, SIG_IGN);
	(void) signal(SIGTSTP, SIG_IGN);
	(void) sig_set(SIGPIPE, onpipe);
	(void) sig_set(SIGUSR1, onusr1);
	(void) sig_set(SIGUSR2, onusr2);

/* Patched by SLA */
	(void) signal(SIGIO, SIG_IGN);
/* End of patch */
	
	sigalarm();
}

void
sigdiscard()
{
	(void) signal(SIGHUP, SIG_IGN);
	(void) signal(SIGINT, SIG_IGN);
	(void) signal(SIGQUIT, SIG_IGN);
	(void) signal(SIGTERM, SIG_IGN);
	(void) signal(SIGALRM, SIG_IGN);
	(void) signal(SIGIO, SIG_IGN);
	(void) signal(SIGUSR1, SIG_IGN);
}

void
backupinfo()
{
	int timeleft, fd;
	struct stat st;
	extern int UserArtRead, UserArtSize, UserArtPost, UserMsgRead, UserMsgPost;
	extern char *systemhost, *makestrdotname();

	restoretimer(0);
	if (logoutprog != NULL) {
		if (*logoutprog == '/') strcpy(tmpbuf, logoutprog);
		else sprintf(tmpbuf, "%s/%s", orgdir, logoutprog);
		if (stat(tmpbuf, &st) == 0 && st.st_uid <= MAX_MENU_UID &&
		    (st.st_mode & S_IFMT) == S_IFREG && (st.st_mode & S_IXUSR)) {
			strcatf(tmpbuf, " %d", (time(NULL) - process.ltime) / 60);
			put_env("PATH=%s:%s/bin:%s", orgdir, orgdir, homedir);
			put_env("USER=%s", process.name);
			put_env("GROUP=%s", usergroup);
			put_env("HOME=%s", homedir);
			put_env("CALLSIGN=%s", callsign);
			put_env("DOTTEDNAME=%s", makestrdotname());
			put_env("HOSTNAME=%s", systemhost);
			put_env("EMAIL=%s", (char *)fromfield(FALSE));
			put_env("%s=%d", USERTIMELEFT, gettimer()/60);
			put_env("%s=%d", USERPRIVLEVEL, privlevel);
			if (remote_addr) put_env("%s=%s", REMOTEADDR, remote_addr);
			if (remote_host) put_env("%s=%s", REMOTEHOST, remote_host);
			if (loginline) put_env("%s=%s", LINE, loginline);
			system(tmpbuf);
		}
	}
	timeleft = gettimer()/60;
	putuserconf(USERTIMELEFT, itoa(timeleft + restoretime));
	putuserconf(LASTCALLTIME, itoa(time(NULL)));
	putuserconf(USERPRIVLEVEL, itoa(privlevel));
	putuserconf(USERGROUP, usergroup);
	if (loggedin) putuserconf(MENUPATH, dirpath);
	if (UserArtRead)
		putuserconf(USERARTREAD, itoa(UserArtRead));
	if (UserArtSize/1024)
		putuserconf(USERARTSIZE, itoa(UserArtSize/1024));
	if (UserArtPost)
		putuserconf(USERARTPOST, itoa(UserArtPost));
	if (UserMsgRead)
		putuserconf(USERMSGREAD, itoa(UserMsgRead));
	if (UserMsgPost)
		putuserconf(USERMSGPOST, itoa(UserMsgPost));

	chkhangfiles(FALSE);
	parsedszlog(FALSE);
	hangnetchat();
	hangnewsrc();
	update_folder();
	/* save alternate user.conf from memory to file if any */
	if (path_altuserconf != NULL) {
		if (body_altuserconf != NULL) {
			if ((fd = open(path_altuserconf, O_WRONLY|O_CREAT|O_TRUNC, 0640)) != -1) {
				write(fd, body_altuserconf, strlen(body_altuserconf));
				close(fd);
			}
		} else unlink(path_altuserconf);
		free(path_altuserconf);
		path_altuserconf = NULL;
	}
	if (timeleft < 2 && (process.flags & CHARGEUSER))
		LOGIT(LOG_WARNING, "User %s: charge time's up", process.name);
}

cleanexit(rval)
	int rval;
{
	purgeinfo(&process);
	dropusr1();
	if (talkaddrfile != NULL) unlink(talkaddrfile);
	if (ftpcachfile != NULL) unlink(ftpcachfile);
	if (pidfile != NULL) unlink(pidfile);
	exit(rval);
}

void
quit(rval)
	int rval;
{
	sigdiscard();
	if (rval == 1) putstr("\007\n*** system error!\n");
	if (globalflags & IMMQUIT)
		putstr("\007\n*** immediately quit(%d). bye!\n", rval);
	else	backupinfo();

	graphclose();
	ttyrestore(getppid() > 1);	/* flush if have parent proccess */
	LOGIT(LOG_NOTICE, "Logout, timeused %s",
	      sec2str(time(NULL) - process.ltime));
	cleanexit(rval);
}


static int maxtimer;
int acttimer;
int timerate = TIMECOUNTER;
int intrrate = TIMECOUNTER;

void
onalarm()
{
	extern int syschatinactive;

	sigalarm();
	if (syschatinactive || (process.flags & INFINITYUSER)) return;
	if ((maxtimer -= timerate) < timerate) {
		if (((process.flags & INCONF) && (process.flags & INFINITYCONF)) ||
		    (globalflags & (SENDINPROGRESS|RECVINPROGRESS))) {
			maxtimer += timerate;
			return;
		}
		if (pid) killpid("onalarm", pid);
		if (globalflags & IMMQUIT)
			putstr("\007\nLogin time limit reached\n");
		else	putstr("\007\n%s\n", sysmsg(MSG_YOURTIMESUP));
		LOGIT(LOG_NOTICE, "Time limit reached");
		quit(0);
	}
	if (globalflags & INACTIVITY) {
		if ((acttimer -= intrrate) < intrrate) {
			if (globalflags & IMMQUIT)
				putstr("\007\nIdle time reached\n");
			else	putstr("\007\n%s\n", sysmsg(MSG_INACTTIMEUP));
			LOGIT(LOG_NOTICE, "Idle time reached");
			quit(0);
		}
	} else acttimer = INACTTIMER * 60;
}

void
onint()
{
	if (orgdir) chdir(orgdir);
	(void)signal(SIGINT, onint);
	longjmp(jmpint, 1);
}

void
onpipe()
{
	if (process.flags & KICKER)
		process.flags &= ~KICKER;
	else	process.flags |= KICKER;
	saveinfo(&process);
	sig_set(SIGPIPE, onpipe);
}

void
onsegv()
{
	LOGIT(LOG_CRIT, "SIGSEGV - It seems to be a BUG!");
	quit(1);
}

void
onhup()
{
	if (orgdir) chdir(orgdir);
	sigdiscard();
	LOGIT(LOG_DEBUG, "SIGHUP");
	if (pid) killpid("onhup", pid);
	if (!(globalflags & IMMQUIT)) backupinfo();
	LOGIT(LOG_NOTICE, "HangUp, timeused %s",
	      sec2str(time(NULL) - process.ltime));
	cleanexit(0);
}

void
onterm()
{
	if (orgdir) chdir(orgdir);
	LOGIT(LOG_DEBUG, "SIGTERM");
	if (pid) killpid("onterm", pid);
	putstr("\007\n*** terminated!\n");
	quit(0);
}

int
setalarm()
{
	struct itimerval it;
	register struct itimerval *itp = &it;

	itp->it_value.tv_sec = itp->it_interval.tv_sec = intrrate;
	itp->it_value.tv_usec = itp->it_interval.tv_usec = 0;
	return (setitimer(ITIMER_REAL, itp, NULL));
}

static time_t savetime = 0;

restoretimer(timer)
	int timer;
{
	if (savetime) {
		time_t t = time(NULL) - savetime;
		savetime = 0;
		if (t > 0) {
			maxtimer -= t;
			if (maxtimer < timerate) maxtimer = timerate;
		}
	}
	if (timer) setalarm();
}

backuptimer()
{
	if (savetime) restoretimer(0);
	alarm(0);
	savetime = time(NULL);
}

int
settimer(maxmin)
	int maxmin;
{
	maxtimer = maxmin * 60 - intrrate;
	acttimer = INACTTIMER * 60;
	return setalarm();
}

int
gettimer()
{
	struct itimerval it;
	getitimer(ITIMER_REAL, &it);
	return (maxtimer + it.it_value.tv_sec);
}

static jmp_buf dontresponse;

static void
waitforesponse()
{
	longjmp(dontresponse, 1);
}

char *
getresponse(maxchars, seconds)
	int maxchars;
	unsigned int seconds;
{
	char *p;
	sigfunc oldint;

	if (!seconds) return NULL;
	oldint = signal(SIGINT, SIG_IGN);
	p = NULL;
	backuptimer();
	if (!setjmp(dontresponse)) {
		(void) signal(SIGALRM, waitforesponse);
		alarm(seconds);
		p = getstr(maxchars, 1, ECHODISABLE);
		alarm(0);
	} else	p = NULL;
	sigalarm();
	restoretimer(1);
	(void) signal(SIGINT, oldint);
	return p;
}

static sigfunc intsave;
static int numopenstreams = 0;

FILE *
sfopen(path, mode)
	char *path, *mode;
{
	FILE *fp;

	if ((fp = fopen(path, mode)) == NULL) return NULL;
	if (!numopenstreams++) intsave = signal(SIGINT, SIG_IGN);
	return fp;
}

int
sfclose(s)
	FILE *s;
{
	int rval = fclose(s);
	if (!--numopenstreams) (void)signal(SIGINT, intsave);
	return rval;
}

int
init_sysinfo()
{
	int timeleft;
	char *ptr;
	time_t lastlogin, newday;
	struct tm tmlogged, *tm;
	extern int dailydlsize, dailyulsize;

	memcpy(&tmlogged, localtime(&process.ltime), sizeof(struct tm));

	/* save alternate user.conf in memory */
	if (process.flags & CHARGEUSER)
		ptr = USERCONF;
	else	ptr = CHARGEUSERCONF;
	path_altuserconf = strdupf("%s/%s", homedir, ptr);
	if ((ptr = loadfile(path_altuserconf)) != NULL)
		body_altuserconf = strdupf(ptr);

	/* Prepare editor work file */
	editorworkfile = strdupf("%s/.editor", homedir);

	/* Prepare ftp cach file */
	ftpcachfile = strdupf("%s/.ftp", homedir);

	talkaddrfile = strdupf("%s/.talk.%s", orgdir, fntty);
	unlink(talkaddrfile);

	/* Get last login time */
	if ((ptr = getuserconf("LASTLOGIN", NULL)) != NULL)
		lastlogin = atoi(ptr);
	else	lastlogin = process.ltime;

	/* Get last logout time */
	if ((ptr = getuserconf(LASTCALLTIME, NULL)) != NULL) {
		lastcalltime = atoi(ptr);
		tm = localtime(&lastcalltime);
	} else {
		lastcalltime = process.ltime;
		tm = NULL;
	}

	/* Get user privlevel */
	if (!privlevel) {
		if ((ptr = getuserconf(USERPRIVLEVEL, NULL)) != NULL)
			privlevel = atoi(ptr);
		else if ((ptr = getsysconf(USERPRIVLEVEL, NULL)) != NULL)
			privlevel = atoi(ptr);
		else	privlevel = DEFPRIVLEVEL;
	}

	/* Find daily timeleft limit */
	maxsessiontime = restoretime = 0;
	if ((timeleft = init_limits()) < 0) {
		if ((ptr = getsysconf(USERTIMELEFT, NULL)) != NULL)
			timeleft = atoi(ptr);
		else	timeleft = DEFTIMELEFT;
	}
	if ((ptr = getuserconf(SESSIONTIME, NULL)) != NULL)
		maxsessiontime = atoi(ptr);
	if ((ptr = getuserconf(DAILYTIMELEFT, NULL)) != NULL)
		timeleft = atoi(ptr);
	if (timeleft == 0) {
		process.flags |= INFINITYUSER;
		timeleft = DEFTIMELEFT;
	}

	dailydlsize = dailyulsize = 0;

	/* Charge user */
	if (process.flags & CHARGEUSER) {
		maxsessiontime = restoretime = 0;
		if ((ptr = getuserconf(USERTIMELEFT, NULL)) != NULL)
			timeleft = atoi(ptr);
	}

	if (tm != NULL &&
	    tm->tm_year == tmlogged.tm_year &&
	    tm->tm_mon  == tmlogged.tm_mon &&
	    tm->tm_mday == tmlogged.tm_mday) {	/* That's day */
		tmlogged.tm_hour = tmlogged.tm_min = tmlogged.tm_sec = 0;
		newday = mktime(&tmlogged);

		if ((process.flags & (CHARGEUSER|INFINITYUSER)) == 0 &&
		    (ptr = getuserconf(USERTIMELEFT, NULL)) != NULL) {
			timeleft = atoi(ptr);
			if (lastlogin < newday && newday < lastcalltime)
				timeleft -= (lastcalltime - newday)/60;
		}
		if ((ptr = getuserconf(DAILYDLSIZE, NULL)) != NULL)
			dailydlsize = atoi(ptr) * 1024;
		if ((ptr = getuserconf(DAILYULSIZE, NULL)) != NULL)
			dailyulsize = atoi(ptr) * 1024;
	} else	newday = 0;	/* New day */

	if (timeleft < 2) return 0;

	putuserconf("LASTLOGIN", itoa(process.ltime));
	putuserconf(LASTCALLTIME, itoa(process.ltime));
	if (!newday) {
		putuserconf(DAILYDLSIZE, "");
		putuserconf(DAILYULSIZE, "");
	}

	if (maxsessiontime > 2 && timeleft > maxsessiontime) {
		restoretime = timeleft - maxsessiontime;
		timeleft = maxsessiontime;
	}
	if ((ptr = getuserconf("TERMINATOR", NULL)) != NULL) {
		if (!strcasecmp(ptr, "Yes"))
			process.flags |= (TERMINATOR|INFINITYCONF);
		else if (!strcasecmp(ptr, "On"))
			process.flags |= (TERMINATOR|INFINITYCONF);
		else if (atoi(ptr))
			process.flags |= (TERMINATOR|INFINITYCONF);
	}
	if (!(process.flags & INFINITYCONF) &&
	    (ptr = getsysconf(INFINITY, NULL)) != NULL) {
		if (!strcasecmp(ptr, "Yes"))
			process.flags |= INFINITYCONF;
		else if (!strcasecmp(ptr, "On"))
			process.flags |= INFINITYCONF;
		else if (atoi(ptr))
			process.flags |= INFINITYCONF;
	}
	if ((ptr = getuserconf(CALLNUMBER, NULL)) != NULL)
		ncalls = atoi(ptr);
	else	ncalls = 0;
	putuserconf(CALLNUMBER, itoa(++ncalls));

	if ((ptr = getuserconf(INTERRUPT, NULL)) != NULL)
		intrrate = atoi(ptr);
	else if ((ptr = getsysconf(INTERRUPT, NULL)) != NULL)
		intrrate = atoi(ptr);
	else	intrrate = TIMECOUNTER;

	if ((ptr = getuserconf(TIMERATE, NULL)) != NULL)
		timerate = atoi(ptr);
	else if ((ptr = getsysconf(TIMERATE, NULL)) != NULL)
		timerate = atoi(ptr);
	else	timerate = intrrate;

	if (remote_host) putuserconf(REMOTEHOST, remote_host);

	return timeleft;
}

int
init_limits()
{
	int timeleft = -1;
	int d1, d2, d3, d4, d5, d6, d7, d8, d9, d10, d11;
	FILE *fp;
	extern int dailydllimit, ulmultiplier, dlulratio;
	extern int ArtReadLimit, ArtPostLimit, ArtSizeLimit;
	extern int MsgReadLimit, MsgPostLimit;

	if ((fp = sfopen(MAINLIMITSCONF, "r")) == NULL)
		if ((fp = sfopen(LIMITSCONF, "r")) == NULL) return -1;
	while (fgets(tmpbuf, sizeof(tmpbuf), fp) != NULL) {
		tmpbuf[sizeof(tmpbuf)-1] = '\0';
		if (tmpbuf[0] == '#' || tmpbuf[0] == '\n') continue;
		if (sscanf(tmpbuf, "%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\n",
			      &d1, &d2, &d3, &d4, &d5, &d6, &d7, &d8, &d9, &d10, &d11) != 11) {
			sfclose(fp);
			badline(LIMITSCONF, tmpbuf);
			return -1;
		}
		if (d1 <= privlevel) {
			timeleft = d2;
			maxsessiontime = d3;
			dailydllimit = d4 * 1024;
			ulmultiplier = d5;
			dlulratio = d6;
			MsgReadLimit = d7;
			MsgPostLimit = d8;
			ArtReadLimit = d9;
			ArtSizeLimit = d10 * 1024;
			ArtPostLimit = d11;
		}
	}
	sfclose(fp);
	return timeleft;
}

dropusr1()
{
	pid_t monpid;
	FILE *fp;
	char buf[40];

	if ((fp = sfopen(TNSMONPID, "r")) != NULL) {
		if (fgets(buf, sizeof(buf), fp) != NULL) {
			buf[sizeof(buf)-1] = '\0';
			monpid = atoi(buf);
			if (monpid && monpid != process.pid)
				kill(monpid, SIGUSR1);
		}
		sfclose(fp);
	}
	return;
}

int
menupassword(menupath)
	char *menupath;
{
	int rval;
	FILE *fp;
	char *p, buf[25], fpath[1024];
	extern void promptpasswd();

	sprintf(fpath, "%s/%s", menupath, MENUPASSWORD);
	if ((fp = sfopen(fpath, "r")) == NULL) return 1;
	buf[0] = '\0';
	fgets(buf, sizeof(buf), fp);
	buf[sizeof(buf)-1] = '\0';
	sfclose(fp);
	if ((p = strchr(buf, '\n')) != NULL) *p = '\0';
	if (buf[0] == '\0') return 1;
	if (*menupath != '/') {
		sprintf(fpath, "%s/%s", orgdir, menupath);
		menupath = fpath;
	}
	putchr('\n');
	if (!display(0, "%s/%s", menupath, MENUPASSSHOW)) {
		if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;37m");
		putstr("%s\n", sysmsg(MSG_MENUPASSWORD));
	}
	promptpasswd(MSG_ENTERMENUPASSWORD);
	p = getstr(25, 0, ECHODISABLE);
	if (!*p || strcmp(buf, p)) {
		warning(MSG_BADMENUPASSWORD, 0);
		anykey();
		rval = FALSE;
	} else	rval = TRUE;
	putchr('\n');
	return rval;
}

settings()
{
	int i;
	char *pt, *pu, *ptr, tpat[40], upat[40], buf[1024];
	FILE *fp;
	struct stat st;

	if ((fp = sfopen(SETUSERFILE, "r")) == NULL &&
	    (fp = sfopen(strippath(SETUSERFILE), "r")) == NULL) return;

	if (fstat(fileno(fp), &st) < 0 || st.st_uid > MAX_MENU_UID ||
	    (st.st_mode & S_IFMT) != S_IFREG) {
		sfclose(fp);
		return;
	}

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		buf[sizeof(buf)-1] = '\0';
		if ((ptr = strrchr(buf, '\n')) != NULL) *ptr = '\0';
		for (ptr = buf; (u_char)*ptr <= 0x20 && *ptr; ptr++);
		if (!*ptr || *ptr == '#') continue;

		if ((pt = strstr(ptr, "T:")) != NULL) {
			for (i = 0, pt += 2; *pt && (u_char)*pt > 0x20 &&
			     i < sizeof(tpat)-1; i++, pt++) tpat[i] = *pt;
			tpat[i] = '\0';
		} else {
			pt = ptr;
			tpat[0] = '\0';
		}

		if ((pu = strstr(ptr, "U:")) != NULL) {
			for (i = 0, pu += 2; *pu && (u_char)*pu > 0x20 &&
			     i < sizeof(upat)-1; i++, pu++) upat[i] = *pu;
			upat[i] = '\0';
		} else {
			pu = ptr;
			upat[0] = '\0';
		}

		if (pu > pt) ptr = pu;
		else ptr = pt;

		if (tpat[0] != '\0' && upat[0] != '\0') {
			if (rematch(tpat, process.tty) &&
			    rematch(upat, process.name))
				setvariables(ptr);
		} else if (tpat[0] != '\0') {
			if (rematch(tpat, process.tty)) setvariables(ptr);
		} else if (upat[0] != '\0') {
			if (rematch(upat, process.name)) setvariables(ptr);
		} else setvariables(ptr);
	}
	sfclose(fp);
	return;
}


static struct area_ent {
	char *menu;
	char area[1024];
	struct area_ent *next;
} *arealist = NULL;

char *
setarea(area)
	char *area;
{
	register struct area_ent *cl, *mem;

	for (cl = mem = arealist; cl != NULL; cl = cl->next) {
		if (!strcmp(dirpath, cl->menu)) {
			if (*area) putuserconf(strippath(dirpath), area);
			return strcpy(cl->area, area);
		}
		mem = cl;
	}
	if (*area) putuserconf(strippath(dirpath), area);
	if ((cl = malloc(sizeof(struct area_ent))) == NULL) {
		LOGIT(LOG_CRIT, "can't allocate memory");
		quit(1);
	}
	cl->next = NULL;
	if ((cl->menu = strdup(dirpath)) == NULL) {
		LOGIT(LOG_CRIT, "can't allocate memory");
		quit(1);
	}
	if (mem == NULL) arealist = cl;
	else mem->next = cl;
	return strcpy(cl->area, area);
}

char *
getarea()
{
	register struct area_ent *cl;

	for (cl = arealist; cl != NULL; cl = cl->next)
		if (!strcmp(dirpath, cl->menu)) return cl->area;
	return setarea(getuserconf(strippath(dirpath), ""));
}

void
externaltalk()
{
	int fd;
	char *ptr;

	if ((fd = open(talkaddrfile, O_RDONLY)) < 0) return;
	tmpbuf[0] = '\0';
	if (read(fd, tmpbuf, sizeof(tmpbuf)) < 0) return;
	close(fd);
	if ((ptr = strchr(tmpbuf, '\n')) != NULL) *ptr = '\0';
	if (strlen(tmpbuf) > 1) invoke_cmd("%s %s", talkprog, tmpbuf);
	unlink(talkaddrfile);
}

/* get connect speed from file */
int
getconnectfile()
{
	int baud;
	FILE *fp;
	char *p;

	(void)sprintf(tmpbuf, CONNECTFILE, fntty);
	if ((fp = fopen(tmpbuf, "r")) == NULL) return 0;

	while (fgets(tmpbuf, sizeof(tmpbuf), fp) != NULL) {
		tmpbuf[sizeof(tmpbuf)-1] = '\0';
		if ((p = strpbrk(tmpbuf, "\r\n")) != NULL) *p = '\0';
		if ((p = strstr(tmpbuf, CONNECT)) == NULL) continue;
		for (p += sizeof(CONNECT)-1; *p == ' ' || *p == '\t'; p++);
		if (*p) {
			baud = atoi(p);
		/* some modems return dotted connect (19.2, 28.8, etc) */
			while (isdigit(*p)) p++;
			if (*p == '.')
				baud = baud * 1000 + atoi(++p) * 100;
			fclose(fp);
			return baud;
		}
	}
	fclose(fp);
	return 0;
}

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

static void
set_remote()
{
	struct utmpent *ut;
	struct hostent *hp;
	struct sockaddr_in remote;
	char hostaddr[256];

	remote_addr = NULL;
	remote_host = NULL;

	*hostaddr = '\0';
	while ((ut = getutmpent(process.name)) != NULL) {
		if (!strcmp(ut->line, process.tty)) {
			(void)strcpy(hostaddr, ut->host);
			break;
		}
	}
	endutmpent();

	if (*hostaddr == '\0') {
		remote_addr = strdup("127.0.0.1");
		remote_host = strdup("localhost");
	} else {
		if (atoi(hostaddr)) {
			remote.sin_addr.s_addr = inet_addr(hostaddr);
			hp = gethostbyaddr((char *)&remote.sin_addr, 4, AF_INET);
		} else {
			if (strchr(hostaddr, '.') == NULL) {
				char *p, hostname[100];
				gethostname(hostname, sizeof(hostname));
				if ((p = strchr(hostname, '.')) != NULL)
					(void)strcat(hostaddr, p);
			}
			hp = gethostbyname(hostaddr);
		}
		if (hp != NULL) {
			remote_host = strdup(hp->h_name);
			memcpy((char *)&remote.sin_addr, hp->h_addr, hp->h_length);
			remote_addr = strdup(inet_ntoa(remote.sin_addr));
		} else {
			remote_host = strdup(hostaddr);
			remote_addr = remote_host;
		}
	}
}
