/*
 *	Copyright (c) 1994,1995 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <time.h>
#include <ctype.h>
#include <unistd.h>
#include <utime.h>

#include "compat.h"
#include "drive.h"
#include "sysmsg.h"
#include "ripwraps.h"
#include "usenet.h"
#include "group.h"
#include "rematch.h"
#include "scandir.h"
#include "loginfo.h"
#include "variables.h"

extern struct loginfo process;
extern char *usersetup(), *workdir;
extern void mail_to_sysop();

char searchstring[MAXSEARCHSTRINGS+1][MAXSTRINGLENGTH+1];
char *hidedirs[64];
char *defuploadpath, *uploadpath;
char *dszlogname = NULL;
int dailydlsize;
int dailyulsize;
int userdlsize = 0;
int userulsize = 0;
int dailydllimit = DEFDLSIZELIMIT * 1024;
int ulmultiplier = DEFULMULTIPLIER;
int dlulratio = DEFULMULTIPLIER;

char *month[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
char working[4] = { '/', '-', '\\', '|'};

static char tmpbuf[1024];
static char *nodesconfile;
time_t newfilestime;

static struct {
	char *mask;		/* regex directory mask */
	char *name;		/* file index name */
	int mode;		/* LISTMODE, see drive.h */
	char *tabl;		/* translating codetable name */
} fileslist[MAXCONFENTRIES];

static struct {
	char *exp;		/* regex files mask */
	char *prg;		/* program to execute */
} viewers[MAXCONFENTRIES];

static struct {
	char dir;		/* transfer direction 'D' or 'U' */
	int batch;		/* batch supported flag */
	char mark[2];		/* bidirections protocol markers */
	char *prt;		/* protocol name */
	char *prg;		/* program to execute */
} transfers[MAXCONFENTRIES];

static char dszmark[2];
static char *commentul;

extern int errno;

void
initfileslist()
{
	int i;
	char *ptr, *line, *mask, *name, *ptr2;

	fileslist[0].mask = NULL;

	if ((ptr = loadfile(MAINFILESLIST)) == NULL &&
	    (ptr = loadfile(FILESLIST)) == NULL) return;

	for (i = 0; i < MAXCONFENTRIES-1 &&
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#') continue;
		mask = ptr;
		while ((u_char)*ptr > 0x20) ptr++;
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr; ptr++);
		if (!*ptr) continue;
		name = ptr;
		while ((u_char)*ptr > 0x20 && *ptr != ',') ptr++;
		for (ptr2 = ptr; (u_char)*ptr2 <= 0x20 && *ptr2; ptr2++);
		if (*ptr2 == ',') {
			for (*ptr2++ = 0; (u_char)*ptr2 <= 0x20 && *ptr2; ptr2++);
			ptr = ptr2;
			while ((u_char)*ptr > 0x20) ptr++;
		} else  ptr2 = NULL;
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr; ptr++);
		fileslist[i].mode = 0;
		if (*name == '!') {
			name++;
			if (!*name) continue;
			fileslist[i].mode |= LISTMODE_EXT;
		}
		fileslist[i].mask = strdup(mask);
		fileslist[i].name = strdup(name);
		if (ptr2 != NULL) {
			if (!strcasecmp(ptr2, "raw"))
				fileslist[i].mode |= LISTMODE_RAW;
			else if (!strcasecmp(ptr2, "all"))
				fileslist[i].mode |= LISTMODE_ALL;
		}
		if (*ptr) fileslist[i].tabl = strdup(ptr);
		else fileslist[i].tabl = NULL;
		i++;
	}
	fileslist[i].mask = NULL;
}

void
initviewers()
{
	int i;
	char *ptr, *line, *exp;

	viewers[0].prg = NULL;

	if ((ptr = loadfile(MAINVIEWERSCONF)) == NULL &&
	    (ptr = loadfile(VIEWERSCONF)) == NULL) return;

	for (i = 0; i < MAXCONFENTRIES-1 &&
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = 0;
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#') continue;
		exp = ptr;
		while ((u_char)*ptr > 0x20) ptr++;
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr != 0; ptr++);
		if (*ptr) {
			viewers[i].exp = strdup(exp);
			viewers[i].prg = strdup(ptr);
			i++;
		}
	}
	viewers[i].prg = NULL;
}

void
inittransfers()
{
	int i, j, batch;
	char *ptr, *ptr2, *line, *prt, dir, mark[2];
	struct stat st;

	transfers[0].prg = NULL;

	if ((ptr = loadfile(MAINFTPCONF)) == NULL &&
	    (ptr = loadfile(FTPCONF)) == NULL) return;

	for (i = 0; i < MAXCONFENTRIES-1 &&
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = 0;
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#') continue;
		dir = *ptr++;
		if (batch = (*ptr == 'B')) ptr++;
		if (*ptr == ',') {
			ptr++;
			mark[0] = *ptr++;
			mark[1] = *ptr++;
		} else	{
			mark[0] = '\0';
			mark[1] = '\0';
		}
		while ((u_char)*ptr <= 0x20 && *ptr != 0) ptr++;
		if (!*ptr) continue;
		prt = ptr;
		while ((u_char)*ptr > 0x20) ptr++;
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr != 0; ptr++);
		if (!*ptr) continue;
		for (ptr2 = ptr, j = 0; (u_char)*ptr2 > 0x20; ptr2++, j++)
			tmpbuf[j] = *ptr2;
		tmpbuf[j] = '\0';
		if (stat(tmpbuf, &st) < 0 || (st.st_mode & S_IFMT) != S_IFREG ||
		    !(st.st_mode & S_IRUSR) || !(st.st_mode & S_IXUSR))
			continue;
		transfers[i].prt = strdup(prt);
		transfers[i].prg = strdup(ptr);
		transfers[i].dir = dir;
		transfers[i].batch = batch;
		transfers[i].mark[0] = mark[0];
		transfers[i].mark[1] = mark[1];
		i++;
	}
	transfers[i].prg = NULL;
}

initfilearea()
{
	int i;
	char *ptr;

	dszlogname = strdupf("%s.%s", DSZLOG, fntty);
	unlink(dszlogname);
	dszmark[0] = '\0';
	dszmark[1] = '\0';

	i = 0;
	hidedirs[i++] = orgdir;
	if ((ptr = getsysconf(HIDEDIRS, NULL)) != NULL) {
		ptr = strdup(ptr);
		while ((ptr = strtok(ptr, " \t,;")) != NULL && i < 63) {
			hidedirs[i++] = ptr;
			ptr = NULL;
		}
	}
	hidedirs[i] = NULL;

	defuploadpath = strdup(getsysconf(UPLOADPATH, workdir));
	uploadpath = NULL;

	if ((commentul = getsysconf("UPLOADINFO", NULL)) != NULL)
		commentul = strdup(commentul);

	if ((ptr = getuserconf(UPLOADPATH, NULL)) != NULL &&
	    isaccess(ptr)) uploadpath = strdup(ptr);

	if ((ptr = getuserconf(DAILYDLLIMIT, NULL)) != NULL)
		dailydllimit = atoi(ptr) * 1024;
	if ((ptr = getuserconf(ULMULTIPLIER, NULL)) != NULL)
		ulmultiplier = atoi(ptr);
	if ((ptr = getuserconf(SYSDLULRATIO, NULL)) != NULL)
		dlulratio = atoi(ptr);
	if ((ptr = getuserconf(USERDLSIZE, NULL)) != NULL)
		userdlsize = atoi(ptr) * 1024;
	if (userdlsize <= 0) userdlsize = 1;
	if ((ptr = getuserconf(USERULSIZE, NULL)) != NULL)
		userulsize = atoi(ptr) * 1024;
	if (userulsize <= 0) userulsize = 1;
	nodesconfile = strdup(getsysconf(NODESCONFILE, DEFNODESCONFILE));
	if ((ptr = getsysconf(XFERLOG, NULL)) != NULL)
		if (!initxferlog(ptr))
			LOGIT(LOG_ERR, "can't open xferlog \"%s\": %m", ptr);
	initfileslist();
	initviewers();
	inittransfers();
}

int
content(path, filename)
	char *path, *filename;
{
	int i, j;
	register char *ptr;
	char temp[1024];
	struct stat st;

	if (!isgrpright(GSID_FILE, strncmp(path, homedir, strlen(homedir)) ?
			path : HOMEDIR, GRF_R))
		return nopermission(GRF_R);

	for (i = 0; viewers[i].prg != NULL; i++)
		if (rematch(viewers[i].exp, filename)) break;
	if (viewers[i].prg == NULL) return -1;

	for (j = 0; j < 3; j++) {
		sprintf(temp, "%s/%s", path, filename);
		if (stat(temp, &st) == 0) break;
		if (!j) {
			strlwr(filename);
			continue;
		}
		for (ptr = filename; *ptr && *ptr != '-' && *ptr != '_'; ptr++);
		if (*ptr == '-') *ptr = '_';
		else if (*ptr == '_') *ptr = '-';
		else j = 3;
	}
	if (j > 2) return 0;

	if (!isgrpright(GSID_FILE, temp, GRF_R))
		return nopermission(GRF_R);

	if (display(1, viewers[i].prg, temp) == 1) anykey();
	return 0;
}

int
download(path, av, ac, wantidx)
	char *path, *av[];
	int ac, wantidx;
{
	register int i, j;
	register char *ptr;
	int lncnt, dlsize, dltime;
	int tmpvar, tmpvar2;
	struct stat st;
	char *token, buf[256], ftpexecbuf[256], *avf[MAXCONFLINES+1], *ave[MAXCONFLINES+10];

	if (!isgrpright(GSID_FILE, strncmp(path, homedir, strlen(homedir)) ?
			path : HOMEDIR, GRF_R))
		return nopermission(GRF_R);

	if (transfers[0].prg == NULL) return -1;

	if (wantidx < 0) ptr = NULL;
	else ptr = av[wantidx];

	if (!((termflags & RIPMOUSE) && token != NULL))
		if ((ptr = prompt_str(MSG_ENTERFILENAME, ptr, 0)) == NULL)
			return 1;

	for (i = lncnt = dlsize = dltime = 0, token = strcpy(buf, ptr);
	     (token = strtok(token, " \t")) != NULL && i < MAXCONFLINES;
	     token = NULL) {
		for (j = 0; j < ac && i < MAXCONFLINES; j++) {
			if (fnmatch(token, av[j], FNM_NOESCAPE)) continue;
			for (tmpvar = 0; tmpvar < 3; tmpvar++) {
				sprintf(tmpbuf, "%s/%s", path, av[j]);
				if (stat(tmpbuf, &st) == 0) break;
				if (!tmpvar) {
					strlwr(av[j]);
					continue;
				}
				for (ptr = av[j]; *ptr && *ptr != '-' && *ptr != '_'; ptr++);
				if (*ptr == '-') *ptr = '_';
				else if (*ptr == '_') *ptr = '-';
				else tmpvar = 3;
			}
			if (tmpvar < 3) {
				int rval = isgrpright(GSID_FILE, tmpbuf, GRF_R);
				if (!lncnt) putchr('\n');
				tmpvar = ((int)st.st_size/128+1)*128*11/(process.baud+1);
				tmpvar2 = (int)st.st_size/1024;
				ptr = sec2str(tmpvar);
				if (termflags & (ANSITERM | RIPTERM)) {
					sprintf(tmpbuf, "\033[1;33m%s \033[0;32m(%dkb, %s)", av[j], tmpvar2, ptr);
					strcpy(ftpexecbuf, "\033[1;31m");
				} else {
					sprintf(tmpbuf, "%s (%dkb, %s)", av[j], tmpvar2, ptr);
					ftpexecbuf[0] = '\0';
				}
				if (!rval) {
					LOGIT(LOG_INFO, "DL \"%s\" (%dkb, %s): permission denied", av[j], tmpvar2, ptr);
					putstr("%s %s- %s\n", tmpbuf, ftpexecbuf, sysmsg(MSG_NOPERMISSION));
				} else if (av[j][0] != '@' &&
					   dltime + tmpvar > gettimer()) {
					LOGIT(LOG_INFO, "DL \"%s\" (%dkb, %s): exceed time limit", av[j], tmpvar2, ptr);
					putstr("%s %s- %s\n", tmpbuf, ftpexecbuf, sysmsg(MSG_EXCTIMELIMIT));
				} else if (av[j][0] != '@' && dailydllimit > 0 &&
					   dlsize + dailydlsize + (int)st.st_size > dailydllimit) {
					LOGIT(LOG_INFO, "DL \"%s\" (%dkb, %s): exceed size limit", av[j], tmpvar2, ptr);
					putstr("%s %s- %s\n", tmpbuf, ftpexecbuf, sysmsg(MSG_EXCSIZELIMIT));
				} else if (av[j][0] != '@' && dlulratio > 0 &&
					   userdlsize > dailydllimit &&
					   (dlsize + (int)st.st_size + userdlsize)/userulsize > dlulratio) {
					LOGIT(LOG_INFO, "DL \"%s\" (%dkb, %s): exceed DL/UL ratio", av[j], tmpvar2, ptr);
					putstr("%s %s- %s\n", tmpbuf, ftpexecbuf, sysmsg(MSG_EXCDLULRATIO));
				} else {
					dltime += tmpvar;
					dlsize += (int)st.st_size;
					avf[i++] = av[j];
					if (termflags & (ANSITERM | RIPTERM))
						putstr("\033[0;36m#%d  %s\n", i, tmpbuf);
					else	putstr("#%d  %s\n", i, tmpbuf);
				}
			} else	{
				clearline();
				if (termflags & (ANSITERM | RIPTERM))
					putstr("\033[1;33m%s \033[32m- \033[31m%s\n", av[j], sysmsg(MSG_CANTOPENFILE));
				else	putstr("%s - %s\n", av[j], sysmsg(MSG_CANTOPENFILE));
			}
			lncnt++;
		}
	}
	if (!i) {
		clearline();
		if (!lncnt) warning(MSG_NOSUCHFILE, 0);
		anykey();
		return 0;
	}
	avf[i] = NULL;
	tmpvar = i;
	if ((ptr = getuserconf(PROTOCOL, NULL)) != NULL) {
		for (i = 0; transfers[i].prg != NULL; i++)
			if (transfers[i].dir == 'D' &&
			    !strcasecmp(ptr, transfers[i].prt)) break;
		if (transfers[i].prg == NULL) {
			putuserconf(PROTOCOL, "");
			ptr = NULL;
		}
	}
	if (ptr == NULL) {
		if (termflags & RIPTERM)
			sprintf(tmpbuf, "((*%s::", sysmsg(MSG_SELECTPROTO));
		else	putchr('\n');
		for (i = j = 0; transfers[i].prg; i++) if (transfers[i].dir == 'D') {
			if (tmpvar > 1 && !transfers[i].batch) continue;
			j++;
			if (termflags & ANSITERM)
				putstr("\033[1;33m%2d \033[0;36m%s\n", j, transfers[i].prt);
			else if (termflags & RIPTERM)
				strcatf(tmpbuf, "%d^m@~%d~ %s,", j, j, transfers[i].prt); 
			else	putstr("%2d) %s\n", j, transfers[i].prt);
		}
		if (!j) {
			if (tmpvar > 1)
				warning(MSG_NOBATCHAVAIL, 0);
			else	warning(MSG_NOPROTOAVAIL, 0);
			anykey();
			return 0;
		}
		if (termflags & RIPTERM) {
			strcat(tmpbuf, "0^m@~0~ Quit))");
			rip_query(0, 0, 0, tmpbuf);
			ripio_flush();
			lncnt = atoi(getstr(0, 1, ECHODISABLE));
		} else	lncnt = prompt_num(MSG_SELECTPROTO, 1, j);
		if (lncnt < 1 || lncnt > j) return 1;
		for (i = j = 0; transfers[i].prg; i++) if (transfers[i].dir == 'D') {
			if (tmpvar > 1 && !transfers[i].batch) continue;
			if (++j == lncnt) break;
		}
	}
	tmpvar2 = i;
	sprintf(tmpbuf, SENDBATCH, fntty);
	unlink(tmpbuf);
	unlink(dszlogname);

	if (strstr(transfers[i].prg, "%s") != NULL)
		sprintf(ftpexecbuf, transfers[i].prg, tmpbuf);
	else	if (strstr(transfers[i].prg, "%d") != NULL)
		sprintf(ftpexecbuf, transfers[i].prg, process.baud);
	else	strcpy(ftpexecbuf, transfers[i].prg);

	for (i = 0, ptr = ftpexecbuf;
	     (ptr = strtok(ptr, " \t")) != NULL; ptr = NULL) {
		ave[i++] = ptr;
		if (i == 1) ave[i++] = strippath(ptr);
	}
	lncnt = i;
	for (j = 0; avf[j]; j++) {
		ave[i] = strdupf("%s/%s", path, avf[j]);
		LOGIT(LOG_NOTICE, "DL \"%s\"", ave[i]);
		i++;
	}
	ave[i] = NULL;

	mimefiles(dirpath, path, &ave[lncnt]);

	dszmark[0] = transfers[tmpvar2].mark[0];
	dszmark[1] = transfers[tmpvar2].mark[1];
	setxferproto(transfers[tmpvar2].prt);
	LOGIT(LOG_NOTICE, "%s started for Download", transfers[tmpvar2].prt);
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;37m");
	sprintf(tmpbuf, "%d fls, %d kb, %s, %s at %d", tmpvar, dlsize/1024,
		sec2str(dltime), transfers[tmpvar2].prt, process.baud);
	putchr('\n');
	putstr(sysmsg(MSG_BEGINDOWNLOAD), tmpbuf);
	putchr('\n');
	j = gettimer();
	globalflags |= SENDINPROGRESS;
	process.flags |= INRECVF;
	chdir(workdir);
	i = run(ave[0], &ave[1]);
	while (ave[lncnt]) free(ave[lncnt++]);
	mimeclear();
	if (i != 0) {
		protobyebye();
		LOGIT(LOG_NOTICE, "%s failed", transfers[tmpvar2].prt);
		putstr("\n%s\n", sysmsg(MSG_DLABORTED));
	}
	if (access(dszlogname, F_OK) < 0) {
		if (i == 0) {
			j = j - gettimer();
			j = dlsize/(j+1);
			LOGIT(LOG_NOTICE, "%s complete. %d kb, %d fls, %d cps",
			    transfers[tmpvar2].prt, dlsize/1024, tmpvar, j);
			add_dl_stat(dlsize);
			xferlog_dlm(path, avf, j);
			putstr("\n%s (CPS:%d)\n", sysmsg(MSG_DLCOMPLETE), j);
		} else	chkhangfiles(TRUE);
	} else	parsedszlog(TRUE);
	setxferproto(NULL);
	globalflags &= ~SENDINPROGRESS;
	sprintf(tmpbuf, SENDBATCH, fntty);
	unlink(tmpbuf);
	return i;
}

char *
OpenFileArea()
{
	char *ptr;

	pleasewait(0);
	ptr = getarea();
	if (ptr != NULL && *ptr != '\0') return ptr;
	warning(MSG_SELECTAREAFIRST, 1);
	return NULL;
}

int
SetStrToSearch()
{
	register i, j;
	char *ptr, *ptr2;

	ptr2 = prompt_str(MSG_ENTERSUBSTR, NULL, 32);
	if (ptr2 == NULL) return 0;
	putchr('\n');
	i = 0;
	while ((ptr = strtok(ptr2, " \t")) && i < MAXSEARCHSTRINGS) {
		strncpy(searchstring[i], ptr, MAXSTRINGLENGTH);
		searchstring[i][MAXSTRINGLENGTH] = '\0';
		if (i) {
			for (j = 0; j < i; j++)
				if (!strcmp(searchstring[j], searchstring[i]))
					break;
			if (j == i) i++;
		} else i++;
		ptr2 = NULL;
	}
	searchstring[i][0] = '\0';
	return i;
}

int
kill_file(nfiles, path, filename)
	int *nfiles;
	char *path, *filename;
{
	int ishome;

	ishome = !strncmp(path, homedir, strlen(homedir));
	sprintf(tmpbuf, "%s/%s", ishome ? HOMEDIR : path, filename);
	if (!isgrpright(GSID_FILE, tmpbuf, GRF_D)) {
		nopermission(GRF_D);
		return -1;
	}
	if (ishome) sprintf(tmpbuf, "%s/%s", path, filename);
	if (unlink(tmpbuf) < 0) {
		LOGIT(LOG_ERR, "%s: can't unlink \"%s\": %m", process.name, tmpbuf);
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;33m%s \033[37m%s\n", filename, strerror(errno));
		else	putstr("%s %s\n", filename, strerror(errno));
		anykey();
		return -1;
	}
	if (nfiles != NULL) (*nfiles)++;
	LOGIT(LOG_WARNING, "Kill file \"%s\"", tmpbuf);
	return 0;
}

int
insert_file(slines, path, filename)
	int *slines;
	char *path, *filename;
{
	register char *ptr;
	int lines, ishome;
	FILE *fp;
	FILE *fpw;
	struct stat st;

	ishome = !strncmp(path, homedir, strlen(homedir));
	sprintf(tmpbuf, "%s/%s", ishome ? HOMEDIR : path, filename);
	if (!isgrpright(GSID_FILE, tmpbuf, GRF_R)) {
		nopermission(GRF_R);
		return -1;
	}
	if (ishome) sprintf(tmpbuf, "%s/%s", path, filename);

	if (termflags & (ANSITERM | RIPTERM))
		putstr("\033[1;33m%s \033[37m", filename);
	else	putstr("%s: ", filename);

	if (stat(tmpbuf, &st) < 0) {
		warning(MSG_NOSUCHFILE, 0);
		anykey();
		return -1;
	}
	if (st.st_size > MAXFATTACHSIZE) {
		warning(MSG_FILETOOLONG, 0);
		anykey();
		return -1;
	}
	if ((fp = fopen(tmpbuf, "r")) == NULL) {
		warning(MSG_CANTOPENFILE, 0);
		anykey();
		return -1;
	}
	if (*slines) fpw = fopen(editorworkfile, "a");
	else fpw = fopen(editorworkfile, "w");
	if (fpw == NULL) {
		fclose(fp);
		warning(MSG_CANTOPENFILE, 0);
		anykey();
		return -1;
	}
	lines = 0;
	while (fgets(tmpbuf, sizeof(tmpbuf), fp) != NULL) {
		tmpbuf[sizeof(tmpbuf)-1] = '\0';
		if ((ptr = strpbrk(tmpbuf, "\r\n")) != NULL) *ptr = '\0';
		if (strlen(tmpbuf) > 128) goto binary_file;
		for (ptr = tmpbuf; *ptr; ptr++)
			if ((u_char)*ptr < 0x20 && *ptr != '\t')
				goto binary_file;
		fprintf(fpw, "%s\n", tmpbuf);
		lines++;
	}
done_file:
	fclose(fp);
	fclose(fpw);
	(*slines) += lines;
	putstr(sysmsg(MSG_MSGLINESINSERT), itoa(lines));
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0;37m");
	putchr('\n');
	anykey();
	return 0;

binary_file:
	putstr("%s ", sysmsg(MSG_FILEBEENCODED));
	rewind(fp);
	fprintf(fpw, "\n<-- uuencoded binary file; name %s size %d -->\n\n",
		filename, (int)st.st_size);
	fprintf(fpw, "begin 644 %s\n", filename);
	lines = uuencode(fp, fpw) + 5;
	fprintf(fpw, "end\n");
	goto done_file;
}

int
selectfiles(path, func, pnum)
	char *path;
	int (*func)(), pnum;
{
	int i, n, lncnt, items, rval, fidx[MAXSCRLINES];
	char ch, *ptr, buf[256];
	struct dir_ent *flst;

	items = lncnt = 0;
	while ((n = scan_dir(path, &flst, files_only, SCAN_SORTBYNAME)) > 0) {
		lncnt = 1;
		for (i = 0; i < n; i++) {
			if (lncnt == 1) {
				lncnt += 2;
				memset(fidx, 0, sizeof(fidx));
				fidx[0] = -1;
				snprintf(tmpbuf, MAXSTRINGLENGTH-8, "%s %s",
					 sysmsg(MSG_SELECTFILESHEAD), makeshowpath(path));
				if (termflags & (ANSITERM | RIPTERM))
					putstr("\033[H\033[J\033[1;37;44m %-72.72s %5d\033[K\033[40m\n\n", tmpbuf, n);
				else {
					clearline();
					putstr("\n %s (%4d)\n\n", tmpbuf, n);
				}
			}
			fidx[lncnt-1] = i+1;
			if (flst[i].date > lastcalltime) ch = '*';
			else ch = ' ';
			ptr = ctime(&flst[i].date)+4;
			ptr[20] = '\0';
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[1;33m%4d \033[31m%c\033[0;32m%8d %s \033[36m%-.40s\n",
				       i+1, ch, (int)flst[i].size, ptr, flst[i].name);
			else	putstr("%4d) %c%8d %s %-.40s\n",
				       i+1, ch, (int)flst[i].size, ptr, flst[i].name);
more_again:
			rval = morenum(pnum, &lncnt, n);
			if (rval == -1) {
				if (termflags & RIPMOUSE) {
					free_dir(&flst, n);
					return items;
				}
				break;
			}
			if (rval > 0) {
				if (rval > n) {
					rval = fidx[(rval-n-1)/2];
					if (rval < 0) {
						free_dir(&flst, n);
						return items;
					}
					if (!rval) {
						lncnt = scrlines;
						goto more_again;
					}
				}
				(*func)(&items, path, flst[rval-1].name);
				goto rescan_list;
			}
			if (rval == -2)	i -= (scrlines - 2) * 2;
			else if (rval == -3) i = n - (scrlines - 2) - 1;
			else if (rval == -4) i = -1;
			if (i < -1) i = -1;
		}
		if (termflags & RIPMOUSE) {
			lncnt = scrlines;
			goto more_again;
		}
		if ((ptr = prompt_str(pnum, NULL, 0)) == NULL) {
			free_dir(&flst, n);
			return items;
		}
		clearline();
		for (ptr = strcpy(buf, ptr);
		     (ptr = strtok(ptr, " \t,;")) != NULL; ptr = NULL) {
			i = atoi(ptr);
			if (i > 0 && i <= n) (*func)(&items, path, flst[i-1].name);
			else if (i == -1 || !strcasecmp(ptr, "All")) {
				for (i = 0; i < n; i++)
					(*func)(&items, path, flst[i].name);
				break;
			} else for (i = 0; i < n; i++) {
				if (!fnmatch(ptr, flst[i].name, FNM_NOESCAPE))
					(*func)(&items, path, flst[i].name);
			}
		}
rescan_list:
		free_dir(&flst, n);
	}
	if (!n) {
		if (lncnt) clearscr();
		warning(MSG_NOFILESAVAIL, 1);
	}
	return items;
}

void
KillFile()
{
	char *path;

	if ((path = OpenFileArea()) == NULL) return;
	if (!isgrpright(GSID_FILE, strncmp(path, homedir, strlen(homedir)) ?
			path : HOMEDIR, GRF_D))
		nopermission(GRF_D);
	else	selectfiles(path, kill_file, MSG_SELECTKILLFILE);
}

time_t
getdirmtime(path)
	char *path;
{
	struct stat st;

	if (*path == '/') strcpy(tmpbuf, path);
	else sprintf(tmpbuf, "%s/%s", dirpath, path);
	if (stat(tmpbuf, &st) < 0 || (st.st_mode & S_IFMT) != S_IFDIR ||
	    !(st.st_mode & S_IRUSR) || !(st.st_mode & S_IXUSR)) return 0;
	return st.st_mtime;
}

static void
setupareadir(dlpath, ulpath)
	char *dlpath, *ulpath;
{
	char buf[1024];

	if (*dlpath != '/') {
		sprintf(buf, "%s/%s", dirpath, dlpath);
		dlpath = buf;
	}
	LOGIT(LOG_INFO, "Change FileArea to \"%s\"", setarea(dlpath));
	if (uploadpath != NULL) {
		free(uploadpath);
		uploadpath = NULL;
		putuserconf(UPLOADPATH, "");
	}
	if (ulpath != NULL) {
		uploadpath = strdup(ulpath);
		putuserconf(UPLOADPATH, uploadpath);
		LOGIT(LOG_INFO, "Change Upload to \"%s\"", uploadpath);
	}
	putrcent(FILE_RC, dlpath, itoa(time(NULL)));
}

/*** FUNCTION FileAreaTree ***/
void
FileAreaTree()
{
	int priv, tmpvar;
	char *ptr, *line, *ptr2, dirtop[1024];
	extern char filesdir[];
	extern u_char *strptr;

	sprintf(tmpbuf, "%s/%s", dirpath, FILEAREACONF);
	if ((ptr = loadfile(tmpbuf)) == NULL) return;

	dirtop[0] = '\0';
	freedirtree();
	for (; (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#') continue;
		ptr2 = ptr;
		while ((u_char)*ptr > 0x20) ptr++;
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr != 0; ptr++);
		if (*ptr) {
			if (!parse_access(ptr, &priv, &tmpvar)) continue;
		} else	priv = 0;
		if (dirtop[0] == '\0') {
			if ((ptr = strchr(ptr2, ':')) != NULL) *ptr++ = '\0';
			tmpvar = getdirmtime(ptr2);
			if (priv > privlevel || !tmpvar ||
			    !isgrpright(GSID_FILE, tmpbuf, GRF_S)) {
				warning(MSG_NOAREASAVAILABLE, 1);
				return;
			}
			strcpy(dirtop, tmpbuf);
			if (ptr != NULL) setdirdescfile(ptr);
			continue;
		}
		if (priv > privlevel) adddirmask(ptr2);
	}
	if (dirtop[0] != '\0') {
		ptr2 = getarea();
		ptr = "";
		if (nchr) {
			ptr = strptr;
			while (*ptr == ' ' || *ptr == '\t') ptr++;
		}
		if (*ptr == '/') {
			ptr = getstr(0, 1, ECHODISABLE);
			if ((tmpvar = strlen(ptr)) > 1) {
				strcat(dirtop, ptr);
				if (ptr[tmpvar-1] != '.' &&
				    ptr[tmpvar-1] != '/' &&
				    getdirmtime(dirtop) &&
				    !isdirmasked(dirtop)) {
					freedirtree();
					setupareadir(dirtop, NULL);
					return;
				}
				freedirtree();
				warning(MSG_NOSUCHAREA, 1);
				return;
			}
			ptr2 = dirtop;
		}
		strcpy(filesdir, ptr2);
		if (godirtree(dirtop) > 0) setupareadir(filesdir, NULL);
	}
	freedirtree();
	return;
}

/*** Function %a: Area change ***/
void
FileAreaChange()
{
	int i, n, lncnt, rval, j, fidx[MAXSCRLINES*2];
	char *ptr, *ptr2, *line, ch;
	time_t date;
	struct {
		char *dlp;
		char *ulp;
		char *desc;
		time_t mtime;
	} farea[MAXCONFLINES];

	if (nchr > 5) nchr = 0;
	sprintf(tmpbuf, "%s/%s", dirpath, FILEAREACONF);
	if ((ptr = loadfile(tmpbuf)) == NULL) return;

	pleasewait(1);
	for (n = i = 0; n < MAXCONFLINES &&
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#' || *ptr == '.') continue;
		farea[n].dlp = ptr;
		while ((u_char)*ptr > 0x20 && *ptr != ',') ptr++;
		for (ptr2 = ptr; (u_char)*ptr2 <= 0x20 && *ptr2; ptr2++);
		if (*ptr2 == ',') {
			for (*ptr2++ = 0; (u_char)*ptr2 <= 0x20 && *ptr2; ptr2++);
			ptr = ptr2;
			while ((u_char)*ptr > 0x20) ptr++;
		} else  ptr2 = NULL;
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr; ptr++);
		if (!*ptr || isdigitstr(ptr)) continue;
		farea[n].desc = ptr;
		if ((ptr = strrchr(farea[n].dlp, '.')) != NULL)
			if (isdigitstr(&ptr[1])) {
				if (atoi(&ptr[1]) > privlevel) continue;
				else *ptr = '\0';
			}
		if (!isgrpright(GSID_FILE, farea[n].dlp, GRF_S)) continue;
		if (ptr2 != NULL && (ptr = strrchr(ptr2, '.')) != NULL)
			if (isdigitstr(&ptr[1])) {
				if (atoi(&ptr[1]) > privlevel) ptr2 = NULL;
				else *ptr = '\0';
			}
		if (ptr2 != NULL && !isgrpright(GSID_FILE, ptr2, GRF_S))
			ptr2 = NULL;
		if (ptr2 != NULL && !strcmp(ptr2, HOMEDIR)) ptr2 = homedir;
		farea[n].ulp = ptr2;
		if (!strcmp(farea[n].dlp, HOMEDIR)) farea[n].dlp = homedir;
		if (!(termflags & RIPTERM) && !(n % 10))
			putstr("\b%c", working[++i & 3]);
		if ((farea[n].mtime = getdirmtime(farea[n].dlp)) != 0) n++;
	}
	if (!n) {
		warning(MSG_NOAREASAVAILABLE, 1);
		return;
	}
	if (nchr) {
		rval = prompt_num(0, 0, n);
		if (rval) setupareadir(farea[rval-1].dlp, farea[rval-1].ulp);
		return;
	}
	lncnt = 1;
	for (i = 0; i < n; i++) {
		if (lncnt == 1) {
			lncnt += 2;
			j = 4;
			memset(fidx, 0, sizeof(fidx));
			fidx[0] = -1;
			snprintf(tmpbuf, MAXSTRINGLENGTH-8, "%s %s",
				 sysmsg(MSG_AFILEAREAHEAD), makeshowpath(dirpath));
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[H\033[J\033[1;37;44m %-72.72s %5d\033[K\033[40m\n\n", tmpbuf, n);
			else {
				clearline();
				putstr("\n %s (%4d)\n\n", tmpbuf, n);
			}
		}
		fidx[j++] = i+1;
		ptr = getrcent(FILE_RC, farea[i].dlp);
		if (ptr != NULL)
			date = atoi(ptr);
		else	date = lastcalltime;
		if (farea[i].mtime > date) ch = '*';
		else ch = ' ';
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;33m%4d \033[31m%c \033[0;36m%-32.32s", i+1, ch, farea[i].desc);
		else	putstr("%4d) %c %-31.31s", i+1, ch, farea[i].desc);
		if (!(i & 1)) {
			putchr(' ');
			continue;
		}
		putchr('\n');
more_again:
		rval = morenum(MSG_SELECTFAREA, &lncnt, n);
		if (rval == -1) return;
		if (rval > 0) {
			if (rval > n) {
				rval = fidx[rval-n-1];
				if (rval < 0) return;
				if (!rval) {
					lncnt = scrlines;
					goto more_again;
				}
			}
			setupareadir(farea[rval-1].dlp, farea[rval-1].ulp);
			return;
		}
		if (rval == -2)	i -= (scrlines - 2) * 4;
		else if (rval == -3) i = n - ((scrlines - 2) * 2) - 1;
		else if (rval == -4) i = -1;
		if (i < -1) i = -1;
	}
	if (n & 1) putchr('\n');
	lncnt = scrlines;
	goto more_again;
}

/*** Function %f: List of files ***/
void
FileList()
{
	char *path;

	if ((path = OpenFileArea()) == NULL) return;
	LOGIT(LOG_INFO, "FileList \"%s\"", path);
	if (!filelist(path)) warning(MSG_NOFILESAVAIL, 1);
}

int
filelist(path)
	char *path;
{
	int fd, rval, lncnt = 1, startidx, fidx[MAXSCRLINES];
	char *av[MAXSTACKSIZE+1];
	char ch, color, flen[8], fdate[8], outline[sizeof(tmpbuf)];
	char *ptr, *ptr2, *line;
	struct stat st;
	time_t clknow;
	struct tm tmnow, *tmfile;
	static char *alloctop = NULL;

	if (nchr) {
		startidx = atoi(getstr(0, 1, ECHODISABLE));
		if (startidx < 1) startidx = 1;
	} else startidx = 1;

list_again:
	if (alloctop != NULL) {
		free(alloctop);
		alloctop = NULL;
	}
	memset(fidx, 0, sizeof(fidx));
	if (globalflags & SEARCHPATH) {
		fidx[0] = -1;
		lncnt++;
	}
	if (lncnt < 2) clearscr();

	if ((fd = openlist(path)) < 0) {
		struct dir_ent *flst;
		if ((rval = scan_dir(path, &flst, files_only, 0)) > 0) {
			register i;
			if (rval > MAXSTACKSIZE) rval = MAXSTACKSIZE;
			for (i = fd = 0; i < rval; i++)
				fd += flst[i].len + 1;
			if ((alloctop = ptr = malloc(fd + 2)) == NULL) {
				free_dir(&flst, rval);
				return -1;
			}
			for (i = 0, line = ptr; i < rval; i++) {
				fd = flst[i].len;
				memcpy(line, flst[i].name, fd);
				line[fd] = '\n';
				line += fd + 1;
			}
			*line = '\0';
			free_dir(&flst, rval);
		} else	ptr = NULL;
	} else {
		if (fstat(fd, &st) < 0 || !st.st_size) {
			close(fd);
			return -1;
		}
		if ((alloctop = ptr = malloc((int)st.st_size + 2)) == NULL) {
			close(fd);
			return -1;
		}
		while ((rval = read(fd, ptr, (int)st.st_size)) < 0 && errno == EINTR);
		close(fd);
		if (rval != st.st_size) return -1;
		if (ptr[rval-1] != '\n') ptr[rval++] = '\n';
		ptr[rval] = '\0';
	}

	clknow = time(NULL);
	memcpy(&tmnow, localtime(&clknow), sizeof(struct tm));

	for (fd = 0; ptr != NULL && fd < MAXSTACKSIZE &&
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		if (line > ptr && *(line-1) == '\r') *(line-1) = 0;
		*line++ = 0;
		if ((rval = istruefile(path, ptr)) < 0) continue;
		if (!rval) {
			if (!(globalflags & (SEARCHFILE|NEWFILES)) &&
			    fd+1 >= startidx) {
				if (termflags & (ANSITERM|RIPTERM))
					putstr("\033[1;36m%s\n", ptr);
				else	putstr("%s\n", ptr);
				lncnt++;
			}
			continue;
		}
		(void)strncpy(outline, ptr, sizeof(outline));
		outline[sizeof(outline)-1] = '\0';
		if (globalflags & SEARCHFILE) {
			int i, j;
			for (i = j = 0; searchstring[i][0] != '\0'; i++)
				if ((ptr2 = __strcasestr(outline, searchstring[i])) != NULL) {
					strnupr(ptr2, strlen(searchstring[i]));
					j++;
				}
			if (!j) continue;
		}
		av[fd] = ptr;
		ptr = outline;
		for (ptr2 = av[fd]; (u_char)*ptr > 0x20; ptr++, ptr2++);
		*ptr2 = '\0';
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr != 0; ptr++);
		if (!*ptr) ptr = nodesconfile;
		color = '2';
		ch = ' ';
		if (!(globalflags & LISTMODE_RAW)) {
			sprintf(tmpbuf, "%s/%s", path, av[fd]);
			if ((rval = stat(tmpbuf, &st)) < 0) {
				sprintf(tmpbuf, "%s/%s", path, strlwr(av[fd]));
				if ((rval = stat(tmpbuf, &st)) == 0)
					strlwr(outline);
			}
			if (rval < 0) {
				if (!(globalflags & LISTMODE_ALL) ||
				    (globalflags & NEWFILES)) continue;
				if (termflags & (ANSITERM | RIPTERM)) color = '1';
				strcpy(flen, "off");
				strcpy(fdate, "line");
			} else {
				if ((st.st_mode & S_IFMT) != S_IFREG) continue;
				if ((globalflags & NEWFILES) &&
				    st.st_mtime < newfilestime) continue;
				if ((int)st.st_size >= 1024000) { /* Over 1 Mb */
					rval = (int)st.st_size >> 20;
					if (!rval) strcpy(flen, "0.9M");
					else if (rval < 10)
						sprintf(flen, "%d.%dM", rval, ((int)st.st_size & 0xFFFFF)/100000);
					else	sprintf(flen, "%3dM", rval);
				} else if (st.st_size > 1024) /* Over 1 Kb */
					sprintf(flen, "%3dk", (int)st.st_size/1024);
				else	sprintf(flen, "%4d", (int)st.st_size);
				tmfile = localtime(&st.st_mtime);
				if (tmfile->tm_year < tmnow.tm_year)
					sprintf(fdate, "%s-%02d", month[tmfile->tm_mon], (tmfile->tm_year > 99) ? tmfile->tm_year - 100 : tmfile->tm_year);
				else	sprintf(fdate, "%s %-2d", month[tmfile->tm_mon], tmfile->tm_mday);
				if (st.st_mtime > lastcalltime)	ch = '*';
			}
		}
		if (!fd && (globalflags & SEARCHPATH)) {
			ptr2 = makeshowpath(path);
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[1;37;44m%.66s\033[K\033[0;40m\n", ptr2);
			else	putstr("%s\n", ptr2);
			if (fd) lncnt++;
		}
		fd++;
		if (fd < startidx) continue;

		if (lncnt > scrlines) lncnt = scrlines;
		fidx[lncnt-1] = fd;

		/* Output file number */
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;37m%4d ", fd);
		else	putstr("%4d) ", fd);

		/* Output file name and description */
		if (strlen(outline) > 14) {
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[1;33m%-.72s\n     ", outline);
			else	putstr("%-.72s\n      ", outline);
			lncnt++;
			strcpy(outline, " ");
		}
		if (globalflags & LISTMODE_RAW) {
			if (termflags & (ANSITERM | RIPTERM))
				sprintf(tmpbuf, "\033[1;33m%-14.14s", outline);
			else	sprintf(tmpbuf, "%-14.14s", outline);
		} else {
			if (termflags & (ANSITERM | RIPTERM))
				sprintf(tmpbuf, "\033[1;33m%-14.14s\033[0;3%cm%4.4s %-6.6s\033[1;31m%c ", outline, color, flen, fdate, ch);
			else	sprintf(tmpbuf, "%-14.14s%4.4s %-6.6s%c ", outline, flen, fdate, ch);
		}
		do {
			register n = strlen(ptr);
			if (globalflags & LISTMODE_RAW) {
				if (n > 59) n = 59;
				ptr2 = NULL;
			} else if (n > 46) {
				for (ptr2 = &ptr[45]; ptr2 != ptr &&
				     *ptr2 != '/' && *ptr2 != '-' &&
				     *ptr2 != '.' && *ptr2 != ',' &&
				     (u_char)*ptr2 > 0x20; ptr2--);
				if (ptr2 == ptr) ptr2 = &ptr[45];
				n = ++ptr2 - ptr;
			} else ptr2 = NULL;
			if (termflags & (ANSITERM | RIPTERM))
				putstr("%s\033[0;36m%.*s\n", tmpbuf, n, ptr);
			else	putstr("%s%.*s\n", tmpbuf, n, ptr);
more_again:
			rval = morenum(MSG_SELECTFILE, &lncnt, fd);

			if (rval == -4) {	/* Get back on top list */
				startidx = 1;
				goto list_again;
			}
			if (rval == -2) {	/* Page Up */
				startidx = (fd - scrlines * 2) + 1;
				if (startidx < 1) startidx = 1;
				goto list_again;
			}
			if (rval == -3) {	/* Bottom list */
				if (globalflags & SEARCHPATH) {
					clearscr();
					free(alloctop);
					alloctop = NULL;
					return fd;
				}
				rval = 0;
			}
			if (rval > 0) {	/* Number entered */
				int kp;
				if (rval > fd) {
					rval = fidx[(rval-fd-1)/2];
					if (rval < 0) {
						setupareadir(path, NULL);
						globalflags &= ~SEARCHPATH;
						free(alloctop);
						alloctop = NULL;
						return fd;
					}
					if (!rval) {
						lncnt = scrlines;
						goto more_again;
					}
				}
				--rval;
				if (!strncmp(path, homedir, strlen(homedir)))
					ptr = HOMEDIR;
				else	ptr = path;
				if (isgrpright(GSID_FILE, ptr, GRF_D))
					kp = TRUE;
				else	kp = FALSE;
				if (!kp) ptr = sysmsg(MSG_SELECTEDFILE);
				else ptr = sysmsg(MSG_SELECTEDKFILE);
				if (termflags & RIPTERM) {
					if (!kp) sprintf(tmpbuf, "((*%s::D@~D~ownload,C@~C~ontents,Q@~Q~uit))", av[rval]);
					else sprintf(tmpbuf, "((*%s::D@~D~ownload,C@~C~ontents,K@~K~ill,Q@~Q~uit))", av[rval]);
					rip_query(0, 0, 0, tmpbuf);
					ripio_flush();
				} else if (termflags & ANSITERM)
					putstr("\r\033[1;33m%s \033[36m%s\033[0;37m", av[rval], ptr);
				else	putstr("\r%s - %s", av[rval], ptr);
				n = chrupr(getask(ECHODISABLE));
				clearline();
				if (n == 'Q') {
					lncnt = scrlines;
					goto more_again;
				}
				savescr();
				if (kp && n == 'K') {
					if (kill_file(NULL, path, av[rval]) < 0 &&
					    (termflags & RIPTERM)) {
						restorescr();
						lncnt = scrlines;
						goto more_again;
					}
					startidx = rval - (scrlines - 4) / 2;
					if (startidx < 1) startidx = 1;
					goto list_again;
				}
				if (n == 'C')
					kp = content(path, av[rval]);
				else	kp = download(path, av, fd, rval);
				if (kp >= 0) {
					if (termflags & RIPTERM) {
						restorescr();
						lncnt = scrlines;
						goto more_again;
					}
					startidx = rval - (scrlines - 4) / 2;
					if (startidx < 1) startidx = 1;
					goto list_again;
				}
				rval = -1;
			}
			if (rval < 0) {	/* No more */
				if (globalflags & SEARCHPATH) clearscr();
				free(alloctop);
				alloctop = NULL;
				return fd;
			}
			if (startidx && lncnt < 2) {
				memset(fidx, 0, sizeof(fidx));
				clearscr();
			}
			ptr = ptr2;
			if (termflags & (ANSITERM | RIPTERM))
			strcpy(tmpbuf, "                                ");
			else
			strcpy(tmpbuf, "                                 ");
		} while (ptr != NULL);
	}

	if (fd) {
		putstr("\007<EOF>\r");
		ptr2 = NULL;
		startidx = 0;
		lncnt = scrlines;
		goto more_again;
	}
	if (alloctop != NULL) {
		free(alloctop);
		alloctop = NULL;
	}
	return 0;
}

int
setnewfilestime()
{
	int val, badformat;
	char *ptr, tmbuf[9];
	struct tm *tm;

	newfilestime = lastcalltime;
	tm = localtime(&newfilestime);
	sprintf(tmbuf, "%02d-%02d-%02d", tm->tm_mday, tm->tm_mon+1, (tm->tm_year > 99) ? tm->tm_year-100 : tm->tm_year);
	val = strlen(tmbuf);
	if ((ptr = prompt_str(MSG_ENTERDATE, tmbuf, val+1)) == NULL) return -1;
	badformat = 0;
	strncpy(tmbuf, ptr, sizeof(tmbuf)-1);
	tmbuf[sizeof(tmbuf)-1] = '\0';
	val = atoi(tmbuf);					/* Day */
	if (val >= 1 && val <= 31) {
		if (val > tm->tm_mday) {	/* previous month */
			if (tm->tm_mon - 1 < 0) {	/* previous year */
				if (tm->tm_year - 1 >= 0) {
					tm->tm_mon = 11;
					tm->tm_year--;
				}
			} else tm->tm_mon--;
		}
		tm->tm_mday = val;
		if ((ptr = strchr(tmbuf, '-')) != NULL) {
			val = atoi(++ptr);			/* Month */
			if (val >= 1 && val <= 12) {
				tm->tm_mon = val - 1;
				if ((ptr = strchr(ptr, '-')) != NULL) {
					val = atoi(++ptr);	/* Year */
					if (val >= 0 && val <= 99) {
						tm->tm_year = val;
						if(val < 38) tm->tm_year += 100;
					} else	badformat = MSG_INVALIDYEAR;
				}
			} else	badformat = MSG_INVALIDMONTH;
		}
	} else	badformat = MSG_INVALIDDAY;
	if (badformat) {
		warning(badformat, 1);
		return -1;
	}
	newfilestime = mktime(tm);
	putchr('\n');
	return 0;
}


/*** Function %n: List of new files ***/
void
NewFiles()
{
	int i, n;
	char *ptr, *line, *av[MAXCONFLINES+1], **ap = av;
	char searchpath[MAXPATHLEN];

	if (!(globalflags & SEARCHPATH)) {
		if ((ptr = OpenFileArea()) == NULL) return;
		if (setnewfilestime() < 0) return;
		LOGIT(LOG_INFO, "NewFiles \"%s\"", ptr);
		globalflags |= NEWFILES;
		filelist(ptr);
		globalflags &= ~NEWFILES;
		return;
	}
	sprintf(tmpbuf, "%s/%s", dirpath, FILEAREACONF);
	if ((ptr = loadfile(tmpbuf)) == NULL) return;

	if (setnewfilestime() < 0) return;
	LOGIT(LOG_INFO, "NewFiles browse");

	for (n = 0; n < MAXCONFLINES &&
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#' || *ptr == '.') continue;
		*ap = ptr;
		while ((u_char)*ptr > 0x20 && *ptr != ',') ptr++;
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr; ptr++);
		if (!*ptr) continue;
		if ((ptr = strrchr(*ap, '.')) != NULL)
			if (isdigitstr(&ptr[1])) {
				if (atoi(&ptr[1]) > privlevel) continue;
				else *ptr = '\0';
			}
		if (!isgrpright(GSID_FILE, *ap, GRF_S)) continue;
		if (!strcmp(*ap, HOMEDIR)) *ap = homedir;
		if (getdirmtime(*ap)) ap++, n++;
	}
	if (n) clearscr();
	for (i = 0; i < n && (globalflags & SEARCHPATH); i++) {
		if (*av[i] == '/') strcpy(searchpath, av[i]);
		else sprintf(searchpath, "%s/%s", dirpath, av[i]);
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\r\033[1;37;44m* Area #\033[33m%-4d ", i+1);
		else	putstr("\r* Area #%-4d ", i+1);
		globalflags |= NEWFILES;
		filelist(searchpath);
		globalflags &= ~NEWFILES;
	}
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0;37;40m");
	return;
}

/*** Function %i: Browse List of New files ***/
void
BrowseNewFiles()
{
	globalflags |= SEARCHPATH;
	NewFiles();
	globalflags &= ~SEARCHPATH;
	return;
}

/*** Function %s: Search file ***/
void
SearchFile()
{
	int i, n;
	char *ptr, *line, *av[MAXCONFLINES+1], **ap = av;
	char searchpath[MAXPATHLEN];

	if (globalflags & SEARCHPATH) {
		sprintf(tmpbuf, "%s/%s", dirpath, FILEAREACONF);
		ptr = loadfile(tmpbuf);
	} else	ptr = OpenFileArea();
	if (ptr == NULL) return;

	if (!SetStrToSearch()) return;

	if (!(globalflags & SEARCHPATH)) {
		LOGIT(LOG_INFO, "SearchFile \"%s\"", ptr);
		globalflags |= SEARCHFILE;
		filelist(ptr);
		globalflags &= ~SEARCHFILE;
		return;
	}
	LOGIT(LOG_INFO, "SearchFile browse");

	for (n = 0; n < MAXCONFLINES &&
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#' || *ptr == '.') continue;
		*ap = ptr;
		while ((u_char)*ptr > 0x20 && *ptr != ',') ptr++;
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr; ptr++);
		if (!*ptr) continue;
		if ((ptr = strrchr(*ap, '.')) != NULL)
			if (isdigitstr(&ptr[1])) {
				if (atoi(&ptr[1]) > privlevel) continue;
				else *ptr = '\0';
			}
		if (!isgrpright(GSID_FILE, *ap, GRF_S)) continue;
		if (!strcmp(*ap, HOMEDIR)) *ap = homedir;
		if (getdirmtime(*ap)) ap++, n++;
	}
	if (n) clearscr();
	for (i = 0; i < n && (globalflags & SEARCHPATH); i++) {
		if (*av[i] == '/') strcpy(searchpath, av[i]);
		else sprintf(searchpath, "%s/%s", dirpath, av[i]);
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\r\033[1;37;44m* Area #\033[33m%-4d ", i+1);
		else	putstr("\r* Area #%-4d ", i+1);
		globalflags |= SEARCHFILE;
		filelist(searchpath);
		globalflags &= ~SEARCHFILE;
	}
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0;37;40m");
	return;
}

/*** Function %e: Browse Search file ***/
void
BrowseSearchFile()
{
	globalflags |= SEARCHPATH;
	SearchFile();
	globalflags &= ~SEARCHPATH;
	return;
}

/*** Function %r: Raw files ***/
void
RawFiles()
{
	char *ptr;

	if ((ptr = OpenFileArea()) != NULL) {
		if (display(1, "|%s %s %s", PROG_LS, PROG_LS_ARGS, ptr) == 1)
			anykey();
	}
}

void
down_load(flagnew)
	int flagnew;
{
	int fd, lncnt, rval;
	char *av[MAXSTACKSIZE+1], **ap = av;
	char *ptr, *path, *line;
	struct stat st;
	static char *alloctop = NULL;

	if (alloctop != NULL) {
		free(alloctop);
		alloctop = NULL;
	}
	if ((path = OpenFileArea()) == NULL) return;

	if ((fd = openlist(path)) < 0) {
		struct dir_ent *flst;
		if ((lncnt = scan_dir(path, &flst, files_only, 0)) > 0) {
			register i;
			if (lncnt > MAXSTACKSIZE) lncnt = MAXSTACKSIZE;
			for (i = fd = 0; i < lncnt; i++)
				fd += flst[i].len + 1;
			if ((alloctop = ptr = malloc(fd + 2)) == NULL) {
				free_dir(&flst, lncnt);
				return;
			}
			for (i = 0, line = ptr; i < lncnt; i++) {
				fd = flst[i].len;
				memcpy(line, flst[i].name, fd);
				line[fd] = '\n';
				line += fd + 1;
			}
			*line = '\0';
			free_dir(&flst, lncnt);
		} else  alloctop = ptr = NULL;
	} else {
		if (fstat(fd, &st) < 0 || !st.st_size) {
			close(fd);
			return;
		}
		if ((alloctop = ptr = malloc((int)st.st_size + 2)) == NULL) {
			close(fd);
			return;
		}
		while ((lncnt = read(fd, ptr, (int)st.st_size)) < 0 && errno == EINTR);
		close(fd);
		if (lncnt != st.st_size) return;
		if (ptr[lncnt-1] != '\n') ptr[lncnt++] = '\n';
		ptr[lncnt] = '\0';
	}
	if (flagnew && setnewfilestime() < 0) {
		if (alloctop != NULL) {
			free(alloctop);
			alloctop = NULL;
		}
		return;
	}
	for (lncnt = 0; ptr != NULL && lncnt < MAXSTACKSIZE &&
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		if (line > ptr && *(line-1) == '\r') *(line-1) = 0;
		*line++ = 0;
		if (istruefile(path, ptr) < 1) continue;
		*ap = ptr;
		while ((u_char)*ptr > 0x20) ptr++;
		*ptr = '\0';
		sprintf(tmpbuf, "%s/%s", path, *ap);
		if ((rval = stat(tmpbuf, &st)) < 0 ||
		    (st.st_mode & S_IFMT) != S_IFREG) {
			sprintf(tmpbuf, "%s/%s", path, strlwr(*ap));
			if ((rval = stat(tmpbuf, &st)) == 0)
				if ((st.st_mode & S_IFMT) != S_IFREG)
					rval = -1;
		}
		if (rval == 0) {
			if (!flagnew || st.st_mtime >= newfilestime)
				ap++, lncnt++;
		}
	}
	if (lncnt < 1) warning(MSG_NOFILESAVAIL, 1);
	else download(path, av, lncnt, -1);
	if (alloctop != NULL) {
		free(alloctop);
		alloctop = NULL;
	}
	return;
}

/*** Function %d: Download files ***/

void
DownLoad()
{
	down_load(0);
}

void
DlNewFiles()
{
	down_load(1);
}

int
updatef(fu_name, fu_size, askdesc)
	char *fu_name;
	int fu_size, askdesc;
{
	int i;
	register char *ptr;
	char *ul_path, fname[40], buf[1024];
	struct stat st;
	FILE *fp;

	/* What is the really uploaded file size */
	if (fu_size == 0) return 0;
	strcpy(buf, workdir);
	strcat(buf, "/");
	strcat(buf, fu_name);
	if (stat(buf, &st) < 0) return 0;
	chmod(buf, 0644);
	if (fu_size > 0 && st.st_size != fu_size) {
		unlink(buf);
		return 0;
	}
	fu_size = (int)st.st_size;

	/* Set file access and modification times to current time */
#ifdef	HAVE_UTIME_NULL
	(void)utime(buf, NULL);
#else
	{
		struct utimbuf tm;
		tm.actime = tm.modtime = time(NULL);
		(void)utime(buf, &tm);
	}
#endif
	strncpy(fname, fu_name, sizeof(fname));
	fname[sizeof(fname)-1] = '\0';
	ptr = fixfname(fname, TRUE);
	if (strlen(ptr) < 2) {
		LOGIT(LOG_WARNING, "%s upload \"%s\": bad filename", process.name, fu_name);
		unlink(buf);
		return 0;
	}
	if (strcmp(fu_name, ptr)) {
		chdir(workdir);
		i = link(fu_name, ptr);
		unlink(fu_name);
		chdir(orgdir);
		if (i < 0) {
			LOGIT(LOG_ERR, "%s UL \"%s\", link to \"%s\": %m",
			    process.name, fu_name, ptr);
			return 0;
		}
		strcpy(fu_name, ptr);
		strcpy(buf, workdir);
		strcat(buf, "/");
		strcat(buf, ptr);
	}

	/* If uploaded file is a message */
	if (askdesc < 0) {
		LOGIT(LOG_NOTICE, "UL message \"%s\" (%d bytes)", fu_name, fu_size);
		i = 0;
		insert_file(&i, workdir, fu_name);
		unlink(buf);
		return i;
	}

	LOGIT(LOG_NOTICE, "UL file \"%s\" (%dkb)", fu_name, fu_size/1024);

	/* Prepare upload area path into ul_path */
	if (uploadpath != NULL) ul_path = uploadpath;
	else ul_path = defuploadpath;

	sprintf(tmpbuf, "%s/%s", !strncmp(ul_path, homedir, strlen(homedir)) ? HOMEDIR : ul_path, fu_name);
	if (!isgrpright(GSID_FILE, tmpbuf, GRF_W)) {
		LOGIT(LOG_WARNING, "%s upload \"%s/%s\": restricted filename",
		    process.name, ul_path, fu_name);
		putstr("WRITE \"%s\": %s. Backup...\n", fu_name, sysmsg(MSG_NOPERMISSION));
		ul_path = workdir;
	} else {
		if (mkdir(ul_path, 0755) < 0 && errno != EEXIST) {
			LOGIT(LOG_ERR, "can't mkdir \"%s\": %m", ul_path);
			if (uploadpath != NULL) {
				free(uploadpath);
				uploadpath = NULL;
				putuserconf(UPLOADPATH, "");
			}
			ul_path = workdir;
		}
		if (access(ul_path, W_OK) < 0) {
			LOGIT(LOG_ERR, "can't upload to \"%s\": Permission denied",
			    ul_path);
			ul_path = workdir;
		}
	}

	/* Move if needed uploaded file into upload area */
	if (strcmp(ul_path, workdir)) {
		if (!invoke_syscmd("%s %s %s/%s", PROG_MV, buf, ul_path, fu_name))
			ul_path = workdir;
	}

	if (!askdesc) return fu_size;

	/* Get decription on file from user */
	putstr("%s \"%s\" (max %d lines)\n", sysmsg(MSG_ENTERDESCFILE), fu_name, MAXDESCLINES);
	putstr("   ---------1---------2---------3---------4---------5\n");
	buf[0] = i = 0;
	do {
		putstr("%d: ", ++i);
		ptr = getstr(50, 0, ECHOENABLE);
		putchr('\n');
		if (*ptr != 0) {
			strcat(buf, ptr);
			if (strlen(ptr) < 50) strcat(buf, " ");
		}
	} while (*ptr && i <= MAXDESCLINES);
	putchr('\n');
	for (ptr = &buf[strlen(buf)-1]; ptr >= buf; ptr--) {
		if (!isspace(*ptr)) break;
		*ptr = '\0';
	}

	/* Append description to the desc file */
	sprintf(tmpbuf, "%s/%s", ul_path, DEFFILESLIST);
	if ((fp = fopen(tmpbuf, "a")) == NULL) {
		LOGIT(LOG_ERR, "can't append to \"%s\": %m", tmpbuf);
		return fu_size;
	}
	flock(fileno(fp), LOCK_EX);
	if (commentul != NULL) {
		strcpy(tmpbuf, commentul);
		(void)insvariables(tmpbuf);
		fprintf(fp, "%s\n", tmpbuf);
	}
	fprintf(fp, "%s\t%s\n", fu_name, buf);
	fflush(fp);
	flock(fileno(fp), LOCK_UN);
	fclose(fp);
	return fu_size;
}

/*** Function %u: Upload files ***/
void
UpLoad()
{
	int ishome;
	char *ul_path;

	if (uploadpath != NULL) ul_path = uploadpath;
	else ul_path = defuploadpath;

	ishome = !strncmp(ul_path, homedir, strlen(homedir));
	if (isgrpright(GSID_FILE, ishome ? HOMEDIR : ul_path, GRF_W))
		up_load(ishome == FALSE);
	else nopermission(GRF_W);
}

int
up_load(askdesc)
	int askdesc;
{
	int i, j, lncnt, rval;
	char *ptr, *ptr2;
	sigfunc intsave;
	char ftpexecbuf[1024], *ave[MAXCONFLINES+1];
	int avs[MAXCONFLINES+1];

	if (transfers[0].prg == NULL) return 0;

	if ((ptr = getuserconf(PROTOCOL, NULL)) != NULL) {
		for (i = 0; transfers[i].prg != NULL; i++)
			if (transfers[i].dir == 'U' &&
			    !strcasecmp(ptr, transfers[i].prt)) break;
		if (transfers[i].prg == NULL) {
			putuserconf(PROTOCOL, "");
			ptr = NULL;
		}
	}
	if (ptr == NULL) {
		if (termflags & RIPTERM)
			sprintf(tmpbuf, "((*%s::", sysmsg(MSG_SELECTPROTO));
		else	putchr('\n');
		for (i = j = 0; transfers[i].prg; i++) if (transfers[i].dir == 'U') {
			j++;
			if (termflags & ANSITERM)
				putstr("\033[1;33m%2d \033[0;36m%s\n", j, transfers[i].prt);
			else if (termflags & RIPTERM)
				strcatf(tmpbuf, "%d^m@~%d~ %s,", j, j, transfers[i].prt); 
			else	putstr("%2d) %s\n", j, transfers[i].prt);
		}
		if (!j) {
			warning(MSG_NOPROTOAVAIL, 1);
			return 0;
		}
		if (termflags & RIPTERM) {
			strcat(tmpbuf, "0^m@~0~ Quit))");
			rip_query(0, 0, 0, tmpbuf);
			ripio_flush();
			lncnt = atoi(getstr(0, 1, ECHODISABLE));
		} else	lncnt = prompt_num(MSG_SELECTPROTO, 1, j);
		if (lncnt < 1 || lncnt > j) return 0;
		for (i = j = 0; transfers[i].prg; i++)
			if (transfers[i].dir == 'U' && ++j == lncnt) break;
	}
	sprintf(tmpbuf, RECVBATCH, fntty);
	unlink(tmpbuf);
	unlink(dszlogname);

	if (strstr(transfers[i].prg, "%s") != NULL)
		sprintf(ftpexecbuf, transfers[i].prg, tmpbuf);
	else	if (strstr(transfers[i].prg, "%d") != NULL)
		sprintf(ftpexecbuf, transfers[i].prg, process.baud);
	else	strcpy(ftpexecbuf, transfers[i].prg);

	ptr2 = ftpexecbuf;
	j = 0;
	while ((ptr = strtok(ptr2, " \t")) != NULL) {
		ave[j++] = ptr;
		if (j == 1) ave[j++] = strippath(ptr);
		ptr2 = NULL;
	}
	if (!transfers[i].batch) {
		if ((ptr = prompt_str(MSG_ENTERULNAME, NULL, MAXFILENAMELEN)) == NULL)
			return 0;
		if (!legalfname(ptr)) return 0;
		ave[j++] = ptr;
	}
	ave[j] = NULL;

	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;37m");
	ptr2 = transfers[i].prt;
	sprintf(tmpbuf, "%s at %d", ptr2, process.baud);
	putchr('\n');
	putstr(sysmsg(MSG_BEGINUPLOAD), tmpbuf);
	putchr('\n');
	LOGIT(LOG_NOTICE, "%s started for Upload", ptr2);
	dszmark[0] = transfers[i].mark[0];
	dszmark[1] = transfers[i].mark[1];
	setxferproto(ptr2);
	lncnt = gettimer();
	globalflags |= RECVINPROGRESS;
	process.flags |= INSENDF;
	chdir(workdir);
	rval = run(ave[0], &ave[1]);
	if (rval != 0) {
		protobyebye();
		LOGIT(LOG_NOTICE, "%s failed", ptr2);
		putstr("\n%s\n", sysmsg(MSG_ULFAILED));
	}
	if (access(dszlogname, F_OK) < 0) {
		if (rval == 0) {
			int ulsize = 0, needfree = 0;
			i = gettimer();
			j = lncnt-i;
			if (j < 60) j = 60;
			i += j;
			settimer(i/60);
			lncnt = 0;
			intsave = signal(SIGINT, SIG_IGN);
			if (ptr != NULL) {
				ulsize = updatef(strcpy(ftpexecbuf, ptr), -1, askdesc);
				ave[lncnt] = ftpexecbuf;
				avs[lncnt] = ulsize;
				lncnt++;
			} else {
				FILE *fp;

				sprintf(ftpexecbuf, RECVBATCH, fntty);
				if ((fp = sfopen(ftpexecbuf, "r")) != NULL) {
					while (fgets(ftpexecbuf, sizeof(ftpexecbuf), fp) != NULL) {
						ftpexecbuf[sizeof(ftpexecbuf)-1] = '\0';
						if ((ptr = strchr(ftpexecbuf, '\n')) != NULL)
							*ptr = '\0';
						if ((ptr = strchr(ftpexecbuf, ' ')) != NULL) {
							int size;
							*ptr++ = '\0';
							size = updatef(ftpexecbuf, atoi(ptr), askdesc);
							if (size) {
								needfree = 1;
								if ((ave[lncnt] = strdup(ftpexecbuf)) == NULL)
									return 0;
								avs[lncnt] = size;
								lncnt++;
								ulsize += size;
							}
						}
					}
					sfclose(fp);
				}
			}
			ave[lncnt] = NULL;
			if (askdesc >= 0 && ulsize)
				xferlog_ulm(ave, avs, ulsize/j);
			if (needfree) while(lncnt) free(ave[--lncnt]);
			(void)signal(SIGINT, intsave);
			putchr('\n');
			psysmsg(MSG_ULCOMPLETE);
			if (!ulsize) {
				putstr(". %s\n", sysmsg(MSG_CANTFINDUL));
				rval = 1;
			} else if (askdesc > 0) {
				LOGIT(LOG_NOTICE, "%s complete. %d kb, %d cps",
				    ptr2, ulsize/1024, ulsize/j);
				add_ul_stat(ulsize);
				putstr(" (CPS:%d)\n%s (%dkb)\n", ulsize/j,
				       sysmsg(MSG_ULSIZEADDED), ulsize/1024);
				putstr("%s (%s)\n", sysmsg(MSG_ULTIMEADDED),
				       sec2str(i));
			}
		} else	if (askdesc > 0) chkhangfiles(askdesc);
	} else	rval = !parsedszlog(askdesc);
	setxferproto(NULL);
	globalflags &= ~RECVINPROGRESS;
	sprintf(tmpbuf, RECVBATCH, fntty);
	unlink(tmpbuf);
	return (rval == 0);
}

/*** Function %g: Good bye ***/
void
GoodBye()
{
	if (!prompt_yn(MSG_CONFIRMBYE, NULL, 'y')) return;
	ftp_disconnect();
	if (!prompt_yn(MSG_LEAVEMESSAGE, sysadmname, 'n'))
		mail_to_sysop();
	if (!display(0, GOODBYE)) {
		putchr('\n');
		putstr(sysmsg(MSG_GOODBYE), process.name);
		putchr('\n');
	}
	quit(0);
}

/*** Setup Function: Language ***/
void
Language()
{
	extern char *langname;

	if (language(NULL) > 0 && !(globalflags & IMMQUIT))
		putuserconf(LANGUAGE, langname);
}

/*** Setup Function: Protocol ***/
void
Protocol()
{
	register i, j;
	int lncnt;

	if (transfers[0].prg == NULL) return;

	if (termflags & RIPTERM)
		sprintf(tmpbuf, "((*%s::", sysmsg(MSG_SELECTPROTO));
	else	putchr('\n');

	for (i = j = 0; transfers[i].prg; i++) if (transfers[i].dir == 'D') {
		j++;
		if (termflags & ANSITERM)
			putstr("\033[1;33m%2d \033[0;36m%s\n", j, transfers[i].prt);
		else if (termflags & RIPTERM)
			strcatf(tmpbuf, "%d^m@~%d~ %s,", j, j, transfers[i].prt);
		else	putstr("%2d) %s\n", j, transfers[i].prt);
	}
	if (!j) {
		warning(MSG_NOPROTOAVAIL, 1);
		return;
	}
	if (termflags & RIPTERM) {
		strcat(tmpbuf, "0^m@~0~ Quit))");
		rip_query(0, 0, 0, tmpbuf);
		ripio_flush();
		lncnt = atoi(getstr(0, 1, ECHODISABLE));
	} else	lncnt = prompt_num(MSG_SELECTPROTO, 1, j);
	if (lncnt < 1 || lncnt > j) {
		putuserconf(PROTOCOL, "");
		return;
	}
	for (i = j = 0; transfers[i].prg; i++)
		if (transfers[i].dir == 'D' && ++j == lncnt) break;
	putuserconf(PROTOCOL, transfers[i].prt);
	return;
}

/*** Setup Function: Realname ***/
void
Realname()
{
	usersetup(MSG_ENTERREALNAME, REALNAME, 32);
}

/*** Setup Function: UserLocation ***/
void
UserLocation()
{
	usersetup(MSG_ENTERUSERLOCATION, USERLOCATION, 25);
}

/*** Setup Function: Organization ***/
void
Organization()
{
	usersetup(MSG_ENTERORGANIZATION, USERORG, 25);
}

/*** Setup Function: Phone Number ***/
void
Phonenumber()
{
	usersetup(MSG_ENTERPHONE, "PHONE", 15);
}

/*** Setup Function: Signature ***/
void
Signature()
{
	usersetup(MSG_ENTERSIGNATURE, SIGNATURE, 0);
}

/*** Setup Function: NewsSort String ***/
void
NewsSort()
{
	char *ptr;

	ptr = usersetup(MSG_ENTERNEWSSORT, NEWSSORT, 4);
	if (newssort != NULL) free(newssort);
	if (ptr == NULL) newssort = NULL;
	else newssort = strdup(ptr);
}

/*** Setup Function: ChPassword ***/
void
ChPassword()
{
	char *ptr;
	extern char *assignpasswd();

	ptr = assignpasswd(1);
	if (ptr == NULL) {
		warning(MSG_PASSWDUNCHANGED, 0);
		anykey();
	} else	putuserconf(PASSWORD, ptr);
	putchr('\n');
}

/*** Function ScrLines ***/
void
Scr_Lines()
{
	int num;

	num = prompt_num(MSG_ENTERSCRLINES, scrlines+1, MAXSCRLINES);
	if (num >= 20) {
		scrlines = num - 1;
		putuserconf(SCRLINES, itoa(num));
	}
}

/*** Function HotKey ***/
void
Hot_Key()
{
	char *ptr;
	if (prompt_yn(MSG_ASKHOTKEY, NULL, 'N')) {
		termflags &= ~HOTKEYON;
		ptr = "No";
	} else {
		termflags |= HOTKEYON;
		ptr = "Yes";
	}
	putuserconf(HOTKEY, ptr);
}

/*** Function ShortMenu ***/
void
Short_Menu()
{
	char *ptr;
	if (prompt_yn(MSG_ASKSHORTMENU, NULL, 'N')) {
		termflags &= ~SHORTMENUON;
		ptr = "No";
	} else {
		termflags |= SHORTMENUON;
		ptr = "Yes";
	}
	putuserconf(SHORTMENU, ptr);
}

/*** Function Mesg ***/
void
Change_Mesg()
{
	char *state;

	if (!prompt_yn(MSG_ASKMESGSTATE, NULL, 'N')) state = "No";
	else state = "Yes";
	putuserconf("MESG", mesgtty(state));
	saveinfo(&process);
}

/*** Function PressAnyKey ***/
void
PressAnyKey()
{
	anykey();
}

/*** Do not show internal default menu ***/
void
ExternalMenu()
{
	globalflags |= EXTERNALMENUFLAG;
}

void
ExternalTalk()
{
	globalflags |= EXTERNALTALK;
}

/*** More prompt off ***/
void
MoreOff()
{
	globalflags |= NOMOREFLAG;
}

/*** More prompt on ***/
void
MoreOn()
{
	globalflags &= ~NOMOREFLAG;
}

void
unknowncmd()
{
	return;
}

add_dl_stat(dlsize)
	int dlsize;
{
	if (dlsize > 0) {
		dailydlsize += dlsize;
		putuserconf(DAILYDLSIZE, itoa(dailydlsize/1024));
		userdlsize += dlsize;
		putuserconf(USERDLSIZE, itoa(userdlsize/1024));
	}
}

add_ul_stat(ulsize)
	int ulsize;
{
	if (ulsize > 0) {
		dailydlsize -= ulsize * ulmultiplier;
		if (dailydlsize < 0) dailydlsize = 0;
		putuserconf(DAILYDLSIZE, itoa(dailydlsize/1024));
		dailyulsize += ulsize;
		putuserconf(DAILYULSIZE, itoa(dailyulsize/1024));
		userulsize += ulsize;
		putuserconf(USERULSIZE, itoa(userulsize/1024));
	}
}

chkhangfiles(askdesc)
	int askdesc;
{

	FILE *fp;
	char ftplogf[256], buf[256], *ptr;
	int restsize, nb;

	mimeclear();
	if (globalflags & RECVINPROGRESS) {
		sprintf(ftplogf, RECVBATCH, fntty);
		if ((fp = sfopen(ftplogf, "r")) != NULL) {
			restsize = 0;
			while (fgets(buf, sizeof(buf), fp) != NULL) {
				buf[sizeof(buf)-1] = '\0';
				if ((ptr = strchr(buf, '\n')) != NULL)
					*ptr = '\0';
				if ((ptr = strchr(buf, ' ')) != NULL) {
					*ptr++ = '\0';
					nb = updatef(buf, atoi(ptr), askdesc);
					if (nb) {
						restsize += nb;
						xferlog_ul(buf, nb, 0);
					}
				}
			}
			sfclose(fp);
			unlink(ftplogf);
			add_ul_stat(restsize);
		}
	}
	if (globalflags & SENDINPROGRESS) {
		sprintf(ftplogf, SENDBATCH, fntty);
		if ((fp = sfopen(ftplogf, "r")) != NULL) {
			restsize = 0;
			while (fgets(buf, sizeof(buf), fp) != NULL) {
				buf[sizeof(buf)-1] = '\0';
				if ((ptr = strchr(buf, '\n')) != NULL)
					*ptr = '\0';
				if ((ptr = strchr(buf, ' ')) != NULL) {
					*ptr++ = '\0';
					nb = atoi(ptr);
					if (nb > 0) {
						restsize += nb;
						LOGIT(LOG_NOTICE, "DL \"%s\" (%dkb)", buf, nb/1024);
						xferlog_dl(buf, nb, 0);
						if (askdesc)
							putstr("%s (%dkb) - Downloaded\n",
							       strippath(buf), nb/1024);
					}
				}
			}
			sfclose(fp);
			unlink(ftplogf);
			add_dl_stat(restsize);
		}
	}
}

int
parsedszlog(askdesc)
	int askdesc;
{
	FILE *fp;
	int lncnt, dlsize, ulsize;
	char res, *ptr, buf[1024], fname[256];
	int byts, bps, cps, errs, flow, last, sn;

	if (dszlogname == NULL || access(dszlogname, F_OK) < 0) return 0;
	if (!dszmark[0] || !dszmark[1]) {
		LOGIT(LOG_ERR, "find \"%s\", but NO char markers, check your \"%s\"!",
		    dszlogname, FTPCONF);
		return -1;
	}
	if ((fp = fopen(dszlogname, "r")) == NULL) {
		LOGIT(LOG_ERR, "can't open \"%s\": %m", dszlogname);
		return -1;
	}
	lncnt = dlsize = ulsize = 0;
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		buf[sizeof(buf)-1] = '\0';
		if ((ptr = strchr(buf, '\n')) != NULL) *ptr = '\0';
		lncnt++;
		if (sscanf(buf, "%c %d %d bps %d cps %d errors %d %d %s %d",
			   &res, &byts, &bps, &cps, &flow, &errs, &last,
			   fname, &sn) != 9) {
			badline(dszlogname, buf);
			continue;
		}
		if (res == dszmark[0]) {	/* Snd */
			dlsize += byts;
			LOGIT(LOG_NOTICE, "DL \"%s\" (%dkb, %dcps, %derrs)",
			    fname, byts/1024, cps, errs);
			xferlog_dl(fname, byts, cps);
			if (askdesc)
				putstr("%s (%dkb, CPS:%d) - Downloaded\n",
				       strippath(fname), byts/1024, cps);
		} else	if (res == dszmark[1]) {	/* Rcv */
			int nb = updatef(fname, byts, askdesc);
			if (nb) {
				xferlog_ul(fname, nb, cps);
				ulsize += nb;
			}
		} else	{			/* Unk */
			LOGIT(LOG_ERR, "\"%s\": unknown marker '%c'", dszlogname, res);
		}
	}
	fclose(fp);
	unlink(dszlogname);
	add_dl_stat(dlsize);
	add_ul_stat(ulsize);
	return ulsize;
}

protobyebye()
{
	register i;
	for (i = 0; i < 12; i++) putchr('\030');
}

int
istruefile(fareadir, filename)
	char *fareadir, *filename;
{
	register char *p = filename;
	struct stat st;
	char fpath[1024];

	if (*p == '#' || *p == '.' || *p == ';') return -1;
	if ((u_char)*p <= 0x20 || *p == '-') return 0;
	while ((u_char)*p > 0x20 && *p != '.') p++;
	if (p[0] == '.') return (p[1] > 0x20 && p[1] != '.');
	sprintf(fpath, "%s/%.*s", fareadir, p - filename, filename);
	if (stat(fpath, &st) < 0) return 0;
	if ((st.st_mode & S_IFMT) == S_IFREG) return 1;
	return -1;
}

int
openlist(path)
	char *path;
{
	register fd, i;
	struct stat st;
	char *ptr, buf[1024];
	extern char *stdfile;

	for (i = 0; fileslist[i].mask != NULL; i++)
		if (rematch(fileslist[i].mask, path)) {
			ptr = fileslist[i].name;
			if (*ptr != '/') {
				sprintf(buf, "%s/%s", path, ptr);
				ptr = buf;
			}
			if (stat(ptr, &st) == 0) break;
		}

	globalflags &= ~(LISTMODE_RAW | LISTMODE_ALL);
	if (fileslist[i].mask == NULL) {
		sprintf(buf, "%s/%s", path, DEFFILESLIST);
		return open(buf, O_RDONLY);
	}
	if (fileslist[i].mode & LISTMODE_EXT) {
		char temp[128];
		if (!(st.st_mode & S_IXUSR)) return -1;
		stdfile = mktemp(strcpy(temp, TMPFILEMASK));
		chdir(path);
		if (!invoke_syscmd(ptr)) {
			chdir(orgdir);
			return -1;
		}
		fd = open(temp, O_RDONLY);
		unlink(temp);
	} else	fd = open(ptr, O_RDONLY);
	globalflags |= (fileslist[i].mode & (LISTMODE_RAW | LISTMODE_ALL));
	return fd;
}

/* mail.c */
extern void MailCheck();
extern void MsgAreaChange();
extern void MsgList();
extern void MsgListNew();
extern void MsgReader();
extern void MsgSearch();
extern void MsgToFile();
extern void MsgKill();
extern void MsgEnter();
extern void MsgForYou();
void BrowseMsgForYou(){ }	/* fake */
void BrowseMsgSearch(){ }	/* fake */

/* usenet.c */
extern void UsenetAreaTree();
extern void UsenetAreaChange();
extern void UsenetArtList();
extern void UsenetReader();
extern void UsenetPost();
extern void UsenetAuthorSearch();
extern void UsenetSubjectSearch();

/* userlist.c */
extern void who_is_there();
extern void user_list();
extern void who_is_who();
extern void callsign_list();
extern void ban_list();

/* login.c */
extern void make_callsign();

/* ipc.c */
extern void who_in_conf();
extern void who_do();
extern void go_conf();
extern void go_talk();
extern void go_unixchat();
extern void go_finger();
extern void go_nettalk();

/* readfiles.c */
extern void readfiles();

/* menu.c */
extern void ripsave();
extern void riprestore();

/* ftp.c */
extern void FTP_open();
extern void FTP_select();
extern void FTP_list();
extern void FTP_raw();
extern void FTP_pwd();
extern void FTP_stat();
extern void FTP_cdup();
extern void FTP_cwd();
extern void FTP_get();
extern void FTP_new();

static struct cmdfunc {
	void (*f)();
	char *key;
} cmdfuncs[] = {
	{	ExternalMenu,		"ExternalMenu"	},
	{	ExternalTalk,		"ExternalTalk"	},
	{	MoreOff,		"MoreOff"	},
	{	MoreOn,			"MoreOn"	},
	{	ripsave,		"RIPSave"	},
	{	riprestore,		"RIPRestore"	},
	{	FileAreaChange,		"FAreaChange"	},
	{	FileAreaTree,		"FAreaTree"	},
	{	SearchFile,		"ViewContent"	}, /* Compatibility */
	{	BrowseSearchFile,	"BViewContent"	}, /* Compatibility */
	{	FileList,		"FileList"	},
	{	DownLoad,		"DownLoad"	},
	{	DlNewFiles,		"DlNewFiles"	},
	{	UpLoad,			"UpLoad"	},
	{	GoodBye,		"GoodBye"	},
	{	NewFiles,		"NewFiles"	},
	{	BrowseNewFiles,		"BNewFiles"	},
	{	RawFiles,		"RawList"	},
	{	SearchFile,		"SearchStr"	},
	{	BrowseSearchFile,	"BSearchStr"	},
	{	KillFile,		"KillFiles"	},
	{	Language,		"Language"	},
	{	Protocol,		"Protocol"	},
	{	Realname,		"Realname"	},
	{	UserLocation,		"UserLocation"	},
	{	Organization,		"Organization"	},
	{	Phonenumber,		"Phone"		},
	{	Signature,		"Signature"	},
	{	NewsSort,		"NewsSort"	},
	{	ChPassword,		"ChPassword"	},
	{	Scr_Lines,		"ScrLines"	},
	{	Hot_Key,		"HotKey"	},
	{	Short_Menu,		"ShortMenu"	},
	{	Change_Mesg,		"Mesg"		},
	{	PressAnyKey,		"PressAnyKey"	},
	{	MailCheck,		"MailCheck"	},
	{	MsgAreaChange,		"MAreaChange"	},
	{	MsgReader,		"ReadMsg"	},
	{	MsgList,		"ListMsg"	},
	{	MsgListNew,		"ListNewMsg"	},
	{	MsgToFile,		"SaveMsg"	},
	{	MsgKill,		"KillMsg"	},
	{	MsgEnter,		"EnterMsg"	},
	{	MsgSearch,		"SearchMsg"	},
	{	MsgForYou,		"YourMsg"	},	/* fake */
	{	BrowseMsgForYou,	"BYourMsg"	},	/* fake */
	{	BrowseMsgSearch,	"BSearchMsg"	},	/* fake */
	{	UsenetAreaTree,		"UAreaTree"	},
	{	UsenetAreaChange,	"UAreaChange"	},
	{	UsenetArtList,		"ListArticles"	},
	{	UsenetReader,		"UsenetReader"	},
	{	UsenetPost,		"PostArticle"	},
	{	UsenetAuthorSearch,	"AuthorSearch"	},
	{	UsenetSubjectSearch,	"SubjectSearch"	},
	{	who_is_there,		"WhoIsThere"	},
	{	user_list,		"UserList"	},
	{	who_is_who,		"WhoIsWho"	},
	{	who_in_conf,		"WhoInConf"	},
	{	who_do,			"WhoDo"		},
	{	go_conf,		"GoInConf"	},
	{	go_talk,		"GoInTalk"	},
	{	go_unixchat,		"GoUnixChat"	},
	{	callsign_list,		"CallSignList"	},
	{	ban_list,		"BanList"	},
	{	make_callsign,		"MakeCallSign"	},
	{	go_finger,		"Finger"	},
	{	go_nettalk,		"Talk"		},
	{	readfiles,		"ReadFiles"	},
	{	quit,			"Quit"		},
	{	FTP_open,		"FTPopen"	},
	{	FTP_select,		"FTPselect"	},
	{	FTP_list,		"FTPlist"	},
	{	FTP_raw,		"FTPraw"	},
	{	FTP_stat,		"FTPstat"	},
	{	FTP_pwd,		"FTPpwd"	},
	{	FTP_cdup,		"FTPcdup"	},
	{	FTP_cwd,		"FTPcwd"	},
	{	FTP_get,		"FTPget"	},
	{	FTP_new,		"FTPnew"	},
	{	mail_to_sysop,		"MailToSysop"	},
	{ 0, 0 },
};

void
(*do_cmd(key))()
	char *key;
{
	struct cmdfunc *p;

	for (p = cmdfuncs; p->f; ++p)
		if (!strcasecmp(key, p->key)) {
			proctitle(p->key);
			LOGIT(LOG_DEBUG, "do_cmd: \"%s\"", p->key);
			return p->f;
		}
	LOGIT(LOG_ERR, "do_cmd: \"%s\": Unknown command", key);
	return unknowncmd;
}
