#include <curses.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/fcntl.h>
#include <pwd.h>
#include <string.h>
#include <signal.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <stdarg.h>

#include "getconf.h"
#include "admin.h"

#define	ADM	"TNSAdmin v1.3"

static char *copy  = "Copyright (c) 1994-1997 The CAD lab of the";
static char *right = "Siberian State Academy of Telecommunication, RU";

extern int errno;
void onterm(), onalrm();

char username[MAXUSRNAMELEN+1];
char *orgdir;
char mytty[10];

struct usrent **usrlist;
int nusers;
int usridx, first_line, scr_line, showmode;
int pagesize;
int markflag;
char header[81];

int Pflag = 0, lflag = -1, yflag = 0;

main(argc, argv)
	int argc;
	char **argv;
{
	char buf[256];
	int op;
	void usage();
	extern char *usrdir;
	extern char *optarg;
	extern int optind, opterr;

	opterr = 0;
	while ((op = getopt(argc, argv, "P:l:y")) != EOF)
		switch (op) {
		case 'y':
			++yflag;
			break;
		case 'l':
			if (!isdigit(*optarg)) usage();
			lflag = atoi(optarg);
			break;
		case 'P':
			if ((Pflag = atoi(optarg)) != 0) break;
		case 'h':
		case '?':
		default:
			usage();
		}

	if (!Pflag && (yflag || lflag >= 0)) usage();

	if (argv[optind] != NULL) {
		orgdir = argv[optind];
		usrdir = argv[optind];
	} else {
		findorgdir();
		findusrdir();
	}
	initcallsign(orgdir);
	initgetconf(orgdir);

	if (Pflag) {
		if (getuid() != 0) {
			fprintf(stderr, "Only root can use this option\n");
			exit(1);
		}
		if (seteuid(0) < 0) {
			fprintf(stderr, "Can't set the effective user ID to root\n");
			exit(1);
		}
		purge_users(Pflag, lflag);
		exit(0);
	}

	printf("Scaning userdir in %s... ", usrdir);
	fflush(stdout);
	if ((nusers = scanusers(&usrlist)) < 0) {
		printf("Failed!\n");
		exit(1);
	}
	if (!nusers) {
		printf("No users!\n");
		exit(1);
	}
	printf("\b\b\b\b\b%d users found\n", nusers);
	fflush(stdout);
	sleep(1);
	printf("Initializing... ");
	fflush(stdout);
	if (initselfinfo() < 0) exit(1);

	if (gethostname(buf, 40) < 0) strcpy(buf, "localhost");
	snprintf(header, sizeof(header)-1, "%s@%s (%s), %s",
		       getlogin(), buf, username, mytty);
	initterm();

	signal(SIGHUP, onterm);
	signal(SIGINT, onterm);
	signal(SIGQUIT, onterm);
	signal(SIGTERM, onterm);
	signal(SIGTSTP, SIG_IGN);
	signal(SIGALRM, onalrm);

	mainloop();

	onterm();
}

initterm()
{
/*	puts(NICECOLOR); */
	COLS = 80;
	if (initscr() == NULL) exit(1);
	puts(NICECOLOR);
	if (LINES < 20) {
		addstr("Must more LINES on term");
		panic(1);
	}
	if (COLS < 80) {
		addstr("Must more COLS on term");
		panic(1);
	}
	pagesize = LINES - MINPAGESIZE;

	cbreak();
	noecho();
	nonl();
/*	leaveok(stdscr, FALSE);
	scrollok(stdscr, FALSE);
*/
	mvaddstr(0, COLS/2 - strlen(header)/2, header);
	mvaddstr(1, 0, "\
    #    User Name                Stat Priv Calls Timeleft  Lastcall");
	mvaddstr(2, 0, "\
================================================================================");

}

void
onterm()
{
	signal(SIGINT, SIG_IGN);
	mvcur(0, COLS - 1, LINES - 1, 0);
	endwin();
	printf(DEFCOLOR);
	exit(0);
}

void
onalrm()
{
	return;
}

panic(rval)
	int rval;
{
	refresh();
	signal(SIGINT, SIG_IGN);
	mvcur(0, COLS - 1, LINES - 1, 0);
	endwin();
	printf(DEFCOLOR);
	exit(rval);
}

findorgdir()
{
	struct passwd *pwd;

	if ((pwd = getpwnam("bbs")) == NULL)
		if ((pwd = getpwnam("tns")) == NULL) {
			fprintf(stderr, "No BBS in this system\n");
			exit(1);
		}
	orgdir = strdup(pwd->pw_dir);
}

initselfinfo()
{
	int i;
	char *ptr, mylogname[25];
	struct passwd *pwd;

	if (!isatty(0))	{
		printf("No control terminal\n");
		return -1;
	}
	if ((ptr = getlogin()) == NULL) {
		printf("Can't find your login name\n");
		return -1;
	}
	if ((pwd = getpwnam(ptr)) == NULL) {
		printf("Can't find passwd entry for `%s'\n", ptr);
		return -1;
	}
	strcpy(mylogname, pwd->pw_name);
	i = 0;
	if (pwd->pw_gecos != NULL)
		for (ptr = pwd->pw_gecos; *ptr && *ptr != ',' &&
		     i < sizeof(username)-1; i++, ptr++) username[i] = *ptr;
	if (!i) {
		printf("Can't find your real name\n");
		return -1;
	}
	username[i] = '\0';
	strcpy(mytty, ttyname(0) + 5);
	return 0;
}

showlist(first)
	int first;
{
	register i, l, y;

	move(Y_FIRST, X_FIRST);
	if (showmode) {
		extern int printw();
		clrtobot();
		if (usridx >= 0) getuserinfo(printw, usrlist[usridx]->u_name);
	} else {
		for (i = first, l = 0; i < nusers && l < pagesize; i++, l++) {
			y = l + Y_FIRST;
			move(y, X_FIRST);
			printw("%5d", i+1);
			move(y, X_CURSOR);
			if (i == usridx) addstr("->");
			else addstr("  ");
			if (usrlist[i]->u_flag & UF_MARKED) addch('*');
			else addch(' ');
			move(y, X_NAME);
			printw("%-24.24s %-5.5s %2d %6d %8d  %-20.20s",
			       usrlist[i]->u_name,
			       flag2str(usrlist[i]->u_flag),
			       usrlist[i]->u_priv,
			       usrlist[i]->u_call,
			       usrlist[i]->u_tleft,
			       usrlist[i]->u_time ?
			       ctime(&usrlist[i]->u_time)+4 : "unknown");
		}
		clrtobot();
	}
	if (markflag) {
		move(LINES-1, 0);
		printw("Marked %d users", markflag);
	}
	if (usridx < 0) mvaddstr(LINES-1, COLS/2 - 7, "Oops! No Users");
	else {
		mvaddstr(LINES-1, COLS/2 - usrlist[usridx]->u_nlen/2,
		      usrlist[usridx]->u_name);
		if (showmode) {
			move(LINES-1, COLS/2+12);
			printw("- %d -", usridx + 1);
		}
		mvaddstr(LINES-1, COLS-14, "h - Help");
	}
}

int
sortbypriv(d1, d2)
	const void *d1;
	const void *d2;
{
	if ((*(struct usrent **)d1)->u_priv >
	    (*(struct usrent **)d2)->u_priv) return -1;
	if ((*(struct usrent **)d1)->u_priv <
	    (*(struct usrent **)d2)->u_priv) return 1;
	return 0;
}

int
sortbycall(d1, d2)
	const void *d1;
	const void *d2;
{
	if ((*(struct usrent **)d1)->u_call >
	    (*(struct usrent **)d2)->u_call) return -1;
	if ((*(struct usrent **)d1)->u_call <
	    (*(struct usrent **)d2)->u_call) return 1;
	return 0;
}

int
sortbytime(d1, d2)
	const void *d1;
	const void *d2;
{
	if ((*(struct usrent **)d1)->u_time >
	    (*(struct usrent **)d2)->u_time) return -1;
	if ((*(struct usrent **)d1)->u_time <
	    (*(struct usrent **)d2)->u_time) return 1;
	return 0;
}

int
sortbytleft(d1, d2)
	const void *d1;
	const void *d2;
{
	if ((*(struct usrent **)d1)->u_tleft >
	    (*(struct usrent **)d2)->u_tleft) return -1;
	if ((*(struct usrent **)d1)->u_tleft <
	    (*(struct usrent **)d2)->u_tleft) return 1;
	return 0;
}

int
sortbyflag(d1, d2)
	const void *d1;
	const void *d2;
{
	if (((*(struct usrent **)d1)->u_flag & UF_FLAGS) >
	    ((*(struct usrent **)d2)->u_flag & UF_FLAGS)) return -1;
	if (((*(struct usrent **)d1)->u_flag & UF_FLAGS) <
	    ((*(struct usrent **)d2)->u_flag & UF_FLAGS)) return 1;
	return 0;
}

int
sortbyuser(d1, d2)
	const void *d1;
	const void *d2;
{
	return (strcmp((*(struct usrent **)d1)->u_name,
		       (*(struct usrent **)d2)->u_name));
}

int
sortbyname(d1, d2)
	const void *d1;
	const void *d2;
{
	register char *s1, *s2;

	if ((s1 = strchr((*(struct usrent **)d1)->u_name, ' ')) == NULL)
		s1 = (*(struct usrent **)d1)->u_name;
	else	s1++;
	if ((s2 = strchr((*(struct usrent **)d2)->u_name, ' ')) == NULL)
		s2 = (*(struct usrent **)d2)->u_name;
	else	s2++;
	return (strcmp(s1, s2));
}

static char *compoundstr;

int
sortcompound(d1, d2)
	const void *d1;
	const void *d2;
{
	static int k[] = { 64, 32, 16, 8, 4, 2, 1 };
	register i, sum = 0, sign;

	for (i = 0; compoundstr[i] && i < 7; i++) {
		sign = 1;
		switch (compoundstr[i]) {
			case 'p': sign = -1;
			case 'P': sum += sign * k[i] * sortbypriv(d1, d2);
				break;
			case 'c': sign = -1;
			case 'C': sum += sign * k[i] * sortbycall(d1, d2);
				break;
			case 'l': sign = -1;
			case 'L': sum += sign * k[i] * sortbytime(d1, d2);
				break;
			case 't': sign = -1;
			case 'T': sum += sign * k[i] * sortbytleft(d1, d2);
				break;
			case 's': sign = -1;
			case 'S': sum += sign * k[i] * sortbyflag(d1, d2);
				break;
			case 'u': sign = -1;
			case 'U': sum += sign * k[i] * sortbyuser(d1, d2);
				break;
			case 'n': sign = -1;
			case 'N': sum += sign * k[i] * sortbyname(d1, d2);
				break;
		}
	}
	return sum;
}

eraseline(i)
	int i;
{
	int x, y;

	getyx(stdscr, y, x);
	if (i) {
		move(y, x-i);
		clrtoeol();
		refresh();
	}
	return 0;
}

char *
inputstr()
{
	int ch;
	int touch = FALSE;
	static int i = 0;
	static char strbuf[MAXUSRNAMELEN+1];

	strbuf[i] = '\0';
	if (i) {
		addstr(strbuf);
		refresh();
	}
	while (1) {
		while((ch = getch()) < ' ' || ch == 127) {
			if (ch == '\r' || ch == '\n') break;
			touch = TRUE;
			if (ch == '\b' || ch == 127) { /* Backspace */
				if (i) {
					i--;
					eraseline(1);
				} else addch('\007');
			} else if (ch == 21) {	/* Control-U, erase line */
				if (i) i = eraseline(i);
				else addch('\007');
			} else addch('\007');
			refresh();
		}
		if (ch == '\r' || ch == '\n') break;
		if (!touch) {
			i = eraseline(i);
			touch = TRUE;
		}
		if (i + 1 <= MAXUSRNAMELEN) {
			strbuf[i++] = ch;
			strbuf[i] = '\0';
			addch(ch);
		} else	addch('\007');
		refresh();
	}
	strbuf[i] = '\0';
	if (i > 0) return strbuf;
	return NULL;
}

mainloop()
{
	int ch;
	char *p;

	markflag = showmode = 0;
home_list:
	usridx = first_line = scr_line = 0;
	showlist(first_line);
	refresh();
	while (1) {
		if (usridx < 0 || nusers <= 0) onterm();
		if ((ch = getch()) == ERR) panic(1);
		switch(ch) {
			case ESC:
				switch(get_arrow_key()) {
					case KEYMAP_UP:
						goto line_up;
					case KEYMAP_DOWN:
						goto line_down;
					case KEYMAP_LEFT:
					case KEYMAP_PAGE_UP:
						goto page_up;
					case KEYMAP_RIGHT:
					case KEYMAP_PAGE_DOWN:
						goto page_down;
					case KEYMAP_HOME:
						goto home_list;
					case KEYMAP_END:
						goto end_list;
					case KEYMAP_INS:
						goto mark_line;
					default:
						mvaddstr(LINES-1, COLS-14, "Bad command\007");
						refresh();
						continue;
				}
				break;

			case 'k':	/* line up */
			case ctrl('P'):
line_up:
				if (usridx - 1 < 0) break; /* was continue */
				usridx--;
				if (--scr_line < 0 && first_line) {
					scr_line = 0;
					if (--first_line < 0) first_line = 0;
				}
				break;

			case 'j':
			case ctrl('N'):	/* line down */
line_down:
				if (usridx + 1 >= nusers) break; /* was continue */
				usridx++;
				if (++scr_line >= pagesize) {
					scr_line--;
					first_line = usridx - scr_line;
				}
				break;

			case '\b':	/* page up */
			case 'b':
			case ctrl('U'):
			case ctrl('B'):
page_up:	
				if (scr_line > 0) usridx -= scr_line;
				else {
					usridx -= pagesize;
					first_line -= pagesize;
					if (usridx < 0 || first_line < 0)
						usridx = first_line = 0;
				}
				scr_line = 0;
				break;

			case ' ':	/* page down */
			case ctrl('D'):
			case ctrl('F'):
page_down:
				if (nusers <= pagesize)
					usridx = scr_line = nusers - 1;
				else {
					if (scr_line < pagesize - 1)
						usridx += pagesize - scr_line - 1;
					else {
						usridx += pagesize;
						first_line = usridx - pagesize + 1;
					}
					if (usridx >= nusers) {
						usridx = nusers - 1;
						first_line = nusers - pagesize;
					}
					scr_line = pagesize - 1;
				}
				break;

			case ctrl('A'):		/* home */
				goto home_list;

			case ctrl('E'):		/* end */
end_list:
				if (nusers <= pagesize)
					usridx = scr_line = nusers - 1;
				else {
					usridx = nusers - 1;
					first_line = nusers - pagesize;
					scr_line = pagesize - 1;
				}
				break;

			case 'i':
mark_line:
				if (usrlist[usridx]->u_flag & UF_MARKED) {
					usrlist[usridx]->u_flag &= ~UF_MARKED;
					markflag--;
				} else {
					usrlist[usridx]->u_flag |= UF_MARKED;
					markflag++;
				}
				goto line_down;

			case '\r':
			case '\n':
				if (isuserexist(usrlist[usridx]->u_name) == UF_UNKNOWN) {
					mvaddstr(LINES-1, COLS-16, "Config absent\007");
					refresh();
					continue;
				}
				if (isuseronline(usrlist[usridx]->u_name)) {
					mvaddstr(LINES-1, COLS-18, "User is on-line\007");
					refresh();
					continue;
				}
				endwin();
				printf(DEFCOLOR);
				fflush(stdout);
				if (edituser(usrlist[usridx]->u_name) != 0) {
					printf("\nUnexpected error!? Press ENTER to continue or ^D to abort...");
					if (getchar() == EOF) onterm();
				}
				if (isuseronline(usrlist[usridx]->u_name)) {
					printf("\nUser is on-line -- your changes take no effect! Press ENTER to continue...");
					if (getchar() == EOF) onterm();
				} else (void) scanuser(usrlist[usridx]);
				initterm();
				break;

			case '\t':
				showmode ^= 1;
				move(Y_FIRST, X_FIRST);
				clrtobot();
				break;

			case 'q':		/* quit */
				onterm();

			case 'p':		/* sort by priv */
				qsort(usrlist, nusers, sizeof(struct usrent *),
				      sortbypriv);
				break;

			case 'c':		/* sort by calls */
				qsort(usrlist, nusers, sizeof(struct usrent *),
				      sortbycall);
				break;

			case 'l':		/* sort by lastcall */
				qsort(usrlist, nusers, sizeof(struct usrent *),
				      sortbytime);
				break;

			case 's':		/* sort by status flags */
				qsort(usrlist, nusers, sizeof(struct usrent *),
				      sortbyflag);
				break;

			case 't':		/* sort by time left */
				qsort(usrlist, nusers, sizeof(struct usrent *),
				      sortbytleft);
				break;

			case 'u':		/* sort by U)ser name */
				qsort(usrlist, nusers, sizeof(struct usrent *),
				      sortbyuser);
				break;

			case 'n':		/* sort by user N)ame */
				qsort(usrlist, nusers, sizeof(struct usrent *),
				      sortbyname);
				break;

			case 'r':		/* compound resort */
				move(LINES-1, 0);
				clrtoeol();
				addstr("Enter string [uUnNsSpPcCtTlL]: ");
				refresh();
				if ((p = inputstr()) != NULL) {
					compoundstr = p;
					qsort(usrlist, nusers, sizeof(struct usrent *),
					      sortcompound);
				}
				break;

			case 'd':
			case 0x7f:		/* delete user(s) */
				move(LINES-1, 0);
				if (markflag)
					printw("Delete %d users?\007", markflag);
				else	addstr("Delete this user?\007");
				refresh();
				if ((ch = getch()) == ERR) panic(1);
				if (ch == 'y' || ch == 'Y' ||
				    ch == '\r' || ch == '\n') {
					if (deleteusers() < 0) {
						move(LINES-1, 0);
						clrtoeol();
						mvaddstr(LINES-1, COLS-17, "Delete failed!\007");
						refresh();
						continue;
					}
				}
				break;

			case 'w':		/* write userlist */
				move(LINES-1, 0);
				clrtoeol();
				addstr("Enter filename: ");
				refresh();
				if ((p = inputstr()) == NULL) break;
				move(LINES-1, 0);
				clrtoeol();
				printw("Writing %s...", p);
				refresh();
				if (writelist(p) < 0) {
					move(LINES-1, 0);
					clrtoeol();
					mvaddstr(LINES-1, COLS-16, "Write failed!\007");
					refresh();
					continue;
				}
				break;

			case '/':		/* find user */
				move(LINES-1, 0);
				clrtoeol();
				addstr("Enter substring: ");
				refresh();
				if ((p = inputstr()) == NULL) break;
				if ((ch = finduser(p)) < 0) {
					move(LINES-1, 0);
					clrtoeol();
					mvaddstr(LINES-1, COLS-14, "Not found\007");
					refresh();
					continue;
				}
				if (ch > first_line + pagesize - 1)
					first_line = ch, scr_line = 0;
				else	scr_line += ch - usridx;
				usridx = ch;
				break;

			case ctrl('L'):		/* refresh screen */
				wrefresh(curscr);
				continue;

			case 'h':		/* help page */
				help_page();
				break;

			default:
				mvaddstr(LINES-1, COLS-14, "Bad command\007");
				refresh();
				continue;
		}
		showlist(first_line);
		refresh();
	}
}

/*
 * Find user by substring. Return index or -1 if not found.
 */
finduser(s)
	char *s;
{
	register i;

	for (i = usridx + 1; i < nusers; i++)
		if ((char *)__strcasestr(usrlist[i]->u_name, s) != NULL) return i;
	return -1;
}

/*
 * Delete users. Return -1 on error.
 */
deleteusers()
{
	int i, n;

	if (markflag == 0) {
		move(LINES-1, 0);
		clrtoeol();
		addstr("Deleting...");
		mvaddstr(LINES-1, COLS/2 - usrlist[usridx]->u_nlen/2,
			 usrlist[usridx]->u_name);
		refresh();
		if (delete_user(usrlist[usridx]->u_name) < 0) return -1;
		free(usrlist[usridx]);
		nusers--;
		if (usridx < nusers)
			memmove(&usrlist[usridx], &usrlist[usridx+1],
				(nusers - usridx) * sizeof(struct usrent *));
		else usridx--;
		return 0;
	}
	i = 0;
	usrlist[usridx]->u_flag |= UF_CURSOR;
	while (i < nusers) {
		if (usrlist[i]->u_flag & UF_MARKED) {
			move(LINES-1, 0);
			clrtoeol();
			addstr("Deleting...");
			mvaddstr(LINES-1, COLS/2 - usrlist[i]->u_nlen/2,
				 usrlist[i]->u_name);
			refresh();
			if (delete_user(usrlist[i]->u_name) >= 0) {
				free(usrlist[i]);
				markflag--;
				nusers--;
				if (i < nusers)
					memmove(&usrlist[i], &usrlist[i+1],
						(nusers - i) * sizeof(struct usrent *));
				continue;
			}
		}
		i++;
	}

	/* Calculate new cursor position */
	for (i = 0; i < nusers; i++) if (usrlist[i]->u_flag & UF_CURSOR) {
		usrlist[i]->u_flag &= ~UF_CURSOR;
		if (i < usridx) {
			first_line = i - scr_line;
			if (first_line < 0) first_line = 0, scr_line = i;
		}
		usridx = i;
		break;
	}
	if (usridx >= nusers) {
		usridx = nusers - 1;
		if (nusers < pagesize) first_line = 0, scr_line = usridx;
		else {
			scr_line = pagesize - 1;
			first_line = usridx - scr_line; 
		}
	} else if (i == nusers) usridx = first_line = scr_line = 0;

	return (markflag ? -1 : 0);
}

static FILE *fp;

void
writeuser(i)
	int i;
{
	fprintf(fp, "%-24.24s %-5.5s %2d %6d %8d  %-20.20s\n",
	       usrlist[i]->u_name,
	       flag2str(usrlist[i]->u_flag),
	       usrlist[i]->u_priv,
	       usrlist[i]->u_call,
	       usrlist[i]->u_tleft,
	       usrlist[i]->u_time ?
	       ctime(&usrlist[i]->u_time)+4 : "unknown");
}

int
printfp(char *fmt, ...)
{
	va_list ap;
	int ret;

	va_start(ap, fmt);
	ret = vfprintf(fp, fmt, ap);
	va_end(ap);

	return ret;
}

int
writelist(filename)
	char *filename;
{
	int i;

	if ((fp = fopen(filename, "w")) == NULL) return -1;
	if (!showmode) {
		fprintf(fp, "User Name                Stat Priv Calls Timeleft  Lastcall\n");
		fprintf(fp, "=======================================================================\n");
	}
	for (i = 0; i < nusers; i++) {
		if (markflag) {
			if (usrlist[i]->u_flag & UF_MARKED) {
				if (showmode) getuserinfo(printfp, usrlist[i]->u_name);
				else writeuser(i);
				usrlist[i]->u_flag &= ~UF_MARKED;
				markflag--;
				if (!markflag) break;
			}
		} else {
			if (showmode) getuserinfo(printfp, usrlist[i]->u_name);
			else writeuser(i);
		}
	}
	fclose(fp);
	return 0;
}

help_page()
{
	move(Y_FIRST, X_FIRST);
	clrtobot();
	mvaddstr(Y_FIRST+1, COLS/2-sizeof(ADM)/2, ADM);
	mvaddstr(Y_FIRST+2, COLS/2-strlen(copy)/2, copy);
	mvaddstr(Y_FIRST+3, COLS/2-strlen(right)/2, right);
	move(Y_FIRST+5, X_FIRST);
	addstr("\
 k, ^P, UP          line Up             j, ^N, DOWN        line Down\n\
 b, BS, LEFT, PgUp  page Up             SPACE, RIGHT, PgDn page Down\n\
 ^A, HOME           home list           ^E, END            end list\n\
 i, INS             mark users          d, DEL             delete user(s)\n\
 ENTER              edit user           TAB                toggle show mode\n\
 w                  write userlist      /                  search username\n\
 u                  sort by User name   n                  sort by user Name\n\
 p                  sort by Privlevel   s                  sort by Status flags\n\
 l                  sort by Last call   c                  sort by Calls count\n\
 t                  sort by Time left   r                  compound resort\n\
 q, ^C              quit\n");
	addstr("\n\
          R regular   C charge   I infinity   B blocked   T terminator\n");
	mvaddstr(LINES-1, COLS/2-7, "Press any key");
	refresh();
	if (getch() == ERR) panic(1);
	move(Y_FIRST, X_FIRST);
	clrtobot();
	return;
}

void
usage()
{
	fprintf(stderr, "%s\n", ADM);
	fprintf(stderr, "usage: tnsadmin [-P days [-l level] [-y]] [ usrdir ]\n");
	fprintf(stderr, "where:\n\
\t-P\tPurge users which doesn't login the \"number\" of days\n\
\t-l\tPurge users with access level <= \"number\"\n\
\t-y\tPurge in quiet mode, assume Yes on all queries\n\
\tusrdir\tPath to alternative bbs users home directory\n");
	exit(1);
}
