/*
** Program:    rays-filter.c
** Version:    1.14
** Purpose:    Mail filter for sendmailversion >= 8.11.0
** Requires:   Sendmail libmilter library
** Written by: R. Butler <butlerra@sbu.ac.uk>
** Reference:  sample.c (distributed with sendmail-8.11.0)
** Date:       15-Sep-2000
** Revised:    08-Mar-2001
**
** Copyright (C) 2001 South Bank University, London.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**
** Description:
**
** The filter examines the message headers, and MIME part headers within
** the body of a multi-part message, which are specified in the 'header_list'. 
** If any of the headers contain any of the strings listed in the 
** 'bad_string_list', then the message will be rejected.  Rejected
** messages will be saved in the directory specified in FILENAME_TEMPLATE.
** The program uses 'syslog' to record its actions.
**
** To compile this program, use a script as follows:
**
** #! /bin/sh
** SRCDIR=/usr/local/src/sendmail/sendmail-8.10.1
** cc -I$SRCDIR/sendmail -I$SRCDIR/include -o $1 \
**    $1.c \
**    $SRCDIR/obj.OSF1.V4.0.alpha/libmilter/libmilter.a \
**    $SRCDIR/obj.OSF1.V4.0.alpha/libsmutil/libsmutil.a \
**    -pthread
**
** To run the program:
**
**     % ./rays-filter -p unix:/var/run/f1.sock &
** or  % ./rays-filter -p inet:3333@localhost &
** or  % ./rays-filter -p inet6:9999@localhost &
**
** Corresponding lines must appear in sendmail.cf, e.g.
**
**     Xrays-filter, S=inet:3333@localhost, F=R, T=S:60s;R:60s;E:5m
**     O InputMailFilter=rays-filter
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <syslog.h>
#include <signal.h>
#include <sys/types.h>
#include <unistd.h>
#include <regex.h>
#include "libmilter/mfapi.h" 
#include "libmilter/milter.h"
#include "rays-filter.h"


/* Sendmail feeds long messages to the filter in chunks.
** MILTER_CHUNK_SIZE tells the program the maximum size of these chunks.
** The 'bodylen' variable gives the actual size of the current chunk.
** It is #defined in 'milter.h' */

/* The program 'overlaps' the chunks to to catch bad strings which are 
** split between chunks.  MAX_OVERLAP_SIZE must be large enough to
** contain a 'boundary' string and the MIME headers following that 
** boundary string. */

#define MAX_OVERLAP_SIZE  1024 

/* A buffer is set up to hold the current chunk of the message body
** and the overlap saved from the previous chunk */

#define BUFFER_SIZE       (MILTER_CHUNK_SIZE + MAX_OVERLAP_SIZE)

/* MAX_ADDRESS_SIZE is the number of characters required to hold
** an e-mail address from the 'From' or 'To' header. */

#define MAX_ADDRESS_SIZE 63

/* MAX_SUBJECT_SIZE is the number of characters required to hold
** a reasonable portion of the 'Subject' header. */

#define MAX_SUBJECT_SIZE 63

/* BOUNDARY_SIZE is the number of characters required to hold
** most or all of the boundary string of a multipart message */

#define BOUNDARY_SIZE 127

/* REJECT_STRING_SIZE is the number of characters required to hold
** a list of bad strings that have been found in the message.
** these are included in the reply sent to the mail client. */

#define REJECT_STRING_SIZE 63

/* REPLY_SIZE is the length of the text part of the SMTP reply
** sent back to the mail client if the message is rejected.
** Should be large enough to hold the REPLY_TEXT and some
** or all of the reject_string. */

#define REPLY_SIZE 127

/* HEADER_NAME_SIZE is the number of characters required to hold
** a reasonable portion of a MIME header name. Only used for 
** reporting purposes, so not essential to hold the whole name. */

#define HEADER_NAME_SIZE 63

/* CONF_BUFFER_SIZE is the size of a buffer used when reading lines
** from the configuration files. */

#define CONF_BUFFER_SIZE 127 

/* ERROR_MSG_SIZE is a suitable size for error messages */

#define ERROR_MSG_SIZE 127

/* FILTER_SIGNATURE is attached to annotations in the log file 
** and the reply to the mail client if the message is rejected */

#define FILTER_SIGNATURE "Ray\'s Mail Filter"

/* Arbitrary constants */

#define FOUND_BAD_STRING  100
#define MALLOC_FAIL       200
#define DISK_WRITE_FAIL   300

/* The following constants may be defined in rays-filter.h 
** If not, these are the default values. */ 

/* FILENAME_TEMPLATE is a template for the name of the log file in
** which the message and annotations will be stored while it is
** being processed (see 'man mkstemp').  The path relative to the
** filter's working directory must be included. Must be in "quotes" */ 

#ifndef FILENAME_TEMPLATE
#define FILENAME_TEMPLATE "rejects/msg.XXXXXX"
#endif

/* MSG_FILENAME_SIZE is a length sufficient to hold the absolute
** path to the working directory plus a filename based on the 
** FILENAME_TEMPLATE */

#ifndef MSG_FILENAME_SIZE
#define MSG_FILENAME_SIZE  255
#endif

/* In the event of the message being rejected by the filter, the
** log file will be retained under a new name, which will include 
** part or all of the username taken from the 'From' header.
** USERNAME_LIMIT is the maximum number of characters from the username
** which will be included in the log file name. */

#ifndef USERNAME_LIMIT
#define USERNAME_LIMIT 16
#endif

/* CONF_FILE_DIR is the location of the configuration files.
** Must be an absolute path and include the trailing '/'.
** Must be in quotes. */

#ifndef CONF_FILE_DIR
#define CONF_FILE_DIR "/usr/local/etc/mail-filter/"
#endif

/* HEADER_LIST_FILE is the name of the configuration file which
** contains the list of headers to be examined for bad strings. */

#ifndef HEADER_LIST_FILE
#define HEADER_LIST_FILE  CONF_FILE_DIR "header-list.conf"
#endif

/* STRING_LIST_FILE is the name of the configuration file which
** contains the list of bad strings to be searched for in the 
** specified headers. */

#ifndef STRING_LIST_FILE
#define STRING_LIST_FILE  CONF_FILE_DIR "string-list.conf"
#endif

/* REPLY_TEXT is the message sent back to the mail client if the
** message is rejected. The bad string(s) which caused the message
** to be rejected will be appended to this text in brackets () */

#ifndef REPLY_TEXT
#define REPLY_TEXT "Message was rejected because it contains signs " \
                   "of a possible virus" 
#endif

/* A suspiciously long 'Date' header may be a sign that somebody
** is trying to exploit the buffer overrun bug which exists  in 
** some versions of Microsoft Outlook.  To guard against this,
** messages with 'Date' headers longer than DATE_LENGTH_LIMIT 
** will be rejected.
** Default: 60  */

#ifndef DATE_LENGTH_LIMIT
#define DATE_LENGTH_LIMIT 60
#endif

/* Boolean TRUE/FALSE -- if not already defined */

#ifndef TRUE
#define FALSE  0
#define TRUE   1
typedef int bool;
#endif

/* Data Structures
**
** The Libmilter library creates an opaque context structure, 'ctx', to
** store context data for the current message.  In addition, this filter
** program defines its own private data structure, 'priv' (see below).
** The Libmilter API includes the following two functions:-
**
** 'smfi_setpriv(ctx, priv)'  :Sets a component of the 'ctx' structure
**                             to point to the 'priv' structure.
**
** 'smfi_getpriv(ctx)'        :Returns a pointer to the 'priv'
**                             structure from the 'ctx' structure.
**
** For convenience, the 'mlfi_getpriv()' function and an appropriate
** cast are wrapped up in the macro MLFIPRIV.
*/

#define MLFIPRIV          ((struct mlfiPriv *) smfi_getpriv(ctx))

struct mlfiPriv
{
   char     *mlfi_fname;
   FILE     *mlfi_fp;
   u_char   overlap[MAX_OVERLAP_SIZE + 1];
   u_char   buffer[BUFFER_SIZE + 1];
   char     to_address[MAX_ADDRESS_SIZE + 1];
   char     from_address[MAX_ADDRESS_SIZE + 1];
   char     subject[MAX_SUBJECT_SIZE + 1];
   char     boundary[BOUNDARY_SIZE + 1];
   char     reject_string[REJECT_STRING_SIZE + 1];
   sfsistat status;
};


/* Functions from sample filter (some modified) */

sfsistat mlfi_envfrom(SMFICTX *ctx, char **envfrom);
sfsistat mlfi_header(SMFICTX *ctx, char *headerf, char *headerv);
sfsistat mlfi_eoh(SMFICTX *ctx);
sfsistat mlfi_body(SMFICTX *ctx, u_char *bodyp, size_t bodylen);
sfsistat mlfi_eom(SMFICTX *ctx);
sfsistat mlfi_close(SMFICTX *ctx);
sfsistat mlfi_abort(SMFICTX *ctx); 
sfsistat mlfi_cleanup(SMFICTX *ctx, bool keep_file);


/* Functions added for this program */

sfsistat mlfi_envrcpt(SMFICTX *ctx, char **envrcpt);
static int open_log_file(struct mlfiPriv *priv);
static char *upcase_string(char *lc_string);
static int check_header_length(SMFICTX *ctx, char *headerf,
                               char *headerv, size_t limit);
static int save_boundary(SMFICTX *ctx, char *headerv);
static int match_pattern(char *string, char *pattern);
static int append_reject_string(SMFICTX *ctx, char *bad_string);
static int find_bad_string(SMFICTX *ctx, char *header_name, char *header_body); 
static int find_header(u_char *start_of_header, size_t *header_length);
static int is_target_header(char *header);
static int scan_header_block(SMFICTX *ctx, u_char *start_of_header);
static int scan_buffer(SMFICTX *ctx);
static int overlap_chunk(SMFICTX *ctx, u_char *bodyp, size_t bodylen);
static int store_overlap(SMFICTX *ctx, u_char *bodyp, size_t bodylen);
static int get_username(SMFICTX *ctx, char *username, size_t *username_length);
static int make_filename(SMFICTX *ctx, char *new_filename, 
                         size_t new_filename_size);
static char *trim_string(char *s);
static int read_config_file(FILE *fp, char *list[]);
static char **config_from_file(char *filename, char *list[]);
static void configure(int signal_name);
static int set_work_dir(char *work_dir);


/* 'header_list' contains the names of the headers that are to be 
** examined to see whether they contain 'bad strings'.
** Read from HEADER_LIST_FILE on startup or on receipt of SIGUSR1. */ 

static char  **header_list = NULL;

/* 'bad_string_list' contains the strings whose presence in any of
** the listed headers will cause the message to be rejected.
** Read from STRING_LIST_FILE on startup or on receipt of SIGUSR1. */

static char  **bad_string_list = NULL; 

/* 'msg_file_template' contains the working directory, specified as
** a command-line parameter, plus '/' plus the FILENAME_TEMPLATE 
** specified above. */

static char msg_file_template[MSG_FILENAME_SIZE + 1] = "\0";


static int open_log_file(struct mlfiPriv *priv)
{
   /* Open a file with a unique name to store this message 
   */

   priv->mlfi_fp = NULL;
   priv->mlfi_fname = strdup(msg_file_template);

   if (priv->mlfi_fname == NULL)
      syslog(LOG_INFO, "Cannot allocate memory for log file name\n");
   else
   {
      if (mktemp(priv->mlfi_fname) == NULL)
         syslog(LOG_INFO, "Cannot make name for log file\n");
      else
      {
         priv->mlfi_fp = fopen(priv->mlfi_fname, "w");
         if (priv->mlfi_fname == NULL)
            syslog(LOG_INFO, "Cannot open log file\n");
      }
   }
   
   return 0;
}


sfsistat mlfi_envfrom(SMFICTX *ctx, char **envfrom)
{
   /* Allocates memory for a private data structure.  Initialises the
   ** data structure (including the 'From' address from the envelope)
   ** and saves a pointer to it in the opaque context structure.
   */

   struct mlfiPriv *priv;

   priv = malloc(sizeof *priv);
   if (priv == NULL)
   {
      syslog(LOG_INFO, "Cannot allocate private memory\n");
      return SMFIS_TEMPFAIL;
   }
   memset(priv, '\0', sizeof *priv);

   priv->status = SMFIS_CONTINUE;
   strcpy(priv->buffer, "");
   strcpy(priv->overlap, "");
   strcpy(priv->boundary, "");
   strcpy(priv->reject_string, " (");
   if ((envfrom != NULL) && (envfrom[0] != NULL))
      strncpy(priv->from_address, envfrom[0], MAX_ADDRESS_SIZE);
   else
      strcpy(priv->from_address, "");
   open_log_file(priv);

   smfi_setpriv(ctx, priv);

   return SMFIS_CONTINUE;
}


sfsistat mlfi_envrcpt(SMFICTX *ctx, char **envrcpt)
{
   /* Saves the 'To' address from the envelope.
   */

   if (MLFIPRIV != NULL)
   {
      if ((envrcpt != NULL) && (envrcpt[0] != NULL))
         strncpy(MLFIPRIV->to_address, envrcpt[0], MAX_ADDRESS_SIZE);
      else
         strcpy(MLFIPRIV->to_address, "");
   }

   return SMFIS_CONTINUE;
}


static int save_boundary(SMFICTX *ctx, char *headerv)
{
   /* If a 'boundary' string is specified in the 'Content-Type'
   ** header, the leading '--' is prepended and the whole is 
   ** saved in the private memory structure.
   */

   regex_t    preg;
   regmatch_t pmatch;
   char       *start_ptr;
   size_t     length = 0, copy_length = 0;
   int        status = 1;

   status = regcomp(&preg, "boundary=", REG_ICASE);
   if (status == 0)
   {
      status = regexec(&preg, headerv, 1, &pmatch, 0);
      if (status == 0)
      {
         start_ptr = headerv + pmatch.rm_eo;
         if (*start_ptr == '\"')
            start_ptr++;
         length = strcspn(start_ptr, " \"\t\n\r\0");
         copy_length = length < (BOUNDARY_SIZE - 2) ? 
                       length : (BOUNDARY_SIZE - 2);
         strcpy(MLFIPRIV->boundary, "--");
         strncat(MLFIPRIV->boundary, start_ptr, copy_length);
      }
   }
   regfree(&preg);

   return status;
}


static char *upcase_string(char *lc_string)
{
   /* Creates a new string which is the all-upper-case equivalent
   ** of 'lc_string', and returns a pointer to it.
   */

   char *uc_string = NULL, *p_uc = NULL;

   uc_string = strdup(lc_string);
   if (uc_string != NULL)
   {
      p_uc = uc_string;
   
      while (*p_uc != '\0')
      {
         *p_uc = toupper(*p_uc);
         p_uc++;
      }
   }

   return uc_string;
}


static int check_header_length(SMFICTX *ctx, char *headerf,
                               char *headerv, size_t limit)
{
   /* Flags the message for rejection if the header value is 
   ** longer than limit.
   */

   char buffer[REJECT_STRING_SIZE + 1];
   int  space_left = 0;

   if (strlen(headerv) > limit)
   {
      if (MLFIPRIV->mlfi_fp != NULL)
      {
         fprintf(MLFIPRIV->mlfi_fp,  
            "-----------------------------------------------------\n" 
            FILTER_SIGNATURE " rejected the message because:\n"
            ":_ %s header is too long\n"
            "-----------------------------------------------------\n", 
            headerf);
      }
      strncpy(buffer, headerf, REJECT_STRING_SIZE);
      space_left = REJECT_STRING_SIZE - strlen(buffer);
      strncat(buffer, " header is too long", space_left);
      append_reject_string(ctx, buffer);
      MLFIPRIV->status = SMFIS_REJECT;
   }
   return 0;
}


sfsistat mlfi_header(SMFICTX *ctx, char *headerf, char *headerv)
{
   /* Stores information from certain headers and checks headers for
   ** bad strings. MLFIPRIV->status is set if a bad string is found
   ** or a memory fault occurs.
   */

   char  *headerf_uc = NULL; 
   int   i = 0;

   if (MLFIPRIV->mlfi_fp != NULL)
      fprintf(MLFIPRIV->mlfi_fp, "%s: %s\r\n", headerf, headerv);

   headerf_uc = upcase_string(headerf); 

   /* If it's the 'Subject' header, save the value for use later */
   if (strcmp(headerf_uc, "SUBJECT") == 0)
      strncpy(MLFIPRIV->subject, headerv, MAX_SUBJECT_SIZE);

   /* If it's the 'Content-Type' header, save the boundary string */
   if (strcmp(headerf_uc, "CONTENT-TYPE") == 0)
      save_boundary(ctx, headerv);
   
   /* If it's the 'Date' header, check for suspicious length */
   if (strcmp(headerf_uc, "DATE") == 0)
      check_header_length(ctx, headerf, headerv, DATE_LENGTH_LIMIT);
   
   free(headerf_uc); 

   /* Compare header name with each of those in 'header_list'.
   ** If a match is found, check the value for bad strings.
   */

   if (is_target_header(headerf))
   {
      switch (find_bad_string(ctx, headerf, headerv))
      {
         case FOUND_BAD_STRING:
            MLFIPRIV->status = SMFIS_REJECT; 
            break;
         case MALLOC_FAIL:
            if (MLFIPRIV->status != SMFIS_REJECT)
               MLFIPRIV->status = MALLOC_FAIL;
            break;
         default:
            break; 
      }
   }

   return SMFIS_CONTINUE;
}


sfsistat mlfi_eoh(SMFICTX *ctx)
{
   /* output the blank line between the header and the body
   */

   if (MLFIPRIV->mlfi_fp != NULL)
      fprintf(MLFIPRIV->mlfi_fp, "\n");
   return SMFIS_CONTINUE;
}


static int match_pattern(char *string, char *pattern)
{
   char       error_msg[ERROR_MSG_SIZE + 1];
   regex_t    preg;
   regmatch_t pmatch;
   int        status;

   status = regcomp(&preg, pattern, (REG_ICASE | REG_NOSUB));
   if (status != 0)
   {
      regerror(status, &preg, error_msg, ERROR_MSG_SIZE);
      syslog(LOG_INFO, "%s\n", error_msg);
   }
   else  
   {
      status = regexec(&preg, string, 0, &pmatch, 0);
      if ((status != 0) && (status != REG_NOMATCH))
      {
         regerror(status, &preg, error_msg, ERROR_MSG_SIZE);
         syslog(LOG_INFO, "%s\n", error_msg);
      }
   }
   regfree(&preg);

   return status;
}


static int append_reject_string(SMFICTX *ctx, char *bad_string)
{
   /* Appends the bad_string to reject_string if it is not already
   ** included and there is sufficient space. 
   */

   int    status = 0;
   size_t space_left = REJECT_STRING_SIZE - strlen(MLFIPRIV->reject_string);
   size_t space_reqd = strlen(bad_string) + 2;

   if (strstr(MLFIPRIV->reject_string, bad_string) == NULL)
   {
      if (space_left > space_reqd)
      {
         if (strcmp(MLFIPRIV->reject_string, " (") != 0)
            strcat(MLFIPRIV->reject_string, ", ");
         strcat(MLFIPRIV->reject_string, bad_string);
      }
   }
   return status;
}


static int find_bad_string(SMFICTX *ctx, char *header_name, char *header_body) 
{
   /* Searches the header for each of the character strings in 
   ** 'bad_string_list'.  Stops and returns FOUND_BAD_STRING as soon
   ** soon as one of the strings is found. Otherwise returns 0.
   */
   
   int  j = 0; 
   int  status = 0;

   while ((bad_string_list[j] != NULL) && (status == 0))
   {
      if (match_pattern(header_body, bad_string_list[j]) == 0) 
      {
         if (MLFIPRIV->mlfi_fp != NULL)
         {
            fprintf(MLFIPRIV->mlfi_fp,  
               "-----------------------------------------------------\n" 
               FILTER_SIGNATURE " found the following pattern:\n"
               ":_ %s\n"
               "in this header:\n"
               "%s: %s\n"
               "-----------------------------------------------------\n", 
               bad_string_list[j], header_name, header_body);
         }
         append_reject_string(ctx, bad_string_list[j]);
         status = FOUND_BAD_STRING;
      }
      j++;
   }
   return status;
}


static int find_header(u_char *start_of_header, size_t *header_length)
{
   /* Given the start of a header, calculates the length.  Returns
   ** 0 normally, 1 if the end of the header block is reached.
   ** Note: Headers may extend over more than one line, in which case 
   ** the second and subsequent lines begin with white-space.  The 
   ** header block is terminated by a blank line - CRLF only.  
   ** (See RFC-822).
   */

   u_char *end_of_header = NULL;
   int    end_header_block = 0;

   end_of_header = start_of_header;
   while (*end_of_header != '\0' && 
          (*end_of_header != '\n' || *(end_of_header + 1) == ' ' || 
           *(end_of_header + 1) == '\t'))
   {
      end_of_header++;
   } 
   *header_length = end_of_header - start_of_header + 1;
   if ((*end_of_header == '\0') || (*header_length < 3))
      end_header_block = 1;   
   return end_header_block;
}


static int is_target_header(char *header)
{
   /* Checks to see whether this header is one of those listed in
   ** the 'header_list'.
   */

   int j = 0;
   int status = 0;

   while ((header_list[j] != NULL) && (status == 0))
   {
      if (match_pattern(header, header_list[j]) == 0) 
         status = 1;
      j++;
   }
   return status;
}


static int scan_header_block(SMFICTX *ctx, u_char *start_of_header)
{
   /* Makes a local copy of each of the headers in the MIME header 
   ** block and compares it with the 'header_list'.  If a match is 
   ** found, checks the header contents against the 'bad_string_list'
   */

   char     *header, *end_of_header, *header_body;
   char     header_name[HEADER_NAME_SIZE] = "\0";
   size_t   header_length = 0, name_length = 0, copy_length = 0;
   int      status = 0, is_end_of_headers = 0;
   
   is_end_of_headers = find_header(start_of_header, &header_length);
   while (is_end_of_headers == 0)
   {
      header = (char *)malloc((header_length + 1) * sizeof(char)); 
      if (header != NULL)
      {  
         memset(header, '\0', (header_length + 1));
         strncpy(header, start_of_header, header_length); 
         if (is_target_header(header))
         {
            header_body = strchr(header, ':');
            if (header_body != NULL)
            {
               name_length = header_body - header;
               copy_length = name_length < HEADER_NAME_SIZE ?
                             name_length : HEADER_NAME_SIZE;
               strncpy(header_name, start_of_header, copy_length); 
               header_body += 2;
               status |= find_bad_string(ctx, header_name, header_body); 
            }
         }
         start_of_header += header_length; 
         is_end_of_headers = find_header(start_of_header, &header_length);
         free(header);
      }
      else
         status = MALLOC_FAIL;       
   }
   return status;
}


static int scan_buffer(SMFICTX *ctx)
{
   /* Scan the overlap + current body chunk for MIME part-header 
   ** blocks beginning with a boundary string.  Calls the 
   ** 'scan_header_block' function to scan each block. 
   ** Returns the status returned by the last call to 
   ** 'scan_header_block'.
   */

   u_char  *header_block;
   u_char  *buffer = MLFIPRIV->buffer;
   char    *boundary = MLFIPRIV->boundary; 
   size_t  i = 0;
   size_t  used_buffer_length;
   int     status = 0;

   /* used_buffer_length = strlen(buffer);
   ** for (i = 0; i < used_buffer_length; i++)
   **   buffer[i] = toupper(buffer[i]);
   */
  
   header_block = buffer; 
   while (header_block = (u_char *)strstr(header_block, boundary))
   { 
      status |= scan_header_block(ctx, header_block); 
      header_block++;
   }
   return status;
}


static int overlap_chunk(SMFICTX *ctx, u_char *bodyp, size_t bodylen)
{
   /* Concatenate the current body chunk with the overlap from the 
   ** previous chunk, in order to re-combine any naughty bits that
   ** were split at a chunk boundary. */
   
   size_t overlap_size = 0;
   size_t copy_length = 0;

   if (strlen(MLFIPRIV->overlap) == 0)
   {
      copy_length = bodylen < BUFFER_SIZE ?
                    bodylen : BUFFER_SIZE;
      strncpy(MLFIPRIV->buffer, bodyp, copy_length);
   }
   else
   {
      strncpy(MLFIPRIV->buffer, MLFIPRIV->overlap, BUFFER_SIZE);
      overlap_size = strlen(MLFIPRIV->buffer);
      copy_length = bodylen < (BUFFER_SIZE - overlap_size) ?
                    bodylen : (BUFFER_SIZE - overlap_size);
      strncat(MLFIPRIV->buffer, bodyp, copy_length);
   }
   if (MLFIPRIV->mlfi_fp != NULL)
   {
      fprintf(MLFIPRIV->mlfi_fp, "Overlap from previous: %7d\n",
              overlap_size);
      fprintf(MLFIPRIV->mlfi_fp, "Buffer contains:       %7d\n",
              strlen(MLFIPRIV->buffer));
   }

   return 0;
}


static int store_overlap(SMFICTX *ctx, u_char *bodyp, size_t bodylen)
{
   /* Store the last part of this chunk so that it can be appended to
   ** the beginning of the next chunk
   */

   u_char *overlap_ptr = 0;

   if (bodylen <= MAX_OVERLAP_SIZE)
      overlap_ptr = bodyp;
   else
      overlap_ptr = bodyp + bodylen - MAX_OVERLAP_SIZE;
   strncpy(MLFIPRIV->overlap, overlap_ptr, MAX_OVERLAP_SIZE);
   if (MLFIPRIV->mlfi_fp != NULL)
   {
      fprintf(MLFIPRIV->mlfi_fp, 
              "Overlap to next chunk: %7d\n"
              "=====================================================\n",
              strlen(MLFIPRIV->overlap));
   }
   
   return 0;
}


sfsistat mlfi_body(SMFICTX *ctx, u_char *bodyp, size_t bodylen)
{
   /* Saves the text of a chunk of the message.  If it is a multipart
   ** message, looks for MIME header blocks and scans then for bad
   ** strings.
   */

   /* Delay for testing */
   /* system("sleep 30"); */

   if (MLFIPRIV->mlfi_fp != NULL)
   {
      if (fwrite(bodyp, bodylen, 1, MLFIPRIV->mlfi_fp) <= 0)
         MLFIPRIV->status = DISK_WRITE_FAIL;
      else
      {
         fprintf(MLFIPRIV->mlfi_fp,
                 "\n=====================================================\n"
                 "Note inserted by " FILTER_SIGNATURE
                 "\n-----------------------------------------------------\n"
                 "This chunk (bodylen):  %7d\n", bodylen);
      }
   }

   /* If there was a boundary string in the 'Content-Type' header,
   ** then this is a multi-part message, and we need to check for
   ** MIME part headers in the message body. */

   if (strcmp(MLFIPRIV->boundary, "") != 0)
   {
      overlap_chunk(ctx, bodyp, bodylen);
      switch (scan_buffer(ctx))
      {
         case FOUND_BAD_STRING:
            MLFIPRIV->status = SMFIS_REJECT; 
            break;
         case MALLOC_FAIL:
            if (MLFIPRIV->status != SMFIS_REJECT)
               MLFIPRIV->status = MALLOC_FAIL;
            break;
         default:
            break; 
      }
      store_overlap(ctx, bodyp, bodylen);
   } 
   return SMFIS_CONTINUE;
}


sfsistat mlfi_eom(SMFICTX *ctx)
{
   /* Take action according to the status of the message */

   static char *message_state[]  = { " Accepted",
                                     " Disk write failed while filtering",
                                     "*Malloc failed while filtering",
                                     "*Rejected"
                                   };
   int    msi;
   bool   keep_file;
   int    status;
   char   reply[REPLY_SIZE + 1];
   size_t space_left;

   switch (MLFIPRIV->status)
   {
      case SMFIS_REJECT:
         strncpy(reply, REPLY_TEXT, REPLY_SIZE);
         space_left = REPLY_SIZE - strlen(reply) - 2;
         strncat(reply, MLFIPRIV->reject_string, space_left);
         strcat(reply, ")");
         smfi_setreply(ctx, "554", "5.7.1", reply);
         msi = 3;
         keep_file = TRUE;
         status = SMFIS_REJECT;
         break;
      case MALLOC_FAIL:
         msi = 2;
         keep_file = TRUE;
         status = SMFIS_TEMPFAIL;
         break;
      case DISK_WRITE_FAIL:
         msi = 1;
         keep_file = FALSE;
         status = SMFIS_CONTINUE;
         break;
      default:
         msi = 0;
         keep_file = FALSE;
         status = SMFIS_CONTINUE;
         break;
   }
#ifdef DEBUG
   printf("%s message:  From: %s  To: %s  Subject: %s\n",
          message_state[msi], MLFIPRIV->from_address, 
          MLFIPRIV->to_address, MLFIPRIV->subject);
#endif    
   syslog(LOG_INFO, "%s message:  From: %s  To: %s  Subject: %s\n",
          message_state[msi], MLFIPRIV->from_address,
          MLFIPRIV->to_address, MLFIPRIV->subject); 

   mlfi_cleanup(ctx, keep_file);

   return status;
}


sfsistat mlfi_abort(SMFICTX *ctx)
{
   /* Action to take if the message is aborted by the sender.
   ** Contrary to what is implied in 'mfapi.h', this function WILL
   ** be called after mlfi_eom if that function returns SMFIS_REJECT.
   ** Therefore take care not to include anything here that might
   ** conflict with actions already taken in mlfi_eom.
   */
  
   mlfi_cleanup(ctx, FALSE);
   return SMFIS_CONTINUE;
}


sfsistat mlfi_close(SMFICTX *ctx)
{
   return SMFIS_ACCEPT;
}


static int get_username(SMFICTX *ctx, char *username, size_t *username_length)
{
   /* Extracts the username part of the 'From' address (before the '@')
   ** that was saved earlier by 'mlfi_envfrom'.
   */ 
   
   char    *at_ptr = NULL;
   char    *name_ptr = NULL;
   int     status = 0;
   size_t  temp_length;

   name_ptr = MLFIPRIV->from_address;
   while ((!isalnum(*name_ptr)) && (*name_ptr != '\0'))
      name_ptr++;
   at_ptr = strchr(name_ptr, '@');
   if (at_ptr != NULL)
   {
      temp_length = (size_t)(at_ptr - name_ptr);
      *username_length = temp_length < USERNAME_LIMIT ? 
                         temp_length : USERNAME_LIMIT;
      strncpy(username, name_ptr, *username_length);
      status = 1;
   }
   else
   {
      strncpy(username, name_ptr, USERNAME_LIMIT);
      *username_length = strlen(username);
      status = 0;
   }
   return status;
}


static int make_filename(SMFICTX *ctx, char *new_filename, 
                         size_t new_filename_size)
{
   /* Constructs a new name for the log file, including the username
   ** from the 'From' header. Returns the length of the new filename,
   ** which will be zero if 'mktemp' fails.
   */

   char    *p1 = MLFIPRIV->mlfi_fname;
   char    *p2 = strrchr(p1, '/');
   char    username[USERNAME_LIMIT + 1];
   size_t  username_length = 0;

   memset(new_filename, '\0', new_filename_size);
   get_username(ctx, username, &username_length); 
   if (username_length == 0)
   {
      strncpy(username, "Unknown", USERNAME_LIMIT);
      username_length = 7;
   }
   if (p2 != NULL)
   {
      strncpy(new_filename, p1, (p2 - p1 + 1));
      strncat(new_filename, username, username_length);
   }
   else
   {
      strncpy(new_filename, username, username_length); 
   }
   strcat(new_filename, ".XXXXXX");
   mktemp(new_filename);
   
   return strlen(new_filename);
}


sfsistat mlfi_cleanup(SMFICTX *ctx, bool keep_file)
{
   sfsistat rstat = SMFIS_CONTINUE;
   struct   mlfiPriv *priv = MLFIPRIV;
   size_t   new_filename_size;
   char     *new_filename;

   if (priv == NULL)
      return rstat;

   if (priv->mlfi_fp != NULL)
   {
      /* close the archive file */
      if (fclose(priv->mlfi_fp) == EOF)
      {
         /* failed; we have to wait until later */
         rstat = SMFIS_TEMPFAIL;
         (void) unlink(priv->mlfi_fname);
      }
      else if (keep_file)
      {
         /* rename the archive file if possible, otherwise delete it */
         new_filename_size = strlen(priv->mlfi_fname) + USERNAME_LIMIT + 2;
         new_filename = (char *)malloc(new_filename_size); 
         if (new_filename != NULL)
         {
            if (make_filename(ctx, new_filename, new_filename_size) > 0)
               rename(priv->mlfi_fname, new_filename);  
            else
               (void) unlink(priv->mlfi_fname);
            free(new_filename);
         }
      }
      else
      {
         /* delete the archive file */
         (void) unlink(priv->mlfi_fname);
      }
   }

   /* release private memory */
   free(priv->mlfi_fname);
   free(priv);
   smfi_setpriv(ctx, NULL);

   return rstat;
}


static char *trim_string(char *s)
{
   char *start = s;
   char *end = s + strlen(s) - 1;

   while (isspace(*start))
      start++;
   while (isspace(*end) && (end > start))
   {
      *end = '\0';
      end--;
   }
   
   return start;
}


static int read_config_file(FILE *fp, char *list[])
{
   /* Reads lines from a configuration file and returns a count
   ** of non-blank, non-comment lines.  If 'list' is not NULL,
   ** stores the data in memory and puts a pointer to it in 'list'.
   */
   
   char conf_buffer[CONF_BUFFER_SIZE + 1];
   char *temp;
   int  line_count = 0;

   while (fgets(conf_buffer, CONF_BUFFER_SIZE, fp))
   {
      temp = trim_string(conf_buffer);
      if ((temp[0] != '\0') && (temp[0] != '#'))
      {
         if (list != NULL)
            list[line_count] = strdup(temp);
         line_count++;
      }
   }
   if (list != NULL)
      list[line_count] = NULL;
   
   return line_count;
}


static char **config_from_file(char *filename, char **old_list)
{
   /* Determines the number of data lines in the specified file,
   ** creates an array of pointers of appropriate size and reads
   ** and stores data from the file.  Removes the existing data
   ** (if any) and frees memory. Returns a pointer to the new
   ** data structure.
   */
  
   int  lines = 0;
   char **new_list = NULL, **p = NULL;
   FILE *conf_fp = fopen(filename, "r");

   if (conf_fp != NULL)
   {
      lines = read_config_file(conf_fp, NULL);
      new_list = (char **)malloc((lines + 1) * sizeof(char *));
      if (new_list != NULL)
      {
         fseek(conf_fp, 0, SEEK_SET);
         read_config_file(conf_fp, new_list);
         fclose(conf_fp);
         if (old_list != NULL)
         {
            p = old_list; 
            while (*p != NULL)
            {
               free(*p);
               p++;
            }
            free(old_list);
         } 
      }
   }
   return new_list; 
}


static void configure(int signal_name)
{
   /* Reads and stores data from configuration files at startup or in
   ** response to a signal.
   */

   int line = 0;

   if (signal_name == SIGUSR1)
      syslog(LOG_INFO, "Received SIGUSR1; reading configuration files\n");
   
   header_list = config_from_file(HEADER_LIST_FILE, header_list);
   if (header_list != NULL)
   {
      line = 0;
      while (header_list[line] != NULL)
         line++;
   }
   syslog(LOG_INFO, "Read %d header names from %s\n",
          line, HEADER_LIST_FILE);

   bad_string_list = config_from_file(STRING_LIST_FILE, bad_string_list);
   if (bad_string_list != NULL)
   {
      line = 0;
      while (bad_string_list[line] != NULL)
         line++;
   }
   syslog(LOG_INFO, "Read %d strings from %s\n",
          line, STRING_LIST_FILE);

   return;
}


static int set_work_dir(char *work_dir)
{
   size_t wd_length = strlen(work_dir);
   size_t ft_length = strlen(FILENAME_TEMPLATE);
   size_t total_length = wd_length + 1 + ft_length;
   int    status = 0;

   if (total_length < MSG_FILENAME_SIZE)
   {
      strcpy(msg_file_template, work_dir);
      strcat(msg_file_template, "/");
      strcat(msg_file_template, FILENAME_TEMPLATE);
      status = 1;
   }
   return status;
}


struct smfiDesc smfilter =
{
   "RaysFilter",    /* filter name */
   SMFI_VERSION,    /* version code -- do not change */
   SMFIF_ADDHDRS,   /* flags */
   NULL,            /* connection info filter */
   NULL,            /* SMTP HELO command filter */
   mlfi_envfrom,    /* envelope sender filter */
   mlfi_envrcpt,    /* envelope recipient filter */
   mlfi_header,     /* header filter */
   mlfi_eoh,        /* end of header */
   mlfi_body,       /* body block filter */
   mlfi_eom,        /* end of message */
   mlfi_abort,      /* message aborted */
   mlfi_close,      /* connection cleanup */
};


int main(int argc, char *argv[])
{
   int        c;
   const char *args = "d:p:";
   void       (*old_signal_handler)(int);

   /* Process command line options */
   while ((c = getopt(argc, argv, args)) != -1)
   {
      switch (c)
      {
         case 'd':
            if (optarg == NULL || *optarg == '\0')
            {
               fprintf(stderr, "%s: Illegal working directory: %s\n", 
                       argv[0], optarg);
               exit(EX_USAGE);
            }
            if (! set_work_dir(optarg))
            {
               fprintf(stderr, 
                  "Working directory name is too long.  Specify a shorter\n");
               fprintf(stderr, 
                  "path or increase MSG_FILENAME_SIZE in \'rays-filter.h\'.\n");
               exit(EX_USAGE);
            }
            break;
         case 'p':
            if (optarg == NULL || *optarg == '\0')
            {
               (void) fprintf(stderr, "%s: Illegal connection: %s\n", 
                              argv[0], optarg);
               exit(EX_USAGE);
            }
            (void) smfi_setconn(optarg);
            break;
      }
   }
   if (smfi_register(smfilter) == MI_FAILURE)
   {
      fprintf(stderr, "%s: smfi_register failed\n", argv[0]);
      exit(EX_UNAVAILABLE);
   }

   openlog(argv[0], LOG_PID, LOG_USER);
   syslog(LOG_INFO, "Program starting\n");
   configure(0); 

   /* We use SIGUSR1 to make the program re-read its configuration files
   ** because the response to SIGHUP is otherwise defined in 'libmilter.a'
   */
   old_signal_handler = signal(SIGUSR1, configure);
   if (old_signal_handler == SIG_ERR)
   {
      fprintf(stderr, 
              "%s: Warning: SIGUSR1 cannot be used to read config files.\n", 
              argv[0]);
   }
   
   return smfi_main();
}

