/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: st_sql_oracle.c,v 1.3.2.2 2005/02/18 18:03:28 anton Exp $ */

#ifdef USE_ORACLE

#include "netams.h"
#include "oci.h"

/*
 * OCI is bit different from MySQl/PG in programming model.
 * First, we must allocate OCI environment, in which all
 * OCI calls will be executed. This can be global var, OCI
 * will take care of proper locking.
 *
 */

#define ORA_PREFETCH_ROWS   100
#define DEF_DATASTR_LEN     128

static OCIEnv *g_env;
/* we need protect initialization of g_env */
static pthread_mutex_t g_env_mtx = PTHREAD_MUTEX_INITIALIZER;
/* here goes Oracle version */
char    oraver[512];
   
typedef struct oracle_conn {
    OCISvcCtx *svcctx;
    OCIError *errh;
	OCIStmt  *stmth;
    OCIServer *srvh; 
    char **row;
} ORAconn;

static int  ora_select(ORAconn *conn, char *query);
static int  ora_execute(ORAconn *conn, char *query);
static char **ora_fetch(ORAconn *conn);
static char **ora_alloc_row(ORAconn *conn);
static void ora_free_row(char **row);
static int  ora_chk_table(ORAconn *conn, char *tname);
static char *ora_conv_to_values(char *line);
static void ora_errprint(dvoid *errhp, ub4 htype, sb4 *errcodep);
static void ora_checkerr(dvoid *errhp, ub4 htype, sword status,
    text *note, sb4 state, text *file, sb4 line);
/* OCI_CHECK(errhp, ub4 errhptype, sb4 status, struct loadctl *ctlp,
 *          OCIfunction());
 * errhp is typically a (OCIError *), and errhptype is OCI_HTYPE_ERROR.
 * errhp in some cases may be an (OCIEnv *), and errhptype is OCI_HTYPE_ENV.
 */
#define OCI_CHECK(errhp, htype, status, ctlp, OCIfunc) \
if (OCI_SUCCESS != ((status) = (OCIfunc))) \
{ \
  checkerr((dvoid *)(errhp), (ub4)(htype), (sword)(status), (text *)0, \
           (sb4)0, (text *)__FILE__, (sb4)__LINE__); \
  if ((status) != OCI_SUCCESS_WITH_INFO) \
    cleanup((struct loadctl *)ctlp, (sb4)1); \
} else

#define OCI_CHK(status, errhp)                                \
  ora_checkerr((dvoid *)errhp, (ub4)(OCI_HTYPE_ERROR), (sword)(status), (text *)0, \
           (sb4)0, (text *)__FILE__, (sb4)__LINE__);

#define FATAL(note, state) \
do \
{ \
  checkerr((dvoid *)0, (ub4)OCI_HTYPE_ERROR, (sword)OCI_SUCCESS,           \
           (text *)(note), (sb4)(state), (text *)__FILE__, (sb4)__LINE__); \
  cleanup((ctlp), (sb4)2); \
} while (0)


//////////////////////////////////////////////////////////////////////////
void *
ora_stOpenSql(Service *s, st_conn_type type)
{
	ServiceStorage_cfg *cfg = (ServiceStorage_cfg *)s->cfg;
//	FILE *debug;
    ORAconn *conn;
	char *st_conn = NULL;
	char *ora_user, *ora_pass, *ora_dbname;
    sword rc;

    /*
     * sanity check.
     */
    if (type > (ST_CONN_TYPES_NUM - 1)) {
        aLog(D_ERR, "Table type greater then ST_CONN_TYPES_NUM!\n");
        return (NULL);
    }

    /* check if this is first call to OCI */
    pthread_mutex_lock(&g_env_mtx);
    if (g_env == NULL) {
        /* create OCI environment */
        if (OCIEnvCreate(&g_env, OCI_THREADED,
                NULL, NULL, NULL, NULL, 0, NULL)) {
            aLog(D_CRIT, "OCI environment creation failed!\n");
            return (NULL);
        }
    }
    pthread_mutex_unlock(&g_env_mtx);

    if (!cfg->username)
        ora_user="netams";
    else
        ora_user=cfg->username;

    if (!cfg->password)
        ora_pass="netams";
    else
        ora_pass=cfg->password;

    if (!cfg->dbname)
        ora_dbname="netams";
    else
        ora_dbname=cfg->dbname;

    conn = (ORAconn *)aMalloc(sizeof(ORAconn));

    /* create error report handle */
    rc = OCIHandleAlloc(g_env, (dvoid **)&conn->errh, OCI_HTYPE_ERROR, 0, NULL);
    OCI_CHK(rc, NULL);

    /* create handle for OCIServerVersion() call */
    rc = OCIHandleAlloc(g_env, (dvoid**)&conn->srvh, OCI_HTYPE_SERVER, 0, NULL);
    OCI_CHK(rc, NULL);
//    rc = OCIServerAttach(conn->srvh, conn->errh,
//        (CONST text*)ora_dbname, strlen(ora_dbname), OCI_DEFAULT);
    rc = OCIServerAttach(conn->srvh, conn->errh,
        (CONST text*)0, 0, OCI_DEFAULT);
    OCI_CHK(rc, conn->errh);

    /* try connect to DB */
    rc = OCILogon(g_env, conn->errh, &conn->svcctx,
        (CONST text*)ora_user, strlen(ora_user),
        (CONST text*)ora_pass, strlen(ora_pass),
        (CONST text*)ora_dbname, strlen(ora_dbname));
    OCI_CHK(rc, conn->errh);

    /* get Oracle version */
    rc = OCIServerVersion(conn->srvh, conn->errh, (text*)oraver,
        sizeof(oraver), OCI_HTYPE_SERVER);
    OCI_CHK(rc, conn->errh);
    
    /* create statement handle */
    rc = OCIHandleAlloc(g_env, (dvoid **)&conn->stmth,
        OCI_HTYPE_STMT, 0, NULL);
    OCI_CHK(rc, NULL);

    rc = ora_chk_table(conn, (st_conn = st_table_name[type]));
    if (rc == 0) {
        aLog(D_CRIT, "Table %s doesn't exist. Please create DB scheme!\n",
            st_conn);
    }

	//res=PQexec(conn,"COMMIT");PQclear(res);
	//res=PQexec(conn,"BEGIN");PQclear(res);
	return (void*)conn;
}

//////////////////////////////////////////////////////////////////////////
void
ora_stCloseSql(void *fd)
{
    ORAconn *conn = (ORAconn *)fd;

    OCILogoff(conn->svcctx, conn->errh);
    OCIServerDetach(conn->srvh, conn->errh, OCI_DEFAULT);
    OCIHandleFree(conn->errh, OCI_HTYPE_ERROR);
    OCIHandleFree(conn->stmth, OCI_HTYPE_STMT);
    OCIHandleFree(conn->srvh, OCI_HTYPE_SERVER);
    	if (conn->row != NULL)
        	ora_free_row(conn->row);
	
	aFree(conn);

}

//////////////////////////////////////////////////////////////////////////
unsigned
ora_stSaveSql(void *fd, char *filename, st_conn_type type)
{
    ORAconn *conn=(ORAconn*)fd;
    FILE *fp;
    char    *q = NULL, line[1024], *values;
    int qsize;
    int res = 0, lines = 0;
    sword rc;

        aDebug(DEBUG_STORAGE, "ora_stSaveSql: open connection 0x%08x\n",
            conn);
        // Oracle doesn't support COPY FROM or
        // LOAD DATA LOCAL INFILE, so we must load file
        // by hand, line by line.
        fp = fopen(filename, "r");
        while(fgets(line, sizeof(line), fp) != NULL) {
            values = ora_conv_to_values(line);
            q = (char*)realloc(q, (qsize = strlen(values) + 256));
            snprintf(q, qsize, "INSERT INTO %s VALUES (%s)",
                st_table_name[type], values);
            aFree(values);
            lines ++;
            if ((rc = ora_execute(conn, q)) != OCI_SUCCESS) {
                aLog(D_WARN, "ora_stSaveSql() failed for %s\n", q);
                OCI_CHK(rc, conn->errh);
                continue;
            }
            //aDebug(DEBUG_STORAGE, "ora_stSaveSql from '%s', line %d\n", filename, lines);
            res ++;
        }
        fclose(fp);
        free(q);

		// commit changes
        if ((rc = ora_execute(conn, "COMMIT")) != OCI_SUCCESS) {
            aLog(D_WARN, "ora_stSaveSql() failed for COMMIT\n");
            OCI_CHK(rc, conn->errh);
        }

        if (lines != res) {
            aLog(D_WARN, "ora_stSaveSql: %d lines read, but only %d inserted!\n",
                lines, res);
        }

    return (res);
}

//////////////////////////////////////////////////////////////////////////
u_char
ora_stLoadSql(void *fd, Message *message)
{
	Message_Read *msg=(Message_Read *)message;
	char query[255], **row;
	ORAconn *conn = (ORAconn*)fd;
	pstat *cps=NULL;
    sword rc;

	//actually there is no "F"low read requests
	switch (msg->prefix){
    case 'M': cps=&(msg->pdata->m); break;
    case 'W': cps=&(msg->pdata->w); break;
    case 'D': cps=&(msg->pdata->d); break;
    case 'H': cps=&(msg->pdata->h); break;
    }

	snprintf(query, 254, "SELECT bytes_in, bytes_out FROM summary "
        "WHERE prefix='%c' AND unit_oid=%u AND policy_oid=%u AND t_from=%lu",
		msg->prefix, msg->netunit, msg->ap, (unsigned long)cps->from);
	rc = ora_select(conn, query);
    if (rc == OCI_SUCCESS) {
        if ((row = ora_fetch(conn)) != NULL) {
            unsigned long long    in, out;
            in = strtoull(conn->row[0], NULL, 0);
            out = strtoull(conn->row[1], NULL, 0);
            cps->in += in;
            cps->out += out;
        }
    }

	aDebug(DEBUG_STORAGE, "SQL<-HDD/%c query %u bytes, fr=%lu in=%llu out=%llu\n",
        msg->prefix, strlen(query), cps->from, cps->in, cps->out);

	return 1;
}

//////////////////////////////////////////////////////////////////////////
u_char
ora_stLgObtainDbData(void *fd)
{
	char *buffer;
	ORAconn *conn = (ORAconn *)fd;

	buffer="SELECT * from login";
	aDebug(DEBUG_LOGIN, "sLgObtainDbData query: '%s'\n", buffer);
	if (ora_select(conn, buffer) != OCI_SUCCESS) { 
		return 0; 
	}

    {	
        char **row;
		sLoginData *logindata;
		unsigned tmp;
		NetUnit *u=NULL;
		oid id;
		
		while((row=ora_fetch(conn)) != NULL){
			sscanf(row[0], "%u", &id);
			if(id && !(u=Units.getUnitById(id))) continue;
			if(u->logindata) {
				logindata=u->logindata;
				aFree(logindata->password);
			} else 
				logindata = (sLoginData*)aMalloc(sizeof(sLoginData));		

			logindata->password=set_string(row[1]);
            sscanf(row[2], "%lu", (unsigned long*)&logindata->inact);
            sscanf(row[3], "%lu", (unsigned long*)&logindata->abs);
            sscanf(row[5], "%lu", (unsigned long*)&logindata->opened);
            sscanf(row[6], "%u", &logindata->ip_from.s_addr);
			memcpy(&logindata->mac_from, ether_aton(row[7]),
                sizeof (struct ether_addr));
            sscanf(row[8], "%u", &tmp); logindata->flags=tmp;
			
			u->logindata=logindata;
		}
    }
	return 1;
}
//////////////////////////////////////////////////////////////////////////
u_char
ora_stQuObtainDbData(void *fd)
{
    char *buffer, **row;
	ORAconn *conn = (ORAconn*)fd;
    sQuotaData *q;
    unsigned tmp;
    NetUnit *u=NULL;
    oid id;

	buffer="SELECT * from quota";
    aDebug(DEBUG_QUOTA, "sQuObtainDbData query: '%s'\n", buffer);
	if (ora_select(conn, buffer) != OCI_SUCCESS) {
		return 0;
	}

    while((row=ora_fetch(conn)) != NULL) {
        sscanf(row[0], "%u", &id);
        if(id && !(u=Units.getUnitById(id))) continue;
        if(u->quotadata) {
            q=u->quotadata;
        } else
            q = (sQuotaData*)aMalloc(sizeof(sQuotaData));
                       	
        sscanf(row[1], "%u", &id); q->policy=PolicyL.getPolicyById(id);
        sscanf(row[3], "%u", &tmp); q->soft_treshold=tmp;
        sscanf(row[4], "%u", &tmp); q->flags=tmp;
        sscanf(row[5], "%lu", (unsigned long*)&q->blocked_time);

        sscanf(row[6], "%u", &q->nso);
        sscanf(row[7], "%u", &q->nho);
        sscanf(row[8], "%u", &q->nro);

        sscanf(row[9], "%llu", &q->h.in);
        sscanf(row[10], "%llu", &q->h.out);
        sscanf(row[11], "%llu", &q->h.sum);

        sscanf(row[12], "%llu", &q->d.in);
        sscanf(row[13], "%llu", &q->d.out);
        sscanf(row[14], "%llu", &q->d.sum);

        sscanf(row[15], "%llu", &q->w.in);
        sscanf(row[16], "%llu", &q->w.out);
        sscanf(row[17], "%llu", &q->w.sum);

        sscanf(row[18], "%llu", &q->m.in);
        sscanf(row[19], "%llu", &q->m.out);
        sscanf(row[20], "%llu", &q->m.sum);
		
		sscanf(row[21], "%u", &id); if (id) q->fw_block_policy=PolicyL.getPolicyById(id); else q->fw_block_policy=NULL;
		sscanf(row[22], "%u", &tmp); q->fw_block_policy_flags=tmp;
			
        u->quotadata=q;
    } 
    return 1;
}

//////////////////////////////////////////////////////////////////////////
#ifdef HAVE_BILLING

u_char ora_stBiObtainDbData(void *fd){
	char *buffer, **row;
	ORAconn *conn = (ORAconn*)fd;
	
	aDebug(DEBUG_BILLING, "SQL->HDD/billing data request\n");
	buffer="SELECT * FROM billing ORDER BY CREATED";
	if (ora_select(conn, buffer) != OCI_SUCCESS) {
		aLog(D_WARN, "oracle select billing failed: \n"); 
		return 0; 
	}

	Account *ac;
	NetUnit *u;
	unsigned t;
    while((row=ora_fetch(conn)) != NULL) {

			sscanf(row[0], "%u", &t);
			newOid(t);
			
			sscanf(row[10], "%u", &t); //created ?
			if(!t) continue; //this account deleted

			sscanf(row[0], "%u", &t);
			ac=new Account();
			ac->id=newOid(t);

//			for(int j=0; j<16; j++) printf("row[%d]=%s\n", j, row[j]); 
			if (row[1][0]) ac->name=set_string(row[1]);
			if (row[2][0]) ac->description=set_string(row[2]);
			
			sscanf(row[3], "%lf", &ac->balance);
			
			sscanf(row[5], "%u", &t); ac->plan=bPlans->Check(t);
			sscanf(row[6], "%lu", (unsigned long*)&ac->plan_ch);
			sscanf(row[7], "%u", &t); ac->nextplan=bPlans->Check(t);
			sscanf(row[8], "%lu", (unsigned long*)&ac->nextplan_ch);
			sscanf(row[9], "%lu", (unsigned long*)&ac->blocked);
			sscanf(row[10], "%lu", (unsigned long*)&ac->created);
			sscanf(row[11], "%lu", (unsigned long*)&ac->changed);
			sscanf(row[12], "%lu", (unsigned long*)&ac->last_fee_ch);
							  
			if (row[13][0]) ac->email=set_string(row[13]);
			if (row[14][0]) ac->password=set_string(row[14]);
			sscanf(row[15], "%u", &t); ac->status=t;
			
			u_short i=0,k=strlen(row[4]);
			while (i*7<k) {
				sscanf(row[4]+i*7, "%06X", &t);
				if ((u=Units.getUnitById(t))) ac->AddUnit(u, ADD); else break;
				i++;
			}

			bAccounts->Add(ac);
	}
	return 1;
}

extern struct FeeCounters FC;

void ora_stBiLoadBdata(void *fd, Account *ac, char prefix) {
	char query[255];
 	
	char **row;
	ORAconn *conn = (ORAconn*)fd;
	
	bstat *bs=NULL;
	time_t t_from=0;
	
	switch (prefix){
		case 'M': t_from=FC.mt; break;
		case 'W': t_from=FC.wt; break;
		case 'D': t_from=FC.dt; break;
		case 'H': t_from=FC.ht; break;
        }

	snprintf(query, 254, "SELECT subplan_oid, bytes_in, bytes_out, pay_in, pay_out FROM bdata WHERE account_oid=%u AND t_from=%lu AND prefix='%c'", ac->id, (unsigned long)t_from, prefix);
	if (ora_select(conn, query) != OCI_SUCCESS) {
		aLog(D_WARN, "oracle select bstat failed: \n"); 
		return; 
	}
	
	bSPlist *bsp;
	unsigned id;
	u_char i;

    while((row=ora_fetch(conn)) != NULL) {
			sscanf(row[0], "%u", &id);
			
			for(i=0,bsp=ac->plan->root;bsp!=NULL;bsp=bsp->next) {
				if(bsp->sp->id==id) {
					switch (prefix){
						case 'M': bs=&ac->data[i].m; break;
						case 'W': bs=&ac->data[i].w; break;
						case 'D': bs=&ac->data[i].d; break;
						case 'H': bs=&ac->data[i].h; break;
					}
					bs->from=t_from;

					sscanf(row[1], "%lld", &bs->in);
					sscanf(row[2], "%lld", &bs->out);
					sscanf(row[3], "%lf", &bs->pay_in);
					sscanf(row[4], "%lf", &bs->pay_out);
					break;
				}
				i++;
			}
		}
	aDebug(DEBUG_BILLING, "Account %s(%06X), '%c' bstat loaded\n", ac->name, ac->id, prefix);
}
#endif // billing

//////////////////////////////////////////////////////////////////////////
// SQL config
/* XXX broken in -stable
void
ora_stConfigGet(void *fd, config_entry *e, schema_entry *h)
{
	char query[255], **row;
	ORAconn *conn = (ORAconn*)fd;     
	
	snprintf(query, 254, "SELECT schema.name,config.id,i_val,s_val,optional "
        "FROM config,schema "
        "WHERE schema.skey=%d AND config.skey=schema.skey "
        "ORDER BY config.skey,config.sequence limit 1",
        e->skey);
	if(ora_select(conn, query) != OCI_SUCCESS) {
		return;
	}
    while ((row = ora_fetch(conn)) != NULL) {
        h->name=(char*)aMalloc(strlen(row[0]));
        strcpy(h->name, row[0]);
        sscanf(row[1], "%d", &e->id);
        sscanf(row[2], "%d", &e->i_val);
        e->s_val=(char*)aMalloc(strlen(row[3]));
        strcpy(e->s_val, row[3]);
    }
    aDebug(DEBUG_PARSE, "key %d read from config database\n", e->skey);
}

void
ora_stConfigPut(Service *s, config_entry *e)
{
}
*/
/*
 * Oracle-specific functions.
 */

/*
 * Convert coma-delimeted string into string
 * which can be used as VALUES opeardn.
 *
 * XXX Can not handle empty fileds!!!
 */
static char *
ora_conv_to_values(char *line)
{
    char    *p, *strleft = line;
#define MAX_QSIZE   2048
    char    *values = (char*)aMalloc(MAX_QSIZE);
    char    *tmp = NULL;

    /*
     * Remove trailing '\n' if any
     */
     if ((p = strrchr(line, '\n')) != NULL)
	*p = '\0';

    /*
     * we use simplest algorithm possible -
     * if first char is not number - treat
     * it as a string.
     */
    while ((p = strchr(strleft,',')) != NULL) {
        *p = '\0';
        tmp = (char*)realloc(tmp, strlen(strleft) + 4);
        if (isdigit(strleft[0]))
            sprintf(tmp, "%s,", strleft);
        else
            sprintf(tmp, "'%s',", strleft);
        strleft = ++p;
        strcat(values, tmp);
    }

    /* use what is left in strleft */
    if (isdigit(strleft[0]))
        tmp = strdup(strleft);
    else {
        tmp = (char*)realloc(tmp, strlen(strleft) + 10);
        sprintf(tmp, "'%s'", strleft);
    }

    strcat(values, tmp);
    free(tmp);

    return (values);    
}

/*
 * Execute given query
 */
static int
ora_execute(ORAconn *conn, char *query)
{
    sword rc;

    aDebug(DEBUG_STORAGE, "ORACLE DB:[0x%08x] Execute: %s\n", conn, query);
	rc = OCIStmtPrepare(conn->stmth, conn->errh,
        (CONST text*)query, strlen(query), OCI_NTV_SYNTAX, OCI_DEFAULT);
    OCI_CHK(rc, conn->errh);
    rc = OCIStmtExecute(conn->svcctx, conn->stmth, conn->errh,
        1, 0, NULL, NULL, 0);
    OCI_CHK(rc, conn->errh);
    return (rc);
}

/*
 * Execute SELECT query, prepare row for results.
 */
static int
ora_select(ORAconn *conn, char *query)
{
    sword rc;
    ub4 prefrows = 0;

    if (conn->row != NULL)
        ora_free_row(conn->row);
    conn->row = NULL;
/*    aDebug(DEBUG_STORAGE, "ORACLE DB:[0x%08x] Execute: %s\n", conn, query); */
	rc = OCIStmtPrepare(conn->stmth, conn->errh,
        (CONST text*)query, strlen(query), OCI_NTV_SYNTAX, OCI_DEFAULT);
    OCI_CHK(rc, conn->errh);
    
    /*
     * Don't prefetch rows since we will defined
     * output vars later.
     */
    rc = OCIAttrSet(conn->stmth, OCI_HTYPE_STMT, &prefrows,
        0, OCI_ATTR_PREFETCH_ROWS, conn->errh);
    OCI_CHK(rc, conn->errh);
    rc = OCIStmtExecute(conn->svcctx, conn->stmth, conn->errh,
        0, 0, NULL, NULL, 0);
    OCI_CHK(rc, conn->errh);
    if (rc == OCI_SUCCESS)
        conn->row = ora_alloc_row(conn);

    return (rc);
}

/*
 * Fetch SELECT results. Returns one row at time.
 * Must be called after ora_select()
 */
static char **
ora_fetch(ORAconn *conn)
{
    sword rc;

    if (conn->row == NULL)
        return (NULL);

    /* data goes to conn->row */
	rc = OCIStmtFetch(conn->stmth, conn->errh, 1, OCI_FETCH_NEXT, OCI_DEFAULT);
    if (rc == OCI_SUCCESS)
        return (conn->row);
    else
        return (NULL);
}

static char **
ora_alloc_row(ORAconn *conn)
{
    ub4     col;
    char    **row;
    OCIParam *param;
    ub2     dtype, dsize;
    ub4     i;
    sword   rc;
    OCIDefine   *define;

    /*
     * Statement was executed and succedeed.
     * We need to figure out how many columns in set
     * and max length of each column.
     */
    rc = OCIAttrGet(conn->stmth, OCI_HTYPE_STMT, (dvoid *)&col,
        0, OCI_ATTR_PARAM_COUNT, conn->errh);
    OCI_CHK(rc, conn->errh);
    row = (char **)aMalloc(sizeof(char *) * (col + 1));
    for (i = 0; i < col; i++) {
        rc = OCIParamGet(conn->stmth, OCI_HTYPE_STMT, conn->errh,
            (dvoid **)&param, i + 1);
        OCI_CHK(rc, conn->errh);
        rc = OCIAttrGet((dvoid*)param, OCI_DTYPE_PARAM, 
            (dvoid*)&dtype, (ub4*)0, OCI_ATTR_DATA_TYPE,
            conn->errh);
        OCI_CHK(rc, conn->errh);
        dsize = DEF_DATASTR_LEN;

        /*
         * Use the retrieved length of dname to allocate an output
         * buffer, and then define the output variable (but only
         * for char/string type columns).
         */
        switch(dtype) {
        case SQLT_CHR:
        case SQLT_STR:
            rc = OCIAttrGet((dvoid*)param, (ub4) OCI_DTYPE_PARAM,
                (dvoid*) &dsize, (ub4 *)0, (ub4) OCI_ATTR_DATA_SIZE,
                conn->errh);
                OCI_CHK(rc, conn->errh);
                /* FALLTHROUGH */
        case SQLT_DAT:
        case SQLT_INT:
        case SQLT_UIN:
        case SQLT_FLT:
        case SQLT_PDN:
        case SQLT_BIN:
        case SQLT_NUM:
            row[i] = (char*)aMalloc(dsize + 1);
            break;
        default:
            dsize = 0;
            row[i]=NULL;
            break;
        }
        rc = OCIDefineByPos(conn->stmth, &define, conn->errh, i + 1,
            row[i], dsize + 1, SQLT_STR, NULL, NULL, NULL, OCI_DEFAULT);
        OCI_CHK(rc, conn->errh);
    }
    row[i] = NULL;

    return (row);
}

static void
ora_free_row(char **row)
{
    for (char **ptr = row; *ptr != NULL; ptr++) {
        aFree(*ptr);
    }
    aFree(row);
}

/*
 * Check if table 'tname' exists.
 * Return 1 if exists, 0 othewise.
 */
static int
ora_chk_table(ORAconn *conn, char *tname)
{
    char query[256], **row;
    sword rc;

    snprintf(query, sizeof(query), "SELECT * FROM cat "
        "WHERE table_name = '%s'", tname);
    rc = ora_select(conn, query);
    row = ora_fetch(conn);
    if (rc == OCI_SUCCESS) {
        aDebug(DEBUG_STORAGE, "ORACLE DB: table %s exists\n",  tname);
        return (1);
    } else {
        return (0);
    }
}

static void
ora_errprint(dvoid *errhp, ub4 htype, sb4 *errcodep)
{
  text errbuf[512];

  if (errhp != NULL)
  {
    sb4  errcode;

    if (errcodep == (sb4 *)0)
      errcodep = &errcode;

    (void) OCIErrorGet((dvoid *)errhp, (ub4) 1, (text *) NULL, errcodep,
                       errbuf, (ub4) sizeof(errbuf), htype);
    aLog(D_WARN, "OCI Error - %.*s\n", 512, errbuf);
  }
}

static void
ora_checkerr(dvoid *errhp, ub4 htype, sword status,
    text *note, sb4 state, text *file, sb4 line)
{
  sb4 errcode = 0;

  if (status == OCI_SUCCESS)
      return;

  (void) aLog(D_WARN, "OCI Error %ld occurred at File %s:%ld\n",
      (long)status, (char *)file, (long)line);

  switch (status)
  {
  case OCI_SUCCESS_WITH_INFO:
      (void) aLog(D_WARN, "Error - OCI_SUCCESS_WITH_INFO\n");
    ora_errprint(errhp, htype, &errcode);
    break;
  case OCI_NEED_DATA:
      (void) aLog(D_WARN, "Error - OCI_NEED_DATA\n");
    break;
  case OCI_NO_DATA:
      (void) aLog(D_WARN, "Error - OCI_NODATA\n");
    break;
  case OCI_ERROR:
    ora_errprint(errhp, htype, &errcode);
    break;
  case OCI_INVALID_HANDLE:
      (void) aLog(D_WARN, "Error - OCI_INVALID_HANDLE\n");
    break;
  case OCI_STILL_EXECUTING:
    (void) aLog(D_WARN, "Error - OCI_STILL_EXECUTE\n");
    break;
  case OCI_CONTINUE:
    (void) aLog(D_WARN, "Error - OCI_CONTINUE\n");
    break;
  default:
    break;
  }
}

#endif
//////////////////////////////////////////////////////////////////////////
