/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: s_storage.c,v 1.67.4.1 2005/02/18 18:02:56 anton Exp $ */

#include "netams.h"

//////////////////////////////////////////////////////////////////////////
void *sStorage(void *s);
void sStorageCancel(void *);
unsigned stRead(Service *s);
void stStore(Service *s);
//////////////////////////////////////////////////////////////////////////
void sStorageInit(Service *s){
	s->cfg = (ServiceStorage_cfg*)aMalloc(sizeof(ServiceStorage_cfg));

	ServiceStorage_cfg *cfg=(ServiceStorage_cfg*)s->cfg;
	cfg->type=UNKNOWN;
	cfg->hostname=cfg->username=cfg->password=cfg->dbname=cfg->socket=NULL;
	cfg->port=0;
	cfg->in=new FIFO(MAX_UNITS*5);
	cfg->out=new FIFO(MAX_UNITS*5);
	cfg->oids=cfg->raw=cfg->summary=NULL;
	cfg->db_path=NULL;
	
	pthread_create(&(s->t_id), NULL, &sStorage, s);
}

//////////////////////////////////////////////////////////////////////////
void sStorageProcessCfg(char *param[], Connection *conn, u_char no_flag){
	ServiceStorage_cfg *cfg=(ServiceStorage_cfg*)conn->service->cfg;

	if (!strcasecmp(param[0], "type")) {
		if(0) ;
#if defined(USE_HASH)
		else if (!strcasecmp(param[1], "hash")) {
			aParse(conn, "storage type is set to HASH\n");
			cfg->type=HASH;
		}
#endif
#if defined(USE_MYSQL)
		else if (!strcasecmp(param[1], "mysql")) {
			aParse(conn, "storage type is set to MySQL\n");
			cfg->type=MY_SQL;
		}
#endif
#if defined(USE_POSTGRES)
		else if (!strcasecmp(param[1], "postgres")) {
			aParse(conn, "storage type is set to Postgres\n");
			cfg->type=POSTGRES;
		}
#endif
#if defined(USE_ORACLE)
		else if (!strcasecmp(param[1], "oracle")) {
			aParse(conn, "storage type is set to Oracle\n");
			cfg->type=ORACLE;
		}
#endif
		else {
			aParse(conn, "storage type is unknown\n");
			cfg->type=UNKNOWN;
		}
	}		
	else if (!strcasecmp(param[0], "user")) {
		if (param[1]!=empty &&
            (cfg->type==MY_SQL || cfg->type==POSTGRES || cfg->type == ORACLE)) {
			if(cfg->username) aFree(cfg->username);
			cfg->username=set_string(param[1]);
			aParse(conn, "username is set to %s\n", cfg->username);
		} 
		else aParse(conn, "cannot set username for this storage type\n");
	}
	else if (!strcasecmp(param[0], "path")) {
		if (param[1]!=empty && cfg->type==HASH ) {
			if(cfg->username) aFree(cfg->username);
			cfg->username=set_string(param[1]);
			aParse(conn, "path is set to %s\n", cfg->username);
		} 
		else aParse(conn, "cannot set path for this storage type\n");
	}
	else if (!strcasecmp(param[0], "password")) {
		if (param[1]!=empty &&
            (cfg->type==MY_SQL || cfg->type==POSTGRES || cfg->type == ORACLE) ) {
			if(cfg->password) aFree(cfg->password);
			cfg->password=set_string(param[1]);
			aParse(conn, "password is set to %s\n", cfg->password);
		} 
		else aParse(conn, "cannot set password for this storage type\n");
	}
	else if (!strcasecmp(param[0], "host")) {
		if (param[1]!=empty && (cfg->type==MY_SQL || cfg->type==POSTGRES)) {
			if(cfg->hostname) aFree(cfg->hostname);
			cfg->hostname=set_string(param[1]);
			aParse(conn, "hostname is set to %s\n", cfg->hostname);
		} 
		else aParse(conn, "cannot set hostname for this storage type\n");
	}
	else if (!strcasecmp(param[0], "dbname")) {
		if (param[1]!=empty &&
            (cfg->type==MY_SQL || cfg->type==POSTGRES || cfg->type == ORACLE) ) {
			if(cfg->dbname) aFree(cfg->dbname);
			cfg->dbname=set_string(param[1]);
			aParse(conn, "dbname is set to %s\n", cfg->dbname);
		} 
		else aParse(conn, "cannot set dbname for this storage type\n");
	}
	else if (!strcasecmp(param[0], "socket")) {
		if (param[1]!=empty && (cfg->type==MY_SQL || cfg->type==POSTGRES) ) {
			if(cfg->socket) aFree(cfg->socket);
			cfg->socket=set_string(param[1]);
			aParse(conn, "socket is set to %s\n", cfg->socket);
		}
		else aParse(conn, "cannot set socket for this storage type\n");
	}
	 else if (!strcasecmp(param[0], "port")) {
		if (param[1]!=empty && (cfg->type==MY_SQL || cfg->type==POSTGRES) ) {
			cfg->port=strtol(param[1], NULL, 16);
			aParse(conn, "port is set to %u\n", cfg->port);
		}
		else aParse(conn, "cannot set port for this storage type\n");
	}
	else aParse(conn, "unknown storage command: %s\n", param[0]);
}
//////////////////////////////////////////////////////////////////////////
void *sStorage(void *ss){
	Service *s=(Service*)ss;

	aLog(D_INFO, "service storage:%u thread started (%p)\n", s->instance, s->cfg);
	pthread_cleanup_push(sStorageCancel, s);
	SET_CANCEL();
	s->t_id=pthread_self();

	ServiceStorage_cfg *cfg=(ServiceStorage_cfg*)s->cfg;
	
	// now we should sleep before starting storage service exec
	if(!(s->flags&SERVICE_RUN)) s->Sleep();

	switch (cfg->type) {
		case HASH:
		#ifdef USE_HASH
			{
				char hash_path[255];
				char *c;

				if (cfg->username) {
                			strcpy(hash_path, cfg->username);
				} else {
					c=strrchr(config_file_name, '/');
					if (c)
						strncpy(hash_path, config_file_name, (c-config_file_name));
					else
						strcpy(hash_path, "./");
				}
				print_to_string(&cfg->raw, "%s/netams-raw.db", hash_path);
				print_to_string(&cfg->summary, "%s/netams-sumary.db", hash_path);
			}
			aLog(D_INFO, "storage:%d working with HASH\n", s->instance);
				
			while (1) {
				//process input queue
				if(stLoadHash(s)) {
					//process output queue
					stSaveHash(s);
				}

				//this is cancel point
				s->Sleep();
			}
		#else 
			aLog(D_WARN, "storage type HASH is not compiled in!\n");
		#endif
			break;
		case POSTGRES:
		case MY_SQL:
        case ORACLE:
#if defined(USE_MYSQL) || defined(USE_POSTGRES) || defined(USE_ORACLE)
			print_to_string(&cfg->oids, "storage.%u.oids", s->instance);
			print_to_string(&cfg->raw, "storage.%u.raw", s->instance);
			print_to_string(&cfg->summary, "storage.%u.summary", s->instance);

			aLog(D_INFO, "storage:%d working with SQL\n", s->instance);
			
			//write previous data to database
			//there is guarantee this data will be written, at least in next writes
			//but no, that it will be counted propely if no database connection available
			stSaveSql(s,cfg->summary,ST_CONN_SUMMARY);

			while (1) {
				//process input queue
				if(stRead(s)) {
					//process output queue
					stStore(s);
				}
				s->Sleep();
			}
		#else	
			aLog(D_WARN, "storage type SQL is not compiled in!\n");
		#endif
			break;
		case UNKNOWN:
		default:
			aLog(D_WARN, "storage type undefined, service will be finished\n");
			break;
	}

	pthread_cleanup_pop(1);
	return NULL;
}

//////////////////////////////////////////////////////////////////////////
void sStorageCancel(void *v){
	Service *s = (Service*)v;
	ServiceStorage_cfg *cfg=(ServiceStorage_cfg*)s->cfg;
	//unregister storage in processor service
	st_unit *p=NULL;
	for(st_unit *st=ProcessorCfg.st_root;st!=NULL;st=st->next) {
		if(st->id==s->instance) {
			aLog(D_INFO, "unregistering storage:%d\n", s->instance);
                        if(st==ProcessorCfg.st_root) {
                                ProcessorCfg.st_root=st->next;
                                if(ProcessorCfg.st_root) {
                                        aLog(D_INFO, "using storage:%u as source for READ and STAT requests\n",ProcessorCfg.st_root->id);
                                }
                        } else
                                p->next=st->next;
                        aFree(st);
			break;
		}
		p=st;
	}
	aLog(D_INFO, "cancelling service storage:%u, flushing %u messages\n", s->instance, cfg->in->num_items);
	switch (cfg->type) {
	#ifdef USE_HASH
		case HASH:
			if(stLoadHash(s)) stSaveHash(s);
			break;
	#endif
#if defined(USE_MYSQL) || defined(USE_POSTGRES) || defined(USE_ORACLE)
		case POSTGRES:
		case MY_SQL:
        case ORACLE:
			if(stRead(s)) stStore(s);
			break;
	#endif
		default: break;
	}

	for(u_char type=0;type<ST_CONN_TYPES_NUM;type++)
		if(cfg->fd[type]) stCloseSql(s,(st_conn_type)type);

	delete cfg->in;
	delete cfg->out;
	if(cfg->username) aFree(cfg->username);
	if(cfg->password) aFree(cfg->password);
	if(cfg->dbname)   aFree(cfg->dbname);
	if(cfg->hostname) aFree(cfg->hostname);
	if(cfg->socket) aFree(cfg->socket);
	if(cfg->oids) 	aFree(cfg->oids);
	if(cfg->raw) 	aFree(cfg->raw);
	if(cfg->summary) aFree(cfg->summary);

	aFree(cfg);
}

//////////////////////////////////////////////////////////////////////////
void sStorageListCfg(Service *s, FILE *f, int nopasswords){
	ServiceStorage_cfg *cfg=(ServiceStorage_cfg*)s->cfg;

	switch (cfg->type) {
		case HASH: fprintf(f, "type hash\n"); 
					if (cfg->username) fprintf(f, "path %s\n", cfg->username); 
					break;

		case MY_SQL: 		
					fprintf(f, "type mysql\n"); 
					if (cfg->socket) fprintf(f, "socket %s\n", cfg->socket);
					if (cfg->hostname) fprintf(f, "host %s\n", cfg->hostname); 
					if (cfg->port && cfg->port!=3306) fprintf(f, "port %u\n", cfg->port);
					if (cfg->username) fprintf(f, "user %s\n", nopasswords?"***":cfg->username); 
					if (cfg->password) fprintf(f, "password %s\n", nopasswords?"***":cfg->password); 
					if (cfg->dbname) fprintf(f, "dbname %s\n", cfg->dbname); 
					break;
		case POSTGRES: 
					fprintf(f, "type postgres\n");
					if (cfg->socket) fprintf(f, "socket %s\n", cfg->socket);
					if (cfg->hostname) fprintf(f, "host %s\n", cfg->hostname);
					if (cfg->port && cfg->port!=5432) fprintf(f, "port %u\n", cfg->port);
					if (cfg->username) fprintf(f, "user %s\n", nopasswords?"***":cfg->username); 
					if (cfg->password) fprintf(f, "password %s\n", nopasswords?"***":cfg->password);
					if (cfg->dbname) fprintf(f, "dbname %s\n", cfg->dbname); 
					break;
    case ORACLE:
					fprintf(f, "type oracle\n"); 
					if (cfg->username) fprintf(f, "user %s\n", nopasswords?"***":cfg->username); 
 					if (cfg->password) fprintf(f, "password %s\n", nopasswords?"***":cfg->password);
					if (cfg->dbname) fprintf(f, "dbname %s\n", cfg->dbname); 
					break;
		default:	break;
	}

	fprintf(f, "\n");
}
//////////////////////////////////////////////////////////////////////////
unsigned stRead(Service *s) {
	ServiceStorage_cfg *cfg=(ServiceStorage_cfg*)s->cfg;
	Message *msg;
	Message_Read *rmsg;
	unsigned res=0;
	NetUnit *u;
	void *fd;
	FILE *oids=NULL;

	if(!cfg->out->TryPop()) return 1;
		
	if(!(fd=stOpenSql(s,ST_CONN_SUMMARY))) {
		return 0;
	}

	while((msg=cfg->out->TryPop())) {
		if (msg->type==MSG_READ) {
			rmsg=(Message_Read*)msg;
			if(!(u=Units.getUnitById(rmsg->netunit))) goto CLEAN;
			aDebug(DEBUG_STORAGE, "ST<-SQL/%c rd st:%u unit:%06X acct:%06X\n", rmsg->prefix, s->instance, rmsg->netunit, rmsg->ap);
			
			if(!u->ap) goto CLEAN; //somehow unit do not have account data

			//protect counters
			pthread_rwlock_wrlock(u->ap->rwlock);
			res=stLoadSql(s,fd,msg);
			pthread_rwlock_unlock(u->ap->rwlock);
			
			if(!res) break; // we cannot read

			if(!oids) {
				oids=fopen(cfg->oids,"a");
				if(!oids) {
					aLog(D_WARN, "Can't create temporary file %s: %s\n", cfg->oids, strerror(errno));
					res=0;
					break;
				}	
				setlinebuf(oids);
			}
			fprintf(oids, "%u,%s\n", u->id, u->name);
		} else {
			aDebug(DEBUG_STORAGE, "message type %d not READ - discarded\n",msg->type);
		}
CLEAN:
		msg=cfg->out->Pop();
		if(!msg->Active()) MsgMgr.Delete(msg); 
	}
	
	if(oids) {
		fclose(oids);
		stSaveSql(s,cfg->oids,ST_CONN_OIDS);
	}
	return res;
}
//////////////////////////////////////////////////////////////////////////
void stStore(Service *s) {
	ServiceStorage_cfg *cfg=(ServiceStorage_cfg*)s->cfg;
	Message *msg;
	Message_Store *smsg;
	FILE *raw = NULL;
	FILE *summary = NULL;
	unsigned num_raw=0;
	unsigned num_summary=0;

	while((msg=cfg->in->TryPop())) {
		if (msg->type==MSG_STORE) {
			smsg=(Message_Store*)msg;
			aDebug(DEBUG_STORAGE, "ST->SQL/%c wr st:%u unit:%06X acct:%06X from:%u to:%u\n", smsg->prefix, s->instance, smsg->netunit, smsg->ap, smsg->data->from, smsg->ts);
			switch (smsg->prefix){
				case 'F':
					if(!raw) {
						raw=fopen(cfg->raw,"a");
						if(!raw) {
							aLog(D_WARN, "Can't create temporary file %s: %s\n", cfg->raw, strerror(errno));
							return;
						}
						setlinebuf(raw);
					}
					fprintf(raw, "%u,%u,%u,%u,%llu,%llu\n", smsg->netunit, smsg->ap, smsg->data->from, smsg->ts, smsg->data->in, smsg->data->out);
					num_raw++;
					break;
				case 'H':
				case 'D':
				case 'W':
				case 'M':
					if(!summary) {
						summary=fopen(cfg->summary,"a");
						if(!summary) {
							aLog(D_WARN, "Can't create temporary file %s: %s\n", cfg->summary, strerror(errno));
							return;
						}
						setlinebuf(summary);
					}
					fprintf(summary, "%c,%u,%u,%u,%llu,%llu\n", smsg->prefix, smsg->netunit, smsg->ap, smsg->data->from, smsg->data->in, smsg->data->out);
					num_summary++;
					break;
			}
		} else {
			aDebug(DEBUG_STORAGE, "message type %d not STORE - discarded\n",msg->type);
 		}
		msg=cfg->in->Pop();
		if(!msg->Active()) MsgMgr.Delete(msg);
	}

	if(raw) {
		fclose(raw);
		aDebug(DEBUG_STORAGE, "SQL->HDD/raw %u queries\n", num_raw);
		stSaveSql(s,cfg->raw,ST_CONN_RAW);
	}	
	if(summary) {
		fclose(summary);
		aDebug(DEBUG_STORAGE, "SQL->HDD/summary %u queries\n", num_summary);
		stSaveSql(s,cfg->summary,ST_CONN_SUMMARY);
	}
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

