/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: s_server.c,v 1.36 2005/01/17 13:13:22 jura Exp $ */

#include "netams.h"

void *sServer(void *s);
void sServerCancel(void *v);
void *sServerConnection(void *c);
void sServerConnectionCancel(void *c_t);

//////////////////////////////////////////////////////////////////////////
typedef struct ServiceServer_cfg {
	unsigned short port;
	u_char onlylocalhost;
	u_short max_conn;
	unsigned num_connections;
	int server_socket;
} ServiceServer_cfg; 
//////////////////////////////////////////////////////////////////////////
void sServerInit(Service *s){
	s->cfg = (ServiceServer_cfg*)aMalloc(sizeof(ServiceServer_cfg));
	
	ServiceServer_cfg *cfg=(ServiceServer_cfg*)s->cfg;
	cfg->port=20000;
	cfg->onlylocalhost=1;
	cfg->max_conn=6;
	cfg->num_connections=0;
	cfg->server_socket=0;
	pthread_create(&(s->t_id), NULL, &sServer, s);
}
//////////////////////////////////////////////////////////////////////////
void sServerProcessCfg(char *param[], Connection *conn, u_char no_flag){

	ServiceServer_cfg *cfg=(ServiceServer_cfg*)conn->service->cfg;

	if (!strcasecmp(param[0], "listen")) {
		unsigned short port_t=strtol(param[1], NULL, 10);
		if (port_t>0 && port_t <65535) { 
			cfg->port=port_t;
			aLog(D_INFO, "server listen port set to %u\n", cfg->port);
			aParse(conn, "listen port set to %u\n", cfg->port);
		}
		else aParse(conn, "listen port must be between 1 and 65535\n");
	}
	else if (!strcasecmp(param[0], "login")) {
		if (!strcasecmp(param[1], "any")) {
			aLog(D_INFO, "server login permitted from any host\n");
			aParse(conn, "login permitted from any host\n");
			cfg->onlylocalhost=0;
		}
		else if (!strcasecmp(param[1], "local") || !strcasecmp(param[1], "localhost")) {
			aLog(D_INFO, "server login permitted from localhost only\n");
			aParse(conn, "login permitted from localhost only\n");
			cfg->onlylocalhost=1;
		}
		else aParse(conn, "login values are [any|local]\n");
	}
	else if (!strcasecmp(param[0], "max-conn")) {
		unsigned short maxconn_t=strtol(param[1], NULL, 10);
		if (maxconn_t>0 && maxconn_t <256) { 
			cfg->max_conn=maxconn_t;
			aLog(D_INFO, "server maximum connections number is set to %u\n", maxconn_t);
			aParse(conn, "maximum connections number is set to %u\n", maxconn_t);
		}
		else aParse(conn, "maximum connections number must be between 1 and 255\n");
	}
}
//////////////////////////////////////////////////////////////////////////
void *sServer(void *ss){

	Service *s=(Service*)ss; 
	int server_socket, connection_socket;
	int status, addrlen, reuseaddr_on=1;
	struct sockaddr_in server_addr;
	struct sockaddr_in connection_addr;
	char conn_name[32];

	aLog(D_INFO, "service server thread started\n");
	
	pthread_cleanup_push(sServerCancel, s);
	pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
	s->t_id=pthread_self();
	ServiceServer_cfg *cfg=(ServiceServer_cfg*)s->cfg;

	// now we should sleep before starting actual server
	if(!(s->flags&SERVICE_RUN)) s->Sleep();

	if ((server_socket=socket(AF_INET,SOCK_STREAM,0))==-1 ) 
		aLog(D_CRIT, "creation of server socket() failed: %d (%s)\n", server_socket, strerror(server_socket));

	bzero(&server_addr, sizeof(server_addr));
	server_addr.sin_family=AF_INET;
	server_addr.sin_addr.s_addr=htonl(cfg->onlylocalhost?INADDR_LOOPBACK:INADDR_ANY);
	server_addr.sin_port=htons((unsigned short)(cfg->port));
	// set option for reuse address
	setsockopt(server_socket, SOL_SOCKET, SO_REUSEADDR, (void*)&reuseaddr_on, sizeof(reuseaddr_on));
 
	// bind socket with serv_addr 
	status=bind(server_socket, (struct sockaddr*)&server_addr, sizeof(server_addr));
	if(status==-1 ) {
		aLog(D_CRIT, "bind of server socket failed: %u (%s) [%lx]\n[maybe NeTAMS is already running? issue 'ps -ax | grep netams']\n", errno, strerror(errno), server_addr.sin_addr.s_addr);
		return NULL;
	}
	status=listen(server_socket, SOMAXCONN);
	if(status==-1) {
		aLog(D_CRIT, "listen of server socket failed: %d (%s)\n", status, strerror(errno));	
		return NULL;
	}
	aLog(D_INFO, "server is listening\n");
	cfg->server_socket=server_socket;

	addrlen=sizeof(connection_addr);
	
	while(1) {
		bzero(&connection_addr, sizeof(connection_addr));
		connection_socket=accept(server_socket,(struct sockaddr*)(&connection_addr),(socklen_t*)(&addrlen));
		if (connection_socket==-1) {
			if(errno!=EINTR)
				aLog(D_CRIT, "accept of connection failed: %d (%s)\n", connection_socket, strerror(connection_socket));
			continue;
		}
		if (cfg->num_connections==cfg->max_conn) {
			aLog(D_WARN, "no more room for aditional connection\n");
			close(connection_socket); 
			continue; 
		}
		sprintf(conn_name, "conn%04d", cfg->num_connections);
		Connection *conn = new Connection(conn_name, connection_socket);
		bcopy(&connection_addr, conn->addr, sizeof(connection_addr));
		Connections.Insert(conn);
		sprintf(conn->name, "conn%04d", conn->conn_id);

		status = pthread_create(&(conn->t_id), NULL, &sServerConnection, conn);
		if (status != 0) 
			aLog(D_WARN, "Creation of connection %d listener thread failed: %d\n", connection_socket, status); 
	}
	return NULL;
	pthread_cleanup_pop(0);
}
//////////////////////////////////////////////////////////////////////////
void sServerCancel(void *v){
	Service *s = (Service*)v;
	ServiceServer_cfg *cfg=(ServiceServer_cfg*)s->cfg;
	
	//cancell all connected clients(e.g noy cInternal and not CONN_FD_VIRT
	aLog(D_INFO, "cancelling connections\n");
	Connections.DeleteAll(cInternal);

	aLog(D_INFO, "cancelling service server:%d\n", s->instance);
	shutdown(cfg->server_socket, SHUT_RDWR);
	close(cfg->server_socket);
	aFree(cfg);
}
//////////////////////////////////////////////////////////////////////////
void *sServerConnection(void *c_t){
	Connection *c = (Connection*)c_t;
	User *u=NULL;
	int i;
	aDebug(DEBUG_SERVER, "connection %s from %s\n", c->getName(), inet_ntoa(c->addr->sin_addr));
	pthread_cleanup_push(sServerConnectionCancel, c);
	pthread_detach(pthread_self());
//	SET_CANCEL();
	pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);

	char *buffer, *ret;
	buffer = (char*) aMalloc(4*1024+1); 

//	setlinebuf(c->stream_w);
//	setlinebuf(c->stream_r);
	c->permissions=UPERM_NONE; c->user=NULL;
	c->t_id=pthread_self();

	fprintf(c->stream_w, "\n"); aShowVersion(c->stream_w);
	c->Flush();
	
	for (i=0; i<3; i++) {
		fprintf(c->stream_w, "%s user login >", aaa_fw_software_name);
		c->Flush();
		ret=fgets(buffer, 4*1024, c->stream_r);
		if (ret==NULL) { if (errno==EINTR) i=4; break; }
		buffer[strlen(buffer)-2]=0;
//		printf("%s\n", buffer);
    
		u=Users.getUser(buffer);
    		
		if ( u && u->getPermissions() >= UPERM_LOGIN_WEB) {
       			fprintf(c->stream_w, "%s user %s password >", aaa_fw_software_name, u->name);
			c->Flush();
        		ret=fgets(buffer, 4*1024, c->stream_r);
        		if (ret==NULL) { if (errno==EINTR) i=4; break; }
        		buffer[strlen(buffer)-2]=0;
//			printf("%s\n", buffer);
         		if ((u->hidden && !u->crypted) || aAuth(u->name, buffer)>0) {
        			fprintf(c->stream_w, "\nWelcome, %s!\n\n", u->real_name?u->real_name:u->name);
				c->Flush();
         	    		c->user=u; 
				c->permissions=u->getPermissions(); 
				i=4;
		 	  	break;
			}
		}
        }

	if (c->permissions > UPERM_NONE) {

		aDebug(DEBUG_SERVER, "user %s authenticated\n", c->user->name);
		LogEvent(ACCESS, 0, c->user->id, 0, "user %s authenticated from %s permissions %s", c->user->name?c->user->name:(char*)"??", inet_ntoa(c->addr->sin_addr), permissions_list[u->getPermissions()]);
		while (i!=PARSE_EXIT && i!=PARSE_KILL && i!=PARSE_SHUTDOWN && i!=PARSE_RELOAD && !feof(c->stream_r)){
			if (!c->service) fprintf(c->stream_w, ">");
			else fprintf(c->stream_w, "%s:%d#", c->service->getName(),c->service->instance);
			c->Flush();
			ret=fgets(buffer, 4*1024, c->stream_r);
			if (ret==NULL) { if (errno==EINTR) i=PARSE_EXIT; break; }
			buffer[strlen(buffer)-2]=0;
			aDebug(DEBUG_SERVER, "> %s\n", buffer);
			i=aCommand(c, buffer);
			bzero(buffer, 4*1024);
		}  
	}

	aFree(buffer);
	pthread_cleanup_pop(1);
	if (i==PARSE_KILL || i==PARSE_SHUTDOWN || i==PARSE_RELOAD) { 
		termination(i);
	}
	return NULL;
}
//////////////////////////////////////////////////////////////////////////
void sServerConnectionCancel(void *c_t){
	Connection *c = (Connection*)c_t;
	aDebug(DEBUG_SERVER, "cancelling connection %s\n", c->getName());
	c->Flush();
	Connections.Delete(c);
	delete c;
}
//////////////////////////////////////////////////////////////////////////
void sServerListCfg(Service *s, FILE *f){
	ServiceServer_cfg *cfg=(ServiceServer_cfg*)s->cfg;
	fprintf(f, "login %s\n", cfg->onlylocalhost?"local":"any");
	fprintf(f, "listen %u\n", cfg->port);
	fprintf(f, "max-conn %u\n", cfg->max_conn);
	fprintf(f, "\n");
}
//////////////////////////////////////////////////////////////////////////
