/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: s_datasource.c,v 1.131.4.3 2005/02/25 18:07:39 anton Exp $ */

#include "netams.h"

void *sDataSource(void *s);
void sDSCancel(void *);
void sDS_RuleIp(u_short flag, Service *s, unsigned short number, char *rule); // flag==0 eq ADD, ==1 eq REMOVE
void AcctFlows(ServiceDS_cfg* cfg);
void NFSourceAdd(ServiceDS_cfg *cfg, char *param, u_char no_flag);
//////////////////////////////////////////////////////////////////////////
//ds's we support
void* ds_ipq(void *ss);
void* ds_ulog(void *ss);
void* ds_ipfw(void *ss);
void* ds_netflow(void *ss);
void* ds_libpcap(void *ss);
void* ds_mail(void *ss);

extern void ds_libpcap_stats(FILE *f, ServiceDS_cfg *cfg);
//////////////////////////////////////////////////////////////////////////
void sDSInit(Service *s){
	s->cfg = (ServiceDS_cfg*)aMalloc(sizeof(ServiceDS_cfg));

	ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;
	cfg->type=PT_UNKNOWN;
	
	cfg->id=s->instance;
	cfg->flags=DS_NONE;

	cfg->total_packets=0;
	cfg->total_flows=0;
	cfg->total_errors=0;
	cfg->timestamp=0;

	cfg->root=NULL;
	cfg->src_cmd=NULL;
	
	//netflow
	cfg->l_addr.s_addr=INADDR_ANY;
	cfg->port=0;
	cfg->nfroot=NULL;
	
	//libpcap
	cfg->pcap_data=NULL;
	
	//ulog
	cfg->nlmask=0;

	cfg->delay=0; cfg->skewdelay=0;
	for (u_short i=0; i<10; i++) { cfg->pc[i]=cfg->bc[i]=0; }

	cfg->IPtree = new IPTree(s->instance);
	cfg->flow = NULL;
        cfg->ds_fifo = NULL;
	cfg->msg=NULL;

	cfg->packet=(unsigned char*)aMalloc(MAX_PKT_SIZE);

	cfg->child=NULL;

	pthread_create(&(s->t_id), NULL, &sDataSource, s);
}

//////////////////////////////////////////////////////////////////////////
void sDSProcessCfg(char *param[], Connection *conn, u_char no_flag){
	ServiceDS_cfg *cfg=(ServiceDS_cfg*)conn->service->cfg;

	if (!strcasecmp(param[0], "type")) {
		if (!strcasecmp(param[1], "ip-traffic")) {
			aParse(conn, "data-source type is set to ip-traffic\n");
			cfg->type=PT_IP_TRAFFIC;
		}
		else if (!strcasecmp(param[1], "netflow")) {
			aParse(conn, "data-source type is set to cisco NetFlow\n");
			cfg->type=PT_NETFLOW_TRAFFIC;
			if (!cfg->port) cfg->port=20001;
			if(cfg->src_cmd) aFree(cfg->src_cmd);
			cfg->src_cmd=set_string("0.0.0.0");
		}
#ifndef NOPCAP
		else if (!strcasecmp(param[1], "libpcap")) {
			aParse(conn, "data-source type is set to libpcap\n");
			cfg->type=PT_LIBPCAP_TRAFFIC;
		}
#endif
		else if (!strcasecmp(param[1], "mailstat")) {
			aParse(conn, "data-source type is set to mailstat\n");
			cfg->type=PT_MAIL_STAT;
		}
		else aParse(conn, "data-source type is invalid\n");
	}
	else if (!strcasecmp(param[0], "rule")) {
		unsigned number=strtol(param[1], NULL, 10);
		ds_rule *t=NULL, *tt=NULL;

		if (cfg->type==PT_MAIL_STAT) return;

		for (t=cfg->root; t!=NULL; t=t->next) {
			if (t->number==number) break;
			tt=t;
		}

		if (no_flag && t) {
			aParse(conn, "rule %d removed\n", number);
			sDS_RuleIp(REMOVE, conn->service, number, t->rule_str);
			if( t->rule_str) aFree(t->rule_str);
			if(cfg->root==t) cfg->root=t->next;
			else tt->next=t->next;
			aFree(t);
		} //chain removal!!!!!!!
		else if (!no_flag && t) {
			sDS_RuleIp(REMOVE, conn->service, number, t->rule_str);
			if(t->rule_str) aFree(t->rule_str);
			t->rule_str=set_string(param[2]);
			aParse(conn, "rule %d replaced to %s\n", number, t->rule_str);
			sDS_RuleIp(ADD, conn->service, number, t->rule_str);
		}
		else if (!t) {
			t=(ds_rule*)aMalloc(sizeof(ds_rule));
			t->rule_str=set_string(param[2]);
			t->number=number;
			t->next=NULL;
			if(cfg->root) tt->next=t; else cfg->root=t;
			aParse(conn, "rule %d set to %s\n", number, t->rule_str);
			sDS_RuleIp(ADD, conn->service, number, t->rule_str);
		}
	}
	else if (!strcasecmp(param[0], "source")) {
		if (cfg->type==PT_MAIL_STAT) {
			if (!strcasecmp(param[1], "sendmail")) {
				cfg->src_cmd=set_string("sendmail");
				aParse(conn, "source is set to sendmail\n");
			}
			else aParse(conn, "source is invalid\n");
			return;
		}
#if defined(LINUX)
		else if (!strcasecmp(param[1], "ipq")) {
			cfg->src_cmd=set_string("ipq");
			cfg->flags=DS_IPQ;
			aParse(conn, "source is set to ipq\n");
		}
		else if (!strcasecmp(param[1], "ulog")) {
			unsigned nlmask=0;
			u_char i=2;
			while(param[i]!=empty && isdigit(param[i][0])) {
				u_char nlgroup=strtol(param[i], NULL, 10);
				if(nlgroup<1 || nlgroup>32)
					aParse(conn, "nlgroup has to be between 1 and 32\n");
				else 
					nlmask|=(1<<(nlgroup-1));
				
				i++;
			}
			if(nlmask && nlmask!=cfg->nlmask) {
				if(cfg->src_cmd) aFree(cfg->src_cmd);
				cfg->src_cmd=set_string("ulog");
				cfg->nlmask=nlmask;
				cfg->flags=DS_ULOG;
				aParse(conn, "source is set to ulog, nlmask=%u\n", cfg->nlmask);
			}
        	}
#endif
#if defined(FREEBSD)
		else if (!strcasecmp(param[1], "tee")) {
			unsigned short port=strtol(param[2], NULL, 10);
			if (!port) port=199;
			aParse(conn, "source is set to tee:%u\n", port);
			cfg->src_cmd=set_string("tee");
			cfg->port=port;
			cfg->flags=DS_TEE;
		}
		else if (!strcasecmp(param[1], "divert")) {
			unsigned short port=strtol(param[2], NULL,10);
			if (!port) port=199;
			aParse(conn, "source is set to divert:%u\n", port);
			cfg->src_cmd=set_string("divert");
			cfg->port=port;
			cfg->flags=DS_DIVERT;
		}
#endif
		else if (cfg->type == PT_NETFLOW_TRAFFIC) {
			if(INADDR_BROADCAST!=inet_addr(param[1])) {
				NFSourceAdd(cfg, param[1], no_flag);
				aParse(conn, "%s %s source %s accept list\n", (no_flag)?"Removing":"Adding", param[1],(no_flag)?"from":"to");
			}
		}
		else if (cfg->type==PT_LIBPCAP_TRAFFIC) {
			aParse(conn, "source is listening interface %s\n", param[1]);
			cfg->src_cmd=set_string(param[1]);
			cfg->port=0;
			if(!strcasecmp(param[2], "promisc")) cfg->flags=DS_PROMISC;
			else cfg->flags=DS_NONE;
		}
		else aParse(conn, "source is invalid or not available\n");
	}
	else if (!strcasecmp(param[0], "clock")) {
		if(cfg->type == PT_NETFLOW_TRAFFIC) {
			if (!strcasecmp(param[1], "remote")) {
				aParse(conn, "using remote flow time\n");
				cfg->flags=DS_CLOCK_REMOTE;
			} else if (!strcasecmp(param[1], "local")) {
				aParse(conn, "using local time\n");
				cfg->flags=DS_NONE;
			} else {
				aParse(conn, "Clock undefined! Defaulting to local time\n");
				cfg->flags=DS_NONE;
			}
		} else {
			aParse(conn, "Setting clock undefined for this data-source type\n");
		}
	}
	else if (!strcasecmp(param[0], "listen")) {
		if (cfg->type == PT_NETFLOW_TRAFFIC) {
			if(INADDR_BROADCAST!=inet_addr(param[1])) {
				inet_aton(param[1], &cfg->l_addr);
				if(param[2]!=empty) cfg->port=strtol(param[2], NULL, 10);
				if(cfg->src_cmd) aFree(cfg->src_cmd);
				cfg->src_cmd=set_string(param[1]);
				aParse(conn, "listening %s:%u\n", inet_ntoa(cfg->l_addr), cfg->port);
			} else {
				aParse(conn, "Command listen undefined for this data-source type\n");
                	}
		}
	}
	else aParse(conn, "unknown data-source command: %s\n", param[0]);
}
//////////////////////////////////////////////////////////////////////////
void *sDataSource(void *ss){
	Service *s=(Service*)ss;

	aLog(D_INFO, "service data-source:%u thread started (%p)\n", s->instance, s->cfg);
	pthread_cleanup_push(sDSCancel, s);
	SET_CANCEL();
	s->t_id=pthread_self();

	ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;

	// now we should sleep before starting data source service exec
	if(!(s->flags&SERVICE_RUN)) s->Sleep();
	
	//init units for this datasource
	//organize tree here
	{
	Units.tries_lock++;
	int err=pthread_rwlock_tryrdlock(Units.rwlock);
	if (err==EBUSY) { Units.tries_lock_failed++; pthread_rwlock_rdlock(Units.rwlock); }

	for (NetUnit *u=Units.root; u!=NULL; u=u->next)
		if(u->checkDSList(s->instance)) u->unit2tree(cfg->IPtree,ADD);

	if (err!=EDEADLK) pthread_rwlock_unlock(Units.rwlock);
	}

	switch (cfg->type) {
		case PT_IP_TRAFFIC:	// here we are assume that arrived data has an IP format
			cfg->ds_fifo = new DS_FIFO(10000);  //fifo should be initialized ealier then flow
			
			#if defined(LINUX) && !defined(IPTBL_NONE)
			if(cfg->flags==DS_IPQ) {
				cfg->flow = new Flow(s,1);
				cfg->child=new Service(SERVICE_DS_IPQ,s->instance);
				pthread_create(&(cfg->child->t_id), NULL, &ds_ipq, ss);
			} else if(cfg->flags==DS_ULOG) {
				cfg->flow = new Flow(s,0);
				cfg->child=new Service(SERVICE_DS_ULOG,s->instance);
				pthread_create(&(cfg->child->t_id), NULL, &ds_ulog, ss);
			}
			#endif
			#ifdef FREEBSD
				cfg->flow = new Flow(s,(cfg->flags==DS_DIVERT));
				cfg->child=new Service(SERVICE_DS_IPFW,s->instance);
				pthread_create(&(cfg->child->t_id), NULL, &ds_ipfw, ss);
			#endif
			Services.Insert(cfg->child);		

			//run account routines
			while(1) {
				AcctFlows(cfg);
 				s->Sleep(CHECK_EXPIRE);
			}
			break;
		case PT_NETFLOW_TRAFFIC:
                        ds_netflow(ss);
			break;
#ifndef NOPCAP
		case PT_LIBPCAP_TRAFFIC:
			cfg->ds_fifo = new DS_FIFO(10000);  //fifo should be initialized ealier then flow
			cfg->flow = new Flow(s,0);
			
			cfg->child=new Service(SERVICE_DS_LIBPCAP,s->instance);
			pthread_create(&(cfg->child->t_id), NULL, &ds_libpcap, ss);
			Services.Insert(cfg->child);

			// run account routines
			while(1) {
                                AcctFlows(cfg);
			 	s->Sleep(CHECK_EXPIRE);
                        }
			break;
#endif
		case PT_MAIL_STAT: 
			ds_mail(ss);
			break;
		default:	// this source type is not catched yet!
			aLog(D_ERR, "Data-source:%u type is undefined or not compiled\n", s->instance);
			break;
		}

	pthread_cleanup_pop(0);
	return NULL;
}
//////////////////////////////////////////////////////////////////////////
void sDSCancel(void *v){
	Service *s = (Service*)v;
	ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;
	
	ds_rule *ptr,*t=cfg->root;
	while(t) {
//		printf("removing %p (%p) %d %s\n", t, t->next, t->number, t->rule_str);
		sDS_RuleIp(REMOVE, s, t->number, t->rule_str);
		ptr=t; t=t->next;
		aFree(ptr);
	}
	if(cfg->child) {
		cfg->child->Shutdown();
		Services.Delete(cfg->child);
		delete cfg->child;
	}

	if(cfg->flow) delete cfg->flow;
	if(cfg->ds_fifo) {
		aLog(D_INFO, "cancelling service data-source:%u, flushing %u messages\n", s->instance, cfg->ds_fifo->num_items);
		s->flags|=SERVICE_RUN;  //this is workaround for auto-units
		AcctFlows(cfg);  //ensure we count collected data
		s->flags&=~SERVICE_RUN; //this is workaround for auto-units

		delete cfg->ds_fifo;
	}
	delete cfg->IPtree;
	
	if(cfg->pcap_data) aFree(cfg->pcap_data);
	if(cfg->packet) aFree(cfg->packet);
	if(cfg->src_cmd) aFree(cfg->src_cmd);
	aFree(cfg);
}

//////////////////////////////////////////////////////////////////////////
void sDSListCfg(Service *s, FILE *f){
	ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;

	if (cfg->type==PT_IP_TRAFFIC){
		fprintf(f, "type ip-traffic\n");
		
		fprintf(f, "source %s", cfg->src_cmd);

		if(cfg->flags==DS_ULOG) {
			for(u_char i=0;i<32;i++)
				if(cfg->nlmask&(1<<i)) fprintf(f, " %u", i+1);
		} 
		else if (cfg->flags==DS_TEE || cfg->flags==DS_DIVERT)
			fprintf(f, " %u", cfg->port);
		
		fprintf(f, "\n");
	}
	else if (cfg->type==PT_NETFLOW_TRAFFIC) {
		in_addr ina;

		fprintf(f, "type netflow\n");
		if (cfg->l_addr.s_addr != INADDR_ANY || cfg->port!=20001) 
			fprintf(f, "listen %s %u\n", inet_ntoa(cfg->l_addr), cfg->port);
		for(NFSource *nf=cfg->nfroot; nf!=NULL; nf=nf->next) {
			ina.s_addr=nf->src_addr;
			fprintf(f, "source %s\n", inet_ntoa(ina));
		}
		if (cfg->flags==DS_CLOCK_REMOTE) fprintf(f, "clock remote\n");
		fprintf(f, "\n");
		return;
	}
	else if (cfg->type==PT_LIBPCAP_TRAFFIC){
		fprintf(f, "type libpcap\n");
		fprintf(f, "source %s %s\n", cfg->src_cmd, (cfg->flags==DS_PROMISC)?"promisc":"");
	}
	else if (cfg->type==PT_MAIL_STAT)
		fprintf(f, "type mailstat\n");
	
	for (ds_rule *r=cfg->root; r!=NULL; r=r->next) 
		if (r->rule_str) fprintf(f, "rule %u \"%s\"\n", r->number, r->rule_str);
	
	fprintf(f, "\n");
}
//////////////////////////////////////////////////////////////////////////
void sDS_RuleIp(u_short flag, Service *s, u_short number, char *rule) {
	ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;
	
	if (cfg->type==PT_IP_TRAFFIC) {
		char *buffer = (char *)aMalloc(255);
		
		switch (flag) {
		case ADD: // add a rule to the system
			#ifdef LINUX
			sprintf(buffer,"/sbin/iptables -I %s 2>&1 >/dev/null", rule);
			#else
			sprintf(buffer,"/sbin/ipfw add %u %s %u %s 2>&1 >/dev/null", number, cfg->src_cmd, cfg->port, rule);
			#endif
			
			break;
		case REMOVE:  // remove a rule from the system
			#ifdef LINUX
			sprintf(buffer, "/sbin/iptables -D %s 2>&1 >/dev/null", rule);
			#else
			sprintf(buffer,"/sbin/ipfw delete %u 2>&1 >/dev/null", number);
			#endif
	
			break;
		default: 
			break;
		}
		system(buffer);
		aFree(buffer);
	}
	aLog(D_INFO, "%s rule %u \"%s\"\n", (flag==ADD)?"adding":"removing", number, rule);
}
//////////////////////////////////////////////////////////////////////////
void NFSourceAdd(ServiceDS_cfg *cfg, char *param, u_char no_flag) {
	NFSource *nf, *p=NULL;
	in_addr ina;
	
	inet_aton(param, &ina);

	for(nf=cfg->nfroot; nf!=NULL; nf=nf->next) {
		if(nf->src_addr==ina.s_addr) break;
		p=nf;
	}

	if(no_flag) { //REMOVE
		if(!nf) return;
		if(cfg->nfroot==nf) cfg->nfroot=cfg->nfroot->next;
		else p->next=nf->next;
		aFree(nf);
	} else { //ADD
		if(nf) return;
		nf=(NFSource*)aMalloc(sizeof(NFSource));
		nf->src_addr=ina.s_addr;
		if(cfg->nfroot==NULL) cfg->nfroot=nf;
		else p->next=nf;
	}
}
//////////////////////////////////////////////////////////////////////////
void cShowDS(Connection *conn){
        Service *s=NULL;

        while((s=Services.getServiceNextByType(SERVICE_DATASOURCE,s))) {
		ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;
		char *pt;
                switch (cfg->type){
                        case PT_IP_TRAFFIC: pt="IP_FILT"; break;
                        case PT_NETFLOW_TRAFFIC: pt="NETFLOW"; break;
                        case PT_LIBPCAP_TRAFFIC: pt="LIBPCAP"; break;
                        default : pt="<\?\?>"; break;
                }

		fprintf(conn->stream_w, " Data-source ID=%d type %s source %s:%u loop %llu average %d mcsec\n", s->instance,  pt, cfg->src_cmd?cfg->src_cmd:"<>", cfg->port, cfg->total_packets, (cfg->total_packets)?(unsigned)(cfg->delay/cfg->total_packets):0);
                int avg_pps=0; for (u_char i=0; i<10; i++) avg_pps+=cfg->pc[i]; avg_pps=avg_pps/9;
                long avg_bps=0; for (u_char i=0; i<10; i++) avg_bps+=cfg->bc[i]; avg_bps=avg_bps/9;
                fprintf(conn->stream_w, "    Perf: average skew delay %lu mcsec, PPS: %u, BPS: %lu\n", cfg->skewdelay, avg_pps, avg_bps);
		//show IP tree
		IPTree *IPtree=cfg->IPtree;
		fprintf(conn->stream_w, "    IP tree: %lu nodes [%u] + %lu dlinks [%u] + %lu unodes [%u] = %lu bytes\n",IPtree->nodes,sizeof(IPTree_node),IPtree->dlink,256*sizeof(IPTree_node*),IPtree->unodes,sizeof(dsUlist),IPtree->nodes*sizeof(IPTree_node)+IPtree->dlink*256*sizeof(IPTree_node*)+IPtree->unodes*sizeof(dsUlist));
		//show flow
		Flow *flow=cfg->flow;
       		if(flow) {
			fprintf(conn->stream_w, "    Flows: %lu/%lu act/inact entries (%lu bytes), %llu flows sent in %llu messages\n", flow->e_used, flow->e_total-flow->e_used, flow->e_total*sizeof(entry),flow->sent_flows,flow->sent_packets);
			//show hash
			//gather flow hash statistic
        		entry *e;
			entry **table=flow->table;
        		unsigned used=0;
        		unsigned max_chain=0;
        		unsigned tmp;
        		unsigned long total=0;

			for(unsigned i=0;i<IPV4_HASH_SIZE;i++) {
                		if(!(e=table[i])) continue;
                		used++;
                		tmp=0;
				for(;e!=NULL;e=e->next) {
					tmp++;
				}
				total+=tmp;
				if(max_chain<tmp) max_chain=tmp;
        		}

			fprintf(conn->stream_w, "    HASH: size=%u, %lu flows hashed, %u nodes used, max chain= %u\n",IPV4_HASH_SIZE,total,used,max_chain);
		}

		//show fifo
		DS_FIFO *fifo=cfg->ds_fifo;
		if(fifo) fprintf(conn->stream_w, "    FIFO: %u/%u used/ready messages, each %u, total %u bytes\n", fifo->num_items,fifo->ready_items, sizeof(IPStatMsg),(fifo->num_items+fifo->ready_items)*sizeof(IPStatMsg));
		
		//show sources info
		if(cfg->type==PT_NETFLOW_TRAFFIC) {
			char buf[32];
			in_addr ina;
			fprintf(conn->stream_w, "    Total: %llu packets, %llu flows, %u errors, timestamp - %s\n", cfg->total_packets, cfg->total_flows, cfg->total_errors, (cfg->timestamp)?timeU2T((time_t)cfg->timestamp, buf):"never");
			for(NFSource *nf=cfg->nfroot; nf!=NULL; nf=nf->next) {
				ina.s_addr=nf->src_addr;
				fprintf(conn->stream_w, "    Source %s: %llu packets, %llu flows, %u errors, timestamp - %s\n", inet_ntoa(ina), nf->packets, nf->flows, nf->errors, (nf->timestamp)?timeU2T((time_t)nf->timestamp, buf):"never");
			}
#ifndef NOPCAP
		} else if (cfg->type==PT_LIBPCAP_TRAFFIC) {
			ds_libpcap_stats(conn->stream_w, cfg);
#endif
		}
	}
}
//////////////////////////////////////////////////////////////////////////
void AcctFlows(ServiceDS_cfg* cfg) {
	DS_FIFO *fifo=cfg->ds_fifo;
	IPTree *IPtree=cfg->IPtree;

	IPStatMsg *msg;
	FlowStatHdr *hdr;
        IPFlowStat *flow;

	while((cfg->msg=fifo->Pop(cfg->msg))) {
		msg=cfg->msg;
 		hdr = &msg->header;
       		if (hdr->version==FLOW_VERSION) {
//              	aDebug(DEBUG_DS_MUX, "fl seq %lu, %d records:\n",hdr->flow_sequence, hdr->count);
                	aDebug(DEBUG_DS_MUX, " Flow v%d:%02X packet seq %lu with %d records:\n", hdr->version, hdr->engine_id, hdr->flow_sequence, hdr->count);
			for (unsigned short i = 0; i < hdr->count; i++) {
                       		flow = &msg->records[i];
#ifdef DEBUG
                       		char *b1, buf_1[32], *b2, buf_2[32]; // for debugging

               			b1=inet_ntoa(flow->srcaddr); strncpy(buf_1, b1, strlen(b1)+1);
               			b2=inet_ntoa(flow->dstaddr); strncpy(buf_2, b2, strlen(b2)+1);

                       		aDebug(DEBUG_DS_MUX, " IP time %lu src:%s,dst:%s,n:%lu,l:%lu,p:%u,sp:%u,dp:%u\n", flow->Last, buf_1, buf_2, flow->dPkts, flow->dOctets, flow->prot, ntohs(flow->srcport), ntohs(flow->dstport));
#endif
               			IPtree->ACCT(flow);
       			}
       		} else aDebug(DEBUG_DS_MUX, " Flow v%d packet unsupported\n", hdr->version);
		TEST_CANCEL( );
	}
}
//////////////////////////////////////////////////////////////////////////
unsigned sDSMeasure(ServiceDS_cfg* cfg, struct timeval *start, unsigned len) {
	u_char i;
	struct timeval stop;
	unsigned diff;

	gettimeofday(&stop, NULL);
         
	diff= (unsigned) ( (stop.tv_sec - start->tv_sec)*1000000 + (stop.tv_usec - start->tv_usec));      
	
	cfg->total_packets++;
	cfg->delay += diff;
	cfg->skewdelay = (unsigned)(cfg->skewdelay * 98 + diff * 2)/100;
	
	i = start->tv_sec % 10;
	cfg->pc[i]++;
	cfg->bc[i] += len;

	i++;
	if (i == 10) i = 0;
	cfg->pc[i] = 0;
	cfg->bc[i] = 0;
	
	aDebug(DEBUG_DS_IP, "ds:%u packet processed in %u mcsec\n",cfg->id, diff);

	return diff;
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
