/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: s_billing.c,v 1.102.4.4 2005/03/24 19:53:14 anton Exp $ */

#include "netams.h"

char *buildforstring=NULL;

PlansL *bPlans=NULL;
SubPlansL *bSubPlans=NULL;
AccountsL *bAccounts=NULL;

Service *Billing=NULL;
Billing_cfg *BillingCfg=NULL;

extern FeeCounters FC;
//////////////////////////////////////////////////////////////////////////////////////////
void *sBilling(void *s);
void sBiCancel(void *);
void sBiCheckDb(int id, Billing_cfg *c);
void sBiSetCfg(char *param[], Connection *conn, Billing_cfg *c);
void sBiSendAlert(NetUnit *u, u_char dir);
void AccountData(Billing_cfg* cfg);
void BiSyncDb(Billing_cfg* cfg);
//////////////////////////////////////////////////////////////////////////////////////////
void sBiInit(Service *s){
//init Plans, SubPlans and Accounts data
	bPlans = new PlansL();
	bSubPlans = new SubPlansL();
	bAccounts = new AccountsL();
	
	Billing_cfg *cfg = (Billing_cfg*)aMalloc(sizeof(Billing_cfg));
	cfg->delay=S_BILLING_DEF_delay;
	cfg->st=NULL;
	cfg->storage=0; 
	cfg->st_cat=ST_NONE;
	cfg->rwlock=(pthread_rwlock_t*)aMalloc(sizeof (pthread_rwlock_t));
	pthread_rwlock_init(cfg->rwlock, NULL);
	cfg->fifo = new FIFO(MAX_UNITS);
	ProcessorCfg.fifo=cfg->fifo;
	s->cfg=cfg;
	BillingCfg=cfg;
	print_to_string(&cfg->bfilename, "billing.%u", s->instance);
	print_to_string(&cfg->sfilename, "bdata.%u", s->instance);
	Billing=s;
	pthread_create(&(s->t_id), NULL, &sBilling, s);
}
//////////////////////////////////////////////////////////////////////////////////////////
void sBiProcessCfg(char *param[], Connection *conn, u_char no_flag){

	Service *s=Billing;
	if(!s) return;
	
	time_t now=FC.now;
	Billing_cfg *cfg=(Billing_cfg*)s->cfg;
	
	pthread_rwlock_wrlock(cfg->rwlock);

	if (!strcasecmp(param[0], "subplan")) {
		SubPlan *sp;

		u_char sp_id=strtol(param[1], NULL, 10);
		if (!sp_id) { goto END; }

		if ((sp=bSubPlans->Check(sp_id))==NULL) { 
			sp=new SubPlan(sp_id);
			bSubPlans->Add(sp);
			aParse(conn, "subplan %u created\n", sp_id);
		} else if (no_flag) {
			if(sp->connected_plans) {
				aParse(conn, "subplan connected to %u plans\n", sp->connected_plans);
			} else {
				bSubPlans->Delete(sp);
				delete sp;
				aParse(conn, "subplan %u deleted\n", sp_id);
			}
			goto END;
		}
	
		if (!strcasecmp(param[2], "fee")) {
			sp->fee=(float)strtod(param[3], NULL);
			aParse(conn, "subplan %u periodic fee set: %f\n", sp->id, sp->fee);
			if (!strcasecmp(param[4], "spread")) {
				if (!strcasecmp(param[5], "monthly")) { sp->spread='M'; aParse(conn, "subplan %u spread monthly\n", sp->id); }
				else if (!strcasecmp(param[5], "daily")) { sp->spread='D'; aParse(conn, "subplan %u spread daily\n", sp->id); }
				else if (!strcasecmp(param[5], "hourly")) { sp->spread='H'; aParse(conn, "subplan %u spread daily\n", sp->id); }
				else aParse(conn, "subplan %u spread specification invalid\n", sp->id);
			}
			else aParse(conn, "subplan %u fee command invalid\n", sp->id);
		} // fee
		else if (!strcasecmp(param[2], "included")) {
			if (!strcasecmp(param[4], "in")) {
				if (!strcasecmp(param[3], "unlimited")) { sp->flags|=UNLIM_IN; sp->inc_in=0; }
				else { sp->flags&=~UNLIM_IN; sp->inc_in=bytesT2Q(param[3]); }
			}
			if (!strcasecmp(param[6], "out")) {
				if (!strcasecmp(param[5], "unlimited")) { sp->flags|=UNLIM_OUT; sp->inc_out=0; }
				else { sp->flags&=~UNLIM_OUT; sp->inc_out=bytesT2Q(param[5]); }
			}
			aParse(conn, "subplan %u included %llu in, %llu out\n", sp_id, sp->inc_in, sp->inc_out);
			aParse(conn, "unlimited %s, %s\n", sp->flags&UNLIM_IN?"in":"none" ,sp->flags&UNLIM_OUT?"out":"none");
		} // included
		else if (!strcasecmp(param[2], "adjust-included")) {
			unsigned adj=0;
			if (!strcasecmp(param[3], "no")) adj=0;
			else if (!strcasecmp(param[3], "yes")) adj=1;
			aParse(conn, "subplan %u adjust included: %s\n", adj?"yes":"no");
			sp->inc_adjust=adj;
		} // adjust-included
		else if (!strcasecmp(param[2], "policy") && param[3]!=empty) {
			Policy *p;
			policy_flag flags=POLICY_FLAG_NONE;
			char *c_param=param[3];

			if (c_param[0]=='!') { flags|=POLICY_FLAG_INV; c_param++; }
			if (c_param[0]=='%') { flags|=POLICY_FLAG_BRK; c_param++; }
			if (c_param[0]=='!') { flags|=POLICY_FLAG_INV; c_param++; }

			if ((p=PolicyL.getPolicy(c_param))) {
				sp->pid=p->id;
				//clear pervious policy flags
				sp->flags&=~(POLICY_FLAG_INV|POLICY_FLAG_BRK); 
				//set new ones
				sp->flags|=flags;
				aParse(conn, "subplan %u policy set to %s%s%s\n", sp->id, (sp->flags&POLICY_FLAG_INV)?"!":"", (sp->flags&POLICY_FLAG_BRK)?"%":"", c_param);
			}
			else aParse(conn, "subplan %u policy unknown\n", sp->id);
		} // policy
		else if (!strcasecmp(param[2], "overdraft")) {
			if (!strcasecmp(param[4], "in")) {
				sp->pay_in=(float)strtod(param[3], NULL);
				sp->overdraft_in=(double)(sp->pay_in)/MEGABYTE;
			}
			if (!strcasecmp(param[6], "out")) {
				sp->pay_out=(float)strtod(param[5], NULL);
				sp->overdraft_out=(double)(sp->pay_out)/MEGABYTE;
			}
			aParse(conn, "subplan %u overdraft %f in, %f out\n", sp->id, sp->pay_in, sp->pay_out);
		} // overdraft

 		else aParse(conn, "subplan %u command invalid\n", sp->id);
	} // subplan
	else if (!strcasecmp(param[0], "plan")) {
		Plan *pl;

		u_char pl_id=strtol(param[1], NULL, 10);
		if ((pl=bPlans->Check(pl_id))==NULL) { 
			pl=new Plan(pl_id);
			bPlans->Add(pl);
			aParse(conn, "plan %06X created\n", pl_id);
		}  else if (no_flag) {
			if(pl->connected_accounts) {
				aParse(conn, "plan connected to %u accounts\n", pl->connected_accounts);
			} else {
				bPlans->Delete(pl);
				delete pl;
				aParse(conn, "plan %u deleted\n", pl_id);
			}
			goto END;
		}

		if (!strcasecmp(param[2], "name") && param[3]!=empty) {
			if (pl->name) aFree(pl->name);
			pl->name=set_string(param[3]);
			aParse(conn, "plan %u name set to %s\n", pl_id, pl->name);
		} //name
		else if (!strcasecmp(param[2], "description")) {
			if (pl->description) aFree(pl->description);
			pl->description=set_string(param[3]);
			aParse(conn, "plan %u description set to \"%s\"\n", pl_id, pl->description);
		} //description
		else if (!strcasecmp(param[2], "subplan")) {
			u_char i=3, j=0; SubPlan *s;
			while ((j=strtol(param[i], NULL, 10))) { 
				if ((s=bSubPlans->Check(j))) {
					if(pl->AddSubPlan(s,ADD))
						aParse(conn, "plan %u subplan %u added\n", pl_id, s->id);
				}
				else aParse(conn, "plan %u subplan %u not exist\n", pl_id, j);
				i++;
			}
		} //subplan
 		else aParse(conn, "plan %u command invalid\n", pl->id);
	} // plan
	else if (!strcasecmp(param[0], "account")) {
		Account  *ac;

		ac=bAccounts->Check(param[1]);
		if (ac==NULL) {
			if(no_flag) goto END;
			ac=new Account();
			ac->setName(param[1]);
			bAccounts->Add(ac);
			aParse(conn, "account %06X created\n", ac->id);
		} else {
			if(no_flag) {
				aParse(conn, "account %06X deleted\n", ac->id);
				ac->status|=ACCOUNT_DELETED;
				ac->created=0;  //this means deleted
				ac->status|=ACCOUNT_NEED_SYNC;
			}
		}
		ac->changed=now;

		if (!strcasecmp(param[2], "name") && param[3]!=empty) {
			ac->setName(param[3]);		
			ac->status|=ACCOUNT_NEED_SYNC;
			aParse(conn, "account %06X name set to %s\n", ac->id, ac->name);
		} //name
		else if (!strcasecmp(param[2], "description") && param[3]!=empty) {
			if (ac->description) aFree(ac->description);
			ac->description=set_string(param[3]);
			ac->status|=ACCOUNT_NEED_SYNC;
			aParse(conn, "account %06X description set to %s\n", ac->id, ac->description);
		} //description
		else if (!strcasecmp(param[2], "email") && param[3]!=empty) {
			if (ac->email) aFree(ac->email);
			ac->email=set_string(param[3]);
			ac->status|=ACCOUNT_NEED_SYNC;
			aParse(conn, "account %06X email set to %s\n", ac->id, ac->email);
		} //email
		else if (!strcasecmp(param[2], "password") && param[3]!=empty) {
			if (ac->password) aFree(ac->password);
			ac->password=set_string(param[3]);
			ac->status|=ACCOUNT_NEED_SYNC;
			aParse(conn, "account %06X password set to %s\n", ac->id, ac->password);
		} //password 
		else if (!strcasecmp(param[2], "plan") && param[3]!=empty) {
		    Plan *pl=bPlans->Check(strtol(param[3], NULL, 10));
			if (pl) {
				ac->Update(pl);
				ac->ChargeFee();
				aParse(conn, "account %06X plan set to %u %s\n", ac->id, pl->id, pl->name?pl->name:"<\?\?>");
			}
			else aParse(conn, "account %06X no plan %s exist\n", ac->id, param[3]);
		} //plan 
		else if (!strcasecmp(param[2], "nextplan") && param[3]!=empty) {
			Plan *pl=bPlans->Check(strtol(param[3], NULL, 10));
			if (pl)	{
				ac->nextplan=pl;
				ac->status|=ACCOUNT_NEED_SYNC;
				ac->nextplan_ch=now;
				aParse(conn, "account %06X next plan set to %u %s\n", ac->id, pl->id, pl->name?pl->name:"<\?\?>");
			} else aParse(conn, "account %06X no plan %s exist\n", ac->id, param[3]);
		} // next plan 
		else if (!strcasecmp(param[2], "beblock")) {
			ac->UpdateStatus(AC_BEBLOCK, now);
			aParse(conn, "account %06X block pending\n", ac->id);
		}
		else if (!strcasecmp(param[2], "block")) {
			ac->UpdateStatus(AC_BLOCK, now);
			aParse(conn, "account %06X blocked\n", ac->id);
		} //block 
		else if (!strcasecmp(param[2], "unblock")) {
			ac->UpdateStatus(AC_UNBLOCK, now);
			ac->ChargeFee();
			aParse(conn, "account %06X unblocked\n", ac->id);
		} //unblock 
		else if (!strcasecmp(param[2], "balance")) {
			double delta;
			char *act;
			delta=(double)strtod(param[4], NULL);

			if (!strcasecmp(param[3], "add")) {
				ac->Balance(delta, BAL_ADD);
				act="ADD";
			}
			else if (!strcasecmp(param[3], "remove")) {
				ac->Balance(delta, BAL_REMOVE);
				act="REMOVE";
			}
			else if (!strcasecmp(param[3], "set")) {
				ac->Balance(delta, BAL_SET);
				act="SET";
			} else {
				aParse(conn, "wrong balance command: %s\n", param[3]);
				goto END;
			}
			
			char action[64];
			sprintf(action,"BALANCE %s %lf ",act, delta);
			sPvmSend(ac, action);

			LogEvent(BILLING, 0, 0, ac->id, "balance %s %lf now %lf", act, delta, ac->balance);
			aParse(conn, "account %s (%06X) balance %s = %lf, now %lf\n", ac->name, ac->id, param[3], delta, ac->balance);
		} //balance 
		else if (!strcasecmp(param[2], "unit")) {
			NetUnit *u=NULL;
			if (!strcasecmp(param[3], "name")) u=Units.getUnit(param[4]);
			else if (!strcasecmp(param[3], "oid")) u=Units.getUnitById(strtol(param[4], NULL, 16));
			if (!u) {
				aParse(conn, "Unknown unit\n");
				goto END;
			}
			if (!strcasecmp(param[5], "add")) {
				if(!u->account) {
					if(ac->AddUnit(u, ADD))
						aParse(conn, "account %06X add unit %06X (%s)\n", ac->id, u->id, u->name?u->name:"<\?\?>");
				} else {
					aParse(conn, "Unit %s(%06X) already belongs to account %s(%06X)\n", u->name, u->id, ac->name, ac->id);
				}
			} // add
			else if (!strcasecmp(param[5], "delete")){
				if(ac->AddUnit(u, REMOVE)) 
					aParse(conn, "account %06X delete unit %06X (%s)\n", ac->id, u->id, u->name?u->name:"<\?\?>");
			} //delete
		} //unit
	} // account		  
	else if (!strcasecmp(param[0], "delay")) {
		unsigned t;
		t=strtol(param[1], NULL, 10);
		if (t<1 || t>60*60*24*7) goto END; // disallow negative and more than one week
		cfg->delay=t;
		aParse(conn, "billing control check delay set to %ld seconds\n", t);
	}
	else if (!strcasecmp(param[0], "storage")) {
		u_char st=strtol(param[1], NULL, 10);
		cfg->storage=st;
		aParse(conn, "billing storage number set to %u\n", st);
		cfg->st_cat=0;
		for(u_char j=2;param[j]!=empty;j++) {
			if (!strcasecmp(param[j], "all")) {
				cfg->st_cat=ST_ALL;
				break;
			}
			if (!strcasecmp(param[j], "plans")) cfg->st_cat|=ST_PLANS;
			if (!strcasecmp(param[j], "subplans")) cfg->st_cat|=ST_SUBPLANS;
		}
	}
	else aParse(conn, "unknown billing command: %s\n", param[0]);

END:	
	pthread_rwlock_unlock(cfg->rwlock);
}
//////////////////////////////////////////////////////////////////////////////////////////

void sBiListCfg(Service *s, FILE *f){ 

	Billing_cfg *cfg=(Billing_cfg*)s->cfg;
	char b1[32], b2[32];

	if (!cfg->rwlock) return;
        pthread_rwlock_rdlock(cfg->rwlock);

	SubPlan *sp;
	Policy *p;
	for (sp=bSubPlans->root; sp!=NULL; sp=sp->next){
		if (sp->spread=='M') fprintf(f, "subplan %u fee %f spread monthly\n", sp->id, sp->fee);
		if (sp->spread=='D') fprintf(f, "subplan %u fee %f spread daily\n", sp->id, sp->fee);
		if (sp->spread=='H') fprintf(f, "subplan %u fee %f spread hourly\n", sp->id, sp->fee);
		fprintf(f, "subplan %u included %s in %s out\n", sp->id, sp->flags&UNLIM_IN?"unlimited":bytesQ2T(sp->inc_in, b1), sp->flags&UNLIM_OUT?"unlimited":bytesQ2T(sp->inc_out, b2));
		if (sp->inc_adjust) fprintf(f, "subplan %u adjust-included yes\n", sp->id);
		if((p=PolicyL.getPolicyById(sp->pid))) fprintf(f, "subplan %u policy %s%s%s\n", sp->id, (sp->flags&POLICY_FLAG_INV)?"!":"", (sp->flags&POLICY_FLAG_BRK)?"%":"", p->name);
		fprintf(f, "subplan %u overdraft %f in %f out\n", sp->id, sp->pay_in, sp->pay_out);
	}

	Plan *pl;
	for (pl=bPlans->root; pl!=NULL; pl=pl->next){
		fprintf(f, "plan %u name %s\n", pl->id, pl->name?pl->name:"<\?\?>");
		fprintf(f, "plan %u description \"%s\"\n", pl->id, pl->description?pl->description:"<\?\?>");
		if (pl->root) {
			fprintf(f, "plan %u subplan ", pl->id);
			for (bSPlist *spl=pl->root; spl!=NULL; spl=spl->next)
				fprintf(f, "%u ", spl->sp->id); 
			
			fprintf(f, "\n");
		}
	}

	if (cfg->delay!=S_BILLING_DEF_delay) fprintf(f, "delay %u\n", cfg->delay);
	fprintf(f, "storage %u", cfg->storage);
	if(cfg->st_cat==ST_ALL) fprintf(f, " all");
	else {
		if (cfg->st_cat&ST_PLANS) fprintf(f, " plans");
		if (cfg->st_cat&ST_SUBPLANS) fprintf(f, " subplans");
	}
	fprintf(f, "\n\n");
	
	pthread_rwlock_unlock(cfg->rwlock);
}

//////////////////////////////////////////////////////////////////////////////////////////
void *sBilling(void *ss){
	Service *s=(Service*)ss;
	Billing_cfg *cfg=(Billing_cfg*)s->cfg;

	aLog(D_INFO, "service billing:%u thread started (%p)\n", s->instance, s->cfg);
	pthread_cleanup_push(sBiCancel, s);
	SET_CANCEL();
	s->t_id=pthread_self();

	// now we should sleep before starting service
	if(!(s->flags&SERVICE_RUN)) s->Sleep();

	cfg->st = Services.getService(SERVICE_STORAGE, cfg->storage);
   	if (cfg->st==NULL) {
		aLog(D_WARN, "billing service requires storage %u to be up, skipping\n", cfg->storage);
		goto S_BILLING_CLEANUP_POP;
        }
	
	//prepare counters
	PrepareFeeCounters();

	// get all accounts information from storage into RAM (bAccounts list)
	while(!(stBiObtainDbData(cfg))) {
		aLog(D_WARN, "Service billing can't obtain accounts from storage:%u\n", cfg->st->instance);
		s->Sleep(10);
	}
	Processor->Wakeup();	//reread units data

	bAccounts->RestoreAccounts(); //restore back Account status
	
	// get all accounts bdata information from storage into RAM (billing data)
	while(!(stBiLoadBdata(cfg))) {
		aLog(D_WARN, "Service billing can't obtain bdata from storage:%u\n", cfg->st->instance);
		s->Sleep(10);
	}

	aLog(D_DEBUG, "checking every %u seconds\n", cfg->delay);

	while (1) {
		
		AccountData(cfg);
		
		s->Sleep(cfg->delay);
	}

S_BILLING_CLEANUP_POP:
	pthread_cleanup_pop(0);
	return NULL;
}
//////////////////////////////////////////////////////////////////////////////////////////
void sBiCancel(void *v){
	Service *s = (Service*)v;
	Billing_cfg *cfg=(Billing_cfg*)s->cfg;
	aLog(D_INFO, "cancelling service billing:%u\n", s->instance);
	
	AccountData(cfg);
	
	stCloseSql(cfg->st,ST_CONN_BILLING);
	stCloseSql(cfg->st,ST_CONN_BDATA);

	ProcessorCfg.fifo=NULL;
	Billing = NULL;
	BillingCfg = NULL;
	
	pthread_rwlock_destroy(cfg->rwlock);
	aFree(cfg->rwlock);
	delete cfg->fifo;
	aFree(cfg->bfilename);
	aFree(cfg->sfilename);
	aFree(cfg);

        //delete Plans, SubPlans and Accounts data
	delete bPlans;		bPlans = NULL;
	delete bSubPlans;	bSubPlans = NULL;
	delete bAccounts;	bAccounts  = NULL;
}
//////////////////////////////////////////////////////////////////////////////////////////
void cShowBillingAccount(Connection *conn, char *param[]){
	
	Service *s=Billing;
	if(!s) return;

	Account *ac, *acx=NULL;
	u_char isfull=0;
	u_char bdata=0;
	char *blocked;

	if (param[2]!=empty) acx=bAccounts->Check(param[2]);
	if (!strcasecmp(param[3], "list")) {
		for (ac=bAccounts->root; ac!=NULL; ac=ac->next) 
			if (!acx || acx==ac) 
				fprintf(conn->stream_w, "%s %06X ", ac->name, ac->id);
			
		fprintf(conn->stream_w, "\n");
		return;
	}
	else if (!strcasecmp(param[3], "full")) isfull=1;
	
	if (!strcasecmp(param[2], "bdata") || !strcasecmp(param[3], "bdata")) bdata=1;

	for (ac=bAccounts->root; ac!=NULL; ac=ac->next){
		if (ac->status&ACCOUNT_DELETED) continue; //this account deleted
		if (!acx || acx==ac) {
			if(ac->status&ACCOUNT_BLOCKED ) blocked="BLOCKED";
			else if (ac->status&ACCOUNT_BEBLOCKED) blocked="BEBLOCKED";
			else blocked="UNBLOCKED";

			fprintf(conn->stream_w, "Name %s (%06X) %s %s bal: %lf plan: %s\n", ac->name, ac->id, blocked, ac->status&ACCOUNT_NEED_SYNC?"NOTSYNC":"SYNC", ac->balance, ac->plan?(ac->plan->name?ac->plan->name:"<\?\?>"):"-");
			if (isfull) {
				fprintf(conn->stream_w, "Plan %s %u %u Nextplan %s %u %u\n", ac->plan?(ac->plan->name?ac->plan->name:"<\?\?>"):"-", ac->plan?ac->plan->id:0, ac->plan_ch, ac->nextplan?(ac->nextplan->name?ac->nextplan->name:"<\?\?>"):"-", ac->nextplan?ac->nextplan->id:0, ac->nextplan_ch);
				fprintf(conn->stream_w, "Changed %u Blocked %u Created %u\n", ac->changed, ac->blocked, ac->created);
				fprintf(conn->stream_w,	"Email %s Password %s\n", ac->email?ac->email:"-", ac->password?ac->password:"-");
			}
			fprintf(conn->stream_w, "  Units: ");
			for (bUlist *bu=ac->bUroot; bu!=NULL; bu=bu->next) {
				NetUnit *u=bu->u;
				fprintf(conn->stream_w, "%s %06X ", u->name?u->name:"<\?\?>", u->id);
			}
			if(bdata && ac->plan) {
				u_char poz=0;
				billing_data *bd;
				fprintf(conn->stream_w, "\n  Bstats:\n");
				fprintf(conn->stream_w, "Account\tSubPlan\tPrefix\tIn\tOut\tPay_in\t\tPay_out\n");
				for(bSPlist *bsp=ac->plan->root;bsp!=NULL; bsp=bsp->next, poz++) {
					bd=&ac->data[poz];
					fprintf(conn->stream_w, "%06X\t%06X\t%c\t%lld\t%lld\t%lf\t%lf\n",ac->id,bsp->sp->id,'H',bd->h.in,bd->h.out,bd->h.pay_in,bd->h.pay_out);
					fprintf(conn->stream_w, "%06X\t%06X\t%c\t%lld\t%lld\t%lf\t%lf\n",ac->id,bsp->sp->id,'D',bd->d.in,bd->d.out,bd->d.pay_in,bd->d.pay_out);
					fprintf(conn->stream_w, "%06X\t%06X\t%c\t%lld\t%lld\t%lf\t%lf\n",ac->id,bsp->sp->id,'W',bd->w.in,bd->w.out,bd->w.pay_in,bd->w.pay_out);
					fprintf(conn->stream_w, "%06X\t%06X\t%c\t%lld\t%lld\t%lf\t%lf\n",ac->id,bsp->sp->id,'M',bd->m.in,bd->m.out,bd->m.pay_in,bd->m.pay_out);

				}
			}
			fprintf(conn->stream_w, "\n");
		} // if
	} // for

}
//////////////////////////////////////////////////////////////////////////////////////////
void cShowBillingPlan(Connection *conn, char *param[]){

	Service *s=Billing;
	if(!s) return;

	unsigned pl_id=0, a=0, b=0;
	if (param[2]!=empty) pl_id=strtol(param[2], NULL, 10);
	if (!strcasecmp(param[3], "accounts")) a=1;
	else if (!strcasecmp(param[3], "list")) b=1; 
	char b1[32], b2[32];

	Plan *pl; SubPlan *sp;
	for (pl=bPlans->root; pl!=NULL; pl=pl->next){
		if (!pl_id || pl_id==pl->id) {
			fprintf(conn->stream_w, "Plan ID %u Name \"%s\" Desc. \"%s\"\n", pl->id, pl->name?pl->name:"", pl->description?pl->description:"");
			if (a && !b) {
				Account *ac;
				for (ac=bAccounts->root; ac!=NULL; ac=ac->next){
					if (ac->plan==pl) fprintf(conn->stream_w, " %s", ac->name?ac->name:"<\?\?>");
				}
				fprintf(conn->stream_w, "\n");
			}
			else if (!b) {	
				Policy *p;
				for (bSPlist *spl=pl->root; spl!=NULL; spl=spl->next) {
					sp=spl->sp;
					if (!sp) continue;
					p=PolicyL.getPolicyById(sp->pid);
					fprintf(conn->stream_w, "\tSubplan ID %u\n", sp->id);
					fprintf(conn->stream_w, "\t  Fee %f, spread: '%c', policy %s(%06X)\n", sp->fee, sp->spread, p?p->name:"\?\?", sp->pid);
					fprintf(conn->stream_w, "\t  Incl. %s in %s out, ", sp->flags&UNLIM_IN?"unlimited":bytesQ2T(sp->inc_in, b1), sp->flags&UNLIM_OUT?"unlimited":bytesQ2T(sp->inc_out, b2));
					fprintf(conn->stream_w, "Over. %f/M in %f/M out\n", sp->pay_in, sp->pay_out);
				}
			}
		}
	}
}
//////////////////////////////////////////////////////////////////////////////////////////
void AccountData(Billing_cfg* cfg) {
	FIFO *fifo=cfg->fifo;
	Message_Store *msg;
	Account *ac;
	NetUnit *u;
	struct FeeCounters *fc;

	fc=PrepareFeeCounters();

	while((msg=(Message_Store*)fifo->Pop())) {
		aDebug(DEBUG_BILLING, "P->Billing unit:%06X acct:%06X from:%u to:%u\n",msg->netunit, msg->ap, msg->data->from, msg->ts);
		// here we might assume that messages goes in ordered way
		// so there is an order in policies
		// but it probably later
		if((u=Units.getUnitById(msg->netunit)) && (ac=u->account)) ac->AccountMessage(msg);
		if(!msg->Active()) MsgMgr.Delete(msg);
	}
	
	//sync data with storage
	BiSyncDb(cfg);

	bAccounts->UpdateAccounts();
	
	bAccounts->ChargeFees();
}

//////////////////////////////////////////////////////////////////////////////////////////
void BiSyncDb(Billing_cfg* cfg) {
	Account *ac=bAccounts->root;
	Account *prev=NULL;
	unsigned num_accounts=0;
	unsigned num_bdatas=0;

	//prepare account data file
	FILE *bfile=fopen(cfg->bfilename,"a");
	if(!bfile) {
		aLog(D_WARN, "Can't create temporary file %s: %s\n", cfg->bfilename, strerror(errno));
		return;
	}
	setlinebuf(bfile);

	//prepare billing data file
	FILE *sfile=fopen(cfg->sfilename,"a");
	if(!sfile) {
		aLog(D_WARN, "Can't create temporary file %s: %s\n", cfg->sfilename, strerror(errno));
	}
	setlinebuf(sfile);

	while(ac) {
		//sync account
		if (ac->status&ACCOUNT_NEED_SYNC) {
			ac->SyncAccount(bfile);
			num_accounts++;
		}

		//sync bdata
		if(ac->status&ACCOUNT_BDATA_NEED_SYNC) {
			num_bdatas += ac->SyncBdata(sfile);
		}

		if(ac->status&ACCOUNT_DELETED) {
			Account *tmp;
			//remove account from list
			if(ac==bAccounts->root) bAccounts->root=ac->next;
			else prev->next=ac->next;
			bAccounts->num_accounts--;
			//free memory
			tmp=ac->next;
			delete ac;
			ac=tmp;
		} else {
			prev=ac;
			ac=ac->next;
		}
	}

	if(num_accounts) {
		fclose(bfile);
		aDebug(DEBUG_BILLING, "SQL->HDD/billing %u queries\n", num_accounts);
		stSaveSql(cfg->st,cfg->bfilename,ST_CONN_BILLING);
	} else 
		fclose(bfile);

	if(num_bdatas) {
		fclose(sfile);
		aDebug(DEBUG_BILLING, "SQL->HDD/bdata %u queries\n", num_bdatas);
		stSaveSql(cfg->st,cfg->sfilename,ST_CONN_BDATA);
	} else 
		fclose(sfile);
}
//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
