/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: net_units.c,v 1.109.4.2 2005/03/22 10:18:55 jura Exp $*/

#include "netams.h"

char *netunit_type_name[NETUNIT_TYPES_NUM]={ "unknown", "user", "host", "cluster", "net", "group" };

in_addr_t AutoAssignIP(Connection *conn);

/////////////////////////////////////////////////////////////////////////
// NetUnit class
NetUnit::NetUnit(netunit_type t) {
	id=0;
	name=NULL;
	parent=NULL;
	type=t;
	next=NULL;
	link=NULL;
	
	ap=fp=NULL;

	dsl_root=NULL;
	monitor=NULL;

	sys_policy=SP_NONE; 
	sys_policy_perm=0;
	
	logindata=NULL;
	quotadata=NULL;

	// for default unit set up we have no bandwidth
	bw=NULL;
	
#ifdef HAVE_BILLING	
	account=NULL;
#endif
	flags=0;
	
	email=password=NULL;
}

NetUnit::~NetUnit() {
	if(type) this->unit2trees(REMOVE);
        DSList *ptr,*dsl=dsl_root;
        while(dsl) {
		ptr=dsl;
                dsl=dsl->next;
                aFree(ptr);
        }
	
	//remove reference to this unit, if it's target for policy
	if(flags&NETUNIT_POLICY_TARGET) PolicyL.DeleteUnitFromTarget(this);
#ifdef HAVE_BILLING
	if(account) account->AddUnit(this, REMOVE);
	else 
#endif
	if(ap) delete ap;
	
	if(fp) delete fp;
	if(logindata) aFree(logindata);
	if(quotadata) aFree(quotadata);
	if(name) aFree(name);
	if(bw) aFree(bw);
	if(email) aFree(email);
	if(password) aFree(password);
}

void NetUnit::setName(char *n){
	if(name) aFree(name);
	name=set_string(n);
	flags|=NETUNIT_CHANGED;
}

match NetUnit::Check(IPFlowStat *flow){
	switch(type) {
		case NETUNIT_HOST:
			return ((NetUnit_host*)this)->Check(flow);
			break;
		case NETUNIT_CLUSTER:
			return ((NetUnit_cluster*)this)->Check(flow);
			break;
		case NETUNIT_NET:
			return ((NetUnit_net*)this)->Check(flow);
			break;
		case NETUNIT_GROUP:
			return ((NetUnit_group*)this)->Check(flow);
			break;
		case NETUNIT_USER:
			return ((NetUnit_user*)this)->Check(flow);
			break;
		default:
			break;						 
	}
	return MATCH_NONE;
}

void NetUnit::unit2tree(IPTree *iptree,u_char flag) {
        switch(type) {
		case NETUNIT_HOST:
			return ((NetUnit_host*)this)->unit2tree(iptree,flag);
			break;
		case NETUNIT_CLUSTER:
			return ((NetUnit_cluster*)this)->unit2tree(iptree,flag);
			break;
		case NETUNIT_NET:
			return ((NetUnit_net*)this)->unit2tree(iptree,flag);
			break;
		case NETUNIT_USER:
			return ((NetUnit_user*)this)->unit2tree(iptree,flag);
			break;
		default: 
			break;
	}
}

void NetUnit::unit2trees(u_char flag) {
	Service *s=NULL;
	IPTree *iptree;

        while((s=Services.getServiceNextByType(SERVICE_DATASOURCE,s))) {
		//we do not work with stopped data-sources
		if(this->checkDSList(s->instance) && (s->flags&SERVICE_RUN)) {
			iptree=((ServiceDS_cfg*)s->cfg)->IPtree;
                	this->unit2tree(iptree,flag);	
		}
	}
	FW_CHECK_CHANGED(time(NULL));
}

void NetUnit::setDSList(unsigned short ds_id){
	DSList *dsl;
	for(dsl=dsl_root;dsl!=NULL;dsl=dsl->next)
		if (dsl->id==ds_id) return; // we will not add existing DS to list
		
	dsl=(DSList*)aMalloc(sizeof(DSList));
	dsl->id=ds_id;
	dsl->next=dsl_root;
	dsl_root=dsl;
}

void NetUnit::clearDSList(unsigned short ds_id){
	DSList *p=NULL;
	for (DSList *dsl=dsl_root;dsl!=NULL;dsl=dsl->next) {
		if (dsl->id==ds_id) {
			if(dsl==dsl_root) dsl_root=dsl->next;
			else p->next=dsl->next;
			aFree(dsl);
			break;
		}
		p=dsl;
	}
}

struct DSList *NetUnit::getDSList(){
	return dsl_root;
}

u_char NetUnit::checkDSList(unsigned short ds_id){
	if(!dsl_root) return 2;
	for(DSList *dsl=dsl_root;dsl!=NULL;dsl=dsl->next)
                if (dsl->id==ds_id) return 1;
	return 0;
}

void NetUnit::SetSysPolicy(SysPolicy sp, u_char flag, time_t now) {
	if(flag) sys_policy|=sp;
	else sys_policy&= ~ sp;
	FW_CHECK_CHANGED(now);
}
/////////////////////////////////////////////////////////////////////////
// NetUnitsList class
NetUnitsList::NetUnitsList(){
	root=last=NULL;
	num_units=0;
	tries_lock=tries_lock_failed=0;
	
	hash=(NetUnit**)aMalloc(UNITS_HASH_SIZE*sizeof(NetUnit*));
	
	rwlock=(pthread_rwlock_t*)aMalloc(sizeof (pthread_rwlock_t));
	pthread_rwlock_init(rwlock, NULL);
}

NetUnitsList::~NetUnitsList(){
	aFree(hash);
	pthread_rwlock_destroy(rwlock);
	aFree(rwlock);
}

void NetUnitsList::Insert(NetUnit *s){
	if(getUnitById(s->id)) return;
	int err=pthread_rwlock_trywrlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_wrlock(rwlock); }
	
	//hash
	unsigned num=UNITS_HASH(s->id);
	s->link=hash[num];
	hash[num]=s;

        //list
        if (root==NULL) root=last=s;
        else if(s->type == NETUNIT_GROUP) {
                s->next         = root;
                root            = s;
        } else {
                last->next      = s;
                last            = s;
        }

	num_units++;
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

void NetUnitsList::Delete(NetUnit *s){
	int err=pthread_rwlock_trywrlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_wrlock(rwlock); }
	

	unsigned num=UNITS_HASH(s->id);
	NetUnit *d, *p=NULL;

	//remove unit from hash
	for(d=hash[num]; d!=NULL; d=d->link) {
		if (d==s) {
			//hash
			if(hash[num]==s) hash[num]=s->link;
			else p->link=s->link;
			break;
		}
		p=d; 
	}
	//remove unit from list and clear parent reference
	for(d=root; d!=NULL; d=d->next)	{
		if (d->parent==s) d->parent=NULL;
		if (d==s) {
			if (s==root && s==last ) root=last=NULL;
			else if (s==root) root=s->next;
			else if (s==last) { last=p; last->next=NULL; }
			else p->next=s->next;

			num_units--;
		}
		p=d;
	}
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	
	if(s->type == NETUNIT_GROUP) Units.SortGroups();
}

void NetUnitsList::Delete(oid id){
	NetUnit *u;
	u=getUnitById(id);
	if (u) Delete(u);
}

NetUnit* NetUnitsList::getUnit(char *name){
	tries_lock++;
	int err=pthread_rwlock_tryrdlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	NetUnit *d=NULL;
	for(d=root; d!=NULL; d=d->next)	
		if (!strcmp(d->name, name)) break;
	
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return d;
}

NetUnit* NetUnitsList::getUnitById(oid id){
	tries_lock++;
	int err=pthread_rwlock_tryrdlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }
	
	NetUnit *d=NULL;
	
	for(d=hash[UNITS_HASH(id)]; d!=NULL; d=d->link)	
		if (d->id==id) break;
	
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return d;
}

NetUnit* NetUnitsList::getUnitByIP(in_addr_t s) {
	tries_lock++;
	int err=pthread_rwlock_tryrdlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }
	
	NetUnit *d=NULL;
	
	for(d=root; d!=NULL; d=d->next) {	
		if (d->type==NETUNIT_USER) {
			NetUnit_user *du = (NetUnit_user*)d;
			if (du->ip.s_addr==s) break;
		}
		if (d->type==NETUNIT_HOST) {
			NetUnit_host *du = (NetUnit_host*)d;
			if (du->ip.s_addr==s) break;
		}
	}	
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return d;
}


void NetUnitsList::DeletePolicyElsewhere(Policy *p){
	NetUnit *d;
	policy_data *cpd;
	tries_lock++;
	u_char del=0;
	int err=pthread_rwlock_trywrlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_wrlock(rwlock); }

	for (d=root; d!=NULL; d=d->next) {
		if(d->ap) {
			for (cpd=d->ap->root; cpd!=NULL; cpd=cpd->next) 
				if (cpd->policy==p) 
					if(!d->ap->Delete(p)) {delete d->ap; d->ap=NULL;}
		}
		if(d->fp) {
			for (cpd=d->fp->root; cpd!=NULL; cpd=cpd->next) 
				if (cpd->policy==p) 
					if(!d->fp->Delete(p)) {delete d->fp; d->fp=NULL; del++;}
			
		}
	}

	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	if(del) FW_CHECK_CHANGED(time(NULL));
}

u_char NetUnitsList::setMon(oid id, void *value){ //if oid=0; and value!=0 then we remove all reference to monitor
	u_char res=0;
	NetUnit *u;
	
	if((u=getUnitById(id))) {
		u->monitor=value;
		return 1;
	}

	int err=pthread_rwlock_tryrdlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	for(u=root; u!=NULL; u=u->next)	{
		if (u->monitor == value) { u->monitor=NULL; res=1;}
	}

	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return res;
}

void NetUnitsList::showMon(FILE *out, void *to, u_char action){ //action: 1-for show, 0- for config
	int err=pthread_rwlock_tryrdlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	NetUnit *d;
	for(d=root; d!=NULL; d=d->next)	{
		if (d->monitor && d->monitor==to && d->name) {
			if (action) fprintf(out, "%s(%06X) ", d->name, d->id);	
			else fprintf(out, "monitor unit %06X\n", d->id);
		}
 	}

	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return;
}

void NetUnitsList::listPasswordsHtml(Connection *conn){
	int err=pthread_rwlock_tryrdlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }
	NetUnit *u;
	for(u=root; u!=NULL; u=u->next)	
		if (u->password) fprintf(conn->stream_w, "%s:%s\n", u->name, crypt(u->password, "$1$"));
		
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return;
}

void NetUnitsList::SortGroups() {
	
	int err=pthread_rwlock_tryrdlock(rwlock);
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

        //clear weights
        for (NetUnit *u=root; u!=NULL && u->type==NETUNIT_GROUP; u=u->next)
                ((NetUnit_group*)u)->level=0;

        //set weights
        for (NetUnit *u=root; u!=NULL && u->type==NETUNIT_GROUP; u=u->next) {
                u_char level=0;
                for(NetUnit *up=u; up!=NULL; up=up->parent, level++) {
                        if(((NetUnit_group*)up)->level<level)
                                ((NetUnit_group*)up)->level=level;
                        if(u == up->parent) {
                                aLog(D_WARN, "NetUnit groups loop, len=%u\n", level);
                                break;
                        }
                }
        }

        NetUnit *p_u    = NULL;
        NetUnit *p_up   = NULL;

        //classical sort
        for (NetUnit *u=root; u!=NULL && u->type==NETUNIT_GROUP; u=u->next) {
                p_up = u;
                for(NetUnit *up=u->next; up!=NULL && up->type==NETUNIT_GROUP; up=up->next) {
                        if(((NetUnit_group*)u)->level < ((NetUnit_group*)up)->level) {
                                //swap it
                                if(u == root) root = up;
                                else p_u->next = up;

                                p_up->next = u;

                                if(up == last) last = u;

                                NetUnit *tmp;
                                //swap ->next
                                tmp=u->next;
                                u->next = up->next;
                                up->next = tmp;

                                //swap units
                                tmp=u;
                                u=up;
                                up=tmp;
                        }
                        p_up = up;
                }
                p_u = u;
        }
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

/////////////////////////////////////////////////////////////////////////
// NetUnit_cluster class
NetUnit_cluster::NetUnit_cluster() : NetUnit(NETUNIT_CLUSTER) {
	root=last=NULL;
	references=0;
	lock=(pthread_mutex_t*)aMalloc(sizeof (pthread_mutex_t));
	pthread_mutex_init(lock, NULL);
}

NetUnit_cluster::~NetUnit_cluster(){
	pthread_mutex_destroy(lock);
	aFree(lock);
}

void NetUnit_cluster::Add(struct in_addr ip){
	pthread_mutex_lock(lock);
	NetUnit_host *d, *s;
	for(d=root; d!=NULL; d=d->next)
		if (d->ip.s_addr==ip.s_addr) { 
			pthread_mutex_unlock(lock);
			return;
			}
	s=new NetUnit_host;
	s->ip=ip;
	s->next=NULL;

	if (root==NULL) root=s;
	else last->next=s;
	last=s; 
	references++;
	pthread_mutex_unlock(lock);
}

void NetUnit_cluster::Remove(struct in_addr ip){
	pthread_mutex_lock(lock);
	NetUnit_host *d, *p=NULL;
	for(d=root; d!=NULL; d=d->next)	{
		if (d->ip.s_addr==ip.s_addr) {
			if (d==root && d==last ) root=last=NULL;
			else if (d==root) root=d->next;
			else if (d==last) { last=p; last->next=NULL; }
			else p->next=d->next;

			references--;
			delete d;
			break;
		}
		p=d; 
	 }
	pthread_mutex_unlock(lock);
}

match NetUnit_cluster::Check(IPFlowStat *flow) {
	match mf=MATCH_NONE;
	NetUnit_host *d;
	pthread_mutex_lock(lock);
	for (d=root; d!=NULL; d=d->next) {
		if (d->ip.s_addr==flow->srcaddr.s_addr) mf|=MATCH_SRC;
		if (d->ip.s_addr==flow->dstaddr.s_addr) mf|=MATCH_DST;
	}
	pthread_mutex_unlock(lock);
	return mf;
}

void NetUnit_cluster::unit2tree(IPTree *iptree,u_char flag) {
	for(NetUnit_host *h=root;h!=NULL;h=h->next)
		iptree->addr2tree(h->ip.s_addr,32,(NetUnit*)this,flag);
}
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
// NetUnit_group class
NetUnit_group::NetUnit_group() : NetUnit(NETUNIT_GROUP){
	references=0;
}

match NetUnit_group::Check(IPFlowStat *flow) {
	NetUnit *t;
	match m,mf=MATCH_NONE;
	
	for (t=Units.root; t!=NULL; t=t->next)	
		if (t->parent==this) {
			m=t->Check(flow);
			if (m&MATCH_SRC) mf|=MATCH_SRC;
			if (m&MATCH_DST) mf|=MATCH_DST;
		}
	return mf;
}

/////////////////////////////////////////////////////////////////////////
// NetUnit_host class
NetUnit_host::NetUnit_host() : NetUnit(NETUNIT_HOST) {
	ip.s_addr=0;
}

match NetUnit_host::Check(IPFlowStat *flow) {
	match mf=MATCH_NONE;

	if (ip.s_addr==flow->srcaddr.s_addr) mf|=MATCH_SRC;
	if (ip.s_addr==flow->dstaddr.s_addr) mf|=MATCH_DST;

	return mf;
}

void NetUnit_host::unit2tree(IPTree *iptree,u_char flag) {
	iptree->addr2tree(ip.s_addr,32,(NetUnit*)this,flag);
}
/////////////////////////////////////////////////////////////////////////
NetUnit_net::NetUnit_net() : NetUnit(NETUNIT_NET) {
	ip.s_addr=0;
	mask.s_addr=INADDR_BROADCAST;
	auto_units_id=0;
}

match NetUnit_net::Check(IPFlowStat *flow){
	match mf=MATCH_NONE;

	if (ip.s_addr==(flow->srcaddr.s_addr&mask.s_addr)) mf|=MATCH_SRC;
	if (ip.s_addr==(flow->dstaddr.s_addr&mask.s_addr)) mf|=MATCH_DST;
	
	return mf;
}

void NetUnit_net::unit2tree(IPTree *iptree,u_char flag) {
        iptree->addr2tree(ip.s_addr, MASK2MLEN(&mask), (NetUnit*)this, flag);
}
/////////////////////////////////////////////////////////////////////////
// NetUnit_user class
NetUnit_user::NetUnit_user() : NetUnit(NETUNIT_USER) {
        ip.s_addr=0;
	real_name=NULL;
}

match NetUnit_user::Check(IPFlowStat *flow) {
	match mf=MATCH_NONE;
	if (ip.s_addr==flow->srcaddr.s_addr) mf|=MATCH_SRC;
	if (ip.s_addr==flow->dstaddr.s_addr) mf|=MATCH_DST;

	return mf;
}

void NetUnit_user::unit2tree(IPTree *iptree,u_char flag) {
	if (ip.s_addr) iptree->addr2tree(ip.s_addr,32,(NetUnit*)this,flag);
}
/////////////////////////////////////////////////////////////////////////
int cUnit(Connection *conn, char *param[], u_char no_flag){
	NetUnit *u;
	u_char i=2;
	u_char type=0;
	char *type_name=NULL;
	oid id=0;
	if (!strcmp(param[2], "name")) u=Units.getUnit(param[3]);
	else if (!strcmp(param[2], "oid")) {
		id=strtol(param[3], NULL, 16);	
		u=Units.getUnitById(id);
	}
	else { aParse(conn, "unit %s name/oid unspecified\n", param[1]); return PARSE_OK; }


	for(u_char j=0;j<NETUNIT_TYPES_NUM;j++) {
		if (strcasecmp(param[1], netunit_type_name[j])==0) {
			type=j;
			type_name=netunit_type_name[type];
			break;
		}
	}
	if(!type) {
		aParse(conn, "unit type unknown\n");
        	return PARSE_OK;
 	}
       
	NetUnit_host *host=(NetUnit_host*)u;
        NetUnit_user *user=(NetUnit_user*)u;
        NetUnit_net *net=(NetUnit_net*)u;
        NetUnit_cluster *cluster=(NetUnit_cluster*)u;
        NetUnit_group *group=(NetUnit_group*)u;

	if (!u && !no_flag) { 
		switch(type) {
			case NETUNIT_HOST:
				host= new NetUnit_host();
				u=(NetUnit*)host;
				break;
			case NETUNIT_USER:
				user= new NetUnit_user();
                                u=(NetUnit*)user;
				break;
			case NETUNIT_NET:
				net= new NetUnit_net();
				u=(NetUnit*)net;
                                break;
			case NETUNIT_GROUP:
				group= new NetUnit_group();
				u=(NetUnit*)group;
                                break;
			case NETUNIT_CLUSTER:
				cluster= new NetUnit_cluster();
				u=(NetUnit*)cluster;
                                break;
			default:
				return PARSE_OK; // we already sure we have correct type
		}
		u->id=newOid(id);
		Units.Insert(u);
		aParse(conn, "unit %s %06X created\n", type_name, u->id);
		
		//add default policies
		if (ProcessorCfg.def) {
			NetUnit *def=ProcessorCfg.def;
			if(def->ap) def->ap->SetForUnit(POLICY_ACCT, u);
			if(def->fp) {
				def->fp->SetForUnit(POLICY_FW, u);
				FW_CHECK_CHANGED(time(NULL));
			}
		}
	} else if (!u && no_flag) {
		aParse(conn, "unit %s does not exist\n", type_name);
		return PARSE_OK;
	} else if (u && no_flag) {
		aParse(conn, "unit %s %06X deleted\n", type_name, u->id);
		Units.Delete(u);
		delete u;
		return PARSE_OK;
	}
	
	while (param[i]!=empty) {
		no_flag=0;
 		if (strcasecmp(param[i], "no")==0) { 
			no_flag=1; 
			i++;
		}
		
            	if (strstr(param[i], "sys-")==param[i]) { // possibly sys-* here
                	SetSysPolicy(param[i], u, conn);
                        i-=1;

                } else if (strcasecmp(param[i], "acct-policy")==0) {
                	PolicyAdd(u, &i, POLICY_ACCT, conn, param, no_flag);

                } else if (strcasecmp(param[i], "fw-policy")==0) {
                	PolicyAdd(u, &i, POLICY_FW, conn, param, no_flag);
	
		} else if (strcasecmp(param[i], "ip")==0) {
			u->unit2trees(REMOVE);
			switch(type) {
				case NETUNIT_CLUSTER:
					struct in_addr addr;
                    			inet_aton(param[i+1], &addr);
                    			if(no_flag) cluster->Remove(addr);
					else cluster->Add(addr);
					break;
				case NETUNIT_HOST:	
                    			if(no_flag) 
						host->ip.s_addr=0;
					else {
						if (!strcasecmp(param[i+1], "auto")) { 
							aFree(param[i+1]);
							host->ip.s_addr=AutoAssignIP(conn);
							param[i+1]=set_string(inet_ntoa(host->ip));
						} 
						else inet_aton(param[i+1], &host->ip);
					}
					break;
				case NETUNIT_USER:
					if(no_flag) 
						user->ip.s_addr=0;
					else {
						if (!strcasecmp(param[i+1], "auto")) { 
							aFree(param[i+1]);
							user->ip.s_addr=AutoAssignIP(conn);
							param[i+1]=set_string(inet_ntoa(user->ip));
						} 
	                    			else inet_aton(param[i+1], &user->ip);
					}
					break;
				case NETUNIT_NET:
					if(no_flag) net->ip.s_addr=0;
					else {
						getAddr(param[i+1], &net->ip, &net->mask);
					}
					break;
				default:
					aParse( conn, "IP for %s not supported\n", type_name);
					break;
			}
                        u->unit2trees(ADD);
                        aParse(conn, "%s %06X ip set: %s\n", type_name, u->id, param[i+1]);
		
		} else if (type == NETUNIT_NET && strcasecmp(param[i], "mask")==0) {
                                u->unit2trees(REMOVE);
                                inet_aton(param[i+1], &net->mask);
				net->ip.s_addr&=net->mask.s_addr; //correct net ip
                                u->unit2trees(ADD);
                                aParse(conn, "net %06X mask set: %s\n", u->id, inet_ntoa(net->mask));

		} else if (strcasecmp(param[i], "name")==0) { 
			u->setName(param[i+1]);
			aParse(conn, "%s %06X name set: %s\n", type_name, u->id, u->name);
		
		} else if (strcasecmp(param[i], "bw")==0) {
			u->bw=setBW(u->bw, param, &i);
			//no need to remove unit from iptrees - this is just bw update
			//here we should set bw limits
			u->unit2trees(ADD);
			char buf[64];
			aParse(conn, "%s %06X allowed bandwidth set: %s\n", type_name, u->id, getBW(u->bw, buf));

		} else if (strcasecmp(param[i], "email")==0) {
               		if(u->email) aFree(u->email);
			if(no_flag) {
				u->email=NULL;
				aParse(conn, "%s %06X email cleared\n", type_name, u->id);
			} else {
                        	u->email=set_string(param[i+1]);
                        	aParse(conn, "%s %06X email set: %s\n", type_name, u->id, u->email);
			}

		} else if (strcasecmp(param[i], "ds-list")==0) { 
			u->unit2trees(REMOVE);
			DSListAdd(u, &i, conn, param, no_flag);
			u->unit2trees(ADD);

		} else if (strcasecmp(param[i], "auto-units")==0 && u->type==NETUNIT_NET) { 
			net->auto_units_id=strtol(param[i+1], NULL, 10);
			aParse(conn, "%s %06X set auto-units ID %u\n", type_name, u->id, net->auto_units_id);

		} else if (strcasecmp(param[i], "parent")==0) {
			NetUnit_group *gr;
			if (no_flag) { 
				u->parent=NULL;
				if(u->type == NETUNIT_GROUP) Units.SortGroups();
				aParse(conn, "%s %06X parent cleared\n", type_name, u->id);
			} else {
				gr=(NetUnit_group*)Units.getUnit(param[i+1]);
				if(u == gr)
					aParse(conn, "group %s (%06X): can't add parent to itself\n", gr->name, gr->id);
                                else if (gr) {
                                        u->parent=gr;
                                        // in case we want add parent for group, we need to sort them
                                        if(u->type == NETUNIT_GROUP) Units.SortGroups();
					aParse(conn, "%s %06X added to group %s\n", type_name, u->id, gr->name);
				}
				else aParse(conn, "group %s for %s %06X not exist\n", param[i+1], type_name, u->id);
			}
			FW_CHECK_CHANGED(time(NULL));

		} else if (strcasecmp(param[i], "no-local-pass")==0) { 
			if (no_flag) 
				u->flags&=~NETUNIT_NLP; 
			else 
				u->flags|=NETUNIT_NLP;
			aParse(conn, "%s %06X no-local-pass flag set: %u\n", type_name, u->id, u->flags&NETUNIT_NLP);	
			i-=1;
		
		} else if (strcasecmp(param[i], "password")==0) {
                	if(u->password) aFree(u->password);
			if(no_flag) {
				u->password=NULL;
				aParse(conn, "Unit %s %06X password cleared\n", type_name, u->id);
			} else {
                        	u->password=set_string(param[i+1]);
                        	aParse(conn, "Unit %s %06X password set: %s\n", type_name, u->id, u->password);
			}

                } else if (type == NETUNIT_USER && strcasecmp(param[i], "real_name")==0) {
                        if(user->real_name) aFree(user->real_name);
			if(no_flag) {
				user->real_name=NULL;
				aParse(conn, "Unit user %06X real_name cleared\n", u->id);
			} else {
                        	user->real_name=set_string(param[i+1]);
                        	aParse(conn, "user %06X real_name set: %s\n", u->id, user->real_name);
			}

		} else if (strcasecmp(param[i], "oid")==0) {
			;//do nothing
		} else {
			aParse(conn, "%s %06X command '%s' unknown\n", type_name, u->id, param[i]); 
			i--;
		}

		i=i+2;
	}
	return PARSE_OK;
}
/////////////////////////////////////////////////////////////////////////
int cShowUnits(Connection *conn, char *param[]){	   // "show units"
	pthread_rwlock_rdlock(Units.rwlock);
	NetUnit *d;
	char *parent_name;

	if (!strcasecmp(param[2], "syspolicy")) {
		u_char print_where_set=0;
		if (!strcasecmp(param[3], "whereset")) print_where_set=1;
		fprintf(conn->stream_w, "%6s | %10s | %10s \n", "OID", "NAME", "SYSPOLICY");
		for (d=Units.root; d!=NULL; d=d->next) {
			if ((print_where_set==1 && d->sys_policy!=SP_NONE) || print_where_set==0){
				fprintf(conn->stream_w, "%06X | %10s | ", d->id, d->name?d->name:"<\?\?>");
				PrintSysPolicy(conn->stream_w, d->sys_policy, d->sys_policy_perm);
				fprintf(conn->stream_w, "\n");
			}
		}
	}

	else if (!strcasecmp(param[2], "email")) {
		fprintf(conn->stream_w, "%6s | %10s | %10s \n", "OID", "NAME", "E-MAIL");
		for (d=Units.root; d!=NULL; d=d->next) {
			if (d->email) {
				fprintf(conn->stream_w, "%06X | %10s | %s\n", d->id, d->name?d->name:"<\?\?>", d->email);
			}
        	}
	}

	else if (!strcasecmp(param[2], "name") && param[3]!=empty) {
		NetUnit *d = Units.getUnit(param[3]);
		if (d!=NULL) {
			if (d->type==NETUNIT_HOST) { 
				NetUnit_host *dh=(NetUnit_host*)d; 
				fprintf(conn->stream_w, "%06X | %10s | %s\n", d->id, d->name?d->name:"<\?\?>", inet_ntoa(dh->ip));
				}
			else if (d->type==NETUNIT_USER) { 
				NetUnit_user *dh=(NetUnit_user*)d; 
				fprintf(conn->stream_w, "%06X | %10s | %s\n", d->id, d->name?d->name:"<\?\?>", inet_ntoa(dh->ip));
				}
			}
	}

	else if (!strcasecmp(param[2], "hash")) {
		//gather units hash statistic
		NetUnit *u;
		unsigned used=0;
		unsigned max_chain=0;
		unsigned tmp;
		unsigned long total=0;

		for(unsigned i=0;i<UNITS_HASH_SIZE;i++) {
			if(!(u=Units.hash[i])) continue;
			used++;
			tmp=0;
			for(;u!=NULL;u=u->link) {
				tmp++;
			}
			total+=tmp;
			if(max_chain<tmp) max_chain=tmp;
		}
		fprintf(conn->stream_w, "Units HASH: size=%u, %lu units hashed, %u nodes used, max chain= %u\n",UNITS_HASH_SIZE,total,used,max_chain);

	} else {	
		u_char show_type=0; //show units <type>
		u_char show_active=0; //show units users active
		for(u_char i=0;i<NETUNIT_TYPES_NUM;i++) {
			if (!strcasecmp(param[2], netunit_type_name[i])){
				show_type = i;	
				break;
			}
		}
		if (!strcasecmp(param[3], "syspolicy")) {
			fprintf(conn->stream_w, "%6s | %10s | %10s \n", "OID", "NAME", "SYSPOLICY");
                	for (d=Units.root; d!=NULL; d=d->next) {
				if(d->type!=show_type) continue;
                        	fprintf(conn->stream_w, "%06X | %10s | ", d->id, d->name?d->name:"<\?\?>");
				PrintSysPolicy(conn->stream_w, d->sys_policy, d->sys_policy_perm);
				fprintf(conn->stream_w, "\n");
			}
			pthread_rwlock_unlock(Units.rwlock);
        		return PARSE_OK;
                }

		if (!strcasecmp(param[3], "active")) show_active=1;

		fprintf(conn->stream_w, "%8s | %6s | %10s | %3s | %10s | %15s |%-20s\n", "TYPE", "OID", "NAME", "NLP", "PARENT", "EMAIL", "PARAMS");
		for(d=Units.root; d!=NULL; d=d->next)	{
			if(show_type && d->type!=show_type) continue;
			if(show_active) {
			 	if(d->type!=NETUNIT_USER) continue; 
				NetUnit_user *h=(NetUnit_user*)d;
				if(h->ip.s_addr==0) continue;
			}

			if (d->parent && d->parent->name) parent_name=d->parent->name; else parent_name="<>";
			fprintf(conn->stream_w, "%8s | %06X | %10s | %3s | %10s | %15s | ", netunit_type_name[d->type], d->id, d->name?d->name:"<\?\?>", d->flags&NETUNIT_NLP?" + ":"", parent_name,d->email?d->email:"");

			switch (d->type) {
			case NETUNIT_HOST: {
				NetUnit_host *h;
				h = (NetUnit_host*)d;
				fprintf(conn->stream_w, "IP: %-12s\n",inet_ntoa(h->ip));
				} break;
			case NETUNIT_USER: {
				NetUnit_user *h;
				h = (NetUnit_user*)d;
                                fprintf(conn->stream_w, "IP: %-12s\n", inet_ntoa(h->ip));
				} break;
			case NETUNIT_CLUSTER: { 
				NetUnit_cluster *h;
				NetUnit_host *t;
				h = (NetUnit_cluster*)d;
                                fprintf(conn->stream_w, "IPs: ");
				for (t=h->root; t!=NULL; t=t->next)	fprintf(conn->stream_w, "%s ", inet_ntoa(t->ip));
				fprintf(conn->stream_w, "\n");
				} break;
			case NETUNIT_GROUP: {
				NetUnit_group *h;
				NetUnit *t; 
				h = (NetUnit_group*)d;
				if(h->email) fprintf(conn->stream_w, "Sub: ");
				for (t=Units.root; t!=NULL; t=t->next)	if (t->parent==d && t->name) fprintf(conn->stream_w, "%s ", t->name);
				fprintf(conn->stream_w, "\n");
				} break;
			case NETUNIT_NET: {
				NetUnit_net *h;
				h = (NetUnit_net*)d;
				fprintf(conn->stream_w, "%s:", inet_ntoa(h->ip));
				fprintf(conn->stream_w, "%s\n", inet_ntoa(h->mask));
				} break;
			default: 
				fprintf(conn->stream_w, "%8s | %06X\n", "<\?\?>", d->id);
				break;
			}
		}
	}

	pthread_rwlock_unlock(Units.rwlock);
	return PARSE_OK;
}

/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
int cShowUnitsList(Connection *conn, u_char isfull, char *p1, char *p2){ // "show list {full}"
	pthread_rwlock_rdlock(Units.rwlock);
	NetUnit *d, *d2=NULL;
	policy_data *cpd;
	char *parent_name, *t_str;
	t_str=(char *)aMalloc(32);

	if (p1!=empty && p2!=empty) {
		if (!strcasecmp(p1, "name")) d2=Units.getUnit(p2);
		else if (!strcasecmp(p1, "oid")) d2=Units.getUnitById(strtol(p2, NULL, 16));
	}

	for (d=Units.root; d!=NULL; d=d->next) {
  		
  		if (p1!=empty && p2!=empty && d!=d2) continue;
  		
  		fprintf(conn->stream_w, "OID: %06X Name: %-10s Type: %-8s", d->id, d->name, netunit_type_name[d->type]);
		if (d->flags&NETUNIT_NLP) fprintf(conn->stream_w, " NLP"); 
		if (d->parent && d->parent->name) parent_name=d->parent->name; else parent_name="<>";
		fprintf(conn->stream_w, " Parent: %-10s", parent_name);

		if (d->dsl_root) {
			fprintf(conn->stream_w, " DS-list: ");
			for (DSList *dsl=d->dsl_root;dsl!=NULL;dsl=dsl->next)
				 fprintf(conn->stream_w, "%u ", dsl->id);
		}

		fprintf(conn->stream_w, "\n");
				
		fprintf(conn->stream_w, " %-11s", "SYST policy ");
		if (d->sys_policy==SP_NONE) fprintf(conn->stream_w, "is not set\n");
		else {
			PrintSysPolicy(conn->stream_w, d->sys_policy, d->sys_policy_perm);
			fprintf(conn->stream_w, "\n");
		} 

		fprintf(conn->stream_w, " %-11s", "  FW policy");
		if (d->fp) {
			fprintf(conn->stream_w, ": %-6s %-10s %-12s %-12s\n",  "OID", "NAME", "CHECK", "MATCH");
			for (cpd=d->fp->root; cpd!=NULL; cpd=cpd->next) {
				fprintf(conn->stream_w, "%-10s %s%s %06X %-10s %-12llu %-12llu\n", "", (cpd->flags&POLICY_FLAG_BRK)?"%":" ", (cpd->flags&POLICY_FLAG_INV)?"!":" ", cpd->policy->id, cpd->policy->name?cpd->policy->name:"<\?\?>", cpd->check, cpd->match);
			}
		}
		else fprintf(conn->stream_w, " list is empty\n");
						  
		fprintf(conn->stream_w, " %-11s", "ACCT policy");

		if (d->ap) {
			pthread_rwlock_rdlock(d->ap->rwlock);
			fprintf(conn->stream_w, ": %-6s %-10s %-12s %-12s\n",  "OID", "NAME", "CHECK", "MATCH");
			for (cpd=d->ap->root; cpd!=NULL; cpd=cpd->next) {
				fprintf(conn->stream_w, "%-10s %s%s %06X %-10s %-12llu %-12llu\n", "", (cpd->flags&POLICY_FLAG_BRK)?"%":" ", (cpd->flags&POLICY_FLAG_INV)?"!":" ", cpd->policy->id, cpd->policy->name?cpd->policy->name:"<\?\?>", cpd->check, cpd->match);
				if (isfull) {
					timeU2T(cpd->flow.from, t_str); fprintf(conn->stream_w, " %-19s %-10s in: %-12llu out: %-12llu\n", cpd->flow.from?t_str:"--.--.---- --:--:--",  "flow", cpd->flow.in, cpd->flow.out);
					timeU2T(cpd->m.from, t_str); fprintf(conn->stream_w, " %-19s %-10s in: %-12llu out: %-12llu\n", t_str,  "month", cpd->m.in, cpd->m.out);
					timeU2T(cpd->w.from, t_str); fprintf(conn->stream_w, " %-19s %-10s in: %-12llu out: %-12llu\n", t_str,  "week", cpd->w.in, cpd->w.out);
					timeU2T(cpd->d.from, t_str); fprintf(conn->stream_w, " %-19s %-10s in: %-12llu out: %-12llu\n", t_str,  "day", cpd->d.in, cpd->d.out);
					timeU2T(cpd->h.from, t_str); fprintf(conn->stream_w, " %-19s %-10s in: %-12llu out: %-12llu\n", t_str,  "hour", cpd->h.in, cpd->h.out);
				}
			}
			pthread_rwlock_unlock(d->ap->rwlock);
		}
		else fprintf(conn->stream_w, " list is empty\n");

		fprintf(conn->stream_w, "\n");
	}
	aFree(t_str);
	pthread_rwlock_unlock(Units.rwlock);
	return PARSE_OK;
}

/////////////////////////////////////////////////////////////////////////
void DSListAdd(NetUnit *u, u_char *i, Connection *conn, char *param[], u_char no_flag){
	char *res, *ptr;

	aDebug(DEBUG_DS_IP, "NetUnit: DSL: adding DS list \"%s\"\n", param[(*i)+1]);

	int k=strlen(param[(*i)+1]);
	unsigned f;

	for (ptr=param[(*i)+1]; (ptr-param[(*i)+1])<k ;ptr=res+1){
		if(no_flag && ( !strcmp(ptr,"all") || !strcmp(ptr,"*")) ) {
			aParse(conn, "ds-list cleared for unit %06X\n", u->id);
			DSList *ptr,*dsl=u->dsl_root;
			while(dsl) {
				ptr=dsl;
				dsl=dsl->next;
				aFree(ptr);
			}
			u->dsl_root=NULL;
			return;
		}
		f=strtol(ptr, &res, 10);
		if (no_flag) {
			aParse(conn, "removing DS %u from list for unit %06X\n", f, u->id);
			u->clearDSList(f);
		}
		else {
			aParse(conn, "adding DS %u to list for unit %06X\n", f, u->id);
			u->setDSList(f);
		}
	}
}

/////////////////////////////////////////////////////////////////////////
in_addr_t AutoAssignIP(Connection *conn) {
	unsigned addr;

	for(AutoAssignEntry *e=ProcessorCfg.auto_assign; e!=NULL; e=e->next) {
		for(addr=ntohl(e->start.s_addr); addr<=ntohl(e->stop.s_addr); addr++) {
			if (!Units.getUnitByIP(htonl(addr))) {
				struct in_addr ip;
				ip.s_addr=htonl(addr);
				aDebug(DEBUG_COMMAND, "selected auto-assign address %s \n", inet_ntoa(ip));
				return htonl(addr);
                       	}
		}
	} 
	aLog(D_WARN, "auto-assign address pool is empty, discarding new request\n");
	aParse(conn, "auto-assign address pool is empty, discarding new request\n");
	return 0;
}
/////////////////////////////////////////////////////////////////////////
