/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: ds_netflow.c,v 1.42 2005/01/17 13:13:21 jura Exp $ */

#include "netams.h"
#include "netflow.h"

//////////////////////////////////////////////////////////////////////////
unsigned sDS_AcctNetFlow(ServiceDS_cfg* cfg, NFSource *nf, struct ip *ip, unsigned long t);
void ds_netflow_cancel(void *ptr);
//////////////////////////////////////////////////////////////////////////
void* ds_netflow(void *ss){
	Service *s=(Service*)ss;
        ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;

        struct sockaddr_in ds;
	socklen_t size_ds = sizeof(ds);
        int *socketid = (int*)aMalloc(sizeof(int));
        unsigned len;
        struct timeval start;
        unsigned char *packet=cfg->packet;
        NFSource *nf;
	
        int reuseaddr_on=1, status;

	*socketid=socket(PF_INET, SOCK_DGRAM, 0);
        if (*socketid<0) { 
		aLog(D_ERR, "failed to create netflow listening socket: %s", strerror(errno));
		goto END;; 
	}
        bzero(&ds, size_ds);

	ds.sin_family = AF_INET;
        ds.sin_addr.s_addr = cfg->l_addr.s_addr;
        ds.sin_port = htons(cfg->port);

        // set options
	status=bigsockbuf(*socketid, SO_RCVBUF, FT_SO_RCV_BUFSIZE);
	if(status<0) {
		aLog(D_ERR, "failed to setsockopt receive buffer: %s", strerror(errno));
		goto END;
	} else {
		aLog(D_INFO, "receive bufer set to %u\n", status);
	}
        status=setsockopt(*socketid, SOL_SOCKET, SO_REUSEADDR, (void*)&reuseaddr_on, sizeof(reuseaddr_on));
	if (status<0) {
                aLog(D_ERR, "failed to setsockopt netflow listening socket: %s", strerror(errno));
                goto END;
        }

        // bind socket with serv_addr
        status=bind(*socketid, (struct sockaddr*)&ds, size_ds);
        if (status<0) { 
		aLog(D_ERR, "failed to bind netflow listening socket: %s", strerror(errno)); 
		goto END; 
	}
	SET_POLL(*socketid);

	pthread_cleanup_push(ds_netflow_cancel, (void*) socketid);
	pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);

	aLog(D_INFO,"NetFlow v5  packet processing for data-source:%u initialized\n",s->instance);
	aLog(D_INFO,"Listening on %s:%u\n", inet_ntoa(cfg->l_addr), cfg->port);
	
        while(1) {
		CHECK_POLL(status);
		if(!status) continue;

		len=recvfrom(*socketid, packet, MAX_PKT_SIZE, 0, (struct sockaddr *)&ds, &size_ds);
		//look for source
		for(nf=cfg->nfroot; nf!=NULL; nf=nf->next)
			if (ds.sin_addr.s_addr==nf->src_addr)  break;

		if(!nf) {
			cfg->total_errors++;
			aLog(D_WARN, "NetFlow packet from unknown source: %s\n", inet_ntoa(ds.sin_addr)); 
			continue; 
		}
                gettimeofday(&start, NULL);
                sDS_AcctNetFlow(cfg, nf, (struct ip*) packet, (unsigned long)start.tv_sec);
		
		nf->packets++;
		nf->timestamp=cfg->timestamp=start.tv_sec;
                
		sDSMeasure(cfg, &start, len);
    	}

	pthread_cleanup_pop(0);
END:	
	aFree(socketid);
	return NULL;
}
//////////////////////////////////////////////////////////////////////////
void ds_netflow_cancel(void *ptr){
	int *socketid=(int*)ptr;
	shutdown(*socketid, SHUT_RDWR);
        close(*socketid);
	aFree(socketid);
}
//////////////////////////////////////////////////////////////////////////
unsigned sDS_AcctNetFlow(ServiceDS_cfg* cfg, NFSource *nf, struct ip *ip, unsigned long time) {
	IPTree *IPtree=cfg->IPtree;
	
	IPStat5Msg *msg = (IPStat5Msg*)ip;
	IPFlow5Stat *flow;
	Flow5StatHdr *hdr = &msg->header;
	hdr->version = ntohs(hdr->version);
	
	if (hdr->version==FLOW_VERSION_5) {
		hdr->count         = ntohs(hdr->count);
		hdr->SysUptime     = ntohl(hdr->SysUptime);
		hdr->unix_secs     = ntohl(hdr->unix_secs);
		hdr->unix_nsecs    = ntohl(hdr->unix_nsecs);
		hdr->flow_sequence = ntohl(hdr->flow_sequence);

//		aDebug(DEBUG_FLOW, " cfg seq %lu, fl seq %lu, %d records:\n", nf->seq_id, hdr->flow_sequence, hdr->count);

		if (!nf->seq_id) nf->seq_id=hdr->flow_sequence;
		else if (nf->seq_id!=hdr->flow_sequence) { 
			if (hdr->count> V5FLOWS_PER_PAK) 
				aLog(D_WARN, "NF:%u packet with wrong number of flows (>V5FLOWS_PER_PAK), received from %02X \n", cfg->id, hdr->engine_id);
			else 
				aLog(D_WARN, "NF:%u awaited seq %lu, received %lu from %02X (%u flows are lost)\n", cfg->id, nf->seq_id, hdr->flow_sequence, hdr->engine_id, hdr->flow_sequence-nf->seq_id);
			nf->seq_id=hdr->flow_sequence; // restoring back
			nf->errors++;
			cfg->total_errors++;
		}
		
		//update counters
		nf->seq_id+=hdr->count;
		nf->flows+=hdr->count;
		cfg->total_flows+=hdr->count;

		aDebug(DEBUG_FLOW, " NetFlow v%u:%02X packet ds:%u seq %lu/%lu with %u records:\n", hdr->version, hdr->engine_id, cfg->id, hdr->flow_sequence, nf->seq_id, hdr->count);

		for (int i = 0; i < hdr->count; i++) {
			flow = &msg->records[i];
#ifdef DEBUG
			char *b1, buf_1[32], *b2, buf_2[32]; // for debugging
			b1=inet_ntoa(flow->srcaddr); strncpy(buf_1, b1, strlen(b1)+1);
			b2=inet_ntoa(flow->dstaddr); strncpy(buf_2, b2, strlen(b2)+1);
			aDebug(DEBUG_FLOW, " Flow if:%u->%u src:%s:%u,dst:%s:%u,n:%lu,l:%lu,p:%u\n", ntohs(flow->input), ntohs(flow->output), buf_1, ntohs(flow->srcport), buf_2, ntohs(flow->dstport), ntohl(flow->dPkts), ntohl(flow->dOctets), (flow->prot));
#endif
			//bring netflow presentation to local flow presentation
			// we need this because we are working in native format without *toh*() conversions
			flow->dPkts=ntohl(flow->dPkts);
			flow->dOctets=ntohl(flow->dOctets);

			if(cfg->flags==DS_CLOCK_REMOTE) { //e.g == CLOCK_REMOTE for now 
				//use flow time
				flow->First=hdr->unix_secs + ntohl(flow->First)/1000 - hdr->SysUptime/1000;
				flow->Last=hdr->unix_secs + ntohl(flow->Last)/1000 - hdr->SysUptime/1000;
			} else {
				flow->First=time-(ntohl(flow->Last)-ntohl(flow->First))/1000;
				flow->Last=time;
			}
			IPtree->ACCT((IPFlowStat *)flow);
		}
	
	}
	else {
		aLog(D_WARN, " NetFlow v%d packet unsupported\n", hdr->version);
		cfg->total_errors++;
		nf->errors++;
	}
	return 0;
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

