#include "config.h"

#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include "nwnet_i.h"
#include "ncplib_i.h"
#include "ncpcode.h"

#ifndef __MAKE_SULIB__

static const char* wchar_encoding = NULL;
static const char* default_encoding = NULL;
static const wchar_t* default_tree = NULL;
static const wchar_t* default_context = NULL;

static NWDSCCODE NWDSXlateFromCtx(NWDSContextHandle ctx, wchar_t* dst,
		size_t maxlen, const void* src);
static NWDSCCODE NWDSXlateToCtx(NWDSContextHandle ctx, void* data, 
		size_t maxlen, const wchar_t* src);
static NWDSCCODE __NWCCGetServerAddressPtr(NWCONN_HANDLE conn, size_t* count, nuint8** data);
		
/* debug fprintf */
#if 1
#define dfprintf(X...)
#else
#define dfprintf(X...) fprintf(X)
#endif

#ifndef HAVE_WCSCPY
/* Do it yourself... Move into specific file?! */
/* we can pass NULL into this procedure */
wchar_t* wcscpy(wchar_t* dst, const wchar_t* src) {
	wchar_t* tmp = dst;
	while ((*dst++ = *src++) != 0) ;
	return tmp;
}
#endif	/* !HAVE_WCSCPY */

#ifndef HAVE_WCSCMP
int wcscmp(const wchar_t* s1, const wchar_t* s2) {
	wint_t c1;
	wint_t c2;
	do {
		c1 = *s1++;
		c2 = *s2++;
		if (!c1)
			break;
	} while (c1 == c2);
	if (c1 < c2)
		return -1;	/* negative */
	return c1 - c2;		/* zero or positive */
}
#endif	/* !HAVE_WCSCMP */

#ifndef HAVE_WCSNCMP
int wcsncmp(const wchar_t* s1, const wchar_t* s2, size_t n) {
	wint_t c1;
	wint_t c2;
	if (!n)
		return 0;
	do {
		int diff;
		
		c1 = *s1++;
		c2 = *s2++;
		if (!c1)
			break;
	} while ((c1 == c2) && --n);
	if (c1 < c2)
		return -1;
	return c1 - c2;
}
#endif	/* !HAVE_WCSNCMP */

#ifndef HAVE_WCSCASECMP
int wcscasecmp(const wchar_t* s1, const wchar_t* s2) {
	wint_t c1;
	wint_t c2;
	do {
		int diff;
		
		c1 = *s1++;
		c2 = *s2++;
		if (!c1)
			break;
		diff = c1 - c2;
		if (diff) {
			if ((c1 >= L'A') && (c1 <= L'Z'))
				c1 += 0x20;
			if ((c2 >= L'A') && (c2 <= L'Z'))
				c2 += 0x20;
			if (c1 != c2) {
				break;
			}
		}
	} while (1);
	if (c1 < c2)
		return -1;
	return c1 - c2;
}
#endif	/* !HAVE_WCSCASECMP */

#ifndef HAVE_WCSNCASECMP
int wcsncasecmp(const wchar_t* s1, const wchar_t* s2, size_t n) {
	wint_t c1;
	wint_t c2;
	if (!n)
		return 0;
	do {
		int diff;
		
		c1 = *s1++;
		c2 = *s2++;
		if (!c1)
			break;
		diff = c1 - c2;
		if (diff) {
			if ((c1 >= L'A') && (c1 <= L'Z'))
				c1 += 0x20;
			if ((c2 >= L'A') && (c2 <= L'Z'))
				c2 += 0x20;
			if (c1 != c2) {
				break;
			}
		}
	} while (--n);
	if (c1 < c2)
		return -1;
	return c1 - c2;
}
#endif	/* !HAVE_WCSNCASECMP */

#ifndef HAVE_WCSLEN
size_t wcslen(const wchar_t* x) {
	const wchar_t* start = x;
	if (x)
		while (*x) x++;
	return x - start;
}
#endif	/* !HAVE_WCSLEN */

#ifndef HAVE_WCSDUP
/* NULL on input -> NULL on output */
wchar_t* wcsdup(const wchar_t* x) {
	size_t l;
	wchar_t* ret;
	
	if (!x) return NULL;
	l = (wcslen(x) + 1) * sizeof(wchar_t);
	ret = (wchar_t*)malloc(l);
	if (ret)
		memcpy(ret, x, l);
	return ret;
}
#endif	/* !HAVE_WCSDUP */

struct RDNEntry {
	size_t typeLen;
	const wchar_t* type;	/* C, S, L, OU, O, CN, SA, ... */
	size_t valLen;
	const wchar_t* val;
	struct RDNEntry* up;
	struct RDNEntry* next;	/* for CN=A+Bindery Type=B... */
};

/* frees entry in case of failure */
static NWDSCCODE __NWDSAddRDN(struct RDNEntry** add, struct RDNEntry* entry, struct RDNEntry*** next) {
	if (entry->typeLen) {
		switch (entry->typeLen) {
			case 11:
				if (!wcsncasecmp(entry->type, L"Common Name", 11)) {
					entry->typeLen = 2;
					entry->type = L"CN";
				}
				break;
			case 12:
				if (!wcsncasecmp(entry->type, L"Country Name", 12)) {
					entry->typeLen = 1;
					entry->type = L"C";
				}
				break;
			case 13:
				if (!wcsncasecmp(entry->type, L"Locality Name", 13)) {
					entry->typeLen = 1;
					entry->type = L"L";
				}
				break;
			case 14:
				if (!wcsncasecmp(entry->type, L"Street Address", 14)) {
					entry->typeLen = 2;
					entry->type = L"SA";
				}
				break;
			case 17:
				if (!wcsncasecmp(entry->type, L"Organization Name", 17)) {
					entry->typeLen = 1;
					entry->type = L"O";
				}
				break;
			case 22:
				if (!wcsncasecmp(entry->type, L"State or Province Name", 22)) {
					entry->typeLen = 1;
					entry->type = L"S";
				}
				break;
			case 24:
				if (!wcsncasecmp(entry->type, L"Organizational Unit Name", 24)) {
					entry->typeLen = 2;
					entry->type = L"OU";
				}
				break;
		}
	}
	if ((entry->typeLen == 1) && (entry->valLen > 2) &&
	    ((*entry->type == L'C') || (*entry->type == L'c'))) {
	    	free(entry);
		return ERR_COUNTRY_NAME_TOO_LONG;
	}
	if (*add) {
		struct RDNEntry** curr = add;
		struct RDNEntry* nextEntry = *curr;
		size_t typeLen = entry->typeLen;
		const wchar_t* type = entry->type;
		
		if (typeLen) {
			if (!nextEntry->typeLen) {
				free(entry);
				return ERR_ATTR_TYPE_NOT_EXPECTED;
			}
			do {
				int prop;
				size_t l;
				int cmp;
			
				nextEntry = *curr;
				if (!nextEntry)
					break;
				l = typeLen;
				prop = 0;
				if (typeLen < nextEntry->typeLen) {
					prop = -1;
				} else if (typeLen > nextEntry->typeLen) {
					l = nextEntry->typeLen;
					prop = 1;
				}
				cmp = wcsncasecmp(type, nextEntry->type, l);
				if (cmp < 0)
					break;
				if (cmp == 0) {
					if (prop < 0)
						break;
					if (!prop) {
						free(entry);
						return ERR_DUPLICATE_TYPE;
					}
				}
				curr = &nextEntry->next;
			} while (1);
		} else {
			if (nextEntry->typeLen) {
				free(entry);
				return ERR_ATTR_TYPE_EXPECTED;
			}
			do {
				nextEntry = *curr;
				if (!nextEntry)
					break;
				curr = &nextEntry->next;
			} while (1);
		}
		entry->next = nextEntry;
		*curr = entry;
		/* '+' rule in action*/
		if (next)
			*next = &((*add)->up);
	} else {
		*add = entry;
		if (next)
			*next = &entry->up;
	}
	return 0;
}

static void __NWDSDestroyRDN(struct RDNInfo* rdn) {
	struct RDNEntry* up;
	
	up = rdn->end;
	while (up) {
		struct RDNEntry* tmp;
		
		tmp = up;
		up = up->up;
		do {
			struct RDNEntry* tmp2;
			
			tmp2 = tmp;
			tmp = tmp->next;
			free(tmp2);
		} while (tmp);
	}
}

/* returned rdn contains pointers to DN! */
static NWDSCCODE __NWDSCreateRDN(struct RDNInfo* rdn, const wchar_t* dn, size_t* trailingDots) {
	NWDSCCODE err = 0;
	int first = 1;
	struct RDNEntry** add;
	size_t dots = 0;
	struct RDNEntry* currEntry = NULL;
		
	rdn->depth = 0;
	rdn->end = NULL;
	/* map empty string to nothing */
	if (!*dn) {
		if (trailingDots)
			*trailingDots = 0;
		return 0;
	}
	add = &rdn->end;
	do {
		wchar_t x;
		
		x = *dn++;
		if (!x) {
			if (dots)
				break;
			if (!currEntry || !currEntry->val) {
				err = ERR_EXPECTED_IDENTIFIER;
				break;	/* end; we did not expect anything */
			}
			currEntry->valLen = dn-currEntry->val-1;
			err = __NWDSAddRDN(add, currEntry, &add);
			currEntry = NULL;
			rdn->depth++;
			break;	/* we have value... */
		} else if (x == '.') {
			if (first || dots) {
				dots++;
			} else if (!currEntry || !currEntry->val) {
				err = ERR_EXPECTED_IDENTIFIER;
				break;
			} else {
				currEntry->valLen = dn-currEntry->val-1;
				err = __NWDSAddRDN(add, currEntry, &add);
				currEntry = NULL;
				rdn->depth++;
				dots=1;
			}
		} else if (x == '+') {
			if (dots || !currEntry || !currEntry->val) {
				err = ERR_EXPECTED_IDENTIFIER;
				break;
			}
			currEntry->valLen = dn-currEntry->val-1;
			err = __NWDSAddRDN(add, currEntry, NULL);
			currEntry = NULL;
		} else if (x == '=') {
			if (dots || !currEntry || !currEntry->val) {
				err = ERR_EXPECTED_IDENTIFIER;
				break;
			}
			if (currEntry->type) {
				err = ERR_EXPECTED_RDN_DELIMITER;
				break;
			}
			currEntry->type = currEntry->val;
			currEntry->typeLen = dn - currEntry->type - 1;
			currEntry->val = NULL;
		} else {
			if (first + dots > 1) {
				err = ERR_EXPECTED_IDENTIFIER;
				break;
			}
			first = dots = 0;
			if (!currEntry) {
				currEntry = (struct RDNEntry*)malloc(sizeof(*currEntry));
				if (!currEntry) {
					err = ERR_NOT_ENOUGH_MEMORY;
					break;
				}
				memset(currEntry, 0, sizeof(*currEntry));
			}
			if (!currEntry->val)
				currEntry->val = dn-1;
			if (x == '\\') {
				x = *dn++;
				if (!x) {
					err = ERR_EXPECTED_IDENTIFIER;
					break;
				}
			}
		}
	} while (!err);
	if (currEntry)
		free(currEntry);
	if (!err) {
		if (trailingDots)
			*trailingDots = dots;
		else if (dots)
			err = ERR_EXPECTED_IDENTIFIER;
	}
	if (err)
		__NWDSDestroyRDN(rdn);
	return err;
}

/* Charset conversion support for libc5 and libc6.0 */
/* libc6.1 contains iconv interface itself (but buggy and non-working) */
static int iconv_88591_to_wchar_t(const char** inp, size_t* inl,
		char** outp, size_t* outl) {
	const char* i;
	size_t il;
	wchar_t *o;
	size_t ol;
	int ret;
	
	ret = 0;
	ol = *outl;
	o = (wchar_t*)*outp;
	il = *inl;
	i = *inp;
	while (il) {
		if (ol < sizeof(*o)) {
			errno = E2BIG;
			ret = -1;
			goto end;
		}
		*o++ = (*i++) & 0xFF;
		il--;
		ol-=sizeof(*o);
		ret++;
	}
end:;
	*inp = i;
	*inl = il;
	*outp = (char*)o;
	*outl = ol;
	return ret;
}

static int iconv_wchar_t_to_88591(const char** inp, size_t* inl,
		char** outp, size_t* outl) {
	const wchar_t* i;
	size_t il;
	char *o;
	size_t ol;
	int ret;
	
	ret = 0;
	ol = *outl;
	o = *outp;
	il = *inl;
	i = (const wchar_t*)*inp;
	while (il >= sizeof(*i)) {
		if (ol < sizeof(*o)) {
			errno = E2BIG;
			ret = -1;
			goto end;
		}
		*o++ = (*i++) & 0xFF;
		il-=sizeof(*i);
		ol-=sizeof(*o);
		ret++;
	}
end:;
	*inp = (const char*)i;
	*inl = il;
	*outp = o;
	*outl = ol;
	return ret;
}

my_iconv_t my_iconv_open(const char* to, const char* from) {
	int (*p)(const char** inp, size_t* inl,
		char** outp, size_t* outl) = NULL;
	my_iconv_t ret;
		
	if (!strcmp(from, "WCHAR_T//")) {
		if (!strcmp(to, "ISO_8859-1//"))
			p=iconv_wchar_t_to_88591;
/*
		if (!strcmp(to, "UTF-8//"))
			return iconv_internal_to_utf8;
*/
	} else if (!strcmp(to, "WCHAR_T//")) {
		if (!strcmp(from, "ISO_8859-1//"))
			p=iconv_88591_to_wchar_t;
/* 
		if (!strcmp(from, "UTF-8//"))
			return iconv_utf8_to_internal;
*/
	}
	/* this conversion is not supported */
	if (!p) {
#ifdef HAVE_ICONV_H
		iconv_t h = iconv_open(to, from);
		if (h == (iconv_t)-1)
			return (my_iconv_t)-1;
		ret = (my_iconv_t)malloc(sizeof(*ret));
		if (!ret) {
			iconv_close(h);
			errno = ENOMEM;
			return (my_iconv_t)-1;
		}
		ret->lowlevel.h = h;
		ret->type = MY_ICONV_LIBC;
		return ret;
#else
		errno = EINVAL;
		return (my_iconv_t)-1;
#endif
	}
	ret = (my_iconv_t)malloc(sizeof(*ret));
	if (!ret) {
		errno = ENOMEM;
		return (my_iconv_t)-1;
	}
	ret->lowlevel.proc = p;
	ret->type = MY_ICONV_INTERNAL;
	return ret;
}

int my_iconv_close(my_iconv_t filter) {
	if (filter->type == MY_ICONV_INTERNAL) {
		;
#ifdef HAVE_ICONV_H		
	} else if (filter->type == MY_ICONV_LIBC) {
		iconv_close(filter->lowlevel.h);
#endif
	}
	free(filter);
	return 0;
}

int my_iconv(my_iconv_t filter, const char** inbuf, size_t* inbytesleft,
		char** outbuf, size_t* outbytesleft) {
	if (filter->type == MY_ICONV_INTERNAL) {
		if (inbuf && outbuf)
			return (filter->lowlevel.proc)(inbuf, inbytesleft, outbuf, outbytesleft);
		else
			return 0;
#ifdef HAVE_ICONV_H			
	} else if (filter->type == MY_ICONV_LIBC) {
		return iconv(filter->lowlevel.h, inbuf, inbytesleft, 
			outbuf, outbytesleft);
#endif			
	} else {
		errno = EBADF;
		return -1;
	}
}

static int iconv_is_wchar_encoding(const char* to, const char* from) {
	static const wchar_t expected[] = L"Test";
	my_iconv_t h;
	size_t il;
	size_t ol;
	const char* i;
	char* o;
	char obuf[40];
	int q;
	
	h = my_iconv_open(to, from);
	if (h == (my_iconv_t)-1)
		return -1;
	dfprintf(stderr, "open ok\n");
	i = "Test";
	il = 4;
	o = obuf;
	ol = sizeof(obuf);
	q = my_iconv(h, &i, &il, &o, &ol);
	my_iconv_close(h);
	if (q == -1)
		return -1;
	dfprintf(stderr, "conv ok, q=%d, il=%d, ol=%d\n", q, il, ol);
	if (sizeof(obuf)-ol != 4 * sizeof(wchar_t))
		return -1;
	dfprintf(stderr, "len ok\n");
	if (memcmp(obuf, expected, 4 * sizeof(wchar_t)))
		return -1;
	dfprintf(stderr, "ok ok\n");
	return 0;
}

static const char* iconv_search_wchar_name(const char* from) {
	static const char *(names[]) = {
		"WCHAR_T//",
		"WCHAR//",
		"UCS4//",
		"UCS4LE//",
		"UCS4LITTLE//",
		"UCS4BE//",
		"UCS4BIG//",
		"ISO-10646//",
		"ISO-10646-LE//",
		"ISO-10646-LITTLE//",
		"ISO-10646-BE//",
		"ISO-10646-BIG//",
		"INTERNAL//",
		NULL };
	const char **x;
	
	/* Yes, it is unbelievable...
	   with glibc up to glibc-2.1.1 (2.1.1 is latest at the time of writting)
	   iconv_open("ASD", "FGH");
	   iconv_open("ASD", "FGH");
	   coredumps in second iconv_open... So it is completely UNUSABLE */
	for (x = names; *x; x++) {
		dfprintf(stderr, "Testing: %s\n", *x);
		if (!iconv_is_wchar_encoding(*x, from))
			break;
	}
	return *x;
}
			
int __NWUUnicodeToInternal(wchar_t* dest, size_t destLen,
		const unicode* src, size_t srcLen, wchar_t* noMap /*ignored*/,
		size_t* len) {
	wchar_t* start = dest;
	wchar_t* stop;
		
	stop = dest + (destLen / sizeof(*dest));
	if (srcLen == (size_t)-1) {
		const unicode* p;
		
		p = src;
		while (*p++);
		srcLen = (p - src) * sizeof(*src);
	}
	while (srcLen >= sizeof(*src)) {
		wint_t chr = WVAL_LH(src++, 0);
		srcLen -= sizeof(*src);
		if (dest < stop) {
			*dest++ = chr;
			continue;
		}
		if (len)
			*len = (dest - start) * sizeof(*dest);
		return E2BIG;
	}
	if (len)
		*len = (dest - start) * sizeof(*dest);
	if (srcLen)
		return EINVAL;
	return 0;
}

int __NWUInternalToUnicode(unicode* dest, size_t destLen,
		const wchar_t* src, size_t srcLen, unicode* noMap, 
		size_t* len) {
	unicode* start = dest;
	unicode* stop;
	
	stop = dest + (destLen / sizeof(*dest));
	if (srcLen == (size_t)-1)
		srcLen = (wcslen(src) + 1) * sizeof(*src);
	while (srcLen >= sizeof(*src)) {
		int err;
		wint_t chr = *src++;
		srcLen -= sizeof(*src);
		if (chr < 0x10000) {
			if (dest < stop) {
				WSET_LH(dest++, 0, chr);
				continue;
			}
			err = E2BIG;
		} else {
			/* TODO: utf-16 add here... */
			if (noMap) {
				unicode* p = noMap;
				
				while (*p && (dest < stop)) {
					*dest++ = *p++;
				}
				if (!*p)
					continue;
				err = E2BIG;
			} else {
				err = EILSEQ;
			}
		}
		if (len)
			*len = (dest - start) * sizeof(*dest);
		return err;
	}
	if (len)
		*len = (dest - start) * sizeof(*dest);
	if (srcLen)
		return EINVAL;
	return 0;
}

int __NWULocalToInternal(my_iconv_t h, wchar_t* dest, size_t destLen,
		const char* src, size_t srcLen, wchar_t* noMap, size_t* len) {
	wchar_t* start = dest;
	int err = 0;
	
	/* put iconv into known state */
	my_iconv(h, NULL, NULL, NULL, NULL);
	if (srcLen == (size_t)-1)
		srcLen = (strlen(src) + 1) * sizeof(*src);
	while (srcLen > 0) {
		size_t n = my_iconv(h, &src, &srcLen, (char**)&dest, &destLen);
		if (n != (size_t)-1)
			break;
		err = errno;
		if ((err == EILSEQ) && noMap) {
			wchar_t* p = noMap;
			
			while (*p && (destLen >= sizeof(*dest))) {
				*dest++ = *p++;
				destLen -= sizeof(*dest);
			}
			if (!*p) {
				src++;
				srcLen -= sizeof(*src);
				continue;
			}
			err = E2BIG;
		}
		break;
	}
	if (len)
		*len = (dest - start) * sizeof(*dest);
	return err;
}

int __NWUInternalToLocal(my_iconv_t h, char* dest, size_t destLen,
		const wchar_t* src, size_t srcLen, char* noMap, size_t* len) {
	int err = 0;
	char* start = dest;
	
	/* GRRRR: why is not INTERNAL available for iconv?! */
	/* put iconv into known state */
	my_iconv(h, NULL, NULL, NULL, NULL);
	if (srcLen == (size_t)-1)
		srcLen = (wcslen(src) + 1) * sizeof(*src);
	while (srcLen > 0) {
		size_t n = my_iconv(h, (const char**)&src, &srcLen, &dest, &destLen);
		if (n != (size_t)-1)
			break;
		err = errno;
		if ((err == EILSEQ) && noMap) {
			char *p = noMap;
			
			while (*p && (destLen >= sizeof(*dest))) {
				*dest++ = *p++;
				destLen -= sizeof(*dest);
			}
			if (!*p) {
				src++;
				srcLen -= sizeof(*src);
				continue;
			}
			err = E2BIG;
		}
		break;
	}
	if (len)
		*len = (dest - start) * sizeof(*dest);
	return err;
}

NWDSCCODE NWDSInitRequester(void) {
	if (!default_encoding)
		default_encoding = strdup("ISO_8859-1//");
	if (!wchar_encoding) {
		wchar_encoding = iconv_search_wchar_name(default_encoding);
		if (!wchar_encoding) {
			wchar_encoding = iconv_search_wchar_name("US-ASCII//");
			/* return error... */
		}
	}
	dfprintf(stderr, "iconv: %s\n", wchar_encoding);
	return 0;
}

static inline NWDSCCODE NWDXIsValid(NWDS_HANDLE dsh) {
	return dsh ? 0 : ERR_NULL_POINTER;
}

static inline NWDSCCODE NWDSIsContextValid(NWDSContextHandle ctx) {
	return ctx ? 0 : ERR_BAD_CONTEXT;
}

static NWDSCCODE NWDXGetConnection(NWDS_HANDLE dsh, NWCONN_HANDLE* result) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	struct list_head* ptr;
	
	err = NWDXIsValid(dsh);
	if (err)
		return err;
	ptr = dsh->conns.next;
	if (list_empty(ptr))
		return ERR_NO_CONNECTION;
	conn = list_entry(ptr, struct ncp_conn, nds_ring);
	/* FIXME: mark connection as 'authentication disabled' ? */
	ncp_conn_use(conn);
	*result = conn;
	return 0;
}

static NWDSCCODE NWDSSetLastConnection(NWDSContextHandle ctx, NWCONN_HANDLE conn) {
	NWCONN_HANDLE connold;
	
	connold = ctx->dck.last_connection.conn;
	if (conn != connold) {
		if (ctx->dck.flags & DCV_DISALLOW_REFERRALS)
			return ERR_NO_REFERRALS;	/* FIXME! Compare with NWClient! */
		if (conn) {
			ncp_conn_store(conn);
			ctx->dck.last_connection.conn = conn;
			ctx->dck.last_connection.state = conn->state;
		} else {
			ctx->dck.last_connection.conn = NULL;
		}
		if (connold)
			ncp_conn_release(connold);
	}
	return 0;
}

static NWDSCCODE NWDSGetConnection(NWDSContextHandle ctx, NWCONN_HANDLE* result) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	conn = ctx->dck.last_connection.conn;
	if (conn) {
		if (conn->state == ctx->dck.last_connection.state) {
			ncp_conn_use(conn);
			/* FIXME: authentication disabled */
			*result = conn;
			return 0;
		}
		ncp_conn_release(conn);
		ctx->dck.last_connection.conn = NULL;
	}
	err = NWDXGetConnection(ctx->ds_connection, &conn);
	if (err)
		return err;
	err = NWDSSetLastConnection(ctx, conn);
	if (err) {
		NWCCCloseConn(conn);
		return err;
	}
	*result = conn;
	return 0;
}

static inline NWDSCCODE NWDSConnectionFinished(NWDSContextHandle ctx, NWCONN_HANDLE conn) {
	NWCCCloseConn(conn);
	return 0;
}

/* FIXME: Internal only! */
static NWDSCCODE NWDXAddConnection(NWDS_HANDLE dsh, NWCONN_HANDLE conn) {
	list_del(&conn->nds_ring);
	conn->state++;
	list_add(&conn->nds_ring, &dsh->conns);
	return 0;
}

NWDSCCODE NWDSAddConnection(NWDSContextHandle ctx, NWCONN_HANDLE conn) {
	return NWDXAddConnection(ctx->ds_connection, conn);
}

static NWDSCCODE __NWDSCreateDSConnection(NWDS_HANDLE *dsh) {
	NWDS_HANDLE tmp;
	
	tmp = (NWDS_HANDLE)malloc(sizeof(*tmp));
	if (!tmp) return ERR_NOT_ENOUGH_MEMORY;
	tmp->dck.tree_name = default_tree ? wcsdup(default_tree) : NULL;
	INIT_LIST_HEAD(&tmp->contexts);
	INIT_LIST_HEAD(&tmp->conns);
	*dsh = tmp;
	return 0;
}
	
static NWDSCCODE __NWDSReleaseDSConnection(NWDS_HANDLE dsh) {
	/* TODO: mutex */
	/* FIXME: ?? conns ?? */
	if (list_empty(&dsh->contexts) && list_empty(&dsh->conns)) {
		if (dsh->dck.tree_name)
			free(dsh->dck.tree_name);
		free(dsh);
	}
	return 0;
}

/* NWDSSetContext supports only 2 transports */
NWDSCCODE NWDSSetTransport(NWDSContextHandle ctx, size_t len, const NET_ADDRESS_TYPE* transports) {
	NWDSCCODE err;
	nuint32* ptr;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	if (len > 20) /* some reasonable limit */
		return NWE_PARAM_INVALID;
	if (len) {
		size_t cnt;
		nuint32* ptr2;
		
		ptr2 = ptr = (nuint32*)malloc(len * sizeof(nuint32));
		if (!ptr)
			return ERR_NOT_ENOUGH_MEMORY;
		for (cnt = len; cnt; cnt--)
			DSET_LH(ptr2++, 0, *transports++);
	} else {
		ptr = NULL;
	}
	if (ctx->dck.transport_types)
		free(ctx->dck.transport_types);
	ctx->dck.transport_types = ptr;
	ctx->dck.transports = len;
	return 0;
}

NWDSCCODE NWDSCreateContextHandle(NWDSContextHandle *ctx) {
	NWDSContextHandle tmp;
	NWDSCCODE err;
	NWDS_HANDLE dsh;
	
	err = __NWDSCreateDSConnection(&dsh);
	if (err)
		return err;	
	tmp = (NWDSContextHandle)malloc(sizeof(*tmp));
	if (!tmp) {
		__NWDSReleaseDSConnection(dsh);
		return ERR_NOT_ENOUGH_MEMORY;
	}
	memset(tmp, 0, sizeof(*tmp));
	INIT_LIST_HEAD(&tmp->context_ring);
	tmp->dck.flags = DCV_DEREF_ALIASES | DCV_XLATE_STRINGS | DCV_CANONICALIZE_NAMES;
	tmp->dck.name_form = 0;
	tmp->dck.last_connection.conn = NULL;
	tmp->dck.local_charset = NULL;
	tmp->dck.confidence = DCV_LOW_CONF;
	tmp->xlate.from = (my_iconv_t)-1;
	tmp->xlate.to = (my_iconv_t)-1;
	tmp->ds_connection = dsh;
	err = NWDSSetContext(tmp, DCK_LOCAL_CHARSET, default_encoding);
	if (err) {
		NWDSFreeContext(tmp);
		return err;
	}
	err = NWDSSetContext(tmp, DCK_NAME_CONTEXT, L"[Root]");
	if (err) {
		NWDSFreeContext(tmp);
		return err;
	}
	{
		static const nuint32 t[] = {NT_IPX, NT_IPX};

		err = NWDSSetTransport(tmp, sizeof(t)/sizeof(t[0]), t);
	}
	if (err) {
		NWDSFreeContext(tmp);
		return err;
	}
	*ctx = tmp;
	return 0;
}

/* must be able to free partially allocated context handle */
NWDSCCODE NWDSFreeContext(NWDSContextHandle ctx) {
	if (ctx->ds_connection) {
		list_del(&ctx->context_ring);
		__NWDSReleaseDSConnection(ctx->ds_connection);
	}
	if (ctx->xlate.from != (my_iconv_t)-1)
		my_iconv_close(ctx->xlate.from);
	if (ctx->xlate.to != (my_iconv_t)-1)
		my_iconv_close(ctx->xlate.to);
	if (ctx->dck.local_charset)
		free(ctx->dck.local_charset);
	if (ctx->dck.transport_types)
		free(ctx->dck.transport_types);
	__NWDSDestroyRDN(&ctx->dck.rdn);
	if (ctx->dck.namectx)
		free(ctx->dck.namectx);
	free(ctx);
	return 0;
}

NWDSContextHandle NWDSCreateContext(void) {
	NWDSContextHandle ctx;
	NWDSCCODE err;
	
	err = NWDSCreateContextHandle(&ctx);
	if (err)
		return (NWDSContextHandle)ERR_CONTEXT_CREATION;
	return ctx;
}

NWDSCCODE NWDSSetContext(NWDSContextHandle ctx, int key, const void* ptr) {
	NWDSCCODE err;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	switch (key) {
		case DCK_FLAGS:
			{
				ctx->dck.flags = *(const nuint32*)ptr;
			}
			return 0;
		case DCK_CONFIDENCE:
			{
				/* confidence can be any value... */
				ctx->dck.confidence = *(const nuint32*)ptr;
			}
			return 0;
		case DCK_NAME_CONTEXT:
			{
				wchar_t *namectx;
				struct RDNInfo ctxRDN;
				
				namectx = (wchar_t*)malloc(sizeof(wchar_t) * 1024);
				if (!namectx)
					return ERR_NOT_ENOUGH_MEMORY;
				err = NWDSXlateFromCtx(ctx, namectx, sizeof(wchar_t)*1024, ptr);
				if (err)
					return err;
				if (!wcscasecmp(namectx, L"[Root]")) {
					ctxRDN.end = NULL;
					ctxRDN.depth = 0;
				} else {
					err = __NWDSCreateRDN(&ctxRDN, namectx, NULL);
					if (err) {
						free(namectx);
						return err;
					}
				}
				__NWDSDestroyRDN(&ctx->dck.rdn);
				memcpy(&ctx->dck.rdn, &ctxRDN, sizeof(ctxRDN));
				free(ctx->dck.namectx);
				ctx->dck.namectx = namectx;
			}
			return 0;
		case DCK_NAME_FORM:
			{
				nuint32 v = *(const nuint32*)ptr;
				nuint32 x;
				
				if (v == DCV_NF_PARTIAL_DOT)
					x = 0;
				else if (v == DCV_NF_FULL_DOT)
					x = 4;
				else if (v == DCV_NF_SLASH)
					x = 2;
				else
					x = 0;
	/* 0: CN=TEST.OU=VC.O=CVUT.C=CZ (NF_PARTIAL_DOT)
	 * 1: TEST.VC.CVUT.CZ
	 * 2: \T=CVUT\C=CZ\O=CVUT\OU=VC\CN=TEST (NF_SLASH)
	 * 3: \CVUT\CZ\CVUT\VC\TEST
	 * 4: CN=TEST.OU=VC.O=CVUT.C=CZ.T=CVUT. (NF_FULL_DOT)
	 * 5: TEST.VC.CVUT.CZ.CVUT.
	 * 6 & 7 == 4 & 5
	 * 8..9:  Structured (1), unsupported (?)
	 * 10..15: Mirror of 8..9
	 * 16..17: Structured (2), unsupported (?)
	 * 18..31: Mirror of 16..17
	 * other:  Only low 5 bits are significant on NW5 (DS 7.28)
	 * Structured (1) and (2) are not supported ATM TODO */
				ctx->dck.name_form = x;
			}
			return 0;
		case DCK_LOCAL_CHARSET:
			{
				my_iconv_t f;
				my_iconv_t t;
				const char* name = ptr;
				
				if (!name)
					return ERR_NULL_POINTER;
				
				if (ctx->dck.local_charset && !strcmp(name, ctx->dck.local_charset))
					return 0;
				f = my_iconv_open(wchar_encoding, ptr);
				if (f == (my_iconv_t)-1)
					return ERR_UNICODE_FILE_NOT_FOUND;
				t = my_iconv_open(ptr, wchar_encoding);
				if (t == (my_iconv_t)-1) {
					my_iconv_close(f);
					return ERR_UNICODE_FILE_NOT_FOUND;
				}
				if (ctx->xlate.from != (my_iconv_t)-1)
					my_iconv_close(ctx->xlate.from);
				ctx->xlate.from = f;
				if (ctx->xlate.to != (my_iconv_t)-1)
					my_iconv_close(ctx->xlate.to);
				ctx->xlate.to = t;
				if (ctx->dck.local_charset)
					free(ctx->dck.local_charset);
				ctx->dck.local_charset = strdup(name);
				return 0;
			}
	}
	return ERR_BAD_KEY;
}

NWDSCCODE NWDSGetContext2(NWDSContextHandle ctx, int key, void* ptr, size_t maxlen) {
	NWDSCCODE err;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	switch (key) {
		case DCK_FLAGS:
			if (maxlen < sizeof(u_int32_t))
				return NWE_BUFFER_OVERFLOW;
			*(u_int32_t*)ptr = ctx->dck.flags;
			return 0;
		case DCK_CONFIDENCE:
			if (maxlen < sizeof(u_int32_t))
				return NWE_BUFFER_OVERFLOW;
			*(u_int32_t*)ptr = ctx->dck.confidence;
			return 0;
		case DCK_NAME_CONTEXT:
			err = NWDSXlateToCtx(ctx, ptr, maxlen, ctx->dck.namectx ? : L"[Root]");
			return err;
		case DCK_RDN:
			if (maxlen < sizeof(ctx->dck.rdn))
				return NWE_BUFFER_OVERFLOW;
			memcpy(ptr, &ctx->dck.rdn, sizeof(ctx->dck.rdn));
			return 0;
		case DCK_LAST_CONNECTION:
			if (maxlen < sizeof(NWCONN_HANDLE))
				return NWE_BUFFER_OVERFLOW;
			*(NWCONN_HANDLE*)ptr = ctx->dck.last_connection.conn;
			return 0;
		case DCK_TREE_NAME:
			{
				static const wchar_t null = 0;
				const wchar_t* src = ctx->ds_connection->dck.tree_name;
				
				if (!src) src = &null;
				err = NWDSXlateToCtx(ctx, ptr, maxlen, src);
			}
			return err;
		case DCK_NAME_FORM:
			if (maxlen < sizeof(nuint32))
				return NWE_BUFFER_OVERFLOW;
			{
				nuint32* nf = ptr;

				if (ctx->dck.name_form == 4)
					*nf = DCV_NF_FULL_DOT;
				else if (ctx->dck.name_form == 2)
					*nf = DCV_NF_SLASH;
				else
					*nf = DCV_NF_PARTIAL_DOT;
			}
			return 0;
	}
	return ERR_BAD_KEY;
}
	
NWDSCCODE NWDSGetContext(NWDSContextHandle ctx, int key, void* ptr) {
	static size_t optlen[] =
		{                     0, sizeof(u_int32_t),
		                      sizeof(u_int32_t),      MAX_DN_BYTES,
		                      0,                 0,
		 sizeof(struct RDNInfo),		 0,
		  sizeof(NWCONN_HANDLE), 	         0,                      
		  		      0,		 0,
		                      0,               128,
				      0,		 0};
				      
	if ((key >= DCK_FLAGS) && (key <= DCK_TREE_NAME))
		return NWDSGetContext2(ctx, key, ptr, optlen[key]);
	else
		return ERR_BAD_KEY;
	
}

#define NWDSBUFT_AUTOSIZE	0x01000000

NWDSCCODE NWDSAllocBuf(size_t len, pBuf_T* buff) {
	nuint8* buffer;
	Buf_T* buf;
	
	*buff = NULL;
	buf = (Buf_T*)malloc(sizeof(*buf));
	
	len = (len + 3) & ~3;
	if (!buf)
		return ERR_NOT_ENOUGH_MEMORY;
	buf->bufFlags = 0;
	if (!len) {
		buf->bufFlags |= NWDSBUFT_AUTOSIZE;
		len = 32;
	}
	buffer = (nuint8*)malloc(len);
	if (!buffer) {
		free(buf);
		return ERR_NOT_ENOUGH_MEMORY;
	}
	/* if (buffer & 3) { panic("Sorry, malloc must return dword aligned value"}; */
	buf->curPos = buf->data = buffer;
	buf->allocend = buf->dataend = buffer + len;
	*buff = buf;
	return 0;
}

/* Get only! */
NWDSCCODE NWDSCreateBuf(Buf_T** buff, void* ptr, size_t len) {
	Buf_T* buf;
	
	*buff = NULL;
	buf = (Buf_T*)malloc(sizeof(*buf));
	if (!buf)
		return ERR_NOT_ENOUGH_MEMORY;
	buf->bufFlags = 0;
	buf->data = NULL;
	buf->curPos = ptr;
	buf->allocend = ptr + len;
	buf->dataend = ptr + len;
	*buff = buf;
	return 0;
}

NWDSCCODE NWDSFreeBuf(Buf_T* buf) {
	if (buf) {
		if (buf->data) {
			free(buf->data);
			buf->data = NULL;
		}
		free(buf);
	}
	return 0;	/* easy... */
}

static inline void NWDSBufStartPut(Buf_T* buf) {
	buf->dataend = buf->allocend;
	buf->curPos = buf->data;
}

static inline void NWDSBufFinishPut(Buf_T* buf) {
	buf->dataend = buf->curPos;
	buf->curPos = buf->data;
}
	
static inline NWDSCCODE NWDSBufGetLE32(Buf_T* buf, nuint32* val) {
	if (buf->curPos + sizeof(*val) <= buf->dataend) {
		*val = DVAL_LH(buf->curPos, 0);
		buf->curPos += sizeof(*val);
		return 0;
	} else {
		buf->curPos = buf->dataend;
		return ERR_BUFFER_EMPTY;
	}	
}

static inline NWDSCCODE NWDSBufPeekLE32(Buf_T* buf, size_t off, nuint32* val) {
	nuint8* q;
	
	q = buf->curPos + off;
	if (q + sizeof(*val) <= buf->dataend) {
		*val = DVAL_LH(q, 0);
		return 0;
	} else {
		return ERR_BUFFER_EMPTY;
	}
}

/* User ID is an abstract type :-) It is 4 bytes long, big-endian
   (to preserve 0x00000001 = [Supervisor]) */
static inline NWDSCCODE NWDSBufGetID(Buf_T* buf, NWObjectID* ID) {
	if (buf->curPos + 4 <= buf->dataend) {
		*ID = DVAL_HL(buf->curPos, 0);
		buf->curPos += 4;
		return 0;
	} else {
		buf->curPos = buf->dataend;
		return ERR_BUFFER_EMPTY;
	}
}

static inline NWDSCCODE NWDSBufGet(Buf_T* buf, void* buffer, size_t len) {
	if (buf->curPos + len <= buf->dataend) {
		memcpy(buffer, buf->curPos, len);
		buf->curPos += (len + 3) & ~3;
		return 0;
	} else {
		buf->curPos = buf->dataend;
		return ERR_BUFFER_EMPTY;
	}
}

static inline void* NWDSBufGetPtr(Buf_T* buf, size_t len) {
	if (buf->curPos + len <= buf->dataend) {
		void* r = buf->curPos;
		buf->curPos += (len + 3) & ~3;
		return r;
	} else {
		buf->curPos = buf->dataend;
		return NULL;
	}
}

static inline void* NWDSBufPeekPtr(Buf_T* buf) {
	if (buf->curPos <= buf->dataend)
		return buf->curPos;
	else
		return NULL;
}

static inline void NWDSBufSeek(Buf_T* buf, void* ptr) {
	buf->curPos = ptr;
}

static inline NWDSCCODE NWDSBufPutLE32(Buf_T* buf, nuint32 val) {
	if (buf->curPos + sizeof(val) <= buf->dataend) {
		DSET_LH(buf->curPos, 0, val);
		buf->curPos += sizeof(val);
		return 0;
	} else {
		/* TODO: honor NWDSBUFT_AUTOSIZE */
		buf->curPos = buf->dataend;
		return ERR_BUFFER_FULL;
	}	
}

static inline void* NWDSBufPutPtr(Buf_T* buf, size_t len) {
	return NWDSBufGetPtr(buf, len);
}

static inline void* NWDSBufPutPtrLen(Buf_T* buf, size_t* len) {
	*len = buf->dataend - buf->curPos;
	return buf->curPos;
}

/* len <= free space !!! */
static inline void NWDSBufPutSkip(Buf_T* buf, size_t len) {
	buf->curPos += (len + 3) & ~3;
}

static inline NWDSCCODE NWDSBufPut(Buf_T* buf, const void* buffer, size_t len) {
	if (buf->curPos + len <= buf->dataend) {
		if (len & 3)
			*(nuint32*)(buf->curPos + (len & ~3)) = 0;
		memcpy(buf->curPos, buffer, len);
		buf->curPos += (len + 3) & ~3;
		return 0;
	} else {
		/* TODO: honor NWDSBUFT_AUTOSIZE */
		return ERR_BUFFER_FULL;
	}
}

#define NWDSFINDCONN_NOCREATE		0x0000
#define NWDSFINDCONN_CREATEALLOWED	0x0001
static NWDSCCODE NWDXFindConnection(NWDS_HANDLE dsh, NWCONN_HANDLE* pconn, Buf_T* addr, int flags) {
	struct list_head* stop = &dsh->conns;
	u_int32_t saddr, saddr2;
	void* p;
	NWDSCCODE err;
	
	err = NWDSBufGetLE32(addr, &saddr);
	if (err)
		return err;
	saddr2 = saddr;
	p = NWDSBufPeekPtr(addr);
	while (saddr--) {
		struct list_head* current;
		nuint32 addrtype;
		nuint32 len;
		void* data;
		
		err = NWDSBufGetLE32(addr, &addrtype);
		if (err)
			break;
		err = NWDSBufGetLE32(addr, &len);
		if (err)
			break;
		data = NWDSBufGetPtr(addr, len);
		if (!data) {
			err = ERR_BUFFER_EMPTY;
			break;
		}
		
		for (current = stop->next; current != stop; current = current->next) {
			NWCONN_HANDLE conn = list_entry(current, struct ncp_conn, nds_ring);
			size_t connaddresses;
			nuint8* conndata;
			/* compare addresses */
			
			if (__NWCCGetServerAddressPtr(conn, &connaddresses, &conndata))
				continue;
			while (connaddresses--) {
				if ((DVAL_LH(conndata, 0) == addrtype) &&
				    (DVAL_LH(conndata, 4) == len) &&
				    !memcmp(conndata+8, data, len)) {
				    	ncp_conn_use(conn);
					*pconn = conn;
					return 0;
				}
				conndata += 4 + ((3 + DVAL_LH(conndata, 4)) & ~3);
			}
			
		}
	}
	if (flags & NWDSFINDCONN_CREATEALLOWED) {
		saddr = saddr2;
		NWDSBufSeek(addr, p);
		while (saddr--) {
			nuint32 addrtype;
			nuint32 len;
			void* data;
			NWCCTranAddr tran;
		
			err = NWDSBufGetLE32(addr, &addrtype);
			if (err)
				break;
			err = NWDSBufGetLE32(addr, &len);
			if (err)
				break;
			data = NWDSBufGetPtr(addr, len);
			if (!data) {
				err = ERR_BUFFER_EMPTY;
				break;
			}
			switch (addrtype) {
				case NT_IPX:	tran.type = NWCC_TRAN_TYPE_IPX; break;
				case NT_UDP:	tran.type = NWCC_TRAN_TYPE_UDP; break;
				default:	goto nextAddr;
			}
			tran.buffer = data;
			tran.len = len;
			err = NWCCOpenConnByAddr(&tran, 0, NWCC_RESERVED, pconn);
			if (!err)
				return 0;	/* TODO: Authenticate connection */
nextAddr:;
		}
	}
	if (err)
		return err;
	return ERR_NO_CONNECTION;	
}

static NWDSCCODE NWDSFindConnection(NWDSContextHandle ctx, NWCONN_HANDLE* pconn, Buf_T* addr, int flags) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
		
	err = NWDXFindConnection(ctx->ds_connection, &conn, addr, flags);
	if (err)
		return err;
	err = NWDSSetLastConnection(ctx, conn);
	if (err) {
		NWCCCloseConn(conn);
		return err;
	}
	*pconn = conn;
	return 0;
	
		
}

static inline NWDSCCODE NWDSGetDSIRaw(NWCONN_HANDLE conn, nuint32 flags, 
		nuint32 DNstyle, NWObjectID id, Buf_T* reply) {
	unsigned char x[16];
	unsigned char rpl[4096];
	size_t rpl_len;
	NWDSCCODE err;
	
	DSET_LH(x, 0, 2);	/* version */
	DSET_LH(x, 4, DNstyle);
	DSET_LH(x, 8, flags);
	DSET_HL(x, 12, id);
	err = ncp_send_nds_frag(conn, DSV_READ_ENTRY_INFO, x, 16, rpl,
		sizeof(rpl), &rpl_len);
	if (!err) {
		dfprintf(stderr, "Reply len: %u\n", rpl_len);
		NWDSBufStartPut(reply);
		err = NWDSBufPut(reply, rpl, rpl_len);
		NWDSBufFinishPut(reply);
	}
	return err;
}

static NWDSCCODE NWDSXlateToCtx(NWDSContextHandle ctx, void* data, 
		size_t maxlen, const wchar_t* src) {
	NWDSCCODE err;
	nuint32 val;
	
	err = NWDSGetContext(ctx, DCK_FLAGS, &val);
	if (err)
		return err;
	if (val & DCV_XLATE_STRINGS) {
		err = __NWUInternalToLocal(ctx->xlate.to, data, maxlen, 
			src, -1, NULL, NULL);
		if (err)
			return ERR_DN_TOO_LONG; /* or INVALID MULTIBYTE */
	} else {
		err = __NWUInternalToUnicode(data, maxlen, src, -1, NULL, NULL);
	}
	return err;
}

static NWDSCCODE NWDSXlateFromCtx(NWDSContextHandle ctx, wchar_t* dst,
		size_t maxlen, const void* src) {
	NWDSCCODE err;
	nuint32 val;
	
	err = NWDSGetContext(ctx, DCK_FLAGS, &val);
	if (err)
		return err;
	if (val & DCV_XLATE_STRINGS) {
		err = __NWULocalToInternal(ctx->xlate.from, dst, maxlen, 
			src, -1, NULL, NULL);
		if (err)
			return ERR_DN_TOO_LONG;	/* or INVALID MULTIBYTE */
	} else {
		err = __NWUUnicodeToInternal(dst, maxlen, src, -1, NULL, NULL);
	}
	return err;
}

static NWDSCCODE NWDSBufCIString(Buf_T* buffer,	wchar_t* data, size_t maxlen) {
	NWDSCCODE err;
	nuint32 len32;
	size_t len;
	unicode* ptr;
	
	err = NWDSBufGetLE32(buffer, &len32);
	if (err)
		return err;
	len = len32;
	if (len & 1)
		return ERR_INVALID_OBJECT_NAME;
	ptr = NWDSBufGetPtr(buffer, len);
	if (!ptr)
		return ERR_BUFFER_EMPTY;
	if (ptr[(len>>1)-1])	/* 0 at the end of string */
		return ERR_INVALID_OBJECT_NAME;
	if (data) {
		err = __NWUUnicodeToInternal(data, maxlen, ptr, len, NULL, &len);
		if (err)
			return ERR_DN_TOO_LONG;
	}
	return 0;
}

static inline NWDSCCODE NWDSBufSkipCIString(Buf_T* buffer) {
	return NWDSBufCIString(buffer, NULL, 0);
}

static NWDSCCODE NWDSGetAttrVal_DIST_NAME(NWDSContextHandle ctx, Buf_T* buffer,
		void* ret, nuint32 replyFmt) {
	NWDSCCODE err;
	nuint32 v;
	wchar_t tmpb[MAX_DN_CHARS+1];
		
	err = NWDSGetContext(ctx, DCK_FLAGS, &v);
	if (err)
		return err;
	/* TYPELESS NAMES: should be OK from server (check NW4.0x) */
	err = NWDSBufCIString(buffer, tmpb, sizeof(tmpb));
	if (err)
		return err;
	if ((v & DCV_CANONICALIZE_NAMES) && (ctx->dck.name_form == 0)) {
		wchar_t abbrev[MAX_DN_CHARS+1];
		
		err = NWDSAbbreviateNameW(ctx, tmpb, abbrev);
		if (!err)
			err = NWDSXlateToCtx(ctx, ret, MAX_DN_BYTES, abbrev);
	} else {
		err = NWDSXlateToCtx(ctx, ret, MAX_DN_BYTES, tmpb);
	}
	return err;
}

static NWDSCCODE NWDSBufPutCIString(Buf_T* buffer, size_t len, const wchar_t* string) {
	NWDSCCODE err;
	size_t maxlen;
	void* lenspace;
	unicode* strspace;
	
	lenspace = NWDSBufPutPtr(buffer, 4);
	if (!lenspace)
		return ERR_BUFFER_FULL;
	strspace = NWDSBufPutPtrLen(buffer, &maxlen);
	err = __NWUInternalToUnicode(strspace, maxlen, string, len, NULL, &maxlen);
	if (err)
		return ERR_BUFFER_FULL;
	DSET_LH(lenspace, 0, maxlen);
	NWDSBufPutSkip(buffer, maxlen);
	return 0;
}

static NWDSCCODE NWDSPutAttrVal_DIST_NAME(NWDSContextHandle ctx, Buf_T* buffer,
		const char* name) {
	NWDSCCODE err;
	nuint32 v;
	wchar_t tmpb[MAX_DN_CHARS+1];
		
	err = NWDSGetContext(ctx, DCK_FLAGS, &v);
	if (err)
		return err;
	err = NWDSXlateFromCtx(ctx, tmpb, sizeof(tmpb), name);
	if (err)
		return err;
	if ((v & DCV_CANONICALIZE_NAMES) && (ctx->dck.name_form == 0)) {
		wchar_t dst[MAX_DN_CHARS+1];
		
		err = NWDSCanonicalizeNameW(ctx, tmpb, dst);
		if (!err)
			err = NWDSBufPutCIString(buffer, -1, dst);
	} else
		err = NWDSBufPutCIString(buffer, -1, tmpb);
	return err;
}

NWDSCCODE NWDSComputeAttrValSize(NWDSContextHandle ctx, Buf_T* buffer,
		enum SYNTAX syntaxID, size_t* valsize) {
	NWDSCCODE err;
	nuint32 le32;
	size_t size = 0;
	
	switch (syntaxID) {
		case SYN_NET_ADDRESS:
			err = NWDSBufPeekLE32(buffer, 4, &le32);
			if (!err)
				size = le32 + sizeof(Net_Address_T);
			break;
		/* FIXME: other syntaxes... */
		default:
			err = ERR_NO_SUCH_SYNTAX;
			break;
	}
	if (!err && valsize)
		*valsize = size;
	return err;
}

static NWDSCCODE NWDSGetAttrVal_NET_ADDRESS(NWDSContextHandle ctx, Buf_T* buffer,
		Net_Address_T* nt) {
	nuint32 type, len;
	NWDSCCODE err;
	void* p;
	
	err = NWDSBufGetLE32(buffer, &type);
	if (err)
		return err;
	err = NWDSBufGetLE32(buffer, &len);
	if (err)
		return err;
	p = NWDSBufGetPtr(buffer, len);
	if (!p)
		return ERR_BUFFER_EMPTY;
	if (nt) {
		nt->addressType = type;
		nt->addressLength = len;
		memcpy(nt->address = ((nuint8*)nt) + sizeof(*nt), p, len);
	}
	return err;
}

NWDSCCODE NWDSGetAttrVal(NWDSContextHandle ctx, Buf_T* buffer,
		enum SYNTAX syntaxID, void* ptr) {
	NWDSCCODE err;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	switch (syntaxID) {
		case SYN_DIST_NAME:
			err = NWDSGetAttrVal_DIST_NAME(ctx, buffer, ptr, ctx->dck.name_form);
			break;
		case SYN_NET_ADDRESS:
			err = NWDSGetAttrVal_NET_ADDRESS(ctx, buffer, ptr);
			break;
		default:
			err = ERR_NO_SUCH_SYNTAX;
			break;
	}
	return err;
}

static NWDSCCODE __NWDSApplyDefaultNamingRule(const struct RDNInfo* rdn) {
	size_t depth = rdn->depth;
	if (depth > 0) {
		struct RDNEntry* level = rdn->end;
		wchar_t* type = L"CN";
		size_t typeLen = 2;	/* wcslen(type) */
		while (--depth) {
			if (!level->typeLen) {
				if (level->next)
					return ERR_INCONSISTENT_MULTIAVA;
				level->type = type;
				level->typeLen = typeLen;
			}
			type = L"OU";
			typeLen = 2;	/* wcslen(type) */
			level = level->up;
		}
		if (!level->typeLen) {
			if (level->next)
				return ERR_INCONSISTENT_MULTIAVA;
			level->type = L"O";
			level->typeLen = 1; /* wcslen(type) */
		}
	}
	return 0;
}

static NWDSCCODE __NWDSCopyRDN(struct RDNEntry** dst, struct RDNEntry* src) {
	while (src) {
		struct RDNEntry* tmp = src;
		struct RDNEntry** dst2 = dst;
		
		do {
			struct RDNEntry* wrt = *dst2 = (struct RDNEntry*)malloc(sizeof(*wrt));
			if (!wrt)
				return ERR_NOT_ENOUGH_MEMORY;
			wrt->type = tmp->type;
			wrt->typeLen = tmp->typeLen;
			wrt->val = tmp->val;
			wrt->valLen = tmp->valLen;
			wrt->up = NULL; /* wrt->next is set later */
			dst2 = &wrt->next;
			tmp = tmp->next;
		} while (tmp);
		*dst2 = NULL;
		src = src->up;
		dst = &(*dst)->up;
	}
	return 0;
}

static NWDSCCODE __NWDSExtractRDN(struct RDNInfo* rdn, wchar_t* dst, int typeLess, size_t tdots) {
	struct RDNEntry* entry;
	
	entry = rdn->end;
	if (entry || tdots) {
		while (entry) {
			struct RDNEntry* en;
		
			en = entry;
			while (en) {
				if (!typeLess && en->typeLen) {
					memcpy(dst, en->type, en->typeLen * sizeof(wchar_t));
					dst += en->typeLen;
					*dst++ = '=';
				}
				memcpy(dst, en->val, en->valLen * sizeof(wchar_t));
				dst += en->valLen;
				en = en->next;
				if (en)
					*dst++ = '+';
			}
			entry = entry->up;
			if (entry)
				*dst++ = '.';
		}
		for (;tdots;tdots--)
			*dst++ = '.';
		*dst++ = 0;
	} else
		wcscpy(dst, L"[Root]");
	return 0;
}

/* DN modification procedures */
NWDSCCODE NWDSRemoveAllTypesW(NWDSContextHandle ctx, const wchar_t* src,
		wchar_t* dst) {
	wchar_t c;
	wchar_t lc = 0;
	wchar_t* dst2 = dst;
	int bdot = 0;
	int mdot = 0;
	int eq = 0;
	
	while ((c = *src++) != 0) {
		if (c == '.') {
			if (lc == '.')
				mdot = 1;
			else if (dst == dst2) {
				if (!lc)
					bdot = 1;
				else 
					return ERR_EXPECTED_IDENTIFIER;
			}
			*dst++ = c;
			dst2 = dst;
			eq = 0;
		} else if (mdot) {
			return ERR_INVALID_DS_NAME;
		} else if (c == '=') {
			if (eq)
				return ERR_EXPECTED_RDN_DELIMITER;
			if (dst == dst2)
				return ERR_EXPECTED_IDENTIFIER;	/* empty before = */
			dst = dst2;
			eq = 1;
		} else if (c == '+') {
			if (dst == dst2)
				return ERR_EXPECTED_IDENTIFIER;	/* empty before + */
			*dst++ = c;
			dst2 = dst;
			eq = 0;
		} else {
			*dst++ = c;
			if (c == L'\\') {
				wchar_t tmp_c;
				
				tmp_c = *src++;
				if (!tmp_c)
					return ERR_INVALID_DS_NAME;
				*dst++ = tmp_c;
			}
		}
		lc = c;
	}
	if ((lc == 0) || (bdot && (dst == dst2)))
		return ERR_INVALID_DS_NAME;
	*dst++ = c;
	return 0;
}

NWDSCCODE NWDSCanonicalizeNameW(NWDSContextHandle ctx, const wchar_t* src,
		wchar_t* dst) {
	struct RDNInfo rdn;
	struct RDNInfo ctxRDN;
	struct RDNEntry* ctxEntry;
	struct RDNEntry** rdnEntry;
	size_t dots;
	size_t oldDepth;
	int abs = 0;
	NWDSCCODE err;
	int typeLess;
	u_int32_t flags;
	
	err = NWDSGetContext(ctx, DCK_FLAGS, &flags);
	if (err)
		return err;
	typeLess = (flags & DCV_TYPELESS_NAMES) ? 1 : 0;
	/* check for specials... */
	if (*src == '[') {
		if (!wcscasecmp(src, L"[Root]") ||
		    !wcscasecmp(src, L"[Supervisor]") ||
		    !wcscasecmp(src, L"[Public]") ||
		    !wcscasecmp(src, L"[Self]") ||
		    !wcscasecmp(src, L"[Creator]") ||
		    !wcscasecmp(src, L"[Inheritance Mask]") ||
		    !wcscasecmp(src, L"[Root Template]") ||
		    !wcscasecmp(src, L"[Nothing]")) {
		    	wcscpy(dst, src);
			return 0;
		}
	}
	if (*src == '.') {
		abs = 1;
		src++;
	}
	err = __NWDSCreateRDN(&rdn, src, &dots);
	if (err)
		return err;
	err = NWDSGetContext2(ctx, DCK_RDN, &ctxRDN, sizeof(ctxRDN));
	if (err) {
		__NWDSDestroyRDN(&rdn);
		return err;
	}
	if (abs) {
		if (dots) {
			if (rdn.depth) {
				__NWDSDestroyRDN(&rdn);
				return ERR_INVALID_OBJECT_NAME;
			}
			dots++;
		} else {
			if (rdn.depth) {
				dots = ctxRDN.depth;
			} else
				dots++;
		}
	}
	if (dots > ctxRDN.depth) {
		__NWDSDestroyRDN(&rdn);
		return ERR_TOO_MANY_TOKENS;
	}
	ctxEntry = ctxRDN.end;
	rdnEntry = &rdn.end;
	oldDepth = rdn.depth;
	rdn.depth = rdn.depth + ctxRDN.depth - dots;
	if (dots > oldDepth) {
		while (dots-- > oldDepth) {
			ctxEntry = ctxEntry->up;
		}
	} else if (dots < oldDepth) {
		while (dots++ < oldDepth) {
			rdnEntry = &(*rdnEntry)->up;
		}
	}
	if (typeLess) {
		/* skip to end, ignore types */
		while (*rdnEntry) {
			rdnEntry = &(*rdnEntry)->up;
			ctxEntry = ctxEntry->up;
		}
	} else {
		/* copy types */
		while (*rdnEntry) {
			struct RDNEntry* re;
			
			re = *rdnEntry;
			/* copy types from context if DN have no types */
			/* and context has types */
			/* If both are typeless, DefaultNamingRule tries to fix it */
			if (!re->typeLen && ctxEntry->typeLen) {
				struct RDNEntry* ctxE = ctxEntry;
				
				while (re) {
					/* context has less items than DN: boom */
					if (!ctxE) {
						err = ERR_INCONSISTENT_MULTIAVA;
						goto returnerr;
					}
					re->typeLen = ctxE->typeLen;
					re->type = ctxE->type;
					/* fix country attribute... */
					if ((re->typeLen == 1) && (re->valLen > 2) &&
					    ((*re->type == L'C') || (*re->type == L'c')))
						re->type = L"O";
					ctxE = ctxE->next;
					re = re->next;
				}	
			}
			rdnEntry = &(*rdnEntry)->up;
			ctxEntry = ctxEntry->up;
		}
	}
	err = __NWDSCopyRDN(rdnEntry, ctxEntry);
	if (!err) {
		if (!typeLess)
			err = __NWDSApplyDefaultNamingRule(&rdn);
		if (!err)
			err = __NWDSExtractRDN(&rdn, dst, typeLess, 0);
	}
returnerr:;	
	__NWDSDestroyRDN(&rdn);
	return err;
}

NWDSCCODE NWDSAbbreviateNameW(NWDSContextHandle ctx, const wchar_t* src,
		wchar_t* dst) {
	struct RDNInfo rdn;
	struct RDNInfo ctxRDN;
	struct RDNEntry* ctxEntry;
	struct RDNEntry** rdnEntry;
	struct RDNEntry** ostop;
	size_t oldDepth;
	NWDSCCODE err;
	int typeLess;
	u_int32_t flags;
	size_t trailingDots;
	size_t adots;
	
	err = NWDSGetContext(ctx, DCK_FLAGS, &flags);
	if (err)
		return err;
	typeLess = (flags & DCV_TYPELESS_NAMES) ? 1 : 0;
	/* check for specials... */
	if (*src == '[') {
		if (!wcscasecmp(src, L"[Root]") ||
		    !wcscasecmp(src, L"[Supervisor]") ||
		    !wcscasecmp(src, L"[Public]") ||
		    !wcscasecmp(src, L"[Self]") ||
		    !wcscasecmp(src, L"[Creator]") ||
		    !wcscasecmp(src, L"[Inheritance Mask]") ||
		    !wcscasecmp(src, L"[Root Template]") ||
		    !wcscasecmp(src, L"[Nothing]")) {
		    	wcscpy(dst, src);
			return 0;
		}
	}
	err = __NWDSCreateRDN(&rdn, src, NULL);
	if (err)
		return err;
	err = NWDSGetContext2(ctx, DCK_RDN, &ctxRDN, sizeof(ctxRDN));
	if (err) {
		__NWDSDestroyRDN(&rdn);
		return err;
	}
	
	trailingDots = 0;
	
	ctxEntry = ctxRDN.end;
	rdnEntry = &rdn.end;
	oldDepth = rdn.depth;

	if (rdn.depth < ctxRDN.depth) {
		while (rdn.depth < ctxRDN.depth--) {
			ctxEntry = ctxEntry->up;
			trailingDots++;
		}
	} else if (rdn.depth > ctxRDN.depth) {
		while (rdn.depth-- >ctxRDN.depth) {
			rdnEntry = &(*rdnEntry)->up;
		}
	}
	ostop = rdnEntry;
	adots = 0;
	while (*rdnEntry) {
		struct RDNEntry* re = *rdnEntry;
	
		adots++;
			
		if (((re->typeLen && ctxEntry->typeLen) &&
		     ((re->typeLen != ctxEntry->typeLen) ||
		      wcsncasecmp(re->type, ctxEntry->type, re->typeLen))) ||
		     ((re->valLen != ctxEntry->valLen) ||
		      wcsncasecmp(re->val, ctxEntry->val, re->valLen))) {
			rdnEntry = &re->up;
			ctxEntry = ctxEntry->up;
			trailingDots += adots;
			ostop = rdnEntry;
			adots = 0;
		} else {
			rdnEntry = &re->up;
			ctxEntry = ctxEntry->up;
		}
	}
	if (ostop == &rdn.end) {
		if (*ostop) {
			ostop = &(*ostop)->up;	/* do 'a..' instead of '.' */
			trailingDots++;
		} else
			trailingDots = 0;	/* emit '[Root]' instead of '.' */
	}
	ctxEntry = *ostop;
	*ostop = NULL;
	if (!err)
		err = __NWDSExtractRDN(&rdn, dst, typeLess, trailingDots);
	*ostop = ctxEntry;
	__NWDSDestroyRDN(&rdn);
	return err;
}

NWDSCCODE NWDSCanonicalizeName(NWDSContextHandle ctx, const char* src,
		char* dst) {
	wchar_t srcW[MAX_DN_CHARS+1];
	wchar_t dstW[MAX_DN_CHARS+1];
	NWDSCCODE err;
	
	err = NWDSXlateFromCtx(ctx, srcW, sizeof(srcW), src);
	if (err)
		return err;
	err = NWDSCanonicalizeNameW(ctx, srcW, dstW);
	if (err)
		return err;
	return NWDSXlateToCtx(ctx, dst, MAX_DN_BYTES, dstW);
}

NWDSCCODE NWDSAbbreviateName(NWDSContextHandle ctx, const char* src,
		char* dst) {
	wchar_t srcW[MAX_DN_CHARS+1];
	wchar_t dstW[MAX_DN_CHARS+1];
	NWDSCCODE err;
	
	err = NWDSXlateFromCtx(ctx, srcW, sizeof(srcW), src);
	if (err)
		return err;
	err = NWDSAbbreviateNameW(ctx, srcW, dstW);
	if (err)
		return err;
	return NWDSXlateToCtx(ctx, dst, MAX_DN_BYTES, dstW);
}

NWDSCCODE NWDSRemoveAllTypes(NWDSContextHandle ctx, const char* src,
		char* dst) {
	wchar_t srcW[MAX_DN_CHARS+1];
	wchar_t dstW[MAX_DN_CHARS+1];
	NWDSCCODE err;
	
	err = NWDSXlateFromCtx(ctx, srcW, sizeof(srcW), src);
	if (err)
		return err;
	err = NWDSRemoveAllTypesW(ctx, srcW, dstW);
	if (err)
		return err;
	return NWDSXlateToCtx(ctx, dst, MAX_DN_BYTES, dstW);
}

#define DS_RESOLVE_V0			     0

#define DS_RESOLVE_ENTRY_ID		0x0001
#define DS_RESOLVE_READABLE		0x0002
#define DS_RESOLVE_WRITEABLE		0x0004
#define DS_RESOLVE_MASTER		0x0008
#define DS_RESOLVE_CREATE_ID		0x0010
#define DS_RESOLVE_WALK_TREE		0x0020
#define DS_RESOLVE_DEREF_ALIASES	0x0040
/*                                      0x0080 */
#define DS_RESOLVE_ENTRY_EXISTS		0x0100
#define DS_RESOLVE_DELAYED_CREATE_ID	0x0200
#define DS_RESOLVE_EXHAUST_REPLICAS	0x0400
#define DS_RESOLVE_ACCEPT_NOT_PRESENT	0x0800
#define DS_RESOLVE_FORCE_REFERRALS	0x1000
#define DS_RESOLVE_PREFFER_REFERRALS	0x2000
#define DS_RESOLVE_ONLY_REFERRALS	0x4000
/* Entry ID?                            0x8000 */

#define DS_RESOLVE_REPLY_LOCAL_ENTRY	0x0001
#define DS_RESOLVE_REPLY_REMOTE_ENTRY	0x0002
#define DS_RESOLVE_REPLY_REFERRAL	0x0004
#define DS_RESOLVE_REPLY_REFERRAL_AND_ENTRY 0x0006

static inline NWDSCCODE __NWDSResolveNameInt(NWDSContextHandle ctx, NWCONN_HANDLE conn, u_int32_t version, u_int32_t flag,
		const char* name, Buf_T* reply, Buf_T* rq) {
	NWDSCCODE err;
	void* p;
	size_t rpl_len;
	
	p = NWDSBufPutPtr(rq, 12);
	if (!p)
		return ERR_BUFFER_FULL;
	DSET_LH(p, 0, version);
	DSET_LH(p, 4, flag);
	DSET_LH(p, 8, ctx->dck.confidence);
	err = NWDSPutAttrVal_DIST_NAME(ctx, rq, name);
	if (err)
		return err;
	err = NWDSBufPutLE32(rq, ctx->dck.transports);
	if (err)
		return err;
	err = NWDSBufPut(rq, ctx->dck.transport_types, ctx->dck.transports * sizeof(nuint32));
	if (err)
		return err;
	err = NWDSBufPutLE32(rq, ctx->dck.transports);
	if (err)
		return err;
	err = NWDSBufPut(rq, ctx->dck.transport_types, ctx->dck.transports * sizeof(nuint32));
	if (err)
		return err;
							
	NWDSBufStartPut(reply);
	p = NWDSBufPutPtrLen(reply, &rpl_len);
	err = ncp_send_nds_frag(conn, DSV_RESOLVE_NAME, rq->data, rq->curPos-rq->data, p,
			rpl_len, &rpl_len);
	if (!err) {
		if (rpl_len < 8)
			err = ERR_INVALID_SERVER_RESPONSE;
		else {
			NWDSBufPutSkip(reply, rpl_len);
		}
	}
	NWDSBufFinishPut(reply);
	return err;
}

NWDSCCODE NWDSResolveNameInt(NWDSContextHandle ctx, NWCONN_HANDLE conn, u_int32_t version, u_int32_t flag, 
		const char* name, Buf_T* reply) {
	NWDSCCODE err;
	Buf_T* rq;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &rq);
	if (err)
		return err;
	err = __NWDSResolveNameInt(ctx, conn, version, flag, name, reply, rq);
	NWDSFreeBuf(rq);
	return err;
}
		
/* server oriented NDS calls */

NWDSCCODE NWDSMapNameToID(NWDSContextHandle ctx, NWCONN_HANDLE conn,
		const char* name, NWObjectID* ID) {
	NWDSCCODE err;
	Buf_T* rp;
	
	err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &rp);
	if (err)
		return err;
	err = NWDSResolveNameInt(ctx, conn, DS_RESOLVE_V0, DS_RESOLVE_CREATE_ID, name, rp);
	if (!err) {
		nuint32 replytype;
		
		err = NWDSBufGetLE32(rp, &replytype);
		if (!err) {
			/* we requested local entry */
			if (replytype != DS_RESOLVE_REPLY_LOCAL_ENTRY)
				err = ERR_INVALID_SERVER_RESPONSE;
			else {
				err = NWDSBufGetID(rp, ID);
			}
		}
	}
	NWDSFreeBuf(rp);
	return err;
}

NWDSCCODE NWDSMapIDToName(NWDSContextHandle ctx, NWCONN_HANDLE conn,
		NWObjectID ID, char* name) {
	NWDSCCODE err;
	Buf_T* b;
	nuint32 replyFmt = 0;
	nuint32 dsiFmt = DSI_ENTRY_DN;
	nuint32 val;
	
	err = NWDSGetContext(ctx, DCK_FLAGS, &val);
	if (err)
		return err;
	if (val & DCV_TYPELESS_NAMES)
		replyFmt |= 1;
	if (val & DCV_DEREF_BASE_CLASS)
		dsiFmt |= DSI_DEREFERENCE_BASE_CLASS;
	/* DCV_DEREF_ALIASES does not have DSI_ counterpart */
	/* DCV_DISALLOW_REFERRALS is N/A for MapIDToName (server centric) */
	/* DCV_ASYNC_MODE: TODO (if someone knows...) */
	replyFmt |= ctx->dck.name_form;
	err = NWDSAllocBuf(MAX_DN_BYTES, &b);
	if (err)
		return err;
	err = NWDSGetDSIRaw(conn, dsiFmt, replyFmt, ID, b);
	if (!err) {
		/* Netware DS 6.02, 7.26, 7.28 and 7.30 (all I tested)
		 * has problem that [Self] maps to 0x040000FF, but
		 * 0x040000FF maps to [Self][Inheritance Mask]. Do you
		 * think that I should add hack to map [Self][Inheritance Mask]
		 * here? Or is it meant in another way and I oversight
		 * something? */
		err = NWDSGetAttrVal_DIST_NAME(ctx, b, name, replyFmt);
	}
	NWDSFreeBuf(b);
	return err;
}

static NWDSCCODE NWDSResolveName99(NWDSContextHandle ctx, NWCONN_HANDLE conn, const char* name, u_int32_t flag,
		NWCONN_HANDLE* resconn, NWObjectID* ID, Buf_T** rplbuf, void*** array, unsigned int* saddr, int* idepth) {
	NWDSCCODE err;
	Buf_T* rp;
	
	*rplbuf = NULL;
	*array = NULL;
	*saddr = 0;
	*resconn = NULL;
	err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &rp);
	if (err)
		return err;
	flag &= ~(DS_RESOLVE_DEREF_ALIASES);
	if (ctx->dck.flags & DCV_DEREF_ALIASES)
		flag |= DS_RESOLVE_DEREF_ALIASES;
		
	err = NWDSResolveNameInt(ctx, conn, DS_RESOLVE_V0, flag, name, rp);
	if (!err) {
		nuint32 replytype;
		nuint32 dummy;
			
		err = NWDSBufGetLE32(rp, &replytype);
		if (!err) switch (replytype) {
			case DS_RESOLVE_REPLY_LOCAL_ENTRY:
				dfprintf(stderr, "Local Entry\n");
				err = NWDSBufGetID(rp, ID);
				if (err)
					break;
				ncp_conn_use(conn);
				*resconn = conn;
				break;
			case DS_RESOLVE_REPLY_REMOTE_ENTRY:
				dfprintf(stderr, "Remote Entry\n");
				err = NWDSBufGetID(rp, ID);
				if (err)
					break;
				err = NWDSBufGetLE32(rp, &dummy);
				if (err)
					break;
				err = NWDSFindConnection(ctx, resconn, rp, NWDSFINDCONN_CREATEALLOWED);
				break;
			case DS_RESOLVE_REPLY_REFERRAL_AND_ENTRY:
			case DS_RESOLVE_REPLY_REFERRAL:
				{
					u_int32_t depth;
					u_int32_t servercount;
					u_int32_t parsedaddresses;
					void** addresses;
					unsigned int pos;
							
					if (replytype == DS_RESOLVE_REPLY_REFERRAL) {
						dfprintf(stderr, "Referrals\n");
						err = NWDSBufGetLE32(rp, &depth);
						if (err)
							break;
					} else {
						u_int32_t dummy;
						NWObjectID objid;
								
						dfprintf(stderr, "Referrals + ID\n");
						err = NWDSBufGetLE32(rp, &dummy);
						if (err)
							break;
						err = NWDSBufGetID(rp, &objid);
						if (err)
							break;
						if (objid != (NWObjectID)~0) {
							*ID = objid;
							ncp_conn_use(conn);
							*resconn = conn;
							dfprintf(stderr, "Got ID\n");
							break;
						}
						depth = 0xFFFF0000; /* ignore all referrals */
					}
					err = NWDSBufGetLE32(rp, &servercount);
					if (err)
						break;
					if (servercount > 1024) { /* some unreasonable value */
						/* return ERR_TOO_MANY_REFERRALS; ??? */
						servercount = 1024;
					}
					if (!servercount) {
						err = ERR_NO_REFERRALS;
						break;
					}
					dfprintf(stderr, "%d servers\n", servercount);
					addresses = (void**)malloc(sizeof(void*) * servercount);
					if (!addresses) {
						err = ERR_NOT_ENOUGH_MEMORY;
						break;
					}
					parsedaddresses = 0;
					for (pos = 0; pos < servercount; pos++) {
						u_int32_t saddr;
						u_int32_t dummy;
						u_int32_t len;
						void* data;
						u_int32_t i;
									
						addresses[parsedaddresses] = NWDSBufPeekPtr(rp);
						err = NWDSBufGetLE32(rp, &saddr);
						if (err)
							break;
						for (i = 0; i < saddr; i++) {
							err = NWDSBufGetLE32(rp, &dummy);
							if (err)
								goto packetEOF;
							err = NWDSBufGetLE32(rp, &len);
							if (err)
								goto packetEOF;
							data = NWDSBufGetPtr(rp, len);
							if (!data) {
								err = ERR_BUFFER_EMPTY;
								goto packetEOF;
							}
						}
						parsedaddresses++;
					}
				packetEOF:;
					if (err) {
						free(addresses);
						break;
					}
					if (!parsedaddresses) {
						free(addresses);
						err = ERR_NO_REFERRALS;
						break;
					}
					*rplbuf = rp;
					*array = addresses;
					*saddr = parsedaddresses;
					*idepth = depth;
				}
				return 0;
			default:
				err = ERR_INVALID_SERVER_RESPONSE;
				dfprintf(stderr, "Invalid server response\n");
				break;
		}
	}
	NWDSFreeBuf(rp);
	return err;
}

NWDSCCODE NWDSResolveName2(NWDSContextHandle ctx, const char* name, u_int32_t flag,
		NWCONN_HANDLE* resconn, NWObjectID* ID) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	Buf_T* rp;
	void** addresses;
	unsigned int servers;
	int depth;
		
	err = NWDSGetConnection(ctx, &conn);
	if (err)
		return err;
	err = NWDSResolveName99(ctx, conn, name, flag, resconn, ID, &rp, &addresses, &servers, &depth);
	if (!err && rp) {
		unsigned int pos;
		unsigned int crmode;
		NWDSCCODE dserr = ERR_ALL_REFERRALS_FAILED;
again:;
		dfprintf(stderr, "Received referrals\n");
		crmode = NWDSFINDCONN_NOCREATE;
again2:;
		for (pos = 0; pos < servers; pos++) {
			NWCONN_HANDLE conn2;
			Buf_T* buf;
			Buf_T* b2;
			void** addr2;
			unsigned int serv2;
			int depth2;
									
			if (!addresses[pos])
				continue;
			err = NWDSCreateBuf(&buf, addresses[pos], 0x7FFFFFF);
			if (err)
				continue;
			err = NWDSFindConnection(ctx, &conn2, buf, crmode);
			NWDSFreeBuf(buf);
			if (err)
				continue;
			dfprintf(stderr, "Processing referrals\n");
			addresses[pos] = NULL;
			err = NWDSResolveName99(ctx, conn2, name, flag, resconn, ID, &b2, &addr2, &serv2, &depth2);
			if (!err) {
				if (!b2) {
					NWCCCloseConn(conn2);
					goto freeExit;
				}
				if (depth2 < depth) {
					free(addresses);
					addresses = addr2;
					NWDSFreeBuf(rp);
					rp = b2;
					depth = depth2;
					servers = serv2;
					NWCCCloseConn(conn);
					conn = conn2;
					goto again;
				}
				dfprintf(stderr, "Referral ignored; %d >= %d\n", depth2, depth);
				free(addr2);
				NWDSFreeBuf(b2);
				err = dserr;
			} else
				dserr = err;
			NWCCCloseConn(conn2);
		}
		if (crmode == NWDSFINDCONN_NOCREATE) {
			dfprintf(stderr, "Connection not found, making new\n");
			crmode = NWDSFINDCONN_CREATEALLOWED;
			goto again2;
		}
freeExit:;
		free(addresses);
		NWDSFreeBuf(rp);
	}
	NWDSConnectionFinished(ctx, conn);
	return err;
}

NWDSCCODE NWDSResolveName(NWDSContextHandle ctx, const char* name,
		NWCONN_HANDLE* resconn, NWObjectID* ID) {
	return NWDSResolveName2(ctx, name, DS_RESOLVE_WRITEABLE,
		resconn, ID);
}

static NWDSCCODE NWDSGetServerNameAddress(NWCONN_HANDLE conn, u_int32_t version,
	u_int32_t nameform, Buf_T* reply) {
	NWDSCCODE err;
	unsigned char rq[8];
	size_t rpl_len;
	void* rpl;
	
	DSET_LH(rq, 0, version);
	DSET_LH(rq, 4, nameform);
	NWDSBufStartPut(reply);
	rpl = NWDSBufPutPtrLen(reply, &rpl_len);
	err = ncp_send_nds_frag(conn, DSV_GET_SERVER_ADDRESS, rq, sizeof(rq), rpl,
		rpl_len, &rpl_len);
	if (!err) {
		if (rpl_len < 8) {
			/* at least 0 bytes name and 0 network addresses */
			err = ERR_INVALID_SERVER_RESPONSE;
		} else {
			NWDSBufPutSkip(reply, rpl_len);
			NWDSBufFinishPut(reply);
		}
	}
	return err;
}

/* NWSDK returns abbreviated typeless name of server ... */
NWDSCCODE NWDSGetServerDN(NWDSContextHandle ctx, NWCONN_HANDLE conn, char* name) {
	NWDSCCODE err;
	Buf_T* reply;
	u_int32_t replyFmt;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &reply);
	if (err)
		return err;
	replyFmt = ctx->dck.name_form;
	if (ctx->dck.flags & DCV_TYPELESS_NAMES)
		replyFmt |= 1;
	/* COMPATIBILITY
	replyFmt |= 1;
	*/
	err = NWDSGetServerNameAddress(conn, 0, replyFmt, reply);
	if (!err) {
		err = NWDSGetAttrVal_DIST_NAME(ctx, reply, name, replyFmt);
		/* COMPATIBILITY
		if (!err)
			err = NWDSAbbreviateName(ctx, name, name);
		*/
	}
	NWDSFreeBuf(reply);
	return err;
}

static NWDSCCODE NWDSGetServerAddressInt(NWCONN_HANDLE conn, size_t* addrcnt,
		Buf_T* naddrs) {
	NWDSCCODE err;
	Buf_T* reply;
	
	err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &reply);
	if (err)
		return err;
	/* version: 0, format: partial dot, typeless */
	err = NWDSGetServerNameAddress(conn, 0, 1, reply);
	if (!err) {
		err = NWDSBufSkipCIString(reply);
		if (!err) {
			nuint32 cnt;
			
			err = NWDSBufGetLE32(reply, &cnt);
			if (!err) {
				if (addrcnt)
					*addrcnt = cnt;
				if (naddrs) {
					NWDSBufStartPut(naddrs);
					/* copy addresses... maybe we can 
					   copy whole rest of buffer
					   as unstructured data block */
					while (cnt--) {
						nuint32 val;
						void* ptr;

						/* address type */
						err = NWDSBufGetLE32(reply, &val);
						if (err) break;
						err = NWDSBufPutLE32(naddrs, val);
						if (err) break;
						/* address length */
						err = NWDSBufGetLE32(reply, &val);
						if (err) break;
						err = NWDSBufPutLE32(naddrs, val);
						if (err) break;

						/* address value */
						ptr = NWDSBufGetPtr(reply, val);
						if (!ptr) {
							err = ERR_BUFFER_EMPTY;
							break;
						}
						err = NWDSBufPut(naddrs, ptr, val);
						if (err)
							break;
					}
					NWDSBufFinishPut(naddrs);
				}
			}
		}
	}
	NWDSFreeBuf(reply);
	return err;
}

NWDSCCODE NWDSGetServerAddress(NWDSContextHandle ctx, NWCONN_HANDLE conn, size_t* addrcnt,
		Buf_T* naddrs) {
	NWDSCCODE err;

	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	return NWDSGetServerAddressInt(conn, addrcnt, naddrs);
}

/* difference? */
NWDSCCODE NWDSGetServerAddress2(NWDSContextHandle ctx, NWCONN_HANDLE conn, size_t* addrcnt,
		Buf_T* naddrs) {
	return NWDSGetServerAddress(ctx, conn, addrcnt, naddrs);
}

static NWDSCCODE __NWCCGetServerAddressPtr(NWCONN_HANDLE conn, size_t* count, nuint8** data) {
	if (!conn->serverAddresses.buffer) {
		NWDSCCODE err;
		Buf_T* bb;
		size_t cnt;
		
		err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &bb);
		if (err)
			return err;
		err = NWDSGetServerAddressInt(conn, &cnt, bb);
		if (err)
			return err;
		/* TODO: lock connection */
		if (conn->serverAddresses.buffer) {
			NWDSFreeBuf(bb);
		} else {
			conn->serverAddresses.buffer = bb;
			conn->serverAddresses.count = cnt;
		}
		/* unlock connection */
	}
	*count = conn->serverAddresses.count;
	*data = NWDSBufPeekPtr(conn->serverAddresses.buffer);
	return 0;
}
#endif	/* !__MAKE_SULIB__ */
